/*
 * BarGauge: Widget to periodically display a value in form of a horizontal bar.
 * based on Athena StripChart Widget.
 *
 * Author: Gabor Herr
 *
 * $Id: BarGauge.c,v 1.1 1992/10/25 22:46:42 gabor Exp $
 */

/***********************************************************
Copyright 1992, by Gabor Herr
Copyright 1987, 1988 by Digital Equipment Corporation, Maynard, Massachusetts,
and the Massachusetts Institute of Technology, Cambridge, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of Gabor Herr or Digital or MIT not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

GABOR HERR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

******************************************************************/

#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/XawInit.h>
#include "BarGaugeP.h"
#include <X11/Xfuncs.h>

#define MS_PER_SEC 1000

/* Private Data */

#define offset(field) XtOffsetOf(BarGaugeRec, field)

static XtResource resources[] = {
    {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
	offset(core.width), XtRImmediate, (XtPointer) 120},
    {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
	offset(core.height), XtRImmediate, (XtPointer) 12},
    {XtNupdate, XtCInterval, XtRInt, sizeof(int),
        offset(bar_gauge.update), XtRImmediate, (XtPointer) 10},
    {XtNminScale, XtCScale, XtRInt, sizeof(int),
        offset(bar_gauge.min_scale), XtRImmediate, (XtPointer) 1},
    {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
        offset(bar_gauge.fgpixel), XtRString, XtDefaultForeground},
    {XtNhighlight, XtCForeground, XtRPixel, sizeof(Pixel),
        offset(bar_gauge.hipixel), XtRString, XtDefaultForeground},
    {XtNgetValue, XtCCallback, XtRCallback, sizeof(XtPointer),
        offset(bar_gauge.get_value), XtRImmediate, (XtPointer) NULL},
};

#undef offset

static void Initialize(), Destroy(), Redisplay();
static Boolean SetValues();
static void repaint_window();

BarGaugeClassRec barGaugeClassRec = {
    { /* core fields */
    /* superclass		*/	(WidgetClass) &simpleClassRec,
    /* class_name		*/	"BarGauge",
    /* size			*/	sizeof(BarGaugeRec),
    /* class_initialize		*/	XawInitializeWidgetSet,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	XtInheritRealize,
    /* actions			*/	NULL,
    /* num_actions		*/	0,
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	XtExposeCompressMultiple |
	                                XtExposeGraphicsExposeMerged,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	NULL,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	NULL,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	NULL,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
    },
    { /* Simple class fields */
    /* change_sensitive		*/	XtInheritChangeSensitive
    }
};

WidgetClass barGaugeWidgetClass = (WidgetClass) &barGaugeClassRec;

/****************************************************************
 *
 * Private Procedures
 *
 ****************************************************************/

static void draw_it();

/*	Function Name: CreateGC
 *	Description: Creates the GC's
 *	Arguments: w - the strip chart widget.
 *                 which - which GC's to create.
 *	Returns: none
 */

static void
CreateGC(w, which)
BarGaugeWidget w;
unsigned int which;
{
  XGCValues	myXGCV;

  if (which & FOREGROUND) {
    myXGCV.foreground = w->bar_gauge.fgpixel;
    w->bar_gauge.fgGC = XtGetGC((Widget) w, GCForeground, &myXGCV);
  }

  if (which & HIGHLIGHT) {
    myXGCV.foreground = w->bar_gauge.hipixel;
    w->bar_gauge.hiGC = XtGetGC((Widget) w, GCForeground, &myXGCV);
  }
}

/*	Function Name: DestroyGC
 *	Description: Destroys the GC's
 *	Arguments: w - the strip chart widget.
 *                 which - which GC's to destroy.
 *	Returns: none
 */

static void
DestroyGC(w, which)
BarGaugeWidget w;
unsigned int which;
{
  if (which & FOREGROUND) 
    XtReleaseGC((Widget) w, w->bar_gauge.fgGC);

  if (which & HIGHLIGHT) 
    XtReleaseGC((Widget) w, w->bar_gauge.hiGC);
}

/* ARGSUSED */
static void Initialize (greq, gnew)
    Widget greq, gnew;
{
    BarGaugeWidget w = (BarGaugeWidget)gnew;

    if (w->bar_gauge.update > 0)
        w->bar_gauge.interval_id = XtAppAddTimeOut( 
					XtWidgetToApplicationContext(gnew),
					w->bar_gauge.update * MS_PER_SEC, 
					draw_it, (XtPointer) gnew);
    CreateGC(w, (unsigned int) ALL_GCS);

    w->bar_gauge.scale = w->bar_gauge.min_scale;
    w->bar_gauge.max_value = 0.0;
    w->bar_gauge.last_value = 0.0;
}
 
static void Destroy (gw)
     Widget gw;
{
     BarGaugeWidget w = (BarGaugeWidget)gw;

     if (w->bar_gauge.update > 0)
         XtRemoveTimeOut (w->bar_gauge.interval_id);
     DestroyGC(w, (unsigned int) ALL_GCS);
}

/*
 * NOTE: This function really needs to recieve graphics exposure 
 *       events, but since this is not easily supported until R4 I am
 *       going to hold off until then.
 */

/* ARGSUSED */
static void Redisplay(w, event, region)
     Widget w;
     XEvent *event;
     Region region;
{
	(void) repaint_window ((BarGaugeWidget)w);
}

/* ARGSUSED */
static void 
draw_it(client_data, id)
XtPointer client_data;
XtIntervalId *id;		/* unused */
{
   BarGaugeWidget w = (BarGaugeWidget)client_data;
   double value;
   
   if (w->bar_gauge.update > 0)
       w->bar_gauge.interval_id =
       XtAppAddTimeOut(XtWidgetToApplicationContext( (Widget) w),
		       w->bar_gauge.update * MS_PER_SEC,draw_it,client_data);

   /* Get the value, stash the point and draw corresponding line. */

   if (w->bar_gauge.get_value == NULL)
       return;

   XtCallCallbacks( (Widget)w, XtNgetValue, (XtPointer)&value );

   /* 
    * Keep w->bar_gauge.max_value up to date, and if this data 
    * point is off the graph, change the scale to make it fit. 
    */
   
   if (value > w->bar_gauge.max_value) {
       w->bar_gauge.max_value = value;
       if (value > 1.0) w->bar_gauge.scale = (int) value;
   }

   w->bar_gauge.last_value = value;
   
   if (XtIsRealized((Widget)w))
       repaint_window(w);
} /* draw_it */

static void 
repaint_window(w)
BarGaugeWidget w;
{
   if (XtIsRealized((Widget)w)) {       

       int x = (int) ( (int)(w->core.width * w->bar_gauge.last_value) / w->bar_gauge.scale);

       XClearWindow( XtDisplay (w), XtWindow (w));
       XFillRectangle(XtDisplay(w), XtWindow(w), w->bar_gauge.fgGC,
		      0, 0, x, w->core.height);

       XFlush(XtDisplay(w));		    /* Flush output buffers */
   }
}

/* ARGSUSED */
static Boolean SetValues (current, request, new)
    Widget current, request, new;
{
    BarGaugeWidget old = (BarGaugeWidget)current;
    BarGaugeWidget w = (BarGaugeWidget)new;
    Boolean ret_val = FALSE;
    unsigned int new_gc = NO_GCS;

    if (w->bar_gauge.update != old->bar_gauge.update) {
	if (old->bar_gauge.update > 0)
	    XtRemoveTimeOut (old->bar_gauge.interval_id);
	if (w->bar_gauge.update > 0)
	    w->bar_gauge.interval_id =
		XtAppAddTimeOut(XtWidgetToApplicationContext(new),
				w->bar_gauge.update * MS_PER_SEC,
				draw_it, (XtPointer)w);
    }

    if ( w->bar_gauge.min_scale > (int) ((w->bar_gauge.max_value) + 1) )
      ret_val = TRUE;
     
    if ( w->bar_gauge.fgpixel != old->bar_gauge.fgpixel ) {
      new_gc |= FOREGROUND;
      ret_val = True;
    }
    
    if ( w->bar_gauge.hipixel != old->bar_gauge.hipixel ) {
      new_gc |= HIGHLIGHT;
      ret_val = True;
    }
    
    DestroyGC(old, new_gc);
    CreateGC(w, new_gc);

    return( ret_val );
}
