/*
 * sysinfo.c		- get kernel info
 *
 * Copyright (c) 1992 by Gabor Herr, completely based on ps & free
 * which is Copyright (c) 1992 Branko Lankester
 *
 */


#include <sys/types.h>
#include <linux/sched.h>
#include "sysinfo.h"

#define	SWP_USED	1
#define	SWP_WRITEOK	3

struct swap_info_struct {
	unsigned long flags;
	struct inode * swap_file;
	unsigned int swap_device;
	unsigned char * swap_map;
	char * swap_lockmap;
	int lowest_bit;
	int highest_bit;
};

#define pg_shift 2

int get_meminfo( struct meminfo *result )
{
    static unsigned short *memmap;
    static unsigned long pages, _mem_map, _nr_buffers;
    static unsigned long _nr_buffer_heads;
    unsigned used=0, freepg=0, shared=0;
    unsigned long buf_pages;
    int i;

    if (memmap == NULL) {
	pages = get_kword(k_addr("_high_memory")) / 4096;
	memmap = (unsigned short *) xmalloc(pages * sizeof *memmap);
	_mem_map = get_kword(k_addr("_mem_map"));
	_nr_buffers = k_addr("_nr_buffers");
	_nr_buffer_heads = k_addr("_nr_buffer_heads");
    }
    buf_pages = get_kword(_nr_buffers) / 4;	/* assumes block size = 1k */
    buf_pages +=		/* add space used by buffer heads */
	get_kword(_nr_buffer_heads) / (4096 / sizeof(struct buffer_head));
    kmemread(memmap, _mem_map, pages * sizeof *memmap);
    for (i = 0; i < pages; ++i)
	switch (memmap[i]) {
	    case 0: ++freepg; break;
	    case MAP_PAGE_RESERVED: break;
	    default: shared += memmap[i] - 1;
	    case 1: ++used;
	}

    result->total = freepg + used << 2;
    result->used  = used - buf_pages << 2;
    result->cache = buf_pages << 2;
    result->free  = freepg << 2;
    result->shared= shared << 2;
}

get_swapinfo( int *swapdevs, struct meminfo *result )
{
    unsigned freepg=0, used, shared;
    int i, sw;
    unsigned char swapmap[4096];
    struct swap_info_struct swapinfo[MAX_SWAPFILES];
    unsigned nr_swapfiles;
    
    nr_swapfiles = get_kword(k_addr("_nr_swapfiles"));
    *swapdevs = nr_swapfiles;

    if (nr_swapfiles == 0) return;

    kmemread(swapinfo, k_addr("_swap_info"), sizeof swapinfo);
    for (sw = 0; sw < nr_swapfiles; ++sw) {
	if (!(swapinfo[sw].flags & SWP_USED))
	    continue;
	freepg = 0; used = 0; shared = 0;
	kmemread(swapmap, swapinfo[sw].swap_map, 4096);
	for (i = 0; i < 4096; ++i)
	    switch (swapmap[i]) {
		case 0: ++freepg; break;
		case 128: break;
		default: shared += swapmap[i] - 1;
		case 1: ++used;
	    }

	result[sw].total = used + freepg << 2;
	result[sw].used  = used << 2;
	result[sw].free  = freepg << 2;
	result[sw].cache = 0;
	result[sw].shared= shared << 2;
    }
}

double get_load( void )
{
  static unsigned long avenrun = 0;
  unsigned long av[3];

  if (avenrun == 0)
    avenrun = k_addr("_avenrun");

  kmemread( av, avenrun, sizeof av );

  return (double)((av[0] * 1000) >> FSHIFT) / 1000;
}

double get_idle()
{
  static struct task_struct *taskp;
  static unsigned _task;
  static unsigned _jiffies;
  static int last_idle_time;
  static int last_jiffies;
  struct task_struct task0;
  int idle_time;
  int jiffies;
  int idle_ticks;
  int ticks;

  if (taskp == NULL) {
    _task = k_addr("_task");
    _jiffies = k_addr("_jiffies");
    kmemread(&taskp, _task, sizeof(struct task_struct*));
  }

  kmemread(&task0, taskp, sizeof(task0));

  idle_time = task0.utime + task0.stime;
  idle_ticks= idle_time - last_idle_time;
  last_idle_time = idle_time;

  jiffies = get_kword(_jiffies);
  ticks   = jiffies - last_jiffies;
  last_jiffies = jiffies;

  if (idle_ticks >= 0)
    return (double) idle_ticks / ticks;
  else
    return 0.0; 
}

