/*
 * vars.c: All the dealing of the irc variables are handled here. 
 *
 *
 * Written By Michael Sandrof
 *
 * Copyright(c) 1990 
 *
 * See the COPYRIGHT file, or do a HELP IRCII COPYRIGHT 
 */
#include <stdio.h>
#include "irc.h"
#include "status.h"
#include "window.h"
#include "lastlog.h"
#include "log.h"
#include "crypt.h"
#include "history.h"
#include "vars.h"
#include "input.h"
#include "ircaux.h"
#include "whois.h"
#include "translat.h"

/* IrcVariable: structure for each variable in the variable table */
typedef struct
{
	char	*name;		/* what the user types */
	int	type;		/* variable types, see below */
	int	integer;	/* int value of variable */
	char	*string;	/* string value of variable */
	void	(*func) ();	/* function to do every time variable is set */
	char	changed;	/* true if variable has changed this session */
	unsigned short	flags;	/* flags for this variable */
}	IrcVariable;

#define	VF_NODAEMON	0x0001

/* the types of IrcVariables */
#define BOOL_TYPE_VAR 0
#define CHAR_TYPE_VAR 1
#define INT_TYPE_VAR 2
#define STRING_TYPE_VAR 3

char	*var_settings[] =
{
	"OFF", "ON", "TOGGLE"};

static	void	exec_warning();
static	void	input_warning();

extern int load_depth; /* For the NOVICE variable. Complain loudlly if turned
			* off manually.
			*/

/*
 * irc_variable: all the irc variables used.  Note that the integer and
 * boolean defaults are set here, which the string default value are set in
 * the init_variables() procedure 
 */
static	IrcVariable irc_variable[] =
{
	"ALWAYS_SPLIT_BIGGEST",	BOOL_TYPE_VAR,	DEFAULT_ALWAYS_SPLIT_BIGGEST,
		NULL, NULL, 0, 0,
	"AUTO_UNMARK_AWAY",	BOOL_TYPE_VAR,	DEFAULT_AUTO_UNMARK_AWAY,
		NULL, NULL, 0, 0,
	"AUTO_WHOWAS",		BOOL_TYPE_VAR,	DEFAULT_AUTO_WHOWAS,
		NULL, NULL, 0, 0,
	"BEEP",			BOOL_TYPE_VAR,	DEFAULT_BEEP,
		NULL, NULL, 0, 0,
	"BEEP_MAX",		INT_TYPE_VAR,	DEFAULT_BEEP_MAX,
		NULL, NULL, 0, 0,
	"BEEP_ON_MSG",		STRING_TYPE_VAR,0,
		NULL, set_beep_on_msg, 0, 0,
	"BEEP_WHEN_AWAY",	INT_TYPE_VAR,	DEFAULT_BEEP_WHEN_AWAY,
		NULL, NULL, 0, 0,
	"BOLD_VIDEO",		BOOL_TYPE_VAR,	DEFAULT_BOLD_VIDEO,
		NULL, NULL, 0, 0,
	"CHANNEL_NAME_WIDTH",	INT_TYPE_VAR,	DEFAULT_CHANNEL_NAME_WIDTH,
		NULL, update_all_status, 0, 0,
	"CLIENT_INFORMATION",	STRING_TYPE_VAR,0,
		NULL, NULL, 0, 0,
	"CLOCK",		BOOL_TYPE_VAR,	DEFAULT_CLOCK,
		NULL, update_all_status, 0, 0,
	"CLOCK_24HOUR",		BOOL_TYPE_VAR,	DEFAULT_CLOCK_24HOUR,
		NULL, reset_clock, 0, 0,
	"CLOCK_ALARM",		STRING_TYPE_VAR,0,
		NULL, set_alarm, 0, 0,
	"CMDCHARS",		STRING_TYPE_VAR,0,
		NULL, NULL, 0, 0,
	"COMMAND_MODE",		BOOL_TYPE_VAR,	DEFAULT_COMMAND_MODE,
		NULL, NULL, 0, 0,
	"CONTINUED_LINE",	STRING_TYPE_VAR,0,
		NULL, set_continued_line, 0, 0,
	"DCC_BLOCK_SIZE",	INT_TYPE_VAR,	DEFAULT_DCC_BLOCK_SIZE,
		NULL, NULL, 0, 0,
	"DEBUG",		INT_TYPE_VAR,	0,
		NULL, NULL, 0, 0,
	"DISPLAY",		BOOL_TYPE_VAR,	DEFAULT_DISPLAY,
		NULL, NULL, 0, 0,
	"ENCRYPT_PROGRAM",	STRING_TYPE_VAR,0,
		NULL, NULL, 0, VF_NODAEMON,
	"EXEC_PROTECTION",	BOOL_TYPE_VAR,	DEFAULT_EXEC_PROTECTION,
		NULL, exec_warning, 0, VF_NODAEMON,
	"FLOOD_AFTER",		INT_TYPE_VAR,	DEFAULT_FLOOD_AFTER,
		NULL, NULL, 0, 0,
	"FLOOD_RATE",		INT_TYPE_VAR,	DEFAULT_FLOOD_RATE,
		NULL, NULL, 0, 0,
	"FLOOD_USERS",		INT_TYPE_VAR,	DEFAULT_FLOOD_USERS,
		NULL, NULL, 0, 0,
	"FLOOD_WARNING",	BOOL_TYPE_VAR,	DEFAULT_FLOOD_WARNING,
		NULL, NULL, 0, 0,
	"FULL_STATUS_LINE",	BOOL_TYPE_VAR,	DEFAULT_FULL_STATUS_LINE,
		NULL, update_all_status, 0, 0,
	"HELP_PAGER",		BOOL_TYPE_VAR,	DEFAULT_HELP_PAGER,
		NULL, NULL, 0, 0,
	"HELP_PATH",		STRING_TYPE_VAR,0,
		NULL, NULL, 0, 0,
	"HELP_PROMPT",		BOOL_TYPE_VAR,	DEFAULT_HELP_PROMPT,
		NULL, NULL, 0, 0,
	"HELP_SERVICE",		STRING_TYPE_VAR,0,
		NULL, NULL, 0, 0,
	"HELP_WINDOW",		BOOL_TYPE_VAR,	DEFAULT_HELP_WINDOW,
		NULL, NULL, 0, 0,
	"HIDE_PRIVATE_CHANNELS",BOOL_TYPE_VAR,	DEFAULT_HIDE_PRIVATE_CHANNELS,
		NULL, update_all_status, 0, 0,
	"HIGHLIGHT_CHAR",	STRING_TYPE_VAR,0,
		NULL, set_highlight_char, 0, 0,
	"HISTORY",		INT_TYPE_VAR,	DEFAULT_HISTORY,
		NULL, set_history_size, 0, 0,
	"HISTORY_FILE",		STRING_TYPE_VAR,0,
		NULL, set_history_file, 0, 0,
	"HOLD_MODE",		BOOL_TYPE_VAR,	DEFAULT_HOLD_MODE,
		NULL, reset_line_cnt, 0, 0,
	"HOLD_MODE_MAX",	INT_TYPE_VAR,	DEFAULT_HOLD_MODE_MAX,
		NULL, NULL, 0, 0,
	"INDENT",		BOOL_TYPE_VAR,	DEFAULT_INDENT,
		NULL, NULL, 0, 0,
	"INPUT_ALIASES",	BOOL_TYPE_VAR,	DEFAULT_INPUT_ALIASES,
		NULL, NULL, 0, 0,
	"INPUT_PROMPT",		STRING_TYPE_VAR,0,
		NULL, set_input_prompt, 0, 0,
	"INPUT_PROTECTION",	BOOL_TYPE_VAR,	DEFAULT_INPUT_PROTECTION,
		NULL, input_warning, 0, 0,
	"INSERT_MODE",		BOOL_TYPE_VAR,	DEFAULT_INSERT_MODE,
		NULL, update_all_status, 0, 0,
	"INVERSE_VIDEO",	BOOL_TYPE_VAR,	DEFAULT_INVERSE_VIDEO,
		NULL, NULL, 0, 0,
	"LASTLOG",		INT_TYPE_VAR,	DEFAULT_LASTLOG,
		NULL, set_lastlog_size, 0, 0,
	"LASTLOG_LEVEL",	STRING_TYPE_VAR,0,
		NULL, set_lastlog_level, 0, 0,
	"LOAD_PATH",		STRING_TYPE_VAR,0,
		NULL, NULL, 0, 0,
	"LOG",			BOOL_TYPE_VAR,	DEFAULT_LOG,
		NULL, logger, 0, 0,
	"LOGFILE",		STRING_TYPE_VAR,0,
		NULL, NULL, 0, VF_NODAEMON,
	"MAIL",			INT_TYPE_VAR,	DEFAULT_MAIL,
		NULL, update_all_status, 0, VF_NODAEMON,
	"MAX_RECURSIONS",	INT_TYPE_VAR,	DEFAULT_MAX_RECURSIONS,
		NULL, NULL, 0, 0,
	"MENU",			STRING_TYPE_VAR,0,
		NULL, set_menu, 0, 0,
	"MINIMUM_SERVERS",	INT_TYPE_VAR,	DEFAULT_MINIMUM_SERVERS,
		NULL, NULL, 0, VF_NODAEMON,
	"MINIMUM_USERS",	INT_TYPE_VAR,	DEFAULT_MINIMUM_USERS,
		NULL, NULL, 0, VF_NODAEMON,
	"NO_CTCP_FLOOD",	BOOL_TYPE_VAR,	DEFAULT_NO_CTCP_FLOOD,
		NULL, NULL, 0, 0,
	"NOTIFY_ON_TERMINATION",BOOL_TYPE_VAR,	DEFAULT_NOTIFY_ON_TERMINATION,
		NULL, NULL, 0, VF_NODAEMON,
	"NOVICE",		BOOL_TYPE_VAR,	1,
		NULL, NULL, 0, 0,
	"SCROLL",		BOOL_TYPE_VAR,	DEFAULT_SCROLL,
		NULL, set_scroll, 0, 0,
	"SCROLL_LINES",		INT_TYPE_VAR,	DEFAULT_SCROLL_LINES,
		NULL, set_scroll_lines, 0, 0,
	"SEND_IGNORE_MSG",	BOOL_TYPE_VAR,	DEFAULT_SEND_IGNORE_MSG,
		NULL, NULL, 0, 0,
	"SHELL",		STRING_TYPE_VAR,0,
		NULL, NULL, 0, VF_NODAEMON,
	"SHELL_FLAGS",		STRING_TYPE_VAR,0,
		NULL, NULL, 0, VF_NODAEMON,
	"SHELL_LIMIT",		INT_TYPE_VAR,	DEFAULT_SHELL_LIMIT,
		NULL, NULL, 0, VF_NODAEMON,
	"SHOW_AWAY_ONCE",	BOOL_TYPE_VAR,	DEFAULT_SHOW_AWAY_ONCE,
		NULL, NULL, 0, 0,
	"SHOW_CHANNEL_NAMES",	BOOL_TYPE_VAR,	DEFAULT_SHOW_CHANNEL_NAMES,
		NULL, NULL, 0, 0,
	"SHOW_END_OF_MSGS",	BOOL_TYPE_VAR,	DEFAULT_SHOW_END_OF_MSGS,
		NULL, NULL, 0, 0,
	"SHOW_NUMERICS",	BOOL_TYPE_VAR,	DEFAULT_SHOW_NUMERICS,
		NULL, NULL, 0, 0,
	"SHOW_STATUS_ALL",	BOOL_TYPE_VAR,	DEFAULT_SHOW_STATUS_ALL,
		NULL, update_all_status, 0, 0,
	"SHOW_WHO_HOPCOUNT", 	BOOL_TYPE_VAR,	DEFAULT_SHOW_WHO_HOPCOUNT,
		NULL, NULL, 0, 0,
	"STATUS_AWAY",		STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_CHANNEL",	STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_CHANOP",	STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_CLOCK",		STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_FORMAT",	STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_HOLD",		STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_HOLD_LINES",	STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_INSERT",	STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_MAIL",		STRING_TYPE_VAR,0,
		NULL, build_status, 0, VF_NODAEMON,
	"STATUS_MODE",		STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_NOTIFY",	STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_OPER",		STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_OVERWRITE",	STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_QUERY",		STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_SERVER",	STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_UMODE",		STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_USER",		STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_USER1",		STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_USER2",		STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_USER3",		STRING_TYPE_VAR,0, NULL, build_status, 0, 0,
	"STATUS_WINDOW",	STRING_TYPE_VAR,0,
		NULL, build_status, 0, 0,
	"SUPPRESS_SERVER_MOTD",	BOOL_TYPE_VAR,	DEFAULT_SUPPRESS_SERVER_MOTD,
		NULL, NULL, 0, 0,
	"TAB",			BOOL_TYPE_VAR,	DEFAULT_TAB,
		NULL, NULL, 0, 0,
	"TAB_MAX",		INT_TYPE_VAR,	DEFAULT_TAB_MAX,
		NULL, NULL, 0, 0,
	"TERM_COMPRESS",	BOOL_TYPE_VAR,	DEFAULT_TERM_COMPRESS,
		NULL, NULL, 0, 0,
	"TERM_PRIORITY",	INT_TYPE_VAR,	DEFAULT_TERM_PRIORITY,
		NULL, NULL, 0, 0,
	"TRANSLATION",		STRING_TYPE_VAR,0,
		NULL, set_translation, 0, 0,
	"UNDERLINE_VIDEO",	BOOL_TYPE_VAR,	DEFAULT_UNDERLINE_VIDEO,
		NULL, NULL, 0, 0,
	"USE_OLD_MSG",		BOOL_TYPE_VAR,	DEFAULT_USE_OLD_MSG,
		NULL, NULL, 0, 0,
	"USER_INFORMATION", 	STRING_TYPE_VAR,0,
		NULL, NULL, 0, 0,
	"USER_WALLOPS",		BOOL_TYPE_VAR,	DEFAULT_USER_WALLOPS,
		NULL, NULL, 0, 0,
	"VERBOSE_CTCP",		BOOL_TYPE_VAR,	DEFAULT_VERBOSE_CTCP,
		NULL, NULL, 0, 0,
	"WARN_OF_IGNORES",	BOOL_TYPE_VAR,	DEFAULT_WARN_OF_IGNORES,
		NULL, NULL, 0, 0,
	null(char *), 0, 0, NULL, NULL, 0, 0};

/*
 * init_variables: initializes the string variables that can't really be
 * initialized properly above 
 */
void	init_variables()
{
	set_string_var(CMDCHARS_VAR, DEFAULT_CMDCHARS);
	set_string_var(LOGFILE_VAR, DEFAULT_LOGFILE);
	set_string_var(SHELL_VAR, DEFAULT_SHELL);
	set_string_var(SHELL_FLAGS_VAR, DEFAULT_SHELL_FLAGS);
	set_string_var(ENCRYPT_PROGRAM_VAR, DEFAULT_ENCRYPT_PROGRAM);
	set_string_var(CONTINUED_LINE_VAR, DEFAULT_CONTINUED_LINE);
	set_string_var(INPUT_PROMPT_VAR, DEFAULT_INPUT_PROMPT);
	set_string_var(HIGHLIGHT_CHAR_VAR, DEFAULT_HIGHLIGHT_CHAR);
	set_string_var(HISTORY_FILE_VAR, DEFAULT_HISTORY_FILE);
	set_string_var(LASTLOG_LEVEL_VAR, DEFAULT_LASTLOG_LEVEL);
	set_string_var(STATUS_FORMAT_VAR, DEFAULT_STATUS_FORMAT);
	set_string_var(STATUS_AWAY_VAR, DEFAULT_STATUS_AWAY);
	set_string_var(STATUS_CHANNEL_VAR, DEFAULT_STATUS_CHANNEL);
	set_string_var(STATUS_CHANOP_VAR, DEFAULT_STATUS_CHANOP);
	set_string_var(STATUS_CLOCK_VAR, DEFAULT_STATUS_CLOCK);
	set_string_var(STATUS_HOLD_VAR, DEFAULT_STATUS_HOLD);
	set_string_var(STATUS_HOLD_LINES_VAR, DEFAULT_STATUS_HOLD_LINES);
	set_string_var(STATUS_INSERT_VAR, DEFAULT_STATUS_INSERT);
	set_string_var(STATUS_MAIL_VAR, DEFAULT_STATUS_MAIL);
	set_string_var(STATUS_MODE_VAR, DEFAULT_STATUS_MODE);
	set_string_var(STATUS_OPER_VAR, DEFAULT_STATUS_OPER);
	set_string_var(STATUS_OVERWRITE_VAR, DEFAULT_STATUS_OVERWRITE);
	set_string_var(STATUS_QUERY_VAR, DEFAULT_STATUS_QUERY);
	set_string_var(STATUS_SERVER_VAR, DEFAULT_STATUS_SERVER);
	set_string_var(STATUS_UMODE_VAR, DEFAULT_STATUS_UMODE);
	set_string_var(STATUS_USER_VAR, DEFAULT_STATUS_USER);
	set_string_var(STATUS_USER1_VAR, DEFAULT_STATUS_USER1);
	set_string_var(STATUS_USER2_VAR, DEFAULT_STATUS_USER2);
	set_string_var(STATUS_USER3_VAR, DEFAULT_STATUS_USER3);
	set_string_var(STATUS_WINDOW_VAR, DEFAULT_STATUS_WINDOW);
	set_string_var(USERINFO_VAR, DEFAULT_USERINFO);
	set_alarm(DEFAULT_CLOCK_ALARM);
	set_beep_on_msg(DEFAULT_BEEP_ON_MSG);
	set_string_var(STATUS_NOTIFY_VAR, DEFAULT_STATUS_NOTIFY);
	set_string_var(CLIENTINFO_VAR, IRCII_COMMENT);
	set_string_var(TRANSLATION_VAR, "ASCII");
	set_translation("ASCII");
	set_string_var(HELP_PATH_VAR, DEFAULT_HELP_PATH);
	set_string_var(HELP_SERVICE_VAR, DEFAULT_HELP_SERVICE);
	set_lastlog_size(irc_variable[LASTLOG_VAR].integer);
	set_history_size(irc_variable[HISTORY_VAR].integer);
	set_history_file(irc_variable[HISTORY_FILE_VAR].string);
	set_highlight_char(irc_variable[HIGHLIGHT_CHAR_VAR].string);
	set_lastlog_level(irc_variable[LASTLOG_LEVEL_VAR].string);
}

/*
 * find_variable: looks up variable name in the variable table and returns
 * the index into the variable array of the match.  If there is no match, cnt
 * is set to 0 and -1 is returned.  If more than one match the string, cnt is
 * set to that number, and it returns the first match.  Index will contain
 * the index into the array of the first found entry 
 */
int	find_variable(org_name, cnt)
char	*org_name;
int	*cnt;
{
	IrcVariable *v,
		    *first;
	int	len,
	index;
	char	*name = null(char *);

	malloc_strcpy(&name,org_name);
	upper(name);
	len = strlen(name);
	index = 0;
	for (first = irc_variable; first->name; first++, index++)

	{
		if (strncmp(name, first->name, len) == 0)
		{
			*cnt = 1;
			break;
		}
	}
	if (first->name)

	{
		if (strlen(first->name) != len)
		{
			v = first;
			for (v++; v->name; v++, (*cnt)++)

			{
				if (strncmp(name, v->name, len) != 0)
					break;
			}
		}
		new_free(&name);
		return (index);
	}
	else

	{
		*cnt = 0;
		new_free(&name);
		return (-1);
	}
}

/*
 * do_boolean: just a handy thing.  Returns 1 if the str is not ON, OFF, or
 * TOGGLE 
 */
int	do_boolean(str, value)
char	*str;
int	*value;
{
	upper(str);
	if (strcmp(str, var_settings[ON]) == 0)
		*value = 1;
	else if (strcmp(str, var_settings[OFF]) == 0)
		*value = 0;
	else if (strcmp(str, "TOGGLE") == 0)
	{
		if (*value)
			*value = 0;
		else
			*value = 1;
	}
	else
		return (1);
	return (0);
}

/*
 * set_var_value: Given the variable structure and the string representation
 * of the value, this sets the value in the most verbose and error checking
 * of manors.  It displays the results of the set and executes the function
 * defined in the var structure 
 */
void	set_var_value(index, value)
int	index;
char	*value;
{
	char	*rest;
	IrcVariable *var;
	int	old;


	var = &(irc_variable[index]);
#ifdef DAEMON_UID
	if (getuid() == DAEMON_UID && var->flags&VF_NODAEMON && value && *value)


	{
		say("You are not permitted to set that variable");
		return;
	}
#endif
	switch (var->type)
	{
	case BOOL_TYPE_VAR:
		if (value && *value && (value = next_arg(value, &rest)))
		{
			old = var->integer;
			if (do_boolean(value, &(var->integer)))

			{
				say("Value must be either ON, OFF, or TOGGLE");
				break;
			}
			if (!var->changed)
			{
				if (old != var->integer)
					var->changed = 1;
			}
			if (var->func)
				(var->func) (var->integer);
			say("Value of %s set to %s", var->name, (var->integer) ?				var_settings[ON] : var_settings[OFF]);
		}
		else
			say("Current value of %s is %s", var->name,
				(var->integer) ?
				var_settings[ON] : var_settings[OFF]);
		break;
	case CHAR_TYPE_VAR:
		if (value && *value && (value = next_arg(value, &rest)))
		{
			if (strlen(value) > 1)
				say("Value of %s must be a single character",
					var->name);
			else
			{
				if (!var->changed)
				{
					if (var->integer != *value)
						var->changed = 1;
				}
				var->integer = *value;
				if (var->func)
					(var->func) (var->integer);
				say("Value of %s set to '%c'", var->name,
					var->integer);
			}
		}
		else
			say("Current value of %s is '%c'", var->name,
				var->integer);
		break;
	case INT_TYPE_VAR:
		if (value && *value && (value = next_arg(value, &rest)))
		{
			int	val;

			if (!is_number(value))
			{
				say("Value of %s must be numeric!", var->name);
				break;
			}
			if ((val = atoi(value)) < 0)
			{
				say("Value of %s must be greater than 0",
					var->name);
				break;
			}
			if (!var->changed)
			{
				if (var->integer != val)
					var->changed = 1;
			}
			var->integer = val;
			if (var->func)
				(var->func) (var->integer);
			say("Value of %s set to %d", var->name, var->integer);
		}
		else
			say("Current value of %s is %d", var->name,
				var->integer);
		break;
	case STRING_TYPE_VAR:
		if (value)
		{
			if (*value)
			{
				if (!var->changed)
				{
					if ((var->string && ! value) ||
					    (! var->string && value) ||
					    strcmp(var->string, value))
						var->changed = 1;
				}
				malloc_strcpy(&(var->string), value);
			}
			else
			{
				if (var->string)
					say("Current value of %s is %s",
						var->name, var->string);
				else
					say("No value for %s has been set",
						var->name);
				return;
			}
		}
		else
			new_free(&(var->string));
		if (var->func)
			(var->func) (var->string);
		say("Value of %s set to %s", var->name, var->string ?
			var->string : "<EMPTY>");
		break;
	}
}

/*
 * set_variable: The SET command sets one of the irc variables.  The args
 * should consist of "variable-name setting", where variable name can be
 * partial, but non-ambbiguous, and setting depends on the variable being set 
 */
/*ARGSUSED*/
void	set_variable(command, args)
char	*command,
	*args;
{
	char	*var;
	int	no_args = 1,
		cnt,
		index;

	if (var = next_arg(args, &args))
	{
		if (*var == '-')
		{
			var++;
			args = null(char *);
		}
		index = find_variable(var, &cnt);
		switch (cnt)
		{
		case 0:
			say("No such variable \"%s\"", var);
			return;
		case 1:
			set_var_value(index, args);
			return;
		default:
			say("%s is ambiguous", var);
			for (cnt += index; index < cnt; index++)
				set_var_value(index, empty_string);
			return;
		}
	/* not supposed to get here anyway
	 * no_args = 0;
	 */
	}
	if (no_args)
	{
		for (index = 0; index < NUMBER_OF_VARIABLES; index++)
			set_var_value(index, empty_string);
	}
}

/*
 * get_string_var: returns the value of the string variable given as an index
 * into the variable table.  Does no checking of variable types, etc 
 */
char	*get_string_var(var)
int	var;
{
	return (irc_variable[var].string);
}

/*
 * get_int_var: returns the value of the integer string given as an index
 * into the variable table.  Does no checking of variable types, etc 
 */
int	get_int_var(var)
int	var;
{
	return (irc_variable[var].integer);
}

/*
 * set_string_var: sets the string variable given as an index into the
 * variable table to the given string.  If string is null, the current value
 * of the string variable is freed and set to null 
 */
void	set_string_var(var, string)
int	var;
char	*string;
{
	if (string)
		malloc_strcpy(&(irc_variable[var].string), string);
	else
		new_free(&(irc_variable[var].string));
}

/*
 * set_int_var: sets the integer value of the variable given as an index into
 * the variable table to the given value 
 */
void	set_int_var(var, value)
int	var;
unsigned int	value;
{
	if (var == NOVICE_VAR && !load_depth && !value)
	{
say("WARNING: Setting NOVICE to OFF enables commands in your client which");
say("         could be used by others on IRC to control your IRC session");
say("         or compromise security on your machine. If somebody has");
say("         asked you to do this, and you do not know EXACTLY why, or if");
say("         you are not ABSOLUTELY sure what you are doing, you should");
say("         immediately /SET NOTIFY ON and ask the IRC operators about");
say("         the commands you have been asked to enter on channel");
say("         #Twilight_Zone.");
	}
	irc_variable[var].integer = value;
}

/*
 * save_variables: this writes all of the IRCII variables to the given FILE
 * pointer in such a way that they can be loaded in using LOAD or the -l switch 
 */
void	save_variables(fp)
FILE *fp;
{
	IrcVariable *var;

	for (var = irc_variable; var->name; var++)
	{
		if (var->changed)
		{
			if (strcmp(var->name, "DISPLAY") == 0)
				continue;
			if (strcmp(var->name, "CMDCHARS") == 0)
				continue;
			fprintf(fp, "SET ");
			switch (var->type)
			{
			case BOOL_TYPE_VAR:
				fprintf(fp, "%s %s\n", var->name, var->integer ?
					var_settings[ON] : var_settings[OFF]);
				break;
			case CHAR_TYPE_VAR:
				fprintf(fp, "%s %c\n", var->name, var->integer);
				break;
			case INT_TYPE_VAR:
				fprintf(fp, "%s %u\n", var->name, var->integer);
				break;
			case STRING_TYPE_VAR:
				if (var->string)
					fprintf(fp, "%s %s\n", var->name,
						var->string);
				else
					fprintf(fp, "-%s\n", var->name);
				break;
			}
		}
	}
	var = &(irc_variable[CMDCHARS_VAR]);
	fprintf(fp, "SET %s %s\n", var->name, var->string);
}

char	*make_string_var(var_name)
char	*var_name;
{
	int	cnt,
		index;
	char	buffer[BIG_BUFFER_SIZE + 1],
	*ret = null(char *);

	if (((index = find_variable(var_name, &cnt)) == -1) ||
	    (cnt > 1) ||
	    stricmp(var_name,irc_variable[index].name))
		return (null(char *));
	switch (irc_variable[index].type)
	{
	case STRING_TYPE_VAR:
		malloc_strcpy(&ret, irc_variable[index].string);
		break;
	case INT_TYPE_VAR:
		sprintf(buffer, "%u", irc_variable[index].integer);
		malloc_strcpy(&ret, buffer);
		break;
	case BOOL_TYPE_VAR:
		malloc_strcpy(&ret, var_settings[irc_variable[index].integer]);
		break;
	case CHAR_TYPE_VAR:
		sprintf(buffer, "%c", irc_variable[index].integer);
		malloc_strcpy(&ret, buffer);
		break;
	}
	return (ret);

}

/* exec_warning: a warning message displayed whenever EXEC_PROTECTION is turned off.  */
static	void	exec_warning(value)
int	value;
{
	if (value == OFF)
	{
		say("Warning!  You have turned EXEC_PROTECTION off");
		say("Please read the /HELP SET EXEC_PROTECTION documentation");
	}
}

static	void	input_warning(value)
int	value;
{
	if (value == OFF)
	{
		say("Warning!  You have turned INPUT_PROTECTION off");
		say("Please read the /HELP ON INPUT, and /HELP SET INPUT_PROTECTION documentation");
	}
}
