/*
 *   Rat in Control - an interactive cognitive experiment
 *.  by M. J. Roberts 
 */

#include <adv3.h>
#include <en_us.h>

/* ------------------------------------------------------------------------ */
/*
 *   GameInfo metadata
 */
PreinitObject
    execute()
    {
        local tab = new LookupTable();
        tab['Name'] = versionInfo.name;
        tab['Version'] = versionInfo.version;
        tab['Byline'] = versionInfo.byline;
        tab['HtmlByline'] = versionInfo.htmlByline;
        tab['AuthorEmail'] = 'Mike Roberts <mjr_@hotmail.com>';
        tab['Desc'] = 'An interactive cognitive experiment, comparing
           the usability of compass directions with the usability
           of relative directions.';
        tab['PresentationProfile'] = 'Multimedia';

        /* write the game information to GameInfo.txt */
        writeGameInfo(tab, 'GameInfo.txt');
    }
;

/*
 *   Game information 
 */
versionInfo: GameID
    name = 'Rat In Control'
    byline = 'by Michael J.&nbsp;Roberts'
    htmlByline = 'by <a href="mailto:mjr_@hotmail.com">'
                 + 'Michael J.&nbsp;Roberts</a>'
    version = '1.0'

    showCredit()
    {
        "Rat In Control
        <br>An Interactive Cognitive Experiment
        <br>by Michael J.\ Roberts
        <br>
        <br>Designed with TADS 3
        <br>";
    }

    showAbout()
    {
        aboutMenu.display();
        "Done. ";
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   The main startup routine.  The interpreter calls this on entry to kick
 *   off the game.  
 */
main(args)
{
    /* randomly choose a navigation mode */
    randomize();
    me.absDirMode = (rand(100) > 50);

    /* set the interpreter window title */
    "<title>Rat In Control</title>";

    /* advise the player of our initial mode, and how to change it */
    "\ \b\b
    [The game has randomly selected <<me.absDirMode ? 'COMPASS' :
    'RELATIVE' >> direction mode for this session.  You can change
    the mode at any time by typing COMPASS or RELATIVE at the
    command prompt.  However, if this is your first time playing,
    we would like to ask that you play through the whole game once
    in the mode that has just been selected, so that we get a good
    range of experimental results for each initial mode.  Please type
    ABOUT at the command prompt for details.]
    \b\b\b
    Press a key to continue... ";
    inputManager.getKey(nil, nil);
    clearScreen();

    /* show the introduction */
    "<q>They sure have beady little eyes, don't they?</q> says
    that creepy guy who's been hanging around with Jen lately.
    <.p>
    Jen takes off her lab coat and hangs it on a hook
    next to the door. <q>Oh, come on, rats are cute,</q> she
    says as the two make their way out the door.
    <.p>
    <q>I don't know.  The way they're always staring at you...</q>
    The door closes behind them and their voices trail away.  The
    lab is silent, finally empty for the night of the scientists and
    lab assistants who are forever making you run through mazes,
    like you have nothing better to do all day than forage for
    cheese.  But no more; your fellow experimental subject Wilbert
    got ahold of a cage key, and tonight's the night: the rats are
    taking over.
    <.p>
    <b>Rat In Control</b>
    <br>An Interactive Cognitive Experiment
    <br>by Michael J.\ Roberts
    <br>First-time subjects, type <<aHref('about', 'ABOUT')>>
    for information about this game.
    <.p>
    Wilbert appears in front of your cage, gripping the ridiculously
    oversized key with one foot.  <q>Your turn, Fred,</q> he says as
    he laboriously hefts the key into the lock and fiddles with it.
    <q>We've got a lot to do, my friend, so let's be as quick as
    we can.</q>  You hear the familiar snap of the lock releasing,
    and the cage door swings open.  <q>There.  Now, your job is
    to find out where the heating vent is.  I'll get the others out,
    so just go look for the heating vent.</q>  He starts moving off,
    then turns back and gives you a stern look.  <q>And don't waste
    your time carrying everything back to your cage!  I know what a
    pack rat you can be, but you might have noticed that we can only
    hold one thing at a time in these very tiny hands we've got, and
    we don't exactly have all night here!</q>  He heads off to help
    the others out of their cages.
    <.p>";

    /* tell the library to run the game */
    gPlayerChar = me;
    runGame(me);
}

/* ------------------------------------------------------------------------ */
/*
 *   The player character actor. 
 */
me: Actor
    isPlayerChar = true
    location = cage
    desc = "You are a black lab rat.  You have a shiny black coat,
            hairless tail, a pointy nose, and prehensile little feet
            that are rather good at gripping things. "

    /* tiny hands: we can only hold one thing at a time */
    bulkCapacity = 1

    /*
     *   My mission status.
     *   
     *   0 -> initial exploration phase; Wilbert has told us to go find the
     *   heating vent, so we're just poking around looking for it.  No
     *   timing pressure in this part.
     *   
     *   1 -> first timed mission: return from heating vent to cage
     *   
     *   2 -> second timed mission: go to toolbox from cage
     *   
     *   3 -> third timed mission: go from toolbox to vent 
     */
    missionStatus = 0

    /* start time for the current mission */
    missionStartTime = nil

    /* start out pointing north */
    pointing = northDirection
;

/* a component of myself takes up no space in my hands */
class MeComponent: Component bulk = 0;

/* some rat parts, for completeness */
+ MeComponent 'shiny black hair/coat' 'coat' "It's a shiny black. ";
+ MeComponent 'hairless naked tail' 'tail' "It's hairless, of course. ";
+ MeComponent 'pointy nose' 'nose' "It's pointy. ";
+ MeComponent 'prehensile left right front back foot/feet' 'feet'
    "They're good at gripping things, but only rather small things. "
    isPlural = true
;

/* ------------------------------------------------------------------------ */
/*
 *   Locations 
 */

/*
 *   the player character's cage
 */

cage: Room 'Your Cage' 'your cage'
    "This is the inside of your cage.  The door is {to-the-north}. "

    north = cageShelf
    out asExit(north)
;

+ ExitPortal ->cageShelf 'cage door/cage' 'cage door'
    "It's gloriously open, and with no lab assistant in sight trying
    to grab you. "
    dobjFor(Close)
    {
        verify() { }
        action() { "No way you're going to risk getting locked in again. "; }
    }
    dobjFor(Open) { verify() { illogical(&alreadyOpen); } }
;

shelfFloor: Floor 'shelf' 'shelf'
    "It extends off {to-the-east} and {to-the-west}. "
    putDestMessage = &putDestSurface
;

/*
 *   Shelf outside cage 
 */
cageShelf: Room 'Shelf, Outside Your Cage' 'the shelf outside your cage'
    "This is a narrow shelf in front of a row of cages, which extends
    off {to-the-east} and {to-the-west}.  Your cage is {to-the-south}.
    {To-the-north}, someone has left a test instrument with
    its top just about aligned at shelf level. "

    south = cage
    north = topOfInstrument

    east: FakeConnector
          { "Wilbert has the cages covered; no need to help. "; }
    west = (east)
    roomParts = [shelfFloor]

    travelerArriving(traveler, origin, connector, backConnector)
    {
        /* do our inherited work first */
        inherited(traveler, origin, connector, backConnector);

        /* if we're in mission 1, finish it and dispatch on mission 2 */
        if (me.missionStatus == 1)
        {
            local delta;
            local score;
            
            /* note the ending time */
            delta = getTime(GetTimeTicks) - me.missionStartTime;

            /* score it */
            score = (delta > 20000 ? 10 : 30 - delta/1000);
            addToScore(score, 'getting from the vent to Wilbert in '
                       + delta + ' milliseconds');

            "<.p>Wilbert sees you arrive.  <q>Have you found the heating
            vent?</q>  You're still catching your breath from your
            sprint here, so you just nod.  <q>Well, it's about time!
            Now, I'm assuming we'll need some kind of tool to get
            into the vent, but I'm not sure which one.  Mr.\ Tails has
            managed - you know, I really hate these stupid names they
            give us! - anyway.  Mr.\ Tails has managed to open one
            of their toolboxes.  I hope you happened to come across
            it, because I want to you to get to the toolbox as fast
            as you can, and pick out the tool we need.  I'll
            meet you at the vent!</q>  He scampers off in a big rush.
            <.p>
            [When you're ready, press a key to begin timing for
            this mission.]
            \n";

            inputManager.getKey(nil, nil);

            /* note the new start time and mission status */
            me.missionStartTime = getTime(GetTimeTicks);
            me.missionStatus = 2;
        }
    }
;

+ EntryPortal ->cage 'cage cage/door' 'cage' "It's your cage.  It's open. "
    owner = me
    dobjFor(Close)
    {
        verify() { }
        action() { "No way you're going to risk getting locked in again. "; }
    }
    dobjFor(Open) { verify() { illogical(&alreadyOpen); } }
;

+ Fixture 'test lab laboratory instrument/equipment' 'test instrument'
    "You can't see much from here.  Besides, you're a lab rat; test
    instruments aren't exactly your favorite things. "
    dobjFor(StandOn) remapTo(TravelVia, topOfInstrument)
    dobjFor(Enter) remapTo(TravelVia, topOfInstrument)
    dobjFor(Board) remapTo(TravelVia, topOfInstrument)
;

/*
 *   Top of the test instrument 
 */

instrumentFloor: Floor 'test lab laboratory rough blue warm featureless
    top/instrument/equipment/piece/meta/surface' 'top of the instrument'
    "The rough blue metal surface of the top is featureless, and that's
    about all you can see, apart from an electrical cord attached to the
    side and running down to the floor. "

    dobjFor(Feel) { action() { "It's pleasantly warm. "; } }
;

topOfInstrument: Room 'On the Test Instrument' 'the test instrument'
    "The top of this piece of equipment is a featureless
    rough metal painted blue, but it radiates a pleasant residual warmth.
    The instrument is pushed up against a shelf {to-the-south}.
    There's an electrical cord connected to the {east-side} of the unit,
    running down to the floor; you could easily climb down. "

    south = cageShelf
    down = cordTop

    roomParts = [instrumentFloor]
;

+ Fixture  'shelf' 'shelf'
    "The test equipment is pushed up against the shelf. "
    dobjFor(StandOn) remapTo(TravelVia, cageShelf)
    dobjFor(Enter) remapTo(TravelVia, cageShelf)
    dobjFor(Board) remapTo(TravelVia, cageShelf)
;

+ cordTop: StairwayDown 'electrical power cord/wire/cable' 'electrical cord'
    "It runs down to the floor.  You could easily climb down. "

    /* 
     *   the cord attaches to the east side, so we have to turn east to
     *   climb down 
     */
    pointingChange = eastDirection
;

+ Distant 'lab laboratory floor' 'lab floor'
    "You can't make out a lot from here, but you could probably climb
    down the electrical cord if you want a closer look. "
;

/*
 *   Floor at foot of instrument 
 */
footOfCart: Room 'Foot of Equipment Cart'
    'the foot of the equipment cart'
    "A wheeled cart carrying a bunch of test equipment towers above the
    floor here.  The cart blocks the way {west}, but you could
    easily climb the electrical cord hanging down the side from one
    of the pieces of equipment above.  The floor is clear {to-the-south},
    {to-the-north}, and {to-the-east}. "

    up = cordBottom
    north = atWall1
    south = nextToCabinet
    east = frontOfCabinet
    roomParts = [defaultFloor]
;

+ Distant 'test lab piece/pieces/laboratory equipment/instrument'
    'test equipment'
    "The equipment is all way up at the top of the cart; you can't
    see much from here. "
;

+ Heavy 'wheeled cart/wheel/wheels' 'cart' "It towers above. "
    dobjFor(Climb)
    {
        /* allow it, but make it less likely than the cord */
        verify() { logicalRank(70, 'should use cord'); }
        action()
        {
            reportBefore('You can\'t climb the cart itself, but
                fortunately you can easily climb the electrical cord. ');
            replaceAction(Climb, cordBottom);
        }
    }
    dobjFor(ClimbUp) asDobjFor(Climb)
;

+ cordBottom: StairwayUp ->cordTop 'electrical power cord/wire/cable'
    'electrical cord'
    "It hangs down from one of the pieces of equipment on the cart.
    You could easily climb it. "

    /* 
     *   the cord attaches to the east side of the equipment, so we have to
     *   turn west to climb it 
     */
    pointingChange = westDirection
;

/*
 *   Corner, next to the cabinet 
 */
nextToCabinet: Room 'Corner' 'the corner'
    "A tall cabinet stands {to-the-east}, its back to the wall
    that's {to-the-south}.  You can scurry {north} and {west}. "

    west = behindCart
    north = footOfCart

    roomParts = [defaultFloor, defaultSouthWall]
;

+ Fixture 'gray metal tall cabinet/side' 'cabinet'
    "All you can see from here is the gray metal side of the cabinet. "
    dobjFor(Open)
        { verify() { illogical('The door isn\'t on this side. '); } }
    dobjFor(Close) asDobjFor(Open)
;

/*
 *   In front of the cabinet 
 */
frontOfCabinet: Room 'Front of Cabinet' 'the front of the cabinet'
    "A tall gray metal cabinet (which is closed) stands {to-the-south}.
    A wall blocks further travel {to-the-east}, but you can go
    {north} and {west}. "

    west = footOfCart
    north = boxesRoom

    roomParts = [defaultFloor, defaultEastWall]
;

+ Fixture 'gray metal tall cabinet/door/doors' 'cabinet'
    "It towers above you, like almost everything else around here.
    The doors are closed. "
    dobjFor(Open)
    {
        verify() { }
        action()
        {
            reportFailure('Alas, the handle is far, far above, and is
                anyway designed for human hands. ');
        }
    }
;

/*
 *   Behind the cart 
 */
behindCart: Room 'Next to Cart' 'the spot next to the cart'
    "A wheeled cart is {to-the-north}, and a wall is {to-the-south}.
    You can travel {east} and {west}. "

    east = nextToCabinet
    west = ratChowRoom

    roomParts = [defaultFloor, defaultSouthWall]
;

+ Heavy 'wheeled cart/wheel/wheels' 'cart' "It towers above. ";

/*
 *   A class for the maze walls 
 */
class MazeWall: Fixture 'short wooden wall' 'wooden wall'
    "It's short by the standards of the humans who run the lab,
    but too tall for you to climb. "
;

/*
 *   First wall location 
 */
atWall1: Room 'Near Wall' 'the spot near the wall'
    "A short wall (short by human standards, anyway)
    blocks the way {to-the-north}, but you can still
    go {east}, {west}, and {south}. "

    east = boxesRoom
    south = footOfCart
    west = stackOfBooks

    roomParts = [defaultFloor]
;
+ MazeWall;

/*
 *   Boxes 
 */
boxesRoom: Room 'Next to Boxes' 'the spot near the boxes'
    "A tall stack of boxes {to-the-north} prevents you from
    going that way, and a wall {to-the-east} is also in the way.
    You can only go {south} and {west}. "

    west = atWall1
    south = frontOfCabinet

    roomParts = [defaultFloor, defaultEastWall]
;

+ Heavy 'tall big cardboard stack/box/boxes' 'stack of boxes'
    "It's just a stack of big cardboard boxes. "
;

+ Thing 'small plastic remote control unit' 'remote control'
    "It's a small plastic remote, with buttons labeled Forward, Backward,
    Left, and Right. "

    initDesc = "There's a small remote control unit leaning up against
                one of the boxes. "
    actionKnown = nil
;
class RemoteButton: Button, Component
    dobjFor(Push)
    {
        action()
        {
            if (gActor.canSee(radioCar))
            {
                location.actionKnown = true;
                moveCar();
            }
            else
            {
                "Nothing obvious happens.  You probably have to be
                right next to <<location.actionKnown ? 'the radio car'
                : 'whatever this controls'>> for it to work. ";
            }
        }
    }
    moveCar() { "The car makes motor noises, but seems unable to
        move in this direction. "; }
;
++ RemoteButton 'forward button' '<q>Forward</q> button';
++ RemoteButton 'left button' '<q>Left</q> button';
++ RemoteButton 'right button' '<q>Right</q> button';
++ RemoteButton 'backward button' '<q>Backward</q> button'
    moveCar()
    {
        "The car's motor whirs, and the car zips away in reverse
        until it's out of sight. ";
        radioCar.moveInto(nil);

        addToScore(10, 'getting past the toy car');
    }
;

/*
 *   Rat chow room 
 */
ratChowRoom: Room 'Rat Chow' 'the spot near the bag of Rat Chow'
    "A big bag of Plurama Rat Chow sits on the floor here, slumped
    against the wall {to-the-south}, blocking further travel
    {to-the-west}.  You can go {north} and {east}. "

    east = behindCart
    north = deskRoom

    roomParts = [defaultFloor, defaultSouthWall]
;

+ Heavy 'big plurama rat chow/bag' 'bag of rat chow'
    "It's a big bag of Rat Chow.  You never seem to get tired of it.
    Unfortunately, it's closed, and it's too big for you to get into. "

    dobjFor(Open)
        { verify() { illogical('There\'s no obvious way to open it. '); } }
    dobjFor(Eat)
    {
        preCond = [touchObj]
        verify() { }
        action() { reportFailure('Much as you\'d like to, there\'s no
            obvious way into the bag.  And besides, you have other things
            you have to deal with right now. '); }
    }
;

/*
 *   Stack of books 
 */
stackOfBooks: Room 'Stack of Books' 'the spot near the stack of books'
    "A big stack of textbooks is piled against the short wooden
    wall (which is {to-the-north}).  A wheeled cart blocks the
    way {to-the-south}, but you can continue along the wall
    {east} and {west}. "

    up = bookStairsBottom
    east = atWall1
    west = toolboxRoom

    roomParts = [defaultFloor]
;
+ Heavy 'wheeled cart/wheel/wheels' 'cart' "It towers above. ";
+ MazeWall;
+ bookStairsBottom: Heavy, Readable, StairwayUp
    'big text stack/book/books/textbooks' 'stack of textbooks'
    "<i>Elementary Rodent Behavior,</i> <i>So You Want to Train
     a Rat,</i> <i>Who Ate My Cheese?,</i> <i>Rat Behavior for
     Drooling Morons,</i> many more.  They all strike you as terribly
     sinister.  You notice that the stack is just about as high as
     the wooden wall, and by some accident has been arranged into
     a stair-step pattern. "

    readDesc = "Like you're going to learn something you don't already
                know. "

    pointingChange = northDirection
;

/*
 *   top of books 
 */
topOfBooks: Room 'Top of Books' 'the top of the books'
    "This is the top of a pile of books, which is right at the
    same height as the short wooden wall {to-the-north}, so you can
    see over the wall.  Now you realize what the wall is: it's
    the wall of the hated maze, where you've spent far too many
    frustrating hours.  Who ever thought mazes were a good idea,
    anyway?  From here, you have a human's-eye view of the entire
    layout. "

    down = bookStairsTop
    north: NoTravelMessage { "Are you kidding?  There's no way you'd
        go in there willingly. " }

    roomParts = [defaultFloor]

    travelerArriving(traveler, origin, connector, backConnector)
    {
        inherited(traveler, origin, connector, backConnector);
        if (!scoredPoints)
        {
            addToScore(10, 'viewing the maze');
            scoredPoints = true;
        }
    }
    scoredPoints = nil
;

+ bookStairsTop: Heavy, Readable, StairwayDown ->bookStairsBottom
    'big text stack/book/books/textbooks' 'stack of textbooks'
    "The top book is <i>The Rat Brain.</i>  The stack forms a
    stairway leading down. "

    readDesc = "You really don't have a lot of interest in reading
                this stuff. "
    
    pointingChange = southDirection
;
+ MazeWall;

+ Distant 'maze/layout' 'maze'
    "You can see the whole layout from here.  Now that you're going
    to escape, there's really no point in memorizing it, but you
    sure wish you'd had this view the times you were stuck down
    in there hunting for the cheese. "
;

/*
 *   Toolbox room
 */
toolboxRoom: Room 'Toolbox' 'the spot next to the toolbox'
    "A big red toolbox is sitting {to-the-west}, blocking the way,
    and {to-the-north} is a short wooden wall.  You can go
    {east} and {south}. "

    east = stackOfBooks
    south = deskRoom

    roomParts = [defaultFloor]

    travelerArriving(traveler, origin, connector, backConnector)
    {
        /* do our inherited work first */
        inherited(traveler, origin, connector, backConnector);

        /* 
         *   if we're on our second mission, finish it and go to the third
         *   mission 
         */
        if (me.missionStatus == 2)
        {
            local delta;
            local score;
            
            /* note the ending time */
            delta = getTime(GetTimeTicks) - me.missionStartTime;

            /* score it */
            score = (delta > 10000 ? 10 : 20 - delta/1000);
            addToScore(score, 'getting to the toolbox in '
                       + delta + ' milliseconds');

            "<.p>Mr.\ Tails is standing on top of the open toolbox.
            <q>Good to see you! What do you need?</q> You describe
            the type of screw you saw on the vent cover.  Mr.\ Tails
            disappears into the toolbox for a moment, then emerges
            with a screwdriver.  <q>Here you go,</q> he says, handing
            you the screwdriver.  He disappears back into the box
            for a moment, and emerges with some kind of small
            power tools.  <q>Get that over to the vent as fast
            as you can!  Now I'm off to the thermostat.  Can't
            have the heat coming on while we're in the ducts!</q>
            He climbs over the edge of the toolbox and leaps, giving
            the box such a kick that it slams shut behind him with
            a loud crash.  <q>Sorry about that!</q> he says as he
            runs off.
            <.p>
            [When you're ready, press a key to begin timing your
            journey back to the vent.]
            \n";

            /* empty my hands */
            foreach (local cur in me.contents)
            {
                if (!cur.ofKind(Component))
                    cur.moveInto(me.location);
            }

            /* get the screwdriver */
            screwdriver.moveInto(me);

            /* wait for a key */
            inputManager.getKey(nil, nil);

            /* note the new start time and mission status */
            me.missionStartTime = getTime(GetTimeTicks);
            me.missionStatus = 3;
        }
    }
;

+ MazeWall;
+ Heavy 'big red metal toolbox' 'toolbox'
    "It's a big red metal toolbox, currently closed. "
    dobjFor(Open)
    {
        verify() { }
        action() { reportFailure('You are not possessed of the necessary
            strength to open it. '); }
    }
;

/*
 *   Next to the desks
 */
deskRoom: Room 'Desks' 'the spot near the desks'
    "{To-the-west}, two desks have been arranged back-to-back, the
    sides facing this way.  The backs aren't quite touching, so there's
    a narrow gap between the desks, big enough for a rat.  A
    wheeled cart blocks the way {to-the-east}, but the floor is
    open {to-the-south} and {to-the-north}. "

    north = toolboxRoom
    south = ratChowRoom
    west = betweenDesks

    rootParts = [defaultFloor]
;
+ Heavy 'side/sides/desk/desks' 'desks'
    "The desks are sitting back-to-back, their sides facing this
    way.  There's a narrow space between the desks. "
    isPlural = true
;
+ spaceBetweenDesks: Enterable ->betweenDesks
    'narrow space/gap' 'narrow space'
    "It's a narrow space between the desks.  It looks like you
    could fit through. "
;

/*
 *   Between the desks 
 */
betweenDesks: Room 'Between the Desks' 'the space between the desks'
    "This space between the two desks is quite narrow, but that's
    not a problem for someone of your size.  Besides, rats like
    confined areas; it makes you feel more secure.  You can go
    {east} and {west}. "

    east = deskRoom
    west = fourCorners
;
+ Heavy 'desk/desks' 'desks'
    "The desks tower above, forming this narrow space. "
    isPlural = true
;

/*
 *   Four Corners (of desks)
 */
fourCorners: Room 'Four Corners' 'the four corners'
    "This is a narrow space at the corner junction of four desks.
    {To-the-east}, two desks are arranged back-to-back, the
    sides facing this way, and {to-the-west}, the same thing.
    The sides are almost touching, but not quite, leaving narrow
    gaps {to-the-north} and {to-the-south}.  The backs aren't
    quite touching, either, leaving spaces between them big enough
    for you to go {east} and {west}. "

    east = betweenDesks
    west = briefcaseDeadEnd
    south = trashcanDeadEnd
    north = betweenDesks2

    roomParts = [defaultFloor]
;
+ Heavy 'desk/desks' 'desks'
    "The four desks are arranged side-by-side and back-to-back, but
    the sides and backs aren't quite touching, leaving narrow gaps
    in four directions. "
    isPlural = true
;
+ Fixture 'narrow space/gap/spaces/gaps' 'narrow gap'
    "The spaces lead off in four directions.  Each one is large
    enough for you to enter. "
    dobjFor(Enter)
    {
        verify() { }
        action() { reportFailure('Just say which way you want to go. '); }
    }
    dobjFor(GoThrough) asDobjFor(Enter)
;

/*
 *   Dead end - blocked by briefcase 
 */
briefcaseDeadEnd: Room 'Dead End at Briefcase'
    'the dead end at the briefcase'
    "The exit {to-the-west} from this narrow gap is blocked by some
    large object - it looks like someone left a briefcase in front
    of the opening.  You can only go {east}. "

    east = fourCorners

    roomParts = [defaultFloor]
;

+ Heavy 'leather briefcase' 'briefcase'
    "It's hard to tell from this angle, but it looks like a big
    leather briefcase. "
;

+ Heavy 'desk/desks' 'desks'
    "The desks tower above, forming this narrow space. "
    isPlural = true
;

/*
 *   Dead end - blocked by trash can
 */
trashcanDeadEnd: Room 'Dead End at Trash Can'
    'the dead end at the trash can'
    "A round metal trash can has been pushed
    up against the opening {to-the-south} from the gap between
    the desks, blocking the way.  You can only go {north}. "

    north = fourCorners

    roomParts = [defaultFloor]
;

+ Heavy 'round metal trash can trashcan' 'trash can'
    "It's a round metal trash can. "
    dobjFor(LookIn)
    {
        action() { "It's too tall to see inside. "; }
    }
;

+ Heavy 'desk/desks' 'desks'
    "The desks tower above, forming this narrow space. "
    isPlural = true
;

/*
 *   Between the desks, part 2
 */
betweenDesks2: Room 'Between the Desks' 'the space between the desks'
    "This space between the desks is comfortably narrow.  You can go
    {north} and {south}. "

    north = openArea
    south = fourCorners

    roomPart = [defaultFloor]
;

+ Heavy 'desk/desks' 'desks'
    "The desks tower above, forming this narrow space. "
    isPlural = true
;

+ radioCar: Heavy, TravelBarrier
    'radio controlled rc radio-controlled toy car' 'toy car'
    "It's a radio-controlled toy car, sitting right up against the
    opening from the gap, blocking the way. "

    initDesc = "{To-the-north}, the opening from the gap is blocked
                by a toy car whose front end is pressed up against the desks.
                You think the car is radio-controlled, because you've seen
                the lab assistants playing with the car by remote
                control.  It's a large scale model, too big for you to
                be able to move or climb over. "

    dobjFor(Board)
    {
        verify() { }
        action() { "Fun as that might be, it doesn't actually have
            a passenger compartment, since it's only a model.  And
            it's too tall for you to climb on top of. "; }
    }
    dobjFor(Enter) asDobjFor(Board)

    canTravelerPass(traveler)
    {
        /* don't allow travel from betweenDesks2 if I'm present */
        return !(isIn(betweenDesks2) && traveler.isIn(betweenDesks2));
    }
    explainTravelBarrier(traveler)
    {
        reportFailure('You can\'t go that way as long as the toy
            car is in the way. ');
    }
;

/*
 *   Open area north of desks 
 */
openArea: Room 'Open Area' 'the open area'
    "This is a large open area of floor.  A couple of desks are
    {to-the-south}, their sides pushed together, but leaving a big
    enough gap for you to enter. "

    south = betweenDesks2
    east = computerRoom
    north = shelvesRoom
    west = chairRoom
    travelBarrier = [radioCar]

    roomPart = [defaultFloor]
;

+ Enterable -> betweenDesks2 'gap/space' 'gap'
    "It's a gap between the desks - narrow, but big enough for you. "
;
    
+ Heavy 'desk/desks' 'desks'
    "The two desks are pushed together at the sides, but there's
    enough space between them for you to enter. "
    isPlural = true
;

/*
 *   Computer Room 
 */
computerRoom: Room 'computer room' 'the computer room'
    "This area is crowded with computer equipment.  A desk is
    {to-the-south}.  The computers so surround you here that the
    only way you can go is {west}. "

    west = openArea

    roomParts = [defaultFloor]
;

+ Heavy 'desk' 'desk' "It's one of the standard-issue lab desks they
    use around here. "
;

+ Heavy 'computer equipment/computers' 'computers'
    "You're a rat; computers aren't exactly your forte. "
    isPlural = true
;

/*
 *   Room with shelves 
 */
shelvesRoom: Room 'Storage Shelves' 'the storage shelves'
    "Tall storage shelves, filled with exotic laboratory glassware and
    reagent bottles, wall off most of this area.  You can go
    {south} and {west} from here. "

    south = openArea
    west = ventRoom

    roomParts = [defaultFloor]
;

+ Fixture 'storage shelf/shelves' 'storage shelves'
    "The shelves tower above you; nothing is in reach, but you can see
    glassware and bottles of chemicals. "
    isPlural = true
;

+ Distant 'exotic lab laboratory reagent
    glassware/bottle/bottles/chemical/chemicals' 'glassware'
    "You can't see much detail from here. "
;

/*
 *   vent room 
 */
ventRoom: Room 'Corner' 'the corner'
    "This is a corner of the lab.  On the floor is a heating vent -
    it's large enough for a dozen rats to walk into at the same time,
    but right now it's covered up with a grating. "


    travelerArriving(traveler, origin, connector, backConnector)
    {
        /* do our inherited work first */
        inherited(traveler, origin, connector, backConnector);

        /* 
         *   if we haven't dispatched the player on the timed mission yet,
         *   do so now 
         */
        if (me.missionStatus == 0)
        {
            addToScore(10, 'finding the heating vent');

            "<.p>Wait a second - this must be the heating vent that
            Wilbert told you to find!  Success!
            <.p>
            Now you just have to get back to the cages and find
            Wilbert, to let him know where the vent is.  Time is
            critical in the escape plan, so you have to get back
            and find Wilbert as quickly as possible.  Hurry!
            <.p>
            [When you're ready, press a key - you're about to be
            timed on your navigational proficiency!]
            \n";

            inputManager.getKey(nil, nil);

            /* note that we're in the first timed mission */
            me.missionStatus = 1;

            /* note the clock start time */
            me.missionStartTime = getTime(GetTimeTicks);
        }
        else if (me.missionStatus == 3)
        {
            local delta;
            local score;
            
            /* note the ending time */
            delta = getTime(GetTimeTicks) - me.missionStartTime;

            /* score it */
            score = (delta > 20000 ? 10 : 30 - delta/1000);
            addToScore(score, 'returning to the heating vent in '
                       + delta + ' milliseconds');

            "<.p>Wilbert and several other of the rats are waiting
            at the vent.  <q>So, you're finally here!</q> Wilbert
            says with his typical impatience.  <q>And you have the
            screwdriver!  Okay, let's get this escape going!</q>
            <.p>
            Congratulations!  You've completed all missions and
            you've escaped the rat lab!
            <.p>";

            /* show the full score */
            libScore.showFullScore();

            /* offer finish options */
            finishGame([finishOptionFullScore]);
        }
    }

    east = shelvesRoom
    south = chairRoom

    roomParts = [defaultFloor, defaultNorthWall, defaultWestWall]
;

+ Fixture 'heating vent' 'heating vent'
    "It's a big vent opening, currently covered with a grating. "
    dobjFor(Open) remapTo(Open, ventGrating)
;

+ ventGrating: Immovable 'heating vent metal cover/grate/grating' 'grating'
    "It's a metal grating covering the vent.  There's no way into
    the vent without removing the grating, but it seems securely
    screwed in place. "

    cannotTakeMsg = 'It\'s securely screwed in place. '
    cannotMoveMsg = 'It\'s securely screwed in place. '
    cannotPutMsg = 'It\'s securely screwed in place. '

    dobjFor(Open)
    {
        verify() { }
        action() { reportFailure('You\'ll have to figure out how to
            remove the grating. '); }
    }
    dobjFor(Remove)
    {
        preCond = [touchObj]
        verify() { }
        action() { reportFailure('You\'ll need a screwdriver to
            do that. '); }
    }
    dobjFor(Take) asDobjFor(Remove)
    dobjFor(Unscrew) remapTo(Unscrew, ventScrews)
;

+ ventScrews: Immovable 'grating screw/screws' 'screws'
    "Several screws hold the grating in place.  No way you can remove
    them without a screwdriver. "
    isPlural = true

    dobjFor(Unscrew)
    {
        verify() { }
        action() { reportFailure('No way you can do that without
            a screwdriver. '); }
    }
    dobjFor(Remove) asDobjFor(Unscrew)
    dobjFor(Take) asDobjFor(Unscrew)
;

/*
 *   Chair Room 
 */
chairRoom: Room 'Next to Chair' 'the area next to the chair'
    "A big office chair sits pushed up against the wall, and a
    desk lies {to-the-south}.  You can go {north} and {east}. "

    north = ventRoom
    east = openArea

    roomParts = [defaultFloor, defaultWestWall]
;

+ Heavy 'desk' 'desk' "It's one of the standard-issue lab desks they
    use around here. "
;

+ Heavy 'big office chair' 'office chair'
    "It's just a big office chair. "
;


/* ------------------------------------------------------------------------ */
/*
 *   Screwdriver - we'll make this available at the proper time
 */
screwdriver: Thing 'small flat head headed flat-headed screwdriver'
    'screwdriver'
    "It's a small screwdriver with a flat head. "

    initDesc = "A small screwdriver is sitting on the floor next
                to the toolbox. "
;
   

/* ------------------------------------------------------------------------ */
/*
 *   ABOUT menu - the background information on the game is rather lengthy,
 *   so we use a menu structure to break it up a bit.  
 */

aboutMenu: MenuItem 'Hello, test subject!'
    fullScreenMode = true
;

+ MenuLongTopicItem 'Background'
    menuContents =
    ('This isn\'t really a game per se, but rather a human cognition
    experiment that grew out of some discussions on the Usenet group '
    + aHref('news:rec.arts.int-fiction', 'rec.arts.int-fiction') 
    + '.  Even so, I\'ve tried to have a little fun with it, and I
    hope my <q>experimental subjects</q> will enjoy it as well.

    <.p>The usenet discussions that led to this game were about the
    relationship between IF navigation conventions and human spatial
    cognition.  Two competing hypotheses were put forward in the
    discussions; the point of this experiment is to try to test
    objectively which hypothesis is a better fit to the way people
    really perceive their surroundings.  You can read more about the
    hypotheses and the design of the experiment by selecting those
    topics from the main ABOUT menu.

    <.p>Even if you don\'t buy the premise of the spatial cognition
    experiment, you might still find some value in this game as a
    separate personal experiment.  People sometimes say they\'d
    like it better if IF used <q>relative</q> directions - <q>ahead
    of you</q> or <q>to your left,</q> rather than <q>to the north</q>
    and the like.  Others say they like things the way they are, with
    descriptions given using compass directions.  There\'s never been
    much of a chance to compare the two methods head-to-head, though;
    very few games have ever used relative directions, and I don\'t
    know of any game that offers both options.  This game gives
    you a chance to compare the two styles <i>in the same game</i>
    - you can play this game in either mode, and you can switch
    modes at any time.  So, you can try it both ways and find out for
    yourself which style you like better as a player.

    <.p>Finally, even if everyone concludes the experiment is bogus,
    some authors might find the source code useful as a guide to
    implementing a relative direction system in their own games.
    This game is written in TADS 3, and the code herein that implements
    the relative direction system should be readily reusable in other
    TADS 3 games.'
    )
;
+ MenuLongTopicItem 'The Hypotheses'
    menuContents =
    ('Every so often, someone on ' + aHref('news:rec.arts.int-fiction',
                                           'rec.arts.int-fiction') +
    ' complains about the long-standing IF convention of using
    compass directions to describe room layouts and to enter travel
    commands.  The complaint is that this is unrealistic; after all,
    who goes around carrying a compass with them everywhere?  Wouldn\'t
    a better convention be either to use <q>relative</q> directions
    (so things are described as <q>to the left</q> or <q>behind you,</q>
    for example), or to dispense with directions entirely and just
    say where you want to go (<q>go to kitchen</q>)?

    <.p>People have been raising this criticism for a long time,
    but the compass convention has nonetheless persisted for something
    like 25 years now.  Some of the critics dismiss this as sheer
    inertia: the earliest adventure games happened to use this
    convention, and adventure games ever since have imitated it,
    the critics say, because authors don\'t have the imagination to
    try other ways. 
    But the empirical evidence suggests otherwise.  For one thing, 
    authors <i>have</i> tried other ways, including both the
    relative directions and named destinations approaches; but
    the alternatives haven\'t stuck.  Perhaps more
    convincing is the fact that the very first text adventure game,
    Colossal Cave Adventure, used named destinations to some extent.
    If inertia were the only factor, then games today ought to be
    using the same mix of compass directions and named destinations
    that Colossal Cave did.  But they don\'t; the compass approach
    seems to be truly more popular than the others.

    <.p>My hypothesis is that the compass approach has survived
    because it\'s a close match to the way people actually
    perceive their physical surroundings.  My claim is that humans
    have an innate faculty for spatial perception - it\'s not something
    we learn in school, not something that\'s all loaded with 
    cultural biases, but a pre-wired piece of our mental hardware
    that\'s pretty much the same in everyone, an ancient faculty
    that probably even pretty lowly creatures possess.  And I think
    we have an intuitive understanding, at this subconscious level,
    that the world around us is fixed in space and that our bodies move
    through it.  In other words, I don\'t think we imagine our
    field of view to be the center of the universe; I don\'t think
    we imagine that when we turn our head 90 degrees left,
    the whole universe is actually turning 90 degrees right.  If
    we do understand this, then it follows that we would mentally
    model the external world as fixed in
    space, and we\'d mentally project ourselves into that model.  If
    that mental model sees our surroundings as fixed in space,
    then it would be simplest if we saw our surroundings as having some
    fixed orientation: there\'s a <q>top of the map</q> in our
    minds\'s eyes.

    <.p>I\'m not suggesting, by the way, that this <q>top of the map</q>
    is aligned with magnetic north.  There\'s no evidence that humans
    have any sensory apparatus that can detect the Earth\'s magnetic
    field, so magnetic north simply isn\'t part of our innate mental
    landscape (we can learn about it, and we can observe and measure
    it with external tools, but that\'s a separate matter from our
    innate cognition).  The top of the map is simply an arbitrary
    direction we choose in making our mental maps of places.  It
    might be some local landmark, like the front door of a house,
    or it might just be some accident of our mindset the first time
    we saw the place.  I\'m also not proposing that the top of the
    map is global for any one person; rather, I think we keep little
    islands of detailed maps in our brains, corresponding to local
    areas we know in detail, and each island has its own arbitrary
    orientation.  I see the IF compass convention as equally
    arbitrary.  The compass directions in IF aren\'t meant to be
    <i>real</i> compass directions; they\'re just a convenient
    short-hand that lets the author and player share a frame of
    reference.  If we view the IF convention this way, the complaint
    about realism - that no one carries a compass everywhere - is missing
    the point: the game directions are arbitrary, and just happen to
    be <i>named</i> like the compass directions.  Sort of like
    the way the players are labeled in bridge games.

    <.p>I have a few thought experiments to suggest in support of
    my hypothesis.  

    <.p>First, imagine yourself sitting in some random
    room in your own house.  Can you point in the rough direction of
    the street outside?  Even if it\'s a room from which you can\'t
    see the street?  Most people can, without even thinking about it,
    and I claim this is because they have a map in their mind that
    has the layout of the house and its immediate surroundings,
    and they know where they are in that map at any given time.

    <.p>Second, have you ever been walking around a largish area you 
    haven\'t been to before, maybe a park or shopping mall, and
    at some point you come across a landmark you remember from
    earlier, but it wasn\'t at all where you thought it would
    be?  Most people have at one time or another had this feeling
    of being all turned around, that they thought they were
    going one way but it turned out they were going another.
    My contention is this feeling comes directly from that mental
    map we have of our surroundings; that feeling is the discovery
    that your mental map doesn\'t line up with reality the way you
    thought it did.  

    <.p>Third, imagine you\'re in a room you haven\'t
    seen before.  You take note of the layout: a couple of bookcases,
    a single door directly in front of you.  Now, you turn around
    180 degrees.  Do you still know there\'s a door, and that it\'s
    behind you?  If so, you obviously have <i>some</i> kind of map
    in your head, because you\'re aware of something that\'s not in
    your field of view.  When you turned around, did you imagine that
    you were forcing the room to spin around while your head remained
    fixed in space?  Or did you feel like it was you who was turning
    while the room remained fixed in space?

    <.p>The competing hypothesis, or at least the main competing
    viewpoint expressed in the course of the newsgroup discussions
    that led to this experiment, is that our brains have a sort of
    instantaneous first-person view of our surroundings, and that
    we rotate our mental model of the world around us every time
    we turn our bodies.  We don\'t have a map per se, but rather
    we remember moment by moment a big set of nearby objects, and
    we keep a <q>vector</q> for each object from our eyes to the
    object\'s position.  So, every time we turn our head, we
    recalculate all of those vector angles.
    This viewpoint holds that we have some mental
    hardware that does these rotational mappings subconsciously,
    which makes them fast and efficient.  This viewpoint also
    holds that we don\'t have a larger map of our surroundings
    in our heads with a sense of the spatial relations of the
    larger unseen context, but that we merely remember a set of
    <q>routes</q> between locations: to get from my bedroom to
    the kitchen, I go forward, right, down the stairs, left, and
    right.  The contention is that we store this list of
    directions, and we don\'t abstract these to a bigger-picture
    map that we store in our heads. ')
;
+ MenuLongTopicItem 'How the Experiment Works'
    'The reasoning behind this experiment is simple: the model that\'s
    closer to our actual internal cognitive model ought to be
    easier for us to use.  To put this in quantifiable terms,
    I claim that a person should be able to perform a task more
    quickly when the task is cast into a model that more closely
    matches their true internal mental model.  For this experiment,
    the measurement is the amount of time it takes to perform a
    given navigational task within an IF setting.

    <.p>Here\'s how this works.  This game uses the usual IF
    convention of locations connected by directional passages.
    The map isn\'t huge but has enough to make it non-trivial.
    The game can be played in two modes: <q>compass directions</q>
    and <q>relative directions.</q>  In compass mode, the game
    uses the traditional IF convention of describing the setting
    by saying that there\'s a passage leading north, one leading
    east, and so on, and travel is performed by entering GO NORTH
    commands and the like.  In RELATIVE mode, the game uses the
    directions AHEAD, BACK, LEFT, and RIGHT.

    <.p>The experiment has several stages, but they all occur
    in the context of the game, so you can just follow the game\'s
    lead as you go and you\'ll be fine.  The first part is meant to
    let you learn your way around the map; there are no time
    measurements here, so there\'s no need to feel rushed.  Once
    you know your way around, the game will present you with a
    few timed navigational tasks.  The idea is to find your
    way to the new location as quickly as possible.  The game will
    let you know when you\'re being timed.  Once you\'ve accomplished
    everything, the game will give you a summary of your scores.

    <.p>The first section, by the way, is meant to address the
    concern I have that most experienced IF players will have a
    natural advantage with the conventional compass directions,
    since they\'re accustomed to them from other games.  The first
    part is meant to give you a chance to get used to the relative mode,
    when you\'re playing in relative mode, to reduce any systematic
    bias from past experience with compass-mode games.

    <.p>The methodology I\'m proposing is that each person should try
    the game in both COMPASS and RELATIVE direction modes, and
    compare their scores in the two modes.  However, the mode you
    choose first is undoubtedly going to affect the outcome of
    the second mode, since you\'ll already know the map when you
    try the experiment again in the second mode.  Therefore, I\'d
    like to ask that you make your first attempt in the mode randomly
    chosen when you first start up the game.  If everyone does this,
    then we should get a good sampling of each initial mode.

    <.p>I have no training in cognitive science, so I can\'t promise
    this experiment will actually tell us anything useful.  In the
    worst case, the source code for this game might be useful to
    authors who want to use relative direction systems in their
    own games.  Feedback from anyone who actually knows something
    about constructing cognitive experiments would be especially
    interesting.

    <.p>Thanks for participating! '
;
+ MenuLongTopicItem 'COMPASS and RELATIVE modes'
    'This game can be played in two modes: COMPASS and RELATIVE.

    <.p>In COMPASS mode, the game uses the traditional IF compass
    directions - NORTH, SOUTH, EAST, WEST.  The layout of each room
    is described in terms of compass directions, and you travel
    from location to location using commands like GO NORTH, GO
    SOUTH, etc.  For convenience, you can abbreviate the travel
    commands to one letter each: N, S, E, W.

    <.p>In RELATIVE mode, the game instead
    uses the directions AHEAD, BACK, LEFT, and RIGHT, in both
    descriptions and command input.  In addition, the game
    keeps track of which way the player character is pointing at
    any given time.  Each time you travel, you change your
    orientation to point in the direction of travel.  So if
    you type GO LEFT, this means you turn left and then walk
    to the location in that direction, so that you\'re now pointed
    90 degrees counter-clockwise from your prior orientation.
    You can also type TURN LEFT, TURN RIGHT, and TURN AROUND
    to turn 90 degrees counter-clockwise, 90 degrees clockwise,
    and 180 degrees, respectively, to adjust your orientation at
    any time.  The game displays all room descriptions in terms
    of your current orientation.

    <.p>In relative mode, you can abbreviate AHEAD as A or F
    (for FORWARD), BACK as B, LEFT as LE or LF, and RIGHT as R.
    (LEFT needs a two-letter abbreviation because L is already
    taken for LOOK.)

    <.p>You can switch modes at any time by typing COMPASS or
    RELATIVE at a command prompt.  For the purposes of the
    experiment, note that we ask that you play through the game
    once in the mode that\'s randomly selected at start-up.
    Replaying once you know the map might skew the results,
    so the results from your first time through are
    probably the most important.  By randomly choosing an initial
    mode, we should get a good mix of people playing for the
    first time in each mode.'
;

+ MenuLongTopicItem 'Scoring'
    'The outcome of the experiment is the set of timings for the
    navigational tasks you\'ll be given in the course of the game.
    When you\'ve completed the game, you\'ll be shown your full
    score, which will show you the timing for each task in milliseconds.
    Those are the most important numbers.

    <.p>For tradition\'s sake, the game also gives you a conventional
    score in points.  There are a couple of simple untimed tasks that
    will let you get a certain number of points, and there are a few
    timed tasks that award points according to how quickly you complete
    the tasks.  Each timed task gives you a base number of points just
    for completing the task, and then gives you bonus points if you\'re
    faster than a benchmark time.  The benchmark is just a random guess
    I made.  The bonus points are on a sliding scale that gives you the
    maximum number of points if you can complete the task in less than
    a second - since this is probably all but impossible, so you
    definitely shouldn\'t feel bad if you don\'t get a perfect score.'
;

/*
 *   Map HELP and HINT to the ABOUT display
 */
VerbRule(Help) 'help' | 'hint' : AboutAction;

