//===========================================================================
// Copyright  1998 Thin Air Enterprises.  All rights reserved.
//===========================================================================
//---------------------------------------------------------------------------
// RichEdit20Format.cpp - implementation file for TRichEdit20 character and
// paragraph formatting classes.
//---------------------------------------------------------------------------
// Much of this code is a direct translation from the Borland C++ Builder 3.0
// VCL Pascal code.  I have corrected one flaw in the design (could not set a
// character attribute for a multiply formatted selection without setting all
// of the attributes identically, i.e., if the selection contained some bold,
// some italic, and some underscored text, setting the bold on or off also
// changed the selection to all/no italic and underscored).  I also added
// support for Rich Edit 2.0 character and paragraph formatting extensions.
//
// Unless you are using Rich Edit 2.0, there is very little point in looking
// at or using this code with one exception:  the way the BCB 1.0 & 2.0 VCL
// applies text formats, you cannot change a single format.  for example,
// if you have text formatted as bold and italics, turning off bold will
// also turn off italics.  this code overcomes that by adding properties
// for each style.  this could be easily back-ported to BCB 1.0.
//---------------------------------------------------------------------------
#include <vcl.h>
#include <vcl\SysUtils.hpp>
#include <vcl\Classes.hpp>
#include <vcl\Forms.hpp>
#include <vcl\SysUtils.hpp>
#include <vcl\ComCtrls.hpp>
#include <vcl\StdCtrls.hpp>

#include <vector>
#include <values.h>

#pragma hdrstop

#include "RichEdit20Format.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
#include "richedit20format.h"
//---------------------------------------------------------------------------
// TTextAttributes2 class
//---------------------------------------------------------------------------
__fastcall TTextAttributes2::TTextAttributes2(TCustomRichEdit* AOwner,
	TAttributeType AttributeType) : TTextAttributes(AOwner, AttributeType)
{
	FRichEdit2 = AOwner;
	FType2 = AttributeType;
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::InitFormat2(Richedit::TCharFormat2A &Format)
{
	::memset(&Format, 0, sizeof(Format));
	Format.cbSize = sizeof(Format);
}
//---------------------------------------------------------------------------
TConsistentAttributes2 __fastcall TTextAttributes2::GetConsistentAttributes2(void)
{
	TCharFormat2A Format;
	TConsistentAttributes2 Result;

	if (FRichEdit2->Handle && FType2 == atSelected) {
		InitFormat2(Format);
		::SendMessage(FRichEdit2->Handle, EM_GETCHARFORMAT,
			(WPARAM) FType2 == atSelected, (LPARAM) &Format);
		if (Format.dwMask & CFM_BOLD) Result << ca2Bold;
		if (Format.dwMask & CFM_COLOR) Result << ca2Color;
		if (Format.dwMask & CFM_FACE) Result << ca2Face;
		if (Format.dwMask & CFM_ITALIC) Result << ca2Italic;
		if (Format.dwMask & CFM_SIZE) Result << ca2Size;
		if (Format.dwMask & CFM_STRIKEOUT) Result << ca2StrikeOut;
		if (Format.dwMask & CFM_UNDERLINE) Result << ca2Underline;
		if (Format.dwMask & CFM_PROTECTED) Result << ca2Protected;
		if (Format.dwMask & CFM_LINK) Result << ca2Link;
		if (Format.dwMask & (CFM_SUPERSCRIPT)) Result << ca2SuperScript;
		if (Format.dwMask &	(CFM_SUBSCRIPT)) Result << ca2SubScript;
		}
	return Result;
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::GetAttributes2(Richedit::TCharFormat2A &Format)
{
	InitFormat2(Format);
	if (FRichEdit2->Handle) ::SendMessage(FRichEdit2->Handle, EM_GETCHARFORMAT,
		(WPARAM) FType2 == atSelected, (LPARAM) &Format);
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::SetAttributes2(Richedit::TCharFormat2A &Format)
{
	int Flag = (FType2 == atSelected) ? SCF_SELECTION : 0;

	if (FRichEdit2->Handle) ::SendMessage(FRichEdit2->Handle, EM_SETCHARFORMAT,
		(WPARAM) Flag, (LPARAM) &Format);
}
//---------------------------------------------------------------------------
Graphics::TColor __fastcall TTextAttributes2::GetBackColor(void)
{
	TCharFormat2A Format;

	GetAttributes2(Format);
	if (Format.dwEffects & CFE_AUTOBACKCOLOR) return clWindow;
	return TColor(Format.crBackColor);
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::SetBackColor(Graphics::TColor Value)
{
	TCharFormat2A Format;

	InitFormat2(Format);
	Format.dwMask = CFM_BACKCOLOR;
	if (Value == clWindow) Format.dwEffects = CFE_AUTOBACKCOLOR;
	else Format.crTextColor = ColorToRGB(Value);
	Format.crBackColor = ColorToRGB(Value);
	SetAttributes2(Format);
}
//---------------------------------------------------------------------------
TFontStyles2 __fastcall TTextAttributes2::GetStyle2(void)
{
	TCharFormat2A Format;

	TFontStyles2 Result;
	GetAttributes2(Format);
	if (Format.dwEffects & CFE_BOLD) Result << fs2Bold;
	if (Format.dwEffects & CFE_ITALIC) Result << fs2Italic;
	if (Format.dwEffects & CFE_UNDERLINE) Result << fs2Underline;
	if (Format.dwEffects & CFE_STRIKEOUT) Result << fs2StrikeOut;
	if (Format.dwEffects & CFE_SUPERSCRIPT) Result << fs2SuperScript;
	if (Format.dwEffects & CFE_SUBSCRIPT) Result << fs2SubScript;

	return Result;
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::SetStyle2(TFontStyles2 Value)
{
	TCharFormat2A Format;

	InitFormat2(Format);
	Format.dwMask = CFM_BOLD | CFM_ITALIC | CFM_UNDERLINE | CFM_STRIKEOUT |
		CFM_SUPERSCRIPT | CFM_SUBSCRIPT;

	if (Value.Contains(fs2Bold)) Format.dwEffects |= CFE_BOLD;
	if (Value.Contains(fs2Italic)) Format.dwEffects |= CFE_ITALIC;
	if (Value.Contains(fs2Underline)) Format.dwEffects |= CFE_UNDERLINE;
	if (Value.Contains(fs2StrikeOut)) Format.dwEffects |= CFE_STRIKEOUT;
	if (Value.Contains(fs2SuperScript)) Format.dwEffects |= CFE_SUPERSCRIPT;
	if (Value.Contains(fs2SubScript)) Format.dwEffects |= CFE_SUBSCRIPT;

	SetAttributes2(Format);
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::Assign(Classes::TPersistent* Source)
{
	// call inherited
	TTextAttributes::Assign(Source);

	// are we assigning from TTextAttributes2?
	TTextAttributes2* ta = dynamic_cast<TTextAttributes2*>(Source);
	if (ta) {
		BackColor = ta->BackColor;
		Style2 = ta->Style2;
		Link = ta->Link;
		}
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::AssignTo(Classes::TPersistent* Dest)
{
	// call inherited
	TTextAttributes::AssignTo(Dest);

	// are we assigning to a TTextAttributes2?
	TTextAttributes2* ta = dynamic_cast<TTextAttributes2*>(Dest);
	if (ta) {
		ta->BackColor = BackColor;
		ta->Link = Link;
		ta->Style2 = Style2;
		}
}
//---------------------------------------------------------------------------
TUnderlineType __fastcall TTextAttributes2::GetUnderlineType(void)
{
	TCharFormat2A Format;

	GetAttributes2(Format);
	if (!(Format.dwEffects & CFE_UNDERLINE)) return utNone;

	switch (Format.bUnderlineType) {
		case CFU_CF1UNDERLINE: 		return utCF1Underline;
		case CFU_UNDERLINE:			return utUnderline;
		case CFU_UNDERLINEDOTTED:	return utDotted;
		case CFU_UNDERLINEDOUBLE:	return utDouble;
		case CFU_UNDERLINEWORD:		return utWord;
		}
	return utNone;
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::SetUnderlineType(TUnderlineType ut)
{
	TCharFormat2A Format;

	InitFormat2(Format);
	Format.dwMask = CFM_UNDERLINETYPE;

	switch (ut) {
		case utCF1Underline:
			Format.bUnderlineType = CFU_CF1UNDERLINE;
			break;
		case utUnderline:
			Format.bUnderlineType = CFU_UNDERLINE;
			break;
		case utDotted:
			Format.bUnderlineType = CFU_UNDERLINEDOTTED;
			break;
		// the next two yield single, unbroken underlines...
		case utDouble:
			Format.bUnderlineType = CFU_UNDERLINEDOUBLE;
			break;
		case utWord:
			Format.bUnderlineType = CFU_UNDERLINEWORD;
			break;
		default:
			Format.bUnderlineType = CFU_UNDERLINENONE;
		}

	SetAttributes2(Format);
}
//---------------------------------------------------------------------------
// the following property handlers allow getting/setting a single character
// attribute without changing other attributes
//
TThreeState __fastcall TTextAttributes2::GetFontStyle(DWORD effect, DWORD mask)
{
	TCharFormat2A Format;

	GetAttributes2(Format);
	if (!(Format.dwMask & mask)) return tsMaybe;
	if (Format.dwEffects & effect) return tsYes;
	return tsNo;
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::SetFontStyle(DWORD effect, DWORD mask,
	TThreeState state)
{
	if (state == tsMaybe) return;		// do nothing

	TCharFormat2A Format;

	InitFormat2(Format);
	Format.dwMask = mask;
	if (state == tsYes) Format.dwEffects |= effect;

	SetAttributes2(Format);
}
//---------------------------------------------------------------------------
TThreeState __fastcall TTextAttributes2::GetBold(void)
{
	return GetFontStyle(CFE_BOLD, CFM_BOLD);
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::SetBold(TThreeState state)
{
	SetFontStyle(CFE_BOLD, CFM_BOLD, state);
}
//---------------------------------------------------------------------------
TThreeState __fastcall TTextAttributes2::GetItalic(void)
{
	return GetFontStyle(CFE_ITALIC, CFM_ITALIC);
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::SetItalic(TThreeState state)
{
	SetFontStyle(CFE_ITALIC, CFM_ITALIC, state);
}
//---------------------------------------------------------------------------
TThreeState __fastcall TTextAttributes2::GetUnderline(void)
{
	return GetFontStyle(CFE_UNDERLINE, CFM_UNDERLINE);
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::SetUnderline(TThreeState state)
{
	SetFontStyle(CFE_UNDERLINE, CFM_UNDERLINE, state);
}
//---------------------------------------------------------------------------
TThreeState __fastcall TTextAttributes2::GetStrikeOut(void)
{
	return GetFontStyle(CFE_STRIKEOUT, CFM_STRIKEOUT);
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::SetStrikeOut(TThreeState state)
{
	SetFontStyle(CFE_STRIKEOUT, CFM_STRIKEOUT, state);
}
//---------------------------------------------------------------------------
TThreeState __fastcall TTextAttributes2::GetSuperScript(void)
{
	return GetFontStyle(CFE_SUPERSCRIPT, CFM_SUPERSCRIPT);
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::SetSuperScript(TThreeState state)
{
	SetFontStyle(CFE_SUPERSCRIPT, CFM_SUPERSCRIPT, state);
}
//---------------------------------------------------------------------------
TThreeState __fastcall TTextAttributes2::GetSubScript(void)
{
	return GetFontStyle(CFE_SUBSCRIPT, CFM_SUBSCRIPT);
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::SetSubScript(TThreeState state)
{
	SetFontStyle(CFE_SUBSCRIPT, CFM_SUBSCRIPT, state);
}
//---------------------------------------------------------------------------
TThreeState __fastcall TTextAttributes2::GetLink(void)
{
	return GetFontStyle(CFE_LINK, CFM_LINK);
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::SetLink(TThreeState state)
{
	SetFontStyle(CFE_LINK, CFM_LINK, state);
}
//---------------------------------------------------------------------------
TThreeState __fastcall TTextAttributes2::GetProtected2(void)
{
	return GetFontStyle(CFE_PROTECTED, CFM_PROTECTED);
}
//---------------------------------------------------------------------------
void __fastcall TTextAttributes2::SetProtected2(TThreeState state)
{
	SetFontStyle(CFE_PROTECTED, CFM_PROTECTED, state);
}
//---------------------------------------------------------------------------
// TParaAttributes2 class
//---------------------------------------------------------------------------
__fastcall TParaAttributes2::TParaAttributes2(TCustomRichEdit* AOwner) :
	TParaAttributes(AOwner)
{
	FRichEdit2 = AOwner;
}
//---------------------------------------------------------------------------
void TParaAttributes2::InitFormat2(PARAFORMAT2& format)
{
	::memset(&format, 0, sizeof(format));
	format.cbSize = sizeof(format);
}
//---------------------------------------------------------------------------
void TParaAttributes2::GetFormat2(PARAFORMAT2& format)
{
	InitFormat2(format);
	if (FRichEdit2->Handle)
		::SendMessage(FRichEdit2->Handle, EM_GETPARAFORMAT, 0, (LPARAM) &format);
}
//---------------------------------------------------------------------------
void TParaAttributes2::SetFormat2(PARAFORMAT2& format)
{
	if (FRichEdit2->Handle)
		::SendMessage(FRichEdit2->Handle, EM_SETPARAFORMAT, 0, (LPARAM) &format);
}
//---------------------------------------------------------------------------
TParaConsistentAttributes __fastcall TParaAttributes2::GetConsistentAttributes(void)
{
	TParaConsistentAttributes pca;
	PARAFORMAT2 format;

	GetFormat2(format);
	DWORD mask = format.dwMask;

	if (mask & PFM_ALIGNMENT) pca << pcaAlignment;
	if (mask & PFM_LINESPACING) pca << pcaLineSpacing;
	if (mask & PFM_NUMBERING) pca << pcaNumbering;
	if (mask & PFM_OFFSET) pca << pcaOffset;
	if (mask & PFM_OFFSETINDENT) pca << pcaOffsetIndent;
	if (mask & PFM_RIGHTINDENT) pca << pcaRightIndent;
	if (mask & PFM_SPACEAFTER) pca << pcaSpaceAfter;
	if (mask & PFM_SPACEBEFORE) pca << pcaSpaceBefore;
	if (mask & PFM_STARTINDENT) pca << pcaStartIndent;
	if (mask & PFM_TABSTOPS) pca << pcaTabStops;

	return pca;
}
//---------------------------------------------------------------------------
TAlignment2 __fastcall TParaAttributes2::GetAlignment2(void)
{
	TParaConsistentAttributes pca = GetConsistentAttributes();
	if (!pca.Contains(pcaAlignment)) return ta2Indeterminate;

	// use inherited for remaining possibilities
	switch (Alignment) {
		case taLeftJustify:		return ta2LeftJustify;
		case taCenter:			return ta2Center;
		case taRightJustify:	return ta2RightJustify;
		}
	return ta2Indeterminate;	// can happen -- full justification is TOM only
}
//---------------------------------------------------------------------------
void __fastcall TParaAttributes2::SetAlignment2(TAlignment2 alignment)
{
	// note: does nothing if not valid (ta2Indeterminate)
	switch (alignment) {
		case ta2LeftJustify:	Alignment = taLeftJustify; break;
		case ta2Center:			Alignment = taCenter; break;
		case ta2RightJustify:	Alignment = taRightJustify; break;
		}
}
//---------------------------------------------------------------------------
TNumberingStyle2 __fastcall TParaAttributes2::GetNumbering2(void)
{
	TParaConsistentAttributes pca = GetConsistentAttributes();
	if (!pca.Contains(pcaNumbering)) return ns2Indeterminate;

	// use inherited for remaining possibilities
	if (Numbering == nsNone) return ns2None;
	if (Numbering == nsBullet) return ns2Bullet;
	return ns2Indeterminate;		// can happen...
}
//---------------------------------------------------------------------------
void __fastcall TParaAttributes2::SetNumbering2(TNumberingStyle2 numberingStyle)
{
	// does nothing if not valid (ns2Indeterminate)
	if (numberingStyle == ns2None) Numbering = nsNone;
	else if (numberingStyle == ns2Bullet) Numbering = nsBullet;
}
//---------------------------------------------------------------------------
int __fastcall TParaAttributes2::GetSpaceBefore(void)
{
	PARAFORMAT2 format;
	GetFormat2(format);

	return format.dwMask & PFM_SPACEBEFORE ? format.dySpaceBefore : 0;
}
//---------------------------------------------------------------------------
void __fastcall TParaAttributes2::SetSpaceBefore(int twips)
{
	PARAFORMAT2 format;

	InitFormat2(format);
	format.dySpaceBefore = twips;
	format.dwMask = PFM_SPACEBEFORE;
	SetFormat2(format);
}
//---------------------------------------------------------------------------
int __fastcall TParaAttributes2::GetSpaceAfter(void)
{
	PARAFORMAT2 format;
	GetFormat2(format);

	return format.dwMask & PFM_SPACEAFTER ? format.dySpaceAfter : 0;
}
//---------------------------------------------------------------------------
void __fastcall TParaAttributes2::SetSpaceAfter(int twips)
{
	PARAFORMAT2 format;

	InitFormat2(format);
	format.dySpaceAfter = twips;
	format.dwMask = PFM_SPACEAFTER;
	SetFormat2(format);
}
//---------------------------------------------------------------------------
TLineSpacing __fastcall TParaAttributes2::GetLineSpacing(void)
{
	PARAFORMAT2 format;
	TLineSpacing ls = lsIndeterminate;

	GetFormat2(format);
	if (format.dwMask & PFM_LINESPACING) {
		switch (format.bLineSpacingRule) {
			case 0: ls = lsSingle;
			case 1: ls = lsOneAndOneHalf;
			case 2: ls = lsDouble;
			}
		}

	return ls;
}
//---------------------------------------------------------------------------
void __fastcall TParaAttributes2::SetLineSpacing(TLineSpacing lineSpacing)
{
	PARAFORMAT2 format;

	InitFormat2(format);

	// note: default == single == 0
	switch (lineSpacing) {
		case lsOneAndOneHalf:	format.bLineSpacingRule = (BYTE) 1; break;
		case lsDouble:			format.bLineSpacingRule = (BYTE) 2; break;
		}

	SetFormat2(format);
}
//---------------------------------------------------------------------------
int __fastcall TParaAttributes2::NextTab(int fromPos)
{
	PARAFORMAT2 format;

	GetFormat2(format);

	// note: we have a valid set of tabstops -- they may, however,
	// be inconsistent since multiple paragraphs may be selected....
	//
	// if only 1 tabstop, return a multiple of that value
	if (format.cTabCount < 2) {
		// note: the "+" outside of the parenthesis is an attempt to force
		// the compiler to not optimize away the expression....  this was
		// documented in BC++ and I *hope* that it still applies....
		return +((fromPos + format.rgxTabs[0]) / format.rgxTabs[0]) *
			format.rgxTabs[0];
		}

	// we will allow an unlimited number of tabstops -- by taking the last two
	// valid tabstops and adding that difference to the last valid tabstop
	// until we reach a value greater than the requested position...
	if (fromPos > format.rgxTabs[format.cTabCount - 1]) {
		int tabBase = format.rgxTabs[format.cTabCount - 1];
		int tabIncr = tabBase - format.rgxTabs[format.cTabCount - 2];
		int diffBase = fromPos - tabBase;
		int nextPos = +((diffBase + tabIncr) / tabIncr) * tabIncr;
		nextPos += tabBase;
		return nextPos;
		}
	// else we find the tab position that follows the passed position parameter
	else for (int i = 0; i < format.cTabCount; i++)
		if (format.rgxTabs[i] > fromPos) return format.rgxTabs[i];

	// if all else fails, return zero
	return 0;
}
//---------------------------------------------------------------------------
int __fastcall TParaAttributes2::PriorTab(int fromPos)
{
	PARAFORMAT2 format;

	GetFormat2(format);

	if (fromPos < 1) return 0;

	// note: we have a valid set of tabstops -- they may, however,
	// be inconsistent since multiple paragraphs may be selected....
	//
	// if only 1 tabstop, return a multiple of that value
	if (format.cTabCount < 2) {
		// note: the "+" outside of the parenthesis is an attempt to force
		// the compiler to not optimize away the expression....  this was
		// documented in BC++ and I *hope* that it still applies....
		return +((fromPos - 1) / format.rgxTabs[0]) * format.rgxTabs[0];
		}

	// we will allow an unlimited number of tabstops -- by taking the last two
	// valid tabstops and adding that difference to the last valid tabstop
	// until we reach a value less than the requested position...
	if (fromPos > format.rgxTabs[format.cTabCount - 1]) {
		int tabBase = format.rgxTabs[format.cTabCount - 1];
		int tabIncr = tabBase - format.rgxTabs[format.cTabCount - 2];
		int diffBase = fromPos - tabBase;
		int nextPos = +((diffBase - 1) / tabIncr) * tabIncr;
		nextPos += tabBase;
		return nextPos;
		}

	// otherwise we find the tab position that follows the passed position parameter
	int pos = 0;
	for (int i = 0; i < format.cTabCount; i++) {
		if (format.rgxTabs[i] >= fromPos) break;
		pos = format.rgxTabs[i];
		}
	return pos;
}
//---------------------------------------------------------------------------
int __fastcall TParaAttributes2::GetFirstIndent2(void)
{
	PARAFORMAT2 format;
	GetFormat2(format);
	return format.dxStartIndent;
}
//---------------------------------------------------------------------------
void __fastcall TParaAttributes2::SetFirstIndent2(int twips)
{
	PARAFORMAT2 format;
	InitFormat2(format);
	format.dwMask = PFM_STARTINDENT;
	format.dxStartIndent = twips;
	SetFormat2(format);
}
//---------------------------------------------------------------------------
int __fastcall TParaAttributes2::GetLeftIndent2(void)
{
	PARAFORMAT2 format;
	GetFormat2(format);
	return format.dxOffset;
}
//---------------------------------------------------------------------------
void __fastcall TParaAttributes2::SetLeftIndent2(int twips)
{
	PARAFORMAT2 format;
	InitFormat2(format);
	format.dwMask = PFM_OFFSET;
	format.dxOffset = twips;
	SetFormat2(format);
}
//---------------------------------------------------------------------------
int __fastcall TParaAttributes2::GetRightIndent2(void)
{
	PARAFORMAT2 format;
	GetFormat2(format);
	return format.dxRightIndent;
}
//---------------------------------------------------------------------------
void __fastcall TParaAttributes2::SetRightIndent2(int twips)
{
	PARAFORMAT2 format;
	InitFormat2(format);
	format.dwMask = PFM_RIGHTINDENT;
	format.dxRightIndent = twips;
	SetFormat2(format);
}
//---------------------------------------------------------------------------
// end RichEdit20Format.cpp
//---------------------------------------------------------------------------
//===========================================================================
// Copyright  1998 Thin Air Enterprises.  All rights reserved.
//===========================================================================
