var ZScheduler = (function() {
    var queue = [];
    return function(thunk) {
        if (thunk) {
            queue.push(thunk);
            setTimeout("ZScheduler()",10);
        } else if (queue.length > 0) {
            queue.shift()();
        }
    };
})();

function ZInterp(iframe, uri, htmlPageBlurb) {
    var VERSION = "0.0-(20060106-8)";

    function ZScreen(iframe) {
        var doc = iframe.contentDocument || document.frames[iframe.name].document,
            notifyResize = null,

            upperSelected = false,
            fontFamily = "serif",
            backgroundColor = "white",
            foregroundColor = "black",
            textBold = false,
            textItalic = false,
            textReverse = false,
            textFixed = false,
            textFixed2 = false,
            screenWidth = 0,
            screenHeight = 0,
            screenCols = 0,
            screenRows = 0,
            upperWin,
            lowerWin,
            lowerObscured;

        (function() {
            var html = doc.getElementsByTagName("html");
            if (html.length == 0) {
                html = doc.createElement("html");
                doc.appendChild(html);
            } else {
                html = html[0];
            }
            var body = html.getElementsByTagName("body");
            if (body.length == 0) {
                body = doc.createElement("body");
                html.appendChild(body);
            } else {
                body = body[0];
            }

            var upperDiv = doc.createElement("span");
            body.appendChild(upperDiv);
            upperDiv.style.backgroundColor = "white";
            upperDiv.style.position = "fixed";
            upperDiv.style.left = 0;
            upperDiv.style.top = 0;
            upperDiv.style.borderStyle = "none";
            upperDiv.style.fontFamily = "monospace";
            upperDiv.style.whiteSpace = "pre";

            var table = doc.createElement("table");
            upperDiv.appendChild(table);
            table.border = 0;
            table.width = iframe.offsetWidth;
            var upperWinResizer = table;
            var tbody = doc.createElement("tbody");
            table.appendChild(tbody);
            var tr = doc.createElement("tr");
            tbody.appendChild(tr);
            var td = doc.createElement("td");
            tr.appendChild(td);
            td.align = "left";
            td.style.whiteSpace = "pre";
            upperWin = td;

            var hr = doc.createElement("hr");
            upperDiv.appendChild(hr);
            hr.style.margin = "0 0 0 0";

            lowerObscured = doc.createElement("div");
            body.appendChild(lowerObscured);
            lowerObscured.style.height = upperWin.offsetHeight;

            var lowerDiv = doc.createElement("div");
            body.appendChild(lowerDiv);
            lowerWin = lowerDiv;

            function onresize() {
                screenWidth = iframe.offsetWidth;
                screenHeight = iframe.offsetHeight;
                upperWinResizer.width = screenWidth;

                var sizer = doc.createElement("div");
                sizer.style.position = "absolute";
                upperWin.appendChild(sizer);
                sizer.innerHTML = "0";
                screenCols = Math.floor(screenWidth/sizer.offsetWidth);
                screenRows = Math.floor(screenHeight/sizer.offsetHeight);
                upperWin.removeChild(sizer);

                if (notifyResize)
                    notifyResize(screenWidth, screenHeight, screenCols, screenRows);
            }
            window.onresize = onresize;
            onresize();
        })();

        var upperLines = [],
            upperX = 0,
            upperY = 0;
        function upperInit(lines) {
            upperWin.innerHTML = "";
            upperLines = [];
            for (var i = 0; i < lines; i++) {
                var line = doc.createTextNode(".");
                upperWin.appendChild(line);
                line.data = "";
                upperLines.push(line);
                upperWin.appendChild(doc.createElement("br"));
            }
            upperX = upperY = 0;
            lowerObscured.style.height = upperWin.offsetHeight;
        }

        this.splitUpper = upperInit;

        this.screenCols = function() {
            return screenCols;
        }

        this.screenRows = function() {
            return screenRows;
        }

        this.upperRows = function() {
            return upperLines.length;
        }

        function upperSetCursor(x,y) {
            upperX = x-1;
            upperY = y-1;
        }

        this.upperSetCursor = upperSetCursor;

        this.upperEraseLine = function() {
            var line = upperLines[upperY];
            if (line && line.data.length > upperX)
                line.data = line.data.substring(0, upperX);
        }

        this.upperEraseWindow = function() {
            upperInit(upperLines.length);
        }

        function upperAddTextWithoutLineBreaks(t) {
            var line = upperLines[upperY];
            if (!line)
                return;
            while (line.data.length < upperX)
                line.data += " ";
            var trailer = "";
            if (line.data.length > upperX+t.length)
                trailer = line.data.substring(upperX+t.length);
            line.data = line.data.substring(0,upperX)+t+trailer;
            upperX += t.length;
        }

        function upperAddText(t) {
            for (;;) {
                var nl = t.indexOf("\n");
                if (nl < 0)
                    return upperAddTextWithoutLineBreaks(t);
                upperAddTextWithoutLineBreaks(t.substring(0,nl));
                upperX = 0;
                upperY++;
                t = t.substring(nl+1);
            }
        }

        this.upperAddText = upperAddText;

        this.setNormalText = function() {
            textBold = false;
            textItalic = false;
            textReverse = false;
            textFixed = false;
        }

        this.setBoldText = function(bold) {
            textBold = bold;
        }

        this.setItalicText = function(italic) {
            textItalic = italic;
        }

        this.setReverseText = function(reverse) {
            textReverse = reverse;
        }

        this.setFixedText = function(fixed) {
            textFixed = fixed;
        }

        this.setFixedText2 = function(fixed) {
            textFixed2 = fixed;
        }

        this.setNotifyResize = function(notify) {
            notifyResize = notify;
        }

        this.addBR = function() {
            if (upperSelected) {
                upperX = 0;
                upperY++;
            } else {
                lowerWin.appendChild(doc.createElement("br"));
            }
        }

        function lowerAddTextWithoutLineBreaks(t) {
            var font = doc.createElement("font");
            lowerWin.appendChild(font);
            if (textBold)
                font.style.fontWeight = "bold";
            if (textItalic)
                font.style.fontStyle = "italic";
            font.style.color = textReverse ? backgroundColor: foregroundColor;
            font.style.backgroundColor = textReverse ? foregroundColor: backgroundColor;
            if (textFixed || textFixed2) {
                font.style.fontFamily = "monospace";
                font.style.whiteSpace = "pre";
            }
            font.appendChild(doc.createTextNode(t));
        }

        function lowerAddText(t) {
            for (;;) {
                var nl = t.indexOf("\n");
                if (nl < 0)
                    return lowerAddTextWithoutLineBreaks(t);
                lowerAddTextWithoutLineBreaks(t.substring(0,nl));
                lowerWin.appendChild(doc.createElement("br"));
                t = t.substring(nl+1);
            }
        }

        this.lowerAddText = lowerAddText;

        this.lowerEraseWindow = function() {
            lowerWin.innerHTML = "";
        }

        this.reset = function() {
            upperInit(0);
            lowerWin.innerHTML = "";
        }

        this.selectWindow = function(upper) {
            upperSelected = upper;
        }

        this.addText = function(t) {
            if (upperSelected)
                upperAddText(t);
            else
                lowerAddText(t);
        }

        this.setCursor = function(x,y) {
            if (upperSelected)
                upperSetCursor(x,y);
        }

        this.eraseLine = function() {
            if (upperSelected)
            upperEraseLine();
        }

        this.getCursorX = function() {
            return upperX+1;
        }

        this.getCursorY = function() {
            return upperY+1;
        }

        this.readLine = function(handler, prompt) {
            var div = doc.createElement("span");
            lowerWin.appendChild(div);
            div.style.fontFamily = "monospace";
            div.style.color = "#406080";
            var pre = doc.createElement("span");
            pre.style.whiteSpace = "pre";
            div.appendChild(pre);
            if (prompt)
                pre.appendChild(doc.createTextNode(prompt));
            div.appendChild(doc.createElement("br"));
            var input = doc.createElement("input");
            div.appendChild(input);
            input.type = "text";
            input.style.width = 0;
            input.style.height = 0;
            input.style.borderStyle = "none";
            input.autocomplete = "off";
            var text = "",
                cursor = doc.createElement("span");
            pre.appendChild(cursor);
            cursor.style.backgroundColor = "orange";
            cursor.innerHTML = "&nbsp;";
            var line = null;
            input.onkeypress = function(event) {
                if (!event)
                    event = window.event || document.frames[iframe.name].event;
                var key = event.keyCode || event.which;
                if (!line) {
                    line = doc.createTextNode(".");
                    pre.insertBefore(line, cursor);
                }
                if (event.ctrlKey || (event.modifiers&2))
                    key &= 31;
                if (key <= 126 && key >= 32) {
                    text += String.fromCharCode(key);
                } else if (key == 8 || key == 127) {
                    if (text.length > 0)
                        text = text.substring(0, text.length-1);
                } else if (key == 10 || key == 13) {
                    line.data = text;
                    cursor.innerHTML = "";
                    div.removeChild(input);
                    doc.onclick = null;
                    handler(text);
                    return;
                } else if (key == 21) {
                    text = "";
                } else if (key == 23) {
                    while (text.length > 0 && text.charAt(text.length-1) == " ")
                        text = text.substring(0, text.length-1);
                    while (text.length > 0 && text.charAt(text.length-1) != " ")
                        text = text.substring(0, text.length-1);
                }
                line.data = text;
                cursor.innerHTML = "&nbsp;";
            }
            ZScheduler(doc.onclick = function() { input.focus(); });
        }

        this.readKey = function(handler, prompt) {
            var elt = doc.createElement("div");
            lowerWin.appendChild(elt);
            if (prompt) {
                var font = doc.createElement("font");
                elt.appendChild(font);
                font.style.backgroundColor = "orange";
                font.appendChild(doc.createTextNode(prompt));
            }
            var input = doc.createElement("input");
            input.type = "text";
            elt.appendChild(input);
            input.style.width = 0;
            input.style.height = 0;
            input.style.borderStyle = "none";
            input.onkeypress = function(event) {
                if (!event)
                    event = window.event || document.frames[iframe.name].event;
                var key = event.keyCode || event.which;
                doc.onclick = null;
                lowerWin.removeChild(elt);
                handler(key);
            }
            ZScheduler(doc.onclick = function() { input.focus(); });
        }

        function appendDiagnostic(t,color) {
            var font = doc.createElement("font");
            lowerWin.appendChild(doc.createElement("br"));
            lowerWin.appendChild(font);
            font.style.fontFamily = "monospace";
            font.style.color = color;
            font.appendChild(doc.createTextNode(t));
        }

        this.debug = function (t) {
            appendDiagnostic(t,"#804020");
        }

        this.errorMsg = function(t) {
            appendDiagnostic(t,"#ff0000");
        }

        function lowerAddHtml(html) {
            var span = doc.createElement("span");
            lowerWin.appendChild(span);
            span.innerHTML = html;
            return span;
        }

        this.lowerAddHtml = lowerAddHtml;
    }

    var zdata,
        zmemory,
        stackFrame = { pc:0, locals:[], stack:[], argc:0, ret:null },
        stack = [],
        stream3 = [],
        addrFactor,
        zscreen = new ZScreen(iframe),
        charTable, invCharTable, extraCharTable,
        saveData = null,
        undoData = null,
        brk,
        debugTrace = false;

    function init() {
        zmemory = [];
        zscreen.reset();
        //zscreen.debug("Version: " + zdata[0]);
        switch (zdata[0]) {
        case 5: addrFactor = 4; break;
        case 8: addrFactor = 8; break;
        default:
            zscreen.errorMsg("Unknown version: " + zdata[0]);
            return;
        }
        //zscreen.debug("Length: " + zdata.length + "," + getWord(26)*addrFactor);
        var dataLength = getWord(26)*addrFactor;
        if (dataLength > zdata.length) {
            zscreen.errorMsg("Invalid file: bad length: " + dataLength);
            return;
        }
        var sum = 0;
        for (var i = 64; i < dataLength; i++)
            sum = (sum+zdata[i])%65536;
        //zscreen.debug("Checksum: " + sum + "," + getWord(28));
        if (sum != getWord(28)) {
            zscreen.errorMsg("Invalid file: bad checksum: " + getWord(28));
            return;
        }
        //zscreen.debug("Dynamic size: " + getWord(14));
        //zscreen.debug("High memory: " + getWord(4));
        //zscreen.debug("Initial PC: " + getWord(6));
        brk = getWord(14);
        zmemory = zdata.slice(0, brk);

        // init header values
        setByte(1,28); // flags
        setByte(16,getByte(16)&2); // flags
        setByte(30,1); // interp number
        setByte(31,65); // interp version
        setByte(32,zscreen.screenRows());
        setByte(33,zscreen.screenCols());
        setWord(34,zscreen.screenCols());
        setWord(36,zscreen.screenRows());
        setByte(38,1);
        setByte(39,1);
        setByte(44,0);
        setByte(45,1);
        setByte(50,1);
        setByte(41,0);
        zscreen.setNotifyResize(function() {
            setByte(32,zscreen.screenRows());
            setByte(33,zscreen.screenCols());
            setWord(34,zscreen.screenCols());
            setWord(36,zscreen.screenRows());
        });

        extraCharTable = {
            155:0x0e4, 156:0x0f6, 157:0x0fc, 158:0x0c4, 159:0x0d6, 160:0x0dc,
            161:0x0df, 162:0x0bb, 163:0x0ab, 164:0x0eb, 165:0x0ef, 166:0x0ff,
            167:0x0cb, 168:0x0cf, 169:0x0e1, 170:0x0e9, 171:0x0ed, 172:0x0f3,
            173:0x0fa, 174:0x0fd, 175:0x0c1, 176:0x0c9, 177:0x0cd, 178:0x0d3,
            179:0x0da, 180:0x0dd, 181:0x0e0, 182:0x0e8, 183:0x0ec, 184:0x0f2,
            185:0x0f9, 186:0x0c0, 187:0x0c8, 188:0x0cc, 189:0x0d2, 190:0x0d9,
            191:0x0e2, 192:0x0ea, 193:0x0ee, 194:0x0f4, 195:0x0fb, 196:0x0c2,
            197:0x0ca, 198:0x0ce, 199:0x0d4, 200:0x0db, 201:0x0e5, 202:0x0c5,
            203:0x0f8, 204:0x0d8, 205:0x0e3, 206:0x0f1, 207:0x0f5, 208:0x0c3,
            209:0x0d1, 210:0x0d5, 211:0x0e6, 212:0x0c6, 213:0x0e7, 214:0x0c7,
            215:0x0fe, 216:0x0f0, 217:0x0de, 218:0x0d0, 219:0x0a3, 220:0x153,
            221:0x152, 222:0x0a1, 223:0x0bf,
            224:63, 225:63, 226:63, 227:63, 228:63, 229:63, 230:63, 231:63,
            232:63, 233:63, 234:63, 235:63, 236:63, 237:63, 238:63, 239:63,
            240:63, 241:63, 242:63, 243:63, 244:63, 245:63, 246:63, 247:63,
            248:63, 249:63, 250:63, 251:63
        };
        // FIXME: specified table

        charTable = [["a","b","c","d","e","f","g","h","i","j","k","l","m",
                      "n","o","p","q","r","s","t","u","v","w","x","y","z"],
                     ["A","B","C","D","E","F","G","H","I","J","K","L","M",
                      "N","O","P","Q","R","S","T","U","V","W","X","Y","Z"],
                     [null,"\n","0","1","2","3","4","5","6","7","8","9",".",
                      ",","!","?","_","#","'","\"","/","\\","-",":","(",")"]];
        if (getWord(52)) {
            for (var i = 0; i < 26; i++) {
                charTable[0][i] = String.fromCharCode(getByte(getWord(52)+i));
                charTable[1][i] = String.fromCharCode(getByte(getWord(52)+26+i));
                if (i > 1)
                    charTable[2][i] = String.fromCharCode(getByte(getWord(52)+52+i));
            }
        }
        invCharTable = {};
        for (var i = 0; i < 26; i++) {
            invCharTable[charTable[0][i].charCodeAt(0)] = i+6;
            invCharTable[charTable[1][i].charCodeAt(0)] = 32+i+6;
            if (i > 0)
                invCharTable[charTable[2][i].charCodeAt(0)] = 64+i+6;
        }

        stackFrame.pc = getWord(6);
    }

    function interpreterLoop() {
        //var debugMsec = new Date().getTime();
        var res, ticks = 32768;
        while (--ticks && (res = executeInsn()) == true)
            ;
        //zscreen.debug("ticks="+(32768-ticks)+",msec="+(new Date().getTime()-debugMsec));
        if (!ticks) {
            ZScheduler(interpreterLoop);
            return;
        }

        if (res == -1)
            return;

        if (false) {
            var elt = zscreen.lowerAddHtml("<br>restart");
            elt.title = "Click to restart";
            elt.onclick = function() {
                init();
                debugTrace = false;
                interpreterLoop();
            }
            elt = zscreen.lowerAddHtml("<br>restart with tracing");
            elt.title = "Click to restart";
            elt.onclick = function() {
                init();
                debugTrace = true;
                interpreterLoop();
            }
        }
    }

    function toHexString(num) {
        var str = "";
        for (;;) {
            str = ["0","1","2","3","4","5","6","7","8","9","a","b","c","d","e","f"][num%16] + str;
            if (num < 16)
                return str;
            num = num>>>4;
        }
    }

    function getByte(addr) {
        if (addr < brk)
            return zmemory[addr];
        else if (addr < zdata.length)
            return zdata[addr];
        else {
            var msg = "Invalid memory read byte: " + addr;
            throw new Error(msg,msg);
        }
    }

    function getWord(addr) {
        if (addr+1 < brk)
            return zmemory[addr]*256 + zmemory[addr+1];
        else if (addr+1 < zdata.length)
            return zdata[addr]*256 + zdata[addr+1];
        else {
            var msg = "Invalid memory read word: " + addr;
            throw new Error(msg,msg);
        }
    }

    function setByte(addr,value) {
        if (addr < brk && value < 256) {
            zmemory[addr] = value;
        } else {
            var msg = "Invalid memory write byte: " + addr + "," + value;
            throw new Error(msg,msg);
        }
    }

    function setWord(addr,value) {
        if (addr < brk-1 && value < 65536) {
            zmemory[addr] = value>>8;
            zmemory[addr+1] = value&255;
        } else {
            var msg = "Invalid memory write word: " + addr + "," + value;
            throw new Error(msg,msg);
        }
    }

    function zopSave() {
        var data = {
            zmemory:zmemory,
            stackFrame:stackFrame, stack:stack, stream3:stream3
        };
        zopRestore(data);
        return data;
    }

    function zopRestore(data) {
        var i;
        if (!data)
            return false;
        zmemory = data.zmemory.slice(0,data.zmemory.length);
        stackFrame = zopRestoreStackFrame(data.stackFrame);
        stack = data.stack.slice(0,data.stack.length);
        for (i in stack)
            stack[i] = zopRestoreStackFrame(stack[i]);
        stream3 = data.stream3.slice(0,data.stream3.length);
        return true;
    }

    function zopRestoreStackFrame(savedFrame) {
        return {
            pc:savedFrame.pc,
            locals:savedFrame.locals.slice(0,savedFrame.locals.length),
            stack:savedFrame.stack.slice(0,savedFrame.stack.length),
            argc:savedFrame.argc,
            ret:savedFrame.ret
        };
    }

    function getString(addr) {
        var string = "",
            state = 0, z;
        function appendCode(code) {
            if (state == 1 || state == 2 || state == 3) {
                string += getString(2*getWord(getWord(24)+2*((state-1)*32+code)));
                state = 0;
            } else if (state == 6) {
                z = code<<5;
                state = 7;
            } else if (state == 7) {
                z |= code;
                if (z == 13)
                    string += "\n";
                else if (z >= 32 && z <= 126)
                    string += String.fromCharCode(z);
                else if (z >= 155 && z <= 251)
                    string += String.fromCharCode(extraCharTable[z]);
                state = 0;
            } else if (code < 6) {
                state = code;
                if (code == 0)
                    string += " ";
            } else if (code == 6 && state == 5) {
                state = 6;
            } else {
                if (state == 4)
                    string += charTable[1][code-6];
                else if (state == 5)
                    string += charTable[2][code-6];
                else
                    string += charTable[0][code-6];
                state = 0;
            }
        }
        for (;;) {
            var word = getWord(addr);
            addr += 2;
            appendCode((word>>10)%32);
            appendCode((word>>5)%32);
            appendCode(word%32);
            if (word&32768)
                break;
        }
        return string;
    }

    function addStream3Buffer(buffer, ch) {
        var index = getWord(buffer);
        setByte(buffer+index+2,ch);
        setWord(buffer,index+1);
    }

    function zopPrintRet() {
        if (stream3.length == 0) {
            zscreen.addBR();
            return true;
        }
        addStream3Buffer(stream3[stream3.length-1], 13);
        return true;
    }

    function zopPrintStream(text) {
        if (stream3.length == 0) {
            zscreen.setFixedText2(getByte(17)&2);
            zscreen.addText(text);
            return true;
        }
        var buffer = stream3[stream3.length-1];
        for (var i = 0; i < text.length; i++) {
            var c = text.charCodeAt(i);
            if (c == 10 || c == 13)
                c == 13;
            else if (c < 32 || (c > 126 && c < 155) || c > 251)
                continue;
            addStream3Buffer(buffer, c);
        }
        return true;
    }

    function zopPrintChar(ch) {
        if (ch == 13)
            zopPrintRet();
        else if (ch >= 32 && ch <= 126)
            zopPrintStream(String.fromCharCode(ch));
        else if (ch >= 155 && ch <= 251)
            zopPrintStream(String.fromCharCode(extraCharTable[ch]));
    }

    function wordToSigned(val) {
        return val < 32768 ? val : val%65536 - 65536;
    }

    function byteToSigned(val) {
        return val < 128 ? val : val%256 - 256;
    }

    function pushStackFrame(args,ret) {
        stack.push(stackFrame);
        stackFrame = {
            pc:args[0]*addrFactor, locals:[], stack:[],
            argc:args.length-1, ret:ret
        };
        if (stackFrame.pc == 0)
            return zopReturn(0);
        var locals = getPcByte(),
            n = locals;
        if (n)
            do {
                stackFrame.locals.push(0);
            } while (--n);
        n = Math.min(args.length-1,stackFrame.locals.length);
        if (n)
            do {
                stackFrame.locals[n-1] = args[n];
            } while (--n);
        return true;
    }

    function getPcByte() {
        var b = getByte(stackFrame.pc);
        stackFrame.pc++;
        return b;
    }

    function getPcWord() {
        var w = getWord(stackFrame.pc);
        stackFrame.pc += 2;
        return w;
    }

    function getPcVariable() {
        return getZopVariable(getPcByte());
    }

    function getZopVariable(code) {
        if (code == 0)
            return stackFrame.stack.pop();
        else if (code < 16)
            return stackFrame.locals[code-1];
        return getGlobal(code-16);
    }

    function getPcString() {
        var s = getString(stackFrame.pc);
        for (;;) {
            var w = getWord(stackFrame.pc);
            stackFrame.pc += 2;
            if (w&32768)
                break;
        }
        return s;
    }

    function getObjectAddr(number) {
        if (number == 0)
            return 0;
        return getWord(10)+14*(number-1)+2*63;
    }

    function removeObject(number) {
        var objAddr = getObjectAddr(number);
        if (objAddr == 0)
            return;
        var parent = getWord(objAddr+6);
        if (parent == 0)
            return;
        var parentAddr = getObjectAddr(parent),
            o = getWord(parentAddr+10);
        if (o == number) {
            setWord(parentAddr+10,getWord(objAddr+8));
        } else {
            for (;;) {
                var oAddr = getObjectAddr(o);
                o = getWord(oAddr+8);
                if (number == getWord(oAddr+8)) {
                    setWord(oAddr+8,getWord(objAddr+8));
                    break;
                }
            }
        }
        setWord(objAddr+6,0);
        setWord(objAddr+8,0);
    }

    function insertObject(number, parent) {
        removeObject(number);
        var objAddr = getObjectAddr(number),
            parentAddr = getObjectAddr(parent);
        if (objAddr == 0 || parentAddr == 0)
            return;
        setWord(objAddr+6,parent);
        setWord(objAddr+8,getWord(parentAddr+10));
        setWord(parentAddr+10,number);
    }

    function getDefaultProp(prop) {
        return getWord(getWord(10)+2*(prop-1));
    }

    function getPropAddr(objAddr, prop) {
        var props = getWord(objAddr+12);
        props += getByte(props)*2+1;
        for (;;) {
            var size = getByte(props);
            //zscreen.debug("getPropAddr:size="+size+",num="+(size&63)+",addr="+props);
            if ((size&63) < prop)
                return 0;
            if (prop == (size&63))
                return props;
            if (size&128) {
                size = getByte(props+1)&63;
                if (size == 0)
                    size = 64;
                props += size+2;
            } else {
                props += (size&64) ? 3 : 2;
            }
        }
    }

    function getPropSize(propAddr) {
        var size = getByte(propAddr);
        if (size&128) {
            size = getByte(propAddr+1)&63;
            return size || 64;
        } else {
            return (size&64) ? 2 : 1;
        }
    }

    function makeTextEncoder() {
        var encoder = {
            state:0,
            word1:0, word2:0, word3:32768,
            pushCode:function(code) {
                if (encoder.state >= 9)
                    return;
                switch (encoder.state++) {
                case 0:
                    word1 = code<<10;
                    word2 = 0;
                    word3 = 32768;
                    break;
                case 1: word1 |= code<<5; break;
                case 2: word1 |= code; break;
                case 3: word2 |= code<<10; break;
                case 4: word2 |= code<<5; break;
                case 5: word2 |= code; break;
                case 6: word3 |= code<<10; break;
                case 7: word3 |= code<<5; break;
                case 8: word3 |= code; break;
                }
            },
            padToEnd:function() {
                while (encoder.state < 9)
                    encoder.pushCode(5);
            }
        }
        return encoder;
    }

    function tokenize(text, parseResult, dict, skipUnknown) {
        dict = dict || getWord(8);
        var textLen = getByte(text+1),
            maxTokens = getByte(parseResult),
            wordSeparators = {};
        //zscreen.debug("dict="+dict+",textLen="+textLen+",maxTokens="+maxTokens+",dict separator count="+getByte(dict));
        for (var i = 0; i < getByte(dict); i++)
            wordSeparators[getByte(dict+1+i)] = true;
        dict += getByte(dict)+1;
        var entrySize = getByte(dict),
            entryCount = wordToSigned(getWord(dict+1)),
            unsorted = entryCount < 0;
        //zscreen.debug("dict entrySize="+entrySize+",entryCount="+entryCount);
        if (unsorted)
            entryCount = -entryCount;
        dict += 3;

        var resultIndex = 0,
            wordState = 0,
            wordWord1, wordWord2, wordWord3, wordLetters, wordIndex;

        function pushWordEncoding(code) {
            switch (wordState++) {
            case 0:
                wordWord1 = code<<10;
                wordWord2 = 0;
                wordWord3 = 32768;
                break;
            case 1: wordWord1 |= code<<5; break;
            case 2: wordWord1 |= code; break;
            case 3: wordWord2 |= code<<10; break;
            case 4: wordWord2 |= code<<5; break;
            case 5: wordWord2 |= code; break;
            case 6: wordWord3 |= code<<10; break;
            case 7: wordWord3 |= code<<5; break;
            case 8: wordWord3 |= code; break;
            }
        }
            
        function finishWord() {
            if (wordState == 0 || resultIndex >= maxTokens)
                return;
            while (wordState < 9)
                pushWordEncoding(5);
            wordState = 0;
            if (unsorted) {
                for (var i = 0; i < entryCount; i++)
                    if (wordWord1 == getWord(dict+i*entrySize)
                        && wordWord2 == getWord(dict+2+i*entrySize)
                        && wordWord3 == getWord(dict+4+i*entrySize)) {
                        setWord(parseResult+2+resultIndex*4,dict+i*entrySize);
                        setByte(parseResult+4+resultIndex*4,wordLetters);
                        setByte(parseResult+5+resultIndex*4,wordIndex);
                        resultIndex++;
                        //zscreen.debug("tokenize:dictionary word:i="+i+","+resultIndex+",index="+wordIndex+",letters="+wordLetters);
                        return;
                    }
            } else {
                var l = 0, h = entryCount, i, w1, w2, w3;
                while (l < h) {
                    i = (l+h)>>1;
                    if (wordWord1 < (w1 = getWord(dict+i*entrySize))) {
                        h = i;
                        continue;
                    } else if (wordWord1 == w1) {
                        if (wordWord2 < (w2 = getWord(dict+2+i*entrySize))) {
                            h = i;
                            continue;
                        } else if (wordWord2 == w2) {
                            if (wordWord3 < (w3 = getWord(dict+4+i*entrySize))) {
                                h = i;
                                continue;
                            } else if (wordWord3 == w3) {
                                setWord(parseResult+2+resultIndex*4,dict+i*entrySize);
                                setByte(parseResult+4+resultIndex*4,wordLetters);
                                setByte(parseResult+5+resultIndex*4,wordIndex);
                                resultIndex++;
                                //zscreen.debug("tokenize:dictionary word:i="+i+","+resultIndex+",index="+wordIndex+",letters="+wordLetters);
                                return;
                            }
                        }
                    }
                    if (l < i)
                        l = i;
                    else
                        break;
                }
            }
            if (!skipUnknown) {
                setWord(parseResult+2+resultIndex*4,0);
                setByte(parseResult+4+resultIndex*4,wordLetters);
                setByte(parseResult+5+resultIndex*4,wordIndex);
            }
            resultIndex++;
            //zscreen.debug("tokenize:unknown word:i="+resultIndex+",index="+wordIndex+",letters="+wordLetters);
        }

        for (var i = 0; i < textLen; i++) {
            var ch = getByte(text+2+i);
            //zscreen.debug("i="+i+",ch="+ch);
            if (ch == 32) {
                finishWord();
            } else if (wordSeparators[ch]) {
                finishWord();
                if (resultIndex < maxTokens) {
                    setWord(parseResult+2+resultIndex*4,0);
                    setWord(parseResult+4+resultIndex*4,1);
                    setWord(parseResult+5+resultIndex*4,i+2);
                    resultIndex++;
                    //zscreen.debug("tokenize:separator:i="+resultIndex+",index="+wordIndex);
                }
            } else {
                if (wordState == 0) {
                    wordLetters = 0;
                    wordIndex = i+2;
                }
                wordLetters++;
                var code = invCharTable[ch];
                if (!code) {
                    pushWordEncoding(5);
                    pushWordEncoding(6);
                    pushWordEncoding((ch>>5)&31);
                    pushWordEncoding(ch&31);
                } else if (code >= 64) {
                    pushWordEncoding(5);
                    pushWordEncoding(code&31);
                } else if (code >= 32) {
                    pushWordEncoding(4);
                    pushWordEncoding(code&31);
                } else {
                    pushWordEncoding(code&31);
                }
            }
            if (resultIndex >= maxTokens)
                break;
        }
        finishWord();
        setByte(parseResult+1,resultIndex);
    }

    function pushVariableOperand(typeCode, operands) {
        switch (typeCode%4) {
        case 0:
            operands.push(getPcWord());
            break;
        case 1:
            operands.push(getPcByte());
            break;
        case 2:
            operands.push(getPcVariable());
            break;
        }
    }

    function getVariableOperands() {
        var types = getPcByte(),
            operands = [];
        pushVariableOperand(types>>6, operands);
        pushVariableOperand(types>>4, operands);
        pushVariableOperand(types>>2, operands);
        pushVariableOperand(types, operands);
        return operands;
    }

    function getGlobal(i) {
        return getWord(getWord(12) + i*2);
    }

    function setGlobal(i,val) {
        setWord(getWord(12) + i*2, val);
    }

    function getVariableOperands2() {
        var types = getPcWord(),
            operands = [];
        pushVariableOperand(types>>14, operands);
        pushVariableOperand(types>>12, operands);
        pushVariableOperand(types>>10, operands);
        pushVariableOperand(types>>8, operands);
        pushVariableOperand(types>>6, operands);
        pushVariableOperand(types>>4, operands);
        pushVariableOperand(types>>2, operands);
        pushVariableOperand(types, operands);
        return operands;
    }

    function zopFatalError(msg) {
        zscreen.errorMsg("Fatal error [pc=" + stackFrame.pc + "]: " + msg);
        zscreen.errorMsg("Stack trace:");
        for (var i = stack.length-1; i >= 0; i--)
            zscreen.errorMsg("...returning to pc=" + stack[i].pc);
        return false;
    }

    function zopResult(val) {
        return zopSetResult(getPcByte(),val);
    }

    function zopSetResult(code,val) {
        //zscreen.debug("result:"+code+","+val);
        if (code == 0)
            stackFrame.stack.push(val);
        else if (code < 16)
            stackFrame.locals[code-1] = val;
        else
            setGlobal(code-16,val);
        return true;
    }

    function zopReturn(val) {
        var ret = stackFrame.ret;
        stackFrame = stack.pop();
        if (ret == null)
            return true;
        else
            return zopSetResult(ret,val);
    }

    function zopPcBranch(test, conditional) {
        //zscreen.debug("branch:pc="+stackFrame.pc+",t="+test+",cond="+conditional);
        var b = getPcByte();
        if (b&128)
            test = !test;
        if (b&64) {
            b = b&63;
        } else {
            b = (b&63)<<8 | getPcByte();
            if (b >= 8192)
                b -= 16384;
        }
        //zscreen.debug("branch:pc="+stackFrame.pc+",offset="+b);
        if (test && conditional)
            return true;
        if (b == 0)
            return zopReturn(0);
        if (b == 1)
            return zopReturn(1);
        stackFrame.pc += b-2;
        return true;
    }

    function zop_je(operands) {
        //zscreen.debug("zop je:"+operands);
        for (var i = 1; i < operands.length; i++)
            if (operands[0] == operands[i])
                return zopPcBranch(true,true);
        return zopPcBranch(false,true);
    }

    function zop_jl(operands) {
        //zscreen.debug("zop jl:"+operands);
        return zopPcBranch(wordToSigned(operands[0]) < wordToSigned(operands[1]), true);
    }

    function zop_jg(operands) {
        //zscreen.debug("zop jg:"+operands);
        return zopPcBranch(wordToSigned(operands[0]) > wordToSigned(operands[1]), true);
    }

    function zop_dec_chk(operands) {
        //zscreen.debug("zop dec_chk:"+operands);
        var val = getZopVariable(operands[0]);
        zopSetResult(operands[0],(--val)&65535);
        return zopPcBranch(wordToSigned(val) < wordToSigned(operands[1]), true);
    }

    function zop_inc_chk(operands) {
        //zscreen.debug("zop inc_chk:"+operands);
        var val = getZopVariable(operands[0]);
        zopSetResult(operands[0],(++val)&65535);
        return zopPcBranch(wordToSigned(val) > wordToSigned(operands[1]), true);
    }

    function zop_jin(operands) {
        //zscreen.debug("zop jin:"+operands);
        var obj = getObjectAddr(operands[0]);
        if (obj == 0)
            return zopPcBranch(false,true);
        return zopPcBranch(getWord(obj+6) == operands[1],true);
    }

    function zop_test(operands) {
        //zscreen.debug("zop test:"+operands);
        return zopPcBranch((operands[0]&operands[1]) == operands[1],true);
    }

    function zop_or(operands) {
        //zscreen.debug("zop or:"+operands);
        return zopResult(operands[0]|operands[1]);
    }

    function zop_and(operands) {
        //zscreen.debug("zop and:"+operands);
        return zopResult(operands[0]&operands[1]);
    }

    function zop_test_attr(operands) {
        //zscreen.debug("zop test_attr:"+operands);
        var obj = getObjectAddr(operands[0]),
            attr = operands[1];
        obj += attr>>3;
        attr = 128 >> (attr&7);
        return zopPcBranch((getByte(obj)&attr) != 0,true);
    }

    function zop_set_attr(operands) {
        //zscreen.debug("zop set_attr:"+operands);
        var obj = getObjectAddr(operands[0]),
            attr = operands[1];
        obj += attr>>3;
        attr = 128 >> (attr&7);
        setByte(obj,getByte(obj)|attr);
        return true;
    }

    function zop_clear_attr(operands) {
        //zscreen.debug("zop clear_attr:"+operands);
        var obj = getObjectAddr(operands[0]),
            attr = operands[1];
        obj += attr>>3;
        attr = 128 >> (attr&7);
        setByte(obj,getByte(obj)&(~attr));
        return true;
    }

    function zop_store(operands) {
        //zscreen.debug("zop store:"+operands);
        return zopSetResult(operands[0],operands[1]);
    }

    function zop_insert_obj(operands) {
        //zscreen.debug("zop insert_obj:"+operands);
        insertObject(operands[0],operands[1]);
        return true;
    }

    function zop_loadw(operands) {
        //zscreen.debug("zop loadw:"+operands);
        return zopResult(getWord(operands[0]+2*operands[1]));
    }

    function zop_loadb(operands) {
        //zscreen.debug("zop loadb:"+operands);
        return zopResult(getByte(operands[0]+operands[1]));
   }

    function zop_get_prop(operands) {
        //zscreen.debug("zop get_prop:"+operands);
        var obj = getObjectAddr(operands[0]);
        if (obj == 0)
            return zopFatalError("get_prop: Object 0");
        var propAddr = getPropAddr(obj, operands[1]);
        if (propAddr == 0)
            return zopResult(getDefaultProp(operands[1]));
        var size = getPropSize(propAddr);
        propAddr += (getByte(propAddr)&128) ? 2 : 1;
        if (size == 1)
            return zopResult(getByte(propAddr));
        else
            return zopResult(getWord(propAddr));
    }

    function zop_get_prop_addr(operands) {
        //zscreen.debug("zop get_prop_addr:"+operands);
        var obj = getObjectAddr(operands[0]);
        if (obj == 0)
            return zopResult(0);
        var prop = getPropAddr(obj, operands[1]);
        return zopResult(prop && prop + ((getByte(prop)&128) ? 2 : 1));
    }

    function zop_get_next_prop(operands) {
        //zscreen.debug("zop get_next_prop:"+operands);
        var obj = getObjectAddr(operands[0]);
        if (obj == 0)
            return zopFatalError("get_prop_addr: Object 0");
        if (operands[1] == 0) {
            var prop = getWord(obj+12);
            prop += getByte(prop)*2+1;
            return zopResult(getByte(prop)&63);
        }
        var prop = getPropAddr(obj, operands[1]);
        if (prop == 0)
            return zopFatalError("get_prop_addr: Invalid prop: Object " + operands[0] + " prop " + operands[1]);
        prop += getPropSize(prop) + ((getByte(prop)&128) ? 2 : 1);
        return zopResult(getByte(prop)&63);
    }

    function zop_add(operands) {
        //zscreen.debug("zop add:"+operands);
        return zopResult((operands[0] + operands[1])&65535);
    }

    function zop_sub(operands) {
        //zscreen.debug("zop sub:"+operands);
        return zopResult((operands[0] - operands[1])&65535);
    }

    function zop_mul(operands) {
        //zscreen.debug("zop mul:"+operands);
        return zopResult((wordToSigned(operands[0])*wordToSigned(operands[1]))&65535);
    }

    function zop_div(operands) {
        //zscreen.debug("zop div:"+operands);
        if (operands[1] == 0)
            return zopFatalError("div: Division by zero");
        return zopResult(Math.floor(wordToSigned(operands[0])/wordToSigned(operands[1]))&65535);
    }

    function zop_mod(operands) {
        //zscreen.debug("zop mod:"+operands);
        if (operands[1] == 0)
            return zopFatalError("mod: Division by zero");
        return zopResult(Math.floor(wordToSigned(operands[0])%wordToSigned(operands[1]))&65535);
    }

    function zop_call_2s(operands) {
        //zscreen.debug("zop call_2s:"+operands);
        return pushStackFrame(operands,getPcByte());
    }

    function zop_call_2n(operands) {
        //zscreen.debug("zop call_2n:"+operands);
        return pushStackFrame(operands,null);
    }

    function zop_set_colour(operands) {
        //zscreen.debug("zop set_colour:"+operands);
        return true;
    }

    function zop_throw(operands) {
        //zscreen.debug("zop throw:"+operands);
        if (operands[1] < stack.length)
            return zopFatalError("throw: No previous catch");
        while (stack.length > operands[1])
            stackFrame = stack.pop();
        return zopReturn(operands[0]);
    }

    function zop_jz(operand) {
        //zscreen.debug("zop jz:"+operand);
        return zopPcBranch(operand == 0, true);
    }

    function zop_get_sibling(operand) {
        //zscreen.debug("zop get_sibling:"+operand);
        var obj = getObjectAddr(operand),
            result = obj && getWord(obj+8);
        zopResult(result);
        return zopPcBranch(result != 0, true);
    }

    function zop_get_child(operand) {
        //zscreen.debug("zop get_child:"+operand);
        var obj = getObjectAddr(operand),
            result = obj && getWord(obj+10);
        zopResult(result);
        return zopPcBranch(result != 0, true);
    }

    function zop_get_parent(operand) {
        //zscreen.debug("zop get_parent:"+operand);
        var obj = getObjectAddr(operand);
        return zopResult(obj && getWord(obj+6));
    }

    function zop_get_prop_len(operand) {
        //zscreen.debug("zop get_prop_len:"+operand);
        return zopResult(getPropSize(operand - ((getByte(operand-1)&128) ? 2 : 1)));
    }

    function zop_inc(operand) {
        //zscreen.debug("zop inc:"+operand);
        return zopSetResult(operand,(getZopVariable(operand)+1)&65535);
    }

    function zop_dec(operand) {
        //zscreen.debug("zop dec:"+operand);
        return zopSetResult(operand,(getZopVariable(operand)-1)&65535);
    }

    function zop_print_addr(operand) {
        //zscreen.debug("zop print_addr:"+operand);
        return zopPrintStream(getString(operand));
    }

    function zop_call_1s(operand) {
        //zscreen.debug("zop call_1s:"+operand);
        return pushStackFrame([operand],getPcByte());
    }

    function zop_remove_obj(operand) {
        //zscreen.debug("zop remove_obj:"+operand);
        removeObject(operand);
        return true;
    }

    function zop_print_obj(operand) {
        //zscreen.debug("zop print_obj:"+operand);
        return zopPrintStream(getString(getWord(getObjectAddr(operand)+12)+1));
    }

    function zop_ret(operand) {
        //zscreen.debug("zop ret:"+operand);
        return zopReturn(operand);
    }

    function zop_jump(operand) {
        //zscreen.debug("zop jump:"+operand);
        stackFrame.pc += wordToSigned(operand)-2;
        return true;
    }

    function zop_print_paddr(operand) {
        //zscreen.debug("zop print_paddr:"+operand);
        return zopPrintStream(getString(operand*addrFactor));
    }

    function zop_load(operand) {
        //zscreen.debug("zop load:"+operand);
        if (operand == 0)
            return zopResult(stackFrame.stack.pop());
        if (operand < 16)
            return zopResult(stackFrame.locals[operand-1]);
        return zopResult(getGlobal(operand-16));
    }

    function zop_call_1n(operand) {
        //zscreen.debug("zop call_1n:"+operand);
        return pushStackFrame([operand],null);
    }

    function zop_rtrue() {
        //zscreen.debug("zop rtrue");
        return zopReturn(1);
    }

    function zop_rfalse() {
        //zscreen.debug("zop rfalse");
        return zopReturn(0);
    }

    function zop_print() {
        //zscreen.debug("zop print");
        return zopPrintStream(getPcString());
    }

    function zop_print_ret() {
        //zscreen.debug("zop print_ret");
        zopPrintStream(getPcString());
        zopPrintRet();
        return zopReturn(1);
    }

    function zop_nop() {
        //zscreen.debug("zop nop");
        return true;
    }

    function zop_restart() {
        //zscreen.debug("zop restart");
        init();
        return true;
    }

    function zop_ret_popped() {
        //zscreen.debug("zop ret_popped");
        return zopReturn(stackFrame.stack.pop());
    }

    function zop_catch() {
        //zscreen.debug("zop catch");
        return zopResult(stack.length);
    }

    function zop_quit() {
        //zscreen.debug("zop quit");
        zscreen.errorMsg("Game finished.");
        return false;
    }

    function zop_new_line() {
        //zscreen.debug("zop new_line");
        return zopPrintRet();
    }

    function zop_verify() {
        //zscreen.debug("zop verify");
        return zopPcBranch(true, false);
    }

    function zop_extended() {
        //zscreen.debug("zop extended");
        return zopFatalError("extended: not implemented: this should never happen");
    }

    function zop_piracy() {
        //zscreen.debug("zop piracy");
        return zopPcBranch(true, false);
    }

    function zop_call_vs(operands) {
        //zscreen.debug("zop call_vs:"+operands);
        return pushStackFrame(operands,getPcByte());
    }

    function zop_storew(operands) {
        //zscreen.debug("zop storew:"+operands);
        setWord(operands[0]+2*operands[1],operands[2]);
        return true;
    }

    function zop_storeb(operands) {
        //zscreen.debug("zop storeb:"+operands);
        setByte(operands[0]+operands[1],operands[2]);
        return true;
    }

    function zop_put_prop(operands) {
        //zscreen.debug("zop put_prop:"+operands);
        var obj = getObjectAddr(operands[0]);
        if (obj == 0)
            return zopFatalError("put_prop: Object 0");
        var propAddr = getPropAddr(obj, operands[1]);
        var size = getPropSize(propAddr);
        propAddr += (getByte(propAddr)&128) ? 2 : 1;
        if (size == 1)
            setByte(propAddr,operands[2]&255);
        else
            setWord(propAddr,operands[2]);
        return true;
    }

    function zop_aread(operands) {
        //zscreen.debug("zop aread:"+operands);
        zscreen.readLine(function(text) {
            var len = Math.min(text.length, getByte(operands[0]));
            setByte(operands[0]+1,len);
            for (var i = 0; i < len; i++)
                setByte(operands[0]+2+i,text.charCodeAt(i)&255);
            if (operands[1])
                tokenize(operands[0],operands[1],0,0);
            zopResult(10);
            interpreterLoop(); //ZScheduler(interpreterLoop);
        }, null);
        return -1;
    }

    function zop_print_char(operands) {
        //zscreen.debug("zop print_char:"+operands);
        zopPrintChar(operands[0]);
        return true;
    }

    function zop_print_num(operands) {
        //zscreen.debug("zop print_num:"+operands);
        return zopPrintStream(""+operands[0]);
    }

    function zop_random(operands) {
        //zscreen.debug("zop random:"+operands);
        if (wordToSigned(operands[0]) <= 0)
            return zopResult(0);
        return zopResult(1+(new Date().getTime())%operands[0]);
    }

    function zop_push(operands) {
        //zscreen.debug("zop push:"+operands);
        stackFrame.stack.push(operands[0]);
        return true;
    }

    function zop_pull(operands) {
        //zscreen.debug("zop pull:"+operands);
        if (stackFrame.stack.length == 0)
            return zopFatalError("pull: Stack underflow");
        return zopSetResult(operands[0],stackFrame.stack.pop());
    }

    function zop_split_window(operands) {
        //zscreen.debug("zop split_window:"+operands);
        zscreen.splitUpper(operands[0]);
        return true;
    }

    function zop_set_window(operands) {
        //zscreen.debug("zop set_window:"+operands);
        zscreen.selectWindow(operands[0] != 0);
        zscreen.setCursor(1,1);
        return true;
    }

    function zop_call_vs2(operands) {
        //zscreen.debug("zop call_vs2:"+operands);
        return pushStackFrame(operands,getPcByte());
    }

    function zop_erase_window(operands) {
        //zscreen.debug("zop erase_window:"+operands);
        switch (wordToSigned(operands[0])) {
        case 0: zscreen.lowerEraseWindow(); break;
        case -1: zscreen.reset(); break;
        case -2:
            zscreen.lowerEraseWindow();
            zscreen.upperEraseWindow();
            break;
        case 1:
        default:
            zscreen.upperEraseWindow();
            break;
        }
        return true;
    }

    function zop_erase_line(operands) {
        //zscreen.debug("zop erase_line:"+operands);
        if (operands[0] == 1)
            zscreen.eraseLine();
        return true;
    }

    function zop_set_cursor(operands) {
        //zscreen.debug("zop set_cursor:"+operands);
        zscreen.setCursor(operands[1],operands[0]);
        return true;
    }

    function zop_get_cursor(operands) {
        //zscreen.debug("zop get_cursor:"+operands);
        setWord(operands[0],zscreen.getCursorY());
        setWord(operands[0]+2,zscreen.getCursorX());
        return true;
    }

    function zop_set_text_style(operands) {
        //zscreen.debug("zop set_text_style:"+operands);
        zscreen.setReverseText(operands[0]&1);
        zscreen.setBoldText(operands[0]&2);
        zscreen.setItalicText(operands[0]&4);
        zscreen.setFixedText(operands[0]&8);
        return true;
    }

    function zop_buffer_mode(operands) {
        //zscreen.debug("zop buffer_mode:"+operands);
        return true;
    }

    function zop_output_stream(operands) {
        //zscreen.debug("zop output_stream:"+operands);
        switch (wordToSigned(operands[0])) {
        case 1: case -1:
        case 2: case -2:
        case 4: case -4:
            return true;
        case 3:
            stream3.push(operands[1]);
            setWord(operands[1],0);
            return true;
        case -3:
            stream3.pop();
            return true;
        }
        return zopFatalError("output_stream: unrecognized stream " + wordToSigned(operands[0]));
    }

    function zop_input_stream(operands) {
        //zscreen.debug("zop input_stream:"+operands);
        if (operands[0] != 0)
            return zopFatalError("input_stream not supported");
        return true;
    }

    function zop_sound_effect(operands) {
        //zscreen.debug("zop sound_effect:"+operands);
        return true;
    }

    function zop_read_char(operands) {
        //zscreen.debug("zop read_char:"+operands);
        zscreen.readKey(function(code) {
            zopResult(code);
            interpreterLoop(); //ZScheduler(interpreterLoop);
        }, null);
        return -1;
    }

    function zop_scan_table(operands) {
        //zscreen.debug("zop scan_table:"+operands);
        var step = 2, get = getWord;
        if (operands.length > 3) {
            step = operands[3]&127;
            if (!(operands[3]&128))
                get = getByte;
        }
        for (var i = 0; i < operands[2]; i++)
            if (get(operands[1]+i*step) == operands[0]) {
                zopResult(operands[1]+i*step);
                return zopPcBranch(true, false);
            }
        zopResult(0);
        return zopPcBranch(false, false);
    }

    function zop_not(operands) {
        //zscreen.debug("zop not:"+operands);
        return zopResult(~operands[0]);
    }

    function zop_call_vn(operands) {
        //zscreen.debug("zop call_vn:"+operands);
        return pushStackFrame(operands,null);
    }

    function zop_call_vn2(operands) {
        //zscreen.debug("zop call_vn2:"+operands);
        return pushStackFrame(operands,null);
    }

    function zop_tokenise(operands) {
        //zscreen.debug("zop tokenise:"+operands);
        tokenize(operands[0],operands[1],operands.length>2?operands[2]:0,operands.length>3?operands[3]:0);
        return true;
    }

    function zop_encode_text(operands) {
        //zscreen.debug("zop encode_text:"+operands);
        var encoder = makeTextEncoder();
        for (var i = 0; i < operands[1]; i++)
            encoder.pushCode(getByte(operands[0]+operands[2]+i));
        encoder.padToEnd();
        setWord(operands[3],encoder.word1);
        setWord(operands[3]+2,encoder.word2);
        setWord(operands[3]+4,encoder.word3);
        return true;
    }

    function zop_copy_table(operands) {
        //zscreen.debug("zop copy_table:"+operands);
        var size = operands[2]&32767,
            forward = (operands[2]&32768) || operands[1] < operands[0];
        if (operands[1] == 0) {
            for (var i = 0; i < size; i++)
                setByte(operands[0]+i,0);
        } else if (forward) {
            for (var i = 0; i < size; i++)
                setByte(operands[1]+i,getByte(operands[0]+i));
        } else {
            for (var i = size-1; i >= 0; i--)
                setByte(operands[1]+i,getByte(operands[0]+i));
        }
        return true;
    }

    function zop_print_table(operands) {
        //zscreen.debug("zop print_table:"+operands);
        var h = operands.length > 2 ? operands[2] : 1,
            skip = operands.length > 3 ? operands[3] : 0;
        for (var j = 0; j < h; j++) {
            for (var i = 0; i < operands[1]; i++)
                zopPrintChar(getByte(operands[0]+i+j*(skip+operands[1])));
            zopPrintRet();
        }
        return true;
    }

    function zop_check_arg_count(operands) {
        //zscreen.debug("zop check_arg_count:"+operands);
        return zopPcBranch(operands[0] <= stackFrame.argc, true);
    }

    function zop_save(operands) {
        //zscreen.debug("zop save:"+operands);
        if (operands && operands.length > 0)
            return zopResult(0);
        saveData = zopSave();
        return zopResult(1);
    }

    function zop_restore(operands) {
        //zscreen.debug("zop restore:"+operands);
        if (operands && operands.length > 0)
            return zopResult(0);
        return zopResult(zopRestore(saveData) ? 2 : 0);
    }

    function zop_log_shift(operands) {
        //zscreen.debug("zop log_shift:"+operands);
        var shift = wordToSigned(operands[1]);
        if (shift >= 0)
            return zopResult(operands[0]<<shift);
        else
            return zopResult(operands[0]>>>(-shift));
    }

    function zop_art_shift(operands) {
        //zscreen.debug("zop art_shift:"+operands);
        var shift = wordToSigned(operands[1]);
        if (shift >= 0)
            return zopResult(operands[0]<<shift);
        else if (wordToSigned(operands[0]) >= 0)
            return zopResult(operands[0]>>(-shift));
        else
            return zopResult((0xffff0000|operands[0])>>(-shift));
    }

    function zop_set_font(operands) {
        //zscreen.debug("zop set_font:"+operands);
        return zopResult(0);
    }

    function zop_save_undo(operands) {
        //zscreen.debug("zop save_undo:"+operands);
        //return zopResult(65535);
        undoData = zopSave();
        return zopResult(1);
    }

    function zop_restore_undo(operands) {
        //zscreen.debug("zop restore_undo:"+operands);
        return zopResult(zopRestore(undoData) ? 2 : 0);
    }

    function zop_print_unicode(operands) {
        //zscreen.debug("zop print_unicode:"+operands);
        return zop_print_char(operands);
    }

    function zop_check_unicode(operands) {
        //zscreen.debug("zop check_unicode:"+operands);
        if (operands[0] == 13
            || (operands[0] >= 32 && operands[0] <= 126)
            || (operands[0] >= 155 && operands[0] <= 251))
            return 3;
        return 0;
    }

    var INSNS = {
        1:zop_je,
        2:zop_jl,
        3:zop_jg,
        4:zop_dec_chk,
        5:zop_inc_chk,
        6:zop_jin,
        7:zop_test,
        8:zop_or,
        9:zop_and,
        10:zop_test_attr,
        11:zop_set_attr,
        12:zop_clear_attr,
        13:zop_store,
        14:zop_insert_obj,
        15:zop_loadw,
        16:zop_loadb,
        17:zop_get_prop,
        18:zop_get_prop_addr,
        19:zop_get_next_prop,
        20:zop_add,
        21:zop_sub,
        22:zop_mul,
        23:zop_div,
        24:zop_mod,
        25:zop_call_2s,
        26:zop_call_2n,
        27:zop_set_colour,
        28:zop_throw,

        128:zop_jz,
        129:zop_get_sibling,
        130:zop_get_child,
        131:zop_get_parent,
        132:zop_get_prop_len,
        133:zop_inc,
        134:zop_dec,
        135:zop_print_addr,
        136:zop_call_1s,
        137:zop_remove_obj,
        138:zop_print_obj,
        139:zop_ret,
        140:zop_jump,
        141:zop_print_paddr,
        142:zop_load,
        143:zop_call_1n,

        176:zop_rtrue,
        177:zop_rfalse,
        178:zop_print,
        179:zop_print_ret,
        180:zop_nop,
        183:zop_restart,
        184:zop_ret_popped,
        185:zop_catch,
        186:zop_quit,
        187:zop_new_line,
        189:zop_verify,
        190:zop_extended,
        191:zop_piracy,

        224:zop_call_vs,
        225:zop_storew,
        226:zop_storeb,
        227:zop_put_prop,
        228:zop_aread,
        229:zop_print_char,
        230:zop_print_num,
        231:zop_random,
        232:zop_push,
        233:zop_pull,
        234:zop_split_window,
        235:zop_set_window,
        236:zop_call_vs2,
        237:zop_erase_window,
        238:zop_erase_line,
        239:zop_set_cursor,
        240:zop_get_cursor,
        241:zop_set_text_style,
        242:zop_buffer_mode,
        243:zop_output_stream,
        244:zop_input_stream,
        245:zop_sound_effect,
        246:zop_read_char,
        247:zop_scan_table,
        248:zop_not,
        249:zop_call_vn,
        250:zop_call_vn2,
        251:zop_tokenise,
        252:zop_encode_text,
        253:zop_copy_table,
        254:zop_print_table,
        255:zop_check_arg_count
    };

    var EXT_INSNS = {
        0:zop_save,
        1:zop_restore,
        2:zop_log_shift,
        3:zop_art_shift,
        4:zop_set_font,
        9:zop_save_undo,
        10:zop_restore_undo,
        11:zop_print_unicode,
        12:zop_check_unicode
    };

    var INSN_NAMES = {
        1:"je",
        2:"jl",
        3:"jg",
        4:"dec_chk",
        5:"inc_chk",
        6:"jin",
        7:"test",
        8:"or",
        9:"and",
        10:"test_attr",
        11:"set_attr",
        12:"clear_attr",
        13:"store",
        14:"insert_obj",
        15:"loadw",
        16:"loadb",
        17:"get_prop",
        18:"get_prop_addr",
        19:"get_next_prop",
        20:"add",
        21:"sub",
        22:"mul",
        23:"div",
        24:"mod",
        25:"call_2s",
        26:"call_2n",
        27:"set_colour",
        28:"throw",

        128:"jz",
        129:"get_sibling",
        130:"get_child",
        131:"get_parent",
        132:"get_prop_len",
        133:"inc",
        134:"dec",
        135:"print_addr",
        136:"call_1s",
        137:"remove_obj",
        138:"print_obj",
        139:"ret",
        140:"jump",
        141:"print_paddr",
        142:"load",
        143:"call_1n",

        176:"rtrue",
        177:"rfalse",
        178:"print",
        179:"print_ret",
        180:"nop",
        183:"restart",
        184:"ret_popped",
        185:"catch",
        186:"quit",
        187:"new_line",
        189:"verify",
        190:"extended",
        191:"piracy",

        224:"call_vs",
        225:"storew",
        226:"storeb",
        227:"put_prop",
        228:"aread",
        229:"print_char",
        230:"print_num",
        231:"random",
        232:"push",
        233:"pull",
        234:"split_window",
        235:"set_window",
        236:"call_vs2",
        237:"erase_window",
        238:"erase_line",
        239:"set_cursor",
        240:"get_cursor",
        241:"set_text_style",
        242:"buffer_mode",
        243:"output_stream",
        244:"input_stream",
        245:"sound_effect",
        246:"read_char",
        247:"scan_table",
        248:"not",
        249:"call_vn",
        250:"call_vn2",
        251:"tokenise",
        252:"encode_text",
        253:"copy_table",
        254:"print_table",
        255:"check_arg_count"
    };

    for (var i = 0; i < 32; i++)
        if (INSN_NAMES[i])
            INSN_NAMES[i+32] = INSN_NAMES[i+64] = INSN_NAMES[i+96] = INSN_NAMES[i+192] = INSN_NAMES[i];
    for (var i = 0; i < 16; i++)
        if (INSN_NAMES[i+128])
            INSN_NAMES[i+144] = INSN_NAMES[i+160] = INSN_NAMES[i+128];

    var EXT_INSN_NAMES = {
        0:"save",
        1:"restore",
        2:"log_shift",
        3:"art_shift",
        4:"set_font",
        9:"save_undo",
        10:"restore_undo",
        11:"print_unicode",
        12:"check_unicode"
    };

    function executeInsn() {
        var pc = stackFrame.pc,
            code = getPcByte(),
            extCode,
            zop,
            operands;
        if (code == 190) {
            extCode = getPcByte();
            zop = EXT_INSNS[extCode];
            operands = getVariableOperands();
        } else if (code >= 224) {
            zop = INSNS[code];
            if (code == 236 || code == 250)
                operands = getVariableOperands2();
            else
                operands = getVariableOperands();
        } else if (code >= 192) {
            zop = INSNS[code%32];
            operands = getVariableOperands();
        } else if (code >= 176) {
            zop = INSNS[code];
        } else if (code >= 128) {
            zop = INSNS[128+code%16];
            operands = code < 144 ? getPcWord()
                : code < 160 ? getPcByte() : getPcVariable();
        } else {
            zop = INSNS[code%32];
            operands = code < 32 ? [getPcByte(),getPcByte()]
                : code < 64 ? [getPcByte(),getPcVariable()]
                : code < 96 ? [getPcVariable(),getPcByte()]
                : [getPcVariable(),getPcVariable()];
        }
        if (!zop)
            return zopFatalError("Illegal opcode:" + code + " at " + pc);
        if (debugTrace)
            zscreen.debug("pc="+toHexString(pc)+":"+(code==190?EXT_INSN_NAMES[extCode]:INSN_NAMES[code])+":"+operands+",frame="+stack.length+",stack="+stackFrame.stack+",locals="+stackFrame.locals+",ret="+stackFrame.ret);
        try {
            return zop(operands);
        } catch (e) {
            return zopFatalError("Aborting: Exception in interpreter: " + e);
        }
    }

    (function() {
        var B64 = {};
        for (var i = 0; i < 26; i++) {
            B64[i+65] = i+1;
            B64[i+97] = i+27;
        }
        for (var i = 48; i < 58; i++)
            B64[i] = i+5;
        B64[43] = 63;
        B64[47] = 64;

        function b64decode(string, handler) {
            var bytes = [], buffer = 0, index = 0, i = 0;
            function iterate() {
                var chunkMax = Math.min(i+32768,string.length),
                    n = chunkMax - i;
                do {
                    var c;
                    if (c = B64[string.charCodeAt(i)]) {
                        c--;
                        switch (++index) {
                        case 1: buffer = c<<2; break;
                        case 2: bytes.push(buffer|(c>>4)); buffer = (c<<4)&255; break;
                        case 3: bytes.push(buffer|(c>>2)); buffer = (c<<6)&255; break;
                        case 4: bytes.push(buffer|c); index = 0; break;
                        }
                    }
                    i++;
                } while (--n);
                if (i >= string.length) {
                    if (index != 0)
                        bytes.push(buffer);
                    handler(bytes);
                } else {
                    zscreen.lowerAddText(".");
                    ZScheduler(iterate);
                }
            }
            ZScheduler(iterate);
        }

        function load(uri) {
            zscreen.lowerAddText("Loading " + uri + "...");
            var x;
            try {
                x = new XMLHttpRequest();
            } catch (e) {
                try {
                    x = new ActiveXObject("Msxml2.XMLHTTP");
                } catch (e) {
                    try {
                        x = new ActiveXObject("Microsoft.XMLHTTP");
                    } catch (e) {
                        zscreen.errMsg("Failed: XMLHttpRequest not available");
                        return;
                    }
                }
            }
            x.onreadystatechange = function() {
                if (x.readyState != 4)
                return;
                if (x.status != 200) {
                    zscreen.errorMsg("Loading failed: " + x.status + " " + x.statusText);
                    return;
                }
                b64decode(x.responseText, function(result) {
                    zdata = result;
                    zscreen.lowerAddText("done.");
                    zscreen.readKey(function() {
                        ZScheduler(function () {
                            init();
                            interpreterLoop();
                        });
                    }, "Press space to start");
                });
            }
            x.open("GET",uri,true);
            x.send(null);
        }

        var ABOUT = "<br><div style=\"text-align:center\"><b>About Muttonate</b> "+VERSION+"</div>"
            +"<p>This interpreter implements a subset of the <a href=\"http://www.ifarchive.org/if-archive/infocom/interpreters/specification\" target=\"_parent\">Z-Machine Specification 1.0</a>."
            +"<p>It recognizes only version 5 and 8 story files.  It makes no attempt to support any Infocom quirks.  Making Infocom games available on the web wouldn't be legal anyhow (if I were to dig that Activision CD I got in 1994 out of somewhere in my closet)."
            +"<p>A single save (to memory) is supported.  Subsequent saves overwrite previous saves."
            +"<p>This interpreter loads base64 encoded story files, since XmlHttpRequest tends to mangle binary data."
            +"<p>Except on fast computers (as of Jan 2006), this interpreter is too slow to be playable with most story files."
            +"<p>Microsoft Internet Explorer (as of Jan 2006) does not appear to support the styles <tt>white-space: pre</tt> and <tt>position: fixed</tt>, which cause the layout to be wrong with that browser."
            +"<p>Except for scrolling appropriately, this interpreter seems to display things reasonably with Safari and Firefox (as of Jan 2006)."
            +"<p>Problems with this interpreter or with the story file can lock up the browser.  Firefox has protection against infinite loops.  Safari does not."
            +"<p>I do not know if this interpreter works on other browsers.  (Except lynx and links, on which it would not work.)"
            +"<p>The <a href=\"http://www.ifarchive.org/if-archive/infocom/interpreters/emacs\" target=\"_parent\">malyon</a> source code and its tracing facility were very helpful in the making of this interpreter.";

        var GPL =
"		    GNU GENERAL PUBLIC LICENSE\n"
+"		       Version 2, June 1991\n"
+"\n"
+" Copyright (C) 1989, 1991 Free Software Foundation, Inc.\n"
+"     59 Temple Place, Suite 330, Boston, MA  02111-1307  USA\n"
+" Everyone is permitted to copy and distribute verbatim copies\n"
+" of this license document, but changing it is not allowed.\n"
+"\n"
+"			    Preamble\n"
+"\n"
+"  The licenses for most software are designed to take away your\n"
+"freedom to share and change it.  By contrast, the GNU General Public\n"
+"License is intended to guarantee your freedom to share and change free\n"
+"software--to make sure the software is free for all its users.  This\n"
+"General Public License applies to most of the Free Software\n"
+"Foundation's software and to any other program whose authors commit to\n"
+"using it.  (Some other Free Software Foundation software is covered by\n"
+"the GNU Library General Public License instead.)  You can apply it to\n"
+"your programs, too.\n"
+"\n"
+"  When we speak of free software, we are referring to freedom, not\n"
+"price.  Our General Public Licenses are designed to make sure that you\n"
+"have the freedom to distribute copies of free software (and charge for\n"
+"this service if you wish), that you receive source code or can get it\n"
+"if you want it, that you can change the software or use pieces of it\n"
+"in new free programs; and that you know you can do these things.\n"
+"\n"
+"  To protect your rights, we need to make restrictions that forbid\n"
+"anyone to deny you these rights or to ask you to surrender the rights.\n"
+"These restrictions translate to certain responsibilities for you if you\n"
+"distribute copies of the software, or if you modify it.\n"
+"\n"
+"  For example, if you distribute copies of such a program, whether\n"
+"gratis or for a fee, you must give the recipients all the rights that\n"
+"you have.  You must make sure that they, too, receive or can get the\n"
+"source code.  And you must show them these terms so they know their\n"
+"rights.\n"
+"\n"
+"  We protect your rights with two steps: (1) copyright the software, and\n"
+"(2) offer you this license which gives you legal permission to copy,\n"
+"distribute and/or modify the software.\n"
+"\n"
+"  Also, for each author's protection and ours, we want to make certain\n"
+"that everyone understands that there is no warranty for this free\n"
+"software.  If the software is modified by someone else and passed on, we\n"
+"want its recipients to know that what they have is not the original, so\n"
+"that any problems introduced by others will not reflect on the original\n"
+"authors' reputations.\n"
+"\n"
+"  Finally, any free program is threatened constantly by software\n"
+"patents.  We wish to avoid the danger that redistributors of a free\n"
+"program will individually obtain patent licenses, in effect making the\n"
+"program proprietary.  To prevent this, we have made it clear that any\n"
+"patent must be licensed for everyone's free use or not licensed at all.\n"
+"\n"
+"  The precise terms and conditions for copying, distribution and\n"
+"modification follow.\n"
+"\n"
+"		    GNU GENERAL PUBLIC LICENSE\n"
+"   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION\n"
+"\n"
+"  0. This License applies to any program or other work which contains\n"
+"a notice placed by the copyright holder saying it may be distributed\n"
+"under the terms of this General Public License.  The \"Program\", below,\n"
+"refers to any such program or work, and a \"work based on the Program\"\n"
+"means either the Program or any derivative work under copyright law:\n"
+"that is to say, a work containing the Program or a portion of it,\n"
+"either verbatim or with modifications and/or translated into another\n"
+"language.  (Hereinafter, translation is included without limitation in\n"
+"the term \"modification\".)  Each licensee is addressed as \"you\".\n"
+"\n"
+"Activities other than copying, distribution and modification are not\n"
+"covered by this License; they are outside its scope.  The act of\n"
+"running the Program is not restricted, and the output from the Program\n"
+"is covered only if its contents constitute a work based on the\n"
+"Program (independent of having been made by running the Program).\n"
+"Whether that is true depends on what the Program does.\n"
+"\n"
+"  1. You may copy and distribute verbatim copies of the Program's\n"
+"source code as you receive it, in any medium, provided that you\n"
+"conspicuously and appropriately publish on each copy an appropriate\n"
+"copyright notice and disclaimer of warranty; keep intact all the\n"
+"notices that refer to this License and to the absence of any warranty;\n"
+"and give any other recipients of the Program a copy of this License\n"
+"along with the Program.\n"
+"\n"
+"You may charge a fee for the physical act of transferring a copy, and\n"
+"you may at your option offer warranty protection in exchange for a fee.\n"
+"\n"
+"  2. You may modify your copy or copies of the Program or any portion\n"
+"of it, thus forming a work based on the Program, and copy and\n"
+"distribute such modifications or work under the terms of Section 1\n"
+"above, provided that you also meet all of these conditions:\n"
+"\n"
+"    a) You must cause the modified files to carry prominent notices\n"
+"    stating that you changed the files and the date of any change.\n"
+"\n"
+"    b) You must cause any work that you distribute or publish, that in\n"
+"    whole or in part contains or is derived from the Program or any\n"
+"    part thereof, to be licensed as a whole at no charge to all third\n"
+"    parties under the terms of this License.\n"
+"\n"
+"    c) If the modified program normally reads commands interactively\n"
+"    when run, you must cause it, when started running for such\n"
+"    interactive use in the most ordinary way, to print or display an\n"
+"    announcement including an appropriate copyright notice and a\n"
+"    notice that there is no warranty (or else, saying that you provide\n"
+"    a warranty) and that users may redistribute the program under\n"
+"    these conditions, and telling the user how to view a copy of this\n"
+"    License.  (Exception: if the Program itself is interactive but\n"
+"    does not normally print such an announcement, your work based on\n"
+"    the Program is not required to print an announcement.)\n"
+"\n"
+"These requirements apply to the modified work as a whole.  If\n"
+"identifiable sections of that work are not derived from the Program,\n"
+"and can be reasonably considered independent and separate works in\n"
+"themselves, then this License, and its terms, do not apply to those\n"
+"sections when you distribute them as separate works.  But when you\n"
+"distribute the same sections as part of a whole which is a work based\n"
+"on the Program, the distribution of the whole must be on the terms of\n"
+"this License, whose permissions for other licensees extend to the\n"
+"entire whole, and thus to each and every part regardless of who wrote it.\n"
+"\n"
+"Thus, it is not the intent of this section to claim rights or contest\n"
+"your rights to work written entirely by you; rather, the intent is to\n"
+"exercise the right to control the distribution of derivative or\n"
+"collective works based on the Program.\n"
+"\n"
+"In addition, mere aggregation of another work not based on the Program\n"
+"with the Program (or with a work based on the Program) on a volume of\n"
+"a storage or distribution medium does not bring the other work under\n"
+"the scope of this License.\n"
+"\n"
+"  3. You may copy and distribute the Program (or a work based on it,\n"
+"under Section 2) in object code or executable form under the terms of\n"
+"Sections 1 and 2 above provided that you also do one of the following:\n"
+"\n"
+"    a) Accompany it with the complete corresponding machine-readable\n"
+"    source code, which must be distributed under the terms of Sections\n"
+"    1 and 2 above on a medium customarily used for software interchange; or,\n"
+"\n"
+"    b) Accompany it with a written offer, valid for at least three\n"
+"    years, to give any third party, for a charge no more than your\n"
+"    cost of physically performing source distribution, a complete\n"
+"    machine-readable copy of the corresponding source code, to be\n"
+"    distributed under the terms of Sections 1 and 2 above on a medium\n"
+"    customarily used for software interchange; or,\n"
+"\n"
+"    c) Accompany it with the information you received as to the offer\n"
+"    to distribute corresponding source code.  (This alternative is\n"
+"    allowed only for noncommercial distribution and only if you\n"
+"    received the program in object code or executable form with such\n"
+"    an offer, in accord with Subsection b above.)\n"
+"\n"
+"The source code for a work means the preferred form of the work for\n"
+"making modifications to it.  For an executable work, complete source\n"
+"code means all the source code for all modules it contains, plus any\n"
+"associated interface definition files, plus the scripts used to\n"
+"control compilation and installation of the executable.  However, as a\n"
+"special exception, the source code distributed need not include\n"
+"anything that is normally distributed (in either source or binary\n"
+"form) with the major components (compiler, kernel, and so on) of the\n"
+"operating system on which the executable runs, unless that component\n"
+"itself accompanies the executable.\n"
+"\n"
+"If distribution of executable or object code is made by offering\n"
+"access to copy from a designated place, then offering equivalent\n"
+"access to copy the source code from the same place counts as\n"
+"distribution of the source code, even though third parties are not\n"
+"compelled to copy the source along with the object code.\n"
+"\n"
+"  4. You may not copy, modify, sublicense, or distribute the Program\n"
+"except as expressly provided under this License.  Any attempt\n"
+"otherwise to copy, modify, sublicense or distribute the Program is\n"
+"void, and will automatically terminate your rights under this License.\n"
+"However, parties who have received copies, or rights, from you under\n"
+"this License will not have their licenses terminated so long as such\n"
+"parties remain in full compliance.\n"
+"\n"
+"  5. You are not required to accept this License, since you have not\n"
+"signed it.  However, nothing else grants you permission to modify or\n"
+"distribute the Program or its derivative works.  These actions are\n"
+"prohibited by law if you do not accept this License.  Therefore, by\n"
+"modifying or distributing the Program (or any work based on the\n"
+"Program), you indicate your acceptance of this License to do so, and\n"
+"all its terms and conditions for copying, distributing or modifying\n"
+"the Program or works based on it.\n"
+"\n"
+"  6. Each time you redistribute the Program (or any work based on the\n"
+"Program), the recipient automatically receives a license from the\n"
+"original licensor to copy, distribute or modify the Program subject to\n"
+"these terms and conditions.  You may not impose any further\n"
+"restrictions on the recipients' exercise of the rights granted herein.\n"
+"You are not responsible for enforcing compliance by third parties to\n"
+"this License.\n"
+"\n"
+"  7. If, as a consequence of a court judgment or allegation of patent\n"
+"infringement or for any other reason (not limited to patent issues),\n"
+"conditions are imposed on you (whether by court order, agreement or\n"
+"otherwise) that contradict the conditions of this License, they do not\n"
+"excuse you from the conditions of this License.  If you cannot\n"
+"distribute so as to satisfy simultaneously your obligations under this\n"
+"License and any other pertinent obligations, then as a consequence you\n"
+"may not distribute the Program at all.  For example, if a patent\n"
+"license would not permit royalty-free redistribution of the Program by\n"
+"all those who receive copies directly or indirectly through you, then\n"
+"the only way you could satisfy both it and this License would be to\n"
+"refrain entirely from distribution of the Program.\n"
+"\n"
+"If any portion of this section is held invalid or unenforceable under\n"
+"any particular circumstance, the balance of the section is intended to\n"
+"apply and the section as a whole is intended to apply in other\n"
+"circumstances.\n"
+"\n"
+"It is not the purpose of this section to induce you to infringe any\n"
+"patents or other property right claims or to contest validity of any\n"
+"such claims; this section has the sole purpose of protecting the\n"
+"integrity of the free software distribution system, which is\n"
+"implemented by public license practices.  Many people have made\n"
+"generous contributions to the wide range of software distributed\n"
+"through that system in reliance on consistent application of that\n"
+"system; it is up to the author/donor to decide if he or she is willing\n"
+"to distribute software through any other system and a licensee cannot\n"
+"impose that choice.\n"
+"\n"
+"This section is intended to make thoroughly clear what is believed to\n"
+"be a consequence of the rest of this License.\n"
+"\n"
+"  8. If the distribution and/or use of the Program is restricted in\n"
+"certain countries either by patents or by copyrighted interfaces, the\n"
+"original copyright holder who places the Program under this License\n"
+"may add an explicit geographical distribution limitation excluding\n"
+"those countries, so that distribution is permitted only in or among\n"
+"countries not thus excluded.  In such case, this License incorporates\n"
+"the limitation as if written in the body of this License.\n"
+"\n"
+"  9. The Free Software Foundation may publish revised and/or new versions\n"
+"of the General Public License from time to time.  Such new versions will\n"
+"be similar in spirit to the present version, but may differ in detail to\n"
+"address new problems or concerns.\n"
+"\n"
+"Each version is given a distinguishing version number.  If the Program\n"
+"specifies a version number of this License which applies to it and \"any\n"
+"later version\", you have the option of following the terms and conditions\n"
+"either of that version or of any later version published by the Free\n"
+"Software Foundation.  If the Program does not specify a version number of\n"
+"this License, you may choose any version ever published by the Free Software\n"
+"Foundation.\n"
+"\n"
+"  10. If you wish to incorporate parts of the Program into other free\n"
+"programs whose distribution conditions are different, write to the author\n"
+"to ask for permission.  For software which is copyrighted by the Free\n"
+"Software Foundation, write to the Free Software Foundation; we sometimes\n"
+"make exceptions for this.  Our decision will be guided by the two goals\n"
+"of preserving the free status of all derivatives of our free software and\n"
+"of promoting the sharing and reuse of software generally.\n"
+"\n"
+"			    NO WARRANTY\n"
+"\n"
+"  11. BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY\n"
+"FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT WHEN\n"
+"OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES\n"
+"PROVIDE THE PROGRAM \"AS IS\" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED\n"
+"OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF\n"
+"MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS\n"
+"TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE\n"
+"PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,\n"
+"REPAIR OR CORRECTION.\n"
+"\n"
+"  12. IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING\n"
+"WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR\n"
+"REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES,\n"
+"INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING\n"
+"OUT OF THE USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED\n"
+"TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY\n"
+"YOU OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER\n"
+"PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE\n"
+"POSSIBILITY OF SUCH DAMAGES.\n"
+"\n"
+"		     END OF TERMS AND CONDITIONS\n"
+"\n"
+"	    How to Apply These Terms to Your New Programs\n"
+"\n"
+"  If you develop a new program, and you want it to be of the greatest\n"
+"possible use to the public, the best way to achieve this is to make it\n"
+"free software which everyone can redistribute and change under these terms.\n"
+"\n"
+"  To do so, attach the following notices to the program.  It is safest\n"
+"to attach them to the start of each source file to most effectively\n"
+"convey the exclusion of warranty; and each file should have at least\n"
+"the \"copyright\" line and a pointer to where the full notice is found.\n"
+"\n"
+"    <one line to give the program's name and a brief idea of what it does.>\n"
+"    Copyright (C) <year>  <name of author>\n"
+"\n"
+"    This program is free software; you can redistribute it and/or modify\n"
+"    it under the terms of the GNU General Public License as published by\n"
+"    the Free Software Foundation; either version 2 of the License, or\n"
+"    (at your option) any later version.\n"
+"\n"
+"    This program is distributed in the hope that it will be useful,\n"
+"    but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
+"    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
+"    GNU General Public License for more details.\n"
+"\n"
+"    You should have received a copy of the GNU General Public License\n"
+"    along with this program; if not, write to the Free Software\n"
+"    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA\n"
+"\n"
+"\n"
+"Also add information on how to contact you by electronic and paper mail.\n"
+"\n"
+"If the program is interactive, make it output a short notice like this\n"
+"when it starts in an interactive mode:\n"
+"\n"
+"    Gnomovision version 69, Copyright (C) year  name of author\n"
+"    Gnomovision comes with ABSOLUTELY NO WARRANTY; for details type `show w'.\n"
+"    This is free software, and you are welcome to redistribute it\n"
+"    under certain conditions; type `show c' for details.\n"
+"\n"
+"The hypothetical commands `show w' and `show c' should show the appropriate\n"
+"parts of the General Public License.  Of course, the commands you use may\n"
+"be called something other than `show w' and `show c'; they could even be\n"
+"mouse-clicks or menu items--whatever suits your program.\n"
+"\n"
+"You should also get your employer (if you work as a programmer) or your\n"
+"school, if any, to sign a \"copyright disclaimer\" for the program, if\n"
+"necessary.  Here is a sample; alter the names:\n"
+"\n"
+"  Yoyodyne, Inc., hereby disclaims all copyright interest in the program\n"
+"  `Gnomovision' (which makes passes at compilers) written by James Hacker.\n"
+"\n"
+"  <signature of Ty Coon>, 1 April 1989\n"
+"  Ty Coon, President of Vice\n"
+"\n"
+"This General Public License does not permit incorporating your program into\n"
+"proprietary programs.  If your program is a subroutine library, you may\n"
+"consider it more useful to permit linking proprietary applications with the\n"
+"library.  If this is what you want to do, use the GNU Library General\n"
+"Public License instead of this License.\n";

        zscreen.lowerAddHtml("<br><i><b><big>Muttonate</big></b> version " + VERSION + "</i><br><small>Copyright &copy; 2006 Q P Liu<br><i>A JavaScript Z-Machine 1.0 interpreter</i></small><br>");

        if (htmlPageBlurb)
            zscreen.lowerAddHtml("<blockquote>"+htmlPageBlurb+"</blockquote>");

        zscreen.lowerAddText("You can read ");
        var about = zscreen.lowerAddHtml("ABOUT");
        zscreen.lowerAddText(" this interpreter or the terms for ");
        var copying = zscreen.lowerAddHtml("COPYING");
        zscreen.lowerAddText(".\n");

        var info = zscreen.lowerAddHtml("");
        about.style.color = "blue";
        about.title = "About Muttonate";
        about.onclick = function() {
            info.innerHTML = ABOUT;
        };
        copying.style.color = "blue";
        copying.title = "Terms For Copying Muttonate";
        copying.onclick = function() {
            info.innerHTML = "<pre>"+GPL+"</pre>";
        };
        if (uri)
            zscreen.readKey(function() {
                info.innerHTML = "";
                about.style.color = null;
                about.title = null;
                about.onclick = null;
                copying.style.color = null;
                copying.title = null;
                copying.onclick = null;
                ZScheduler(function() { load(uri); });
            }, "Press space to load "+uri);
    })();
}
