! Pathmaker for Triform
! Based upon copyrighted code created by Jim Fisher
!
! License: 
! This work is licensed under the Creative Commons Attribution-ShareAlike License
! (http://creativecommons.org/licenses/by-sa/1.0/).
!
! In summary: you must credit the original author(s); if you alter, transform, or 
! build upon this software, you may distribute the SOURCE FORM of the resulting 
! work only under a license identical to this one. Note that the ShareAlike clause 
! does not affect the way in which you distribute the COMPILED FORM of works built upon
! this software. Copyright remains with the original author(s), from whom you
! must seek permission if you wish to vary any of the terms of this license.
! The author(s) would also welcome bug reports and enhancement suggestions.
!
!
!
! The PathMaker object will determine a path between two rooms.  Two starting objects 
! are taken by the determine_path routine. These can be rooms or objects or characters. 
! Passage through doors is resolved, but if the doors require keys, then the starting 
! object (the one first passed in) must be an object that contains the needed keys.
! This allows an NPC with the appropriate key to successfully take a path that an NPC
! without the appropriate key could not.
!
! Sample use:
!
!	PathMaker.determine_path(player, bathroom);
!
!	PathMaker.determine_path(bathroom, bedroom);
!
! The return values for this routine are:
!	
!	-1 : There is no connecting path between the two given objects.
!	-2 : Ran out of workspace while trying to calculate path.  Define the constant
!			PATHDEPTH with a larger than default value.
!	-3 : Although there WAS a path correctly determined, there was NOT enough space in 
!			the path property to contain it. 
!	room obj : The first connected room that must be traveled to in order to finally reach
!			the destination.
!
! Additionally, once the path has been correctly determined, the size property contains 
! the size and the path property contains the final path.
!
! There is a limited amount of scratch workspace used to determine a correct path. If 
! there is not enough space, the path will not be findable. The constant PATHDEPTH
! is used to allocate this scratch space. By default it is set to 30, but can be 
! defined at a greater value in your program's source.
!
! TO DO:	PathMaker.compare_paths(bathroom, bedroom, bathroom, kitchen, flag);
!		where flag can compare the lengths or find the room where the two paths intersect

message " Processing library extension 3PathMaker...";

!======================================================================================
! C O D E   section (for code that falls between VERBLIB and GRAMMAR)
!======================================================================================

	#Ifndef PATHDEPTH;
	Constant PATHDEPTH 30;
	#Endif;

	Array __pathmaker_room-->PATHDEPTH;

	Array __pathmaker_distance->PATHDEPTH;

	Constant USE_PathMaker;

	Object PathMaker
		with	path 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0,
		end_ptr	0,
		cur_ptr	0,
		size 0,
		npc	PathMaker, !--self if not defined...strange, but it will work
		find_room_distance
		[ room t;

			for(t=0:t<self.end_ptr:t++) {
				if(__pathmaker_room-->t==room) 
					return __pathmaker_distance->t;
			}
			return -1;
		],
		add
		[ room level;

			if(self.find_room_distance(room)==-1) {
				__pathmaker_room-->self.end_ptr=room;
				__pathmaker_distance->self.end_ptr=level;
				self.end_ptr++;
			}
		],
		determine_path
		[ npc_cntx endroom t startroom;

			if(parent(npc_cntx)==0) {
				startroom=npc_cntx; !--room was passed in instead of a character
				self.npc=startroom;
			}
			else {
				startroom=npc_cntx.visibility_ceiling;
				self.npc=npc_cntx;
			}

			!--an object was specified instead of a room... determine

			if(parent(endroom)~=0) endroom=endroom.location;

			if(startroom==endroom) return 0;

			for(t=0:t<self.end_ptr:t++) { 	!-- first, clear everything out
				__pathmaker_room-->t=0;
				__pathmaker_distance->t=0;
			}

			self.end_ptr=self.cur_ptr=0;
			self.add(startroom,0); 
			t=self.process(endroom);

			if(t<0) return t; !--- Errors? -1: not reachable; -2: out of workspace

!---all was good, let's fill in the path property and return (either an error or the first room in the list...)

			return self.unwind_path(); 
		],
		process
		[ endroom r cr cl t;

			while(self.cur_ptr<self.end_ptr) {
				cr=__pathmaker_room-->(self.cur_ptr);
				cl=self.find_room_distance(cr);

				objectloop(t in compass && t provides door_dir) {
					if(cr provides (t.door_dir)) {
						r=cr.(t.door_dir);
						if(r ofclass Room || r ofclass Door) {

								if (r ofclass Door) r = r.door_to(self.npc);

								!--success.  return needed size for path

								self.add(r,cl+1);
								if(r==endroom) return cl+1;

								if(self.end_ptr>PATHDEPTH)
									return -2; !--Out of space
								}
							}
						}
						self.cur_ptr++;
					}
					return -1; !--completed search, with no possible path found
				],
		unwind_path
		[ pos d room r t;

!--called once if the search algorithm has completed successfully,
!--to find the shortest path and put it in the property pointed to by self.path

			room=__pathmaker_room-->(self.end_ptr-1); !--locate the endroom
			self.size=pos=self.find_room_distance(room); !--size of path 
					
			if((self.#path/WORDSIZE)<pos) return -3; !--path too small
				while(pos>0) {
					self.&path-->(pos-1)=room;
					objectloop(d in compass && d provides door_dir) {
						if(room provides (d.door_dir)) {
							r=room.(d.door_dir);
							if(r~=0) {
								if (r ofclass Door) r = r.door_to(self.npc);
								t=self.find_room_distance(r);
								if(t>-1 && t<pos) break;
							}
						}
					}
				room=r;
				pos--;
			}
			return self.&path-->0;
		];