/*
 RAP 1.0 Planbase

 Reactive Agent Planner for TADS
 (the Text Adventure Development System)
 ----------------------------------------


 Default Planbase
 ----------------

 Standard definitions for common Rap conditions and actions


 defines:

  rCarrying: rCond   //  is actor carrying object?
  rGettable: rCond   //  is object a gettable item?
  rReachable: rCond  //  is object within reachable scope?
  rNotHeld: rCond  //  is object not currently held by another actor?
  rOpen: rCond       //  is object open?
  rUnlocked: rCond   //  is object unlocked?
  rIn: rCond         //  is actor inside room?


  rNoAction: rAct    // actor does nothing
  rGet: rAct         // actor gets an item
  rOpen: rAct        // actor opens a door or openable
  rUnlock: rAct      // actor unlocks a lockableDoorway or lockable
  rAskFor: rAct      // actor asks player (or other actor) for object

  rBuildMap: function // creates default movement plans at compile time
  rBuildMapPlan: function // OVERRIDE THIS to add new default
                          // behaviour for movement plans


 modifies:

  room               // adds rBot* directions for robot navigation
                     // intended for extension to rIn condition
                     // to allow default map creation

                     // adds traveldesc property (default to thedesc)
                     // adds rFromList and rMovePlan properties

*/


modify room

 // robot direction properties - currently only test property type of
 // normal exits, and return their value only if it is 2 (an object)
 // ie, this guarantees that no player-movement trigger code will be
 // accidentally run by a robot building a map


 rFromList = []  // list of directions for machine-generated movement plans
                 // created by rBuildMap function

 rBotNorth =
 {
  local x;
  x:=proptype(self,&north);

  if (x=2) return (self.north); else return (nil);
  }

 rBotSouth =
 {
  local x;
  x:=proptype(self,&south);

  if (x=2) return (self.south); else return (nil);
  }


 rBotEast =
 {
  local x;
  x:=proptype(self,&east);

  if (x=2) return (self.east); else return (nil);
  }

 rBotWest =
 {
  local x;
  x:=proptype(self,&west);

  if (x=2) return (self.west); else return (nil);
  }

 rBotNE =
 {
  local x;
  x:=proptype(self,&ne);

  if (x=2) return (self.ne); else return (nil);
  }

 rBotNW =
 {
  local x;
  x:=proptype(self,&nw);

  if (x=2) return (self.nw); else return (nil);
  }

 rBotSE =
 {
  local x;
  x:=proptype(self,&se);

  if (x=2) return (self.se); else return (nil);
  }

 rBotSW =
 {
  local x;
  x:=proptype(self,&sw);

  if (x=2) return (self.sw); else return (nil);
  }

 rBotUp =
 {
  local x;
  x:=proptype(self,&up);

  if (x=2) return (self.up); else return (nil);
  }

 rBotDown =
 {
  local x;
  x:=proptype(self,&down);

  if (x=2) return (self.down); else return (nil);
  }

 rBotIn =
 {
  local x;
  x:=proptype(self,&in);

  if (x=2) return (self.in); else return (nil);
  }

 rBotOut =
 {
  local x;
  x:=proptype(self,&out);

  if (x=2) return (self.out); else return (nil);
  }


// travel description - name of room as used for travel messages
// defaults to thedesc, but this may need to be overridden
// for each room to get capitalisation and phrasing right

 traveldesc =
 {
   self.thedesc;
 }

;

rNoAction: rAct
  sdesc = "rNoAction"
  rPreDesc(a,p) =
  {
    a.rIdleDesc;
  }
  rActionCode(a,p) = {a.rIdleCode;}
;



rCarrying: rCond
 sdesc = "rCarrying"
 rTrue(a,p) =
 {
   return (a.isCarrying(p));
 }
 rPlans(a,p) =
 {
   return(
     [
       [rBe rGettable p
        rBe rReachable p
        rBe rNotHeld p
        rDo rGet p]
     ]
   );
 }
;


rGettable: rCond
 sdesc = "rGettable"
 rTrue(a,p) =
 {

   // simple test - is it an item? 

   return (isclass(p,item));
 }

 // no plans for making non-items gettable in this release

;


rGet: rAct
 sdesc = "rGet"
 rPreDesc(a,p) =  "\b\^<<a.thedesc>> picks up <<p.thedesc>>."
 rActionCode(a,p)=
 {
   p.moveInto(a);
 }
;

rReachable: rCond
 sdesc = "rReachable"
 rTrue(a,p) = 
 {
   return (p.isReachable(a));
 }

 rPlans(a,p) =
 {
   // find closet room-level location of object
   local l;
   l:=p;
   while (not isclass(l,room)) l:=l.location;
   return(
    [
      [rBe rIn l]
    ]
   );
 }

;

rNotHeld: rCond
 sdesc = "rNotHeld"
 rTrue(a,p) =
 {
   // default implementation - false if carried by player, true otherwise
   // plan is to ask player for object
   // should be redirected to object itself in a more final release

   if (Me.isCarrying(p))
     return (nil);
   else return (true);
    
 }

 rPlans(a,p) =
 {

   if (Me.isCarrying(p))
    return(
     [
       [rDo rAskFor [Me p]]
     ]
    );
   else return(nil);
 }
;

rAskFor: rAct
 sdesc = "rAskFor"
 rActionCode(a,p) = {}  // asking other actors is not yet implemented
 rPreDesc(a,p) =
 {
   local obj, askee;

   // parameter is a two-item list, [actorToAsk objectToAskFor]
   // or an object (objectToAskFor), assuming actorToAsk is Me 

   if (datatype(p)=2)
   {
     
     obj:=p;
     askee:=Me;
   }
   else
   {
     askee:=p[1];
     obj:=p[2];
   }
   "\^<<a.thedesc>> asks ";

   if (askee=Me) "you"; else askee.thedesc;
   " for <<obj.thedesc>>.";

 }
;

rOpen: rCond, rAct
 sdesc = "rOpen"

  rTrue(a,p) =
  {
    if (isclass(p,doorway) or isclass(p,openable))
      return (p.isopen);
    else
      return (nil);
  }
 rPlans(a,p) =
 {
   return (
    [
      [rBe rUnlocked p
       rBe rReachable p
       rDo rOpen p]
    ]
   );
 }


 rPreDesc(a,p) = "\b\^<<a.thedesc>> opens <<p.thedesc>>."
 rActionCode(a,p) =
 {
   p.isopen := true;
   if (isclass(p,doorway)) p.otherside.isopen := true;
 }
;

rUnlocked: rCond
 sdesc = "rUnlocked"
 rTrue(a,b) =
 {
   return (b.islocked = nil);
 }

 rPlans(a,p) =
  {
    return (
     [
       [rBe rCarrying p.key
        rBe rReachable p
        rDo rUnlock p]
     ]
    );
  }

;

rUnlock: rCond
 sdesc = "rUnlock"
 rPreDesc(a,p) = "\b\^<<a.thedesc>> unlocks <<p.thedesc>>."
 rActionCode(a,p) =
 {
   p.islocked:=nil;
   if (isclass(p,doorway)) p.otherside.islocked:=nil;
 }
;


rIn: rCond
 sdesc = "rIn"
 rTrue(a,p) =
 {
   return (a.isIn(p));
 }

 rPlans(a,p) =
 {
  return (p.rMovePlan);
 }

;


rGo: rAct
 sdesc = "rGo"
 rPreDesc(a,p) =  {}

 rAction(a,p) =
 {
    local oldloc;
    oldloc:=a.location;
    if (a.rLoudAction(self,p)) a.rTravelToDesc(p);
    a.moveInto(p);
    if (a.rLoudAction(self,p)) a.rTravelFromDesc(oldloc);

 }
;


// rBuildMapPlan: called by rBuildMap
// replace this function to add new default move plan behaviour
// returns a valid RAP plan list for entering room from the specified
// entry (where entry is either an adjacent room, or an obstacle/doorway
// located in an adjacent room)

// Currently only rooms and doorways are supported.
// Other exit types may be added at a later date.


rBuildMapPlan: function (thisRoom,thisEntry)
{
  local thisPlan;

  if (isclass(thisEntry,room))

    thisPlan:= [
                rBe rIn thisEntry
                rDo rGo thisRoom
               ];

   else if (isclass(thisEntry,doorway))
     thisPlan := [
                  rBe rOpen thisEntry
                  rBe rIn thisEntry.location
                  rDo rGo thisRoom
                 ];

   return (thisPlan);

}

// rBuildMap: creates rFromList list for all rooms
// used for automatic movement plan generation
// run this at compile time in preinit function

rBuildMap: function
{
  local thisRoom, thisExit, dest, tempList, j, thisEntry, thisPlan;
  "\nrBuildMap: generating entry lists\n";
  for (thisRoom:=firstobj(room);thisRoom;thisRoom:=nextobj(thisRoom,room))
  {
    "\n<<thisRoom.sdesc>>: ";
    for (thisExit:=1;thisExit<=12;thisExit++)
    {
      switch (thisExit)
      {
        case 1: dest := thisRoom.rBotNorth; break;
        case 2: dest := thisRoom.rBotSouth; break;
        case 3: dest := thisRoom.rBotEast; break;
        case 4: dest := thisRoom.rBotWest; break;
        case 5: dest := thisRoom.rBotNE; break;
        case 6: dest := thisRoom.rBotNW; break;
        case 7: dest := thisRoom.rBotSE; break;
        case 8: dest := thisRoom.rBotSW; break;
        case 9: dest := thisRoom.rBotUp; break;
        case 10: dest := thisRoom.rBotDown; break;
        case 11: dest := thisRoom.rBotIn; break;
        case 12: dest := thisRoom.rBotOut; break;
      }

      // for each valid robot movement, add thisRoom to the
      // destination's rFromList

      if (dest)
      {
        if (isclass(dest,room))
        {
          "<<dest.sdesc>> ";
          dest.rFromList += thisRoom;
//          rToString(dest.rFromList); " ";
        }
        else if (isclass (dest,doorway))
        {
          "door_to <<dest.doordest.sdesc>> ";
          dest.doordest.rFromList += dest;

        }
      }

    } // for thisExit

  }  // for thisRoom


  // for each room, construct rMovePlan from rFromList

  "\nrBuildMap: producing default plans\n";
  for (thisRoom:=firstobj(room);thisRoom;thisRoom:=nextobj(thisRoom,room))
  {
    thisRoom.rMovePlan:=[];
    tempList:=thisRoom.rFromList;
    for (j:=1;j<=length(tempList);j++)
    {
      thisEntry:=tempList[j];

      thisPlan := rBuildMapPlan(thisRoom,thisEntry);

      thisRoom.rMovePlan += [ thisPlan ];
    }
     "\nrBuildMap: <<thisRoom.sdesc>> = ";  rToString(thisRoom.rMovePlan); "\n";

  }

}
