/* Copyright (c) 1999, 2000 by Kevin Forchione.  All Rights Reserved. */
/*
 *  TADS ADV.T/STD.T LIBRARY EXTENSION
 *  SCOPE.T				
 *  version 2.0
 *
 *	    scope.t provides an enhancement of the "visibility" and 
 *      "reachability" concepts established in ADV.T. The scope() 
 *      function will return a list of all objects which are valid 
 *      for a vantage for a particular scopelevel.
 *
 *----------------------------------------------------------------------
 *  REQUIREMENTS
 *
 *      + HTML TADS 2.2.6 or later
 *      + Should be #included after ADV.T and STD.T 
 *
 *----------------------------------------------------------------------
 *  IMPORTANT LIBRARY INTERFACE AND MODIFICATION
 *
 *      + Modifies thing isVisible() and verifyRemove() methods
 *      + Modifies deepverb validXoList() and validXo() methods.
 *
 *----------------------------------------------------------------------
 *  COPYRIGHT NOTICE
 *
 *  	You may modify and use this file in any way you want, provided that
 *		if you redistribute modified copies of this file in source form, the
 *   	copies must include the original copyright notice (including this
 *   	paragraph), and must be clearly marked as modified from the original
 *   	version.
 *
 *------------------------------------------------------------------------------
 *  REVISION HISTORY
 *
 *		07-May-99:	Creation.
 *      13-May-99:  Added ceiling limit to scope()
 *                  Fixed bug involving selection of floatingItem
 *                  Fixed bug with scopeCeiling() so that top level
 *                      locations return themselves.
 *                  Simplified isScope() logic by using find command.
 *      14-May-99:  Broke out scoping verbs into a separate file
 *                      scopedbg.t
 *					Added special scopelist to thing and the 
 *                      addToScope() function to add objects in 
 *                      this list to scope.
 *      20-May-99:  Again, fixed bug involving selection of
 *                      floatingItem.
 */
 
#define __SCOPE_MODULE_

#define SCOPE_OBJTREE  1
#define SCOPE_VISIBLE   2
#define SCOPE_REACHABLE 3

scope: function;
scopeCeiling: function;
scopeList: function;
addToScope: function;
isScopedef: function;
inScope: function;

/*
 *	scope: function( vantage, limit, level )
 *
 *	Function returns a list of all objects that are valid for a given vantage
 *	and scope level. 
 *
 *	First it determines the scopeCeiling for a given vantage and scope level. 
 *	This becomes the first entry in the list. 
 *
 *  limit can be useful in cases where the author wishes to limit the
 *  scope to within a select location, even though the scope level might
 *  extend beyond the location.
 *
 *	Next it calls scopeList with the location, vantage, scope ceiling, and 
 *	scope level and adds the return to the list. 
 *
 *	Next each object in global.floatingList is checked using scopeList and 
 *	the results are added to the list.
 *
 *	Finally, the special vantage.scopelist is added to scope.
 */
scope: function( vantage, limit, level )
{
    local c, o, loc, ceiling, lst := [];
 
    ceiling := scopeCeiling( vantage, limit, level );
        
    loc := ceiling;

    lst += ceiling;
    lst += scopeList( loc, vantage, ceiling, level );

    /*
     *  Each floatingItem object is evaluated. If it has the same
     *  scope ceiling as the vantage or its scope ceiling is the
     *  vantage then it is added to scope along with any appropriate
     *  contents.
     */
	c := global.floatingList;
	o := car( c );
	while( o )
	{
		loc := o;
		ceiling := scopeCeiling( o, limit, level );
		if ( ceiling = lst[1] or ceiling = vantage )
		{
			lst += o;
			lst += scopeList( loc, o, ceiling, level );
		}
		
		c := cdr( c );
		o := car( c );
	}
	
	if ( level <> SCOPE_OBJTREE )
		lst := addToScope( lst, vantage );
    
    return lst;
}

/* 
 *	scopeCeiling: function( vantage, limit, level )
 *
 *	Function returns the scopeCeiling for a given vantage and scope level. The
 *	scopeCeiling is the highest level location that the vantage has access to
 *	for the given scope level.
 *
 *  limit acts to constrain the function a given scope ceiling "height",
 *  but the function will always return a valid scope ceiling for a
 *  valid location.
 *
 */
scopeCeiling: function( vantage, limit, level )
{
    local loc;
    
    if (vantage = limit )
    	return vantage;
    	
    if (vantage.location)
    	loc := vantage.location;
    else
    	loc := vantage;
    while( loc and loc.location ) 
   	{
   	    if ( loc = limit )
   	        break;
   	        
   	    if ( level = SCOPE_VISIBLE 
   	    and not isScopedef( loc, level ))
            break;
            
        if ( level = SCOPE_REACHABLE 
        and not isScopedef( loc, level ))
            break;
            
        loc := loc.location;
    }
    return loc;
}

/*
 *  scopeList: function( loc, vantage, ceiling, level )
 *
 *  Function returns a list of all valid objects for a given vantage, scope
 *	ceiling and scope level. It recurses down each object in the location's
 *	contents.
 */
scopeList: function( loc, vantage, ceiling, level )
{
    local ret := [];
    local i, lst, len;

    /* don't look in "nil" objects */
    if (loc = nil)
        return ret;

	/*
	 *	The location is added to the list if,
	 *		a. it is the vantage or the scope ceiling
	 *		b. it meets the requirements of a scope definition for the given
	 *		   scope level.
	 */
	if (loc = vantage 
	or  loc = ceiling
	or  isScopedef( loc, level ))
    {
        lst := loc.contents;
        len := length(lst);
        ret += lst;
        for (i := 1 ; i <= len ; ++i)
        {
            /* recurse into this object */
      		ret += scopeList( lst[i], vantage, ceiling, level );
        }
    }

    return ret;
}

/*
 *  addToScope: function( scope, vantage )
 *
 *  Function adds special vantage scopelist objects to the scope. This function
 *	provides an alternative mechanism to the normal library's "bubble-up/filter-
 *	down" accessibility mechanisms. An object in the vantage's scopelist will be
 *	added to scope regardless of the verb. This mechanism can be used the same
 *	way validActor is used, except that the vantage need not be the actor.
 *
 *	Function ensures that the object to be added isn't already part of scope. If
 *	it is not then it will be added. 
 */
addToScope: function( scope, vantage )
{
    local i, len := length(vantage.scopelist);
    
    if (len)
    {
        for (i := 1; i <= len; ++i)
        {
            if (not find(scope, vantage.scopelist[i]))
            {
                scope += vantage.scopelist[i];
            }
        }
    }
    return scope;
}

/*
 *  isScopedef: function( loc, level )
 *
 *  Function returns true if the location fits the specified scope
 *  level definition; otherwise it returns nil. This allows scope-
 *  related definitions to be maintained and modified in one central
 *  function.
 */
isScopedef: function( loc, level )
{
    if (level = SCOPE_OBJTREE)
        return true;
        
    if (level = SCOPE_VISIBLE
    and (not isclass(loc, openable)
        or (isclass(loc, openable) and loc.isopen)
        or loc.contentsVisible))
            return true;

	if (level = SCOPE_REACHABLE
    and (not isclass(loc, openable)
        or (isclass(loc, openable) and loc.isopen)))
            return true;
    
    return nil;
}

/*
 *  inScope: function( vantage, target, limit, level )
 *
 *  Function returns true if the target is within the scope list
 *  produced for the given scope level for the vantage; otherwise
 *  it returns nil.
 */
inScope: function( vantage, target, limit, level )
{	
	local value := scope( vantage, limit, level );
	if ( find( value, target ) ) 
	    return true;
	return nil;
}

/*----------------------------------------------------------------------
 *  NECESSARY SCOPING CHANGES TO THING AND DEEPVERB OBJECTS
 *--------------------------------------------------------------------*/

/*
 *	Modifications to thing to incorporate scoping mechanisms.
 */
modify thing
	/*
	 *	scopelist is a special list that can be modified in the author's code to
	 *	add objects to scope outside of the normal "bubble-up/filter-down"
	 *	scoping mechanism. Objects added to this list will be in scope
	 *	regardless of the scope level and therefore available for all actions
	 *	which use scope to determine their validXoList() and validXo() methods
	 *	and should therefore be used only for very exceptional cases.
	 *
	 *	scopelist should never include the object itself or anything ordinarily
	 *	within the scope of the object, as these are handled by normal scoping 
	 *	processes and will already be part of scope.
	 */
    scopelist = []
	/*
 	 *	Modifying verifyRemove() to 'bubble up' verGrab() check, but only as
 	 *	high as thescope ceiling. This allows us to grab things when we're
 	 *	inside closed locations, such as enterables.
 	 */
    replace verifyRemove(actor) =
    {
        /*
         *   Check with each container to make sure that the container
         *   doesn't object to the object's removal.
         */
        local loc, ceiling;

		ceiling := scopeCeiling( actor, nil, SCOPE_REACHABLE );

        loc := self.location;
        while (loc)
        {
        	if (loc = ceiling)
      			break;
            if (loc <> actor)
                loc.verGrab(self);
            loc := loc.location;
        }
    }
    /*
     *	isVisible() now incorporates scope mechanism.
     */
    isVisible(vantage) =
    {
        return( inScope( vantage, self, nil, SCOPE_VISIBLE ) );
    }
;

/*
 *	Modifying deepverb to use the new SCOPING algorithms.
 */
modify deepverb
    replace validDo(actor, obj, seqno) =
    {
        return inScope( actor, obj, nil, SCOPE_REACHABLE );
    }
    replace validDoList(actor, prep, iobj) =
    {
        return scope( actor, nil, SCOPE_VISIBLE );
    }
    replace validIo(actor, obj, seqno) =
    {
        return inScope( actor, obj, nil, SCOPE_REACHABLE );
    }
;

/*
 *	Modifying inspectVerb to use the new SCOPING algorithms.
 */
modify inspectVerb
    replace validDo(actor, obj, seqno) =
    {
        return inScope( actor, obj, nil, SCOPE_VISIBLE );
    }
;
