#pragma C-

/*
 *  Actor: fixeditem, movableActor
 *
 *  A character in the game.  The maxweight property specifies
 *  the maximum weight that the character can carry, and the maxbulk
 *  property specifies the maximum bulk the character can carry.  The
 *  actorAction(verb, directObject, preposition, indirectObject)
 *  method specifies what happens when the actor is given a command by
 *  the player; by default, the actor ignores the command and displays
 *  a message to this effect.  The isCarrying(object)
 *  method returns true if the object is being carried by
 *  the actor.  The actorDesc method displays a message when the
 *  actor is in the current room; this message is displayed along with
 *  a room's description when the room is entered or examined.  The
 *  verGrab(object) method is called when someone tries to
 *  take an object the actor is carrying; by default, an actor won't
 *  let other characters take its possessions.
 *  
 *  If you want the player to be able to follow the actor when it
 *  leaves the room, you should define a follower object to shadow
 *  the character, and set the actor's myfollower property to
 *  the follower object.  The follower is then automatically
 *  moved around just behind the actor by the actor's moveInto
 *  method.
 *  
 *  The isHim property should return true if the actor can
 *  be referred to by the player as "him," and likewise isHer
 *  should be set to true if the actor can be referred to as "her."
 *  Note that both or neither can be set; if neither is set, the actor
 *  can only be referred to as "it," and if both are set, any of "him,''
 *  "her," or "it'' will be accepted.
 */

//ACTUR
class Actor: fixeditem, movableActor
;

/*
 *  basicMe: Actor
 *
 *  A default implementation of the Me object, which is the
 *  player character.  adv.t defines basicMe instead of
 *  Me to allow your game to override parts of the default
 *  implementation while still using the rest, and without changing
 *  adv.t itself.  To use basicMe unchanged as your player
 *  character, include this in your game:  "Me: basicMe;".
 *  
 *  The basicMe object defines all of the methods and properties
 *  required for an actor, with appropriate values for the player
 *  character.  The nouns "me" and "myself'' are defined ("I''
 *  is not defined, because it conflicts with the "inventory"
 *  command's minimal abbreviation of "i" in certain circumstances,
 *  and is generally not compatible with the syntax of most player
 *  commands anyway).  The sdesc is "you"; the thedesc
 *  and adesc are "yourself," which is appropriate for most
 *  contexts.  The maxbulk and maxweight properties are
 *  set to 10 each; a more sophisticated Me might include the
 *  player's state of health in determining the maxweight and
 *  maxbulk properties.
 */

//BASME
class basicMe: Actor, floatingItem
    roomCheck( v ) = { return( self.location.roomCheck( v )); }
    noun = 'me' 'myself'
    sdesc = "you"
    thedesc = "yourself"
    adesc = "yourself"
    ldesc = "You look about the same as always. "
    maxweight = 10
    maxbulk = 10
    verDoFollow( actor ) =
    {
        if ( actor = self ) "You can't follow yourself! ";
    }
    actorAction( verb, dobj, prep, iobj ) = 
    {
    }
    travelTo( room ) =
    {
        if ( room )
        {
	    if ( room.isobstacle )
	    {
	        self.travelTo( room.destination );
	    }
	    else if ( not ( self.location.islit or room.islit ))
	    {
	        darkTravel();
	    }
	    else
	    {
                if ( self.location ) self.location.leaveRoom( self );
                self.location := room;
                room.enterRoom( self );
	    }
        }
    }
    moveInto( room ) =
    {
        self.location := room;
    }
    ioGiveTo(actor, dobj) =
    {
	"You accept <<dobj.thedesc>> from <<actor.thedesc>>.";
	dobj.moveInto(Me);
    }

    // these properties are for the format strings
    fmtYou = "you"
    fmtYour = "your"
    fmtYoure = "you're"
    fmtYoum = "you"
    fmtYouve = "you've"
    fmtS = ""
    fmtEs = ""
    fmtHave = "have"
    fmtDo = "do"
    fmtAre = "are"
    fmtMe = "me"
;

/*
 *  basicNumObj: object
 *
 *  This object provides a default implementation for numObj.
 *  To use this default unchanged in your game, include in your
 *  game this line:  "numObj: basicNumObj".
 */

//BASC#
class basicNumObj: object   // when a number is used in a player command,
    value = 0               //  this is set to its value
    sdesc = "<<value>>"
    adesc = "a number"
    thedesc = "the number <<value>>"
    verDoTypeOn( actor, io ) = {}
    doTypeOn( actor, io ) = { "\"Tap, tap, tap, tap...\" "; }
    verIoTurnTo( actor ) = {}
    ioTurnTo( actor, dobj ) = { dobj.doTurnTo( actor, self ); }
;

/*
 *  basicStrObj: object
 *
 *  This object provides a default implementation for strObj.
 *  To use this default unchanged in your game, include in your
 *  game this line:  "strObj: basicStrObj".
 */
//BASCS
class basicStrObj: object   // when a string is used in a player command,
    value = ''              //  this is set to its value
    sdesc = "\"<<value>>\""
    adesc = "\"<<value>>\""
    thedesc = "\"<<value>>\""
    verDoTypeOn( actor, io ) = {}
    doTypeOn( actor, io ) = { "\"Tap, tap, tap, tap...\" "; }
    doSynonym('TypeOn') = 'EnterOn' 'EnterIn' 'EnterWith'
    verDoSave( actor ) = {}
    saveGame(actor) =
    {
        if (save( self.value ))
	{
            "Save failed. ";
	    return nil;
	}
        else
	{
            "Saved. ";
	    return true;
	}
    }
    doSave( actor ) =
    {
	self.saveGame(actor);
	abort;
    }
    verDoRestore( actor ) = {}
    restoreGame(actor) =
    {
        if (restore( self.value ))
	{
            "Restore failed. ";
	    return nil;
	}
        else
	{
            "Restored.\b";
	    scoreStatus();
	    Me.location.lookAround(true);
	    return true;
	}
    }
    doRestore( actor ) =
    {
	self.restoreGame(actor);
        abort;
    }
    verDoScript( actor ) = {}
    startScripting(actor) =
    {
        logging( self.value );
        "Writing script file. ";
    }
    doScript( actor ) =
    {
	self.startScripting(actor);
        abort;
    }
    verDoSay( actor ) = {}
    doSay( actor ) =
    {
        "Okay, \""; say( self.value ); "\".";
    }
;

/*
 *  beditem: chairitem
 *
 *  This object is the same as a chairitem, except that the player
 *  is described as lying on, rather than sitting in, the object.
 */

//BEDTM
class beditem: chairitem
    ischair = nil
    isbed = true
    sdesc = "bed"
    statusPrep = "on"
    outOfPrep = "out of"
    doLieon(actor) =
    {
        "Okay, %you're% now lying on <<self.thedesc>>.";
	actor.travelTo(self);
    }
;
    
/*
 *  behindHider: hider
 *
 *  This is just like an underHider, except that objects are hidden
 *  behind this object.  Objects to be behind this object should have their
 *  behindLoc property set to point to this object.
 *  
 *  The objects hidden with behindHider must be of class hiddenItem.
 */

//BHIDE
class behindHider: hider
    behindCont = []
    verDoLookbehind(actor) = {}
    doLookbehind(actor) =
    {
	if (self.behindCont = nil)
	    "There's nothing else behind <<self.thedesc>>. ";
	else
	    self.behindCont := self.searchObj(actor, self.behindCont);
    }
;
    
/*
 *  buttonitem: fixeditem
 *
 *  A button (the type you push).  The individual button's action method
 *  doPush(actor), which must be specified in
 *  the button, carries out the function of the button.  Note that
 *  all buttons have the noun "button" defined.
 */

//BTTNS
class buttonitem: fixeditem
    noun = 'button'
    plural = 'buttons'
    verDoPush( actor ) = {}
;

/*
 *  chairitem: fixeditem, nestedroom, surface
 *
 *  Acts like a chair:  actors can sit on the object.  While sitting
 *  on the object, an actor can't go anywhere until standing up, and
 *  can only reach objects that are on the chair and in the chair's
 *  reachable list.  By default, nothing is in the reachable
 *  list.  Note that there is no real distinction made between chairs
 *  and beds, so you can sit or lie on either; the only difference is
 *  the message displayed describing the situation.
 */

//CHRTM
class chairitem: fixeditem, nestedroom, surface
    reachable = ([] + self) // list of all containers reachable from here;
                            //  normally, you can only reach carried items
                            //  from a chair, but this makes special allowances
    ischair = true          // it is a chair by default; for beds or other
                            //  things you lie down on, make it false
    outOfPrep = "out of"
    roomAction( actor, v, dobj, prep, io ) =
    {
        if ( dobj<>nil and v<>inspectVerb )
            checkReach( self, actor, v, dobj );
        if ( io<>nil and v<>askVerb and v<>tellVerb )
            checkReach( self, actor, v, io );
	pass roomAction;
    }
    enterRoom( actor ) = {}
    noexit =
    {
        "%You're% not going anywhere until %you%
        get%s% <<outOfPrep>> <<thedesc>>. ";
        return( nil );
    }
    verDoBoard( actor ) = { self.verDoSiton( actor ); }
    doBoard( actor ) = { self.doSiton( actor ); }
    verDoSiton( actor ) =
    {
        if ( actor.location = self )
        {
            "%You're% already on "; self.thedesc; "! ";
        }
    }
    doSiton( actor ) =
    {
        "Okay, %you're% now sitting on "; self.thedesc; ". ";
        actor.travelTo( self );
    }
    verDoLieon( actor ) =
    {
        self.verDoSiton( actor );
    }
    doLieon( actor ) =
    {
        self.doSiton( actor );
    }
;

/*
 *  clothingItem: item
 *
 *  Something that can be worn.  By default, the only thing that
 *  happens when the item is worn is that its isworn property
 *  is set to true.  If you want more to happen, override the
 *  doWear(actor) property.  Note that, when a clothingItem
 *  is being worn, certain operations will cause it to be removed (for
 *  example, dropping it causes it to be removed).  If you want
 *  something else to happen, override the checkDrop method;
 *  if you want to disallow such actions while the object is worn,
 *  use an exit statement in the checkDrop method.
 */

//CLOTM
class clothingItem: item
    checkDrop =
    {
        if ( self.isworn )
	{
	    "(Taking off "; self.thedesc; " first)\n";
	    self.isworn := nil;
	}
    }
    doDrop( actor ) =
    {
        self.checkDrop;
	pass doDrop;
    }
    doPutIn( actor, io ) =
    {
        self.checkDrop;
	pass doPutIn;
    }
    doPutOn( actor, io ) =
    {
        self.checkDrop;
	pass doPutOn;
    }
    doGiveTo( actor, io ) =
    {
        self.checkDrop;
	pass doGiveTo;
    }
    doThrowAt( actor, io ) =
    {
        self.checkDrop;
	pass doThrowAt;
    }
    doThrowTo( actor, io ) =
    {
        self.checkDrop;
	pass doThrowTo;
    }
    doThrow( actor ) =
    {
        self.checkDrop;
	pass doThrow;
    }
    moveInto( obj ) =
    {
        /*
	 *   Catch any other movements with moveInto; this won't stop the
	 *   movement from happening, but it will prevent any anamolous
	 *   consequences caused by the object moving but still being worn.
	 */
        self.isworn := nil;
	pass moveInto;
    }
    verDoWear( actor ) =
    {
        if ( self.isworn )
        {
            "%You're% already wearing "; self.thedesc; "! ";
        }
        else if ( not actor.isCarrying( self ))
        {
            "%You% %do%n't have "; self.thedesc; ". ";
        }
    }
    doWear( actor ) =
    {
        "Okay, %you're% now wearing "; self.thedesc; ". ";
        self.isworn := true;
    }
    verDoUnwear( actor ) =
    {
        if ( not self.isworn )
        {
            "%You're% not wearing "; self.thedesc; ". ";
        }
    }
    verDoTake(actor) =
    {
        if (self.isworn) self.verDoUnwear(actor);
	else pass verDoTake;
    }
    doTake(actor) =
    {
        if (self.isworn) self.doUnwear(actor);
	else pass doTake;
    }
    doUnwear( actor ) =
    {
        "Okay, %you're% no longer wearing "; self.thedesc; ". ";
        self.isworn := nil;
    }
    doSynonym('Unwear') = 'Unboard'
;

/*
 *  container: item
 *
 *  This object can contain other objects.  The iscontainer property
 *  is set to true.  The default ldesc displays a list of the
 *  objects inside the container, if any.  The maxbulk property
 *  specifies the maximum amount of bulk the container can contain.
 */

//CNTNR
class container: item
    maxbulk = 10            // maximum bulk the container can contain
    isopen = true           // in fact, it can't be closed at all
    iscontainer = true      // Item can contain other items
    ldesc =
    {
        if ( self.contentsVisible and itemcnt( self.contents ) <> 0 )
        {
            "In "; self.thedesc; " %you% see%s% "; listcont( self ); ". ";
        }
        else
        {
            "There's nothing in "; self.thedesc; ". ";
        }
    }
    verIoPutIn( actor ) =
    {
    }
    ioPutIn( actor, dobj ) =
    {
        if (addbulk( self.contents ) + dobj.bulk > self.maxbulk )
        {
            "%You% can't fit that in "; self.thedesc; ". ";
        }
        else
        {
	    dobj.doPutIn( actor, self );
        }
    }
    verDoLookin( actor ) = {}
    doLookin( actor ) =
    {
        self.ldesc;
    }
;

/*
 *  darkroom: room
 *
 *  A dark room.  The player must have some object that can act as a
 *  light source in order to move about and perform most operations
 *  while in this room.  Note that the room's lights can be turned
 *  on by setting the room's lightsOn property to true;
 *  do this instead of setting islit, because islit is
 *  a method which checks for the presence of a light source.
 */

//DRKRM
class darkroom: room        // An enterable area which might be dark
    islit =                 // true ONLY if something is lighting the room
    {
        local rem, cur, tot, i;

	if ( self.lightsOn ) return( true );

	rem := global.lamplist;
	tot := length( rem );
	i := 1;
	while ( i <= tot )
	{
	    cur := rem[i];
	    if ( cur.isIn( self ) and cur.islit ) return( true );
	    i := i + 1;
	}
	return( nil );
    }
    roomAction( actor, v, dobj, prep, io ) =
    {
        if ( not self.islit and not v.isDarkVerb )
	{
	    "%You% can't see a thing. ";
	    exit;
	}
	else pass roomAction;
    }
    statusLine =
    {
        if ( self.islit ) pass statusLine;
	else "In the dark.";
    }
    lookAround( verbosity ) =
    {
        if ( self.islit ) pass lookAround;
	else "It's pitch black. ";
    }
    noexit =
    {
        if ( self.islit ) pass noexit;
	else
	{
	    darkTravel();
	    return( nil );
	}
    }
    roomCheck( v ) =
    {
        if ( self.islit or v.isDarkVerb ) return( true );
	else
	{
	    "It's pitch black.\n";
	    return( nil );
	}
    }
;

/*
 *  decoration: fixeditem
 *
 *  An item that doesn't have any function in the game, apart from
 *  having been mentioned in the room description.  These items
 *  are immovable and can't be manipulated in any way, but can be
 *  referred to and inspected.  Liberal use of decoration items
 *  can improve a game's playability by helping the parser recognize
 *  all the words the game uses in its descriptions of rooms.
 */

//DCR8T
class decoration: fixeditem
    ldesc = "That's not important."
    dobjGen(a, v, i, p) =
    {
        if (v <> inspectVerb)
	{
	    "\^<<self.thedesc>> isn't important.";
	    exit;
	}
    }
    iobjGen(a, v, d, p) =
    {
        "\^<<self.thedesc>> isn't important.";
	exit;
    }
;

/*
 *  dialItem: fixeditem
 *
 *  This class is used for making "dials," which are controls in
 *  your game that can be turned to a range of numbers.  You must
 *  define the property maxsetting as a number specifying the
 *  highest number to which the dial can be turned; the lowest number
 *  on the dial is always 1.  The setting property is the dial's
 *  current setting, and can be changed by the player by typing the
 *  command "turn dial to number."  By default, the ldesc
 *  method displays the current setting.
 */

//DILTM
class dialItem: fixeditem
    maxsetting = 10 // it has settings from 1 to this number
    setting = 1     // the current setting
    ldesc =
    {
        caps(); self.thedesc; " can be turned to settings
        numbered from 1 to << self.maxsetting >>. It's
        currently set to << self.setting >>. ";
    }
    verDoTurn( actor ) = {}
    doTurn( actor ) =
    {
        askio( toPrep );
    }
    verDoTurnTo( actor, io ) = {}
    doTurnTo( actor, io ) =
    {
        if ( io = numObj )
        {
            if ( numObj.value < 1 or numObj.value > self.maxsetting )
            {
                "There's no such setting! ";
            }
            else if ( numObj.value <> self.setting )
            {
                self.setting := numObj.value;
                "Okay, it's now turned to "; say( self.setting ); ". ";
            }
            else
            {
                "It's already set to "; say( self.setting ); "! ";
            }
        }
        else
        {
            "I don't know how to turn "; self.thedesc;
            " to that. ";
        }
    }
;

/*
 *  distantItem: fixeditem
 *
 *  This is an item that is too far away to manipulate, but can be seen.
 *  The class uses dobjGen and iobjGen to prevent any verbs from being
 *  used on the object apart from inspectVerb; using any other verb results
 *  in the message "It's too far away."  Instances of this class should
 *  provide the normal item properties:  sdesc, ldesc, location,
 *  and vocabulary.
 */

//DSTTM
class distantItem: fixeditem
    dobjGen(a, v, i, p) =
    {
        if (v <> inspectVerb)
        {
            "It's too far away.";
            exit;
        }
    }
    iobjGen(a, v, d, p) = { self.dobjGen(a, v, d, p); }
;

/*
 *  doorway: fixeditem, obstacle
 *
 *  A doorway is an obstacle that impedes progress when it is closed.
 *  When the door is open (isopen is true), the user ends up in
 *  the room specified in the doordest property upon going through
 *  the door.  Since a doorway is an obstacle, use the door object for
 *  a direction property of the room containing the door.
 *  
 *  If noAutoOpen is not set to true, the door will automatically
 *  be opened when the player tries to walk through the door, unless the
 *  door is locked (islocked = true).  If the door is locked,
 *  it can be unlocked simply by typing "unlock door", unless the
 *  mykey property is set, in which case the object specified in
 *  mykey must be used to unlock the door.  Note that the door can
 *  only be relocked by the player under the circumstances that allow
 *  unlocking, plus the property islockable must be set to true.
 *  By default, the door is closed; set isopen to true if the door
 *  is to start out open (and be sure to open the other side as well).
 *  
 *  otherside specifies the corresponding doorway object in the
 *  destination room (doordest), if any.  If otherside is
 *  specified, its isopen and islocked properties will be
 *  kept in sync automatically.
 */

//DRWAY
class doorway: fixeditem, obstacle
    isdoor = true           // Item can be opened and closed
    destination =
    {
        if ( self.isopen ) return( self.doordest );
	else if ( not self.islocked and not self.noAutoOpen )
	{
	    self.isopen := true;
	    if ( self.otherside ) self.otherside.isopen := true;
	    "(Opening << self.thedesc >>)\n";
	    return( self.doordest );
	}
	else
	{
	    "%You%'ll have to open << self.thedesc >> first. ";
	    setit( self );
	    return( nil );
	}
    }
    verDoOpen( actor ) =
    {
        if ( self.isopen ) "It's already open. ";
	else if ( self.islocked ) "It's locked. ";
    }
    doOpen( actor ) =
    {
        "Opened. ";
	self.isopen := true;
	if ( self.otherside ) self.otherside.isopen := true;
    }
    verDoClose( actor ) =
    {
        if ( not self.isopen ) "It's already closed. ";
    }
    doClose( actor ) =
    {
        "Closed. ";
	self.isopen := nil;
	if ( self.otherside ) self.otherside.isopen := nil;
    }
    verDoLock( actor ) =
    {
        if ( self.islocked ) "It's already locked! ";
	else if ( not self.islockable ) "It can't be locked. ";
	else if ( self.isopen ) "%You%'ll have to close it first. ";
    }
    doLock( actor ) =
    {
        if ( self.mykey = nil )
	{
	    "Locked. ";
	    self.islocked := true;
	    if ( self.otherside ) self.otherside.islocked := true;
	}
	else
            askio( withPrep );
    }
    verDoUnlock( actor ) =
    {
        if ( not self.islocked ) "It's not locked! ";
    }
    doUnlock( actor ) =
    {
        if ( self.mykey = nil )
	{
	    "Unlocked. ";
	    self.islocked := nil;
	    if ( self.otherside ) self.otherside.islocked := nil;
	}
	else
	    askio( withPrep );
    }
    verDoLockWith( actor, io ) =
    {
        if ( self.islocked ) "It's already locked. ";
	else if ( not self.islockable ) "It can't be locked. ";
	else if ( self.mykey = nil )
	    "%You% %do%n't need anything to lock it. ";
	else if ( self.isopen ) "%You%'ll have to close it first. ";
    }
    doLockWith( actor, io ) =
    {
        if ( io = self.mykey )
	{
	    "Locked. ";
	    self.islocked := true;
	    if ( self.otherside ) self.otherside.islocked := true;
	}
	else "It doesn't fit the lock. ";
    }
    verDoUnlockWith( actor, io ) =
    {
        if ( not self.islocked ) "It's not locked! ";
	else if ( self.mykey = nil )
	    "%You% %do%n't need anything to unlock it. ";
    }
    doUnlockWith( actor, io ) =
    {
        if ( io = self.mykey )
	{
	    "Unlocked. ";
	    self.islocked := nil;
	    if ( self.otherside ) self.otherside.islocked := nil;
	}
	else "It doesn't fit the lock. ";
    }
    ldesc =
    {
	if ( self.isopen ) "It's open. ";
	else
	{
	    if ( self.islocked ) "It's closed and locked. ";
	    else "It's closed. ";
	}
    }
;

/*
 *  fixeditem: thing
 *
 *  An object that cannot be taken or otherwise moved from its location.
 *  Note that a fixeditem is sometimes part of a movable object;
 *  this can be done to make one object part of another, ensuring that
 *  they cannot be separated.  By default, the functions that list a room's
 *  contents do not automatically describe fixeditem objects (because
 *  the isListed property is set to nil).  Instead, the game author
 *  will generally describe the fixeditem objects separately as part of
 *  the room's ldesc.  
 */

//FIXTM
class fixeditem: thing      // An immovable object
    isListed = nil          // not listed in room/inventory displays
    isfixed = true          // Item can't be taken
    weight = 0              // no actual weight
    bulk = 0
    verDoTake( actor ) =
    {
        "%You% can't have "; self.thedesc; ". ";
    }
    verDoTakeOut( actor, io ) =
    {
        self.verDoTake( actor );
    }
    verDoDrop( actor ) =
    {
        "%You% can't drop "; self.thedesc; ". ";
    }
    verDoTakeOff( actor, io ) =
    {
        self.verDoTake( actor );
    }
    verDoPutIn( actor, io ) =
    {
        "%You% can't put "; self.thedesc; " anywhere. ";
    }
    verDoPutOn( actor, io ) =
    {
        "%You% can't put "; self.thedesc; " anywhere. ";
    }
    verDoMove( actor ) =
    {
        "%You% can't move "; self.thedesc; ". ";
    }
    verDoThrowAt(actor, iobj) =
    {
        "%You% can't throw <<self.thedesc>>.";
    }
;

/*
 *  floatingItem: object
 *
 *  This class doesn't do anything apart from mark an object as having a
 *  variable location property.  It is necessary to mark all such
 *  items by making them a member of this class, so that the objects are
 *  added to global.floatingList, which is necessary so that floating
 *  objects are included in validDoList and validIoList values (see
 *  the deepverb class for a description of these methods).
 */

//FLOTM
class floatingItem: object
;

/*
 *  fooditem: item
 *
 *  An object that can be eaten.  When eaten, the object is removed from
 *  the game, and global.lastMealTime is decremented by the
 *  foodvalue property.  By default, the foodvalue property
 *  is global.eatTime, which is the time between meals.  So, the
 *  default fooditem will last for one "nourishment interval."
 */

//FOOTM
class fooditem: item
    verDoEat( actor ) =
    {
        self.verifyRemove( actor );
    }
    doEat( actor ) =
    {
        "That was delicious! ";
        global.lastMealTime := global.lastMealTime - self.foodvalue;
        self.moveInto( nil );
    }
    foodvalue = { return( global.eatTime ); }
;

/*
 *  follower: Actor
 *
 *  This is a special object that can "shadow" the movements of a
 *  character as it moves from room to room.  The purpose of a follower
 *  is to allow the player to follow an actor as it leaves a room by
 *  typing a "follow" command.  Each actor that is to be followed must
 *  have its own follower object.  The follower object should
 *  define all of the same vocabulary words (nouns and adjectives) as the
 *  actual actor to which it refers.  The follower must also
 *  define the myactor property to be the Actor object that
 *  the follower follows.  The follower will always stay
 *  one room behind the character it follows; no commands are effective
 *  with a follower except for "follow."
 */

//FLLWR
class follower: Actor
    sdesc = { self.myactor.sdesc; }
    isfollower = true
    ldesc = { caps(); self.thedesc; " is no longer here. "; }
    actorAction( v, d, p, i ) = { self.ldesc; exit; }
    actorDesc = {}
    myactor = nil   // set to the Actor to be followed
    verDoFollow( actor ) = {}
    doFollow( actor ) =
    {
        actor.travelTo( self.myactor.location );
    }
    dobjGen(a, v, i, p) =
    {
        if (v <> followVerb)
	{
	    "\^<< self.myactor.thedesc >> is no longer here.";
	    exit;
	}
    }
    iobjGen(a, v, d, p) =
    {
        "\^<< self.myactor.thedesc >> is no longer here.";
	exit;
    }
;

/*
 *   The "global" object is the dumping ground for any data items that
 *   don't fit very well into any other objects.  The properties of this
 *   object that are particularly important to the objects and functions
 *   are defined here; if you replace this object, but keep other parts
 *   of this file, be sure to include the properties defined here.
 */

//GLOBL
global: object
    turnsofar = 0                            // no turns have transpired so far
    score = 0                            // no points have been accumulated yet
    maxscore = 100                                    // maximum possible score
    verbose = nil                             // we are currently in TERSE mode
    start = startroom		// Unless you change this, player starts here.
    lamplist = []              // list of all known light providers in the game
;

/*
 *  hiddenItem: object
 *
 *  This is an object that is hidden with one of the hider classes. 
 *  A hiddenItem object doesn't have any special properties in its
 *  own right, but all objects hidden with one of the hider classes
 *  must be of class hiddenItem so that initSearch can find
 *  them.
 */

//HIDTM
class hiddenItem: object
;

/*
 *  hider: item
 *
 *  This is a basic class of object that can hide other objects in various
 *  ways.  The underHider, behindHider, and searchHider classes
 *  are examples of hider subclasses.  The class defines
 *  the method searchObj(actor, list), which is given the list
 *  of hidden items contained in the object (for example, this would be the
 *  underCont property, in the case of an underHider), and "finds"
 *  the object or objects. Its action is dependent upon a couple of other
 *  properties of the hider object.  The serialSearch property,
 *  if true, indicates that items in the list are to be found one at
 *  a time; if nil (the default), the entire list is found on the
 *  first try.  The autoTake property, if true, indicates that
 *  the actor automatically takes the item or items found; if nil, the
 *  item or items are moved to the actor's location.  The searchObj method
 *  returns the list with the found object or objects removed; the
 *  caller should assign this returned value back to the appropriate
 *  property (for example, underHider will assign the return value
 *  to underCont).
 *  
 *  Note that because the hider is hiding something, this class
 *  overrides the normal verDoSearch method to display the
 *  message, "You'll have to be more specific about how you want
 *  to search that."  The reason is that the normal verDoSearch
 *  message ("You find nothing of interest") leads players to believe
 *  that the object was exhaustively searched, and we want to avoid
 *  misleading the player.  On the other hand, we don't want a general
 *  search to be exhaustive for most hider objects.  So, we just
 *  display a message letting the player know that the search was not
 *  enough, but we don't give away what they have to do instead.
 *  
 *  The objects hidden with one of the hider classes must be
 *  of class hiddenItem.
 */

//HEYED
class hider: item
    verDoSearch(actor) =
    {
	"%You%'ll have to be more specific about how %you% want%s%
	to search that. ";
    }
    searchObj(actor, list) =
    {
	local found, dest, i, tot;

	/* see how much we get this time */
	if (self.serialSearch)
	{
	    found := [] + car(list);
	    list := cdr(list);
	}
	else
	{
	    found := list;
	    list := nil;
	}

	/* set it(them) to the found item(s) */
        if (length(found) = 1)
	    setit(found[1]);    // only one item - set 'it'
	else
	    setit(found);       // multiple items - set 'them'

	/* figure destination */
	dest := actor;
	if (not self.autoTake) dest := dest.location;
	
	/* note what we found, and move it to destination */
	"%You% find%s% ";
	tot := length(found);
	i := 1;
	while (i <= tot)
	{
	    found[i].adesc;
	    if (i+1 < tot) ", ";
	    else if (i = 1 and tot = 2) " and ";
	    else if (i+1 = tot and tot > 2) ", and ";
	    
	    found[i].moveInto(dest);
	    i := i + 1;
	}

	/* say what happened */
	if (self.autoTake) ", which %you% take%s%. ";
	else "! ";

	if (list<>nil and length(list)=0) list := nil;
	return(list);
    }
    serialSearch = nil             /* find everything in one try by default */
    autoTake = true               /* actor takes item when found by default */
;

/*
 *  item: thing
 *
 *  A basic item which can be picked up by the player.  It has no weight
 *  (0) and minimal bulk (1).  The weight property should be set
 *  to a non-zero value for heavy objects.  The bulk property
 *  should be set to a value greater than 1 for bulky objects, and to
 *  zero for objects that are very small and take essentially no effort
 *  to hold---or, more precisely, don't detract at all from the player's
 *  ability to hold other objects (for example, a piece of paper).
 */

//EYETM
class item: thing
    weight = 0
    bulk = 1
;
    
/*
 *  keyedLockable: lockable
 *
 *  This subclass of lockable allows you to create an object
 *  that can only be locked and unlocked with a corresponding key.
 *  Set the property mykey to the keyItem object that can
 *  lock and unlock the object.
 */

//KYLOK
class keyedLockable: lockable
    mykey = nil     // set 'mykey' to the key which locks/unlocks me
    doLock( actor ) =
    {
        askio( withPrep );
    }
    doUnlock( actor ) =
    {
        askio( withPrep );
    }
    doLockWith( actor, io ) =
    {
        if ( self.isopen )
        {
            "%You% can't lock << self.thedesc >> when it's open. ";
        }
        else if ( io = self.mykey )
        {
            "Locked. ";
            self.islocked := true;
        }
        else "It doesn't fit the lock. ";
    }
    doUnlockWith( actor, io ) =
    {
        if ( io = self.mykey )
        {
            "Unlocked. ";
            self.islocked := nil;
        }
        else "It doesn't fit the lock. ";
    }
;

/*
 *  keyItem: item
 *
 *  This is an object that can be used as a key for a keyedLockable
 *  or lockableDoorway object.  It otherwise behaves as an ordinary item.
 */

//KEYTM
class keyItem: item
    verIoUnlockWith( actor ) = {}
    ioUnlockWith( actor, dobj ) =
    {
        dobj.doUnlockWith( actor, self );
    }
    verIoLockWith( actor ) = {}
    ioLockWith( actor, dobj ) =
    {
        dobj.doLockWith( actor, self );
    }
;

/*
 *  lightsource: item
 *
 *  A portable lamp, candle, match, or other source of light.  The
 *  light source can be turned on and off with the islit property.
 *  If islit is true, the object provides light, otherwise it's
 *  just an ordinary object.  Note that this object provides a doTurnon
 *  method to provide appropriate behavior for a switchable light source,
 *  such as a flashlight or a room's electric lights.  However, this object
 *  does not provide a verDoTurnon method, so by default it can't be
 *  switched on and off.  To create something like a flashlight that should
 *  be a lightsource that can be switched on and off, simply include both
 *  lightsource and switchItem in the superclass list, and be sure
 *  that lightsource precedes switchItem in the superclass list,
 *  because the doTurnon method provided by lightsource should
 *  override the one provided by switchItem.  The doTurnon method
 *  provided here turns on the light source (by setting its isActive
 *  property to true, and then describes the room if it was previously
 *  dark.
 */

//LITSC
class lightsource: item
    islamp = true
    doTurnon(actor) =
    {
	local waslit := actor.location.islit;

	// turn on the light
	self.isActive := true;
	"You switch on <<thedesc>>";

	// if the room wasn't previously lit, and it is now, describe it
	if (actor.location.islit and not waslit)
	{
	    ", lighting the area.\b";
	    actor.location.enterRoom(actor);
	}
	else
	    ".";
    }
;

/*
 *  lockable: openable
 *
 *  A container that can be locked and unlocked.  The islocked
 *  property specifies whether the object can be opened or not.  The
 *  object can be locked and unlocked without the need for any other
 *  object; if you want a key to be involved, use a keyedLockable.
 */

//LOKBL
class lockable: openable
    verDoOpen( actor ) =
    {
        if ( self.islocked )
        {
            "It's locked. ";
        }
        else pass verDoOpen;
    }
    verDoLock( actor ) =
    {
        if ( self.islocked )
        {
            "It's already locked! ";
        }
    }
    doLock( actor ) =
    {
        if ( self.isopen )
        {
            "%You%'ll have to close "; self.thedesc; " first. ";
        }
        else
        {
            "Locked. ";
            self.islocked := true;
        }
    }
    verDoUnlock( actor ) =
    {
        if ( not self.islocked ) "It's not locked! ";
    }
    doUnlock( actor ) =
    {
        "Unlocked. ";
        self.islocked := nil;
    }
    verDoLockWith( actor, io ) =
    {
        if ( self.islocked ) "It's already locked. ";
    }
    verDoUnlockWith( actor, io ) =
    {
        if ( not self.islocked ) "It's not locked! ";
    }
;

/*
 *  lockableDoorway: doorway
 *
 *  This is just a normal doorway with the islockable and
 *  islocked properties set to true.  Fill in the other
 *  properties (otherside and doordest) as usual.  If
 *  the door has a key, set property mykey to the key object.
 */

//LOKDR
class lockableDoorway: doorway
    islockable = true
    islocked = true
;

/*
 *   "me" is the player's actor.  Pick up the default definition, basicMe,
 *   from "adv.t".
 */

//MEEEE
Me: basicMe
;

/*
 *  movableActor: qcontainer
 *
 *  Just like an Actor object, except that the player can
 *  manipulate the actor like an ordinary item.  Useful for certain
 *  types of actors, such as small animals.
 */

//MVACT
class movableActor: qcontainer // A character in the game
    isListed = nil          // described separately from room's contents
    weight = 10             // actors are pretty heavy
    bulk = 10               // and pretty bulky
    maxweight = 50          // Weight that can be carried at once
    maxbulk = 20            // Number of objects that can be carried at once
    isactor = true          // flag that this is an actor
    roomCheck( v ) = { return( self.location.roomCheck(v)); }
    actorAction( v, d, p, i ) =
    {
        caps(); self.thedesc; " doesn't appear interested. ";
        exit;
    }
    isCarrying( obj ) = { return( obj.isIn( self )); }
    actorDesc =
    {
        caps(); self.adesc; " is here. ";
    }
    verGrab( item ) =
    {
        caps(); self.thedesc; " is carrying "; item.thedesc;
        " and won't let %youm% have it. ";
    }
    verDoFollow( actor ) =
    {
        "But "; self.thedesc; " is right here! ";
    }
    moveInto( obj ) =
    {
        if ( self.myfollower ) self.myfollower.moveInto( self.location );
	pass moveInto;
    }
    // these properties are for the format strings
    fmtYou = "he"
    fmtYour = "his"
    fmtYoure = "he's"
    fmtYoum = "him"
    fmtYouve = "he's"
    fmtS = "s"
    fmtEs = "es"
    fmtHave = "has"
    fmtDo = "does"
    fmtAre = "is"
    fmtMe = { self.thedesc; }
    askWord(word, lst) = { return(nil); }
    verDoAskAbout(actor, iobj) = {}
    doAskAbout(actor, iobj) =
    {
	local lst, i, tot;

	lst := objwords(2);       // get actual words asked about
	tot := length(lst);
	if ((tot = 1 and (find(['it' 'them' 'him' 'her'], lst[1]) <> nil))
	    or tot = 0)
	{
	    "\"Could you be more specific?\"";
	    return;
	}

	// try to find a response for each word
	for (i := 1 ; i <= tot ; ++i)
	{
	    if (self.askWord(lst[i], lst))
	        return;
        }

	// didn't find anything to talk about
	self.disavow;
    }
    disavow = "\"I don't know much about that.\""
    verIoPutIn(actor) =
    {
        "If you want to give that to << thedesc >>, just say so.";
    }
    verIoGiveTo(actor) =
    {
	if (actor = self)
	    "That wouldn't accomplish anything!";
    }
    ioGiveTo(actor, dobj) =
    {
	"\^<<self.thedesc>> rejects the offer.";
    }

    // move to a new location, notifying player of coming and going
    travelTo(room) =
    {
	/* do nothing if going nowhere */
	if (room = nil) return;
	
        /* notify player if leaving player's location (and it's not dark) */
        if (self.location = Me.location and self.location.islit)
            self.sayLeaving;

        /* move to my new location */
        self.moveInto(room);

        /* notify player if arriving at player's location */
        if (self.location = Me.location and self.location.islit)
            self.sayArriving;
    }

    // sayLeaving and sayArriving announce the actor's departure and arrival
    // in the same room as the player.
    sayLeaving = "\n\t\^<<self.thedesc>> leaves the area."
    sayArriving = "\n\t\^<<self.thedesc>> enters the area."

    // this should be used as an actor when ambiguous
    preferredActor = true
;

/*
 *  nestedroom: room
 *
 *  A special kind of room that is inside another room; chairs and
 *  some types of vehicles, such as inflatable rafts, fall into this
 *  category.  Note that a room can be within another room without
 *  being a nestedroom, simply by setting its location property
 *  to another room.  The nestedroom is different from an ordinary
 *  room, though, in that it's an "open" room; that is, when inside it,
 *  the actor is still really inside the enclosing room for purposes of
 *  descriptions.  Hence, the player sees "Laboratory, in the chair."
 *  In addition, a nestedroom is an object in its own right,
 *  visible to the player; for example, a chair is an object in a
 *  room in addition to being a room itself.  The statusPrep
 *  property displays the preposition in the status description; by
 *  default, it will be "in," but some subclasses and instances
 *  will want to change this to a more appropriate preposition.
 *  outOfPrep is used to report what happens when the player
 *  gets out of the object:  it should be "out of" or "off of" as
 *  appropriate to this object.
 */

//NROOM
class nestedroom: room
    statusPrep = "in"
    outOfPrep = "out of"
    islit =
    {
        if ( self.location ) return( self.location.islit );
        return( nil );
    }
    statusLine =
    {
	"<<self.location.sdesc>>, <<self.statusPrep>> <<self.thedesc>>\n\t";
    }
    lookAround( verbosity ) =
    {
        self.statusLine;
	self.location.nrmLkAround( verbosity );
    }
    roomDrop( obj ) =
    {
        if ( self.location = nil or self.isdroploc ) pass roomDrop;
	else self.location.roomDrop( obj );
    }
;

/*
 *   The numobj object is used to convey a number to the game whenever
 *   the player uses a number in his command.  For example, "turn dial
 *   to 621" results in an indirect object of numobj, with its "value"
 *   property set to 621.
 */

//NUMBJ
numObj: basicNumObj  // use default definition from adv.t
;

/*
 *  obstacle: object
 *
 *  An obstacle is used in place of a room for a direction
 *  property.  The destination property specifies the room that
 *  is reached if the obstacle is successfully negotiated; when the
 *  obstacle is not successfully negotiated, destination should
 *  display an appropriate message and return nil.
 */

//OBSTC
class obstacle: object
    isobstacle = true
;

/*
 *  openable: container
 *
 *  A container that can be opened and closed.  The isopenable
 *  property is set to true.  The default ldesc displays
 *  the contents of the container if the container is open, otherwise
 *  a message saying that the object is closed.
 */

//OPENB
class openable: container
    contentsReachable = { return( self.isopen ); }
    contentsVisible = { return( self.isopen ); }
    isopenable = true
    ldesc =
    {
    	caps(); self.thedesc;
    	if ( self.isopen )
	{
	    " is open. ";
	    pass ldesc;
	}
    	else
	{
	    " is closed. ";

	    /* if it's transparent, list its contents anyway */
	    if (isclass(self, transparentItem)) pass ldesc;
	}
    }
    isopen = true
    verDoOpen( actor ) =
    {
        if ( self.isopen )
	{
	    caps(); self.thedesc; " is already open! ";
	}
    }
    doOpen( actor ) =
    {
        if (itemcnt( self.contents ))
	{
	    "Opening "; self.thedesc; " reveals "; listcont( self ); ". ";
	}
	else "Opened. ";
	self.isopen := true;
    }
    verDoClose( actor ) =
    {
        if ( not self.isopen )
	{
	    caps(); self.thedesc; " is already closed! ";
	}
    }
    doClose( actor ) =
    {
        "Closed. ";
	self.isopen := nil;
    }
    verIoPutIn( actor ) =
    {
        if ( not self.isopen )
	{
	    caps(); self.thedesc; " is closed. ";
	}
    }
    verDoLookin( actor ) =
    {
        /* we can look in it if either it's open or it's transparent */
        if (not self.isopen and not isclass(self, transparentItem))
           "It's closed. ";
    }
;

/*
 *  qcontainer: container
 *
 *  A "quiet" container:  its contents are not listed when it shows
 *  up in a room description or inventory list.  The isqcontainer
 *  property is set to true.
 */

//QCNTR
class qcontainer: container
    isqcontainer = true
;

/*
 *  readable: item
 *
 *  An item that can be read.  The readdesc property is displayed
 *  when the item is read.  By default, the readdesc is the same
 *  as the ldesc, but the readdesc can be overridden to give
 *  a different message.
 */

//REDTM
class readable: item
    verDoRead( actor ) =
    {
    }
    doRead( actor ) =
    {
        self.readdesc;
    }
    readdesc =
    {
        self.ldesc;
    }
;

/*
 *  room: thing
 *
 *  A location in the game.  By default, the islit property is
 *  true, which means that the room is lit (no light source is
 *  needed while in the room).  You should create a darkroom
 *  object rather than a room with islit set to nil if you
 *  want a dark room, because other methods are affected as well.
 *  The isseen property records whether the player has entered
 *  the room before; initially it's nil, and is set to true
 *  the first time the player enters.  The roomAction(actor,
 *  verb, directObject, preposition, indirectObject) method is
 *  activated for each player command; by default, all it does is
 *  call the room's location's roomAction method if the room
 *  is inside another room.  The lookAround(verbosity)
 *  method displays the room's description for a given verbosity
 *  level; true means a full description, nil means only
 *  the short description (just the room name plus a list of the
 *  objects present).  roomDrop(object) is called when
 *  an object is dropped within the room; normally, it just moves
 *  the object to the room and displays "Dropped."  The firstseen
 *  method is called when isseen is about to be set true
 *  for the first time (i.e., when the player first sees the room);
 *  by default, this routine does nothing, but it's a convenient
 *  place to put any special code you want to execute when a room
 *  is first entered.  The firstseen method is called after
 *  the room's description is displayed.
 */

//RUMES
class room: thing
    /*
     *   'reachable' is the list of explicitly reachable objects, over and
     *   above the objects that are here.  This is mostly used in nested
     *   rooms to make objects in the containing room accessible.  Most
     *   normal rooms will leave this as an empty list.
     */
    reachable = []
	
    /*
     *   roomCheck is true if the verb is valid in the room.  This
     *   is a first pass; generally, its only function is to disallow
     *   certain commands in a dark room.
     */
    roomCheck( v ) = { return( true ); }
    islit = true            // rooms are lit unless otherwise specified
    isseen = nil            // room has not been seen yet
    enterRoom( actor ) =    // sent to room as actor is entering it
    {
        self.lookAround(( not self.isseen ) or global.verbose );
        if ( self.islit )
	{
	    if (not self.isseen) self.firstseen;
	    self.isseen := true;
	}
    }
    roomAction( a, v, d, p, i ) =
    {
        if ( self.location ) self.location.roomAction( a, v, d, p, i );
    }

    /*
     *   Whenever a normal object (i.e., one that does not override the
     *   default doDrop provided by 'thing') is dropped, the actor's
     *   location is sent roomDrop(object being dropped).  By default, 
     *   we move the object into this room.
     */
    roomDrop( obj ) =
    {
        "Dropped. ";
	obj.moveInto( self );
    }

    /*
     *   Whenever an actor leaves this room, we run through the leaveList.
     *   This is a list of objects that have registered themselves with us
     *   via addLeaveList().  For each object in the leaveList, we send
     *   a "leaving" message, with the actor as the parameter.  It should
     *   return true if it wants to be removed from the leaveList, nil
     *   if it wants to stay.
     */
    leaveList = []
    addLeaveList( obj ) =
    {
        self.leaveList := self.leaveList + obj;
    }
    leaveRoom( actor ) =
    {
        local tmplist, thisobj, i, tot;

        tmplist := self.leaveList;
	tot := length( tmplist );
	i := 1;
        while ( i <= tot )
        {
	    thisobj := tmplist[i];
            if ( thisobj.leaving( actor ))
                self.leaveList := self.leaveList - thisobj;
            i := i + 1;
        }
    }
    /*
     *   lookAround describes the room.  If verbosity is true, the full
     *   description is given, otherwise an abbreviated description (without
     *   the room's ldesc) is displayed.
     */
    nrmLkAround( verbosity ) =      // lookAround without location status
    {
        local l, cur, i, tot;

        if ( verbosity )
        {
            "\n\t"; self.ldesc;

            l := self.contents;
	    tot := length( l );
	    i := 1;
            while ( i <= tot )
            {
	        cur := l[i];
                if ( cur.isfixed ) cur.heredesc;
                i := i + 1;
            }
        }
        "\n\t";
        if (itemcnt( self.contents ))
        {
            "You see "; listcont( self ); " here. ";
        }
        listcontcont( self ); "\n";

        l := self.contents;
	tot := length( l );
	i := 1;
        while ( i <= tot )
        {
	    cur := l[i];
            if ( cur.isactor )
            {
                if ( cur <> Me )
                {
                    "\n\t";
                    cur.actorDesc;
                }
            }
            i := i + 1;
        }
    }
    statusLine =
    {
        self.sdesc; "\n\t";
    }
    lookAround( verbosity ) =
    {
        self.statusLine;
        self.nrmLkAround( verbosity );
    }
    
    /*
     *   Direction handlers.  The directions are all set up to
     *   the default, which is no travel allowed.  To make travel
     *   possible in a direction, just assign a room to the direction
     *   property.
     */
    north = { return( self.noexit ); }
    south = { return( self.noexit ); }
    east  = { return( self.noexit ); }
    west  = { return( self.noexit ); }
    up    = { return( self.noexit ); }
    down  = { return( self.noexit ); }
    ne    = { return( self.noexit ); }
    nw    = { return( self.noexit ); }
    se    = { return( self.noexit ); }
    sw    = { return( self.noexit ); }
    in    = { return( self.noexit ); }
    out   = { return( self.noexit ); }
    
    /*
     *   noexit displays a message when the player attempts to travel
     *   in a direction in which travel is not possible.
     */
    noexit = { "%You% can't go that way. "; return( nil ); }
;

/*
 *  searchHider: hider
 *
 *  This is just like an underHider, except that objects are hidden
 *  within this object in such a way that the object must be looked in
 *  or searched.  Objects to be hidden in this object should have their
 *  searchLoc property set to point to this object.  Note that this
 *  is different from a normal container, in that the objects hidden within
 *  this object will not show up until the object is explicitly looked in
 *  or searched.
 *  
 *  The items hidden with searchHider must be of class hiddenItem.
 */

//SHIDE
class searchHider: hider
    searchCont = []
    verDoSearch(actor) = {}
    doSearch(actor) =
    {
	if (self.searchCont = nil)
	    "There's nothing else in <<self.thedesc>>. ";
	else
	    self.searchCont := self.searchObj(actor, self.searchCont);
    }
    verDoLookin(actor) =
    {
	if (self.searchCont = nil)
	    pass verDoLookin;
    }
    doLookin(actor) =
    {
	if (self.searchCont = nil)
	    pass doLookin;
	else
	    self.searchCont := self.searchObj(actor, self.searchCont);
    }
;

/*   startroom is just a blank placeholder so that I don't get error messages
 *   when writing games, but I left it in for your convenience.
 */

//STTRM
startroom: room
;

/*
 *   strobj works like numobj, but for strings.  So, a player command of
 *     type "hello" on the keyboard
 *   will result in a direct object of strobj, with its "value" property
 *   set to the string 'hello'.
 *
 *   Note that, because a string direct object is used in the save, restore,
 *   and script commands, this object must handle those commands.
 */

//STRBJ
strObj: basicStrObj     // use default definition from adv.t
;

/*
 *  surface: item
 *
 *  Objects can be placed on a surface.  Apart from using the
 *  preposition "on" rather than "in'' to refer to objects
 *  contained by the object, a surface is identical to a
 *  container.  Note: an object cannot be both a
 *  surface and a container, because there is no
 *  distinction between the two internally.
 */

//SURFR
class surface: item
    issurface = true        // Item can hold objects on its surface
    ldesc =
    {
        if (itemcnt( self.contents ))
        {
            "On "; self.thedesc; " %you% see%s% "; listcont( self ); ". ";
        }
        else
        {
            "There's nothing on "; self.thedesc; ". ";
        }
    }
    verIoPutOn( actor ) = {}
    ioPutOn( actor, dobj ) =
    {
        dobj.doPutOn( actor, self );
    }
;

/*
 *  switchItem: fixeditem
 *
 *  This is a class for things that can be turned on and off by the
 *  player.  The only special property is isActive, which is nil
 *  if the switch is turned off and true when turned on.  The object
 *  accepts the commands "turn it on" and "turn it off,'' as well as
 *  synonymous constructions, and updates isActive accordingly.
 */

//SWITM
class switchItem: fixeditem
    verDoSwitch( actor ) = {}
    doSwitch( actor ) =
    {
        self.isActive := not self.isActive;
        "Okay, "; self.thedesc; " is now switched ";
        if ( self.isActive ) "on"; else "off";
        ". ";
    }
    verDoTurnon( actor ) =
    {
        /*
           You can't turn on something in the dark unless you're carrying
           it.  You also can't turn something on if it's already on.
        */
	if (not actor.location.islit and not actor.isCarrying(self))
	    "It's pitch black.";
        else if ( self.isActive )
            "It's already turned on! ";
    }
    doTurnon( actor ) =
    {
        self.isActive := true;
        "Okay, it's now turned on. ";
    }
    verDoTurnoff( actor ) =
    {
        if ( not self.isActive ) "It's already turned off! ";
    }
    doTurnoff( actor ) =
    {
        self.isActive := nil;
        "Okay, it's now turned off. ";
    }
;

/*
 *  theFloor is a special item that appears in every room (hence
 *  the non-standard location property).  This object is included
 *  mostly for completeness, so that the player can refer to the
 *  floor; otherwise, it doesn't do much.  Dropping an item on the
 *  floor, for example, moves it to the current room.
 */

//TFLOR
theFloor: beditem, floatingItem
    noun = 'floor' 'ground'
    sdesc = "ground"
    adesc = "the ground"
    statusPrep = "on"
    outOfPrep = "off of"
    location =
    {
        if ( Me.location = self )
            return( self.sitloc );
        else
            return( Me.location );
    }
    locationOK = true        // suppress warning about location being a method
    doSiton( actor ) =
    {
        "Okay, %you're% now sitting on "; self.thedesc; ". ";
        self.sitloc := actor.location;
        actor.moveInto( self );
    }
    doLieon( actor ) =
    {
        self.doSiton( actor );
    }
    ioPutOn( actor, dobj ) =
    {
        dobj.doDrop( actor );
    }
    ioPutIn( actor, dobj ) =
    {
        dobj.doDrop( actor );
    }
;

/*
 *  thing: object
 *
 *  The basic class for objects in a game.  The property contents
 *  is a list that specifies what is in the object; this property is
 *  automatically set up by the system after the game is compiled to
 *  contain a list of all objects that have this object as their
 *  location property.  The contents property is kept
 *  consistent with the location properties of referenced objects
 *  by the moveInto method; always use moveInto rather than
 *  directly setting a location property for this reason.  The
 *  adesc method displays the name of the object with an indefinite
 *  article; the default is to display "a" followed by the sdesc,
 *  but objects that need a different indefinite article (such as "an"
 *  or "some") should override this method.  Likewise, thedesc
 *  displays the name with a definite article; by default, thedesc
 *  displays "the" followed by the object's sdesc.  The sdesc
 *  simply displays the object's name ("short description") without
 *  any articles.  The ldesc is the long description, normally
 *  displayed when the object is examined by the player; by default,
 *  the ldesc displays "It looks like an ordinary sdesc."
 *  The isIn(object) method returns true if the
 *  object's location is the specified object or the object's
 *  location is an object whose contentsVisible property is
 *  true and that object's isIn(object) method is
 *  true.  Note that if isIn is true, it doesn't
 *  necessarily mean the object is reachable, because isIn is
 *  true if the object is merely visible within the location.
 *  The thrudesc method displays a message for when the
 *  player looks through the object (objects such as windows would
 *  use this property).  The moveInto(object) method
 *  moves the object to be inside the specified object.
 *  To make an object disappear, move it into nil.
 */

//THANG
class thing: object
    weight = 0
    bulk = 1
    isListed = true         // shows up in room/inventory listings
    contents = []           // set up automatically by system - do not set
    verGrab( obj ) = {}
    Grab( obj ) = {}
    adesc =
    {
        "a "; self.sdesc;   // default is "a <name>"; "self" is current object
    }
    pluraldesc =            // default is to add "s" to the sdesc
    {
	self.sdesc; "s";
    }
    thedesc =
    {
        "the "; self.sdesc; // default is "the <name>"
    }
    multisdesc = { self.sdesc; }
    ldesc = { "It looks like an ordinary "; self.sdesc; " to %me%."; }
    readdesc = { "%You% can't read "; self.adesc; ". "; }
    actorAction( v, d, p, i ) =
    {
        "You have lost your mind. ";
        exit;
    }
    contentsVisible = { return( true ); }
    contentsReachable = { return( true ); }
    isIn( obj ) =
    {
        local myloc;

        myloc := self.location;
        if ( myloc )
        {
            if ( myloc = obj ) return( true );
            if ( myloc.contentsVisible ) return( myloc.isIn( obj ));
        }
        return( nil );
    }
    thrudesc = { "%You% can't see much through "; self.thedesc; ".\n"; }
    moveInto( obj ) =
    {
        local loc;

	/*
	 *   For the object containing me, and its container, and so forth,
	 *   tell it via a Grab message that I'm going away.
	 */
	loc := self.location;
	while ( loc )
	{
	    loc.Grab( self );
	    loc := loc.location;
	}

        if ( self.location )
            self.location.contents := self.location.contents - self;
        self.location := obj;
        if ( obj ) obj.contents := obj.contents + self;
    }
    verDoSave( actor ) =
    {
        "Please specify the name of the game to save in double quotes,
        for example, SAVE \"GAME1\". ";
    }
    verDoRestore( actor ) =
    {
        "Please specify the name of the game to restore in double quotes,
        for example, RESTORE \"GAME1\". ";
    }
    verDoScript( actor ) =
    {
        "You should type the name of a file to write the transcript to
        in quotes, for example, SCRIPT \"LOG1\". ";
    }
    verDoSay( actor ) =
    {
        "You should say what you want to say in double quotes, for example,
        SAY \"HELLO\". ";
    }
    verDoPush( actor ) =
    {
        "Pushing "; self.thedesc; " doesn't do anything. ";
    }
    verDoWear( actor ) =
    {
        "%You% can't wear "; self.thedesc; ". ";
    }
    verDoTake( actor ) =
    {
        if ( self.location = actor )
        {
            "%You% already %have% "; self.thedesc; "! ";
        }
        else self.verifyRemove( actor );
    }
    verifyRemove( actor ) =
    {
  	/*
	 *   Check with each container to make sure that the container
	 *   doesn't object to the object's removal.
	 */
        local loc;

        loc := self.location;
        while ( loc )
        {
            if ( loc <> actor ) loc.verGrab( self );
            loc := loc.location;
        }
    }
    isVisible( vantage ) =
    {
        local loc;

        loc := self.location;
        if ( loc = nil ) return( nil );

	/*
	 *   if the vantage is inside me, and my contents are visible,
	 *   I'm visible 
	 */
	if (vantage.location = self and self.contentsVisible)
	    return true;

        /* if I'm in the vantage, I'm visible */
        if ( loc = vantage ) return( true );

        /*
         *   if its location's contents are visible, and its location is
         *   itself visible, it's visible
         */
        if ( loc.contentsVisible and loc.isVisible( vantage )) return( true );

        /*
         *   If the vantage has a location, and the vantage's location's
         *   contents are visible (if you can see me I can see you), and
         *   the object is visible from the vantage's location, the object
         *   is visible
         */
        if ( vantage.location <> nil and vantage.location.contentsVisible and
         self.isVisible( vantage.location ))
            return( true );

        /* all tests failed:  it's not visible */
        return( nil );
    }
    cantReach( actor ) =
    {
        if ( self.location = nil )
        {
            if ( actor.location.location )
               "%You% can't reach that from << actor.location.thedesc >>. ";
            return;
        }
        if ( not self.location.isopenable or self.location.isopen )
            self.location.cantReach( actor );
        else "%You%'ll have to open << self.location.thedesc >> first. ";
    }
    isReachable( actor ) =
    {
        local loc;

        /* if the object is in the room's 'reachable' list, it's reachable */
        if (find( actor.location.reachable, self ) <> nil )
            return( true );

        /*
         *   If the object's container's contents are reachable, and the
         *   container is reachable, the object is reachable.
         */
        loc := self.location;
	if (find( actor.location.reachable, self ) <> nil )
	    return( true );
	if ( loc = nil ) return( nil );
	if ( loc = actor or loc = actor.location ) return( true );
	if ( loc.contentsReachable )
	    return( loc.isReachable( actor ));
	return( nil );
        return( nil );
    }
    doTake( actor ) =
    {
        local totbulk, totweight;

        totbulk := addbulk( actor.contents ) + self.bulk;
        totweight := addweight( actor.contents );
        if ( not actor.isCarrying( self ))
            totweight := totweight + self.weight + addweight(self.contents);

        if ( totweight > actor.maxweight )
            "%Your% load is too heavy. ";
        else if ( totbulk > actor.maxbulk )
            "%You've% already got %your% hands full. ";
        else
        {
            self.moveInto( actor );
            "Taken. ";
        }
    }
    verDoDrop( actor ) =
    {
        if ( not actor.isCarrying( self ))
        {
            "%You're% not carrying "; self.thedesc; "! ";
        }
        else self.verifyRemove( actor );
    }
    doDrop( actor ) =
    {
        actor.location.roomDrop( self );
    }
    verDoUnwear( actor ) =
    {
        "%You're% not wearing "; self.thedesc; "! ";
    }
    verIoPutIn( actor ) =
    {
        "%You% can't put anything into "; self.thedesc; ". ";
    }
    circularMessage(io) =
    {
        local cont;

	"%You% can't put <<thedesc>> in <<io.thedesc>>, because
	<<io.thedesc>> is <<io.location = self ? "already" : ""
        >> <<io.location.issurface ? "on" : "in">> <<io.location.thedesc>>";
	for (cont := io.location ; cont <> self ; cont := cont.location)
	{
	    ", which is ";
            if (cont.location = self) "already ";
            "<<cont.location.issurface ? "on" : "in"
            >> <<cont.location.thedesc>>";
	}
	".";
    }
    verDoPutIn( actor, io ) =
    {
        if ( io = nil ) return;

        if ( self.location = io )
        {
            caps(); self.thedesc; " is already in "; io.thedesc; "! ";
        }
        else if (io = self)
        {
            "%You% can't put "; self.thedesc; " in itself! ";
        }
        else if (io.isIn(self))
	    self.circularMessage(io);
        else
            self.verifyRemove( actor );
    }
    doPutIn( actor, io ) =
    {
        self.moveInto( io );
        "Done. ";
    }
    verIoPutOn( actor ) =
    {
        "There's no good surface on "; self.thedesc; ". ";
    }
    verDoPutOn( actor, io ) =
    {
        if ( io = nil ) return;

        if ( self.location = io )
        {
            caps(); self.thedesc; " is already on "; io.thedesc; "! ";
        }
	else if (io = self)
        {
            "%You% can't put "; self.thedesc; " on itself! ";
        }
	else if (io.isIn(self))
	    self.circularMessage(io);
        else
	    self.verifyRemove( actor );
    }
    doPutOn( actor, io ) =
    {
        self.moveInto( io );
        "Done. ";
    }
    verIoTakeOut( actor ) = {}
    ioTakeOut( actor, dobj ) =
    {
        dobj.doTakeOut( actor, self );
    }
    verDoTakeOut( actor, io ) =
    {
        if ( io <> nil and not self.isIn( io ))
        {
            caps(); self.thedesc; " isn't in "; io.thedesc; ". ";
        }
	self.verDoTake(actor);         /* ensure object can be taken at all */
    }
    doTakeOut( actor, io ) =
    {
        self.doTake( actor );
    }
    verIoTakeOff( actor ) = {}
    ioTakeOff( actor, dobj ) =
    {
        dobj.doTakeOff( actor, self );
    }
    verDoTakeOff( actor, io ) =
    {
        if ( io <> nil and not self.isIn( io ))
        {
            caps(); self.thedesc; " isn't on "; io.thedesc; "! ";
        }
	self.verDoTake(actor);         /* ensure object can be taken at all */
    }
    doTakeOff( actor, io ) =
    {
        self.doTake( actor );
    }
    verIoPlugIn( actor ) =
    {
        "%You% can't plug anything into "; self.thedesc; ". ";
    }
    verDoPlugIn( actor, io ) =
    {
        "%You% can't plug "; self.thedesc; " into anything. ";
    }
    verIoUnplugFrom( actor ) =
    {
        "It's not plugged into "; self.thedesc; ". ";
    }
    verDoUnplugFrom( actor, io ) =
    {
        if ( io <> nil ) { "It's not plugged into "; io.thedesc; ". "; }
    }
    verDoLookin( actor ) =
    {
        "There's nothing in "; self.thedesc; ". ";
    }
    verDoLookthru( actor ) =
    {
        "%You% can't see anything through "; self.thedesc; ". ";
    }
    verDoLookunder( actor ) =
    {
        "There's nothing under "; self.thedesc; ". ";
    }
    verDoInspect( actor ) = {}
    doInspect( actor ) =
    {
        self.ldesc;
    }
    verDoRead( actor ) =
    {
        "I don't know how to read "; self.thedesc; ". ";
    }
    verDoLookbehind( actor ) =
    {
        "There's nothing behind "; self.thedesc; ". ";
    }
    verDoTurn( actor ) =
    {
        "Turning "; self.thedesc; " doesn't have any effect. ";
    }
    verDoTurnWith( actor, io ) =
    {
        "Turning "; self.thedesc; " doesn't have any effect. ";
    }
    verDoTurnTo( actor, io ) =
    {
        "Turning "; self.thedesc; " doesn't have any effect. ";
    }
    verIoTurnTo( actor ) =
    {
        "I don't know how to do that. ";
    }
    verDoTurnon( actor ) =
    {
        "I don't know how to turn "; self.thedesc; " on. ";
    }
    verDoTurnoff( actor ) =
    {
        "I don't know how to turn "; self.thedesc; " off. ";
    }
    verIoAskAbout( actor ) = {}
    ioAskAbout( actor, dobj ) =
    {
        dobj.doAskAbout( actor, self );
    }
    verDoAskAbout( actor, io ) =
    {
        "Surely, %you% can't think "; self.thedesc; " knows anything
        about it! ";
    }
    verIoTellAbout( actor ) = {}
    ioTellAbout( actor, dobj ) =
    {
        dobj.doTellAbout( actor, self );
    }
    verDoTellAbout( actor, io ) =
    {
        "It doesn't look as though "; self.thedesc; " is interested. ";
    }
    verDoUnboard( actor ) =
    {
        if ( actor.location <> self )
        {
            "%You're% not in "; self.thedesc; "! ";
        }
        else if ( self.location=nil )
        {
            "%You% can't get out of "; self.thedesc; "! ";
        }
    }
    doUnboard( actor ) =
    {
        if ( self.fastenitem )
	{
	    "%You%'ll have to unfasten "; actor.location.fastenitem.thedesc;
	    " first. ";
	}
	else
	{
            "Okay, %you're% no longer in "; self.thedesc; ". ";
            self.leaveRoom( actor );
	    actor.moveInto( self.location );
	}
    }
    verDoAttackWith( actor, io ) =
    {
        "Attacking "; self.thedesc; " doesn't appear productive. ";
    }
    verIoAttackWith( actor ) =
    {
        "It's not very effective to attack with "; self.thedesc; ". ";
    }
    verDoEat( actor ) =
    {
        caps(); self.thedesc; " doesn't appear appetizing. ";
    }
    verDoDrink( actor ) =
    {
        caps(); self.thedesc; " doesn't appear appetizing. ";
    }
    verDoGiveTo( actor, io ) =
    {
        if ( not actor.isCarrying( self ))
        {
            "%You're% not carrying "; self.thedesc; ". ";
        }
        else self.verifyRemove( actor );
    }
    doGiveTo( actor, io ) =
    {
        self.moveInto( io );
        "Done. ";
    }
    verDoPull( actor ) =
    {
        "Pulling "; self.thedesc; " doesn't have any effect. ";
    }
    verDoThrowAt( actor, io ) =
    {
        if ( not actor.isCarrying( self ))
        {
            "%You're% not carrying "; self.thedesc; ". ";
        }
        else self.verifyRemove( actor );
    }
    doThrowAt( actor, io ) =
    {
        "%You% miss%es%. ";
        self.moveInto( actor.location );
    }
    verIoThrowAt( actor ) =
    {
        if ( actor.isCarrying( self ))
        {
            "%You% could at least drop "; self.thedesc; " first. ";
        }
    }
    ioThrowAt( actor, dobj ) =
    {
        dobj.doThrowAt( actor, self );
    }
    verDoThrowTo( actor, io ) =
    {
        if ( not actor.isCarrying( self ))
        {
            "%You're% not carrying "; self.thedesc; ". ";
        }
        else self.verifyRemove( actor );
    }
    doThrowTo( actor, io ) =
    {
        "%You% miss%es%. ";
        self.moveInto( actor.location );
    }
    verDoThrow( actor ) =
    {
        if ( not actor.isCarrying( self ))
        {
            "%You're% not carrying "; self.thedesc; ". ";
        }
        else self.verifyRemove( actor );
    }
    doThrow( actor ) =
    {
        "Thrown. ";
        self.moveInto( actor.location );
    }
    verDoShowTo( actor, io ) =
    {
    }
    doShowTo( actor, io ) =
    {
        if ( io <> nil ) { caps(); io.thedesc; " isn't impressed. "; }
    }
    verIoShowTo( actor ) =
    {
        caps(); self.thedesc; " isn't impressed. ";
    }
    verDoClean( actor ) =
    {
        caps(); self.thedesc; " looks a bit cleaner now. ";
    }
    verDoCleanWith( actor, io ) = {}
    doCleanWith( actor, io ) =
    {
        caps(); self.thedesc; " looks a bit cleaner now. ";
    }
    verDoMove( actor ) =
    {
        "Moving "; self.thedesc; " doesn't reveal anything. ";
    }
    verDoMoveTo( actor, io ) =
    {
        "Moving "; self.thedesc; " doesn't reveal anything. ";
    }
    verIoMoveTo( actor ) =
    {
        "That doesn't get us anywhere. ";
    }
    verDoMoveWith( actor, io ) =
    {
        "Moving "; self.thedesc; " doesn't reveal anything. ";
    }
    verIoMoveWith( actor ) =
    {
        caps(); self.thedesc; " doesn't seem to help. ";
    }
    verDoTypeOn( actor, io ) =
    {
        "You should say what you want to type in double quotes, for
        example, TYPE \"HELLO\" ON KEYBOARD. ";
    }
    verDoTouch( actor ) =
    {
        "Touching "; self.thedesc; " doesn't seem to have any effect. ";
    }
    verDoPoke( actor ) =
    {
        "Poking "; self.thedesc; " doesn't seem to have any effect. ";
    }
    verDoBreak(actor) = {}
    doBreak(actor) =
    {
	"You'll have to tell me how to do that.";
    }
    genMoveDir = { "%You% can't seem to do that. "; }
    verDoMoveN( actor ) = { self.genMoveDir; }
    verDoMoveS( actor ) = { self.genMoveDir; }
    verDoMoveE( actor ) = { self.genMoveDir; }
    verDoMoveW( actor ) = { self.genMoveDir; }
    verDoMoveNE( actor ) = { self.genMoveDir; }
    verDoMoveNW( actor ) = { self.genMoveDir; }
    verDoMoveSE( actor ) = { self.genMoveDir; }
    verDoMoveSW( actor ) = { self.genMoveDir; }
    verDoSearch( actor ) =
    {
        "%You% find%s% nothing of interest. ";
    }

    /* on dynamic construction, move into my contents list */
    construct =
    {
	self.moveInto(location);
    }

    /* on dynamic destruction, move out of contents list */
    destruct =
    {
	self.moveInto(nil);
    }

    /*
     *   Make it so that the player can give a command to an actor only
     *   if an actor is reachable in the normal manner.  This method
     *   returns true when 'self' can be given a command by the player. 
     */
    validActor = (self.isReachable(Me))
;

/*
 *  transparentItem: item
 *
 *  An object whose contents are visible, even when the object is
 *  closed.  Whether the contents are reachable is decided in the
 *  normal fashion.  This class is useful for items such as glass
 *  bottles, whose contents can be seen when the bottle is closed
 *  but cannot be reached.
 */

//TRATM
class transparentItem: item
    contentsVisible = { return( true ); }
    ldesc =
    {
        if (self.contentsVisible and itemcnt(self.contents) <> 0)
        {
            "In "; self.thedesc; " %you% see%s% "; listcont(self); ". ";
        }
        else
        {
            "There's nothing in "; self.thedesc; ". ";
        }
    }
    verGrab( obj ) =
    {
        if ( self.isopenable and not self.isopen )
            "%You% will have to open << self.thedesc >> first. ";
    }
    doOpen( actor ) =
    {
        self.isopen := true;
        "Opened. ";
    }
    verDoLookin(actor) = {}
    doLookin(actor) = { self.ldesc; }
;

/*
 *  underHider: hider
 *
 *  This is an object that can have other objects placed under it.  The
 *  objects placed under it can only be found by looking under the object;
 *  see the description of hider for more information.  You should
 *  set the underLoc property of each hidden object to point to
 *  the underHider.
 *  
 *  Note that an underHider doesn't allow the player to put anything
 *  under the object during the game.  Instead, it's to make it easy for the
 *  game writer to set up hidden objects while implementing the game.  All you
 *  need to do to place an object under another object is declare the top
 *  object as an underHider, then declare the hidden object normally,
 *  except use underLoc rather than location to specify the
 *  location of the hidden object.  The behindHider and searchHider
 *  objects work similarly.
 *  
 *  The objects hidden with underHider must be of class hiddenItem.
 */

//UHIDE
class underHider: hider
    underCont = []         /* list of items under me (set up by initSearch) */
    verDoLookunder(actor) = {}
    doLookunder(actor) =
    {
	if (self.underCont = nil)
	    "There's nothing else under <<self.thedesc>>. ";
	else
	    self.underCont := self.searchObj(actor, self.underCont);
    }
;

/*
 *  vehicle: item, nestedroom
 *
 *  This is an object that an actor can board.  An actor cannot go
 *  anywhere while on board a vehicle (except where the vehicle goes);
 *  the actor must get out first.
 */

//VHICL
class vehicle: item, nestedroom
    reachable = ([] + self)
    isvehicle = true
    verDoEnter( actor ) = { self.verDoBoard( actor ); }
    doEnter( actor ) = { self.doBoard( actor ); }
    verDoBoard( actor ) =
    {
        if ( actor.location = self )
        {
            "%You're% already in "; self.thedesc; "! ";
        }
	else if (actor.isCarrying(self))
	{
	    "%You%'ll have to drop <<thedesc>> first!";
	}
    }
    doBoard( actor ) =
    {
        "Okay, %you're% now in "; self.thedesc; ". ";
        actor.moveInto( self );
    }
    noexit =
    {
        "%You're% not going anywhere until %you% get%s% out of ";
	  self.thedesc; ". ";
        return( nil );
    }
    out = ( self.location )
    verDoTake(actor) =
    {
        if (actor.isIn(self))
	    "%You%'ll have to get <<self.outOfPrep>> <<self.thedesc>> first.";
	else
	    pass verDoTake;
    }
    dobjGen(a, v, i, p) =
    {
        if (a.isIn(self) and v <> inspectVerb and v <> getOutVerb
	    and v <> outVerb)
	{
	    "%You%'ll have to get out of << thedesc >> first.";
	    exit;
	}
    }
    iobjGen(a, v, d, p) =
    {
        if (a.isIn(self) and v <> putVerb)
	{
	    "%You%'ll have to get out of << thedesc >> first.";
	    exit;
	}
    }
;

/*
 *   The "version" object defines, via its "sdesc" property, the name and
 *   version number of the game.  Change this to a suitable name for your
 *   game.
 */

//VERSN
version: object
    sdesc = "A TADS Adventure
      \n Developed with TADS, the Text Adventure Development System.\n"
;

