#charset "us-ascii"
#include <adv3.h>
#include <en_us.h>

//------------------------------------------------------------------------------
// The sitting room.

sittingRoom: Room 'Sitting Room'
    "This boxy little room is furnished in a fussy, old-fashioned style. The
    one modern touch is a wide-screen TV, with a couch facing it. A low coffee
    table sits in front of the couch, and a fireplace is set in another wall.
    There\'s a door to the south, and the open doorway to the hall is on the
    north. "
    north = hall
    south = frontDoorInside
    vocabWords = 'boxy little living sitting sitting-room/room/lounge'
;

+ RoomPartItem, EntryPortal ->hall 'open doorway' 'open doorway'
    "The open doorway leads to the north. "
    cannotOpenMsg = 'It\'s already open. '
    nothingThroughMsg = 'It leads to the hall. '  
    specialNominalRoomPartLocation = defaultNorthWall
    specialDesc = "An open doorway leads through the north wall. "
;

+ Chair, Heavy 'leather shiny scuffed depressing brown shade/couch/sofa/settee' 
    'couch'
    "The couch, which is easily long enough for you to lie on, is set facing
    the wide-screen TV. It\'s covered in a rather depressing shade of brown
    leather, which has become more than a little shiny in some places, and
    scuffed in others. "
    allowedPostures = [sitting, standing, lying]
    nothingUnderMsg = 'You get down on your hands and knees to peer under the
        couch but see nothing apart from the floor and a whole lot of dust and
        fluff. '
    lookInDesc = "Searching your settee at home can frequently prove rewarding,
        since there's often the odd dime or quarter or lost bunch of keys
        fallen behind the cushions, but Mrs. Pepper's couch proves obstinately
        devoid of such interesting finds. "
    bulkCapacity = 100
    maxSingleBulk = 100
;

+ Decoration 'whole lot/dust/fluff' 'dust'
    "They look much like dust and fluff always do. "
    notImportantMsg = 'The dust couldn\'t be less important if it tried. '
    isMassNoun = true    
    canMatchThem = true
;

+ Container, Fixture 'gas fire place/fireplace/gas' 'fireplace'
    "The fireplace has an artificial gas log in it, and a mantel above. "
    bulkCapacity = 50
    maxSingleBulk = 50
    
    dobjFor(Burn)
    {
        verify() {}
        check()  {   failCheck(cannotTurnOnMsg);  }
    }
    
    cannotEnterMsg = 'It\'s a bit on the small side for you, and you can\'t see
        anything there you want. Even if it were the chimney were wide enough for
        you to climb up (very doubtful) you\'d end up filthy, and your mother
        would freak out when she saw you. '
    cannotTurnOffMsg = 'It isn\'t lit. '
    cannotTurnOnMsg = 'It\'s quite warm enough in here, and you\'re a bit wary
        of gas fires. '
    cannotExtinguishMsg = (cannotTurnOffMsg)
;

++ Decoration 'gray bark wrinkled artificial gas log' 'gas log'
    "The gas log is gray. Its bark is wrinkled in an unconvincing way. "
;

+ mantel: Surface, CustomFixture 'mantel/mantelpiece/shelf' 'mantel'
    "The mantel is a shelf about four feet long, which runs along the wall
    above the fireplace. "
    cannotTakeMsg = 'It\'s firmly fixed in place. '
    bulkCapacity = 35
    maxSingleBulk = 35
;

++ framedPhoto:  Thing 'framed photo/photograph/picture/frame' 'framed photo'
    "The photo is of Mrs.\ Pepper and a man you\'ve never seen. She is standing
    with her hand on his shoulder; he is sitting in a wheelchair, gazing up at
    her with an expression that\'s hard to read --- something between fondness
    and brave dread. She\'s grimacing at the camera, showing her teeth in a way
    that was probably meant to be a smile. "
    bulk = 15
;

class PicturePart: Hidden, Decoration
    notImportantMsg = 'That\'s just part of the photo --- you can\'t actually do
        much with it. '
    
    discovered = (framedPhoto.described)    
    #ifdef __DEBUG
    dobjFor(Snarf) { verify() {logicalRank(70, 'decoration'); }}
    dobjFor(Pow) { verify() {logicalRank(70, 'decoration'); } }
    #endif
;

+++ mrsPPic: PicturePart 'ghastly mrs. pepper/woman' 'Mrs. Pepper'
    "At least she wasn\'t wearing one of her ghastly wigs when the photo was
    taking, but that doesn\'t make her look much nicer, and her grimace looks
    positively sinister. "
    isHer = true
    isProperName = true    
;

++++ PicturePart 'positively sinister predatory smile/grimace/teeth' 'smile'    
    "The way she's showing her teeth was probably meant to be a smile, but
    there\'s something almost predatory about it. "
    canMatchThem = true
    ownedBy = location
;

++++ PicturePart 'hand/claw' 'hand'
    "Mrs. Pepper\'s hand is resting on the man's shoulder. A closer look
    suggests that it is almost clutching it. "
;

+++ PicturePart '(mrs.) (pepper\'s) mr. unfamiliar late
    (pepper)/man/husband/body'     'man'
    "You\'ve never seen him before, but you guess it might be Mrs. Pepper\'s
    late husband. He's gazing up at her with an expression somewhere between
    fondness and brave dread. "
    isHim = true    
    /* 
     *   If both this object at the pictorial representation of Mrs. Pepper 
     *   appear in the resolve list (as they will if the target of the 
     *   command is "Mrs. Pepper"), remove this object from this resolve 
     *   list as the player must mean the other.
     */
    filterResolveList(lst, action, whichObj, np, requiredNum)
    {
        local idx = lst.indexWhich({x: x.obj_ == self});
        if(idx && lst.indexWhich({x: x.obj_ == mrsPPic}))
            lst = lst.removeElementAt(idx);
        
        return lst;
    }
;

++++ PicturePart 'brave expression/fondness/dread' 'man\'s expression'
    "It\'s hard to read: it looks somewhere between fondness and brave dread. "
    ownedBy = location
;

++++ PicturePart 'slightly stooped unremarkable shoulder' 'man\'s shoulder'
    "It\'s perhaps slightly stooped, but otherwise quite unremarkable apart from
    the hand that\'s resting on it. "
    ownedBy = location 
;

++++ PicturePart 'chair/wheelchair' 'wheelchair in the photo'
    "So far as you\'re concerned, it\'s just a wheelchair. In any case most of
    it is concealed by the man\'s body. "
    ownedBy = location
;

+ frontDoorInside: LockableWithKey, Door ->frontDoorOutside
    'front deadbolt lock/door*doors*locks' 'front door'
    "The front door has a peephole. Also, it's furnished with one of those
    annoying deadbolt locks that has to be unlocked with a key even when
    you're on the inside. "    
    keyList = [keyBunch]
    knownKeyList = [keyBunch]
    notAContainerMsg = iobjMsg('There\'s no need to put anything in that. ')
    dobjFor(Open) {
        verify() {
            gReveal('too-high');
            inherited;
        }
    }
;

// The peephole is mentioned when you're on the porch, so it's implemented here:
++ peephole: Component 'peep brass peephole/hole/cover' 'peephole'
    "It has a small brass cover which can be opened to allow you to look out
    through it. "
    dobjFor(Open) asDobjFor(LookThrough)
    nothingThroughMsg = 'You open the cover for a moment and peer cautiously
        through the peephole. You
        can\'t see much, just a rather uninteresting view of the porch, but at
        least there\'s no sign of Mrs. Pepper returning. '
    /* 
     *   If the player's just referred to me, the next time he refers to 
     *   COVER in a command, he probably means this object
     */
    beforeAction()
    {
        if(gDobj == self || gIobj == self)
            vocabLikelihood = 20;
        else
            vocabLikelihood = 0;
            
    }
;

+ Platform, Heavy 'dark wood low coffee table*tables*furniture' 'coffee table'
    "The coffee table is made of dark wood. "
    cannotJumpOverMsg = 'You\'d better not; the last time you tried that at
        home you nearly crashed into the TV, and your father warned you that if
        you\'d broken it it would have cost you your pocket money for the next
        thirty-seven years. '    
    bulkCapacity = 85
    maxSingleBulk = 85
    nothingUnderMsg = 'You can see at a glance that there\'s nothing under the
        coffee table, unless you count the floor. '
    cannotPutUnderMsg = iobjMsg('There\'s no good reason for putting anything
        there. ')
;

++ dummiesBook: Readable, Consultable
    'yellow thick spell spell-casting paperback dummies book/index/cover'
    'yellow book'
    "The book is a <<isDamp ? 'slightly damp' : 'thick'>> paperback with a
    yellow cover. It\'s entitled <i>Spell-Casting for Dummies.</i><.reveal
    book-read> "
    dobjFor(Read) {  preCond = inherited + objHeld   }
    bulk = 11

    firstRead = true
    readDesc 
    {
        if (firstRead) 
        {
            firstRead = nil;
            "As you open the book, a piece of paper falls out and flutters to
            the floor. ";
            codePaper.moveInto(getOutermostRoom);
            return;
        }
        "There\'s an index at the back of the book, where you might be able to
        look up specific topics if any interest you. Ignoring the index for the
        moment, you flip through the book at random, and encounter ";
        bookContents.doScript();
    }
    bookContents: ShuffledEventList {[
        'a description of how to make voodoo dolls out of Beanie Babies. A
        footnote attributes the technique to Martha Stewart. ',
        'a recipe for a love potion, the principle ingredients of which seem to
        be onions and vacuum cleaner lint. ',
        'some charts that explain how to navigate by the stars while riding a
        broomstick (but no advice on how to make the broomstick fly in the first
        place). ',
        'a heavily underlined passage discussing the habits of garden elves. ',
        'a numbingly complex description of Polynesian astrology, complete with
            pages of smudgy diagrams. ',
        'an entire chapter on warts. '
    ]}
    dobjFor(Open) asDobjFor(Read)
    dobjFor(Close) 
    {
        verify() {}
        check() 
        {
            failCheck ('There\'s no need to close the book; it falls closed
                quite naturally each time you finish reading something. ');
        }
    }
    dobjFor(ConsultAbout) 
    {
        preCond = inherited + objHeld
        action() 
        {
            if (firstRead) 
            {
                firstRead = nil;
                "As you open the book, a piece of paper falls out and flutters
                to the floor.<.reveal book-read> ";
                codePaper.moveInto(sittingRoom);                
                return;
            }
            inherited;
        }
    }
    
    beforeAction()
    {
        if(gDobj == self || gIobj == self)
            vocabLikelihood = 20;
        else
            vocabLikelihood = 0;            
    }
;

+++ConsultTopic [elf, pamphlet]
    "You thumb through the index and find that elves are listed at several
    points in the book. The longest passage reveals that elves today are fond
    of visiting or even dwelling in urban gardens, and can sometimes be coaxed
    with small gifts to make the garden more fertile, lush, and verdant. The
    book also warns that separating a garden elf from the soil for any length
    of time will cause it to sicken and die. "
;

+++ConsultTopic [tRain, tWater]
    "You thumb through the index and find a listing for <.q>rain,
    summoning.<./q> On that page, the book explains that rain can be induced to
    fall by wearing a garment or head-covering imbued with the proper power
    while pouring water on the ground and holding a bamboo stick. The spell for
    adding this power to an ordinary garment is quite intimidatingly complex,
    sad to say, and requires several ingredients that you not only don\'t have
    but have never even heard of. <.reveal need-bamboo>"
;

+++ConsultTopic [tMusic]
    "You thumb through the index and find a brief entry for <.q>music, uses
    of.<./q> This entry reads, <.q>see <.s>locks, magical.<./s><./q> "
;

+++ConsultTopic [blackWig, auburnWig, blondWig, greenWig, tGarments]
    "You thumb through the index. There\'s nothing under <.q>wigs,<./q> but
    purely by chance you spot an entry for <.q>garments, magical, uses of.<./q>
    This section of the book proves to be devoted almost entirely to full-page,
    exotically lighted photos of pointed black hats with wide brims --- but a
    short paragraph reveals that the working of a spell usually requires both
    the wearing of a magically enhanced garment and the performance of some
    other action, such as shooting off firecrackers or plucking the feathers
    from a recently deceased chicken. "
;

// To do: This ConsultTopic will need to be changed if we alter the design of
// the lock.
+++ ConsultTopic @tMagicLock
    "You thumb through the index and find a listing for <.q>locks,
    magical.<./q> On that page is printed the following verse:\b
    \tBehold I stand at the door and play\n
    \tA melody it must obey\n
    \tTo free the lock that I was glad\n
    \tTo conjure up when rightly clad. "
;

+++ ConsultTopic @tKeys
    "You thumb through the index. There\'s no entry for keys, but on
    exploring further you find an entry for <.q>locks, magical.<./q> 
    On that page is printed the following verse:\b
    \tBehold I stand at the door and play\n
    \tA melody it must obey\n
    \tTo free the lock that I was glad\n
    \tTo conjure up when rightly clad. "
;

+++ ConsultTopic @tMagic
    "You thumb through the index, but there\'s no specific entry for that ---
    quite likely because the whole book is devoted to magic! "
;

+++ ConsultTopic @tWitches
    "Flipping through the table of contents, you locate a very long chapter
    on that subject, which proves to contain extremely tedious, boring
    biographies of famous witches, none of whom you\'ve ever heard of. "
;

+++ ConsultTopic @tSleepSpell
    "You thumb through the index and find a listing for <.q>sleep,
    inducing.<./q> On the indicated page, the book explains that to put someone
    to sleep, you need to be wearing a magically prepared garment or
    head-covering and chant a brief incantation while wielding the Sign of the
    Scarlet Fish. Unfortunately, the incantation is written in an alphabet
    you\'ve never seen before<<brush.descPrinted ? ', although it may be in the
        same script as the writing on the hairbrush' : ''>>. It looks rather
    like Arabic, but with a lot more squiggles. "
;

+++ ConsultTopic @tSwerveSpell
    "You thumb through the index and after hunting for a bit, find a listing
    for <.q>vehicles, wheeled, causing to swerve.<./q> The indicated page
    seems to have been torn out, however. "
;

+++ ConsultTopic @tJimAikin
    "You thumb through the index and find a listing for <.q>Aikin, Jim.<./q>
    On the indicated page, you learn that Jim Aikin is an expert on electronic
    music technology and the author of two science fiction novels (both,
    alas, out of print). He was introduced to Interactive Fiction by playing
    Adventure on a gunmetal-gray Kaypro, which had a whopping 64 kilobytes
    of RAM. He's mentioned in a book on spell-casting because of his utterly
    mystifying and almost supernatural ability to teach young cellists the
    difference between <i>staccato</i> and <i>piu mosso.</i> "
;

+++ ConsultTopic @tEricEve
    "You hunt through the index and happen upon a listing for <.q>Eve,
    Eric.<./q> On the indicated page, you learn that Eric Eve is a New
    Testament Scholar working at Oxford University, mentioned in a book on
    magic because he\'s the author of an impenetrable monograph called <i>The
    Jewish Context of Jesus\' Miracles</i>. He was introduced to Interactive
    Fiction more years ago than he cares to admit by playing Adventure on a
    quaint home computer called a Lynx, which boasted an impressive 32
    kilobytes of RAM (later upgraded to an excessive 96K, parts of which could
    only be utilized effectively by peforming wizardry with machine-code). "
;

+++ ConsultTopic @tWarts
    topicResponse = (wartsChapter.desc)
;

+++ ConsultTopic @tAstrology
    topicResponse = (diagrams.desc)
;

+++ ConsultTopic @tLovePotion
    topicResponse = (recipe.desc)
;

+++ diagrams: Component 
    'smudgy complex polynesian (pages)/diagrams/astrology/description' 
    'smudgy diagrams'
    "The diagrams look like something out a geometry lesson from hell, and the
    description is every bit as incomprehensible as your math teacher. "
    isPlural = true
;

+++ Component 'astral navigation charts' 'astral navigation charts'
    "In the absence of a flying broomstick, you find them of little interest. "
    isPlural = true
;

+++ recipe: Component 'love recipe for potion recipe/potion' 'recipe' 
    "The principle ingredients for the love potion seem to be onions and vacuum
    cleaner lint, but the text goes on to warn that this recipe is
    particularly effective on older women. The prospect of Mrs. Pepper
    drooling all over you is so horrific that you vow to go nowhere near either
    ingredient while you're anywhere near her house. "
;

+++ Component 'heavily underlined passage' 'heavily underlined passage'
    "The passage reveals that elves today are fond of visiting or even dwelling
    in urban gardens, and can sometimes be coaxed with small gifts to make the
    garden more fertile, lush, and verdant. The book also warns that separating
    a garden elf from the soil for any length of time will cause it to sicken
    and die. "
;

+++ wartsChapter: Component 'entire chapter on warts/chapter' 'chapter on warts'
    "Funnily enough, you've never been that keen on warts, so you decide to
    give that chapter a miss. "
;


++ remote: ComplexContainer 'remote control/remote' 'remote control'
    "The remote is studded with buttons, but two of them are especially large
    and appear well-thumbed. One is red, the other blue. Around the back is
    a sort of plastic hatch cover. "
    disambigName = 'plastic hatch cover on the remote control'
    subContainer: ComplexComponent, RestrictedContainer, OpenableContainer 
    {
        openStatusReportable = (isOpen)
        vocabWords = 'plastic hatch cover'
        desc = "The plastic hatch cover is round the back of the remote control.
            "
        validContents = [battery]
        dobjFor(Remove) asDobjFor(Open)
        dobjFor(Doff) asDobjFor(Open)
        dobjFor(Open)
        {
            preCond = [new ObjectPreCondition(lexicalParent, objHeld) ]
        }
        canBeMagicallyLocked = nil
    }
    bulk = 7
    iobjFor(TurnOnWith) {
        check () {
            if (gDobj != tv) failCheck ('You can\'t turn that on using the
                remote control. ');
            if (tv.isOn) failCheck ('The TV is already switched on. ');
        }
        action() {
            replaceAction(Push, redButton);
        }
    }
    
    /* 
     *   The player probably means the remote rather than the peephole when 
     *   he refers to COVER, unless he's just referred to the peephole
     */
    beforeAction()
    {
        if(gDobj == self || gIobj == self || gDobj == subContainer || gIobj ==
           subContainer)           
            subContainer.vocabLikelihood = 20;        
        else
            subContainer.vocabLikelihood = 0;        
    }
;

+++ redButton: Button, Component 'red on off on-off button*buttons' 'red button'
    "The red button says <.q>ON<./q> in large square letters. "
    collectiveGroups = [buttonGroup]
    dobjFor(Push)
    {
        preCond = [new ObjectPreCondition(remote, objHeld) ]
        
        action()
        {
            if(battery.isIn(remote) && getOutermostRoom == sittingRoom)
            {
                tv.isOn = !tv.isOn;
                if(tv.isOn)
                    "The TV comes on, showing <<tv.channelDesc[tv.curChannel]>>
                    ";
                else
                    "The TV goes off. ";
                
            }
            else
                mainReport(&pushButtonNoEffectMsg);
        }
    }
;

+++ blueButton: Button, Component 'blue channel chan changing button*buttons' 
    'blue button'
    "The blue button says <.q>CHAN<./q> in large square letters. "    
    collectiveGroups = [buttonGroup]
    dobjFor(Push)
    {
        preCond = [new ObjectPreCondition(remote, objHeld) ]
        
        action()
        {
            if(battery.isIn(remote) && tv.isOn && getOutermostRoom ==
               sittingRoom)
                tv.advanceChannel();
            else
                mainReport(&pushButtonNoEffectMsg);
        }
    }
        
;

+++ buttonGroup: CollectiveGroup, Component '*buttons' 'buttons'
    "The main buttons of interest are the red button marked ON and the blue
    button marked CHAN. "
;

+++ battery: Thing 'men horse horses horseback sword swords aa battery*batteries' 
    'AA battery'
    "On the battery is a tiny product logo of some men on horseback brandishing
    swords. Beneath the horses\' hooves is the word CHARGER! "
    subLocation = &subContainer
    bulk = 3
;

+ tv: Heavy 'wide screen wide-screen video/tv/telly/television' 'wide-screen TV'
    desc()
    {
        "The wide-screen TV is the newest item in the room; it looks like it
        might be a fairly recent acquistion. ";
        if(isOn)
            "It's currently showing <<channelDesc[curChannel]>> ";
        else
            "It\'s off. ";
    }
    isOn = nil
    curChannel = 1
    channelDesc = [
        'the weather forecast. It seems that the fine weather is set to
        continue for several days, with no immediate prospect of rain. ',
        
        'an episode of the long-running sitcom <q>Fiends,</q> about a
        flat-sharing coven of witches. ',        
        
        'a news broadcast. Apparently the President wants to invade
        Liechtenstein as the next step in the War on Error, but no one in the
        administration knows where it is. Or at least, that\'s the impression
        you\'re left with. Naturally, they claim to know precisely where it is,
        and the newscaster is too cowardly or too oblivious to point out that
        they obviously don\'t. ',        
        
        'some boring reality TV show: <i>Celebrity Gardeners\' Cooking Song
        Contest.</i> ',
            
        'a badly colorized movie from the 1930s, in which W.\ C.\ Fields is
        chasing Mae West around the room with a pool cue. ',
        
        'a commercial for a new automobile fueled entirely by wind power ---
        the Ford Gasbag. ',
        
        'the quarter-finals of the International Paint-Drying contest; so far
        as you can tell the red emulsion from Canada looks like beating the
        Chilean white gloss by as much as thirty minutes (or so the
        commentators reckon), but it may be another hour or two until a certain
        result is achieved. You find it difficult to tear your eyes away from
        this one. ',
        
        'the latest episode of your favorite sci-fi show; too bad you don\'t
        have time to stay and watch it. ',
        
        'a current affairs program all about the politics of waste disposal. '
    ]
   
    channels = static channelDesc.length
    
    advanceChannel()
    {
        curChannel++;
        if(curChannel > channels)
            curChannel = 1;
        "The display on the TV changes to show <<channelDesc[curChannel]>> ";
    }
    
    cannotTurnOnMsg = (isOn ? 'It\'s already on. ' : useRemoteMsg)
    cannotTurnOffMsg = (isOn ? useRemoteMsg : 'It isn\'t on. ')
    useRemoteMsg = 'It looks like you have to use the remote to do that. '    
    nothingBehindMsg = 'Not much there but the wall. '
    shouldNotBreakMsg = 'You\'d be in <i>real</i> trouble if you broke that. '
    
    dobjFor(TurnOnWith) {
        check() {
            if (gIobj != remote) failCheck ('You can\'t use that to turn on
                the TV. ');
        }
        // The remote will handle the action.
    }
;

codePaper: Readable, Thing 'paper/piece' 'piece of paper'
    "On the piece of paper someone has written, in block capitals with a stubby
    pencil, the phrase <.q>THE MIND POWER OF GLAMOROUS HAIR CLUSTERS.<./q>
    Below this is another phrase, which oddly enough is total gibberish:
    <.q>JAH ZQSU WDYHF DI OKGZDFDPX AGQF TKPXJHFX.<./q> "
    bulk = 2
    isPaper = true
;

/*
 *   The solution of the wig cryptogram is as follows:
        A = G
        B = L
        C = T
        D = U
        E = H
        F = I
        G = O
        H = A
        I = Q
        J = V
        K = B
        L = K
        M = Z
        N = S
        O = D
        P = W
        Q = C
        R = F
        S = X
        T = J
        U = P
        V = E
        W = Y
        X = M
        Y = N
        Z = R
 *
 *   Not all of these letters are present in either set of text, but all of 
 *   the letters in the wig labels (except V and K) are present in the 
 *   phrase on the paper. The fact that the V in VEHICLES and the K in 
     LOCKING are missing should not be too grave a difficulty.
 *
 *   The wig labels are as follows:
 *
 *      MAGICAL LOCKING
        ZGOQTGK KDTBQSO   
 *
 *      SUMMON RAIN
        XPZZDS FGQS
 *
 *      MAKE WHEELED VEHICLES SWERVE
        ZGBH YAHHKHU EHAQTKHX XYHFEH
 *
 *      CAUSE SLEEP
        TGPXH XKHHW
 */

MultiLoc, Enterable ->hall 'long narrow hall' 'hall'
    "You can't see very far into the hall from here, but it looks quite long
    and narrow. "
    locationList = [kitchen, sittingRoom]
    getFaces = [hall]
    dobjFor(LookIn) asDobjFor(Enter)
;


//------------------------------------------------------------------------------
// The hall -- a not-very-interesting room.

hall: Room 'Hall'
    "This hall is little more than a narrow passage connecting the rooms to
    north and south. On the east side a flight of stairs leads up to the floor
    above, and there\'s a<< liftDoorDown.isOpen ? 'n opening' : ' door'>> to
    the west, with a black button next to it at waist height. "
    vocabWords = 'narrow hall/room/passage'
    south = sittingRoom
    east asExit(up)
    up = hallStairsUp
    west = liftDoorDown
    in asExit(west)
    north = kitchen
;

+ hallStairsUp: StairwayUp 'narrow steep stairway/staircase/stairs/flight' 
    'staircase'
    "The stairway is narrow and steep. "
;

+ liftDoorDown: LiftDoor     
    initiallyOpen = true        
;

+ LiftButton
    myDoor = liftDoorDown
;

class LiftButton: RoomPartItem, Button, Fixture 
    'little brass black plate/button*buttons' 
    'black button'
    "It\'s an ordinary black button, set in a little brass plate. "
    dobjFor(Push)
    {
        action() {  elevator.summon(myDoor);  }
    }
    specialNominalRoomPartLocation = defaultWestWall
    specialDesc = "A black button is set at waist-height on the west wall, just
        next to <<myDoor.theName>>. "    
    vocabLikelihood = 10
;

//------------------------------------------------------------------------------
// The kitchen.

kitchen: Room 'Kitchen'
    "The kitchen is small, and equipped with the usual appliances, including a
    cupboard, a sink, a stove, and a refrigerator, though most of the space is
    taken up by the unusually bulky wooden table standing in the middle of the
    room. The back door leads north, while to the south is an open doorway. A
    flight of stairs on the west side of the room leads down, roughly opposite
    the window in the east wall. "
    
    roomFirstDesc = "You glance nervously around as your feet land on the
        floor, but so far as you can tell the house really is empty now --- at
        least, you can't hear anybody about.\b
        
        Compared to your parents' kitchen, Mrs.\ Pepper's seems, well, a bit mean
        and cramped, just like its owner. Most of the space is taken up by the
        old wooden table standing right in the middle. Beneath the lip of the
        table is a wide, shallow drawer. Completing the furnishings are a
        cupboard, a sink, a stove, and a refrigerator. The back door leads out
        to the north, while to the south is an open doorway, its door long since
        removed. The window through which you've just climbed is set in the
        east wall, and on the opposite side of the room a flight of steps
        leads down into darkness. "
    east = kitchenWindowInside
    west asExit(down)
    down = kitchenStairs
    south = hall
    north = backDoorInside    
    vocabWords = 'kitchen/room'
;

+ backDoorInside: LockableWithKey, Door ->backDoorOutside 
    'back deadbolt lock/door*doors*locks' 'back door'
    "The back door is furnished with one of those annoying deadbolt locks that
    has to be unlocked with a key even when you're on the inside. "
    keyList = [keyBunch]
    knownKeyList = [keyBunch]    
    notAContainerMsg = iobjMsg('There\'s no need to put anything in that. ')
    dobjFor(Open) {
        verify() {
            gReveal('too-high');
            inherited;
        }
    }
;

+ RoomPartItem, EntryPortal, Exitable -> hall 
    'open (door) (s) (south) doorway/frame' 
    'open doorway'
    "A close inspection suggests that the door that once hung here was removed
    several years ago (although whether 'several' means two or ten you have no
    idea). Now, not even the hinges remain, leaving a bare frame. "
    nothingThroughMsg = 'You can\'t see much through the doorway, just some
        kind of hall. '
    cannotOpenMsg = 'It\'s already open. '
    specialNominalRoomPartLocation = defaultSouthWall
    specialDesc = "An open doorway leads through the south wall. "
;

+ applianceGroup: CollectiveGroup 'usual kitchen *appliances' 'appliances'
    "In addition to the cupboard and the sink, there's a stove and a fridge. "
;

+ OpenableContainer, CustomImmovable 'usual (kitchen) white ancient 
    refrigerator/fridge*appliances' 'fridge'
    "The refrigerator is your basic white model. Now and again it emits wheezing,
        chuckling noises. "
    collectiveGroups = [applianceGroup]
    cannotEnterMsg = 'It\'s too small, and too cold. '
    bulkCapacity = 50
    maxSingleBulk = 50
;

++ CustomImmovable 'stale moldy mouldy mold/mould/loaf/bread*food' 
    'stale loaf of bread'
    "It looks like one of your more successful school biology experiments ---
    the one where you\'re meant to grow a culture of mold. "
    cannotTakeMsg = 'You\'d rather not touch it. '
    cannotEatMsg = 'Even the boiled cabbage in the school cafeteria is more
        appetizing. '
    isListed = true
    isListedInContents = true
;

++ CustomImmovable 'runny crusty blue ripe over-ripe large wedge/cheese*food' 
    'large wedge of blue cheese'
    "The cheese is runny in some spots and dried and crusty in other spots,
    all at the same time.
    Its odor is remarkable: If it gets much riper, it'll probably come when you
    call it. "
    isListed = true
    isListedInContents = true
    cannotTakeMsg = 'You\'d rather leave it where it is. '
    cannotEatMsg = 'The thought of it turns your stomach. '
;

+++ Odor 'horrible overpowering vile stench/smell' 'stench'
    sourceDesc = "Phewww! It smells horrible. "
    descWithSource = "The cheese smells absolutely vile. "
    hereWithSource = "There\'s an overpowering smell of over-ripe cheese. "
;

+ ComplexContainer, CustomFixture 'usual stove/oven/cooker*appliances' 'stove'
    "The stove is not as nice as the one your family has, and not as clean
    either. Bits of this and that have dribbled down the front and dried
    there. "
    subContainer: ComplexComponent, OpenableContainer 
    {   
        bulkCapacity = 50
        maxSingleBulk = 50
    }
    subSurface: ComplexComponent, Surface
    {
        bulkCapacity = 50
        maxSingleBulk = 50
    }
    cannotEnterMsg = 'It\'s too small, and you don\'t want to cook yourself. '
    cannotStandOnMsg = 'You\'re not sure it\'s safe to. '
    cannotSitOnMsg = (cannotStandOnMsg)
    cannotLieOnMsg = 'It\'s not big enough. '
    cannotTurnOffMsg = 'It doesn\'t appear to be on. '
    cannotTurnOnMsg = 'You\'ve had it drummed in to you that stoves are
        <i>not</i> things to be played with, and you have no plans to cook
        anything here right now. '
    collectiveGroups = [applianceGroup]
    
;

+ kitchenWindowInside:  RoomPartItem, TravelWithMessage, Door ->kitchenWindow 
    'window*windows' 'window'
    "It\'s set about three feet off the ground in the east wall. "

    specialNominalRoomPartLocation = defaultEastWall
    specialDesc = "A window is set in the east wall. "
    travelBarrier = 
    [
        new PushSkateBarrier('You can hardly push {the obj/him} out
        through the window; the window\'s too high up for that! '),
        
        binBarrier
    ]
    
    travelDesc = "You climb back out through the window, using the trash can to
        step safely back down into the driveway. "
    nothingThroughMsg = 'The window is quite high up, so you can\'t see much
        through it apart from patch of blue sky. '
    dobjFor(GetOutOf) asDobjFor(Enter)
;

+ Distant 'cloudless blue patch/sky' 'sky'
    "It's blue and cloudless. "
;

+ kitchenStairs: TravelWithMessage, StairwayDown, EventList
    'flight/steps/stairway/staircase/stairs' 'staircase'
    "The stairway leads steeply downward and disappears into the gloom.
    <<lightDesc()>> "
    lightDesc()
    {
        switch(cellar.senseAmbientMax([sight]))
        {
        case 0:
            case 1: "There's no light at all down there. "; break;
            case 2: "The light down there looks very dim. "; break;
            case 3: break;
        }
    }
    
    eventList = [ 'You tiptoe down the stairs into the cellar. It certainly is
        dark down here.... '
        ]
    
;

+ cabinetDrawers: Fixture 'cabinet drawers' 'cabinet drawers'
    "The drawers in the cabinets look pretty much the way you\'d expect.
    There\'s also a single drawer mounted under the edge of the table. "
    drawerHintMsg = 'You poke around among the cabinet drawers for a minute,
        and spot nothing useful. But your eye is drawn to the single drawer
        tucked away beneath the kitchen table. '
    dobjFor(Search) asDobjFor(Open)
    dobjFor(LookIn) asDobjFor(Open)
    dobjFor(Open) {
        verify() {}
        check() {
            failCheck (drawerHintMsg);
        }
    }    
;

+ ComplexContainer, Heavy 'big square old scratched bulky wooden kitchen 
    edge/table*tables' 'kitchen table'
    "It\'s a big square wooden table, old and scratched. Beneath one edge is a
    single drawer. "
    subSurface: ComplexComponent, Platform 
    { 
        stagingLocations = [lexicalParent.location]
        bulkCapacity = 1000
        maxSingleBulk = 500
        down asExit(out)
    }
    subUnderside: ComplexComponent, Underside 
    { 
        bulkCapacity = 500
        maxSingleBulk = 400
        lookInLister: undersideLookUnderLister
        {
            showListEmpty(pov, parent)
            {
                "The only thing under the table is its drawer. ";
            }
        }
    }
    
    dobjFor(StandOn) remapTo(StandOn, subSurface)
    dobjFor(SitOn) remapTo(SitOn, subSurface)
    dobjFor(LieOn) remapTo(LieOn, subSurface)
    dobjFor(Board) remapTo(StandOn, subSurface)
    dobjFor(GetOffOf) remapTo(GetOffOf, subSurface)
;

++ Component, OpenableContainer 'single plain wooden wide shallow drawer' 'drawer'
    "It\'s just a plain wooden drawer. "
    dobjFor(Pull) asDobjFor(Open)
    dobjFor(Push) asDobjFor(Close)
    cannotLockMsg = 'It doesn\'t have a lock. '
    cannotUnlockMsg = (cannotLockMsg)
;

+++ penLight: RestrictedContainer, OpenableContainer, Flashlight 
    'slim small flash/switch/opening/penlight/light/flashlight/torch' 
    'small flashlight'
    bulk = 7
    validContents = [battery]
    
    desc {
        if ((location == cellar) && (!bulb.isLit))
            "You can\'t see any such thing. ";
        else
            "The flashlight is slim, and about four inches long. There\'s a
            switch on one side, and an opening in the end through which a
            battery can be inserted. ";
    }
    dobjFor(Switch) {
        verify() {}
        check() {
            if (isOn) failCheck ('The flashlight is already on. ');
            if (location != me) 
                failCheck('You poke at the switch with your thumb, but it seems
                    the switch is spring-loaded. The moment you let go of it,
                    it snaps back to the <.s>off<./s> position. ');
        }
        action () {
            if(!battery.isIn(self))
                "You thumb the switch, but the flashlight fails to
                    turn on. <.reveal torch-dead>";
            else
            replaceAction (TurnOn, self);
        }
    }
    dobjFor(TurnOn)
    {
        check() {
            if (location != me) 
                failCheck('You poke at the switch with your thumb, but it seems
                    the switch is spring-loaded. The moment you let go of it,
                    it snaps back to the <.s>off<./s> position. ');
        }
        action()
        {
            if(!battery.isIn(self))
                "You thumb the switch, but the flashlight fails to
                    turn on. <.reveal torch-dead>";
            else
            {
                inherited; 
                achievement.awardPointsOnce();
            }
        }
    }
    
    iobjFor(PutIn) { preCond = [touchObj, objOpen] 
        action() {
            inherited;
            makeOpen(nil);
            "You put the battery in the flashlight and close it. ";
        }
    }
    
    notifyRemove(obj)
    {
        if(obj == battery && isOn)
        {
            "The flashlight goes out. ";
            makeOn(nil);
        }
    }
    
    /* 
     *   Handling for picking up the flashlight while carrying the pot has 
     *   to be added here, because the customization in the modify Thing 
     *   block in pepper.t won't be used in an object that has its own 
     *   notifyMoveInto.
     */
    notifyMoveInto(newCont) {
        if (newCont == me) {
            if (pot.location == me) {
                "You find that it\'s impossible to carry anything else
                while holding the heavy pot. ";
                exit;
            }
            return;
        }
        if (!isOn) return;
        /* 
         *   So now we know that the flashlight is on, which means it must 
         *   be in the PC's about to be moved to somewhere OTHER than the PC.
         */
        if ((me.location == cellar) && (!bulb.isLit)) {
            "As you\'re about to set down the flashlight, it occurs to you that
            its spring-loaded switch will cause it to go off, leaving you in
            the dark (with no way to find the flashlight again). Better not
            drop it in a dark place. ";
            exit;
        }
        makeOn(nil);
        "As the flashlight leaves your hand, the spring-loaded switch snaps to the
        <.s>off<./s> position. ";
    }
    
    openStatusReportable = (isOpen)
    canBeMagicallyLocked = nil
    achievement: Achievement { +2 "getting the small flashlight to work" }
;

+ RoomPartItem, OpenableContainer, CustomFixture 
    'battered usual kitchen cabinet/cupboard*appliances' 'cabinet'
    "It\'s seen better days, but it still appears functional. " 
    cannotTakeMsg = 'It may look a bit battered, but it\'s fixed firmly to
        the wall. '
    specialNominalRoomPartLocation = defaultNorthWall
    specialDesc = "A battered kitchen cabinet is built into the north wall. "
    collectiveGroups = [applianceGroup]
    bulkCapacity = 40
    maxSingleBulk = 40
;

++ butterKnife: Thing 'stainless steel blunt small (butter) blade/knife*knives' 
    'butter knife'
    "It's a small knife, made of stainless steel. It looks as if it should cut
    through butter readily enough, but it probably wouldn't make much of an
    impression on a rump steak. "
    bulk = 7
    iobjFor(MoveWith)
    {
        verify() {}
        check()
        {
            if(gDobj != keyBunch || !keyBunch.isIn(brakeLever))
                failCheck('{You/he} hardly need{s} to use {the iobj/him} to
                    move {that dobj/him} around. ');
        }
    }
    
    dobjFor(Move)
    {
        action()
        {
            if(isIn(brakeLever) && keyBunch.isIn(brakeLever))
            {
                "By wiggling the knife around behind the brake lever you manage
                to dislodge the bunch of keys, which promptly falls onto the
                floor. ";
                keyBunch.moveInto(getOutermostRoom);
                butterKnife.moveInto(gActor);
                keyBunch.achievement.awardPointsOnce();
            }
            else
                inherited;
        }     
    }
    
    dobjFor(Pull) maybeRemapTo(isIn(brakeLever), Move, self)
    dobjFor(Push) maybeRemapTo(isIn(brakeLever), Move, self)
    iobjFor(TurnWith) {
        verify() {}
    }
;

/* 
 *   'glass' can be a noun or an adjective here, since this object could be 
 *   referred to as a "chipped glass" or a "glass tumbler"
 */

++ chippedGlass: Thing
    '(brown) oily chipped full empty glass swirls/glass/tumbler/water' 'glass'
    "It\'s chipped on one edge. It\'s currently <<getState.desc()>><<(waterType
        == tapWater) ? ', which is rather brown, with oily swirls curling
            across the surface' : ''>>. "
    allStates = [glassFull, glassEmpty]
    getState = glassEmpty
    waterType = nil
    bulk = 15
    disambigName = 'chipped glass'
    /* 
     *   We're going to sneakily use the available makeDamp method to fill 
     *   the glass if you put it in the basin or the sink while the faucet 
     *   is flowing:
     */
    makeDamp (obj) {
        if (getState == glassFull) return;
        getState = glassFull;
        
        if (obj != birdbath && obj != birdbathWater) {
            waterType = tapWater;
            "As the faucet is running, the glass quickly fills with water. ";
        }
        else {
            waterType = freshWater;
            birdbath.achievement.awardPointsOnce();
            "You fill the glass with fresh water from the birdbath. ";
        }
    }

    dobjFor(Pour) asDobjFor (Empty)
    dobjFor(Empty)
    {
        verify() {}
        action()
        {
            /* 
             *   Emptying the glass is equivalent to pouring its contents 
             *   onto the ground.
             */
            replaceAction(PourOnto, self, getOutermostRoom.roomFloor);               
        }
    }
    
    /* 
     *   The player may try 'put water in pot', which would cause the glass 
     *   itself to be moved into the pot container:
     */
    notifyMoveInto(newCont) {
        if ((newCont == pot) && (getState != glassEmpty)) {
            replaceAction(PourInto, self, pot);
        }
        else inherited(newCont);
    }
    dobjFor(Drink)
    {
        verify()
        {
            if(getState == glassEmpty)
                illogicalNow(emptyMsg);
        }
        action()
        {
            "You drink all the water in the glass; it tastes <<tasteDesc>>. ";
            getState = glassEmpty;
            waterType = nil;
        }
    }
    
    dobjFor(Taste)
    {
        verify()
        {
            if(getState == glassEmpty)
                illogicalNow(emptyMsg);
        }
        action() { "The water in the glass tastes <<tasteDesc>>. "; }
    }

    dobjFor(PourInto)
    {
        verify()
        {
            if(getState == glassEmpty)
                illogicalNow(emptyMsg);
        }
        
        action()
        {
            /* 
             *   The elf will object if you try to pour tap water into the 
             *   pot, so if we've reached the action stage and the gIobj is 
             *   the pot, we know that the water it rainwater. So we'll let 
             *   the elf handle it. However, elf.addWater doesn't empty the 
             *   glass (since it can also be reached by summoning rain 
             *   directly on the elf), so we'll do that here:
             */
            if (gIobj == pot) {
                elf.addWater;
                getState = glassEmpty;
                waterType = nil;
                return;
            }
            if (gIobj == holes)
                "{You/he} pour{s} all the water from <<theName>> into one of
                the holes; it rapidly seeps away into the ground. ";
            else
                
                "{You/he} pour{s} all the water from <<theName>> into {the
                iobj/him}. ";
            getState = glassEmpty;
            waterType = nil;
        }
    }
    
    dobjFor(PourOnto)
    {
        verify()
        {
            if(getState == glassEmpty)
                illogicalNow(emptyMsg);
        }
        check()
        {
            if (((gIobj == elf) || (gIobj == pot)) && (!elf.isWatered)
                && (waterType == tapWater))
            {
                gReveal('need-water');
                failCheck('When you lean over the pot and start to tip the
                    glass of tap water into it, the elf writhes and convulses
                    as if an electrical current were shooting through him.
                    <.q>Nooooo!<./q> he cries in a quivering little voice.
                    <.q>Nasty chemicals! Nooo, please. Need nice fresh water,
                    not nasty chemical water.<./q> He falls back, sobbing. ');
            }
        }
        action()
        {
            if ((gIobj == pot) || (gIobj == elf)) {
                elf.addWater;
                getState = glassEmpty;
                waterType = nil;
                return;
            }
            "{You/he} pour{s} all the water from <<theName>> onto {the
            iobj/him}. ";
            getState = glassEmpty;
            waterType = nil;
            if (gIobj.ofKind(defaultFloor)) return;
            /*  
             *   Still here? Then we must be outdoors. The second condition 
             *   below traps the possibility that the player has somehow 
             *   managed to get the elf to wear the wig:
             */
            if (!blondWig.isWorn || (blondWig.location != me)) 
                return;
            /*  
             *   Okay, we've just poured water on the ground while wearing 
             *   the wig:
             *
             *   Next check if the PC is carrying the bamboo. If not the 
             *   rain-making spell won't quite work, so we'll display a 
             *   message describing its failing near-success.
             */
            if(bambooStick.location != me)
            {
                "Within moments after the water splashes onto the ground, a small
                and rather wispy cloud gathers over your head. For a few moments
                you fancy you feel a fine drizzle on your face, but it\'s over
                almost as soon as it\'s begun, The cloud dissipates,
                leaving a clear blue sky and no sign of moisture anywhere. ";
                return;
            }
            "Within moments after the water splashes onto the ground, a very
            small but dense gray cloud gathers over your head. The brief rumble
            of thunder sounds more like the rattling of a sheet of tin in a
            cheap theatrical production than like real thunder --- but all the
            same, the cloud dumps a downpour of rain onto the ground for about
            ten seconds before dissipating as abruptly as it gathered. As the
            day is warm, the water on the ground swiftly evaporates in tendrils
            of steam. ";
            rainHasBeenSummoned = true;
            
            /* 
             *   Is the elf here? If so, he will be watered, irrespective of 
             *   whether we're in the back yard or not. Bear in mind, 
             *   however, he might ALREADY have been watered.
             */
            local r = getOutermostRoom;
            if ((r == elf.getOutermostRoom) && (!elf.isWatered)) {
                elf.addWater;
                return;
            }
            
            /* Next question: Are we in the back yard? */
            if (getOutermostRoom != backYard) return;
            
            /* Has the birdbath _already_ been filled? */
            if (birdbath.containsWater) return;
            
            /* 
             *   Still here? Then we've just filled the birdbath with rain 
             *   water:
             */
            birdbath.containsWater = true;
            birdbathWater.moveInto(birdbath);
            basinGrime.moveInto(nil);
            "A good bit of water has fallen into the birdbath, however --- so
            much that it\'s in no danger of evaporating. The birdbath is now
            full of fresh rainwater. ";
        }
    }
    rainHasBeenSummoned = nil
    iobjFor(PutIn) {
        verify() {}
        check() {
            if (gDobj != birdbathWater)
                failCheck ('Putting that in the glass probably wouldn\'t
                    accomplish very much. ');
        }
        action() {
            replaceAction(FillWith, self, gDobj);
        }
    }
    dobjFor(FillWith)
    {
        verify()
        {
            if(getState == glassFull)
                illogicalAlready('The glass is already full. ');
        }
    }
    iobjFor(MoveWith) {
        verify() {}
        check() {
            if (gDobj != birdbathWater) failCheck ('There doesn\'t
                seem to be any way to scoop up {the dobj/him} with
                the glass. ');
        }
        action() {
            replaceAction (FillWith, self, birdbathWater);
        }
    }
    dobjFor(ThrowAt)
    {
        verify()
        {
            illogical('You\'d better not start throwing the glass around; you
                might break it. ');
        }
    }
    
    okayDropMsg = '{You/he} put{s} {the dobj/him} down carefully. '
    tasteDesc = (waterType == tapWater ? 'stale, flat, and rather bitter' :
                 'nice and fresh')    
    
    emptyMsg = 'The glass is empty. '
    
    shouldNotBreakMsg = 'You\'d better not; apart from the danger of cutting
        yourself on the sharp edges, you might not be able to find another
        glass. '
;

+ sink: RoomPartItem, Container, CustomFixture 
    'large (kitchen) usual porcelain sink/basin*appliances' 'sink'
    "The large porcelain sink is built against the east wall, near the window.
    A faucet is mounted above the sink. "
    specialNominalRoomPartLocation = defaultEastWall
    specialDesc = "A large sink nestles against the east wall. "
    collectiveGroups = [applianceGroup]
    bulkCapacity = 50
    maxSingleBulk = 50
    iobjFor(PourInto) { verify() { } }
    cannotCleanMsg = 'That\'s definitely a job for Mrs. Pepper. '
    cannotEnterMsg = 'It\'s a sink, not a bath! '
    cannotBoardMsg = (cannotEnterMsg)
    
    notifyInsert(obj, newCont) {
        if(kitchenTap.isOn)
            obj.makeDamp(self);        
    }
    dobjFor(TurnOn) remapTo(TurnOn, kitchenTap)
;

++ kitchenTap: Faucet 'chrome burnished gloss/faucet/tap/spigot' 'faucet'
    "The chrome of the faucet has been burnished to a terrifying gloss. "
    
    wetLocation = sink
    
;

modify defaultGround
    iobjFor(PourOnto) { verify() { } check() {} }
;

modify Thing
    iobjFor(PourOnto)
    {
        verify() {}
        check()
        {
            failCheck(cannotPourOntoMsg);
        }
        action() { gDobj.makeDamp(self); }
    }
   cannotPourOntoMsg = iobjMsg( getOutermostRoom.ofKind(OutdoorRoom) ? 'Pouring
       the water there won\'t achieve much. ':  'Emptying the glass here would
           only make a mess. ')
;


glassFull: ThingState 'full of water'
    desc = (listName_)
    stateTokens = ['full', 'from', 'water']
;

glassEmpty: ThingState
    desc = (stateTokens[1])
    stateTokens = ['empty']
;

enum freshWater, tapWater;

class Faucet: Switch, Component 'running tap/faucet/water/jet/spout' 'faucet'
    
    dobjFor(Open) asDobjFor(TurnOn)
    dobjFor(Close) asDobjFor(TurnOff)
    makeOn(stat)
    {
        inherited(stat);
        
        /* 
         *   Don't display any report if the faucet is turned on through an 
         *   implicit action.
         */
        if(gAction.isImplicit)
            return;
        
        "You turn <<theName>> <<stat ? 'on' : 'off'>>, and
        <<stat ? 'a jet of water starts to flow from the spout' : 'the water
            stops running'>>. ";
        if(isOn && underObj && underObj.getState == glassEmpty)
        {
            underObj.getState = glassFull;
            underObj.waterType = tapWater;
            "\^<<underObj.theName>> under <<theName>> fills up with water. ";
        }
        
        if(isOn && underObj)
            underObj.makeDamp(self);
        
        if(stat)
        {
            foreach(local obj in wetLocation.contents)
                obj.makeDamp(self);
        }
    }
    
    /* The location - basin or sink - that this faucet puts water in when on */
    wetLocation = nil
    vocabLikelihood = (isOn ? 10 : 5)
    dobjFor(Drink) {
        preCond = []
        verify() {
            if (!isOn) illogicalNow ('There\'s no water coming from the faucet. ');
        }
        action() {
            "You stick your face under the faucet and manage to get a sip of water.
            Your mother always tells you to use a glass, but what does she know? ";
        }
    }
    examineStatus()
    {
        if(isOn)
            "A jet of water is streaming from the spout. ";
    }
    
    dobjFor(Turn) asDobjFor(Switch)
    
    iobjFor(PutUnder)
    {
        verify() 
        { 
            if(gDobj && gDobj == underObj)
                illogicalAlready('{The dobj/he} {is} already under {the
                    iobj/him}. ');
        }
        check()
        {
            if(isOn && gDobj != chippedGlass)
                failCheck('You don\'t want to soak {the dobj/him}. ');
            
            if(isOn && gDobj == chippedGlass && chippedGlass.getState ==
               glassFull)
                failCheck('{The dobj/he} {is} already full. ');
        }
        
        action()
        {         
            
            "You hold {the dobj/him} under {the iobj/him} "; 
            if(isOn)
            {
                "until {it dobj/he} fill{s} up with water. ";
                gDobj.getState = glassFull;
                gDobj.waterType = tapWater;
            }
            else if(gDobj == chippedGlass)
            {
                "but since {the iobj/he} {is}n\'t running, {the dobj/he}
                remain{s} empty. ";
                new Fuse (self, &moveGlass, 2);
                underObj = gDobj;
            }
            else
                " for a few moments, but since this doesn't achieve anything
                you take {it dobj/him} away again. ";
        }
        
    }
    
    moveGlass()
    {
        if(underObj.isIn(getOutermostRoom) && me.isIn(getOutermostRoom))
            "You move <<underObj.theName>> out from under <<theName>>. ";
        underObj = nil;
        
    }
    
    iobjFor(FillWith)
    {
        verify() {}
        action()
        {
            if(!isOn)
                tryImplicitAction(TurnOn, self);
            
            if(underObj != gDobj)
                tryImplicitAction(PutUnder, gDobj, self);
        }
    }
    
    underObj = nil
    
    allStates = [tapOff, tapRunning]
    getState = (isOn ? tapRunning : tapOff)    
;

tapOff: ThingState
;

tapRunning: ThingState
    stateTokens = ['running', 'jet', 'water']
;

//------------------------------------------------------------------------------
// The cellar -- an important and interesting place. (Also dark.)

modify DefaultWall
    iobjFor(ThrowAt) {
        verify () {
            if (gDobj && gDobj != glassJar) inherited;
        }
        check () {
            if (gDobj != glassJar) inherited;
        }
    }
;

cellar: DarkRoom 'Cellar'
    "The cellar is dim and dismal. A flight of stairs leads up. Near the foot
    of the stairs a slim chain dangles from the ceiling. Off to one side squats
    a heavy piece of machinery, and a plank shelf has been built into the
    opposite wall. "
    up = cellarStairs
    roomDarkName = 'Cellar (in pitch darkness)'
    roomDarkDesc = "You can\'t see anything in here, apart from the dim outline
        of the stairs leading back up. <.reveal cellar-dark>"
    
    roomBeforeAction()
    {
        if(gActionIs(Drop) && senseAmbientMax([sight]) < 2)
            failCheck('You\'d better not drop anything down here while it\'s 
                dark; you might not be able to find it again. ');
    }
    
    vocabWords = 'cellar/room'
;

+ bulbChain: Fixture 'slim chain' 'chain'
    "The chain dangles down from the ceiling. At its upper end, a bare
    lightbulb is mounted on a rafter. "
    dobjFor (Pull) {
        verify () {}
        check () 
        {
            if(bulb.isLit && !penLight.isLit)
                failCheck('If you plunge yourself into darkness now you might
                    not be able to find either the flashlight or the chain
                    again in the dark. ');
        }
        action () {
            "You pull the chain, and the cellar ";
            if (bulb.isLit) {
                bulb.makeLit(nil);
                "is plunged into near-darkness. ";
                if (penLight.isLit)
                    "The flashlight continues to provide a little illumination,
                    however. ";
                return;
            }
            bulb.makeLit(true);
            "is illuminated by a bare lightbulb mounted on a rafter at the
            upper end of the chain. ";
        }
    }
    cannotMoveMsg = 'The chain swings to and fro a few times, eventually coming
        to rest. '
    
;

+ bulb: Distant, LightSource 'light bare lightbulb/bulb' 'bulb'
    "It's bare and <<isLit ? 'lit' : 'unlit'>>. "
    tooDistantMsg = 'You can\'t reach it. '
    isLit = nil
;

+ Distant 'plain wooden rafter*rafters' 'rafter'
    "It's just a plain wooden rafter. "
    tooDistantMsg = 'You can\'t reach it. '
;


+ cellarStairs: StairwayUp ->kitchenStairs
    'dim outline/flight/staircase/stairs' 'staircase'
    "The stairs lead up to the kitchen. "
    brightness = 1    
;

+ plankShelf: OutOfReach, Surface, Fixture 'plank shelf' 'shelf'
    "The shelf is built high against one
    wall.<<glassJar.discover>><<brooch.discover>> "
    canObjReachContents (obj)
    {
        if ((gActionIn(AttackWith, MoveWith)) && (gIobj == rake)) 
            return true;
        
        if (obj.isIn(trashcan.subSurface) 
            && (obj.posture == standing) && (trashcan.nominalLocation == self)) 
            return true;
        
        return obj.isIn(wheelchair) && obj.posture == standing &&
            wheelchair.nominalLocation == self;
    }
    iobjFor(PutUnder) remapTo(MoveTo, DirectObject, self)
    iobjFor(MoveTo) { verify() { } }  
    cannotReachFromOutsideMsg(dest)
    {
        gMessageParams(dest);
        return 'You can\'t reach {the dest/him} because the shelf is too high. ';
    }
    bulkCapacity = 50
    maxSingleBulk = 50
;

++ glassJar: Hidden, Container 
    'broken glass dirty dusty jar/lid/remains/cylinder' 'glass jar'
    "The glass jar seems to have been used at one time to hold paint. The
    interior surface is splattered with flecks of white, and a few dribbles
    have flowed down the outside and dried. <<getState.desc>>"
    bulk = 20
    bulkCapacity = 10
    maxSingleBulk = 10
    getState = jarDusty
    material = glass
    isOpen = (getState == jarBroken)
    allStates = [jarDusty, jarSound, jarBroken]
    cannotPourIntoMsg = (getState.cannotPourIntoMsg)
    cannotOpenMsg = 'The lid seems to be stuck. '
    
    cannotTouchThroughMsg = 'That would be difficult, since the jar isn\'t
        open. '
    
    dobjFor(Shake) {
        preCond = [objHeld]
        check() {
            if (getState == jarBroken) failCheck ('Shaking the bits of
                broken glass would be fairly pointless. ');
        }
        action() {
            "You give the jar a shake. Whatever is inside it makes a metallic
            rattling noise. ";
        }
    }
    dobjFor(Drop) {
        action() {
            if ((me.location != wheelchair) || (me.posture != standing))
                inherited;
            else
            {
                getState = jarBroken;
                moveInto(cellar);
                "The jar plummets to the floor and breaks. ";
            }
        }
    }
    dobjFor(Break)
    {
        preCond = [objHeld]
        verify()
        {
            if(getState == jarBroken)
                illogicalAlready('{The dobj/he} {is} already broken. ');
        }
        action()
        {
            "You smash the jar against the
            <<getOutermostRoom.ofKind(OutdoorRoom) ? 'ground' : 'nearest wall'>>,
            and pieces of
            glass fly in all directions, narrowly missing you. You
            are left holding the remains of the jar, an open half-cylinder of
            glass with the lid still stuck firmly in place on one end. A gold
            brooch lies nestled in the open cylinder. ";
            getState = jarBroken;
            
            /* In case the jar is broken before it's cleaned. */
            brooch.giveName; 
        }
    }
    dobjFor(Open) {
        verify() {
            if (getState == jarBroken) illogicalAlready('You\'ve already
                opened the jar (by breaking it). ');
        }
        check() {
            if (!location == me)
                failCheck ('To do that, you\'d need to be holding it. ');
            else failCheck (cannotOpenMsg);
        }
    }
    dobjFor(Attack) asDobjFor(Break)
    dobjFor(MoveWith) asDobjFor(AttackWith)
    dobjFor(AttackWith) {
        verify() {}
        check() {
            if ((gIobj != rake) && (location == plankShelf))
                failCheck ('You can\'t quite reach
                    the jar with {the iobj/him}. ');
            if (getState == jarBroken) failCheck ('There\'s surely no reason
                    to do that. You\'ve already broken it. ');
            if (location != plankShelf)
                failCheck ('The jar\'s lid proves too stubborn for that to
                work. ');
        }
        action() {
            "You knock the jar from the shelf with the rake. It falls to the
            floor, and pieces of glass fly in all directions, narrowly missing
            you. In the remains of the jar, an open half cylinder of glass, lies
            a gold brooch! ";
            getState = jarBroken;
            moveInto(getOutermostRoom);
            brooch.giveName;
            achievement.awardPointsOnce();
        }
    }                 

    dobjFor(Clean)
    {
        verify()
        {
            if(getState == jarSound)
                illogicalAlready('You\'ve already cleaned it. ');
            if(getState == jarBroken)
                illogicalNow('There\'s not much point trying to clean a broken
                    jar.' );
        }
        
        action()
        {
            getState = jarSound;
            "You wipe the dust of the jar, giving you a much better look at
            what's inside, which turns out to be a gold brooch. ";
            brooch.giveName;
        }
    }
    
    dobjFor(Take)
    {
        action() 
        {
            inherited;
            achievement.awardPointsOnce();
        }
    }
    achievement: Achievement { +3 "getting the glass jar down from the shelf" }
    
    throwTargetHitWith(projectile, path)
    {
        if(moved)
            inherited(projectile, path);
        else
        {
            gMessageParams(projectile);
            "{The projectile/he} just misses the jar, bounces off the wall
            behind, and lands on the floor. ";
            projectile.moveInto(getOutermostRoom);
        }
        
    }
    dobjFor(ThrowAt) {
        preCond = [objHeld]
        verify() {}
        check() 
        {
            if (getState == jarBroken) 
                failCheck('You\'ve already broken the jar --- throwing it
                    hardly seems necessary. ');
        }
        
        action() 
        {
            local r = getOutermostRoom;
            moveInto(r);
            if (!gIobj.ofKind(DefaultWall)) 
            {
                "You toss the jar at {the iobj/him}, but miss. ";
                return;
            }
            getState = jarBroken;
            "You smash the jar against the nearest wall, and pieces of
            glass fly in all directions, narrowly missing you. Lying on the
            <<getOutermostRoom.ofKind(OutdoorRoom) ? 'ground' : 'floor'>>
            are the shattered remains of the jar, an open half-cylinder of
            glass with the lid still stuck firmly in place on one end. A gold
            brooch lies nestled in the open cylinder. ";
        }
    }
;

+++ jarDusty: ThingState
    desc = "The jar is quite dusty on the outside. "
    cannotPourIntoMsg = (jarSound.cannotPourIntoMsg)
    stateTokens = ['dirty', 'dusty']
;

+++ jarSound: ThingState 
    desc = ""
    cannotPourIntoMsg = '{You/he} {can\'t} pour anything into {the iobj/him}
        while {it\'s dobj/he\'s} closed. '
;

+++ jarBroken: ThingState 'broken'
    stateTokens = ['broken', 'remains']
    desc = "The jar is now broken. "
    cannotPourIntoMsg = 'There\'s not much point trying to pour anything into
        {the iobj/him} now that {it\'s dobj/he\'s} broken. '
;

/* Including the misspelling 'broach' because some players might use it: */
+++ brooch: Hidden, Wearable
    'antique small gold metallic golden brooch/broach/object/jewelry' 
    'small metallic object'
    "It's an antique gold brooch, roughly round and about an inch and a half in
    diameter.<<giveName>> "
    initDesc = "It's a little hard to make out through the dusty glass; it's
        obviously not very large, possibly round in shape, and a little yellowy
        in colour. "
    isInInitState = (isIn(glassJar) && glassJar.getState == jarDusty)
    bulk = 5
    isKnown = (gRevealed('brooch'))
    ownedBy = (gRevealed('brooch') ? neighbour : nil)
    hasBeenNamed = nil
    
    /* 
     *   giveName is called by the neighbour object, and is useful because 
     *   you might perchance tell her about the shiny object you saw in the 
     *   jar before you actually know it's a brooch, in which case she'll 
     *   tell you.
     */
   
    giveName { name = 'gold brooch'; hasBeenNamed = true; }
;

+ ceilingHoles: Decoration '(small) holes' 'holes'
    "The holes are small. The braided metal cable runs up through them into
        the ceiling. Really, they don't look important at all. "
    isPlural = true
;

+ machinery: CustomFixture
    '(metal) (elevator) heavy sturdy piece/machinery/machine/housing/motor' 
    'machinery'
    "The innards of the machine are concealed by a metal housing. On one side
    is a large vertically mounted wheel, around both sides of which a sturdy
    cable runs up through the ceiling. Not far from the cable a black-coated
    wire dangles down from the ceiling. <<isFixed ? 'The wire is attached to a
        screw on the top of the machine. ' : 'The lower end of the wire swings
            free not far from a screw that protrudes from the top of the
            machine. '>><<nothingBehindMsg>> "
    isMassNoun = true
    isFixed = nil
    nothingBehindMsg = 'Mounted in the north wall just behind the machine is a
    small metal door. '
    cannotStandOnMsg = 'Although you probably could clamber up on top of the
        machine, it doesn\'t look a particularly secure perch, let alone a
        comfortable one. '
    cannotSitOnMsg = (cannotStandOnMsg)
    cannotLieOnMsg = 'You might just about manage to lie on top of the
        machinery but it would be <i>extremely</i> uncomfortable and utterly
        pointless. '
    cannotTakeMsg = 'It looks impossible to move the machinery and unwise to
        try. '
    cannotJumpOverMsg = 'You\'d rather not risk it; there are too many things
        to collide with or get caught up on. '
    cannotClimbMsg = (cannotStandOnMsg)
    feelDesc = "It feels pretty solid. "
    nothingUnderMsg = 'The machinery is resting on the floor; you can\'t see
        under it. '
    cannotCloseMsg = 'It\'s not open. '
    cannotOpenMsg = 'You don\'t have the tools. '    
    cannotCleanMsg = dobjMsg('It does look quite dirty, but you don\'t have the
        right cleaning materials to hand, and you were never one for domestic
        chores. ')
    cannotAttachToMsg = iobjMsg('You\'ll have to specify what part of the
        machinery you want to attach {that dobj/him} to. ')
    dobjFor(TurnOff)
    {
        verify()
        {
            if(!(machinery.isFixed && circuitBreaker.isOn))
                illogicalAlready('It isn\'t on. ');
        }
        action()   {   replaceAction(TurnOff, circuitBreaker);  }
        
    }
    
    dobjFor(TurnOn)
    {
        verify()
        {
            if(machinery.isFixed && circuitBreaker.isOn)
                illogicalAlready('It seems to be on already.' );
        }
        check()
        {
            if(!machinery.isFixed) {
                gReveal('machine-repair');
                failCheck('That might be a good idea if you could work out how.
                    ');
            }
        }
        action()   {  replaceAction(TurnOn, circuitBreaker);   }
    }
;

++ Component 'large (vertical) vertically mounted wheel' 'wheel'
    "The wheel is mounted vertically on the side of the machine. A thick metal
    cable loops around it, extending upward through the ceiling. "    
    dobjFor(Turn)
    {
        verify() {}
        action()  { "It proves to be too stiff to turn by hand. ";  }
    }
    dobjFor(Move) asDobjFor(Turn)
    dobjFor(Push) asDobjFor(Turn)
    dobjFor(Pull) asDobjFor(Turn)
;

++ Component 'sturdy thick braided (metal) cable' 'braided metal cable'
    "The thick braided metal cable loops around the wheel on the side of the
    machine. From both sides of the wheel, it extends upward through small
    holes in the ceiling. "
    
    dobjFor(Move)
    {
        verify() {}
        action() { "It turns out to be too stiff to move by hand. "; }
    }

    dobjFor(Push) asDobjFor(Move)
    dobjFor(Pull) asDobjFor(Move)
    dobjFor(MoveWith) asDobjFor(Move)
    dobjFor(MoveTo) asDobjFor(Move)
    dobjFor(PushTravel) asDobjFor(Move)

    
    dobjFor(Climb) {
        verify() {}
        check() {
            failCheck('There\'s nothing visible at the upper end, just a couple
                of small holes in the ceiling. Anyway, it\'s too slippery to
                climb. ');
        }
    }
;

++ SimpleNoise 'low humming sound' 'low humming sound'
    "A low humming sound comes from the machinery. "
    isEmanating = (machinery.isFixed && circuitBreaker.isOn)
;

++ wire: Component
    'black-coated coated insulating black gleam bare metal material/wire' 
    'black wire'
    "The wire dangles down from the ceiling. It\'s covered with black insulating
    material, except for an inch or so at the lower end, which has the gleam of
    bare metal. <<machinery.isFixed ? 'The end of the wire is attached to the
        screw on the top plate of the machinery. ' : ''>>"
    dobjFor(Move) asDobjFor(Take)
    dobjFor(Pull) asDobjFor(Take)
    dobjFor(Take) {
        verify() {}
        check() {
            if (machinery.isFixed) failCheck ('There\'s no reason to disconnect
                the wire from the machine. ');
        }
        action() {
            "You grip the end of the wire between two fingers and give it a
            little tug. It seems to be securely attached to something up in the
            ceiling, but this end is dangling loose --- not the usual situation,
            with electrical wires. ";
        }
    }
    iobjFor(AttachTo) {
        verify() {}
        check() 
        {
            if (gDobj != machineScrew) 
                failCheck ('{That dobj/he} {is} not something that can be
                    attached to the wire. ');
        }
    }       
    dobjFor(PutOn) remapTo(AttachTo, self, IndirectObject)
    dobjFor(AttachTo) {
        verify() {}
        check() 
        {
            if (gIobj != machineScrew)
                failCheck ('You can\'t attach the wire to that. ');
            
            if (machinery.isFixed)
                failCheck ('You\'ve already attached the wire to the screw. ');
            
            if (screwdriver.location != me)
                failCheck ('To do that, you\'ll need a screwdriver. ');
            
            // local n = me.countHeld();
            if (me.countHeld > 1)
            {
                failCheck ('It appears you\'ll need both hands. Trying to hold
                    anything other than the screwdriver while doing that is
                    just too awkward. ');
            }
        }
        
        action() 
        {
            machinery.isFixed = true;
            achievement.awardPointsOnce();
            "Using the screwdriver, you attach the bare metal end of the wire
            to the screw on the top of the machine. ";
        }
    }
    achievement: Achievement { +5 "attaching the wire to the machine" }
    
    /* verifyTouch() is called by the modified version of touchObj */
    verifyTouch(obj)
    {
        if(obj == me && circuitBreaker.isOn)
        {
            illogicalNow('Touching the wire gives you a severe shock, so you
                let go of it at once.<.reveal shock> ');            
        }
    }
    
    feelDesc()
    {
        if(gRevealed('shock'))
        {
            "The wire fails to shock you. Evidently the switch on the wall is
            the circuit breaker. ";
            circuitBreaker.name = 'circuit breaker';
        }
        else
            inherited;
    }
;

++ machineScrew: Component 'screw' 'screw'
    "The screw protrudes from the top plate of the machine housing.
    <<machinery.isFixed ? 'The end of the wire is attached to the screw. ' :
      ''>>"
    dobjFor(AttachTo) 
    {
        verify() {}
        check() 
        {
            if (gIobj != wire) 
                failCheck ('You can\'t attach the screw to that. ');
            
            failCheck('The screw isn\'t movable. You\'ll need to attach the
                wire to the screw, not the other way around. ');
        }
    }
    iobjFor(PutOn) asIobjFor(AttachTo)
    iobjFor(AttachTo) 
    {
        verify() {}
        check() 
        {
            if (gDobj != wire)
                failCheck ('You can\'t attach that to the screw. ');
        }
    }
    
    cannotMoveMsg = 'You can\'t budge it. '
    
    dobjFor(TurnWith) {
        verify() {}
        
        check() 
        {
            if (gIobj == butterKnife) 
                failCheck ('Unfortunately, it\'s a Phillips-head screw. ');
            if (gIobj != screwdriver) 
                failCheck ('That\'s not something you can turn the screw
                    with. ');
        }
        
        action() 
        {
            "You give the screw a little twist with the screwdriver. ";
        }
    }
    dobjFor(Move)
    {
        verify() {}
        action() { reportFailure(cannotMoveMsg); }
    }
    dobjFor(Turn) asDobjFor(Move)
    dobjFor(Push) asDobjFor(Move)
    dobjFor(Pull) asDobjFor(Move)
;

+ RoomPartItem, OpenableContainer, CustomFixture 
    'small metal grey gray recessed compartment/box/door*doors' 
    name = (isOpen ? 'small metal box' : 'small metal door')
    desc = "It's <<isOpen ? 'a small metal box, recessed into the wall behind a
        small gray door, and containing a large red switch' : 'a small, gray,
            metal door set into the north wall'>>. A thick cable runs up from
        the <<isOpen ? 'box' : 'door'>> to the ceiling. "
    bulkCapacity = 1
    openingLister: openableOpeningLister
    {
        showListEmpty(pov, parent) 
        { 
            "Opening the metal door reveals a small box containing a large red
            switch. ";
        }
    }
    lookInDesc = "The recessed box contains a large red switch. "
    cannotJumpOverMsg = &mustBeJokingMsg
    cannotEnterMsg = 'You\'re not an ant! '
;

++ circuitBreaker: RoomPartItem, Switch, CustomFixture 
    'circuit large red breaker/switch*switches' 'large red switch'
    "It's <<aName>>, mounted on the wall in a gray box, and currently
    in the <.s><<isOn ? 'up (on)' : 'down (off)'>><./s> position."
    isOn = true
    makeOn(stat)
    {
        inherited(stat);
        if(stat && machinery.isFixed)
            "A low humming sound starts coming from the machinery. ";
        if(!stat && machinery.isFixed)
            "The machinery falls silent. ";
    }
    dobjFor(Move) asDobjFor(Switch)
    dobjFor(Pull) asDobjFor(Switch)
    dobjFor(Push) asDobjFor(Switch)
    dobjFor(Throw) asDobjFor(Switch)
    
    /* to allow 'move switch up', etc. */
    dobjFor(PushTravel)
    {
        verify()
        {
            local dirn = gAction.dirMatch.dir;
            if(dirn not in (upDirection, downDirection))
                inherited;            
        }
        action()
        {
            if(gAction.dirMatch.dir == upDirection)
                replaceAction(TurnOn, self);
            else
                replaceAction(TurnOff, self);
        }
    }
    bulk = 1
;

+ Decoration 'thick gray grey insulated cable' 'gray cable'
    "The thick gray cable looks heavily insulated; it runs up out of the circuit
    breaker box and off into the ceiling. "
    notImportantMsg = 'That sort of cable is best left alone; it looks like it
        could carry quite a current. '
;

+ wheelchair: Chair, TravelPushable 'battered old (wheel) wheelchair/chair' 
    'wheelchair'
    "It may still be functional, but it has clearly seen better days, and has
    hardly been treated with tender loving care. Part of the fabric is torn,
    and several spots of rust are appearing, not least on the brake lever. "
    bulkCapacity = 60
    maxSingleBulk = 60
    bulk = 200
    initDesc()
    {
        desc();
        "<.p>You recall your mother telling you how poor Mr.\ Pepper was found
        dead at the foot of the cellar stairs following a tragic accident
        involving his wheelchair. After your various encounters with his widow,
        you find yourself wondering just how accidental his accident actually
        was. ";
    }
    actorInPrep 
    {
        if (me.posture == standing) 
            return 'on';
        return 'in';
    }
    specialDesc() 
    {
        if(nominalLocation == nil)
            "A battered old wheelchair slumps in one corner. ";
        else
            "The old wheelchair is resting just by <<nominalLocation.theName>>.
            ";            
    }
    down asExit(out)
    nominalLocation = nil
    cannotTakeMsg = 'The old wheelchair would be far too awkward to carry. '
    isInInitState = (!described)
    
    lookInDesc()
    {
        if(!keyBunch.moved)
            "There seems to be something trapped behind the brake lever. ";
    }
    
    dobjFor(MoveTo)
    {
        verify()
        {
            if(gIobj && gIobj == nominalLocation)
                illogicalAlready('{The dobj/he} {is} already there. ');
        }
        
        check()
        {
            if(!brakeLever.isPulled)
                failCheck('Something is stopping the wheelchair from moving ---
                    probably the brake. <.reveal chair-brake>');
        }
        
        action()
        {
            nominalLocation = gIobj;
            "{You/he} manage{s} to push {the dobj/him} over to {the iobj/him}.
            <.reveal chair-moved>";
        }
    }
    
    dobjFor(PushTravel)
    {
        check()
        {
            checkDobjMoveTo();
            inherited;
        }
    }
    
    dobjFor(Push)
    {
        check() { checkDobjMoveTo(); }
        action()
        {
            "You'll need to be more specific about where you want to push it. ";
        }
    }
    
    dobjFor(Move) asDobjFor(Push)
    
    dobjFor(StandOn)
    {
        check()
        {
            if(brakeLever.isPulled)
                failCheck('The wheelchair starts to roll away as soon as you
                    try to stand on it. ');
        }
    }
    cannotJumpOverMsg = 'This doesn\'t look a good place to practice your
        limited acrobatic skills. '
    cannotCleanMsg = dobjMsg('Whatever for? It\'s not as if Mr. Pepper\'s going
        to use it any time soon! ')
    nothingUnderMsg = 'You peer under the wheelchair and see nothing but the
        floor. '
    cannotOpenMsg = 'It\'s as open as it\'s ever going to be. '
    
    canReachFromInside(obj, dest) 
    { 
        if(nominalLocation != nil && dest.isOrIsIn(nominalLocation))
            return true;
            
        return nil; 
    }
;

++ Decoration 'dirty black torn fabric' 'fabric'
    "The fabric covering the wheelchair is not only torn but dirty. "   
    dobjFor(Tear)  {  verify() { inherited Thing; }  }
    cannotTearMsg = 'Although the black fabric is already torn in places, it
        turns out to be too tough for you to be able to tear a piece off. '
    cannotCleanMsg = dobjMsg(location.cannotCleanMsg)
;

++ Decoration 'rust spots/patches/rust' 'rust'
    "There are patches of rust on all the exposed metal parts of the wheelchair.
    "
    canMatchThem = true
;

++ brakeLever: Lever, RestrictedRearSurface, Component 
    'short brake rusty exposed metal rubber lever/grip/part' 'brake lever'
    "It's just a short lever with a worn rubber grip at the end. The exposed
    metal part is beginning to look a bit rusty. Apparently the brake is
    <<isPulled ? 'off' : 'on'>>. <<contents.length() > 0 ? somethingJamming :
      ''>>  "
    
    somethingJamming()
    {
        something.makePresent();
        something.desc();
    }
    dobjFor(Doff) asDobjFor(Release)
    dobjFor(Wear) asDobjFor(Push)
    dobjFor(Pull)
    {
        check()
        {
            if(contents.length() > 0)
                failCheck('The lever is jammed; something seems to be stopping
                    it moving. ');
        }
    }
    validContents = [butterKnife]
    iobjFor(PutBehind)
    {
        action()
        {
            if(keyBunch.isIn(self))
            {
                if(!keyBunch.discovered)
                {
                    "The knife blade encounters an obstacle, which turns out to
                    be a bunch of keys. ";
                    keyBunch.discover();
                }
                replaceAction(MoveWith, keyBunch, gDobj);                
            }
            else
                inherited;
        }
    }

    dobjFor(Release) {
        preCond = [touchObj]
        verify() {}
        check() 
        {
            if (isPulled) 
                failCheck ('The brake has already been released. ');
        }
        action()  {  replaceAction (Pull, self);   }
    }
    dobjFor(Throw) asDobjFor(Move)
    dobjFor(Move)
    {
        remap()
        {
            return [isPulled ? PushAction : PullAction, self];
        }
    }
    
    dobjFor(TurnOff) asDobjFor(Pull)
    dobjFor(TurnOn) asDobjFor(Push)
    dobjFor(LookUnder) asDobjFor(LookBehind)
    
    cannotPutBehindMsg(obj) 
    { 
        if(obj==keyBunch)
            return 'Having gone to all the trouble to retrieve the keys from
                behind the brake lever, you hardly want to put them back
                there!';
        return inherited(obj); 
    }
   
;

+++ keyBunch: Hidden, Key 'key bunch/keys/ring/keyring' 'bunch of keys'
    "There are several keys on the key ring. "
    canMatchThem = true
    discover()
    {
        inherited;
        something.makePresentIf(nil);
        reportAfter('It looks like they must have fallen there from the
            owner\'s pocket. ');
    }
    
    dobjFor(Take)
    {
        check()
        {
            if(!moved)
                failCheck('The keys seem to be stuck behind the lever; you\'ll
                    need something to move them with, something thin enough to
                    fit behind the lever but sturdy enough to dislodge the
                    keys. ');
        }       
    }

    dobjFor(MoveWith)
    {
        verify()
        {
            if(moved)
                illogicalAlready('Now that you\'ve freed the keys from the
                    wheelchair, you don\'t need anything to move them with. ');
        }
        check()
        {
            if(gIobj != butterKnife)
                failCheck('{The iobj/he} do{es}n\'t look at all likely to do
                    the job. ');
        }
        action()
        {
            "You poke the knife behind the brake lever, and manage to dislodge
            the bunch of keys, which promptly falls onto the ground. ";
            moveInto(getOutermostRoom);
            achievement.awardPointsOnce();
        }
    }
    
    achievement: Achievement { +2 "recovering the bunch of keys" }
    
    #ifdef __DEBUG
    checkDobjSnarf() {}
    #endif
;

++ something: PresentLater, Decoration 'something' 'something'
    "<<notImportantMsg>>"
    notImportantMsg = 'It looks as if something may be jamming the lever from
        underneath, but you can\'t quite see what\'s back there without looking
        behind the lever. '
;

//------------------------------------------------------------------------------

stairUpBarrier: PushTravelBarrier    
    explainTravelBarrier(traveler)
    {
        "You can hardly push <<traveler.obj_.theName>> up the stairs. ";
    }    
;

stairDownBarrier: PushTravelBarrier
    explainTravelBarrier(traveler)
    {
        "It's not safe to try pushing <<traveler.obj_.theName>> down the
        stairs. ";
    }    
; 

skateUpBarrier: VehicleBarrier
    explainTravelBarrier(traveler)
    {
        "You've yet to master the art of skateboarding up a staircase. ";
    }    
;

skateDownBarrier: VehicleBarrier
    explainTravelBarrier(traveler)
    {
        "You suspect that trying to skateboard down a staircase would end in
        disaster. ";
    }    
;

modify StairwayUp
    travelBarrier = [stairUpBarrier, skateUpBarrier]
;

modify StairwayDown 
    travelBarrier = [stairDownBarrier, skateDownBarrier]    
;

class PushSkateBarrier: PushTravelBarrier
    construct(msg)
    {
        msg_ = msg;
    }
    msg_ = nil
    
    /* 
     *   This method is defined in such a way that we can use {the 
     *   obj/him} as a parameter substitution parameter in the msg we pass 
     *   to the constructor.
     */
    
    explainTravelBarrier(traveler)
    {
        local obj = traveler.obj_;
        gMessageParams(obj);
        reportFailure(msg_);
    }   
    
;

class RideSkateBarrier: VehicleBarrier
    construct(msg)
    {
        msg_ = msg;
    }
    msg_ = nil
    
    /* 
     *   This method is defined in such a way that we can use {the 
     *   obj/him} as a parameter substitution parameter in the msg we pass 
     *   to the constructor.
     */
    
    explainTravelBarrier(traveler)
    {
        local obj = traveler;
        gMessageParams(obj);
        reportFailure(msg_);
    } 
    
;

binBarrier: TravelBarrier
    canTravelerPass(traveler)
    {
        return trashcan.nominalLocation == kitchenWindow
            && trashcanLid.isIn(trashcan.subSurface);
    }
    
    explainTravelBarrier(traveler)
    {
        "You poke your head out the window and have a look around. But
        it\'s too long a drop to the ground, and <<trashcan.nominalLocation ==
          kitchenWindow ? 'without its lid, the trash can doesn\'t provide a
              safe step down' : 'the trashcan is too far away to be of
                  help'>>. ";
        
    }
;

modify TouchObjCondition
    verifyPreCondition(obj)
    {
        inherited(obj);
        obj.verifyTouch(sourceObj);
    }    
;

modify Thing
    verifyTouch(obj) { }
;


//------------------------------------------------------------------------------
// The elevator.

elevator: Room 'Small Compartment'
    "This chamber is not much larger than a closet<<hasBeenUsed ? '' : ', and
        in the right place for a coat closet --- but strangely, there are no
        coathangers'>>. The way out is to the east, next to a waist-height
    black button. "
    remoteDesc(pov)
    {
        "The chamber is not much larger than a closet, but there are no
        coathangers, and no hanging-rail. ";
    }
    vocabWords = 'small elevator/lift/chamber/closet/compartment'
    east = elevatorDoor
    out asExit(east)
    hasBeenUsed = nil
    
    /* 
     *   newDoor is the door object which will become the other side of 
     *   elevatorDoor, the elevator door as it is seen from the inside. By 
     *   changing which door the elevator door is linked to, we change where 
     *   an actor will go when leaving the elevator.
     */
    
    summon(newDoor)
    {
        local reportDoor = me.isIn(elevator) ? elevatorDoor : newDoor;
        gReveal('button-pushed');
        if(!isWorking)
        {
            mainReport(&pushButtonNoEffectMsg);
            return;
        }
        
        if(newDoor == elevatorDoor.masterObject)
        {
            if(elevatorDoor.isOpen)
                mainReport(&pushButtonNoEffectMsg);
            else
            {
                elevatorDoor.makeOpen(true);
                "\^<<reportDoor.theName>> slides open. ";
            }
        }
        else
        {
            if(me.isIn(elevator) && elevatorDoor.isOpen)
                "\^<<elevatorDoor.theName>> slides shut; a few moments later
                there is a brief humming noise, accompanied by a momentary
                sensation of <<newDoor == liftDoorUp ? 'heaviness' :
                  'lightness'>>. Then ";
            else
                "There's a short hum and then ";

            elevatorDoor.makeOpen(nil);
            elevatorDoor.masterObject = newDoor;
            elevatorDoor.initializeThing();
            hasBeenUsed = true;
            "<<reportDoor.theName>> slides open. ";
            elevatorDoor.makeOpen(true);
            achievement.awardPointsOnce();
        }
    }
    
    isWorking = (machinery.isFixed && circuitBreaker.isOn)
    achievement: Achievement { +2 "using the elevator" }

    dobjFor(Enter)
    {
        preCond = [objVisible]
        verify()
        {
            if(gActor.isIn(self))
                illogicalAlready('You\'re already there. ');
        }
        action()
        {
            replaceAction(TravelVia, self);
        }
    }
;

+ elevatorDoor: LiftDoor ->liftDoorDown
    'metal steel door*doors' 'steel door'   
    canBeMagicallyLocked = nil
;

+ LiftButton
    dobjFor(Push)
    {
        action() {  elevator.summon(myDoor);  }
    }
    myDoor = (elevatorDoor.masterObject == liftDoorUp ? liftDoorDown : liftDoorUp)
    specialNominalRoomPartLocation = defaultEastWall
    specialDesc = "A black button is set at waist-height on the east wall, just
        next to the exit. " 
    vocabLikelihood = 10
;

elevatorConnector: DistanceConnector [elevator, hall];

class LiftDoor: IndirectLockable, Door 
    'narrow steel sliding slot/doorway/frame/doorframe/opening/door*doors' 
    name = (isOpen ? 'opening' : 'steel door')
    disambigName = (isOpen ? 'doorway' : 'steel door')
    desc = "<<isOpen ? 'The opening is an ordinary doorway. Within the
        doorframe is a narrow slot that runs all the way around the frame,
        in a way that suggests this might be a sliding
        door. ' : 'The door has no visible knob or handle. It\'s
            recessed slightly in the wall. ' >>"
    cannotUnlockMsg = 'It doesn\'t have any obvious lock. '
    cannotLockMsg = (cannotUnlockMsg)
    canBeMagicallyLocked = nil
    dobjFor(Close) 
    {
        verify() {}
        check() 
        {
            failCheck('There\'s no obvious way to do that. ');
        }
    }
    dobjFor(LookThrough)
    {
        verify()
        {
            if(!isOpen)
                illogicalNow('You can\'t see anything through {the dobj/him};
                    it\'s firmly closed and quite opaque. ');
        }
        
        action()
        {
            "Through {the dobj/him} you see a chamber not much larger than a
            closet. ";
        }
    }
    examineStatus {}
    makeOpen(stat)
    {
        inherited(stat);
        if(stat)
            elevatorConnector.moveIntoAdd(location);
        else
            elevatorConnector.moveOutOf(location);
    }
    vocabLikelihood = 5
;

