/* Copyright (c) 1998 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  plantug.t - underground
Function
  
Notes
  
Modified
  08/16/98 MJRoberts  - Creation
*/

/* ------------------------------------------------------------------------ */
/*
 *   Atrium 
 */
atrium: room
    sdesc = "Atrium"
    ldesc =
    {
        "The polished marble walls of this vast chamber slant slightly
        inward as they rise hundreds of feet to the glass panes of the
        ceiling.  Lights from the towers above the surface twinkle
        like Christmas tree decorations through the glass.  An elevator
        is to the west, and passages lead north and southeast. ";

        if (atrium_fireHose.isIn(self))
            "\n\tA fire hose hangs down from the ceiling, the end of the
            hose just reaching the floor. ";
            
    }
    teeterNoted = nil
    teeterwallerPostFollow =
    {
        inherited.teeterwallerPostFollow;
        if (teeterwaller.location = inCage)
        {
            "\bAfter a few minutes, Mr.\ Teeterwaller comes sliding
            down the hose at high speed, and crashes into the floor.
            He collects himself off the floor and spends a few moments
            brushing off dust. ";

            if (!self.teeterNoted)
            {
                "\b<q>What a novel idea,</q> Teeterwaller tells you.
                <q>I wondered what you were thinking, dragging that
                hose everywhere.</q> ";
                self.teeterNoted := true;
            }
        }
    }
    firstClimb = true
    se = oldCorridor
    west = atrium_elevator
    in = atrium_elevator
    north = constructionHall
    up =
    {
        if (atrium_fireHose.isIn(self))
        {
            if (self.firstClimb)
            {
                "Climbing hundreds of feet up a fire hose
                might seem like a daunting task to some people.  It's
                a good thing you've been working out.\b";
                self.firstClimb := nil;
            }
            
            if (teeterwaller.isIn(self))
                "<q>I'll follow you after you get to the top,</q>
                Teeterwaller says.\b";

            "You laboriously climb the hose.  It seems a lot further
            going up than it did coming down.\b";

            if (inCage.xcoord != 60 || inCage.ycoord != 50)
            {
                "Upon reaching the top, you're horrified to discover
                that the cage isn't positioned above the opening.  You
                don't want to risk walking over the glass, so you
                reluctantly climb back down. ";
                return nil;
            }
            else
            {
                "You reach the top and are relieved to find the cage
                parked next to the opening in the glass.  You grab a pole
                and drag yourself onto the platform.\b";
                return inCage;
            }
        }
        else
        {
            "There's no way up. ";
            return nil;
        }
    }
    teeterwallerPreFollow(rm) =
    {
        /* we've already said what happens if we're going to the cage */
        if (rm != inCage)
            inherited.teeterwallerPreFollow(rm);
    }
    enterRoom(actor) =
    {
        /* do default operations */
        inherited.enterRoom(actor);

        /* award points for arriving here the first time */
        maybeIncscore(10, 'entering the atrium', self);
    }
;

class elevatorDoorItem: doorway
    noAutoOpen = true
    isopen = nil
    noun = 'door'
    adjective = 'sliding' 'metal' 'elevator'
    sdesc = "elevator door"
    adesc = "an elevator door"
    ldesc = "The sliding door is currently <<
             self.isopen ? "open" : "closed">>. "
    doOpen(actor) =
    {
        "The door is controlled automatically by the elevator; you can't
        open it manually. ";
    }
    doClose(actor) =
    {
        "The door is controlled automatically by the elevator; you can't
        close it manually. ";
    }
    otherside = inElevator_door
;

atrium_elevator: elevatorDoorItem
    noun = 'elevator'
    sdesc = "elevator"
    adesc = "an elevator"
    location = atrium
    ldesc = "The elevator has a sliding metal door, which is <<
             self.isopen ? "open" : "closed">>, and a round button
             just to the right of the door. "
    doordest = inElevator
;

atrium_elevatorButton: elevatorButtonItem
    adjective = 'elevator'
    sdesc = "elevator button"
    adesc = "an elevator button"
    adjective = 'round'
    location = atrium
    buttonDest = atrium
    ldesc = "It's a small round button next to the door. <<self.litDesc>> "
;

atrium_fireHose: fixeditem
    noun = 'hose'
    adjective = 'fire'
    sdesc = "fire hose"
    ldesc = "The fire hose extends up to the glass ceiling high above.
             The nozzle is just touching the floor. "
    verDoClimb(actor) = { }
    doClimb(actor) = { actor.travelTo(atrium.up); }
    doSynonym('Climb') = 'Climbup'
    verDoTake(actor) =
    {
        "The fire hose will not extend any further. ";
    }
    doSynonym('Take') = 'Pull' 'Move'
;

atrium_fireHoseNozzle: fixeditem
    location = atrium_fireHose
    noun = 'nozzle'
    adjective = 'fire' 'hose'
    sdesc = "fire hose nozzle"
    ldesc = "The nozzle directs the flow of water from the hose. Mounted
             near the back of the nozzle is a lever. "
    verDoPointAt(actor, iobj) =
    {
        "The hose isn't going anywhere in its current position. ";
    }
;

atrium_fireHoseLever: fixeditem
    location = atrium_fireHose
    noun = 'lever'
    adjective = 'fire' 'hose'
    sdesc = "fire hose lever"
    ldesc = "The lever is attached to the nozzle. "
    verDoPull(actor) = { }
    doPull(actor) =
    {
        "Nothing happens.  The water flow is probably cut off by
        the way the hose is bent over the glass at the top of this
        chamber. ";
    }
;    

atrium_walls: decoration
    noun = 'wall' 'walls'
    adjective = 'polished' 'marble'
    sdesc = "marble walls"
    location = atrium
    adesc = "a marble wall"
    isThem = true
    ldesc = "The walls are tiled with huge slabs of polished marble. "
;

atrium_roof: distantItem
    noun = 'ceiling' 'roof' 'pane' 'panes'
    adjective = 'glass'
    sdesc = "glass ceiling"
    location = atrium
    ldesc = "The glass ceiling is high above. "
;

atrium_lights: distantItem
    noun = 'light' 'lights' 'tower' 'towers'
    adjective = 'tower'
    sdesc = "tower lights"
    adesc = "a tower light"
    isThem = true
    ldesc = "The lights twinkle through the glass. "
    location = atrium
;

/* ------------------------------------------------------------------------ */
/*
 *   Inside the atrium elevator 
 */
inElevator: room
    sdesc = "Elevator"
    ldesc = "This elevator has a sliding door to the east, next to which is
             a control panel, with <<
             inElevator_controlPanel.ldescSuffix>> The door is <<
             inElevator_door.isopen ? "open" : "closed">>. "
    east = inElevator_door
    out = inElevator_door

    // doordest specifies the adjacent location
    doordest = atrium

    // our eventual doordest, when we're moving
    travelDest = nil

    // number of turns remaining until we reach our destination, while moving
    travelTime = 2

    // room daemon
    notifier =
    {
        local btnlist := [inElevator_buttonUpper
                          inElevator_buttonLower
                          atrium_elevatorButton
                          outsideElevator_button];
        local cur;
        local i;
        
        /* if the doors are open, there's nothing to do */
        if (inElevator_door.isopen)
            return;

        /* if the knob is pulled out, the elevator isn't running */
        if (inElevator_knob.isPulled)
            return;

        /* 
         *   if we're already in motion, decrement the travel time
         *   remaining; if we've reached our destination, finish travel
         *   and open the doors 
         */
        if (self.doordest = nil)
        {
            /* decrement the remaining travel time */
            self.travelTime--;

            /* if we've finished travel, arrive */
            if (self.travelTime = 0)
            {
                /* arrive */
                self.doordest := self.travelDest;
                self.travelDest := nil;

                /* mention it if the player's here */
                switch(parserGetMe().location)
                {
                case atrium:
                case outsideElevator:
                    if (self.doordest = parserGetMe().location)
                        "\bThe elevator door slides open. ";
                    break;

                default:
                    "\bYou can feel the elevator come to a stop, and the
                    door slides open. ";
                    break;
                }

                /* open the door */
                inElevator_door.setIsopen(true);

                /* turn off all of the destination buttons */
                for (i := 1 ; i <= length(btnlist) ; ++i)
                    btnlist[i].isLit := nil;
            }

            /* done */
            return;
        }

        /* check the buttons to see if travel should begin */
        for (i := 1 ; i <= length(btnlist) ; ++i)
        {
            /* get this button */
            cur := btnlist[i];

            /* if it's lit, start travel */
            if (cur.isLit)
            {
                self.beginTravel(cur.buttonDest);
                break;
            }
        }
    }

    // begin travel to a particular button's destination
    beginTravel(dest) =
    {
        /* mention the change if the player is present */
        if (parserGetMe().isIn(self))
            "\bYou can feel the elevator start moving. ";

        /* set up the travel */
        self.doordest := nil;
        self.travelDest := dest;
        self.travelTime := 2;
    }

    // push a button assigned to a particular destination
    pushButton(btn, dest) =
    {
        /* if we're already at the destination, do nothing */
        if (dest = self.doordest)
        {
            "The button lights momentarily as you push it, then turns
            off as soon as you release it. ";
            return;
        }

        /* mention the change */
        "The button lights up. ";

        /* turn on the button */
        btn.isLit := true;
    }
;

inElevator_controlPanel: fixeditem
    sdesc = "control panel"
    noun = 'panel'
    adjective = 'control'
    location = inElevator
    ldesc = "The panel has <<self.ldescSuffix>> "
    ldescSuffix = "buttons labeled <font size=-1>UPPER LEVEL</font>,
                   <font size=-1>LOWER LEVEL</font>,
                   <font size=-1>DOOR OPEN</font>, and
                   <font size=-1>DOOR CLOSE</font>, plus a red knob labeled
                   <font size=-1>PUSH TO RUN/PULL TO STOP</font>.  The knob
                   is currently <<
                   inElevator_knob.isPulled ? "pulled out" : "pushed in">>. "

;

inElevator_knob: fixeditem
    sdesc = "red knob"
    noun = 'knob'
    adjective = 'red'
    location = inElevator
    ldesc = "The knob is labeled <font size=-1>PUSH TO RUN/PULL
             TO STOP</font>. It's currently <<
             self.isPulled ? "pulled out" : "pushed in">>. "
    isPulled = true
    verDoPush(actor) =
    {
        if (!self.isPulled)
            "It's already pushed all the way in. ";
    }
    doPush(actor) =
    {
        "You push the knob all the way in. ";
        self.isPulled := nil;

        /* if the elevator is running, note the continuation */
        if (inElevator.doordest = nil)
            "You feel the elevator begin moving again. ";
    }
    verDoPull(actor) =
    {
        if (self.isPulled)
            "It's already pulled all the way out. ";
    }
    doPull(actor) =
    {
        "You pull the knob all the way out. ";
        self.isPulled := true;

        /* if the elevator is running, note that we're stopping */
        if (inElevator.doordest = nil)
            "You feel the elevator lurch to a stop. ";
    }
;

class elevatorButtonItem: fixeditem
    noun = 'button'
    plural = 'buttons'
    sdesc = "<<self.buttonLabel>> button"
    ldesc =
    {
        "The button is labeled <font size=-1><<self.buttonLabel>></font>. ";
        self.litDesc;
    }
    litDesc =
    {
        if (self.isLit)
            "It's currently glowing. ";
    }
    location = inElevator
    isLit = nil
    verDoPush(actor) = { }
    doPush(actor) =
    {
        if (inElevator.doordest = self.buttonDest)
        {
            /* 
             *   we're at the button's destination; just open the door if
             *   it's not already open 
             */
            if (!inElevator_door.isopen)
            {
                "The elevator door slides open. ";
                inElevator_door.setIsopen(true);
            }
            else
                "Nothing happens. ";
        }
        else if (inElevator_knob.isPulled || self.isLit)
            "Nothing happens. ";
        else
            inElevator.pushButton(self, self.buttonDest);
    }
;

inElevator_buttonLower: elevatorButtonItem
    adjective = 'lower' 'level'
    buttonLabel = "LOWER LEVEL"
    buttonDest = atrium
;

inElevator_buttonUpper: elevatorButtonItem
    adjective = 'upper' 'level'
    buttonLabel = "UPPER LEVEL"
    buttonDest = outsideElevator
;

inElevator_buttonOpen: elevatorButtonItem
    adjective = 'open' 'door'
    buttonLabel = "DOOR OPEN"
    doPush(actor) =
    {
        /* open the doors only if the elevator isn't in motion */
        if (inElevator.doordest = nil)
            "Nothing happens. ";
        else
        {
            /* if they're not already open, mention the change */
            if (inElevator_door.isopen)
                "Click. ";
            else
                "The door slides open. ";

            /* 
             *   Open the doors.  Do this even if they're already open, so
             *   that we reset the timer each time they push the button. 
             */
            inElevator_door.setIsopen(true);
        }
    }
;

inElevator_buttonClose: elevatorButtonItem
    adjective = 'close' 'door'
    buttonLabel = "DOOR CLOSE"
    doPush(actor) =
    {
        /* if the doors are already closed, do nothing */
        if (!inElevator_door.isopen)
            "Nothing happens. ";
        else
        {
            /* close the door */
            inElevator_door.setIsopen(nil);

            /* mention it */
            "The door slides closed. ";
        }
    }
;

inElevator_door: elevatorDoorItem
    location = inElevator
    doordest = { return self.location.doordest; }
    otherside =
    {
        switch (self.doordest)
        {
        case atrium:
            return atrium_elevator;

        case outsideElevator:
            return outsideElevator_door;

        default:
            return nil;
        }
    }
    doorTimer = 3
    closeDoors =
    {
        /* if the door timer has expired, close the doors */
        if (self.doorTimer = 0)
        {
            /* close the doors */
            self.setIsopen(nil);

            /* tell the player, if they're nearby */
            switch(parserGetMe().location)
            {
            case inElevator:
                "\bThe elevator door slides closed. ";
                break;

            case atrium:
            case outsideElevator:
                if (self.doordest = parserGetMe().location)
                    "\bThe elevator door slides closed. ";
                break;
            }
        }
        else
        {
            /* decrement the timer */
            self.doorTimer--;
        }
    }
    setIsopen(setting) =
    {
        local wasopen := self.isopen;
        
        /* inherit the default operation */
        inherited.setIsopen(setting);
        
        /* 
         *   if opening the door, set a daemon to close it in a few turns;
         *   otherwise, cancel the daemon 
         */
        if (setting)
        {
            /* reset the timer */
            self.doorTimer := 3;

            /* start the daemon if necessary */
            if (!wasopen)
                notify(self, &closeDoors, 0);
        }
        else
            unnotify(self, &closeDoors);
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Hallway under construction 
 */
constructionHall: room
    sdesc = "Construction area"
    ldesc = "This wide hallway is filled with dust and debris from
             a construction project.  A foreman is looking at a
             blueprint, and several construction workers are demolishing
             parts of the hallway.  The hallway opens to the south, and
             continues north past the contruction. "
    south = atrium
    north =
    {
        if (self.checkWearingHardhat(parserGetMe()))
        {
            "The foreman waves you by.  You carefully navigate through
            the construction debris, following the hallway as it turns
            a corner and continues west.\b";
            return northHall;
        }
        else
        {
            "The foreman blocks your way. <q>I can't let anyone through
            here without a hard hat,</q> he shouts over the construction
            noise. ";
            return nil;
        }
    }
    checkWearingHardhat(actor) =
    {
        return ((whiteHardhat.isIn(actor) && whiteHardhat.isworn)
                || (yellowHardhat.isIn(actor) && yellowHardhat.isworn));
                 
    }
;

constructionHall_workers: Actor
    sdesc = "construction workers"
    adesc = "some construction workers"
    isThem = true
    noun = 'workers' 'workman' 'workmen'
    adjective = 'construction'
    location = constructionHall
    ldesc = "The workers are demolishing parts of the hallway. "
    actorAction(v, d, p, i) =
    {
        "The workers ignore you. ";
        exit;
    }
    disavow = "The workers ignore you. "
    actorDesc = ""
;

constructionHall_debris: fixeditem
    location = constructionHall
    noun = 'debris' 'dust' 'lumber' 'wood' 'tile' 'tiles' 'piece' 'pieces'
           'paint' 'chip' 'chips' 'fixtures'
    adjective = 'paint' 'electrical'
    sdesc = "construction debris"
    ldesc = "The hallway is filled with debris&mdash;pieces of lumber,
             tiles, chips of paint, electrical fixtures.  None of it
             looks to be in usable condition. "
    verDoLookin(actor) = { "You find nothing of interest. "; }
    doSynonym('Lookin') = 'Search'
;

constructionHall_foreman: Actor
    sdesc = "foreman"
    noun = 'foreman'
    actorDesc = ""
    ldesc = "The foreman is studying a blueprint. "
    actorAction(v, d, p, i) =
    {
        "The foreman can't seem to hear what you're saying. ";
        exit;
    }
    disavow = "The foreman can't seem to hear you over the
               construction noise. "
    location = constructionHall
;

constructionHall_blueprint: readable
    sdesc = "blueprint"
    noun = 'blueprint'
    location = constructionHall_foreman
    ldesc = "The foreman won't let you get a very good look at it. "
;

/* ------------------------------------------------------------------------ */
/*
 *   North end of hallway 
 */
northHall: room
    sdesc = "Hallway"
    ldesc = "This is the west end of a wide east-west hallway. A sliding
             metal door, labeled <q>Lab 7,</q> leads north, and a white
             door, labeled <q>Observation Room,</q> leads west. "
    east =
    {
        "You follow the hallway as it turns a corner and continues south,
        then carefully make your way through a huge pile of construction
        debris littering the hallway.\b";
        return constructionHall;
    }
    north = northHall_labDoor
    west = northHall_obsDoor

    haveMovedScientist = nil
    enterRoom(actor) =
    {
        /*
         *   If the scientist isn't here yet, and they've seen the scene
         *   from the air vent, set up the scientist and technician.
         */
        if (!self.haveMovedScientist && shaftDuct_ventCover.isDone)
        {
            /* move the characters into the lab */
            lab7_scientist.moveInto(lab7);
            lab7_technician.moveInto(lab7);

            /* 
             *   note that we've set things up, since we don't want to do
             *   this again later 
             */
            self.haveMovedScientist := true;
        }

        /* do the normal work */
        inherited.enterRoom(actor);
    }
;

northHall_labDoor: doorway
    noAutoOpen = true
    isopen = nil
    noun = 'door'
    adjective = 'sliding' 'metal' 'north'
    sdesc = "sliding metal door"
    location = northHall
    doordest = lab7
    otherside = lab7_door
    ldesc =
    {
        "The door is <<self.isopen ? "open" : "closed">> and
        is labeled <q>L<font size=-1>AB</font>
        7&mdash;R<font size=-1>ESTRICTED</font>.</q>
        A small slot, about the right size to accept a credit
        card, is next to the door. ";
    }
    doOpen(actor) = { "You can't manage to move it at all. "; }
    verDoMove(actor) = { "You are unable to move the door. "; }
    doSynonym('Move') = 'Push' 'Pull'

    closeDoor =
    {
        /* note the change is they're in sight of the door */
        switch(parserGetMe().location)
        {
        case northHall:
        case lab7:
            "\bThe metal door slides closed. ";
            break;
        }

        /* close the door */
        self.setIsopen(nil);
    }
    setIsopen(setting) =
    {
        /* 
         *   if we're opening the door, and it's not already open, set a
         *   timer for closing the door 
         */
        if (setting && !self.isopen)
            notify(self, &closeDoor, 2);

        /* do the normal work */
        inherited.setIsopen(setting);
    }
;

class lab7DoorSlotItem: fixeditem
    sdesc = "small slot"
    noun = 'slot'
    adjective = 'small'
    verIoPutIn(actor) = { }
    ioPutIn(actor, dobj) =
    {
        if (dobj.isCardSized)
        {
            /* 
             *   if we're outside the lab, and the lab is occupied, don't
             *   proceed 
             */
            if (!shaftDuct_ventCover.isDone || lab7_scientist.isIn(lab7))
            {
                if (teeterwaller.isIn(self.location))
                    "Teeterwaller stops you. <q>Wait a moment,</q> he says.
                    <q>I think I hear voices in there.  We'd better wait
                    until they're gone.</q> ";
                else
                    "As you're about to slip <<dobj.thedesc>> into the
                    slot, you hear a voice from the other side of the door.
                    You decide not to try opening it until whoever's there
                    has left. ";
                
                /* do not continue */
                return;
            }

            /* 
             *   if we're in the lab, and the guards have shown up, don't
             *   let them out 
             */
            if (observationRoom_guards.isIn(observationRoom))
            {
                "Given that a troop of guards is outside the door
                preparing to capture you, you'd rather find an alternative
                way out of here. ";
                return;
            }

            /* note that it fits */
            "You slip <<dobj.thedesc>> into the slot; it's a good
            fit";

            /* check to see what happens */
            switch(dobj)
            {
            case labIdCard:
                /* the lab ID opens the door */
                if (northHall_labDoor.isopen)
                    ". Nothing happens, so you pull it back out. ";
                else
                {
                    ".  A soft beep emanates from the slot, and the metal
                    door slides open. ";
                    northHall_labDoor.setIsopen(true);
                }
                break;

            default:
                /* other cards have no effect */
                ", but nothing seems to happen, so you pull it back out. ";
                break;
            }
        }
        else
            "That's not a good fit. ";
    }
;

northHall_labDoorSlot: lab7DoorSlotItem
    location = northHall
;

northHall_obsDoor: doorway
    isopen = nil
    noun = 'door'
    adjective = 'white' 'west'
    sdesc = "white door"
    location = northHall
    doordest = observationRoom
    otherside = observationRoom_door
;

/* ------------------------------------------------------------------------ */
/*
 *   Observation room 
 */

observationRoom: room
    sdesc = "Observation room"
    ldesc =
    {
        "The lighting in this room is muted, presumably to reduce
        glare on the window that covers the north wall. ";

        observationRoom_window.thrudesc;

        "A white door leads east. ";
    }
    east = observationRoom_door

    notifyState = 0
    notifier =
    {
        /* 
         *   Proceed only when the player is present - we want to make
         *   sure they see the whole scene.  Also ensure that the
         *   scientist is present; earlier scenes must be completed before
         *   the scientist is ready for this scene.  
         */
        if (!parserGetMe().isIn(self) || !lab7_scientist.isIn(lab7))
            return;

        switch(self.notifyState)
        {
        case 0:
            "\bThrough the glass, you see the scientist walk to the
            silver crate.  She does something
            you can't see to the side of the crate, and the top slowly
            swings open.  The woman reaches inside and removes a black
            disk, about the size of a half-dollar coin.  She holds it up
            and looks at it, then closes the crate's lid. ";

            lab7_disk.moveInto(lab7_scientist);
            break;

        case 1:
            "\bThrough the glass, you see the scientist walk to the
            workbench near the center
            of the room and carefully place the black disk on a stand
            on the workbench.  The technician pulls a cart carrying an
            electronic device over to the workbench, and attaches a pair
            of electrodes from the device to the stand, on opposite sides
            of the disk. ";

            lab7_disk.moveInto(lab7_stand);
            lab7_cart.moveInto(lab7);
            lab7_electrodes.moveInto(lab7);
            break;

        case 2:
            "\bA woman's voice issues from a hidden speaker; you realize
            it's the scientist in the lab, talking to the technician.
            <q>Okay, we'll
            start at low voltages and work our way up.  Ready?</q>
            \bThe technician carries a hand-held meter to the lab bench,
            points it at the stand, and nods to the scientist.
            \b<q>Here we go, then.  One percent.</q>  She makes some
            adjustments to the electronic device.  You can't be sure,
            but it looks like the disk glows faintly for a moment.
            \b<q>Negative point zero zero three gee,</q> the technician
            says. ";

            lab7_device_dial.setting := 1;
            lab7_meter.moveInto(lab7_technician);
            break;

        case 3:
            "\bThe scientist's voice is audible through a hidden speaker:
            <q>Two percent.</q>  She adjusts the electronic device, and
            you again see a faint, momentary glow from the disk.
            \b<q>Negative point zero one gee,</q> the technician says. ";

            lab7_device_dial.setting := 2;
            break;

        case 4:
            "\b<q>Five percent,</q> the scientist says.  She adjusts the
            electronic device, and you see a clear blue glow from the disk;
            the glow fades after a few moments.
            \b<q>Negative point zero eight gee,</q> the technician says. ";

            lab7_device_dial.setting := 5;
            break;

        case 5:
            "\bYou hear a warbling sound through the hidden speakers.
            The scientist reaches into the pocket of her lab coat and
            takes out a cellular phone.  <q>Yes?</q>  She listens to
            the phone. <q>Affirmative.  We've just started, but the
            anti-gravity effect is confirmed.</q>  She listens some
            more.  <q>I understand.  We're on our way.</q>
            \bShe turns to the technician.  <q>They're ready to start
            the omicron beam again and they need us to monitor the
            scattering.  Leave everything set up here and we'll resume
            this when we're done with the beam test.</q>
            \bThe two walk briskly to the door and leave the lab. ";

            /* remove them from the lab */
            lab7_scientist.moveInto(nil);
            lab7_technician.moveInto(nil);
            break;
        }

        /* advance my state */
        self.notifyState++;
    }
    enterRoom(actor) =
    {
        /* do the normal work */
        inherited.enterRoom(actor);

        /* start the room daemon */
        notify(self, &notifier, 0);
    }
    leaveRoom(actor) =
    {
        /* do the normal work */
        inherited.leaveRoom(actor);

        /* stop the room daemon */
        unnotify(self, &notifier);
    }

    /* compute the list of objects visible, including the lab contents */
    getVisibleList(actor) =
    {
        /* 
         *   compute the normal list, and add the list of objects visible
         *   in the lab 
         */
        return (inherited.getVisibleList(actor) 
                + visibleList(lab7, actor));
    }

    /* items in the observation room are visible from the lab */
    isVisible(vantage) =
    {
        /* items in the observation room are visible from the lab */
        if (vantage = lab7)
            return true;

        /* otherwise, inherit the default */
        return inherited.isVisible(vantage);
    }

    /* 
     *   provide an alternative cantReach for items that the player
     *   attempts to manipulate from the lab 
     */
    cantReach(actor) =
    {
        if (actor.isIn(lab7))
            "That's on the other side of the window. ";
        else
            inherited.cantReach(actor);
    }
;

observationRoom_window: fixeditem, seethruItem
    noun = 'window' 'glass'
    adjective = 'north'
    location = observationRoom
    sdesc = "north window"
    doBreak(actor) = { "That would be too dangerous. "; }
    verDoKnock(actor) = { "You'd rather not draw attention to yourself. "; }
    ldesc =
    {
        "The window occupies most of the north wall. ";
        self.thrudesc;
    }
    thrudesc =
    {
        "On the other side of the window is a huge chamber, a hundred
        feet high and twice as wide; pieces of electronic equipment
        are scattered throughout the area, and a workbench is in the
        center of the room. ";

        if (lab7_crate.isIn(lab7))
            "Near the workbench is a silver crate the
            size of a refrigerator lying on its back. ";

        if (lab7_people.isIn(lab7))
            "Several workers in isolation suits and gas masks are
            visible through the glass. ";

        if (lab7_scientist.isIn(lab7))
            "A woman, whom you take to be a scientist because of her
            white lab coat, and a technician, wearing a blue jump-suit,
            are working with the equipment. ";
    }

    isVisible(vantage) =
    {
        /* I'm not visible from the lab (to avoid ambiguity) */
        if (vantage = lab7)
            return nil;

        /* otherwise, use the default handling */
        return inherited.isVisible(vantage);
    }
;

observationRoom_door: doorway
    isopen = nil
    noun = 'door'
    adjective = 'white' 'west'
    sdesc = "white door"
    location = observationRoom
    doordest = northHall
    otherside = northHall_obsDoor
;

/*
 *   Basic guard class - we'll instantiate one of these in each room with
 *   guards in an adjacent room.  Each instance needs an overriding ldesc
 *   to describe where the guards are.  
 */
class guardsItem: fixeditem
    noun = 'guard' 'guards' 'jump-suits' 'jumpsuits' 'helmet'
    adjective = 'black'
    sdesc = "guards"
    adesc = "several guards"
    isThem = true
;

/*
 *   guards in the observation room - these become visible from the lab
 *   when the end-game begins 
 */
observationRoom_guards: guardsItem
    ldesc = "Several guards, dressed in black jump-suits and black
             helmets, are watching you from the observation room. "
;

/* ------------------------------------------------------------------------ */
/*
 *   Guard daemon - we start this daemon when the guards show up.  We'll
 *   just let the player's current location handle it.  
 */
guardDaemon: function(param)
{
    /* let the player's location handle it */
    parserGetMe().location.roomGuardDaemon;
}

/* ------------------------------------------------------------------------ */
/*
 *   Lab 7 
 */
lab7: room
    sdesc = "Laboratory"
    ldesc = "This is a cavernous room, a hundred feet tall and twice as
             wide.  Pieces of electronic equipment are scattered throughout
             the chamber.  In the center of the room is a workbench, on
             which is a plastic stand; a pair of electrodes are connected
             to the stand, and the electrodes are connected with wires to
             an electronic device on a cart next to the workbench.  Near
             the workbench is a silver crate, the size of a
             refrigerator sitting on its back.
             \n\tAn air vent is visible in the center of the ceiling.
             A glass window in the south wall looks out into an observation
             room.
             \n\tA sliding metal door leads south. "
    south = lab7_door

    enterRoom(actor) =
    {
        /* do the normal work */
        inherited.enterRoom(actor);

        /* award some points */
        maybeIncscore(10, 'entering the lab', self);
    }

    /* compute the list of objects visible, including the lab contents */
    getVisibleList(actor) =
    {
        /* 
         *   compute the normal list, and add the list of objects visible
         *   in the lab 
         */
        return (inherited.getVisibleList(actor) 
                + visibleList(observationRoom, actor));
    }

    /* 
     *   items in the lab are visible from the observation room and from
     *   the air vent, when the vent cover is open 
     */
    isVisible(vantage) =
    {
        /* items in the lab are visible from the observation room */
        if (vantage = observationRoom)
            return true;

        /* items in the lab are also visible from the open air vent */
        if (vantage = shaftDuct && shaftDuct_ventCover.isopen)
            return true;

        /* otherwise, inherit the default behavior */
        return inherited.isVisible(vantage);
    }

    /* 
     *   provide an appropriate cantReach message for items that the
     *   player attempts to manipulate from the observation room or from
     *   the air vent 
     */
    cantReach(actor) =
    {
        if (actor.isIn(observationRoom))
            "That's on the other side of the window. ";
        else if (actor.isIn(shaftDuct) && shaftDuct_ventCover.isopen)
            "That's a hundred feet below. ";
        else
            inherited.cantReach(actor);
    }

    teeterwallerPreFollow(rm) =
    {
        /*
         *   If we're traveling to the air duct, we're already providing
         *   plenty of explanatory transition text, so don't bother with
         *   the normal "teeterwaller comes with you" message.  Keep the
         *   normal message in other cases, though. 
         */
        if (rm != shaftDuct)
            inherited.teeterwallerPreFollow(rm);
    }

    roomGuardDaemon =
    {
        local msg :=
        [
            'The guards in the observation room bang on the glass. '
            'Someone pounds on the door from the other side. '
            'The guards make threatening gestures at you through
            the window. '
            'Guards run in and out of the observation room. '
        ];

        /* just display a random message about the guards */
        "\b";
        say(msg[rand(length(msg))]);
    }

    activateEndgame(blankBefore) =
    {
        /* if the guards aren't already here, start the end-game */
        if (!observationRoom_guards.isIn(observationRoom))
        {
            if (blankBefore)
                "\b";
            
            "A siren sounds from a loudspeaker somewhere above you,
            and a voice echoes through the room: <q>Intruder alert,
            laboratory seven.  Section black, commence procedure one
            one delta.</q>
            \bThrough the window in the south wall, you see several guards
            dressed in black jump-suits and black helmets rush into the
            observation room.  They see you and start motioning to one
            another.
            \bTeeterwaller takes two steps backwards.  <q>Uh, oh...</q> ";

            /* 
             *   start the guard daemon; this will keep the guards
             *   animated through the end-game 
             */
            setdaemon(guardDaemon, nil);

            /* move the guards into the observation room */
            observationRoom_guards.moveInto(observationRoom);

            /* 
             *   move the guards onto the stairs - we'll put guards on the
             *   even levels, where they're visible but not immediately
             *   accessible from the levels the player can traverse during
             *   the end-game 
             */
            towerStair2_guards.moveInto(towerStair2);
            towerStair4_guards.moveInto(towerStair4);
            towerRoof_guards.moveInto(towerRoof);

            if (!blankBefore)
                "\b";
        }
    }
;

lab7_window: fixeditem, seethruItem
    location = lab7
    sdesc = "south window"
    noun = 'window' 'glass'
    adjective = 'glass' 'south'
    ldesc = { self.thrudesc; }
    thrudesc =
    {
        "An observation room is on the other side of the glass. ";
        if (observationRoom_guards.isIn(observationRoom))
            "Several guards are watching you through the window. ";
    }
   
    isVisible(vantage) =
    {
        /* I'm not visible from the observation room (to avoid ambiguity) */
        if (vantage = observationRoom)
            return nil;

        /* otherwise, use the default handling */
        return inherited.isVisible(vantage);
    }
    verDoBreak(actor) = { "That would be too dangerous. "; }
;

lab7_equipment: fixeditem
    location = lab7
    sdesc = "equipment"
    adesc = "some equipment"
    noun = 'piece' 'pieces' 'equipment' 'machines'
    adjective = 'electronic'
    ldesc = "Numerous pieces of equipment are scattered throughout
             the area, all of the machines incomprehensible. "
;

lab7_people: fixeditem
    location = lab7
    sdesc = "workers"
    adesc = "some workers"
    isThem = true
    noun = 'workers' 'people' 'persons' 'suit' 'suits' 'mask' 'masks'
    adjective = 'isolation' 'gas'
    ldesc = "They're dressed in isolation suits. "
;

lab7_crate: fixeditem
    noun = 'crate'
    adjective = 'silver'
    sdesc = "silver crate"
    ldesc = "The crate is made of a very shiny silver metal.  It's
             about the size of a refrigerator sitting on its back. "
    verDoOpen(actor) = { }
    doOpen(actor) =
    {
        "There's no obvious way to open the crate.  You can find no
        latches, buttons, catches, or other mechanisms, and there's
        not even a visible seam anywhere. ";
    }
    verDoMove(actor) =
    {
        "It's extremely heavy; it would take several more people to
        move it. ";
    }
    doSynonym('Move') = 'Pull' 'Push'
;

lab7_doorSlot: lab7DoorSlotItem
    location = lab7
;

lab7_door: doorway
    noAutoOpen = true
    sdesc = "sliding metal door"
    noun = 'door'
    adjective = 'sliding' 'metal' 'south'
    location = lab7
    ldesc = "The door is <<self.isopen ? "open" : "closed">>.  A small
             slot is next to the door. "
    otherside = northHall_labDoor
    doordest = northHall
    doOpen(actor) = { "You can't manage to move it at all. "; }
    verDoMove(actor) = { "You are unable to move the door. "; }
    doSynonym('Move') = 'Push' 'Pull'
;

lab7_ventCover: distantItem
    noun = 'cover' 'vent'
    adjective = 'vent'
    location = lab7
    sdesc = "vent cover"
    ldesc =
    {
        "The vent cover is ";
        if (self.isopen)
            "open, creating an opening in the ceiling. ";
        else
            "closed. ";
    }

    isVisible(vantage) =
    {
        /* I'm not visible from the air shaft (to avoid ambiguity) */
        if (vantage = shaftDuct)
            return nil;

        /* otherwise, use the default handling */
        return inherited.isVisible(vantage);
    }
;

lab7_scientist: fixeditem
    noun = 'scientist' 'woman' 'coat'
    adjective = 'white' 'lab' 'knee-length'
    sdesc = "scientist"
    ldesc = "She's wearing a knee-length white lab coat, which is
             how you know she's a scientist. "
;

lab7_technician: fixeditem
    noun = 'technician' 'jumpsuit' 'jump-suit' 'suit' 'man'
    adjective = 'blue' 'jump'
    sdesc = "technician"
    ldesc = "He's wearing a blue jump-suit. "
;

lab7_disk: item
    noun = 'disk' 'disc'
    adjective = 'black'
    sdesc = "black disk"
    ldesc = "It's about the size of a half-dollar coin, and is made of
             a dull black material. "
;

lab7_workbench: fixeditem, surface
    noun = 'workbench' 'bench' 'rubber'
    adjective = 'work'
    location = lab7
    sdesc = "workbench"
    ldesc =
    {
        "It's the size of a large desk, but taller, and the top is
        coated with rubber.  A plastic stand is in the center of the
        top surface";

        if (lab7_electrodes.isIn(self.location))
            ", and a pair of electrodes are attached at either side
            of the stand's top platform. ";

        /* show contents only if there's something on me */
        self.optionalListContents;
    }
;

lab7_fixeditem: fixeditem
    verDoTake(actor) = { "You'd rather not disturb that. "; }
    doSynonym('Take') = 'Move' 'Push' 'Pull'
;    

lab7_stand: lab7_fixeditem, surface
    noun = 'stand' 'platform' 'leg' 'legs'
    adjective = 'plastic'
    sdesc = "plastic stand"
    location = lab7_workbench
    ldesc =
    {
        "It's a round platform, a few inches in diameter, on
        top of four legs about nine inches high. ";
        if (lab7_electrodes.isIn(lab7))
            "A pair of electrodes are attached at either side of the
            platform. ";

        /* show the contents if necessary */
        self.optionalListContents;
    }
    ioPutOn(actor, dobj) =
    {
        if (dobj = lab7_disk)
            inherited.ioPutOn(actor, dobj);
        else
            "You'd rather not disturb the experiment; the only thing
            that the scientist put on the stand was the black disk. ";
    }
    verIoDetachFrom(actor) =
    {
        "You'd rather not disturb that. ";
    }
    Grab(obj) =
    {
        /* do any inherited work */
        inherited.Grab(obj);
        
        /* 
         *   if they're taking the disk and the guards aren't present yet,
         *   activate the guards - we don't want to allow leaving the lab
         *   with the disk except by entering the end-game 
         */
        lab7.activateEndgame(nil);
    }
;

lab7_cart: lab7_fixeditem
    noun = 'cart' 
    sdesc = "cart"
    ldesc = "On the cart is an electronic device of some kind. "
;

lab7_device: lab7_fixeditem
    noun = 'device'
    adjective = 'electronic'
    location = lab7_cart
    sdesc = "electronic device"
    adesc = "an electronic device"
    ldesc = "The device is about the same size and shape as a video
             recorder, but the case is unpainted metal and the only
             obvious controls are a dial and a red button. "
;

lab7_device_dial: fixeditem, dialItem
    noun = 'dial'
    sdesc = "dial"
    location = lab7_device
    minsetting = 1
    setting = 1
    maxsetting = 100
    doTurnTo(actor, iobj) =
    {
        /* do the normal work */
        inherited.doTurnTo(actor, iobj);
        
        /* the first time they turn the dial, bring in the guards */
        lab7.activateEndgame(true);
    }
;

lab7_device_button: fixeditem
    noun = 'button'
    adjective = 'red'
    sdesc = "red button"
    location = lab7_device
    ldesc = "It's the kind you push to activate something. "
    verDoPush(actor) = { }
    doPush(actor) =
    {
        /* if the disk isn't on the platform, we do nothing */
        if (!lab7_disk.isIn(lab7_stand))
        {
            "A spark jumps between the electrodes for a fraction of a
            second, then disappears. ";
            return;
        }

        /* 
         *   depending on how high the meter is set, we have different
         *   effects 
         */
        if (lab7_device_dial.setting < 10)
            "A spark jumps across the electrodes.  The disk glows faintly
            for a moment, then returns to normal. ";
        else if (lab7_device_dial.setting < 50)
            "A spark flashes between the electrodes.  The disk glows
            bright blue, and for a moment you feel very dizzy and
            light-headed.  The disk's glow fades after a moment, and
            the dizziness passes. ";
        else if (lab7_device_dial.setting < 95)
            "A searing white arc of electricity jumps across the electrodes,
            and the black disk starts glowing with an intense blue light.
            You suddenly feel disoriented, and the room seems to spin around.
            You lean on the workbench to keep your balance.  After several
            seconds, the disorientation subsides; you notice that the disk
            has stopped glowing. ";
        else
        {
            "Darkness suddenly falls over the room as all of the lights
            go out; a second later, the electrodes blaze with a blinding
            flash of white light and make a loud pop like a balloon bursting.
            The spark fades, and the disk glows with an intense but ghostly
            blue light.
            \bYou suddenly feel like you're tumbling sideways, and it
            feels like something is pulling your feet and your head in
            opposite directions.  You reach out for something to hold,
            but find nothing in reach.
            \bJust as you think you're going to hit the ground, you realize
            that the floor is twenty feet away and getting further.  You're
            falling, but somehow you're falling toward the ceiling.  The
            room is spinning rapidly, and you see that Mr.\ Teeterwaller
            is falling alongside you, his eyes closed and his arms flailing
            wildly; beside him the brightly-glowing disk is serenly
            floating in mid-air.
            \bYou keep falling upwards for several seconds, moving directly
            toward a large air vent on the ceiling.  You draw closer and
            closer to the vent, and you brace yourself for impact.  In a
            moment the vent is only a few feet away, but now you can see
            that you're not moving very quickly after all, and you easily ";
            if (!lab7_ventCover.isopen)
            {
                "pull open the cover and ";
                lab7_ventCover.isopen := true;
                shaftDuct_ventCover.isopen := true;
            }
            "maneuver yourself into the opening.
            \bTeeterwaller crashes into the ceiling just outside
            the vent, but you manage to grab his arm and
            help him through the opening.
            \bYou find yourself in a large air duct.  The disk floats
            up through the opening; its glow gradually fades,
            and as it does, you, Teeterwaller, and the disk settle to
            the floor of the duct.  Gravity returns to normal.\b";

            lab7_disk.moveInto(shaftDuct);
            parserGetMe().travelTo(shaftDuct);

            incscore(10, 'escaping the lab with the disk');
        }
    }
;

lab7_electrodes: lab7_fixeditem
    noun = 'electrodes' 'pair' 'cable' 'cables' 'wire' 'wires'
    sdesc = "electrodes"
    adesc = "an electrode"
    isThem = true
    ldesc = "The electrodes are connected to the device on the cart,
             and the ends are attached to opposite sides of the stand. "
    verDoDetach(actor) = { "You'd rather not disturb that. "; }
    verDoDetachFrom(actor, iobj) = { self.verDoDetach(actor); }
;

lab7_meter: item
    noun = 'meter'
    adjective = 'hand-held'
    sdesc = "hand-held meter"
    ldesc = "You can't see any detail from here. "
;

/*
 *   guards in the lab - they move in as soon as we levitate away to the
 *   air shaft 
 */
lab7_guards: guardsItem
    ldesc = "Dozens of guards are swarming around the lab below, looking
             up at you. "
;


/* ------------------------------------------------------------------------ */
/*
 *   Old corridor southeast of atrium
 */
oldCorridor: room, noFloorRoom
    sdesc = "Old corridor"
    ldesc = "The proportions of this passage, the workmanship
             of the marble floor, the wire-caged light fixtures all
             give this corridor an antique appearance.  A large room
             lies to the northwest, and silvery doors lie to the east,
             west, and south; the door to the south is labeled
             <q><font size=-1>MAINTENANCE</font>,</q> and the door
             to the east is marked
             <q><font size=-1>RECORDS</font>.</q> "
    nw = atrium
    east = oldCorridor_eastDoor
    south = oldCorridor_southDoor
    west = oldCorridor_westDoor
;

oldCorridor_floor: decoration
    noun = 'floor' 'patterns'
    adjective = 'marble' 'geometrical'
    sdesc = "marble floor"
    location = oldCorridor
    ldesc = "The floor is inlaid with an elaborate geometrical pattern,
             purely for decoration. "
;

oldCorridor_lights: decoration
    noun = 'light' 'lights' 'fixtures' 'cage' 'cages' 'bulb' 'bulbs'
    adjective = 'light' 'wire'
    sdesc = "light fixture"
    location = oldCorridor
    ldesc = "There are a few fixtures set into the ceiling. They're
             just bare low-wattage bulbs covered by wire cages. "
;

class oldCorridor_door_class: doorway
    location = oldCorridor
    sdesc = "silvery door"
    noun = 'door' 'grooves' 
    adjective = 'silvery' 'vertical' 'metal'
    plural = 'doors'
    isopen = nil
    ldesc = "The door is made of a strange silvery metal, whiter and
             shinier than aluminum but not mirror-smooth, and patterned
             with parallel vertical grooves spaced a quarter-inch apart<<
             self.doorLabel != nil
             ? '; it\'s labeled <q><font size=-1>' + self.doorLabel
               + '</font>.</q>'
             : '.'
             >> It's currently <<self.isopen ? "open" : "closed">>. "
    doordest = oldCorridor
    doorLabel = nil
;

oldCorridor_eastDoor: oldCorridor_door_class
    sdesc = "east door"
    adjective = 'east'
    doordest = recordsRoom
    otherside = recordsRoom_door
    doorLabel = 'RECORDS'
    noAutoOpen = { return recordsRoom.isOccupied; }
    doOpen(actor) =
    {
        if (recordsRoom.isOccupied)
            "You start to open the door, but then you hear a voice
            from the other side.  Since you wouldn't want to have to
            explain why you're here, you decide to keep a low profile
            for now. ";
        else
            inherited.doOpen(actor);
    }
    doKnock(actor) =
    {
        if (recordsRoom.isOccupied)
            "Better not; you're afraid you might get an answer. ";
        else
            inherited.doOpen(actor);
    }
;


/* a dummy key to keep doors locked */
nonExistentKey: object
;

oldCorridor_westDoor: oldCorridor_door_class
    sdesc = "west door"
    adjective = 'west'
    doordest = storageRoom
    otherside = storageRoom_door
;

oldCorridor_southDoor: oldCorridor_door_class
    sdesc = "south door"
    adjective = 'south'
    doordest = maintRoom
    otherside = maintRoom_door
    doorLabel = 'MAINTENANCE'
;

/* ------------------------------------------------------------------------ */
/*
 *   Storage room 
 */
storageRoom: room
    sdesc = "Storage room"
    ldesc = "This room is almost filled with stacks of wooden crates
             from floor to ceiling.  A door leads east; above the door
             is a small air vent which is blowing a steady stream of <<
             maintRoom_vent.isSucking ? "musty-smelling" : "fresh"
             >> air. "
    east = storageRoom_door
    out = storageRoom_door
;

class oldCorridorVentItem: fixeditem
    noun = 'vent'
    adjective = 'air'
    sdesc = "air vent"
    adesc = "an air vent"
    ldesc = "It's a small vent, only a few inches high; it's blowing <<
             maintRoom_vent.isSucking ? "musty-smelling" : "fresh">> air. "
    verDoOpen(actor) = { "The vent is firmly closed. "; }
    doSynonym('Open') = 'Lookin' 'Search'
;

storageRoom_vent: oldCorridorVentItem
    location = storageRoom
;

storageRoom_door: oldCorridor_door_class
    location = storageRoom
    otherside = oldCorridor_westDoor
;

storageRoom_crates: fixeditem
    sdesc = "crates"
    adesc = "some crates"
    isThem = true
    location = storageRoom
    noun = 'crate' 'crates' 'box' 'boxes'
    adjective = 'wood' 'wooden'
    ldesc = "The crates are neatly stacked from floor to ceiling.
             The crates all look alike; you see no markings that
             distinguish one from another. "
    verDoTake(actor) =
    {
        "The crates are packed in so tightly that you couldn't easily
        move one without disturbing all of them. ";
    }
    doSynonym('Take') = 'Move' 'Pull' 'Push' 'Open'
    verDoLookin(actor) = { "The crates are all closed. "; }
    doSynonym('Lookin') = 'Search' 
;

/* ------------------------------------------------------------------------ */
/*
 *   Records Room 
 */

recordsRoom: room
    sdesc = "Records room"
    ldesc = "This long, narrow chamber might better be called a corridor
             than a room.  The north and south walls are lined with
             file drawers from floor to ceiling; there must be thousands
             of drawers.  A door lies to the west; a small vent is
             over the door. "
    west = recordsRoom_door
    out = recordsRoom_door
    isOccupied = true
    enterRoom(actor) =
    {
        /* award some points on the first entry */
        if (!self.isseen)
            incscore(10, 'entering the records room');

        /* do the normal stuff */
        inherited.enterRoom(actor);
    }
    teeterwallerPreFollow(rm) =
    {
        if (briefcase.isIn(self))
        {
            if (recordsRoom_drawers.isDone)
                "<q>I think we've seen enough here,</q> Teeterwaller
                says as he follows you out of the room.\b";
            else
                "<q>We must come back here,</q> Teeterwaller says,
                stuffing a few last files into his briefcase as he
                follows you out of the room.  <q>There's much more
                I want to see here.</q>\b";
        }
        else
            inherited.teeterwallerPreFollow;
    }
    teeterwallerPostFollow =
    {
        /* do the normal stuff */
        inherited.teeterwallerPostFollow;

        /* snatch the briefcase */
        if (briefcase.isIn(self) && !briefcase.isIn(teeterwaller))
        {
            briefcase.moveInto(teeterwaller);
            "\bTeeterwaller has a look of awe on his face. <q>This
            is amazing,</q> he says. <q>I never thought they'd just
            file everything away like this.</q>  He picks up the
            briefcase. <q>This is perfect.</q>
            \bHe starts opening drawers and going through files. ";
        }
    }
    teeterIdleCnt = 0
    teeterIdleAction =
    {
        if (briefcase.isIn(teeterwaller))
        {
            "\bTeeterwaller looks through a drawer, pulling a couple of
            files and looking them over";

            switch(self.teeterIdleCnt)
            {
            case 0:
                ". <q>Very interesting,</q> he mutters, putting the files
                in his briefcase. ";
                break;
                
            case 1:
                ". <q>Amazing,</q> he mutters, putting the files in his
                briefcase. ";
                break;

            case 2:
                ". <q>This is astonishing,</q> he says. <q>I've been
                looking for this type of information for <i>years</i>,
                but all I've ever been able to find is the vaguest of
                hints, just bits and pieces.  This is <i>everything</i>
                I've been looking for, all in one place.</q> ";
                break;

            case 3:
                ". <q>This is it!  Look!</q> he says excitedly, showing
                you the file.  It's marked <q>Omegatron Corp.\ (Project
                <q>Tyche</q>)</q> and is dated in 1970.  He flips through
                the file for you; it contains electronic diagrams and
                technical data, none of which makes sense to you but
                clearly is important to Teeterwaller. ";

                if (tycheSpec.isAskedAbout)
                    "\b<q>I don't know if I ever told you about Project
                    Tyche,</q> he says. <q>I worked on it around twenty-five
                    years ago.  We were building an embedded controller
                    system for some industrial equipment.  Part of the
                    project involved these logic chips that seemed almost
                    magical at the time.  They were far ahead of anything
                    we'd seen.  I started wondering about them recently,
                    so I asked our corporate archives people for a copy
                    of the specification document for the project.  I got
                    the spec, but the parts about the logic chips were
                    missing. ";
                else
                    "\b<q>Remember the missing sections from the Project
                    Tyche specification document?  The parts with the
                    information I was looking for about the logic chips? ";

                "Well, <i>this</i> is the missing section.  Plus, it has
                information about how these people,</q> he looks
                around the room at the file drawers, <q>whoever it is who's
                keeping these records, planted this information with
                our project.</q> ";

                tycheSpec.isExtraInfoKnown := true;
                break;

            default:
                ", stuffing them into his briefcase. ";
                break;
            }
            self.teeterIdleCnt++;
        }
    }
;

recordsRoom_vent: oldCorridorVentItem
    location = recordsRoom
;

recordsRoom_door: oldCorridor_door_class
    location = recordsRoom
    otherside = oldCorridor_eastDoor
;

briefcase: openable
    isopen = nil
    location = recordsRoom
    sdesc = "briefcase"
    noun = 'briefcase'
    dobjGen(a, v, i, p) =
    {
        if (v != inspectVerb && v != askVerb && v != tellVerb
            && self.isIn(teeterwaller))
        {
            "Mr.\ Teeterwaller won't let you. ";
            exit;
        }
    }
    iobjGen(a, v, d, p) = { self.dobjGen(a, v, d, p); }
;

labIdCard: readable
    isCardSized = true
    location = recordsRoom
    noun = 'card' 'id' 'badge'
    adjective = 'id' 'red'
    sdesc = "red ID card"
    ldesc = "The ID is about the same size as an ordinary credit card,
             but twice as thick.  It's made of red plastic, and reads:
             \b\t<font size=-1><b>RESTRICTED</b>
             \n\t10746331
             </font> "
;

recordsRoom_drawers: fixeditem
    noun = 'drawers' 'file' 'files'
    adjective = 'file'
    location = recordsRoom
    sdesc = "file drawers"
    isThem = true
    adesc = "a file drawer"
    ldesc = "The file drawers are marked with some bizarre numbering
             system that makes no sense at all.  You could probably just
             pick a drawer at random and look in it. "
    nextFile = 1
    isDone = nil
    verDoOpen(actor) = { }
    doOpen(actor) =
    {
        local records :=
        [
         ['Microprocessor Transfer Study'
          '<tt>The rapid pace of transfers over the past five years has
           clearly strained several absorption factors and is not producing
           the benefits that accrued in prior stages.  The original recipients
           bear partial responsibility through their dubious practices, but
           fundamentally we have simply transferred technology too rapidly.
           The schedule adjustments we have initiated should stabilize the
           market within two years and restore full profitability.</tt>']
         ['Embargo Schedule - Medical'
          '<tt>Agreements with Mercer &amp; Co., Pfalk Laboratories,
          Seth and Gurney, Milnes renewed through FY 2015.  All below listed
          substances are embargoed to all parties through contract expiration,
          pending timely receipt of annual suppression fees as negotiated.
          \bNN 373 AS1 (wide spectrum influenza antiviral)
          \nA901/902 PF (macular regenerative)
          \nX-6080114 (bacteriophage - BUT NOTE use as car wax approved
                       for 1999, see Schedule - Automotive,
                       Cleaning &amp; Cosmetic)
          \b</tt>...and a long list of similar entries follows.']
         ['MB-7'
          'This file contains numerous technical diagrams of a jet
           airliner, including everything from the passenger compartment
           floor plan to the parts list for the fan blades in the engine.']
         ['HYB-303592'
          'This file has pages and pages of random letters and numbers,
           with every line filled out to exactly the same length (around
           seventy characters) and every page with the same number of
           lines (around fifty).  The only pattern you can see is that
           every line starts with a capital <q>M.</q>']
         ['Project Mercury'
          '<tt>Suspicions of a military source for switching power supply
           design for secondary power unit have been dispelled.  Confirmed
           reports from three M9 operatives indicate zero progress on
           reverse engineering for entire power supply assembly.
           M9-Sigma and M9-Xi independently report P/S assembly is
           considered to be ejection seat apparatus (i.e., M8 disinformation
           evidently successful).</tt>']
         ['Origin Survey'
          '<tt>Radio surveys recently completed (1959.12.10 to 1960.01.09)
           focused on the two primary candidate stars, based on the latest
           cartography results.  Beta Eridani was completely negative in
           H-alpha and pi-band.  Results on Epsilon Aquilae are inconclusive.
           The project team is awaiting practical application of high-speed
           signal processor design currently in analysis by electronics
           division.</tt>']
         ['Facilities Move Requests'
          '<tt>To: All Managers
           \nFrom: Facilities
           \nSubject: Move Requests
           \nDate: 1991.05.14
           \bThe new Conners City facility will open in approximately two
           months, and Facilities has already received over seven thousand
           discrete move requests.  Due to the volume of requests, we have
           established a policy of processing moves through September only
           for groups currently in the White Bluffs and Shawlow locations.
           PLEASE NOTE that Security approval MUST be included on all move
           requests for Black, Orange, and Red Sectors, and for ALL
           personnel in White Bluffs.</tt>']
         ['New 5-inch format'
          '<tt>The storage systems group is seeking approval for the 180k
           format to replace the original 150k format.  Our analysis is
           that the proliferation risk is minimal, since the higher density
           involves only a few small algorithmic changes and does not require
           the release of any improved manufacturing techniques.  Apparently
           the third-party software developer has been unable to improve
           their own storage efficiency and needs the extra capacity; we feel
           that the risk would be greater if we helped the software vendor,
           in light of Psychology\'s projections of 9.2 imitation rank
           and 9.7 antisocial rank for this vendor\'s leadership.</tt>']
        ];

        if (self.nextFile <= length(records))
        {
            local cur;

            cur := records[self.nextFile++];
            "You pick a drawer at random and open it.  The drawer is loaded
            with files; you choose one with a typewritten label
            reading <q><<cur[1]>></q> and look through it.
            \b<<cur[2]>>
            \bTeeterwaller snatches the file. <q>This one looks <<
            ['interesting' 'extremely interesting' 'important'
             'curious' 'intriguing' 'fascinating'][rand(6)]>>,</q>
             he says as he stuffs the file into his briefcase. ";
        }
        else
        {
            "You pick a drawer at random and open it, and you browse through
            some of the files, but they're all gibberish. ";

            /* note that we've seen all of the messages */
            self.isDone := true;
        }
    }
    doSynonym('Open') = 'Lookin' 'Search'
;

/* ------------------------------------------------------------------------ */
/*
 *   Maintenance room 
 */
maintRoom: room
    sdesc = "Maintenance room"
    ldesc = "This is a small, musty-smelling room whose only exit is the
             door to the north.  A cabinet occupies one wall.  Opposite the
             cabinet is a large air vent, below which is a lever. "
    north = maintRoom_door
    out = maintRoom_door
    addGreenGas =
    {
        if (maintRoom_vent.isSucking)
        {
            "The green gas is quickly drawn into the vent";
            if (recordsRoom.isOccupied)
            {
                "; a few moments later, from the hall, you hear loud coughing
                and yelling. ";
                if (maintRoom_door.isopen)
                    "Down the hall, one of the doors bursts open, and two
                    men and a woman run out into the hall, holding their
                    hands over their mouths.  They run off down the hall. ";
                else
                    "The voices recede into the distance. ";

                recordsRoom.isOccupied := nil;
                recordsRoom_door.setIsopen(true);
            }
            else
                ". ";
        }
        else
            "The gas gradually dissipates. ";
    }
;

maintRoom_lever: fixeditem
    location = maintRoom
    sdesc = "lever"
    noun = 'lever'
    ldesc = "The lever is mounted below the vent, and moves
             left and right.  The left end is labeled <q>Outside,</q>
             and the right end is labeled <q>Recycled.</q>
             The lever is currently at the <<
             maintRoom_vent.isSucking ? "right" : "left" >> limit
             of its travel. "
    verDoMove(actor) = { }
    doMove(actor) =
    {
        maintRoom_vent.isSucking := !maintRoom_vent.isSucking;
        "You pull the lever all the way <<
        maintRoom_vent.isSucking ? "right" : "left" >>. ";

        if (maintRoom_vent.isSucking)
            "The vent starts drawing in air. ";
        else
            "The vent stops drawing in air. ";
    }
    doSynonym('Move') = 'Pull' 'Push'
;

maintRoom_vent: fixeditem
    location = maintRoom
    sdesc = "air vent"
    adesc = "an air vent"
    noun = 'vent'
    adjective = 'large' 'air'
    isSucking = nil
    ldesc =
    {
        "The vent is mounted near the top of the wall; it's about
        three feet square. ";
        if (self.isSucking)
            "The vent is generating considerable suction, steadily
            drawing in air. ";
    }
    verDoOpen(actor) = { "It's firmly fixed in place. "; }
    verDoLookin(actor) =
    {
        "There does not appear to be anything in the air vent. ";
    }
    doSynonym('Lookin') = 'Search' 'Lookthru'
    verDoEnter(actor) =
    {
        "Unfortunately, you won't be able to do any air shaft
        spelunking through this vent. ";
    }
    doSynonym('Enter') = 'Board' 'Gothrough'
;

maintRoom_cabinet: fixeditem, openable
    isopen = nil
    location = maintRoom
    sdesc = "cabinet"
    noun = 'cabinet'
    ldesc =
    {
        "The cabinet occupies the entire wall";
        if (self.isopen)
        {
            "; it's open, ";
            if (itemcnt(self.contents) != 0)
                "and contains <<listcont(self)>>. ";
            else
                "but is empty. ";
        }
        else
            "; it's closed. ";
    }
;

maintRoom_door: oldCorridor_door_class
    location = maintRoom
    otherside = oldCorridor_southDoor
;

/* ------------------------------------------------------------------------ */
/*
 *   Shaft beneath the tower stairs
 */
towerShaft: endGameRoom, noFloorRoom
    sdesc = "Underground, in shaft"
    ldesc = "Concrete walls line this cylindrical shaft surrounding
             the tower.  The stairs end their descent here, about
             a story beneath the surface, but the shaft continues far
             below. A metal grating completely covers the shaft at the
             bottom of the stairs, providing a floor.
             \n\tA narrow opening in the wall of the shaft leads west. "
    up =
    {
        "Following the stairs around the tower, you cross through an opening
        in the grating and emerge into the night air.\b";
        
        return towerStair1;
    }
    down =
    {
        "The stairs go no further down. ";
        return nil;
    }
    west = shaftHall
    addGreenGas = "The gas quickly blows up out of the shaft. "
;

towerShaft_opening: fixeditem
    noun = 'opening'
    adjective = 'narrow'
    sdesc = "narrow opening"
    location = towerShaft
    ldesc = "The opening leads west. "
    verDoEnter(actor) = { }
    doEnter(actor) = { actor.travelTo(self.location.west); }
    doSynonym('Enter') = 'Gothrough' 'Board'
;

towerShaft_wall: fixeditem
    noun = 'wall'
    adjective = 'concrete' 'shaft'
    location = towerShaft
    sdesc = "concrete wall"
    ldesc = "The concrete wall curves around the tower. "
;

towerShaft_tower: fixeditem
    noun = 'tower'
    adjective = 'cylindrical'
    location = towerShaft
    sdesc = "tower"
    ldesc = "It's a white metal cylinder in the center of the shaft. "
    verDoEnter(actor) = { "You see no entrance here. "; }
    verDoClimb(actor) = { "You might want to try the stairs. "; }
    doSynonym('Climb') = 'Climbup'
;

towerShaft_stairs: fixeditem
    noun = 'stair' 'stairway' 'stairs' 'staircase'
    location = towerShaft
    sdesc = "stairway"
    ldesc = "The stairs spiral around the tower, and lead up out of the
             shaft. "
    verDoClimb(actor) = { }
    doClimb(actor) = { actor.travelTo(self.location.up); }
    doSynonym('Climb') = 'Climbup'
;

class towerGratingItem: fixeditem
    noun = 'grating' 'shaft' 'floor' 'ground'
    adjective = 'metal' 'concrete' 'floor'
    sdesc = "metal grating"
    ldesc = "The grating provides a floor at the bottom of the stairs. "
    verDoLookthru(actor) = { }
    doLookthru(actor) =
    {
        "You can only see that the shaft continues for some distance
        below. ";
    }
    doSynonym('Lookthru') = 'Lookin' 'Search'
    verDoMove(actor) = { "It's firmly fixed in place. "; }
    doSynonym('Move') = 'Push' 'Pull' 'Open' 'Take'
    ioShineOn(actor, dobj) =
    {
        if (dobj.islamp && dobj.islit)
            "The light from <<dobj.thedesc>> disappears into the gloom
            of the shaft below. ";
        else
            "That reveals nothing new. ";
    }
    ioSynonym('ShineOn') = 'ShineThru'
;

towerShaft_grating: towerGratingItem
    location = towerShaft
;

/* ------------------------------------------------------------------------ */
/*
 *   Hallway west of shaft 
 */
shaftHall: endGameRoom
    sdesc = "Narrow passage"
    ldesc = "This low, narrow tunnel was probably never intended as
             a walkway; it's more like a ventilation shaft than a
             corridor.  The tunnel becomes an aluminum duct to the
             west, and several small vents connect through the north
             and south walls.  The passage opens into a larger area
             to the east. "
    east = towerShaft
    west = shaftDuct
    addGreenGas = "The gas quickly blows toward the shaft to the east. "
;

shaftHall_vents: fixeditem
    noun = 'vent' 'vents'
    adjective = 'air' 'small'
    location = shaftHall
    sdesc = "air vents"
    adesc = "an air vent"
    isThem = true
    ldesc = "The vents are circular and vary in size from a few inches
             to a foot in diameter. "
    verIoPutIn(actor) =
    {
        "None of the vents would be a very good place to put that. ";
    }
    verDoEnter(actor) = { "None of the vents are large enough. "; }
    doSynonym('Enter') = 'Board' 'Gothrough'
    verDoLookin(actor) = { }
    doLookin(actor) = { "You see nothing in any of the vents. "; }
    doSynonym('Lookin') = 'Search'
;

shaftHall_duct: fixeditem
    noun = 'duct'
    sdesc = "aluminum duct"
    adesc = "an aluminum duct"
    ldesc = "The duct leads to the west.  It's more cramped than this
             part of the passage, but is still big enough to move through. "
    location = shaftHall
    verDoEnter(actor) = { }
    doEnter(actor) = { actor.travelTo(self.location.west); }
;

/* ------------------------------------------------------------------------ */
/*
 *   During the endgame, we need to make sure we don't lose the black
 *   disk, so we'll have Mr. Teeterwaller pick it up and carry it if the
 *   player should leave it behind in these rooms. 
 */
class endGameRoom: room
    teeterwallerPreFollow(rm) =
    {
        /* 
         *   If the disk is about to be left behind, make teeterwaller
         *   grab it; otherwise, just do the normal work. 
         */
        if (lab7_disk.isIn(self) && !lab7_disk.isIn(parserGetMe()))
        {
            "Mr.\ Teeterwaller picks up the black disk and hands it
            to you. <q>We don't want to lose track of this,</q> he says
            as he catches up to you.\b";

            lab7_disk.moveInto(parserGetMe());
        }
        else
            inherited.teeterwallerPreFollow(rm);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   ducting, further west of the shaft 
 */
shaftDuct: endGameRoom
    sdesc = "Air duct"
    ldesc =
    {
        "This aluminum air duct is just large enough to crawl through here,
        but splits into several smaller ducts to the west, none of which
        are large enough to permit further passage.  The duct widens
        to the east. ";

        if (shaftDuct_ventCover.isopen)
            "In the floor is a square opening, two feet on a side,
            whose cover has been swung out and down; a cavernous
            chamber below is visible through the opening. ";
        else
            "A vent cover, two feet on a side, is over an opening
            in the floor. ";
    }
    east = shaftHall
    west =
    {
        "The main duct splits into several smaller ducts to the west;
        none of the smaller ducts are large enough to enter. ";
        return nil;
    }
    down =
    {
        if (shaftDuct_ventCover.isopen)
        {
            "The floor is a hundred feet below. ";
            return nil;
        }
        else
        {
            "The vent cover is closed. ";
            return nil;
        }
    }
    addGreenGas = "The gas quickly blows out of the duct to the east. "

    /* compute the list of objects visible, including the lab contents */
    getVisibleList(actor) =
    {
        local ret;
        
        /* compute the normal list */
        ret := inherited.getVisibleList(actor);

        /* add the lab contents, if the vent cover is open */
        if (shaftDuct_ventCover.isopen)
            ret += visibleList(lab7, actor);

        /* return the result */
        return ret;
    }

    roomGuardDaemon =
    {
        /*
         *   If we haven't moved the guards into the lab yet, do so now 
         */
        if (!lab7_guards.isIn(lab7))
        {
            "\bThrough the opening, you see the door to the lab slide
            open, and dozens of guards rush into the lab, looking up
            at you from the floor far below. ";

            lab7_guards.moveInto(lab7);
        }
        else
        {
            local msg :=
            [
                'The guards in the lab rush around busily, several keeping
                a close watch on you. '
                'The guards in the lab make hand signals to one another. '
                'Through the opening, you see the guards in the lab watching
                you. '
            ];

            "\b";
            say(msg[rand(length(msg))]);
        }
    }
;

shaftDuct_ducts: fixeditem
    noun = 'duct' 'ducts'
    adjective = 'smaller'
    location = shaftDuct
    sdesc = "smaller ducts"
    adesc = "a smaller duct"
    isThem = true
    ldesc = "There are six smaller ducts, all too small to enter. "
    verIoPutIn(actor) =
    {
        "None of the ducts would be a good place to put that. ";
    }
    verDoEnter(actor) = { "None of the ducts are large enough. "; }
    doSynonym('Enter') = 'Board' 'Gothrough'
    verDoLookin(actor) = { }
    doLookin(actor) = { "You see nothing in any of the ducts. "; }
    doSynonym('Lookin') = 'Search'
;


shaftDuct_ventCover: fixeditem, openable
    isopen = nil
    noun = 'cover' 'vent'
    adjective = 'vent'
    location = shaftDuct
    sdesc = "vent cover"
    ldesc =
    {
        if (self.isopen)
            "The vent cover has been swung out and down, leaving the
            opening uncovered.  You could probably squeeze through
            the opening. ";
        else
            "The vent cover is hinged so that it can be swung open for
            access through the opening.  It is currently closed. ";
    }
    hasBeenOpened = nil
    isDone = nil
    notifierState = 0
    notifier =
    {
        local disp;
        
        /* 
         *   if the player hasn't seen the bits in the narrow passage yet,
         *   don't proceed until they have 
         */
        if (!narrowPassage_space.isseen)
            return;

        /* 
         *   Note whether we're displaying messages - do so only if the
         *   player is present and the vent is open.  The action will
         *   proceed even if we're not printing messages.
         */
        disp := (parserGetMe().isIn(self.location) && self.isopen);

        /* see what I'm doing */
        switch(self.notifierState)
        {
        case 0:
            if (disp)
                "\bYou hear some voices from the room below,
                but you can't make out what they're saying. ";
            break;

        case 1:
            if (disp)
                "\bSeveral workers wearing isolation suits and gas masks,
                together carrying a large silver crate the size of a
                refrigerator, walk into view in the room below.  They
                put the crate on the floor. ";

            lab7_crate.moveInto(lab7);
            break;

        case 2:
            if (disp)
                "\bOne of the workers visible below walks slowly around the
                silver crate, scanning the crate with a hand-held device.
                He makes two circuits of the crate, then clips the device
                to his belt and takes off his mask.  After a few moments,
                the other workers all remove their masks as well. ";
            break;

        case 3:
            if (disp)
                "\bThrough the opening in the floor, you see the workers
                file out of the room to the south. ";

            /* the people are gone now */
            lab7_people.moveInto(nil);

            /* we're done - turn off the notifier */
            unnotify(self, &notifier);

            /* set a flag indicating that I've been fully seen */
            self.isDone := true;
            break;
        }

        /* advance to the next state */
        self.notifierState++;
    }
    doOpen(actor) =
    {
        "You swing the vent cover down and out, leaving the opening
        uncovered. ";
        self.isopen := true;
        lab7_ventCover.isopen := true;
        shaftDuct_chamber.moveInto(shaftDuct);

        /* start the daemon the first time we open it */
        if (!self.hasBeenOpened)
        {
            notify(self, &notifier, 0);
            self.hasBeenOpened := true;
        }
    }
    doClose(actor) =
    {
        "You swing the vent cover back into position over the opening. ";
        self.isopen := nil;
        lab7_ventCover.isopen := nil;
        shaftDuct_chamber.moveInto(nil);
    }
    doEnter -> shaftDuct_opening
    doLookin -> shaftDuct_opening
    doSearch -> shaftDuct_opening
    doLookthru -> shaftDuct_opening
;

shaftDuct_chamber: fixeditem
    noun = 'chamber'
    adjective = 'cavernous'
    sdesc = "cavernous chamber"
    ldesc = { shaftDuct_opening.ldesc; }
;

shaftDuct_opening: fixeditem
    noun = 'opening'
    adjective = 'square'
    sdesc = "square opening"
    location = shaftDuct
    ldesc =
    {
        if (shaftDuct_ventCover.isopen)
        {
            "A cavernous space the size of an airplane hangar is
            visible through the opening; the floor is a hundred feet
            below.  Numerous strange pieces of equipment are scattered
            throughout, and a workbench is in the center of the room. ";

            if (lab7_crate.isIn(lab7))
                "A silver crate the size of a refrigerator in its
                back is next to the workbench. ";

            if (lab7_people.isIn(lab7))
                "\bSeveral workers, dressed in isolation suits and
                gas masks, are visible in the room below. ";

            if (lab7_scientist.isIn(lab7))
                "\bA woman, whom you take to be a scientist because of
                her white lab coat, and a technician, who's wearing a
                blue jump-suit, are visible in the room below. ";
        }
        else
            "The vent cover is closed. ";
    }
    verIoPutIn(actor) = { }
    ioPutIn(actor, dobj) =
    {
        "You'd rather not; it's a long way down. ";
    }
    verDoEnter(actor) = { }
    doEnter(actor) = { actor.travelTo(self.location.down); }
    doSynonym('Inspect') = 'Lookin' 'Lookthru' 'Search'
    doOpen -> shaftDuct_ventCover
    doClose -> shaftDuct_ventCover
;

