/*
 * Polyadventure
 *
 * A remake of the various versions of the classic Adventure game by Don
 * Woods and Willie Crowther, based on their sources.  Currently, the 350,
 * 550, and 551-point versions are implemented.  See the file "ccr-help.t"
 * for more information.
 *
 * Please document all changes in the history so we know who did what.
 *
 * This source code is copylefted under the terms of the GNU Public
 * License.  Essentially, this means that you are free to do whatever
 * you wish with this source code, provided you do not charge any
 * money for it or for any derivative works.
 *
 *
 * Contributors (see history.t for current e-mail addresses)
 *
 *      djp     In real life:   David J. Picton
 *
 * Modification History
 *
 * 13-Jan-03    djp     Version 2.09 rev C: bugfixes and code tidy-up
 *                      Changes in this version:
 *                      * Moved the 'extra' items (torn scrap, base-metal
 *                        rings) to this file, as a preparation for the
 *                        extended game version (701+ points).
 *
 * 12-Aug-03    bjs     Version 2.11: added 580-point mode.
 *
 * 23-Jan-04    djp     Version 3.00.  Addition of a new game mode (701+).
 *
 * 15-Jan-05    djp     Version 3.01: bugfix release.
 *                      * The green rod is now correctly described as green,
 *                        including the star color when it is spun.
 *                      * Fixed READ GREEN PENDANT.
 *
 * 12-Apr-05    djp     Version 3.10: Bugfixes and enhancements
 * 7-May-05     djp     Version 3.10 Rev. A
 *                      * Minor grammatical correction to ldesc of lead ball.
 *
 * 28-Jun-05    djp     Version 3.11: Enhancements and bugfixes
 *                      * Two objects have been added - a brass sphere which 
 *                        can be transmuted into an iridium sphere.  Note that
 *                        this is an optional treasure.  The game will 
 *                        finish with a declared "maximum" score of 581 points
 *                        if the room containing the brass sphere is not
 *                        unlocked.  If the room is unlocked, the maximum
 *                        score goes up to 593 points.
 *                      * The description of the damaged key now contains
 *                        a hint that it may be useful somewhere.
 *
 * 9-Aug-05     djp     Version 3.12: Enhancements and bugfixes
 *                      * If the steel keys are attached to the keyring, 
 *                        wearing the steel keys is now equivalent to wearing
 *                        the keyring. 
 *
 * 16-Jul-06    djp     Version 3.20: Bugfixes and extensions
 *                      * Added a second oblong box (in White-level machine
 *                        chamber, which the player can now reach). 
 *                      * Added code for rubbing gold and ceramic pendants 
 *                        (it toggles the strictcontrol property - initially 
 *                        nil.  If set to true, it causes the player to
 *                        remain at the same Transindection level after
 *                        spinning a rod (terrestrial destinations only)
 *                      * Added a cryptic note to the Manual about rubbing the 
 *                        pendant.
 *                      * Implemented the spinnable_rod and waveable_rod 
 *                        classes as appropriate.
 *                      * Added various objects (elevator key, copper bracelet,
 *                        multicolored rod, eight-pointed star, platinum bars,
 *                        a second oblong box)
 *
 * 12-Nov-06    djp     Version 3.20 Rev. A - minor corrections/enhancements
 *                      * Added a new feature to the multicolored rod, which
 *                        (as its name suggests) can now travel to different
 *                        Transindection levels. 
 * 20-Oct-07    djp     Version 3.20 Rev. C
 *                      * The eight pointed star now has an initial
 *                        description.
 *                      * Changed the description of the copper bracelet.
 *                        (Now plain - an obviously elvish bracelet ought
 *                        to be worth something, even in a poor condition)
 *    
 * 30-Mar-09    djp     Version 3.20 Rev. D
 *                      * Fixed a coding error in the verification method for
 *                        wearing the steel key.
 *
 * 21-Nov-10    djp     Version 3.21
 *                      * Coding changes to allow 'chapters' to be referred to
 *                        in the plural, e.g. READ CHAPTERS
 *                      * Minor changes to wording of pendant wearproc
 *                        code, to avoid overuse of the phrase 'you feel'
 *                        
 * 22-Oct-13    djp     Version 3.21a
 *                      * Vocab for multicolored rod allows spelling
 *                        'multicoloured'
 *
 */

ring_note: readable
    game551 = true
    game550 = true
    mass = 0
    sdesc = "torn scrap of paper"
    noun = 'paper' 'scrap'
    adjective = 'small' 'torn'
    ldesc = {
        "It's a small scrap of paper which seems to have been
        torn from a large piece.  The following cryptic message is
        written on it: \n\b";
        if (global.newgame) {
            "C=1\n
            B=2\n
            G=3\n";            // copper, brass, gold rings
            if(global.game701)
                "M=3\n";       // mithril ring

        }
        else
            "H=3\n
            R=3\n";       // helmet, mithril ring
        "Total 3: basic protection. \n
        Total 6: retaliation. \n";
        if (global.game701) {
            "Total 9: search and destroy. \b";
        // i.e. 3 points will protect you against the Dwarves' knives,
        // 6 will kill off the dwarves by sending the knives back.
        // 9 will destroy all dwarves after they try to attack you.
            if (global.game701p) {
                if (not self.isread)
                    "At first sight, the back of the paper appears to be
                    blank - until you notice that the bottom edge has
                    been folded over.  You unfold it, revealing the
                    words:\b ";
                else
                    "The back of the scrap also bears a message:\b ";
                "Pendant keywords: ANA, KATA, PHR...\n";
                "Unfortunately the bottom right hand corner is missing,
                so you can only see the first three letters of the third
                pendant keyword. ";
            }
        }
        self.isread := true;
    }
    location = Octagonal_Room
    location550 = Vault
    location701 = Octagonal_Room
;

// Class for objects which give protection against the dwarves' knives when
// worn.  The name is for historical reasons - the objects need not be rings
// (but should be wearable).

class protect_ring: thing
    verDoRub(actor) = {}
    doRub(actor) = {
        if(self.protection) {
            local weaplist := '',count := 0;
            if (self.isworn and self.location = actor) {
                "You rub <<self.thedesc>>";

                if(axe.isIn(actor)) {
                    weaplist := 'axe';
                    count++;
                }
                if(sword.isIn(actor)) {
                    if(weaplist = '')weaplist := 'sword';
                    else weaplist += ' and sword';
                    count++;
                }
                if(singing_sword.isIn(actor)) {
                    if(weaplist = '')weaplist := 'sword';
                    else if(sword.isIn(actor)) weaplist += 's';
                    else weaplist += ' and sword';
                    count++;
                }
                if(count > 0) {
                   " and you notice that your <<weaplist>> ";
                   if (count > 1) "seem"; else "seems";
                   " to be repelled away from it! ";
                }
                else ". Nothing unusual happens. ";
            }
            else "You rub <<self.thedesc>>.  Nothing exciting happens. ";
        }
        else pass doRub;
    }
;

copper_ring: clothingItem, protect_ring
    game551 = true
    mass = 1
    protection = 1
    sdesc = "small copper ring"
    ldesc = "It's a plain copper ring, of no obvious value. "
    noun = 'ring'
    plural = 'rings'
    adjective = 'small' 'copper'
    location = Cloak_Pit_2
;

brass_ring: clothingItem, protect_ring
    game551 = true
    mass = 1
    protection = 2
    sdesc = "small brass ring"
    ldesc = "It's a plain brass ring, of no obvious value. "
    noun = 'ring'
    plural = 'rings'
    adjective = 'small' 'brass'
    location = Cloak_Pit_3
;

/* There is also a gold ring - this is a treasure and was in the original
game, therefore it is defined in ccr-it1.t. */

transindection_color: function(index) {
    switch(index) {
                        case -3: return 'yellow'; 
                        case -2: return 'purple';
                        case -1: return 'orange';
                        case 0: return 'red';
                        case 1: return 'blue';
                        case 2: return 'green';
                        case 3: return 'white';
                        case 4: return 'magenta';
                        default: return 'black';
    }
}

class pendantItem: clothingItem
    plural = 'pendants'
    doWear(actor) = {
        inherited clothingItem.doWear(actor);
        if((self.location = actor) and self.isworn) self.wearproc(actor);
    }    
    wearproc(actor) = {
        if ((self.location = actor) and self.isworn) 
            "You experience a strange sensation as you put the chain around 
            your neck.  It seems to whisper the words \"Ana\" and \"Kata\" to
            you. ";
        if ((cloak.location = actor) and cloak.isworn)
            "It occurs to you that it would be a good idea to hide the
            pendant, so you slip it under the cloak where it won't be
            seen. ";
    }
    doInspect(actor) = {
        if ((cloak.location = actor) and cloak.isworn and
        (self.location = actor) and self.isworn)
            "\b(Taking the pendant out of the cloak to examine it)\b";
        inherited.doInspect(actor);
        if ((cloak.location = actor) and cloak.isworn and
        (self.location = actor) and self.isworn)
            "\b(Concealing the pendant under the cloak again.)\b";
    }
    doRead(actor) = {
        if ((cloak.location = actor) and cloak.isworn and
        (self.location = actor) and self.isworn)
            "\b(Taking the pendant out of the cloak to examine it)\b";
        inherited readable.doRead(actor);
        if ((cloak.location = actor) and cloak.isworn and
        (self.location = actor) and self.isworn)
            "\b(Concealing the pendant under the cloak again.)\b";
    }
    // cleaning is synonymous with rubbing
    doClean(actor) = { self.doRub(actor); }
    doCleanWith(actor,io) = { self.doRub(actor,io); }
;

pendant: pendantItem, CCR_treasure_item, hiddenItem
    game701p = true
    bonustreasure = true    // initially excluded from scoring-object lists!
    targloc = Me            // player should keep it with him
    contloc = nil
    sdesc = "gold pendant"
    pendantlist = [pendant, pendant2, pendant3]
    anaproplist = [&ana, &ana2, &ana3]
    kataproplist = [&kata, &kata2, &kata3]
    ldesc = {
        local actor := getActor(&verbActor);
        local toploc := toplocation(self);
        local ndots, lowdot, middledot, highdot;
        local pendantobj, anaprop, kataprop, i, l;

        // On the planet Zarkon the pendant will be blank unless it's
        // been Zarkonalized, or you're wearing another pendant which has
        // been.
        if (toploc.Zarkalonroom and not self.Zarkonalized) {
            local o := firstobj(pendantItem),zpend := nil;
            while (o) {
               if((o.location = actor) and o.isworn and o.Zarkalonized) 
                    zpend := true;
               o := nextobj(o,pendantItem);
            }
            if(not zpend) {
               "Strangely, the disk of your pendant is now completely
               featureless.  You suspect that it won't work here. ";
            }
        }
        "It's a disk of gold with a chain attached, so that you
        could wear it round your neck.  The diameter is about two inches. ";
        if(self.Zarkalonized) 
             "One side of the disk bears a slanted checkerboard pattern, and
             the other is plain";
        else
             "The disk is plain";
        ndots := 1;
        
        l := length(pendantlist);
        // Check that the gold pendant is worn.  (If not, it only shows one
        // dot.)
        if((self.location = actor) and self.isworn) {
            // Determine the actual class in which the default 
            // kata/ana methods are defined (in case a subsequent
            // modify block has been coded for the room class.  We assume that
            // all methods are defined in the same block of code.)
            if (room.transclass = nil)
                room.transclass:= defined(room, &kata, DEFINED_GET_CLASS); 
            // step through all the pendants to determine whether any of their
            // associated travel properties are defaulted or defined by
            // the room.  If they are defined, the pendant regards 
            // transindection as possible (although the destination room may 
            // sometimes be blocked.)
            for (i := 1; i <= l; i++) {
                pendantobj := pendantlist[i];
                if ((pendantobj.location = actor) and pendantobj.isworn) {
                    anaprop := anaproplist[i];
                    kataprop := kataproplist[i];

                    if (defined(toploc,kataprop,DEFINED_GET_CLASS) <> 
                    room.transclass) {
                        lowdot := toploc.analevel - 1;
                    }
                    if (defined(toploc,anaprop,DEFINED_GET_CLASS) <> 
                    room.transclass) {
                        highdot := toploc.analevel + 1;
                    }                        
                }
            }
        }
        if (lowdot <> nil) ndots++;
        if (highdot <> nil) ndots++;
        middledot := toploc.analevel;
        switch(ndots) {
            case 1:
                " except for a single ";
                say (transindection_color(middledot));
                " dot with a circle around it. ";
                break;
            case 2:
                " except for two dots colored ";
                if (lowdot <> nil) {
                    say (transindection_color(lowdot));
                    " and ";
                    say (transindection_color(middledot));
                }
                else {
                    say (transindection_color(middledot));
                    " and ";
                    say (transindection_color(highdot));
                }
                ". There is a circle around the ";
                say (transindection_color(middledot)); 
                " dot. ";
                break;
            case 3:
                " except for three dots colored ";
                say (transindection_color(lowdot));
                ", ";
                say (transindection_color(middledot));
                " and ";
                say (transindection_color(highdot));
                ". There is a circle around the ";
                say (transindection_color(middledot));
                " dot. ";
                break;
        }
    }
    noun = 'pendant' 'amulet' 'medallion' 'dots'
    adjective = 'gold' 
    searchLoc = Blue_Upper_Trans_Room_Rockfalls
    // stop the player from taking off the pendant while away from Red
    // home transindection level (0), or in the blocked-off area.
    verDoUnwear(actor) = {
        local toploc := toplocation(actor);
        if(self.isworn and ((toploc.analevel <> 0) or toploc.isolated))
            self.notrans(actor);
        else pass verDoUnwear;
    }
    verDoPutIn(actor, io) = {
        if(io =StrangeMachine) 
            inherited.verifyRemove(actor);
        else
            pass verDoPutIn;
    }
    verifyRemove(actor) = {
        local toploc := toplocation(actor);
        if(self.isworn and ((toploc.analevel <> 0) or toploc.isolated))
            self.notrans(actor);
        else pass verifyRemove;
    }
    notrans(actor) = {
        local toploc := toplocation(actor);
        if(self.location = actor) {
            if(toploc.analevel = 0)
                "A voice whispers in your ear: \"Security error 10A: removal
                of the Pendant is not recommended in isolated areas.\" ";
            else {
                "A voice whispers in your ear: \"Security error 10B: removal
                of the Pendant is not recommended in this area.\" ";
            }
            "This sounds like good advice, so you decide not to take off
            the pendant. ";
        }
    }
    // this property flags the rooms which can only be reached by
    // transindection movements
    trans_scene(...) = { 
        local toploc;
        if (argcount > 0) toploc := toplocation(getarg(1));
        else toploc := toplocation(self);
        if (toploc = nil) return nil;
        return ((toploc.analevel <> 0) or toploc.isolated);
    }
    wearproc(actor) = {
        inherited.wearproc(actor);
        if(not self.firstworn) {
            P(); I();
            "A few seconds later, you sense that something is wrong.  The words
            \"Pendant Error 3: Recharge required\" seem to pass 
            through your mind, and you look again at the pendant.  Strangely, 
            it no longer appears to be made of gold - in fact it's now a plain
            two-inch disk of tarnished brass! ";
            tarnished_pendant.moveInto(self.location);
            tarnished_pendant.isworn := true;
            self.firstworn := true;
            self.moveInto(nil);
        }
    }
    // Wizards can purloin the pendant without needing to recharge it.
    doPurloin(actor) = {
        self.firstworn := true;
        pass doPurloin;
    }

    doRub(actor) = {
        if ((self.location = actor) and self.isworn) {
            self.isrubbed := true;
            self.strictcontrol := (not self.strictcontrol);
            if(self.strictcontrol)
                "The pendant seems to whisper to your mind: \"Strict control
                enabled.  Unregistered objects will not be permitted to 
                mediate Transindection movements.\" ";
            else
                "The pendant seems to whisper to your mind: \"Strict control
                disabled.  Mediation of Transindection movements by
                unregistered objects will be notified.\" ";
        }
        else inherited.doRub(actor);
    }
;

crumpled_sheet: item, readable
    game701p = true
    sdesc = "crumpled sheet of paper"
    ldesc = {
        "It's a crumpled sheet of paper, apparently torn from a larger
        piece.  You can make out the following: \b
        \"...pus population levels in Green level necessit... \n
        ... due to objections from orange-level occupants al... \n
        ... upper transindection chamber is now restricted t... \n
        ...nd cave occupants have agreed to block access to lower ch... \n
        ...pi have breached the stone wall but hopefully lack the abilit...\n
        ...nze and gold pendants are restricted to Red and Blue levels in... \n
        ...wever gold pendants will always permit return travel to Red... \n 
        ...amaged ceramic disks can interact with unregistered Transin ...\n
        ...nforunately this area is prone to rockfalls due to faulting... \n
        ...l funding is found for repairs the access to the Control Roo... \n
        ...all keys are now kept in the Orange Level Office, and a searc... \n
        ...not possible to change pendant travel keywords, but we hope... \n
        ...ismiss all talk of transindectional travel as old cave lore. \"";

        self.isread := true;
    }
    noun = 'sheet' 'paper'
    adjective = 'crumpled' 
    location = Upper_Trans_Room
;

transindection_key: keyItem, clothingItem, hiddenItem
    game701p = true
    sdesc = "oddly-shaped steel key"
    adesc = "an <<self.sdesc>>"
    ldesc = "It's a steel key with a strange, complex shape.  A neck-chain 
        is attached to the key. "
    noun = 'key'
    adjective = 'steel' 'odd' 'oddly' 'shaped' 'oddly-shaped'
    searchLoc = Wumpus
    verDoWear(actor) = {
        if(self.location = set_of_keys) 
            self.location.verDoWear(actor);
        else
            pass verDoWear;
    }
    doWear(actor) = {
        if(self.location = set_of_keys) 
            self.location.doWear(actor);
        else
            pass doWear;
    }
;

iron_key: keyItem, readable
    game701p = true
    sdesc = "iron key"
    adesc = {"an "; self.sdesc;}
    ldesc = "It's a rather large, rusty iron key.  On close examination
        you can make out the words \"Control Room\". "
    noun = 'key'
    adjective = 'iron' 'large'
;
pendant2: pendantItem, CCR_treasure_item, readable  /* green pendant */
    game701p = true
    bonustreasure = true    // initially excluded from scoring-object lists!
    sdesc = "green pendant"
    ldesc = {
        "It's a green disk on a chain.  One side bears the words
        \"Type IIA: extended area\"";
        if(self.Zarkalonized)
             ", and the other side bears a motif resembling a slanted
             checkerboard. ";
        else
             ". ";
        "Like the orange pendant, this one
        has a disk of the same size as the gold pendant, and an
        identical neck-chain. ";
    }
    noun = 'pendant' 'amulet' 'medallion'
    adjective = 'green'
    targloc = Me
    contloc = nil
    wearproc(actor) = {
        if ((self.location = actor) and self.isworn) 
            "You experience a strange sensation as you put the chain around 
            your neck.  It seems to whisper the word \"Ana\" to you, and
            you somehow feel an awareness that the scope of your gold 
            pendant has been expanded ... ";
        if ((cloak.location = actor) and cloak.isworn)
            "It occurs to you that it would be a good idea to hide the
            pendant, so you slip it under the cloak where it won't be
            seen. ";
    }
;
pendant3: pendantItem, CCR_treasure_item  /* orange pendant */
    game701p = true
    bonustreasure = true    // initially excluded from scoring-object lists!
    sdesc = "orange pendant"
    adesc = "an orange pendant"
    ldesc = {
        if(self.Zarkalonized)
           "It's an orange disk, bearing a slanted checkerboard motif on 
           one side and ";
        else 
           "It's a completely featureless orange disk on a chain, ";
        "otherwise similar to the gold pendant.  In fact the orange disk is 
        exactly the same size as the gold disk, and the neck-chain is 
        identical.  ";
    }
    noun = 'pendant' 'amulet' 'medallion'
    adjective = 'orange'
    location = Blue_Level_Hooks // in case the player somehow gets the
                                // topaz without obtaining the orange pendant
    targloc = Me
    contloc = nil
    wearproc(actor) = {
        if ((self.location = actor) and self.isworn) 
            "You experience a strange sensation as you put the chain around 
            your neck.  It seems to whisper the word \"Kata\" to you, and
            you somehow feel an awareness that the scope of your gold
            pendant has been expanded, albeit in a limited area ... ";
        if ((cloak.location = actor) and cloak.isworn)
            "It occurs to you that it would be a good idea to hide the
            pendant, so you slip it under the cloak where it won't be
            seen. ";
    }
;

broken_key: keyItem
    game701p = true
    sdesc = "damaged key"
    ldesc = "This was an identical copy of your iron key, but is now 
            badly damaged and certainly won't fit the lock for which it was 
            intended. "
    noun = 'key'
    adjective = 'broken' 'damaged'
;
    
broken_pendant: pendantItem //almost useless, disappears if charged
    game701p = true
    sdesc = "broken ceramic pendant"
    ldesc = "This looks like another Transindection pendant, but the disk is
            broken and I doubt whether it will work. "
    noun = 'pendant' 'amulet' 'medallion'
    adjective = 'broken' 'shattered' 'damaged' 'ceramic'
    wearproc(actor) = {
        if ((self.location = actor) and self.isworn) 
            "You experience a slight tingling sensation as you put the broken
            pendant around your neck, but nothing else.  I suspect that it's 
            no use for anything much ... ";
        if ((cloak.location = actor) and cloak.isworn)
            "It occurs to you that it would be a good idea to hide the
            pendant, so you slip it under the cloak where it won't be
            seen. ";
    }    
    doRub(actor) = {
        if ((self.location = actor) and self.isworn) {
            self.isrubbed := true;
            self.strictcontrol := (not self.strictcontrol);
            if(self.strictcontrol)
                "The broken pendant seems to whisper to your mind: \"Strict 
                control enabled.  Unregistered objects will not be permitted to
                mediate Transindection movements.\" ";
            else
                "The broken pendant seems to whisper to your mind: \"Strict 
                control disabled.\" ";;
        }
        else inherited.doRub(actor);
    }
;

discolored_disk: item 
    // This object is important in the Zarkalon tower puzzle (where the
    // Red-level pendant-charging machine must be 'ana'ed into a locked
    // cupboard at Green level, causing a spin-rod in the cupboard to be
    // 'kata'ed to Blue level.
    //
    // If it is placed in a pendant-charging machine, any rods at the
    // next Transindection level are 'kata'ed to the room.  The machine
    // will be 'ana'ed if there isn't already a pendant-charging machine in 
    // the destination room.
    //
    game701p = true
    sdesc = "discolored aluminum disk"
    ldesc = {
        if (memo.isread)
            "This is obviously the crude replica pendant described in the
            Blue-level memo.  It seems to have caused the pendant-charging 
            machine to malfunction, such that it \"ana\"ed itself to Green
            level and presumably \"kata\"ed the gray rod from Green level to
            Blue level. ";
        else
            "It's a discolored aluminum disk.  The size of the disk, and
            the remnants of a now-melted neck-chain suggest that this was
            once a Transindection pendant. ";
    }
    noun = 'disk' 'disc' 'pendant' 'amulet' 'medallion'
    plural = 'pendants'
    adjective = 'discolored' 'melted' 'aluminum' 'aluminium'
    verDoWear(actor) = {"The disk once had a neck-chain, but it has almost
        completely melted away.  You can't wear it.";}
    location = Green_Bird_Chamber
    // Code to handle PUT DISK IN MACHINE.  The object is to re-enact
    // the scenario described in the memo.
    machinespecial = {
        local actor := getActor(&currentActor), moved := nil;
        local macloc := StrangeMachine.location;
        local toploc, dest, topdest, macloclist := StrangeMachine.loclist;
        local n := 0, o, i := 0, l := length(global.spinrodlist);
        toploc := toplocation(self);  // Current top level room
        dest := toploc.machinetrans;  // where the machine will actually go
        topdest := toplocation(dest); // The top level destination room
        for (i := 1; i <= l; i++) {
            o := global.spinrodlist[i];
            if (o.isInside(topdest)) n++;
        }
        if(topdest = nil) {
            "Nothing happens.  You take the disk out again. ";
            return;
        }
        "Shortly after you place the disk into the slot, something very
        odd happens.  The room shimmers, and you feel as if you are trapped
        in a kind of bubble, unable to escape! ";
        "The room then changes.  Although you are unable to leave the
        bubble, you find that you have been transported to the next
        Transindection level! ";
        P();
        actor.travelTo(topdest);
        P();
        "After about half a minute, there is a brilliant flash of light.
        When your eyes start working again, you find yourself back in the
        original room: ";

        P();
        if (n > 0) {
            for (i := 1; i <= l; i++) {
                o := global.spinrodlist[i];
                if (o.isInside(topdest) and not o.isInside(actor)) 
                    o.moveInto(toploc);
            }
            if (find(macloclist,dest) = nil) {
                moved := true;
                macloclist -= macloc;
                macloclist += dest;
                StrangeMachine.moveLoclist(macloclist);
            }
        }
        if(moved) 
            self.moveInto(dest);
        else 
            self.moveInto(toploc);
        // Kata the player back, then tell him what's happened.
        actor.travelTo(toploc);
        if(n = 1) {
            P();"You notice that a rod has now been \"Kata\"ed down from 
            the next level up, ";
        }
        else if (n > 1) {
            P();"You notice that <<n>> rods have now been \"Kata\"ed down from 
            the next level up, ";
        }
        else if ((n = 0) and memo.isread) {
            P();"No rods have been transported into this room, and the machine
            hasn't been transported either.  Maybe the machine won't \"Ana\"
            itself unless there's a rod to \"Kata\" from the next
            level. ";
        }
        if (moved and (n > 0))
            "and the machine has disappeared.  I guess that it's \"Ana\"ed 
            itself to the next level. ";
        else if (n > 0)
            "and the machine is here but looks translucent - no, it's
            solid again.  I guess that it tried to
            \"Ana\" itself, but rebounded back to this level because there's 
            another machine at the next one. ";
    }   
;

dull_pendant: pendantItem
    game701p = true
    sdesc = "dull green pendant"
    ldesc = "The color has now changed.  The pendant is now a dull green. "
    noun = 'pendant' 'amulet' 'medallion'
    adjective = 'dull' 'green' 
    wearproc(actor) = {
        if ((self.location = actor) and self.isworn) 
            "You feel nothing unusual as you put the dull pendant around
            your neck.  I suspect that it doesn't work any more ...";
        if ((cloak.location = actor) and cloak.isworn)
            "It occurs to you that it would be a good idea to hide the
            pendant, so you slip it under the cloak where it won't be
            seen. ";
    }    
;

tarnished_pendant: pendantItem
    sdesc = "tarnished pendant"
    ldesc = "This pendant no longer appears to be made of gold!  It now looks
        like a plain two-inch disk of tarnished brass.  "
    noun = 'pendant' 'amulet' 'medallion'
    adjective = 'tarnished' 'brass'
    wearproc(actor) = {
        if ((self.location = actor) and self.isworn) 
            "You feel nothing unusual as you put the tarnished pendant around
            your neck.  I suspect that it doesn't work any more ...";
        if ((cloak.location = actor) and cloak.isworn)
            "It occurs to you that it would be a good idea to hide the
            pendant, so you slip it under the cloak where it won't be
            seen. ";
    }    
;

livingRoomKey: keyItem
    game701p = true
    sdesc = "aluminum key"
    adesc = {"an "; self.sdesc;}
    ldesc = "It's a small key, made of aluminum. "
    noun = 'key' 'object'
    adjective = 'small' 'aluminum' 'aluminium' 'metallic'
;

wumpi_book: readable
    game701p = true
    sdesc = "history book"
    ldesc = "This book appears to have been written by a human called 
        \"Jedediah Smith\", and is entitled \"History of the Wumpi in Bedquilt
        Cave\".  It is divided into four chapters: Early History, 
        Civilization of the Wumpi, Escape of the Rapscallion, and Achievements
        of the Elvish Masters.  A handwritten note on the title page reads:
        \"Protected by Elvin Sorcery.  Do not remove.\" "
    noun = 'book' 'history' 'bedquilt' 'cave'
    adjective = 'history'
    moveInto(loc) = {
        local i, o, len, chaplist = [Chapter1, Chapter2, Chapter3, Chapter4];
        len := length(chaplist);
        for (i := 1; i <= len; i++) {
            o := chaplist[i];
            o.moveInto(loc);
        } 
        pass moveInto;
    }
    location = Bookcase 
    chapsread = 0 // Number of chapters read. 
;

class Chapter: fixeditem, readable
    game701p = true
    verDoTake(actor) = {"That's part of the book. ";}
    verDoPutIn(actor,io) = {self.verDoTake;}
    ldesc = {
        local actor := getActor(&verbActor);
        if (not self.isread) {
            self.isread := true;
            wumpi_book.chapsread++;
            if(wumpi_book.chapsread = 4) {
                "A small metallic object falls out of the book and onto the
                floor. "; P();
                livingRoomKey.moveInto(toplocation(self));
            }                  
        }
        self.text;             
        P();
    }
    noun = 'chapter'
    plural = 'chapters'
;

Chapter1: Chapter
    sdesc = "Early History"
    adjective = 'early' 'history' '1'
    text = {
        "You learn that there are two rival groups of
        elves - the Mountain Elves and the Castle Elves.  The
        Castle Elves once lived in the area of the Octagonal
        Room, but most of them migrated to the Castle after rock-falls made 
        this area unsafe.  The remaining elves occupied another area of the
        cave, and came to be known as the Mountain elves.  Around this time, 
        the Wumpi at Green level were thought to present a threat to other 
        Transindection levels, and the cave management at each Level was asked
        to block off all access to the Transindection Chambers. ";
    }
    location = Bookcase
;

compoundWord 'civilization' 'of' 'civilization-of';
compoundWord 'civilisation' 'of' 'civilization-of';
Chapter2: Chapter
    sdesc = "Civilization of the Wumpi"
    adjective = 'civilization-of' 'wumpi' '2'
    text = {
        "This chapter describes how the Wumpi began to change their ways.  It
        would appear that they no longer eat human flesh, and live in peace
        with elves and humans. ";
    }
    location = Bookcase
;
Chapter3: Chapter
    sdesc = "Escape of the Rapscallion"
    adjective = 'escape' 'rapscallion' '3'
    text = {
        "This chapter tells of one renegade Wumpus who decided to
        continue with the \"Old Ways\".  He turned up at Red level, and it
        was suspected that he had found one of the lost gold pendants.  The
        elves planned to recapture the Wumpus and return him to Green level,
        but this plan was cancelled when the
        Green-level Mountain elves fell out with the Castle Elves.
        In case the Wumpus ever returned, the Wumpi were ordered to secure the
        exit from their quarters with a remotely-operated lock.  It is in use 
        to this day. ";

        P();
        "The Castle Elves decided to contain the problem by confining the
        Wumpus to Red level.  They sent their chief engineer, Eldrand, to
        de-activate both of the Transindection Chambers and all of the
        gold pendants.  Unfortunately he found it necessary to do 
        this from the Control Room, which was known to be unsafe.  He never 
        returned from his mission, having apparently died in the Control 
        Room.  The Castle Elves learned that the Lower Transindection Room
        no longer worked at Green level, and assumed that Eldrand's mission
        had succeeded. ";
        if (transindection_Button.knowgreenfail)
            "(Of course, you know that he succeeded only in disabling the
            Lower Transindection Chamber at green level ";
        else
            "(Of course, you know that he failed ";
        "- and the Wumpi now know this as well.) ";

        P();
        "The book then goes on to speculate on the reasons why the Castle
        Elves never mounted a follow-up mission.  It concludes that they
        were concerned to prevent their Transindection pendants from falling
        into the wrong hands, then refers the reader to the following
        chapter ... ";
    }
    location = Bookcase
;
Chapter4: Chapter
    sdesc = "Achievements of the Elvish Masters"
    adjective = 'achievements' 'elvish' '4'
    text = {
        "The final chapter gives you cause for concern.
        It appears that the Mountain Elves at Green level seem to have become 
        very militaristic in recent years, and have taken over most of the 
        cave and the surrounding area.  It all started with the discovery of 
        three sealed burial chambers, containing 114 elven crowns, 101 suits
        of mithril silver chain-mail and 79 steel swords.   This made the
        elves almost invincible, even when greatly outnumbered.  However, 
        they failed to conquer the Castle, due to the ability of the 
        Castle Elves to bring in unlimited supplies using some form of
        teleportation magic.  It has long been suspected that Transindection
        was used. ";
        P();
        "(You feel sure of one thing.  There would be unpleasant consequences
        if the Green-level Mountain Elves managed to get hold of any
        Transindection pendants.  You'd better watch your step here.) ";
    }
    location = Bookcase
;

manual: readable
    game701p = true
    sdesc = "control room manual"
    ldesc = {
        "The manual is written in English, Dwarvish and Elvish.  
        You can't read Dwarvish and Elvish, and the English version is no
        better - it's full of incomprehensible jargon.  ";

        // If the scrawled note or pencilled note are referred to, read just
        // the relevant section.  (However, if the manual hasn't been read,
        // both sections are shown).

        // Hint to rub the pendant

        if (pendant.isrubbed or broken_pendant.isrubbed)
             "There are many written notes.  Some of them now seem to
             make sense - for example, you realize what is meant by
             \"oscillatory contact\" to prevent \"mediation [of Transindection
             movements] by unregistered objects\".  But the majority are
             still quite incomprehensible. ";
        else
            "There are many written notes, most of which make little sense to 
            you. For example, one section refers to the use of \"oscillatory 
            contact\" with a \"portable mediator\" in order to prevent 
            \"unwanted mediation by unregistered objects\". ";

        
        if(((objwords(1)[1] != 'pencilled') and
          (objwords(1)[1] != 'penciled')) or (not self.isread)) {

            if (black_rod.isspun or gray_rod.isspun or multicolored_rod.isspun
            or DisplayBoards.spanseen) {
                if(self.misunderstood) {
                    self.misunderstood := nil;
                    "A scrawled note seems to read:
                    \"Borrowed a spare Astral Staff\". This ";
                    if (black_rod.isspun) 
                        "almost certainly ";
                    else
                        "possibly ";
                    "refers to a rod like the one you found.  You then slap 
                    your forehead in embarrassment.  The first sentence was in
                    the past tense, and so was the second: the writer \"span 
                    the Staff over the Green-level fissure to gain entry to 
                    the Orange Stone Chamber.\" ";
                }
                else {
                    "An exception is a scrawled note 
                    which reads 
                    \"Borrowed a spare Astral Staff.\" A rod like the one you 
                    found? ";
                    if (black_rod.isspun) 
                        "Almost certainly, ";
                    else
                        "Maybe, ";
                    "and you read with 
                    interest that the rod was then \(spun\)! The writer
                    \"span the Staff over the Green-level fissure to gain
                    entry to the Orange Stone Chamber.\" ";
                }
            }
            else {
                "A scrawled note starts \"Borrowed a spare Astral 
                Staff.\"  Could this be a rod with a star on an end, like the
                one you found in the Debris room?  You read the next sentence 
                as an instruction, although it would be an impossible task: 
                \"Span the Staff over the Green-level fissure to gain entry to
                the Orange Stone Chamber.\"  ";
                self.misunderstood := true;
            }
            P(); I();
        }
    
        // pencilled note (re topaz)

        if((objwords(1)[1] != 'scrawled') or (not self.isread)) {
            if(((objwords(1)[1] = 'pencilled') or
            (objwords(1)[1] = 'penciled')) and self.isread)
                "A section of the manual refers ";
            else
                "Another section also draws your attention.  It refers ";
            "to an emergency procedure for gaining access to a Maintenance 
            Room.  It would appear that the Eldrand-Fitzgerald Topaz must be 
            brought close to the west wall of the room, which will then become 
            \"permeable\".  This has been crossed out, and a penciled note 
            underneath seems to hint at the reason: ";
            P();
            "\"Topaz activation threshold = 1.5 Fzg (following damage) \n
            Permeation field just outside Maintenance Room = 1.1 
            Fzg (maximum)\"\n A further note has been added underneath: ";
            P();
            "\"Try aqua mithrilis (N.B. gold content unknown; reaction rate
            slow if more than 1%)\" ";
            P();
        }   
        self.isread := true;
    }
    noun = 'manual'
    adjective = 'control' 'room' 'control-room' 'scrawled' 'pencilled' 
    'penciled' 'note' //'note' defined as adjective to avoid conflicts
;

topaz: CCR_treasure_item
    game701p = true
    bonustreasure = true
    sdesc = "topaz"
    ldesc = "It's a large, almost perfect crystal - except for a slight
        chip off one corner. "
    noun='topaz'
    adjective = 'large' 
    location = Green_Stone
    verDoTouch(actor) = {
        if((self.location = Green_Stone) and Green_Stone.isglowing)
            "Your hand passes straight through <<self.thedesc>>! ";
        else pass verDoTouch;
    }
    verDoRub(actor) = {
        if((self.location = Green_Stone) and Green_Stone.isglowing)
            "Your hand passes straight through <<self.thedesc>>! ";
        else pass verDoRub;
    }
;

lead_sphere: item
    game701p = true
    sdesc = "lead sphere"
    ldesc = {
        if(iridium_sphere.moved or platinum_sphere.moved or not
        Zarkalon_Tower_Plaque.isread)
            "It's a shiny lead sphere, about three inches in
            diameter. ";         
        else
            "You look at the sphere with disappointment.  Instead of the
            valuable platinum sphere you were hoping for, this is a three-inch
            ball of lead.  Its surface is unusually shiny, but it's certainly
            not worth much. ";
    }
    // DJP - corrected It's to Its in last sentence.
    noun = 'sphere' 'ball'
    adjective = 'lead'
    location = Doorsite_SmallRoom
;

platinum_sphere: CCR_treasure_item
    game701p = true
    bonustreasure = true
    sdesc = "platinum sphere"
    ldesc = "It's a very shiny sphere of platinum, about three inches
        in diameter! "
    noun = 'sphere' 'ball'
    adjective = 'platinum'
    location = nil 
;

brass_sphere: item
    game701p = true
    sdesc = "brass sphere"
    ldesc = {
        if(iridium_sphere.moved or platinum_sphere.moved or not
        Zarkalon_Tower_Plaque.isread)
            "It's a shiny brass sphere, about three inches in
            diameter. ";         
        else
            "You look at the sphere with disappointment.  Instead of the
            valuable platinum sphere you were hoping for, this is a three-inch
            ball of brass.  Its surface is unusually shiny, but it's
            not particularly valuable. ";
    }
    // DJP - corrected It's to Its in last sentence.
    noun = 'sphere' 'ball'
    adjective = 'brass'
    location = Green_Doorsite_SmallRoom
;

iridium_sphere: CCR_treasure_item
    game701p = true
    bonustreasure = true
    sdesc = "iridium sphere"
    adesc = "an iridium sphere"
    ldesc = "It's a very shiny sphere of iridium, about three inches
        in diameter! "
    noun = 'sphere' 'ball'
    adjective = 'iridium'
    location = nil 
;

compoundWord 'power' 'pack' 'powerpack';
compoundWord 'power' 'packs' 'powerpacks';

class oblong_box_class: item 
// Power pack for machine in blue-level Zarkalon tower.
    game701p = true
    sdesc = "oblong box"
    adesc = "an oblong box"
    pluraldesc = "oblong boxes"
    noun = 'box' 'powerpack' 'power-pack'
    plural = 'boxes' 'powerpacks' 'power-packs'
    adjective = 'oblong'
    ldesc = "It's a featureless rectangular box, made from a nondescript 
    grey metal.  It measures about 6 by 4 by 2 inches. "
    isEquivalent = true
;
oblong_box_1: oblong_box_class
    location = Green_Maintenance_Room
;
oblong_box_2: oblong_box_class
    location = White_Machine_Chamber
;

gray_rod: item, waveable_rod, spinnable_rod 
               // This rod is exactly the same as the black rod in most
               // respects.
    mass = 2
    sdesc = "gray rod"
    ldesc = {
        "It's a three foot gray rod with a ";
        if(self.isupgraded) "shiny ";
        else "rusty ";
        "star on an end. ";
    }
    location = Blue_Bird_Chamber
    noun = 'rod' 'wand'  // DJP - removed 'star' from noun list
    plural = 'rods' 'wands'
    // 'rusty' is changed dynamically - DJP
    adjective = 'gray' 'grey' 'rusty' 'star' 'magic'
    mycolor = 'green'
    myhome = Green_Bird_Chamber
    myhome2 = [In_Bird_Chamber, Blue_Bird_Chamber, Green_Bird_Chamber]
;
red_rod: item, spinnable_rod
               //N.B. the Zarkalon rods work only when spun
    mass = 2
    sdesc = "red rod"
    ldesc = {
        "It's a two foot red rod with a dull metal star on an end. ";
    }
    noun = 'rod' 'wand'
    plural = 'rods' 'wands'
    adjective = 'red' 'star' 'magic'
    // This rod is for spinning only.
    doWave(actor) = {
        "Nothing happens, and you somehow sense that this rod requires a
        different kind of motion. ";
    }
    mycolor = 'red'
    location = Zarkalon_Tower_Basement
    myhome = Zarkalon_Tower_Basement
    // Not possible to block Transindection movements on Zarkalon
    myhome2 = [Zarkalon_Tower_Basement, Zarkalon_Tower_Basement,
    Zarkalon_Tower_Basement]
;
green_rod: item, spinnable_rod
                //N.B. the Zarkalon rods work only when spun.
                //This is the useful rod.
    mass = 2
    sdesc = "green rod"
    ldesc = {
        "It's a two foot green rod with a dull metal star on an end. ";
    }
    noun = 'rod' 'wand'
    plural = 'rods' 'wands'
    adjective = 'green' 'star' 'magic'
    // This rod is for spinning only.
    doWave(actor) = {
        "Nothing happens, and you somehow sense that this rod requires a
        different kind of motion. ";
    }
    mycolor = 'green'
    location = GreenZarkalonCabinet
    myhome = Green_Zarkalon_Tower_Basement
    // Not possible to block Transindection movements on Zarkalon
    myhome2 = [Green_Zarkalon_Tower_Basement, Green_Zarkalon_Tower_Basement,
    Green_Zarkalon_Tower_Basement]
;

multicolored_rod: item, spinnable_rod
    // Similar to Zarkalon rods (spinning only) but with a special
    // feature.  A star must be attached first.
    mass = 2
    sdesc = "multicolored rod"
    ldesc = {
        if(self.activated)
            "It's a two-foot multicolored rod with an eight-pointed star on an
            end. ";
        else
           "It's a two-foot multicolored rod without a star on either end. ";
    }
    noun = 'rod' 'wand'
    plural = 'rods' 'wands'
    adjective = 'multicolored' 'multicoloured' 'magic'
    needs_activation = true
    // This rod is similar to the Zarkalon ones - for spinning only.
    doWave(actor) = {
        "Nothing happens, and you somehow sense that this rod requires a
        different kind of motion. ";
        if(not self.activated)
            "You also sense that something is missing - a star on the end,
            perhaps?  ";
    }
    mycolor = transindection_color(myhome.analevel)
    location = nil // appears when white button is pressed 

    myhome = White_Plaque_Room_2
    myhome2 = [Plaque_Room_2, Blue_Plaque_Room_2, Green_Plaque_Room_2]
    verIoAttachTo(actor) = {}
    ioAttachTo(actor,dobj) = {
        if(not (dobj = eight_pointed_star))
             "You can't attach that to a rod. ";
        else {
            "The star fits perfectly.  In fact, it's such a snug fit that
            it won't come off again. ";   
            self.activated := true;
            dobj.moveInto(nil);
            addword(self,&adjective,'star');
            addword(self,&adjective,'eight-pointed');
            addword(self,&adjective,'shiny');
        } 
    }
    verIoPutOn(actor) = {}
    ioPutOn(actor,dobj) = {self.ioAttachTo(actor,dobj);}
    verDoRub(actor) = {}
    doRub(actor) = {
        "As you rub the rod, the ";
        switch (self.myhome) {
            case Plaque_Room_2:
                self.myhome := Blue_Plaque_Room_2;
                break;
            case Blue_Plaque_Room_2:
                self.myhome := Green_Plaque_Room_2;
                break;
            case Green_Plaque_Room_2:
                self.myhome := White_Plaque_Room_2;
                break;
            case White_Plaque_Room_2:
                self.myhome := Plaque_Room_2;
                break;
            default:
        } 
        "<<self.mycolor>> section of the rod glows brightly for a few 
        seconds. ";
    }
;
eight_pointed_star: initmess
    game701p = true
    sdesc = "eight-pointed star"
    adesc = "an eight-pointed star"
    initdesc = "A shiny eight-pointed star is lying in the debris. "
    ldesc = "It's a shiny eight-pointed star.  It looks as if it is designed
    to be attached to something. "
    noun = 'star' // to avoid disambiguation problems, 'star' removed from
                  // noun list of rods
    adjective = 'eight-pointed' 'shiny' 'metallic'
    location = nil
    fromloc = Debris
;

memo: item,readable 
    // To explain why the rod and the machine have swapped
    // Transindection levels.
    sdesc = "memo"
    ldesc = {
        self.isread := true;
        "It reads: \n
        \"DISAPPEARANCE OF MEDALLION-CHARGING MACHINE ";
        P();
        "An enquiry is to be held following the incident last week involving
        the apparent loss of the \"medallion-charging\" machine.  Past
        experience suggests that the device may have translocated itself to 
        another placement; if so, it may be found in due time.  The occurrence
        was caused by an unpermissioned attempt to charge a crude replica
        medallion.  The grey wand which appeared in place of the machine
        is to be left in situ, and the Travertine Room will remain 
        closed to visitors until the finale of the investigation.\" 
        No doubt a junior employee will take the blame for the failings of
        his (or her) superiors ... ";
    }
    noun = 'memo'
    location = Blue_Bird_Chamber
;    

elevator_key: keyItem, readable
    game701p = true
    sdesc = "small shiny key"
    ldesc= {
        "It's a small, shiny key.  A label attached to the key reads
         \"elevator\". ";
    }
    noun = 'key'
    adjective = 'shiny' 'small'
    location = Blue_Debris_Room
;

copper_bracelet: clothingItem, readable
    game701p = true
    sdesc = "copper bracelet"
    ldesc = "It's a plain copper bracelet.
        Due to its battered condition, I don't think it would be
        valuable. "
    noun = 'bracelet'
    adjective = 'copper' 'battered'
    location = nil // found by searching rocks in White-level Octagonal Room
;

platinum_bars: CCR_treasure_item
    game701p = true
    bonustreasure = true // added to treasure list when found
    sdesc = "platinum bars"
    adesc = "platinum bars"
    isThem = true
    ldesc = "I didn't think they belonged in this game, but they must be
        worth an absolute fortune. "
    noun = 'bars' 'platinum'
    adjective = 'platinum'
    location = White_Plaque_Room
;
