#include <string.h>
#include <time.h>

#include <windows.h>
#include "wintads.h"
#include "winio.h"
#include "global.h"
#include "htads\os.h"
#include "htads\run.h"
#include "htads\trd.h"

extern VOID TweakMenus(VOID);

DWORD os_main_shell(LPDWORD pdw);

void cleanup_stuff(void)
{
	reset_screen();			// Print an exit message
	fThreadRunning = FALSE;	// No more threads

	// We stop the presses if the user prefers, if we're a bound game, or if
	// the player signalled a break using the close button.
	if (validBinding || prefsTADS.fCloseOnEnd || fBreakSignalled)
		PostMessage(hwndFrame, WM_CLOSE, 0, 0);
	else {					// Otherwise, reset the text engine
		xtext_end();
		xtext_init();
		mainwin_caret_changed(FALSE);
		MyShowScrollBar(FALSE);
		ClearStatus();
	  	xtext_resize(0, 0, (short)(clientRect.right -
	  		clientRect.left), (short)lClientHeight);
		InvalidateRect(hwndClient, NULL, FALSE);
		TweakMenus();
	}
	SetWindowText(hwndFrame, "WinTADS");
}

DWORD os_main_shell(LPDWORD pdw)
{
	#define NUMOPTIONS (4)
	char *argv[NUMOPTIONS] = {NULL, NULL, NULL, NULL}; /* "-tp" for stress-testing */
	char buf[2048], *p;

	argv[0] = pszExecutable;
	argv[1] = "-m700000";	/* Size of the cache */
	argv[2] = prefsTADS.doublespace ? "-double+" : "-double-";

	/* If we don't have a zcodename, then assume the game is bound to the executable */
	if (zcodename[0] != 0) {
		/* We need to see if zcodename is actually a .gam file. I'll assume that,
		   if the file doesn't end in .gam, it's a saved game file. */
		p = zcodename + strlen(zcodename) - 1;
		while (*p != '.' && p != zcodename)
			p--;
		/* If there's no extension, assume it's still a .gam file */
		if (*p == '.') {
			if (stricmp(p, ".gam") != 0) {
				sprintf(buf, "-r%s", zcodename);
				argv[NUMOPTIONS-1] = buf;
			}
			else argv[NUMOPTIONS-1] = zcodename;
		}
		else argv[NUMOPTIONS-1] = zcodename;

		/* Also, while we're at it, set the default saved game directory
		   to be the same as the zcodename's directory */
		strcpy(prefsTADS.szSavePath, zcodename);
		p = os_get_root_name(prefsTADS.szSavePath);
		*p = '\0';
	}
		
	fThreadRunning = TRUE;	// Let people know we're going
	TweakMenus();			// Get the menus set up
	initialize_screen();
	SetStatusTextRight("0/0");

	if (zcodename[0] != 0)
		trdmain(NUMOPTIONS, argv, NULL, "");
	else trdmain(NUMOPTIONS - 1, argv, NULL, "");

	cleanup_stuff();

	return 0;
}

void os_term(int status)
{
	cleanup_stuff();
	ExitThread(status);
}

/* Do a printf() to the output buffer */
void os_printf(const char *fmt, ...)
{
    va_list argptr;

	va_start(argptr, fmt);
	os_vprintf(fmt, argptr);
	va_end(argptr);
}

void os_vprintf(const char *fmt, va_list argptr)
{
	char	buf[512];
	
	vsprintf(buf, fmt, argptr);
	display_string(buf);
}

/* Flush output. */
void os_flush()
{
    /*fflush( stdout ); No need under OS/2 */
}

/* Like gets(): return a string with no terminal newline. */
unsigned char *os_gets(unsigned char *buf, size_t buflen)
{
	int readpos, ret;
	
	runstat(0); /* Pretend this was called from the TADS engine before os_gets() */
	
	readpos = 0;
	ret = input_line(buflen-1, buf, 0, &readpos);
	if (ret == -1)
		return NULL;
	buf[readpos] = '\0';
    return buf;
}

/* Return the character which will be sucked from the output stream to indicate
	boldface on/off. 2 means on, 1 off. The HILIGHT_ON/OFF macros are defined
	in WinIo.h */
int os_hilite(int flag)
{
	if (flag == 2)
		return HILIGHT_ON;
	else
		return HILIGHT_OFF;
}

void os_clear_screen()
{
	clear_text_window();
}

/* Open an existing file or, if the file doesn't exist, create a
    new one. */
osfildef *osfoprwb(const char *fname, os_filetype_t typ)
{
    osfildef *fp;

    /* try opening an existing file in read/write mode */
    fp = fopen(fname, "r+b");

    /* if that failed, create a new file in read/write mode */
    if (fp == 0)
        fp = fopen(fname, "w+b");

    /* return the file */
    return fp;
}


/* Put a random number into seed. It should be as random as possible, because it's
	used as a seed for TADS's RNG in non-deterministic mode. */
void os_rand(long *seed)
{
	time_t val = time(NULL);
	*seed = (long)val;
}

void os_defext(char *buf, const char *ext)
{
    char *p, *n, tmp[1024];
    char *defpath;

    /*
     * Prepend default path
     */
   if (!stricmp(ext, "sav"))
	defpath = getenv("TADSSAVE");
   else if (!stricmp(ext, "gam"))
	defpath = getenv("TADSGAME");
   else
        defpath = NULL;

   if (defpath) {
   	/*
	 * Look for slashes.  If there are any, don't mess with name.
	 */
	n = buf;
	while (*n) {
		if (*n == '/')
			break;
		n++;
	}
	if (!*n) {
		strcpy(tmp, defpath);
		if (defpath[strlen(defpath)] != '/')
			strcat(tmp, "/");
		strcat(tmp, buf);
		strcpy(buf, tmp);
	}  
   }

   p = buf+strlen(buf);
   while ( p>buf )
     {
       p--;
       if ( *p=='.' ) return;      /* already has an extension */
       if ( *p=='/' || *p=='\\' || *p==':'
          ) break;    /* found a path */
     }
   strcat( buf, "." );              /* add a dot */
   strcat( buf, ext );              /* add the extension */
}

/*
 *   Add an extension, even if the filename currently has one 
 */
void os_addext(char *fn, const char *ext)
{
    strcat(fn, ".");
    strcat(fn, ext);
}

/* 
 *   os_remext(fn) removes the extension from fn, if present.  The buffer
 *   at fn should be modified in place.  If no extension is present, no
 *   action should be taken.  For systems without an analogue of
 *   extensions, this routine should do nothing.  
 */
void os_remext(char *fn)
{
    char *p = fn+strlen(fn);
    while ( p>fn )
    {
        p--;
        if ( *p=='.' )
        {
            *p = '\0';
            return;
        }
        if ( *p=='/' || *p=='\\' || *p==':') return;
    }
}

/*
 *   Get a pointer to the root name portion of a filename.
 */
char *os_get_root_name(char *buf)
{
    char *rootname;

    /* scan the name for path separators */
    for (rootname = buf ; *buf != '\0' ; ++buf)
    {
        switch(*buf)
        {
        case '/':
        case '\\':
        case ':':
            /* 
             *   These are all path separators - for now, assume the root
             *   will start at the next character after this separator.
             *   If we find another separator later, we'll forget about
             *   this one and use the later one instead. 
             */
            rootname = buf + 1;
            break;

        default:
            /* other characters are not path separators, so ignore them */
            break;
        }
    }

    /* return the most recent root name candidate */
    return rootname;
}

/* Opens the given .exe file and seeks to the end of the executable part of
	it, on the assumption that a datafile is to be found there. */
osfildef *os_exeseek(const char *exefile, const char *typ)
{
    FILE     *fp;
    unsigned  long seekpt;
    unsigned  long check;
    unsigned  long startofs;
    char      typbuf[4];

    /* If we're interested in seeing if the executable has a data file
       appended to it, well, TADS/2 has already taken care of that! */
    if (!(strcmp(exefile, pszExecutable) || memcmp(typ, "TGAM", 4))) {
    	if (!validBinding) return ((FILE *)0);
    	if (!(fp = fopen(exefile, "rb"))) return ((FILE *)0);
		fseek(fp, zcodepos, SEEK_SET);
		fread(&check, sizeof(check), 1, fp);
        fread(typbuf, sizeof(typbuf), 1, fp);
        return fp; 
    }

    /* open the file and seek to the very end */
    if (!(fp = fopen(exefile, "rb"))) return((FILE *)0);
    fseek(fp, 0L, SEEK_END);

    /* look through tagged blocks until we find the type we want */
    for (;;)
    {
        /* seek back past the descriptor block */
        fseek(fp, -12L, SEEK_CUR);
        seekpt = ftell(fp);

        /* read the elements of the descriptor block */
        if (fread(&check, sizeof(check), 1, fp) != 1
            || fread(typbuf, sizeof(typbuf), 1, fp) != 1
            || fread(&startofs, sizeof(startofs), 1, fp) != 1)
            break;

        /* check the signature to make sure we're looking at a valid block */
        if (check != ~seekpt)
            break;

        /* seek to the start of the data for this resource */
        fseek(fp, startofs, SEEK_SET);

        /* if this is the one we want, return it, otherwise keep looking */
        if (!memcmp(typ, typbuf, sizeof(typbuf)))
        {
            /* check the header to make sure it matches */
            if (fread(&check, sizeof(check), 1, fp) != 1
                || fread(typbuf, sizeof(typbuf), 1, fp) != 1
                || check != ~startofs
                || memcmp(typ, typbuf, sizeof(typbuf)))
                break;
            return fp;
        }
    }

    /* we didn't find anything - close the file and return failure */
    fclose(fp);
    return((FILE *)0);
}

/* Get a string resource from, well, the resource file */
int os_get_str_rsc(int id, char *buf, size_t buflen)
{
	// All of the string resources are located in the string
	// table, beginning with strix_TADSRes and going in order
	LoadString(hInst, strix_TADSRes + id - 1, buf, buflen);
	return 0;
}

/* Pause and wait for a keystroke. This is called between the display of a startup
	error and shutdown. */
void os_expause()
{
	reset_screen();
}

/* Pause and wait for a keystroke. This is called between normal game termination and
	shutdown. */
void os_waitc()
{
	reset_screen();
}

/* Wait for a character and return it. os_getc() is only called when a game
   calls inputevent() or inputkey(); with the advent of TADS 2.2.7, those
   functions can now handle extended keys like the arrow or function keys.
   Normally WinTADS doesn't pass those along to the input stream; while
   we're waiting for a single keystroke, we set a global flag to tell WinTADS
   to pass along extended keys. */
int os_getc()
{
	int ret_val;

	fWantsExtendedKeys = TRUE;
	ret_val = input_character(0);
	fWantsExtendedKeys = FALSE;
	return ret_val;
}

/* For os_getc_raw(), we don't want the CMD_xxxx codes */
int os_getc_raw(void)
{
	int ret_val;

	ret_val = input_character(0);
	return ret_val;
}

/* This takes two arguments, an osfildef* and a length, and tries to suck an external code
   resource from the osfildef*. Return NULL for error. */
int (*os_exfld(osfildef *fp, unsigned len))(void *)
{
	int (*result)(void); /* pointer to memory containing code */
	
	result = NULL;
	return result;
}

/* This takes one argument, a filename, and returns a pointer to a function
   int func(void). It is used for external code resources. Return NULL for error. */
int (*os_exfil(const char *filenm))(void)
{
	int (*result)(void); /* pointer to memory containing code */
	
	result = NULL;
	return result;
}

/* Call the function. Return nonzero for error. */
int os_excall(int (*funcptr)(void *), void *arg)
{
	return -1;
}

/* Check for any event that the user might send as a "break" signal. */
int os_break()
{
	if (fBreakSignalled)
		return 1;
	return 0;
}

/* Set the cursor to or from a wait image. */
void os_csr_busy(int val)
{
	if (val)
		fWaitCursor = TRUE;
	else
		fWaitCursor = FALSE;
}

/* Set a file to a given type. */
void os_settype(const char *filenm, int typeval)
{
	/* do nothing */
}

/* Check to see if a save file was specified for restoring at startup. If so,
	put its name in filbuf and return TRUE. Note: in TADS/2 this is
	unnecessary, as the front-end takes care of things */
int os_paramfile(char *filbuf)
{
	return 0;
}

/* Prompt the user for a filename, and store it in buf. Return nonzero for error. */
int os_askfile(const char *prompt, char *buf, int bufsiz, int prompt_type,
			   os_filetype_t file_type)
{
	SHORT type = 0; /* 0: saved game; 1: .gam; 2: transcript */
	BOOL  fWriting;
	SHORT fReturn;
	PCHAR p;
	
	if (prompt_type == OS_AFP_SAVE)
		fWriting = TRUE;
	else fWriting = FALSE;
	if (file_type == OSFTSAVE)
		type = 0;
	else if (file_type == OSFTLOG || file_type == OSFTTEXT)
		type = 2;
	
	{
		SHORT QueryFile(CHAR *buf, SHORT bufsize, BOOL fWriting, SHORT iType, const CHAR *szPrompt);
		fReturn = QueryFile(buf, (SHORT)bufsiz, fWriting, type, prompt);
	}
	if (!fReturn) {
		p = buf;			// Nasty hack: Change '\\' to '/' to
		while (*p != 0) {	//  avoid '\t' '\n' problems
			if (*p == '\\') {
				*p = '/';
			}
			p++;
		}
	}
	return fReturn;
}

/* Set right side of status line to "numer/denom" */
void os_score(int numer, int denom)
{
	char buf[64];

	if (denom < 0)
		strcpy(buf, "");
	else
		sprintf(buf, "%d/%d", numer, denom);
	os_strsc(buf);
}

/* Set right side of status line to str. */
void os_strsc(const char *str)
{
	SetStatusTextRight(str);
}

/* 0: story window. 1: status line. 2: Ldesc mode? */
void os_status(int flag)
{
	select_status_mode(flag);
}

int os_get_status(void)
{
	return get_status_mode();
}

/*
 *   Get the temporary file path.  This should fill in the buffer with a
 *   path prefix (suitable for strcat'ing a filename onto) for a good
 *   directory for a temporary file, such as the swap file.
 */
/* Because I'm lazy, this function will automatically dump the temp files
   into the program's home directory. */
void os_get_tmp_path(char *s)
{
/*  int i;

  strcpy(s, tmpnam(NULL));
  for (i = strlen(s) - 1; i >= 0; i--)
    if (s[i] == '/' || s[i] == '\\')
      break;

  s[i + 1] = 0;*/
  strcpy(s, pszHomePath);
}

// Returns 0 if a file already exists
int osfacc(const char *fname)
{
	HANDLE hFile;
	
	hFile = CreateFile(fname, GENERIC_READ, FILE_SHARE_READ, (LPSECURITY_ATTRIBUTES)NULL,
		OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
	if (hFile == INVALID_HANDLE_VALUE) return 1;
	CloseHandle(hFile);
	return 0;
}

// We don't have a [MORE] prompt for TADS, since the display window
// handles all of that for us.
void os_more_prompt(void) {}

// The mouse functions won't do anything under WinTADS
void os_moushow(void) {}
void os_mouhide(void) {}

// Set up the time zone
void os_tzset(void)
{
	tzset();
}

// Translate HTML 4 codes
void os_xlat_html4(unsigned int html4_char,
                   char *result, size_t result_size)
{
    /* presume we're going to return a single byte */
    result[1] = '\0';

    /* check the character to see what range it's in */
    if (html4_char < 128)
    {
        /* 
         *   it's in the ASCII range, so assume that it's part of the
         *   ASCII subset of every Windows character set -- return the
         *   value unchanged and in the current default character set 
         */
		result[0] = (char)html4_char;
	}
	else if (html4_char >= 160 && html4_char <= 255)
	{
        /*
         *   It's in the 160-255 range, which is the ISO Latin-1 range for
         *   HTML 4.  These map trivially to Windows code page 1252
         *   characters, so leave the value unchanged and use the ANSI
         *   character set.  
         */
		result[0] = (char)html4_char;
        return;
    }
    else
    {
        static struct
        {
            unsigned int html_val;
            unsigned int win_val;
            int          win_charset;
        } mapping[] =
        {
            /*
             *   IMPORTANT: This table must be sorted by html_val, because
             *   we perform a binary search on this key.  
             */

            /* 
             *   Latin-2 characters and additional alphabetic characters.
             *   Most of these characters exist only in the Windows
             *   Eastern European code page, but a few are in the common
             *   Windows extended code page set (code points 128-159) and
             *   a few are only in the ANSI code page.  
             */
            { 258, 0xC3, HTML4_EASTEUROPE },                    /* Abreve */
            { 259, 0xE3, HTML4_EASTEUROPE },                    /* abreve */
            { 260, 0xA5, HTML4_EASTEUROPE },                     /* Aogon */
            { 261, 0xB9, HTML4_EASTEUROPE },                     /* aogon */
            { 262, 0xC6, HTML4_EASTEUROPE },                    /* Cacute */
            { 263, 0xE6, HTML4_EASTEUROPE },                    /* cacute */
            { 268, 0xC8, HTML4_EASTEUROPE },                    /* Ccaron */
            { 269, 0xE8, HTML4_EASTEUROPE },                    /* ccaron */
            { 270, 0xCF, HTML4_EASTEUROPE },                    /* Dcaron */
            { 271, 0xEF, HTML4_EASTEUROPE },                    /* dcaron */
            { 272, 0xD0, HTML4_EASTEUROPE },                    /* Dstrok */
            { 273, 0xF0, HTML4_EASTEUROPE },                    /* dstrok */
            { 280, 0xCA, HTML4_EASTEUROPE },                     /* Eogon */
            { 281, 0xEA, HTML4_EASTEUROPE },                     /* eogon */
            { 282, 0xCC, HTML4_EASTEUROPE },                    /* Ecaron */
            { 283, 0xEC, HTML4_EASTEUROPE },                    /* ecaron */
            { 313, 0xC5, HTML4_EASTEUROPE },                    /* Lacute */
            { 314, 0xE5, HTML4_EASTEUROPE },                    /* lacute */
            { 317, 0xBC, HTML4_EASTEUROPE },                    /* Lcaron */
            { 318, 0xBE, HTML4_EASTEUROPE },                    /* lcaron */
            { 321, 0xA3, HTML4_EASTEUROPE },                    /* Lstrok */
            { 322, 0xB3, HTML4_EASTEUROPE },                    /* lstrok */
            { 323, 0xD1, HTML4_EASTEUROPE },                    /* Nacute */
            { 324, 0xF1, HTML4_EASTEUROPE },                    /* nacute */
            { 327, 0xD2, HTML4_EASTEUROPE },                    /* Ncaron */
            { 328, 0xF2, HTML4_EASTEUROPE },                    /* ncaron */
            { 336, 0xF5, HTML4_EASTEUROPE },                    /* odblac */
            { 337, 0xD5, HTML4_EASTEUROPE },                    /* Odblac */
            { 338,  140, HTML4_ANSI },                           /* OElig */
            { 339,  156, HTML4_ANSI },                           /* oelig */
            { 340, 0xC0, HTML4_EASTEUROPE },                    /* Racute */
            { 341, 0xE0, HTML4_EASTEUROPE },                    /* racute */
            { 344, 0xD8, HTML4_EASTEUROPE },                    /* Rcaron */
            { 345, 0xF8, HTML4_EASTEUROPE },                    /* rcaron */
            { 346, 0x8C, HTML4_EASTEUROPE },                    /* Sacute */
            { 347, 0x9C, HTML4_EASTEUROPE },                    /* sacute */
            { 350, 0xAA, HTML4_EASTEUROPE },                    /* Scedil */
            { 351, 0xBA, HTML4_EASTEUROPE },                    /* scedil */
            { 352,  138, HTML4_DEFAULT },                       /* Scaron */
            { 353,  154, HTML4_DEFAULT },                       /* scaron */
            { 354, 0xDE, HTML4_EASTEUROPE },                    /* Tcedil */
            { 355, 0xFE, HTML4_EASTEUROPE },                    /* tcedil */
            { 356, 0x8D, HTML4_EASTEUROPE },                    /* Tcaron */
            { 357, 0x9D, HTML4_EASTEUROPE },                    /* tcaron */
            { 366, 0xD9, HTML4_EASTEUROPE },                     /* Uring */
            { 367, 0xF9, HTML4_EASTEUROPE },                     /* uring */
            { 368, 0xDB, HTML4_EASTEUROPE },                    /* Udblac */
            { 369, 0xFB, HTML4_EASTEUROPE },                    /* udblac */
            { 376,  159, HTML4_ANSI },                            /* Yuml */
            { 377, 0x8F, HTML4_EASTEUROPE },                    /* Zacute */
            { 378, 0x9F, HTML4_EASTEUROPE },                    /* zacute */
            { 379, 0xAF, HTML4_EASTEUROPE },                      /* Zdot */
            { 380, 0xBF, HTML4_EASTEUROPE },                      /* zdot */
            { 381, 0x8E, HTML4_EASTEUROPE },                    /* Zcaron */
            { 382, 0x9E, HTML4_EASTEUROPE },                    /* zcaron */

            /* Latin-extended B */
            { 402, 166, HTML4_SYMBOL },                           /* fnof */

            /* Latin-2 accents */
            { 710, 136, HTML4_ANSI },                             /* circ */
            { 711, 0xA1, HTML4_EASTEUROPE },                     /* caron */
            { 728, 0xA2, HTML4_EASTEUROPE },                     /* breve */
            { 729, 0xFF, HTML4_EASTEUROPE },                       /* dot */
            { 731, 0xB2, HTML4_EASTEUROPE },                      /* ogon */
            { 732, 152, HTML4_ANSI },                            /* tilde */
            { 733, 0xBD, HTML4_EASTEUROPE },                     /* dblac */

            /* Greek letters */
            { 913, 'A', HTML4_SYMBOL },                          /* Alpha */
            { 914, 'B', HTML4_SYMBOL },                           /* Beta */
            { 915, 'G', HTML4_SYMBOL },                          /* Gamma */
            { 916, 'D', HTML4_SYMBOL },                          /* Delta */
            { 917, 'E', HTML4_SYMBOL },                        /* Epsilon */
            { 918, 'Z', HTML4_SYMBOL },                           /* Zeta */
            { 919, 'H', HTML4_SYMBOL },                            /* Eta */
            { 920, 'Q', HTML4_SYMBOL },                          /* Theta */
            { 921, 'I', HTML4_SYMBOL },                           /* Iota */
            { 922, 'K', HTML4_SYMBOL },                          /* Kappa */
            { 923, 'L', HTML4_SYMBOL },                         /* Lambda */
            { 924, 'M', HTML4_SYMBOL },                             /* Mu */
            { 925, 'N', HTML4_SYMBOL },                             /* Nu */
            { 926, 'X', HTML4_SYMBOL },                             /* Xi */
            { 927, 'O', HTML4_SYMBOL },                        /* Omicron */
            { 928, 'P', HTML4_SYMBOL },                             /* Pi */
            { 929, 'R', HTML4_SYMBOL },                            /* Rho */
            { 931, 'S', HTML4_SYMBOL },                          /* Sigma */
            { 932, 'T', HTML4_SYMBOL },                            /* Tau */
            { 933, 'U', HTML4_SYMBOL },                        /* Upsilon */
            { 934, 'F', HTML4_SYMBOL },                            /* Phi */
            { 935, 'C', HTML4_SYMBOL },                            /* Chi */
            { 936, 'Y', HTML4_SYMBOL },                            /* Psi */
            { 937, 'W', HTML4_SYMBOL },                          /* Omega */
            { 945, 'a', HTML4_SYMBOL },                          /* alpha */
            { 946, 'b', HTML4_SYMBOL },                           /* beta */
            { 947, 'g', HTML4_SYMBOL },                          /* gamma */
            { 948, 'd', HTML4_SYMBOL },                          /* delta */
            { 949, 'e', HTML4_SYMBOL },                        /* epsilon */
            { 950, 'z', HTML4_SYMBOL },                           /* zeta */
            { 951, 'h', HTML4_SYMBOL },                            /* eta */
            { 952, 'q', HTML4_SYMBOL },                          /* theta */
            { 953, 'i', HTML4_SYMBOL },                           /* iota */
            { 954, 'k', HTML4_SYMBOL },                          /* kappa */
            { 955, 'l', HTML4_SYMBOL },                         /* lambda */
            { 956, 'm', HTML4_SYMBOL },                             /* mu */
            { 957, 'n', HTML4_SYMBOL },                             /* nu */
            { 958, 'x', HTML4_SYMBOL },                             /* xi */
            { 959, 'o', HTML4_SYMBOL },                        /* omicron */
            { 960, 'p', HTML4_SYMBOL },                             /* pi */
            { 961, 'r', HTML4_SYMBOL },                            /* rho */
            { 962, 'V', HTML4_SYMBOL },                         /* sigmaf */
            { 963, 's', HTML4_SYMBOL },                          /* sigma */
            { 964, 't', HTML4_SYMBOL },                            /* tau */
            { 965, 'u', HTML4_SYMBOL },                        /* upsilon */
            { 966, 'f', HTML4_SYMBOL },                            /* phi */
            { 967, 'c', HTML4_SYMBOL },                            /* chi */
            { 968, 'y', HTML4_SYMBOL },                            /* psi */
            { 969, 'w', HTML4_SYMBOL },                          /* omega */
            { 977, 'J', HTML4_SYMBOL },                       /* thetasym */
            { 978, 161, HTML4_SYMBOL },                          /* upsih */
            { 982, 'v', HTML4_SYMBOL },                            /* piv */

            /* additional punctuation */
            { 8211, 150, HTML4_DEFAULT },                        /* ndash */
            { 8212, 151, HTML4_DEFAULT },                        /* mdash */
            { 8216, 145, HTML4_DEFAULT },                        /* lsquo */
            { 8217, 146, HTML4_DEFAULT },                        /* rsquo */
            { 8218, 130, HTML4_DEFAULT },                        /* sbquo */
            { 8220, 147, HTML4_DEFAULT },                        /* ldquo */
            { 8221, 148, HTML4_DEFAULT },                        /* rdquo */
            { 8222, 132, HTML4_DEFAULT },                        /* bdquo */
            { 8224, 134, HTML4_DEFAULT },                       /* dagger */
            { 8225, 135, HTML4_DEFAULT },                       /* Dagger */
            { 8226, 149, HTML4_DEFAULT },                         /* bull */
            { 8230, 133, HTML4_DEFAULT },                       /* hellip */
            { 8240, 137, HTML4_DEFAULT },                       /* permil */
            { 8242, 162, HTML4_SYMBOL },                         /* prime */
            { 8243, 178, HTML4_SYMBOL },                         /* Prime */
            { 8249, 139, HTML4_DEFAULT },                       /* lsaquo */
            { 8250, 155, HTML4_DEFAULT },                       /* rsaquo */
            { 8254, '`', HTML4_SYMBOL },                         /* oline */
            { 8260, 164, HTML4_SYMBOL },                         /* frasl */

            /* letter-like symbols */
            { 8465, 193, HTML4_SYMBOL },                         /* image */
            { 8472, 195, HTML4_SYMBOL },                        /* weierp */
            { 8476, 194, HTML4_SYMBOL },                          /* real */
            { 8482, 153, HTML4_DEFAULT },                        /* trade */
            { 8501, 192, HTML4_SYMBOL },                       /* alefsym */
            
            /* arrows */
            { 8592, 172, HTML4_SYMBOL },                          /* larr */
            { 8593, 173, HTML4_SYMBOL },                          /* uarr */
            { 8594, 174, HTML4_SYMBOL },                          /* rarr */
            { 8595, 175, HTML4_SYMBOL },                          /* darr */
            { 8596, 171, HTML4_SYMBOL },                          /* harr */
            { 8629, 191, HTML4_SYMBOL },                         /* crarr */
            { 8656, 220, HTML4_SYMBOL },                          /* lArr */
            { 8657, 221, HTML4_SYMBOL },                          /* uArr */
            { 8658, 222, HTML4_SYMBOL },                          /* rArr */
            { 8659, 223, HTML4_SYMBOL },                          /* dArr */
            { 8660, 219, HTML4_SYMBOL },                          /* hArr */
            
            /* mathematical operators */
            { 8704, 34, HTML4_SYMBOL },                         /* forall */
            { 8706, 182, HTML4_SYMBOL },                          /* part */
            { 8707, '$', HTML4_SYMBOL },                         /* exist */
            { 8709, 198, HTML4_SYMBOL },                         /* empty */
            { 8711, 209, HTML4_SYMBOL },                         /* nabla */
            { 8712, 206, HTML4_SYMBOL },                          /* isin */
            { 8713, 207, HTML4_SYMBOL },                         /* notin */
            { 8715, '\'', HTML4_SYMBOL },                           /* ni */
            { 8719, 213, HTML4_SYMBOL },                          /* prod */
            { 8721, 229, HTML4_SYMBOL },                           /* sum */
            { 8722, '-', HTML4_SYMBOL },                         /* minus */
            { 8727, '*', HTML4_SYMBOL },                        /* lowast */
            { 8730, 214, HTML4_SYMBOL },                         /* radic */
            { 8733, 181, HTML4_SYMBOL },                          /* prop */
            { 8734, 165, HTML4_SYMBOL },                         /* infin */
            { 8736, 208, HTML4_SYMBOL },                           /* ang */
            { 8743, 217, HTML4_SYMBOL },                           /* and */
            { 8744, 218, HTML4_SYMBOL },                            /* or */
            { 8745, 199, HTML4_SYMBOL },                           /* cap */
            { 8746, 200, HTML4_SYMBOL },                           /* cup */
            { 8747, 242, HTML4_SYMBOL },                           /* int */
            { 8756, '\\', HTML4_SYMBOL },                       /* there4 */
            { 8764, '~', HTML4_SYMBOL },                           /* sim */
            { 8773, '@', HTML4_SYMBOL },                          /* cong */
            { 8776, 187, HTML4_SYMBOL },                         /* asymp */
            { 8800, 185, HTML4_SYMBOL },                            /* ne */
            { 8801, 186, HTML4_SYMBOL },                         /* equiv */
            { 8804, 163, HTML4_SYMBOL },                            /* le */
            { 8805, 179, HTML4_SYMBOL },                            /* ge */
            { 8834, 204, HTML4_SYMBOL },                           /* sub */
            { 8835, 201, HTML4_SYMBOL },                           /* sup */
            { 8836, 203, HTML4_SYMBOL },                          /* nsub */
            { 8838, 205, HTML4_SYMBOL },                          /* sube */
            { 8839, 202, HTML4_SYMBOL },                          /* supe */
            { 8853, 197, HTML4_SYMBOL },                         /* oplus */
            { 8855, 196, HTML4_SYMBOL },                        /* otimes */
            { 8869, '^', HTML4_SYMBOL },                          /* perp */
            { 8901, 215, HTML4_SYMBOL },                          /* sdot */
            { 8968, 233, HTML4_SYMBOL },                         /* lceil */
            { 8969, 249, HTML4_SYMBOL },                         /* rceil */
            { 8970, 235, HTML4_SYMBOL },                        /* lfloor */
            { 8971, 251, HTML4_SYMBOL },                        /* rfloor */
            { 9001, 225, HTML4_SYMBOL },                          /* lang */
            { 9002, 241, HTML4_SYMBOL },                          /* rang */

            /* geometric shapes */
            { 9674, 224, HTML4_SYMBOL },                           /* loz */

            /* misceallaneous symbols */
            { 9824, 170, HTML4_SYMBOL },                        /* spades */
            { 9827, 167, HTML4_SYMBOL },                         /* clubs */
            { 9829, 169, HTML4_SYMBOL },                        /* hearts */
            { 9830, 168, HTML4_SYMBOL }                       /* diamonds */
        };
        int hi, lo;

        /*
         *   Find the character in the mapping table 
         */
        lo = 0;
        hi = sizeof(mapping)/sizeof(mapping[0]) - 1;
        for (;;)
        {
            int cur;
            
            /* if there's nothing left in the table, we didn't find it */
            if (lo > hi)
            {
                /* use the invalid character in the default character set */
                result[0] = '*';
                break;
            }

            /* split the difference */
            cur = lo + (hi - lo)/2;

            /* is this the one we're looking for? */
            if (mapping[cur].html_val == html4_char)
            {
                /* this is it */
				result[0] = HTML4_CHARACTER;
				// Notice that I add 1 to the character set. This is because one
				// of the character sets is numbered '0', which the text engine
				// then mistakes for the end of the translation. Whups. When I
				// decode the charater set in display_char() (in WinIo.c) I
				// subtract one from it again
				result[1] = mapping[cur].win_charset + 1;
                result[2] = (char)mapping[cur].win_val;
				result[3] = 0;
                break;
            }
            else if (mapping[cur].html_val < html4_char)
            {
                /* we want something higher - look in the upper half */
                lo = (cur == lo ? lo + 1 : cur);
            }
            else
            {
                /* we want something smaller - look in the lower half */
                hi = (cur == hi ? hi - 1 : cur);
            }
        }
    }
}

// Tell what our capabilities are
int os_get_sysinfo(int code, void *param, long *result)
{
	switch(code) {
	case SYSINFO_HTML:
	case SYSINFO_JPEG:
	case SYSINFO_PNG:
	case SYSINFO_WAV:
	case SYSINFO_MIDI:
	case SYSINFO_WAV_MIDI_OVL:
	case SYSINFO_WAV_OVL:
	case SYSINFO_PREF_IMAGES:
	case SYSINFO_PREF_SOUNDS:
	case SYSINFO_PREF_MUSIC:
	case SYSINFO_PREF_LINKS:
	case SYSINFO_MPEG:
	case SYSINFO_MPEG1:
	case SYSINFO_MPEG2:
	case SYSINFO_MPEG3:
		/* 
		*   we don't support any of these features - set the result to 0
		*   to indicate this 
		*/
		*result = 0;

		/* return true to indicate that we recognized the code */
		return TRUE;

	default:
		/* we don't recognize other codes */
		return FALSE;
	}
}

/*
 *   Receive notification that a character mapping file has been loaded.
 *   WinTADS isn't going to do anything with this information.
 */
void os_advise_load_charmap(char *id, char *ldesc, char *sysinfo)
{
}

/*
 *   Generate a filename for a character mapping table.  On Windows, the
 *   filename is always simply "win" plus the internal ID plus ".tcp".  
 */
void os_gen_charmap_filename(char *filename, char *internal_id,
                             char *argv0)
{
    char *p;
    char *rootname;
    size_t pathlen;
    
    /* find the path prefix of the original executable filename */
    for (p = rootname = argv0 ; *p != '\0' ; ++p)
    {
        if (*p == '/' || *p == '\\' || *p == ':')
            rootname = p + 1;
    }

    /* copy the path prefix */
    pathlen = rootname - argv0;
    memcpy(filename, argv0, pathlen);

    /* if there's no trailing backslash, add one */
    if (pathlen == 0 || filename[pathlen - 1] != '\\')
        filename[pathlen++] = '\\';

    /* add "win_", plus the character set ID, plus the extension */
    strcpy(filename + pathlen, "win_");
    strcat(filename + pathlen, internal_id);
    strcat(filename + pathlen, ".tcp");
}

/*
 *   Get the current system high-precision timer.  This function returns a
 *   value giving the wall-clock ("real") time in milliseconds, relative
 *   to any arbitrary zero point.  It doesn't matter what this value is
 *   relative to -- the only important thing is that the values returned
 *   by two different calls should differ by the number of actual
 *   milliseconds that have elapsed between the two calls.  On most
 *   single-user systems, for example, this will probably return the
 *   number of milliseconds since the user turned on the computer.
 */
long os_get_sys_clock_ms(void)
{
    if (timeZero == 0)
        timeZero = GetTickCount();
    return (long)(GetTickCount() - timeZero);
}

/*
 *   Sleep for a while.  This should simply pause for the given number of
 *   milliseconds, then return.  Since the TADS engine runs in a separate
 *   thread, I can do the nasty thing and just loop for the required number
 *   of milliseconds before returning.
 */
void os_sleep_ms(long delay_in_milliseconds)
{
    Sleep((DWORD)delay_in_milliseconds);
}

/*
 *   Get an input event.  If use_timeout
 *   is false, this routine should simply wait until one of the events it
 *   recognizes occurs, then return the appropriate information on the
 *   event.  If use_timeout is true, this routine should return
 *   OS_EVT_TIMEOUT after the given number of milliseconds elapses if no
 *   event occurs first.
 *   
 *   This function is not obligated to obey the timeout.  If a timeout is
 *   specified and it is not possible to obey the timeout, the function
 *   should simply return OS_EVT_NOTIMEOUT.  The trivial implementation
 *   thus checks for a timeout, returns an error if specified, and
 *   otherwise simply waits for the user to press a key.  
 */
int os_get_event(unsigned long timeout_in_milliseconds, int use_timeout,
                 os_event_info_t *info)
{
	int	ret_val;

	fWantsExtendedKeys = TRUE;
	if (use_timeout) {
		ret_val = input_character((int)(timeout_in_milliseconds));
		if (ret_val == -1)		// We timed out
			return OS_EVT_TIMEOUT;
	}
	else {
		ret_val = input_character(0);
	}
	(*info).key[0] = ret_val;
	if (ret_val == 0) {
		ret_val = input_character(0);
		(*info).key[1] = ret_val;
	}
	fWantsExtendedKeys = FALSE;
	return OS_EVT_KEY;
}

/*
 * os_input_dialog() is so complex, I simply refer you to osifc.h, where
 * it is defined.
 */
int os_input_dialog(int icon_id, const char *prompt, int standard_button_set,
                    const char **buttons, int button_count,
                    int default_index, int cancel_index)
{
	UINT	flags = 0;
	int		retval;

	// The standard button set we can handle w/a MessageBox
	if (standard_button_set) {
		switch (standard_button_set) {
		case OS_INDLG_OK:
			flags = MB_OK;
			break;
		case OS_INDLG_OKCANCEL:
			flags = MB_OKCANCEL;
			break;
		case OS_INDLG_YESNO:
			flags = MB_YESNO;
			break;
		case OS_INDLG_YESNOCANCEL:
			flags = MB_YESNOCANCEL;
			break;
		default:
			// I don't recognize the button set, so return an err
			return 0;
		}
		switch (icon_id) {
		case OS_INDLG_ICON_WARNING:
			flags |= MB_ICONWARNING;
			break;
		case OS_INDLG_ICON_INFO:
			flags |= MB_ICONINFORMATION;
			break;
		case OS_INDLG_ICON_QUESTION:
			flags |= MB_ICONQUESTION;
			break;
		case OS_INDLG_ICON_ERROR:
			flags |= MB_ICONERROR;
			break;
		}
		switch (default_index) {
		case 1:
			flags |= MB_DEFBUTTON1;
			break;
		case 2:
			flags |= MB_DEFBUTTON2;
			break;
		case 3:
			flags |= MB_DEFBUTTON3;
			break;
		case 4:
			flags |= MB_DEFBUTTON4;
			break;
		}
		flags |= MB_SYSTEMMODAL;   // Specify the box's modality
		retval = MessageBox(GetFocus(), prompt, "WinTADS Message",
			flags);
		switch (standard_button_set) {
		case OS_INDLG_OKCANCEL:
			if (retval == IDCANCEL) return 2;
		case OS_INDLG_OK:
			return (retval == IDOK);
		case OS_INDLG_YESNOCANCEL:
			if (retval == IDCANCEL) return 3;
		case OS_INDLG_YESNO:
			if (retval == IDNO) return 2;
			return (retval == IDYES);
		}
	}
	// Otherwise, use a stdio-approach
	while (TRUE) {
		int i;
		char buf[256];
		const char *p, *cur;
		char *resp;
		int match_cnt, last_found;

		// If there's no button count, no response is possible
		if (button_count == 0)
			return 0;
		
		// Print a new line
		outformat("\\n");
		outformat((char *)prompt);
		outformat(" ");

		// Print the prompt
		for (i = 0; i < button_count; i++) {
			if (i != 0)
				outformat("/");
			cur = buttons[i];
			// Look for '&' shortcuts and, if found, put parens
			// around the letter in question
			for (p = cur; *p != '&' && *p != '\0'; p++);

			if (*p == '&') {
                // reformat the response string
                sprintf(buf, "%.*s(%c)%s", (int)(p - cur), cur, *(p+1), p+2);

                // show it
                outformat(buf);
            }
            else {
				display_string((char *)cur);
			}
		}
		// if we're in HTML mode, switch to input font
		if (tio_is_html_mode())
			outformat("<font face='TADS-Input'>");

		/* read the response */
		getstring(" >", buf, sizeof(buf));

		/* if we're in HTML mode, close the input font tag */
		if (tio_is_html_mode())
			outformat("</font>");

        /* skip any leading spaces in the reply */
		for (resp = buf; isspace(*resp); ++resp);

		/* if it's one character, check it against the shortcut keys */
		if (strlen(resp) == 1) {
			/* scan the responses */
			for (i = 0 ; i < button_count ; ++i) {
	            /* look for a '&' in this button */
				for (p = buttons[i] ; *p != '&' && *p != '\0' ; ++p) ;

				/* if we found the '&', check the shortcut */
				if (*p == '&' && toupper(*(p+1)) == toupper(*resp)) {
			        /*
					 *   this is the one - return the current index
					 *   (bumping it by one to get a 1-based value)
					 */
					return i + 1;
				}
			}
		}
		for (i = 0, match_cnt = 0 ; i < button_count ; ++i) {
			const char *p1;
			const char *p2;

			/*
			 *   compare this response to the user's response; skip any
			 *   '&' in the button label
			 */
			for (p1 = resp, p2 = buttons[i] ; *p1 != '\0' && *p2 != '\0' ;
				++p1, ++p2)
			{
				/* if this is a '&' in the button label, skip it */
				if (*p2 == '&')
					++p2;

				/* if these characters don't match, it's no match */
	            if (toupper(*p1) != toupper(*p2))
					break;
			}

	        /*
			 *   if we reached the end of the user's response, we have a
			 *   match in the leading substring - count it and remember
			 *   this as the last one, but keep looking, since we need to
			 *   make sure we don't have any other matches
			 */
	        if (*p1 == '\0')
			{
	            ++match_cnt;
				last_found = i;
			}
		}

		/*
		 *   if we found exactly one match, return it (adjusting to a
		 *   1-based index); if we found more or less than one match, it's
		 *   not a valid response, so start over with a new prompt
		 */
		if (match_cnt == 1)
	        return last_found + 1;
	}
}


/*
 *   Set the game title.  The output layer calls this routine when a game
 *   sets its title (via an HTML <title> tag, for example).  If it's
 *   convenient to do so, the OS layer can use this string to set a window
 *   caption, or whatever else makes sense on each system.  Most
 *   character-mode implementations will provide an empty implementation,
 *   since there's not usually any standard way to show the current
 *   application title on a character-mode display.  
 */
void os_set_title(const char *title)
{
	SetWindowText(hwndFrame, title);
}
