#charset "us-ascii"
#include "adv3.h"

/*

Tads-3 Sense Connector Filter

Steve Breslin, 2004-2005

====

Licence:

Everyone is free to use this, but please alert me to your findings, and
provide me with all improvements and modifications.

====

If your game makes extensive use of SenseConnectors, you should consider
using this library patch.

The main library places all game objects directly or indirectly
connected via SenseConnectors into the "containment table" -- a
(frequently quite large) data sturcture which is constructed and
processed several times per turn. The containment table can be much
wider than the objects in scope -- the objects that the player can
actually sense. Because of the extensive computation involved in
constructing the containment (and from that, the sense path), games that
have a lot of SenseConnectors can exhibit significant processing
slowdown, even on the most modern machine.

The purpose of this patch is to decrease the size of the containment
table by stopping SenseConnectors from adding their locations to the
table in certain cases. (Basically, we filter when the SenseConnector's
locations and their contents cannot possibly "shine light" on objects
which could possibly be sensed by the Point Of View.)

This is good because the smaller the containment table, the less work
later for the computation-heavy sense-path resolution.

There are two different implementations of this patch: the normal
default version and the "Light" version. The "Light" version is slightly
simpler than the normal default version. It is faster, but breaks with
the library slightly. If you have light passing through connectors to
illuminate otherwise darkened objects, that illumination will not work
under certain conditions. As a precaution, if parts of your game use
dimished light or darkness, you should use the full version of Sense
Connector Filter instead. (This only really matters if you're also using
Sense Connectors connecting to darkened locations.) If, however, all
your rooms are lit, or the dark rooms are not connected by
SenseConnectors to the rest of the game-map, the "Light" version will
produce faster runtime performance.

====

The algorithm is fairly simple:

Before SenseConnector.addDirectConnections() begins adding its
containers, it adds a transparency information object to the vector
libGlobal.transObjs. This object encapsulates the SenseConnector's
net transparency information for all senses. (Its net transparency is
the transparency of that connector filtered by the transparency of all
the connectors backchain.)

When SenseConnector.addDirectConnections finishes adding its
containers, it removes its transparency object from the transObjs
vector.

The method SenseConnector.addDirectConnections(tab) is responsible for
adding the connector's locations to the table, so that's what we
modify.

The normal version:

Before SenseConnector.addDirectConnections() decides to add its
containers, it checks each of its non-opaque senses (from
transSensingThru()) against the transparency information in the
transObjs vector. If, on comparison, it fails to find at least one
non-opaque sense, it does add its containers, but will not allow the
containers' further connections to add further containers.

The "light" version:

Before SenseConnector.addDirectConnections() decides to add its
containers, it checks each of its non-opaque senses (from
transSensingThru()) against the transparency information in the
transObjs vector. If, on comparison, it fails to find at least one
non-opaque sense, it does not adds its containers.

====

Notes:

This patch is still somewhat experimental. It is meant to work in all
situations, but if you find a situation which this module doesn't handle
properly, definitely please let me know!

For simplicity, this module only considers the basic senses defined by
the library: sight, smell, sound, and touch.

The library already relativizes the connection table to the POV. In the
library's version of the connection table, an object's presence in the
table does not mean that the table contains all objects possibly
sensible by this object. This is true with this module as well; we only
remark that our filtering mechanism does not stray from the library
model.

*/

/* use #define LIGHT to use the "Light" version of this patch. */

// #define LIGHT

#ifndef LIGHT

/* LIGHT is not defined: there follows the "normal default" patch. */

modify libGlobal
    /* transObjs keeps track of all the sense connectors in the current
     * chain.
     */
    transObjs = static new Vector(4)

    /* SCFiltering is a boolean flag, which we turn on when we've
     * reached a fully opaque sense connector.
     */
    SCFiltering = nil
;

modify SenseConnector
    addDirectConnections(tab)
    {
        /* add myself */
        tab[self] = true;
            
        /* add my contents */
        foreach (local cur in contents)
        {
            if (tab[cur] == nil)
                cur.addDirectConnections(tab);
        }

//--!! This is where the modification comes.

        local myInfoObject;

        /* If we're filtering Sense Connectors, that means we don't let
         * them add their further locations if their net transparency
         * for the current sense path is opaque for all senses.
         */
        if (libGlobal.SCFiltering)
            return;

        /* Create a new transInfoObject which encapsulates our
         * transparence information.
         *
         * At this point, the myInfoObject reflects our transparency of
         * each sense.
         */
        myInfoObject = new transObjInfo(self);

        /* If there is one or more sense-connector further up the call
         * chain, we calculate net the transparency of the last
         * (closest) one compared with the new one. (The last one
         * already encapsulates the net transparency of the entire
         * chain of connectors, just as the current one will by the end
         * of the following code block.)
         *
         * If the net transparency is opaque for all senses, we turn on
         * SCFiltering.
         */
        if (libGlobal.transObjs.length())
        {
            local lastTrans = libGlobal.transObjs[libGlobal.transObjs.length];

            /* We iterate through the senses which are not already
             * opaque, and update the myInfoObject.
             */
            foreach(local curSense in lastTrans.nonOpaques)
                if (myInfoObject.nonOpaques.indexOf(curSense))
                    myInfoObject.updateSense(curSense, lastTrans);

            /* At this point, myInfoObject reflects the net
             * transparency of all connectors in the chain.
             */

            /* If there is no non-opaque sense passable by this connector,
             * we turn filtering on.
             */
            if (myInfoObject.nonOpaques.length() == 0)
            {
                libGlobal.SCFiltering = true;
            }
        }
 
       /* We're going to add our locations. The call to
         * cur.addDirectConnections(tab) can wind up calling yet
         * another sense connector. So we'll record our
         * transparency information in the libGlobal.transObjs
         * vector.
         */
        libGlobal.transObjs.append(myInfoObject);

        /* The following process is what the current library does,
         * without our mechanism.
         */
        foreach (local cur in locationList)
        {
            if (tab[cur] == nil)
                cur.addDirectConnections(tab);
        }

        /* Now that we've added all our locations (which have added
         * everything in them including any sense connectors,
         * recursively), we must remove our transObjInfo from the
         * transObjs vector.
         * That way, we won't interfere in the sense connectors
         * which aren't being connected to the POV by us.
         */
        libGlobal.transObjs.removeElementAt(libGlobal.transObjs.length());

        /* Finally, we turn off SCFiltering, in case we turned it on.
         */
        libGlobal.SCFiltering = nil;
    }
;

#else

/* LIGHT is defined: there follows the "Light" patch. */


modify libGlobal
    transObjs = static new Vector(3)
;

modify SenseConnector
    addDirectConnections(tab)
    {
        /* add myself */
        tab[self] = true;
            
        /* add my contents */
        foreach (local cur in contents)
        {
            if (tab[cur] == nil)
                cur.addDirectConnections(tab);
        }

//--!! This is where the modification comes.

        local myInfoObject;

        /* Create a new transInfoObject which encapsulates our
         * transparence information.
         *
         * At this point, the myInfoObject reflects our transparency of
         * each sense.
         */
        myInfoObject = new transObjInfo(self);

        /* If there is one or more sense-connector further up the call
         * chain, we calculate net the transparency of the last
         * (closest) one compared with the new one. (The last one
         * already encapsulates the net transparency of the entire
         * chain of connectors, just as the current one will by the end
         * of the following code block.)
         */
        if (libGlobal.transObjs.length())
        {
            local lastTrans = libGlobal.transObjs[libGlobal.transObjs.length];

            /* We iterate through the senses which are not already
             * opaque, and update the myInfoObject.
             */
            foreach(local curSense in lastTrans.nonOpaques)
                if (myInfoObject.nonOpaques.indexOf(curSense))
                    myInfoObject.updateSense(curSense, lastTrans);

            /* At this point, myInfoObject reflects the net
             * transparency of all connectors in the chain.
             */

            /* If there is no non-opaque sense passable by this connector,
             * we return without adding our locations.
             */
            if (myInfoObject.nonOpaques.length() == 0)
            {
                return;
            }
        }
 
       /* We're going to add our locations. The call to
         * cur.addDirectConnections(tab) can wind up calling yet
         * another sense connector. So we'll record our
         * transparency information in the libGlobal.transObjs
         * vector.
         */
        libGlobal.transObjs.append(myInfoObject);

        /* The following process is what the current library does,
         * without our mechanism.
         */
        foreach (local cur in locationList)
        {
            if (tab[cur] == nil)
            cur.addDirectConnections(tab);
        }

        /* Now that we've added all our locations (which have added
         * everything in them including any sense connectors,
         * recursively), we must remove our transObjInfo from the
         * transObjs vector.
         * That way, we won't interfere in the sense connectors
         * which aren't being connected to the POV by us.
         */
         libGlobal.transObjs.removeElementAt(libGlobal.transObjs.length());
    }
;

#endif

/* The transObjInfo object remains the same no matter which version of
 * the patch we're using.
 */
transObjInfo: object
    construct(connector)
    {
        nonOpaques = new Vector(4, allSenses);
        senseTable = new LookupTable();

        /* populate the senseTable and nonOpaques vector. */
        foreach(local curSense in allSenses)
            updateTable(curSense, connector.transSensingThru(curSense));
    }

    senseTable = nil
    nonOpaques = nil

    updateTable(sense, transparency)
    {
        /* We're not calculating attenuation. We'll consider this
         * transparency level to be equivalent to transparent for
         * our purposes.
         */
        if (transparency == attenuated)
            transparency = transparent;
        senseTable[sense] = transparency;
        if (transparency == opaque)
            nonOpaques.removeElement(sense);
    }

    updateSense(sense, obj)
    {
        local myTrans = senseTable[sense];
        local otherTrans = obj.senseTable[sense];

        /* two non-transparents equals opaque */
        if (myTrans != transparent && otherTrans != transparent)
                updateTable(sense, opaque);

        /* one non-transparent equals obscured (or distant); it doesn't
         * matter which we call it for our purposes.
         */
        else if (myTrans != transparent || otherTrans != transparent)
                updateTable(sense, obscured);
    }
    allSenses = [sight, smell, sound, touch]
;


