#charset "us-ascii"
/* 
 *  Copyright (c) 2005 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 Flags Library Extension
 *
 *  flags.t
 *  Version 2.0
 *
 *  This file defines a Flag and BitFlag class that allows 
 *  the programmer to work with bit flags that are larger 
 *  than the 32-bit limit.
 */

#include "flags.h"

/*
 *  An object representing a flag. 
 *  This flag encapsulates its implementation
 *  details in a collection object and supplies
 *  methods for bitwise, shift, evaluation, and 
 *  comparison purposes. 
 */
class Flag: object
{
    /*
     *  The maximum bit number supported in
     *  the collection. The default is 63.
     *  Bits are numbered starting with 0.
     */
    maxBitNum       = static ((collSize * bitsPerElem) - 1)

    /*
     *  The maximum element number supported in
     *  the collection. The default is 1.
     *  Elements are numbered starting with 0.
     */
    maxElemNum      = static (collSize - 1)

    /*
     *  The size of the collection required to
     *  support the maximum number of bit flags.
     *  An author should change this value to 
     *  accomodate the number of flags he wishes 
     *  to support.
     */
    collSize   = 2

    /*
     *  The number of bits in a single element
     *  of the bit collection.
     */
    bitsPerElem     = 32
        
    /*
     *  The collection instance used to store
     *  the bits representing this flag.
     */
    collection      = nil

    /*
     *  Get the bit value (0 or 1) at location bitNum 
     *  in the flag. Bits are numbered beginning at 0.
     */
    getBitVal(bitNum)
    {
        local elem, bit;

#ifdef  __FLAG_VALIDATE__
        if (bitNum < 0 || bitNum > maxBitNum)
            throw new FlagException('Invalid bit number');
#endif  // __FLAG_VALIDATE__

        elem    = bitNum / bitsPerElem;
        bit     = bitNum % bitsPerElem;

        /*
         *  if the bit is "on" at this position
         *  return 1; otherwise return 0.
         */
        if (((1 << bit) & getElemVal(elem)) != 0)
            return 1;
        else return 0;
    }

    /*
     *  Set the bit at bitNum "on" (true) or "off" (nil). 
     *  Use of this method ensures that a bit is correctly
     *  set in the collection. Bits are numbered beginning 
     *  at 0.
     */
    setBitVal(bitNum, on)
    {
        local elem, bit, val;

#ifdef  __FLAG_VALIDATE__
        if (bitNum < 0 || bitNum > maxBitNum)
            throw new FlagException('Invalid bit number');
#endif  // __FLAG_VALIDATE__

        elem    = bitNum / bitsPerElem;
        bit     = bitNum % bitsPerElem;
        val     = (1 << bit);

        /*
         *  the bit is "off" at this position
         */
        if ((val & getElemVal(elem)) == 0)
        {
            /*
             *  Set the bit "on" at this position
             */
            if (on)
                setElemVal(elem, val);
        }
        /*
         *  the bit is "on" at this position
         */
        else
        {
            /*
             *  Set the bit "off" at this position
             */
            if (!on)
                setElemVal(elem, -val);
        }
    }

    /*
     *  Get the integer value of the collection element
     *  at location elemNum in the flag. Elements are 
     *  numbered beginning at 0.
     */
    getElemVal(elem)
    {
#ifdef  __FLAG_VALIDATE__
        if (elem < 0 || elem > maxElemNum)
            throw new FlagException('Invalid element number');
#endif  // __FLAG_VALIDATE__

        return collection[elem + 1];    
    }

    /*
     *  Adds the integer value to the collection element
     *  at location elemNum in the flag. Elements are 
     *  numbered beginning at 0.
     */
    setElemVal(elem, val)
    {
#ifdef  __FLAG_VALIDATE__
        if (elem < 0 || elem > maxElemNum)
            throw new FlagException('Invalid element number');
#endif  // __FLAG_VALIDATE__

        collection[elem + 1] += val;
    }

    /*
     *  Turns on the bits of this flag 
     *  to those indicated by flag.
     */
    setBits([flags])
    {
        collection = (or(flags...)).collection;
        return self;
    }

    /*
     *  Turns off the bits of this flag
     *  to those indicated by flag.
     */
    clearBits([flags])
    {
        flags = flags.mapAll({x: x.not()});
        collection = (and(flags...)).collection;
        return self;
    }

    /*
     *  Returns the integer (1/0) representing the true/nil 
     *  value of the flag or of the specified bit within 
     *  the flag. 
     *
     *  If no bit num argument is provided, returns
     *  1 if any of the flag's bits are turned on;
     *  otherwise 0 is returned.
     *
     *  If a bit num argument is providedr returns the 
     *  integer (1/0) value representing the logical 
     *  value for the specified bit. 
     *
     *  Bits are numbered from 0.
     */
    ival([bitNum])
    {
        bitNum = bitNum.car();
        if (bitNum != nil)
        {
            return getBitVal(bitNum);
        }
        else
        {
            for (local i = 0; i < collSize; ++i)
                if (getElemVal(i) > 0)
                    return 1;
            return 0;
        }
    }

    /*
     *  Returns the logical (true/nil) value of the 
     *  flag or of the specified bit within the flag. 
     *
     *  If no bit num argument is provided, returns
     *  true if any of the flag's bits are turned on;
     *  otherwise nil is returned.
     *
     *  If a bit num argument is providedr returns the 
     *  logical value for the specified bit. 
     *
     *  Bits are numbered from 0.
     */
    lval([num])
    {
        return ival(num...) == 1 ? true : nil;
    }

    /*
     *  Returns a flag that is the result
     *  of a bitwise COMPLEMENT (~).
     */
    not()
    {
        local newFlag;

        newFlag = new Flag();
        for (local elemNum = 0; elemNum < collSize; ++elemNum)
            newFlag.setElemVal(elemNum, ~(getElemVal(elemNum)));
        return newFlag;
    }

    /*
     *  Returns a new flag whose bits are shifted
     *  left by distance dist; fills with zero bits 
     *  on the right-hand side.
     */
    lsh(dist)
    {
        local newFlag, elemNum, val, mask;

#ifdef  __FLAG_VALIDATE__        
        if (dist < 0 || dist > maxBitNum)
            throw new FlagException('Invalid shift value');
#endif  // __FLAG_VALIDATE__

        mask    = ~(~0 << dist);
        newFlag = new Flag();
        for (elemNum = collSize - 1; elemNum > 0; --elemNum)
        {
            val = getElemVal(elemNum - 1) >> (bitsPerElem - dist);
            val &= mask;
            newFlag.setElemVal(elemNum, 
                ((getElemVal(elemNum) << dist) 
                    | val));
        }
        newFlag.setElemVal(elemNum, (getElemVal(elemNum) << dist));
        return newFlag;
    }

    /*
     *  Returns a new flag whose bits are shifted
     *  right by distance dist; fills with zero bits 
     *  on the right-hand side.
     */
    rsh(dist)
    {
        local newFlag, elemNum, val, mask;
        
#ifdef  __FLAG_VALIDATE__
        if (dist < 0 || dist > maxBitNum)
            throw new FlagException('Invalid shift value');
#endif  // __FLAG_VALIDATE__

        mask    = ~(~0 << (bitsPerElem - dist));
        newFlag = new Flag();
        for (elemNum = 0; elemNum < collSize - 1; ++elemNum)
        {
            val = getElemVal(elemNum) >> dist;
            val &= mask;
            newFlag.setElemVal(elemNum, 
                ((getElemVal(elemNum + 1) << (bitsPerElem - dist))
                    | val));
        }
        newFlag.setElemVal(elemNum, 
            ((getElemVal(elemNum) >> dist) & mask));
        return newFlag;
    }

    /*
     *  Returns a flag that is the result
     *  of a bitwise AND (&).
     */
    and_(flag)
    {
        local newFlag;

        newFlag = new Flag();
        for (local elemNum = 0; elemNum < collSize; ++elemNum)
            newFlag.setElemVal(elemNum, 
                (getElemVal(elemNum) & flag.getElemVal(elemNum)));
        return newFlag;
    }

    /*
     *  Returns a flag that is the result
     *  of a bitwise AND (&) for each element
     *  of flag.
     */
    and([flags])
    {
        local newFlag = self;
        for (local i = 1; i <= flags.length(); ++i)
            newFlag = newFlag.and_(flags[i]);
        return newFlag;
    }

    /*
     *  Returns a flag that is the result
     *  of a bitwise OR (|).
     */
    or_(flag)
    {
        local newFlag;

        newFlag = new Flag();
        for (local elemNum = 0; elemNum < collSize; ++elemNum)
            newFlag.setElemVal(elemNum, 
                (getElemVal(elemNum) | flag.getElemVal(elemNum)));
        return newFlag;
    }

    or([flags])
    {
        local newFlag = self;
        for (local i = 1; i <= flags.length(); ++i)
            newFlag = newFlag.or_(flags[i]);
        return newFlag;
    }

    /*
     *  Returns a flag that is the result
     *  of a bitwise XOR (^).
     */
    xor_(flag)
    {
        local newFlag;

        newFlag = new Flag();
        for (local elemNum = 0; elemNum < collSize; ++elemNum)
            newFlag.setElemVal(elemNum, 
                (getElemVal(elemNum) ^ flag.getElemVal(elemNum)));
        return newFlag;
    }

    xor([flags])
    {
        local newFlag = self;
        for (local i = 1; i <= flags.length(); ++i)
            newFlag = newFlag.xor_(flags[i]);
        return newFlag;
    }

    /*
     *  Returns true if the integer value of 
     *  each elemnt of this flag is equal to 
     *  the the integer value of each element 
     *  of the flag argument.
     */
    equals(flag)
    {
        for (local i = 0; i < collSize; ++i)
        {
            if (getElemVal(i) != flag.getElemVal(i))
                return nil;
        }
        return true;
    }

    /*
     *  Compares the bits of this flag with those 
     *  of flag and returns:
     *  
     *   1  this flag's bit value is greater than flag
     *   0  this flag's bit value is equal to flag
     *  -1  this flag's bit value is less than flag
     */
    comp(flag)
    {
        for (local elemNum = collSize - 1; elemNum > -1; --elemNum)
        {
            if (getElemVal(elemNum) > flag.getElemVal(elemNum))
                return 1;
            else if (getElemVal(elemNum) < flag.getElemVal(elemNum))
                return -1;
        }
        return 0;
    }

    /* 
     *  Converts the flag to a single-quote
     *  string binary representation -- a "bit string".
     */
    toBitString([bitSep])
    {
        local num, eStr, fStr = '';

        bitSep  = bitSep.car();

        for (local elemNum = 0; elemNum < collSize; ++elemNum)
        {
            num = getElemVal(elemNum);
            eStr = '';
            for (local i = 1; i <= 4; ++i)
            {
                local str, val;

                val = (num & 255);
                str = toString(val,2);
                str = ('00000000'.substr(1, 8 - str.length()) + str);
                eStr = (str + eStr);
                if (i < 4 && bitSep)
                    eStr = (' ' + eStr);
                num >>= 8;
            }
            
            fStr = (eStr + fStr);
            if (elemNum < collSize)
                fStr = (' ' + fStr);
        }
        return fStr;
    }

    /* 
     *  Displays the binary representation 
     *  of this flag as built by toSString().
     */
    disp()
    {
        "\b<<toBitString(true)>>\n";
    }

    /*
     *  An abstract method for returning a Flag 
     *  by converting the bit string into the 
     *  appropriate flag value.
     */
    cvtBitString(str)
    {
        local newFlag, bitNum, elemNum, val;

#ifdef  __FLAG_VALIDATE__
        if (str.length() > maxBitNum + 1)
            throw new FlagException('Bit String longer than maximum number of bits');
#endif  // __FLAG_VALIDATE__

        newFlag = new Flag();

        /*
         *  Initialize to -1 so that when we start the
         *  iteration the first increment will set this 
         *  value to zero.
         */
        bitNum = -1;

        for (local i = str.length(); i > 0; --i)
        {
            /*
             *  White spaces don't count
             */
            if (str.substr(i,1) == ' ')
                continue;

            bitNum++;

            /* 
             *  For values other than '1' we simply 
             *  continue the iteration as there is 
             *  nothing to add to the flag.
             */
            if (str.substr(i,1) != '1')
                continue;

            elemNum = bitNum / bitsPerElem;
            val     = 1 << (bitNum % bitsPerElem);

            /*
             *  Set the bit at bitNum to "on"
             */
            newFlag.setElemVal(elemNum, val);
        }
        return newFlag;
    }
    xcvtBitString(str)
    {
        local newFlag, bitNum;

#ifdef  __FLAG_VALIDATE__
        if (str.length() > maxBitNum + 1)
            throw new FlagException('Bit String longer than maximum number of bits');
#endif  // __FLAG_VALIDATE__

        newFlag = new Flag();

        /*
         *  Initialize to -1 so that when we start the
         *  iteration the first increment will set this 
         *  value to zero.
         */
        bitNum = -1;

        for (local i = str.length(); i > 0; --i)
        {
            /*
             *  White spaces don't count
             */
            if (str.substr(i,1) == ' ')
                continue;

            bitNum++;

            /* 
             *  For values other than '1' we simply 
             *  continue the iteration as there is 
             *  nothing to add to the flag.
             */
            if (str.substr(i,1) != '1')
                continue;

            /*
             *  Set the bit at bitNum to "on"
             */
            newFlag.setBitVal(bitNum, true);
        }
        return newFlag;
    }

    /*
     *  Returns the count of "on" bits for the flag
     *  or if an elem argument is supplied, the count
     *  of "on" bits for that element of the flag.
     */
    bitCount([elem])
    {
        local start, len, num, cnt = 0;

        elem = elem.car();
        if (elem)
        {
#ifdef  __FLAG_VALIDATE__
            if (0 <= elem && elem < collSize)
            {
                start   = elem;
                len     = elem + 1;
            }

            else throw new FlagException('Invalid element number');
#else
            start   = elem;
            len     = elem + 1;
#endif  // __FLAG_VALIDATE__

        }
        else
        {
            start   = 0;
            len     = collSize;
        }
        for (local elemNum = start; elemNum < len; ++elemNum)
        {
            num = getElemVal(elemNum);

            while(num)
            {
                cnt++;
                num &= (num - 1);
            }
        }

        return cnt;
    }

    /*
     *  Constructor for the new flag
     */
    construct([flags])
    {
        local len;

        collection  = new Vector(collSize);
        collection.fillValue(0, 1, collSize);

        len = flags.length();

        if (len)
        {
            if (len == 1
                && dataType(flags.car()) == TypeSString)
                flags = cvtBitString(flags...);

            setBits(flags...);
        }
    }
}

/*
 *  An object representing a bit flag. 
 *  This flag has only one bit set to on,
 *  the one indicated by bitNum. Bits are 
 *  numbered from 0.
 */
class BitFlag: Flag
{
    bitNum          = nil
    elemNum         = nil
    elemVal         = nil

    /*
     *  Get the bit value (0 or 1) at location bitNum 
     *  in the flag. Bits are numbered beginning at 0.
     *  
     *  A bitFlag has only 1 bit set "on", indicated 
     *  by the object's bitNum value.
     */
    getBitVal(bitNum)
    {
        if (bitNum == self.bitNum)
            return 1;
        else return 0;
    }

    /*
     *  Set the bit at bitNum "on" (true) or "off" (nil). 
     *  Use of this method ensures that a bit is correctly
     *  set in the collection. Bits are numbered beginning 
     *  at 0.
     *
     *  A bitFlag cannot have it's bit values set after 
     *  it's initialization, so this method does nothing.
     */
    setBitVal(bitNum, on) {}

    /*
     *  Get the integer value of the collection element
     *  at location elemNum in the flag. Elements are 
     *  numbered beginning at 0.
     */
    getElemVal(elem)
    {
#ifdef  __FLAG_VALIDATE__
        if (elem < 0 || elem > maxElemNum)
            throw new FlagException('Invalid element number');
#endif  // __FLAG_VALIDATE__

        if (elemNum == elem)
            return elemVal;
        else return 0;
    }

    /*
     *  Adds the integer value to the collection element
     *  at location elemNum in the flag. Elements are 
     *  numbered beginning at 0.
     *
     *  A BitFlag can't have it's element values set. So this
     *  method does nothing.
     */
    setElemVal(elem, val) {}

    /*
     *  A BitFlag can't have it's bits set. We simply
     *  return the original bit flag.
     */
    setBits([flags]) { return self; }

    /*
     *  A BitFlag can't have it's bits cleared. We simply
     *  return the original bit flag.
     */
    clearBits([flags]) { return self; }

    construct([num])
    {
        local elem, bit;

        num = num.car();

        if (num != nil)
            bitNum = num;

#ifdef  __FLAG_VALIDATE__
        if (bitNum < 0 || bitNum > maxBitNum)
            throw new FlagException('Invalid bit number');
#endif  // __FLAG_VALIDATE__
        
        elem    = bitNum / bitsPerElem;
        bit     = bitNum % bitsPerElem;

        elemNum = elem;
        elemVal = (1 << bit);
    }
}

class FlagException: RuntimeError
{
    construct(msg)
    {
        inherited(0);

        exceptionMessage = msg;
    }
    throwExc(msg)
    {
        throw createInstance(msg);
    }
}