#charset "us-ascii"
#include <adv3.h>
#include <en_us.h>

// Padlock.t: A TADS 3 implementation of padlocks and padlockable things.
// by Greg Boettcher
// v1.00 2016-11-29

// This extension is freeware. If you use it, I would appreciate attribution, 
// although I'm not going to insist on it. By the same token, this comes with no warranty.

// I tried to test this pretty thoroughly, but of course there may be bugs I missed.
// Feel free to send bugs/suggestions to paradoxgreg@X.com, where X = gmail. 

// A demo, padlockdemo.t, is available from wherever you got this code.

// The classes you are most likely to need are:
// KeyedPadlock: for padlocks that can be opened with a key
// IndirectPadlock: for padlocks that can't be opened in any obvious way
// PadlockableDoor: for padlockable doors
// PadlockableContainer: for padlockable containers. This derives from ComplexContainer;
//   see below for more info.

// This extension requires SimpleAttachable.t, which is not part of the standard
// TADS 3 adv3 library, but is an extension that comes with the TADS 3 Author's Kit.
// You should find it in \lib\extensions wherever you installed the Author's Kit.

// Design features that you should know about this extension:
// 1. For the Padlockable class that defines all things to which you can attach a 
//    padlock, you will likely run into problems if you create Padlockable objects that 
//    are not Openable. This is because, the more I worked on this, the more I could 
//    think of no good reason why anyone would want to put padlocks on anything 
//    unopenable, and it simplified my effort to exclude that scenario from my design.
// 2. I also made a choice to not try to design for the scenario where a door or
//    container can be locked both by a padlock and also by anything else 
//    non-padlock-related. In technical terms, this means that you should not create 
//    a door/container that is both Padlockable and LockableWithKey, for example, 
//    unless you are prepared to put in some work ironing out some problems.
//    (If you do this and test it thoroughly, let me know and we can 
//    come out with a version 2.0 of this extension with you as co-author!)
// 3. Rules for doors that represent the other side of a PadlockableDoor:
//    (a) Such doors must somehow inherit from both Lockable and Door. 
//        ("IndirectLockable, Door" is a good choice.)
//    (b) Such doors must not belong to LockableWithKey or be lockable by means of 
//        anything other than a padlock. See #2 above. I did not design for 
//        this scenario.
//    (c) Such doors *may* be PadlockableDoors themselves. That is, I tried to make it
//        so you can implement the (atypical?) scenario where a door can have 
//        padlocks affixed to it on both sides. In full disclosure, I didn't test this 
//        a lot, so let me know if you run into problems.

// -------------------------------------------------------------------------------------

// Padlock: 
// A base class for padlocks, with no implementation for unlocking.
// Rather than using this directly, first consider KeyedPadlock or 
// IndirectPadlock below.
// Or, if you want to create a combination padlock, it probably wouldn't
// be too hard to create a subclass for that.

class Padlock: Lockable, SimpleAttachable, Openable, Thing
  // -----------------
  // Open, Close, Lock, Unlock handling
  // -----------------
  // A padlock can only be either open/unlocked or closed/locked. Zealously 
  // enforce this by overriding makeLocked and makeOpen as follows. 
  makeLocked(stat, [args]) {
    inherited(stat);
    
    local avoidRecursion = nil;
    if (args.length && args[1] == true) {
      avoidRecursion = true;
    }
    if (!avoidRecursion) {
      makeOpen((stat ? nil : true), true);
    }
  }
  makeOpen(stat, [args]) {
    inherited(stat);
    
    local avoidRecursion = nil;
    if (args.length && args[1] == true) {
      avoidRecursion = true;
    }
    if (!avoidRecursion) {
      makeLocked((stat ? nil : true), true);
    }
  }
  dobjFor(Open) {
    verify() { verifyDobjUnlock(); }
    preCond = (nilToList(inherited()) - [objUnlocked])
    check() { checkDobjUnlock(); }
    action() { actionDobjUnlock(); }
  }
  dobjFor(Close) {
    verify() { verifyDobjLock(); }
    check() { checkDobjLock(); }
    action() { actionDobjLock(); }
  }
  autoUnlockOnOpen = nil
  // There are no prerequisites for locking a padlock, no matter how hard
  // it may be to unlock.
  dobjFor(Lock) {
    verify() {
      if (!isOpen) {
        illogicalAlready(&alreadyLockedMsg);
      }
      else {
        logicalRank(140, 'unlocked');
      }
    }
    preCond = (nilToList(inherited()) - [objClosed])
    action() {
      makeOpen(nil);
      //makeLocked(true); // redundant now
      defaultReport(&okayLockMsg);
    }
  }
  dobjFor(Unlock) {
    verify() {
      if (isOpen) {
        illogicalAlready(&alreadyUnlockedMsg);
      }
      else {
        logicalRank(140, 'locked');
      }
    }
    preCond = (nilToList(inherited()) - [objUnlocked])
    // Subclasses can override this
    action() {
      askForIobj(UnlockWith);
    }
  }
  // This is a prime thing to be modified by subclasses that actually do take keys.
  // i.e., LOCK PADLOCK WITH X
  dobjFor(LockWith) {
    verify() {
      if (isLocked) {
        illogicalAlready(&alreadyLockedMsg);
      }
      else {
        logicalRank(140, 'locked');
      }
    }
  }
  // i.e., LOCK X WITH PADLOCK
  iobjFor(LockWith) {
    verify() {
      if (gDobj && gDobj.ofKind(Padlockable)) {
        logicalRank(150, 'very suitable padlock');
      }
      else {
        logicalRank(110, 'possibly unsuitable padlock');
      }
    }
    check {
      if (!gDobj.ofKind(Padlockable)) {
        reportFailure(padlockMessages.cannotLockUnpadlockable);
        exit;
      }
    }
    action {
      // If all the preconditions have succeeded thus far, including 
      // (1) you can touch the padlockable object, (2) the padlockable
      // object is closed, and (3) you are holding the padlock,
      // then I can think of no reason why this should fail. 
      gIobj.moveInto(gDobj.targetForDetachingPadlocks);
      gIobj.attachTo(gDobj.targetForDetachingPadlocks);
      gIobj.makeOpen(nil);
      say(padlockMessages.okayLockWithPadlock);
    }
  }
  // -----------------
  // Attach/Detach/Fasten/Unfasten handling
  // -----------------
  dobjFor(AttachTo) {
    verify() {
      if (location != nil && location.ofKind(Attachable) 
          && location.attachedObjects.indexOf(self) != nil) {
        illogicalAlready(&alreadyAttachedMsg);
      }
    }
    preCond = (nilToList(inherited()) + [objUnlocked])
  }
  dobjFor(DetachFrom) {
    preCond = (nilToList(inherited()) + [objUnlocked])
    action() {
      inherited();
      // Override the default of "detach it and it winds up on the floor."
      // That's a fine default for large things, but not for small things.
      // However, make an exception if we are implicitly detaching because the 
      // player typed "TAKE PADLOCK", as otherwise we would get an awkward message.
      local parentActionIsTake = nil;
      local parent = gAction.parentAction;
      while (parent != nil) {
        if (parent.baseActionClass == TakeAction) {
          parentActionIsTake = true;
        }
        parent = parent.parentAction;
      }
      if (!parentActionIsTake) {
        moveInto(gActor);
      }
    }
  }
  // Not sure why Fasten/Unfasten aren't synonymous with Attach/Detach.
  // At any rate, change this.
  dobjFor(Fasten) {
    //preCond { return inherited() - objClosed; }
    verify() {
      if (location != nil && location.ofKind(Attachable) 
          && location.attachedObjects.indexOf(self) != nil) {
        illogicalAlready(&alreadyAttachedMsg);
      }
    }
    action() {
      askForIobj(FastenTo);
    }
  }
  dobjFor(FastenTo) asDobjFor(AttachTo)
  dobjFor(Unfasten) asDobjFor(Detach)
  dobjFor(UnfastenFrom) asDobjFor(DetachFrom)
  // -----------------
  // Examine handling
  // -----------------
  // Co-opting openStatus, which typically reports only one thing, but now will be
  // assigned to report on three things: whether we're (1) open, (2) locked, and 
  // (3) attached to something (and our reporting of this may depend on whether that
  // thing is open or closed).
  openStatus {
    return padlockMessages.padlockStatusMsg(self);
  }
  // Getting rid of currentlyLocked/currentlyUnlocked messages, since openStatus is handling this
  lockStatusReportable = nil
  // -----------------
  // Misc behavior
  // -----------------
  initializeThing() {
    // Inheritance from SimpleAttachable will make sure that, if the padlock's
    // initial location is padlockable, the padlock will be attached to that location.
    inherited;
    
    //// copying from SimpleAttachable, just in case it's needed
    //if(location && location.isMajorItemFor(self)) {
    //  attachTo(location);
    //}
    
    // isOpen and isLocked need to always have opposite values for this class.
    // Probably redundant, but just making sure.
    if (isOpen) { makeLocked(nil); }
    else if (!isOpen) { makeLocked(true); }
  }
;

// -------------------------------------------------------------------------------------

// KeyedPadlock: 
// This is the class to use for a padlock that can be opened with a key.

class KeyedPadlock: LockableWithKey, Padlock
  // LockableWithKey wants to ask for a key when you type "lock x", but 
  // for a padlock we want to override that.
  dobjFor(Lock) {
    action() {
      makeOpen(nil);
      //makeLocked(true); // redundant now
      defaultReport(&okayLockMsg);
    }
  }
;

// IndirectPadlock: 
// This is the class to use for a padlock that can't be opened in any direct way.

class IndirectPadlock: IndirectLockable, Padlock
  // Must override Lock for reasons similar to above. Locking a padlock is easy.
  dobjFor(Lock) {
    check() { }
    action() {
      makeOpen(nil);
      //makeLocked(true); // redundant now
      defaultReport(&okayLockMsg);
    }
  }
;

// -------------------------------------------------------------------------------------

// Padlockable: 
// A a mix-in class. Before using this class directly, first consider 
// PadlockableDoor and PadlockableContainer below.
// This mix-in class is for any item that can be locked with a padlock. Objects that 
// inherit from it should also be made to inherit from a Thing subclass as well.

// This implementation assumes that Padlockable things cannot be locked by anything
// other than a padlock. (Nor can Padlockable things be *linked* to anything that's 
// lockable by means of anything other than a padlock. E.g. you can't have a door 
// that's Padlockable on one side but LockableWithKey on the other side.)

class Padlockable: SimpleAttachable, Lockable
  // -----------------
  // Attach/Detach/Fasten/Unfasten handling
  // -----------------
  // Not sure why Fasten/Unfasten aren't synonymous with Attach/Detach.
  // At any rate, change this.
  iobjFor(FastenTo) asIobjFor(AttachTo)
  // To coordingate with SimpleAttachable.t. By default, padlocks and only padlocks 
  // can be attached to this.
  isMajorItemFor(obj) {
    return obj.ofKind(Padlock);
  }
  iobjFor(PutOn) maybeRemapTo((gDobj && gDobj.ofKind(Padlock)), AttachTo, gDobj, self)
  // -----------------
  // Open/Close handling
  // -----------------
  dobjFor(Open) {
    preCond = (nilToList(inherited()) - [objUnlocked] + [objNotPadlocked])
  }
  dobjFor(Close) {
    // Some authors may prefer to include objNotPadlocked as a precondition here,
    // but I decided I like letting the player try to close a door even with padlocks
    // attached. Their attempt will fail.
    //preCond = (nilToList(inherited()) - [objUnlocked] + [objNotPadlocked])
    check {
      if (attachedPadlocks.length > 0) {
        reportFailure(padlockMessages.cannotCloseDueToLocalPadlocks(self));
        exit;
      }
      if (otherSide && otherSide.attachedPadlocks && otherSide.attachedPadlocks.length > 0) {
        reportFailure(padlockMessages.cannotCloseDueToRemotePadlocks(self));
        exit;
      }
    }
  }
  // -----------------
  // Lock/Unlock
  // -----------------
  // Completely override the definition of isLocked! Remember, this class assumes 
  // that Padlockable things cannot be locked by anything other than padlocks.
  isLocked() {
    if (isOpen) {
      return nil;
    }
    if (attachedPadlocks.length > 0) {
      return true;
    }
    if (otherSide && otherSide.attachedPadlocks && otherSide.attachedPadlocks.length > 0) { 
      return true;
    }
    return nil;
  }
  dobjFor(Unlock) {
    check() {
      if (attachedPadlocks.length == 0) {
        reportFailure(&unknownHowToUnlockMsg);
        exit;
      }
    }
    action() {
      // Just in case. Hopefully this won't happen.
      if (attachedPadlocks.length == 0) {
        say(&unknownHowToUnlockMsg);
      }
      else {
        replaceAction(DetachFrom, attachedPadlocks, targetForDetachingPadlocks);
      }
    }
  }
  dobjFor(Lock) {
    // default verify and preCond are fine for this, but the inherited action is wrong.
    action {
      askForIobj(LockWith);
    }
  }
  dobjFor(LockWith) {
    // must override the "does not appear to take a key" message.
    verify() {
      logicalRank(110, 'somewhat logical to lock');
    }
  }
  targetForDetachingPadlocks = (self)
  targetForBeingOpen = (self)
  // By default, not initially locked. But if there is a padlock on this at the start 
  // of the game, the Padlock class's InitializeThing method will set the state of this 
  // door to locked.
  initiallyLocked = nil
  // -----------------
  // Examine handling
  // -----------------
  // Co-opting openStatus, which typically reports only one thing, but now will be
  // assigned to report on three things: whether we're (1) open, (2) locked, and 
  // (3) holding any padlocks (and our reporting of this may depend on whether those
  // padlocks are open or closed).
  openStatus {
    return padlockMessages.padlockableStatusMsg(self);
  }
  // Getting rid of currentlyLocked/currentlyUnlocked messages, since openStatus is handling this
  lockStatusReportable = nil
  // -----------------
  // Misc behavior
  // -----------------
  attachedPadlocks() {
    local attachedPadlocks = [];
    for (local i = 1; i <= attachedObjects.length; i++) {
      if (attachedObjects[i].ofKind(Padlock)) {
        attachedPadlocks += attachedObjects[i];
      }
    }
    return attachedPadlocks;
  }
  initializeThing() {
    inherited();
    
    //// Get list of padlocks attached to both this and the other side
    //// Sadly, we can't rely on attachedPadlocks() because that list may not be done
    //// initializing yet. So, changing tack:
    //local padlocksAttachedToSelf = [];
    //local padlocksAttachedToOtherSide = [];
    //for (local obj = firstObj(Padlock, ObjInstances); obj != nil;
    //     obj = nextObj(obj, Padlock, ObjInstances))
    //{
    //  if (obj.location.getIdentityObject() == self) {
    //    padlocksAttachedToSelf += obj;
    //  }
    //  if (self.otherSide != nil && obj.location.getIdentityObject() == self.otherSide) {
    //    padlocksAttachedToOtherSide += obj;
    //  }
    //}
    //
    //if (padlocksAttachedToSelf.length > 0 || padlocksAttachedToOtherSide.length > 0) {
    //  makeOpen(nil);
    //  //makeLocked(true); // redundant now
    //  if (self.otherSide != nil) {
    //    otherSide.makeOpen(nil);
    //    otherSide.makeLocked(true);
    //  }
    //}
  }
;

// -------------------------------------------------------------------------------------

// PadlockableDoor: 
// The class to be used for a door that can be locked with a padlock.
// Assuming you are implementing the opposite site of a padlockable door,
// you need to make sure that that opposite site inherits from IndirectLockable 
// (or at least Lockable) and Door, or it probably won't behave as you expect.

class PadlockableDoor: Padlockable, Door
;

// OtherSideOfPadlockableDoor: 
// For your convenience. Opposite sides of padlockable doors must inherit from 
// IndirectLockable (or at least Lockable) and Door, or they won't work right.

class OtherSideOfPadlockableDoor: IndirectLockable, Door
;

modify Door
  dobjFor(Close) {
    check {
      if (otherSide && otherSide.attachedPadlocks && otherSide.attachedPadlocks.length > 0) {
        reportFailure(padlockMessages.cannotCloseDueToRemotePadlocks(self));
        exit;
      }
      inherited();
    }
  }
;

// -------------------------------------------------------------------------------------

// PadlockableContainer:
// The class to use for containers that can be locked with padlocks.
// PadlockableContainer has to be a ComplexContainer, and as such must have 
// a subSurface and subContainer deriving from appropriate classes 
// (PadlockableContainerSubsurface and PadlockableContainerOpenableSubcontainer). 
// See example below, which is taken from padlockdemo.t.

/*
box: PadlockableContainer
  'box/container' 'box'
  @lockRoom
  desc = "It's a box. "
  subSurface: PadlockableContainerSubsurface {
  }
  subContainer: PadlockableContainerOpenableSubcontainer {
  }
;
*/

class PadlockableContainer: Padlockable, ComplexContainer
  attachedPadlocks = (self.subSurface.attachedPadlocks)
  targetForDetachingPadlocks = (self.subSurface)
  targetForBeingOpen = (self.subContainer)
;

// -------------------------------------------------------------------------------------

// PadlockableContainer's complexComponent subclasses

class PadlockableContainerSubsurface: Padlockable, ComplexComponent, Surface
  // -----------------
  // Open/Close handling
  // -----------------
  iobjFor(PutOn) {
    verify() {
      if (gDobj && !gDobj.ofKind(Padlock)) {
        illogicalNow(&notASurfaceMsg); // change this message later, it's a little misleading
      }
    }
    action() {
      if (gDobj && gDobj.ofKind(Padlock)) {
        replaceAction(AttachTo, gDobj, self);
      }
      else {
        inherited();
      }
    }
  }
  // This works in conjunction with SimpleAttachable to dictate that you can
  // attach a padlock to this
  targetForDetachingPadlocks = (self)
  targetForBeingOpen = (location.subContainer)
  isMajorItemFor(obj) {
    return obj.ofKind(Padlock);
  }
;

class BasicPadlockableContainerSubcontainer: Padlockable, ComplexComponent
  attachedPadlocks = (location.attachedPadlocks)
  targetForDetachingPadlocks = (location.subSurface)
  targetForBeingOpen = (self)
;

class PadlockableContainerOpenableSubcontainer: BasicPadlockableContainerSubcontainer, OpenableContainer
;

// Why you would want to padlock a container that can't be closed? This hasn't been tested
// and probably wouldn't work.
class PadlockableContainerSubcontainer: BasicPadlockableContainerSubcontainer, Container;
;

// -------------------------------------------------------------------------------------
// MajorAttachmentLister
// -------------------------------------------------------------------------------------

// This section may be important to you if you are particular about how your padlocks appear 
// in room descriptions and object descriptions. I will highlight two main options here.
// 
// Option 1. This is default but adds extra burden to you as the author to enhance your
// room descriptions. How to do this is exemplified in padlockdemo.t. 
// This is the option to choose if you like the output below and are willing to enhance 
// your descriptions of rooms and lockables to make it happen. I like the output below,
// and that is why I made this the default option.
// 
// >LOOK AROUND
// Room
// This is a room. To the east stands a door, which is locked shut by a large padlock.
// >X DOOR
// It's a heavy door. It's currently locked shut by a large padlock.
// 
// Option 2. This is more consistent with adv3 classes Attachable and SimpleAttachable.
// It is also easier (provided you make the modifications suggested below), as it 
// doesn't require as much work for you to make your room/object descriptions quite 
// so dynamic. Please note, this option did not get tested much.
// 
// >LOOK AROUND
// Room
// This is a room. To the east stands a door.
// A large padlock is attached to the door.
// >X DOOR
// It's a heavy door. It's currently locked. 
// A large padlock is attached to the door.
// >X PADLOCK
// It's a large padlock. It's currently locked.
// The large padlock is attached to a door.

// Option 1: Simply don't modify this code. This means your room descriptions
// need to be more robust.

modify MajorAttachmentLister
  isListed(obj) { return (parent_.isListedAsMajorFor(obj) && !obj.ofKind(Padlock)); }
  //// standard adv3 implementation:
  //isListed(obj) { return parent_.isListedAsMajorFor(obj); }
;

// Option 2: modify the above by putting this into your code, reinstating the adv3 default:

/*
modify MajorAttachmentLister
  isListed(obj) { return parent_.isListedAsMajorFor(obj); }
;
*/

// -------------------------------------------------------------------------------------
// SimpleAttachmentLister
// -------------------------------------------------------------------------------------

// I don't recommend changing this unless you dislike the default below and want to
// change it.
// 
// Padlock.t default:
// >X PADLOCK
// It's a large padlock. It's currently closed, and is secured to the door, locking it.
// 
// If you went back to adv3 defaults, it would be:
// >X PADLOCK
// It's a large padlock. It's currently closed, and is secured to the door, locking it.
// The large padlock is attached to a door.

modify SimpleAttachmentLister
  isListed(obj) { return !parent_.ofKind(Padlock) && parent_.isListedAsAttachedTo(obj); }
;

// -------------------------------------------------------------------------------------
// Preconditions
// -------------------------------------------------------------------------------------

objNotPadlocked: PreCondition
  checkPreCondition(obj, allowImplicit)
  {
    // get list of attached padlocks    
    local attachedPadlocks = obj.attachedPadlocks();
    //local attachedPadlocks = [];
    //for (local i = 1; i <= obj.attachedObjects.length; i++) {
    //  if (obj.attachedObjects[i].ofKind(Padlock)) {
    //    attachedPadlocks += obj.attachedObjects[i];
    //  }
    //}
    
    // If no padlocks attached, nothing to do.
    if (attachedPadlocks.length == 0) {
      return nil;
    }
    
    // Try to remove all the padlocks from the object
    if (allowImplicit) {
      for (local i = 1; i <= attachedPadlocks.length; i++) {
        // for each padlock, try to implicitly detach it
        local padlock = attachedPadlocks[i];
        if (tryImplicitAction(DetachFrom, padlock, padlock.location)) {
          // If the implicit detachment failed, abort.
          if (padlock.location == obj.targetForDetachingPadlocks) {
            exit;
          }
        }
      }
      // if all the implicit detachments succeeded, tell caller we succeeded
      return true;
    }
    
    // recalculate list of attached padlocks for reporting
    attachedPadlocks = obj.attachedPadlocks();
    //attachedPadlocks = [];
    //for (local i = 1; i <= obj.attachedObjects.length; i++) {
    //  if (obj.attachedObjects[i].ofKind(Padlock)) {
    //    attachedPadlocks += obj.attachedObjects[i];
    //  }
    //}
    // report failure
    reportFailure(padlockMessages.mustRemovePadlocksMsg(attachedPadlocks, obj));
    exit;
  }
;


// -------------------------------------------------------------------------------------
// English language-specific code
// -------------------------------------------------------------------------------------

// Note: I suppose some might say that it's good to decouple language-specific
// code from logic-intensive code. By that measure, I failed pretty bad here. 
// Oh well, so sue me.

padlockMessages: object
  cannotCloseDueToLocalPadlocks(obj) {
    gMessageParams(obj);
    return 
      '{You/he} tr{ies/ied} to close {the obj/him}, but {you/he} 
        {can\'t} fully shut {it obj/him} because of '
      + sayList(obj.attachedPadlocks, &theName) 
      + ' attached to {it obj/him}. ';
  }
  cannotLockUnpadlockable() {
    return '{You/he} {can\'t} see any easy way to use 
      {the iobj/him} to lock {that dobj/him}. ';
  }
  okayLockWithPadlock() {
    return 'Okay, {you/he} attach{es/ed} {the iobj/him} to {the dobj/him},
      then snap{s/ed actor} {it iobj/him} shut. ';
  }
  cannotCloseDueToRemotePadlocks(obj) {
    gMessageParams(obj);
    return 
      '{You/he} tr{ies/ied} to close {the obj/him}, but {you/he} 
        {can\'t} fully shut {it obj/him}, apparently because of something
        on the other side. ';
  }
  // padlockStatusMsg is used for when the player examines a padlock. 
  // Specifically, it is used for the adv3 library's Padlock.openStatus, and thus 
  // must be an independent clause without starting capitalization or closing 
  // punctuation.
  // We are co-opting this to also convey locked status and attached padlock status.
  padlockStatusMsg(obj) {
    // start by reporting "it's open" or "it's closed"
    local msg = obj.itIsContraction + ' '
      + (obj.isOpen ? 'currently open' : 'currently locked shut');
    
    if (obj.location && obj.location.ofKind(Padlockable)) {
      // next report what it's attached to
      if (obj.isOpen) {
        msg += ', and is hanging on ' + obj.location.theName;
      }
      else {
        msg += ', and is secured to ' + obj.location.theName;
        
        // if the location is locked shut due to the padlock, report that.
        if (!obj.location.targetForBeingOpen.isOpen) {
          msg += ', locking ' + obj.location.itObj;
        }
      }
    }
    return msg;
  }
  // padlockableStatusMsg is used for when the player examines a padlockable object. 
  // Specifically, it is used for the adv3 library's Padlockable.openStatus, and thus 
  // must be an independent clause without starting capitalization or closing 
  // punctuation.
  // We are co-opting openStatus to also convey locked status and attached status.
  // This should work for both doors and padlockable containers. 
  // I can't imagine a reason to make something padlockable without also making it 
  // openable, so I'm not accounting for that scenario.
  padlockableStatusMsg(obj) {
    local attachedLockedPadlocks = [];
    local attachedHangingPadlocks = [];
    for (local i = 1; i <= obj.attachedPadlocks.length; i++) {
      if (obj.attachedPadlocks[i].isOpen) {
        attachedHangingPadlocks += obj.attachedPadlocks[i];
      }
      else {
        attachedLockedPadlocks += obj.attachedPadlocks[i];
      }
    }
    
    local msg;
    // If obj is closed, say so, and add info about attached *locked* padlocks:
    if (!obj.isOpen) {
      msg = obj.itIsContraction + ' currently closed';
      if (attachedLockedPadlocks.length > 0) {
        msg += ', and ' + obj.verbToBe + ' locked shut by '
          + sayList(attachedLockedPadlocks, &aName);
      }
    }
    // If obj is open, say so, and add info about attached *locked* padlocks:
    else {
      msg = obj.itIsContraction + ' currently open';
      if (attachedLockedPadlocks.length > 0) {
        msg += ', while at the same time there '
          + isOrAre(attachedLockedPadlocks) + ' '
          + sayList(attachedLockedPadlocks, &aName)
          + ' locked shut upon ' + obj.itObj;
      }
    }
    // add info about attached *non-locked* padlocks:
    if (attachedHangingPadlocks.length > 0) {
      msg += '. There ';
      msg += isOrAre(attachedHangingPadlocks);
      msg += ' '
        + (attachedLockedPadlocks.length > 0 ? 'also ' : '')
        + sayList(attachedHangingPadlocks, &aName)
        + ' hanging open upon ' + obj.itObj;
    }
    return msg;
  }
  // briefPadlockableStatusMsg is intended for room descriptions, when authors
  // want the status of their padlockable items to be mentioned in the room description.
  // This is particularly likely to be helpful for padlockable doors, and maybe
  // also for containers that are so large that their locked state belongs in
  // the room description. briefPadlockableStatusMsg is like padlockableStatusMsg, 
  // except (1) there's no need to begin with "it's", and (2) for room 
  // descriptions we may want the status message to be a little more brief.
  briefPadlockableStatusMsg(obj) {
    local attachedLockedPadlocks = [];
    local attachedHangingPadlocks = [];
    for (local i = 1; i <= obj.attachedPadlocks.length; i++) {
      if (obj.attachedPadlocks[i].isOpen) {
        attachedHangingPadlocks += obj.attachedPadlocks[i];
      }
      else {
        attachedLockedPadlocks += obj.attachedPadlocks[i];
      }
    }
    
    local msg = '';
    // If obj is closed, say so, and add info about all attached padlocks:
    if (!obj.isOpen) {
      if (attachedLockedPadlocks.length > 0) {
        msg += 'locked shut by '
          + sayList(attachedLockedPadlocks, &aName);
      }
      else {
        msg += 'closed';
      }
      if (attachedHangingPadlocks.length > 0) {
        msg += ', and '
          + (attachedLockedPadlocks.length > 0 ? 'also ' : '')
          + obj.verbToHave + ' '
          + sayList(attachedHangingPadlocks, &aName)
          + ' hanging on ' + obj.itObj;
      }
    }
    // If obj is open, say so, and add info about all attached padlocks:
    else {
      msg = 'open';
      if (attachedLockedPadlocks.length > 0 ||
          attachedHangingPadlocks.length > 0) {
        msg += ', yet ';
      }
      if (attachedLockedPadlocks.length > 0) {
        msg += obj.verbToHave + ' '
          + sayList(attachedLockedPadlocks, &aName)
          + ' locked shut on ' + obj.itObj;
      }
      if (attachedHangingPadlocks.length > 0) {
        msg += (attachedLockedPadlocks.length > 0 ? ', and also ' : '')
          + obj.verbToHave + ' '
          + sayList(attachedHangingPadlocks, &aName)
          + ' hanging on ' + obj.itObj;
      }
    }
    return msg;
  }
  mustRemovePadlocksMsg(padlockList, obj) {
    return
	  '{You/he} {must} detach '
	  + sayList(padlockList, &theName)
	  + ' before {it actor/he} {can} do that. ';
  }
  isOrAre(theList) {
    if (gameMain.usePastTense) {
      return (theList.length == 1 ? 'was' : 'were');
    }
    return (theList.length == 1 ? 'is' : 'are');
  }
  sayList(theList, prop) {
    local msg = '';
    for (local i = 1; i <= theList.length; i++) {
      local item = theList[i];
      msg += item.(prop);
      if (theList.length >= 3 && i == theList.length - 1) {
        msg += ', and ';
      }
      else if (i == theList.length - 1) {
        msg += ' and ';
      }
      else if (i < theList.length) {
        msg += ', ';
      }
    }
	return msg;
  }
;

modify playerActionMessages
  // Override the standard "seems to be locked" message for two different scenarios:
  // (1) the door (or other padlockable) is locked by visible padlocks;
  // (2) the door (or other padlockable) is locked because of padlocks on its opposite
  // side. In either of these cases, let's make the "seems to be locked" message
  // a bit more descriptive.
  cannotOpenLockedMsg {
    if (gDobj && gDobj.attachedPadlocks && gDobj.attachedPadlocks.length > 0) {
      return '{You/he} tr{ies/ied} to open {the dobj/him}, but {it\'s dobj/him} locked by '
        + padlockMessages.sayList(gDobj.attachedPadlocks, &theName) + '. ';
    }
    if (gDobj && gDobj.otherSide && gDobj.otherSide.attachedPadlocks 
        && gDobj.otherSide.attachedPadlocks.length > 0) {
      return '{You/he} tr{ies/ied} to open {the dobj/him}, but {it dobj/he} seem{s/ed} to 
          be blocked by something on the other side. ';
    }
    return inherited;
  }
  
  // Improve the message that results when you detach a padlock as a consequence
  // of having typed UNLOCK.
  okayDetachFromMsg {
    local parentActionIsUnlock = nil;
    local parent = nil;
    if (gAction && gAction.parentAction) {
      parent = gAction.parentAction;
    }
    while (parent != nil) {
      if (parent.baseActionClass == UnlockAction || 
          parent.baseActionClass == UnlockWithAction) {
        parentActionIsUnlock = true;
      }
      parent = parent.parentAction;
    }
    
    if (!parentActionIsUnlock) {
      return inherited;
    }
    else {
      return 'Okay, {you/he} detach{es/ed} {the dobj/him} from {the iobj/him}. ';
    }
  }
;

