#charset "us-ascii"

/*
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *  Based on code provided by Michael J. Roberts.
 *
 *  This file is part of the PROTEUS, the TADS 3 Utility Classes Package
 *
 *  Reflect.t
 *
 *  Provides a basic implementation for mapping from symbol to string
 *  and string to symbol using the TADS 3 Symbol Table generated at pre-
 *  compile time.
 */

/* include the Proteus header */
#include "proteus.h"

/* ------------------------------------------------------------------------ */
/*
 *  Special Class Mappings.
 *
 *  It's easily extensible for other intrinsic classes or even user-
 *  defined classes - you simply put in a mapToSString() to produce a
 *  special mapping for each class for which you want a special 
 *  mapping.
 */

/*
 *  Modifications of Object, the Abstract Datatype from which all 
 *  other TADS objects derive.
 */
modify Object
{
    /*
     *  Map this object to a single-quoted string. The variable 
     *  argument list flgs is used to determine whether we are 
     *  mapping the object's symbol, obj reference, structure, 
     *  or value.
     *
     *  If a variable list argument is passed it should be of the format:
     *      mapControl
     *      objRefStyle
     *      strHtmlStyle
     *      
     */
    mapToSString([flgs]) 
    {
        switch(flgs.car())
        {
            case MapObjReference:
                return mapObjReferenceToSString(flgs.cdr()...);

            case MapObjStructure:
                return mapObjStructureToSString(flgs.cdr()...);

            case MapObjValue:
                return mapObjValueToSString(flgs.cdr()...);

            default:
                return mapObjReferenceToSString(flgs.cdr()...);
        }
    }

    mapObjReferenceToSString([flgs])
    {
        switch(flgs.car())
        {
            case MapObjRefSym:
                return mapObjRefSymToSString();

            case MapObjRefTag:
                return mapObjRefTagToSString();

            case MapObjRefAHref:
                return mapObjRefAHrefToSString();

            default:
                return mapObjRefSymToSString();
        }
    }

    /*
     *  Map this object's symbol to a single-quoted string. If the 
     *  symbol is not a key in the global symbols table then this 
     *  will return nil.
     */
    mapObjRefSymToSString() { return gSymbols.getSString(self); }

    /*
     *  Map this object's reference, obtained using 
     *  getObjTagAndSym(), to a single-quoted string.
     */
    mapObjRefTagToSString() { return getObjTagAndSym(); }

    /*
     *  Map this object's reference in aHref format to
     *  a single-quoted string.
     */
    mapObjRefAHrefToSString()
    {
        local href, txt;

        if (isDynamOrAnon())
        {
            return getObjTagAndSym();
        }
        else
        {
            href = gAHrefMetaCmd.getHref(getMetaCmdReference());
            txt = getObjTagAndSym();

            return aHref(href, txt);
        }
    }

    /*
     *  Map this object's structure to a single-quoted string. The
     *  object structure is generated using the getInherStrucList() method
     *  and the resulting list converted to string format.
     */
    mapObjStructureToSString([flgs])
    {
        local lst;

        lst = getInherStrucList();

        return lst.mapObjValueToSString(flgs...);
    }
    
    /*
     *  Map this object's value to a single-quoted string.
     */
    mapObjValueToSString([flgs])
    {
        switch(flgs.car())
        {
            case MapObjRefSym:
                return mapObjRefSymToSString();

            case MapObjRefTag:
                return mapObjRefTagToSString();

            case MapObjRefAHref:
                return mapObjRefAHrefToSString();

            default:
                return mapObjRefSymToSString();
        }
    }
}

#ifdef _BIGNUM_H_

modify BigNumber
{
    /*
     *  Map the BigNumber value to a single-quoted string
     *  limiting the value to 12 significant digits.
     */
    mapObjValueToSString([flgs]) { return formatString(12); }
}

#endif /* _BIGNUM_H_ */

modify ByteArray
{
    /*
     *  We map the ByteArray value using the CharsetDisplay characterset
     *  and the class' mapToString() method.
     */
    mapObjValueToSString([flgs]) 
    {
        local ch = getCharacterSet(CharsetDisplay);
        if (ch == nil)
            ch = 'utf8';
        return mapToString(new CharacterSet(ch));
    }
}

modify CharacterSet
{
    /*
     *  Map CharacterSet values using the class' getName() method.
     */
    mapObjValueToSString([flgs]) 
    {
        return getName();
    }
}

modify Collection
{
    /*
     *  Map Collection values by iterating across the collection
     *  and adding the values to a list object, then mapping the 
     *  list values to a single-quoted string.
     */
    mapObjValueToSString([flgs])
    {
        local len, lst = [];

        len = length();
        if (len == nil)
            len = 0;

        for (local i = 1; i <= len; ++i)
            lst += self[i];

        return lst.mapObjValueToSString(flgs...);
    }
}

modify List
{
    /* 
     *  Map List values by building a string composed of 
     *  the elements of the list as converted using the 
     *  gSymbols toSString() method.
     */
    mapObjValueToSString([flgs]) 
    {
        local len, str;
        
        str = '[';

        len = length();
        if (len == nil)
            len = 0;

        for (local i = 1; i <= len; ++i)
        {
            str += String.toSString(self[i], MapObjValue, flgs...);
            if (i < len)
                str += ', ';
        }

        str += ']';

        return str;
    }
}

modify LookupTable
{
    /*
     *  Map LookupTable values by combining key lists and value 
     *  lists into [key, value] subelements and then mapping the
     *  newly created list to a single-quoted string.
     */
    mapObjValueToSString([flgs])
    {
        local keyList, valList, newList = [], ele;

        keyList = keysToList();
        if (keyList == nil)
            keyList = [];

        valList = valsToList();
        if (valList == nil)
            valList = [];

        for (local i = 1; i <= keyList.length(); ++i)
        {
            ele = [];
            ele += keyList[i];
            ele += valList[i];

            newList += [ele];
        }

        return newList.mapObjValueToSString(flgs...);
    }
}

modify RexPattern
{
    /* 
     *  Map RexPattern values by retrieving the original 
     *  pattern string and then mapping the string using
     *  Htmlify arguments.
     */
    mapObjValueToSString([flgs])
    {
        local str;

        str = getPatternString();
        if (str == nil)
            str = '';

        return str.mapObjValueToSString(HtmlifyTranslateWhitespace);
    }
}

modify String
{
    /*
     *  This method maps val based on its dataType(). 
     */
    toSString(val, [flgs])
    {
        switch(dataType(val))
        {
            case TypeNil:
            case TypeTrue:
            case TypeInt:
                flgs = flgs.subset({x: (dataType(x) == TypeInt 
                    && x > 1 && x < 37)});
                return toString(val, flgs...);

            case TypeObject:
            case TypeSString:
            case TypeList:
                /* objects that use special intrinsic class mappings */
                return val.mapToSString(flgs...);

            case TypeProp:
                return '&' + gSymbols.getSString(val);
                
            case TypeFuncPtr:
            case TypeEnum:
                /* a value that might have a global symbol - look it up */
                return gSymbols.getSString(val);

            default:
                /* it's not a known type */
                return '????';
        }
    }

    /*
     *  Map String values to a single-quoted string.
     */
    mapObjValueToSString([flgs]) 
    {
        local str = '\'';
        str += htmlify(flgs.cdr()...);
        str += '\'';
        return str;
    }
}

modify Vector
{
    /*
     *  Map Vector values to a list, then map the list values
     *  to a single-quoted string.
     */
    mapObjValueToSString([flgs]) { return toList.mapObjValueToSString(flgs...); }
}

modify TadsObject
{
    /*
     *  
     */
    mapObjRefTagToSString() { return getObjTagAndSym(); }

    /*
     *  Map this object's reference in aHref form 
     *  to a single-quoted string.
     */
    mapObjRefAHrefToSString()
    {
        local href, txt;

        href = gAHrefMetaCmd.getHref(getMetaCmdReference());
        txt = getObjTagAndSym();

        return aHref(href, txt);
    }
}

/* 
 *  Modify the ProteusException class to display 
 * stack trace information.
 */
modify ProteusException
{
    stack_ = []

    construct()
    {
        local frList, str = '';
        
        frList = t3GetStackTrace();
        
        exceptionMessage += '\bStack Trace:';
        
        foreach (local fr in frList)
        {
            str += '\n\t';
            str += formatStackFrame(fr, true);
        }
        
        exceptionMessage += ' ';
        exceptionMessage += str;
        exceptionMessage += '\n';
    }
    
    /*
     *  Format a stack frame object (of class T3StackInfo).
     *
     *  "Borrowed" from Reflect.t  
     */
    formatStackFrame(fr, includeSourcePos)
    {
        local ret;
        
        /* see what kind of frame we have */
        if (fr.func_ != nil)
        {
            /* it's a function */
            ret = String.toSString(fr.func_);
        }
        else if (fr.obj_ != nil)
        {
            /* it's an object.property */
            ret = String.toSString(fr.self_) + '.' 
                + String.toSString(fr.prop_).substr(2);
        }
        else
        {
            /* no function or object - must be a system routine */
            ret = '(System)';
        }

        /* if it's not a system routine, add the argument list */
        if (!fr.isSystem())
        {
            /* add the open paren */
            ret += '(';

            /* add the arguments */
            for (local i = 1, local len = fr.argList_.length() ;
                 i <= len ; ++i)
            {
                /* if it's not the first one, add a comma */
                if (i != 1)
                    ret += ', ';

                /* add this value */
                ret += String.toSString(fr.argList_[i]);
            }

            /* add the close paren */
            ret += ')';

            /* if desired, add the source location */
            if (includeSourcePos && fr.srcInfo_ != nil)
                ret += ' ' + fr.srcInfo_[1] + ', line ' + fr.srcInfo_[2];
        }

        /* return the result */
        return ret;
    }
}