#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the PROTEUS, the TADS 3 Utility Classes Package
 *
 *  Structure comp.t
 *
 *  Provides a mechanism for comparing the structures of two objects.
 */

#include "proteus.h"
 
/*
 *  Function returns an enumerator based on a comparison of the two
 *  objects structures. 
 */
analyzeStructure(thisStructure, objStructure)
{
    local svThisStruc, svObjStruc;
    local baseStruc;
    
    svThisStruc = thisStructure;
    svObjStruc  = objStructure;
    
    /* the objects are structurally the same object */
    if (thisStructure == objStructure)
        return tautological;
        
    /*
     *  the objects might be different instances of the same
     *  classes, we remove the first element of the first
     *  subelement and compare.
     */
    thisStructure[1] = thisStructure[1].removeElementAt(1);
    objStructure[1]  = objStructure[1].removeElementAt(1);
    
    if (thisStructure == objStructure)
        return equivalent;
    /*
     *  the objects might share the same superclass list, but
     *  have disparate directly defined properties, we remove 
     *  the first subelement and compare.
     */
    thisStructure = thisStructure.removeElementAt(1);
    objStructure  = objStructure.removeElementAt(1);
    
    if (thisStructure == objStructure)
        return contingent;
        
    thisStructure   = svThisStruc;
    objStructure    = svObjStruc;
    
    local intsctList = thisStructure.intersect(objStructure);
    if (intsctList == thisStructure)
        return trait;
    else if (intsctList == objStructure)
        return extension;
    else 
    {
        /* get the structure of TadsObject */
        baseStruc   = TadsObject.getInherStrucList();
        
        /* remove TadsObject structure */
        intsctList -= baseStruc;
        
        if (intsctList.length() != 0)
            return shared;
    }
        
    return disparate;
}
    
modify Object
{
    /*
     *  Returns an analysis of the structure of thisObj when compared
     *  with that of obj. The return value will be one of the
     *  enumerators: tautological, equivalent, contingent, trait,
     *  extension, shared, or disparate.
     */
    hasStructureOf(obj)
    {
        local thisStructure, objStructure;
        
        thisStructure = getInherStrucList();
        objStructure  = obj.getInherStrucList();
        
        return analyzeStructure(thisStructure, objStructure);
    }
    
    /*
     *  Returns an analysis of the structure of thisObj when compared
     *  with that of obj. The return value will be one of the
     *  enumerators: tautological, equivalent, contingent, trait,
     *  extension, shared, or disparate.
     *
     *  This method only examines the object's state structure.
     */    
    hasStateStructureOf(obj)
    {
        local thisStructure, objStructure;
        
        thisStructure   = getInherStateStrucList();
        objStructure    = obj.getInherStateStrucList();
        
        return analyzeStructure(thisStructure, objStructure);
    }
    
    /*
     *  Method returns true if the state of thisObj is the same as the
     *  state of obj; otherwise returns nil. This is determined by
     *  comparing the property list of thisObj with the property list of
     *  obj, and then comparing the property list of obj with the
     *  property list of thisObj. 
     *
     *  This method doesn't take the object's state structure into its
     *  consideration. The two objects might have disparate structures,
     *  but have the same state.
     *
     *  The method also considers an object that defines a property
     *  as nil the same as an object that does not define the property.
     */
    hasStateOf(obj, [bypass])
    {
        local thisStructure;

        thisStructure   = getInherStateStrucList();
        
        for (local i = 1; i <= thisStructure.length(); ++i)
        {
            local propList;
            
            propList = thisStructure[i][2];
            
            foreach (local prop in propList)
            {
                if (bypass.indexOf(prop))
                    continue;
                    
                if (self.(prop) != obj.(prop)) 
                    return nil;
            }
        }
        
        thisStructure   = obj.getInherStateStrucList();
        
        for (local i = 1; i <= thisStructure.length(); ++i)
        {
            local propList;
            
            propList = thisStructure[i][2];
            
            foreach (local prop in propList)
            {
                if (bypass.indexOf(prop))
                    continue;
                    
                if (self.(prop) != obj.(prop)) 
                    return nil;
            }
        }
        
        return true;
    }

	/*
      *   Indicates whether some other object is "equal to" this one.
      *   One object is equal to another if they shar the same symbolic 
      *   ancestors and the two objects' states are equal.
      */
	equals(obj)
	{
        local v1, v2;

        if (isDynamOrAnon())
            v1 = getFirstSymAncestorList();
        else
            v1 = [] + self;

        if (obj.isDynamOrAnon())
            v2 = getFirstSymAncestorList();
        else
            v2 = [] + self;

		/* the two objects do not have the same symbolic traits */
		if (v1 != v2)
			return nil;

		/* compare the states of the two objects */
		return hasStateOf(obj);
	}

    /*
     *  Method returns a state delta list if the state of thisObj is not 
     *  the same as the state of obj; otherwise returns an empty list.
     *
     *  The structure of the list is [[prop,[thisObjVal, objVal], ...]
     *  
     *  This is determined by comparing the property list of thisObj 
     *  with the property list of obj, and then comparing the property 
     *  list of obj with the property list of thisObj. 
     *
     *  This method doesn't take the object's state structure into its
     *  consideration. The two objects might have disparate structures,
     *  but have the same state.
     *
     *  The method also considers an object that defines a property
     *  as nil the same as an object that does not define the property.
     */
    getStateDelta(obj, [bypass])
    {
        local thisStructure, propList;
        local thisVal, objVal, ele, delta = [];

        thisStructure   = getInherStateStrucList();
        
        for (local i = 1; i <= thisStructure.length(); ++i)
        {
            propList = thisStructure[i][2];
            
            foreach (local prop in propList)
            {
                if (bypass.indexOf(prop))
                    continue;
                    
                thisVal = self.(prop);
                objVal  = obj.(prop);
                ele     = [prop, [thisVal, objVal]];
                
                if (thisVal != objVal) 
                    delta = delta.appendUnique([ele]);
            }
        }
        
        thisStructure   = obj.getInherStateStrucList();
        
        for (local i = 1; i <= thisStructure.length(); ++i)
        {
            propList = thisStructure[i][2];
            
            foreach (local prop in propList)
            {
                if (bypass.indexOf(prop))
                    continue;
                    
                thisVal = self.(prop);
                objVal  = obj.(prop);
                ele     = [prop, [thisVal, objVal]];
                
                if (thisVal != objVal) 
                    delta = delta.appendUnique([ele]);
            }
        }
        
        return delta;
    }
}