
#       whoisd - whois Internet daemon
#
#	$RCSfile: whoisd.pl,v $
#	$Revision: 2.3 $
#	$Author: ambrose $
#	$Date: 1996/11/08 09:53:39 $
#

#
# Usage:
#
# whoisd [-V] [-D] [-p port] [[WhoisOptions] SearchString]
#
# where:
#
# -D       - run as daemon instead of from inetd 
#            Note: -D should always be set, inetd functionality is untested
# -p port  - bind to port 'port'
# -V       - run in debug/verbose mode
# 
# whoisd will not run as daemon if a SearchString is specified
# but instead perform a lookup for the SearchString
# WhoisOptions are the options that are used for the whois lookup,
# see 'whois' for more information on which options can be used.
#
#
# signals:
#
# *doesn't work (yet):*
#
# USR1 toggles the acceptance of network updates
# HUP  reread config file


PERL5OPTIONS

require "flush.pl";

$SOL_SOCKET = 0xffff;
$SO_REUSEADDR = 0x0004;

@INC = ("LIBDIR", @INC);

require "rconf.pl";

require "dbopen.pl";
require "dbclose.pl";
require "dbmatch.pl";
require "cldb.pl";

require "enread.pl";
require "enwrite.pl";
require "enkeys.pl";

require "misc.pl";
require "serial.pl";

require "syslog.pl";
require "template.pl";
require "whoisqry.pl";
require "processdata.pl";

#
# If we get a SIGALRM, exit. Used in the read loop, to make sure people
# do not keep the connection open, and open and open ....

sub alarmhandler {

   &exitwhoisd(0,"% Timeout... Closing connection");

}

sub quithandler {

   &exitwhoisd(0,"% Server has been (administratively) shutdown... Closing connection\n% Please try again later");

}

$QUIT=0;

sub quitdaemonhandler {
   
   $QUIT=$_[0];
   
}

$NOUPDATES=0;

sub toggleupdates {

    $NOUPDATES=!$NOUPDATES;
    
    if ($NOUPDATES) {
       &syslog(AUDITLOG, "stopped network updates");
    }
    else {
       &syslog(AUDITLOG, "restarted network updates");
    }

}


sub exitwhoisd {
   local($exitcode, $msg)=@_;
   
   local($*)=1;
    
   $msg=~ s/^\n+//;
   $msg=~ s/\n+$//;
   
   if ((!$opt_D) || (@ARGV)) {
      print STDOUT "\n", $msg, "\n\n\n" if ($msg);
   }
   else {
      print NS "\n", $msg, "\n\n\n" if ($msg);
      &flush(NS);
      close(NS);
   }
   
   exit($exitcode);
   
}

#
# sometimes we need to exit earlier ...

sub exitonerror {
   local($msg) = @_;

   local($*)=1;
    
   $msg=~ s/^\n+//;
   $msg=~ s/\n+$//;
   $msg=~ s/\n/\n\% ERROR\: /g;

   &exitwhoisd(0, "\% ERROR: ".$msg);
   
}



#
# check for authorized access...
#

sub allowupdate {
    local($rhost,*accesslist)=@_;
    
    foreach $updfrom (@accesslist) {
    
       print STDERR "($$) checking $rhost=~ /", $updfrom, "/\n" if $opt_V;
       
       if (($updfrom!~ /^\s*$/) &&
           ($rhost=~ /^$updfrom$/)) {
          
          print STDERR "($$) update from: $rhost accepted, regular: $updfrom\n" if $opt_V;
          
          return 1;
       
       }
       
    }
    
    print STDERR "($$) update from: $rhost rejected\n" if $opt_V;
    
    return 0;
}

#
# Here is some guess work about what file to open....

sub findsplitdbs {
    local(*keys, @preselection)=@_;
    
    local(@searchdb)=();
    
    # print STDERR join(" ", keys %CLASSLESSDBS), "\n";
    
    @preselection=grep($CLASSLESSDBS{$_}, @preselection) if (scalar(grep(/^$VALIDPREFIXKEY$/o, @keys))==scalar(@keys));
    
    #
    # and now we are going to do something really smart...
    # and trickey things
             
    local(@upperkeys)=@keys;
    grep(tr/a-z/A-Z/, @upperkeys);
    
    #            
    # see comment below why we need this
    local(@tmpkeys,@tmpupperkeys);
    local($tmp);
                
    foreach $j (@preselection) {
                   
       #
       # we need to do this since we might have changed 
       # the @keys by the grep's and calling routines
                   
       @tmpkeys=@keys;
       @tmpupperkeys=@upperkeys;   
                   
       # print STDERR "db: ",$j, "\n";
       
       if ($j eq "rt") {
             next if (grep($_ !~ /^$VALIDIP4PREFIXKEY$/o, @tmpkeys));
                      
             push(@searchdb, $j);
             next;
                      
       }
       elsif ($j eq "in") {
                      
          next if (grep(!(($tmp=$_) &&
                        (($tmp=~ /^$VALIDIP4PREFIXKEY$/o) ||
                        (&isnetname($tmp)))), @tmpupperkeys));
                      
                  push(@searchdb, $j);
                  next;
                      
       }
       elsif ($j eq "i6") {
          
          #print STDERR "keys: ", join(" ", @tmpupperkeys), "\n";
          #print STDERR "keys: ", $tmpupperkeys[0]=~ /^[\:\dA-Z]*[\:][\:\dA-Z]+(\/\d+)?$/, "\n";
                      
          next if (!(grep(/^$VALIDIP6PREFIXKEY$/, @tmpkeys) ||
                     grep(&isnetname($tmp), @tmpupperkeys))) ;
                      
                  push(@searchdb, $j);
                  next;
                      
       }
       elsif ($j eq "ir") {
             next if (grep(!((/^$VALIDIP4PREFIXKEY$/o) ||
                             (&isdomname($_))), @tmpkeys));
                      
             push(@searchdb, $j);
             next;
                                      
       }            
       elsif ($j eq "an") {
          next if (grep(!&isasnum($_), @tmpupperkeys));
                      
          push(@searchdb, $j);
          next;

       }
       elsif ($j eq "am") {
          next if (grep(!&isasmacro($_), @tmpupperkeys));
                      
          push(@searchdb, $j);
          next;

       }
       elsif ($j eq "cm") {
          next if (grep(!&iscommunity($_), @tmpupperkeys));
                      
          push(@searchdb, $j);
          next;

       }
       elsif ($j eq "dn") {
          next if (grep(!&isdomname($_), @tmpkeys));
                      
          push(@searchdb, $j);
          next;

       }
       elsif ($j=~ /^pn|ro$/) {
                      
             #print STDERR "-", scalar(grep(&isname($_), @tmpkeys));
             #print STDERR "-", scalar(grep(&isnichandle($_), @tmpupperkeys));
             #print STDERR "-", scalar(@keys), "-\n";
                      
             next if (grep( !(($tmp=$_) &&
                              ((&isname($tmp)) ||
                               (&isnichandle($tmp)) ||
                               (&isemail($tmp)))), @tmpupperkeys));
                                
             push(@searchdb, $j);
             next;
                                
       }
       elsif ($j eq "mt") {
          next if (grep(!&ismaintainer($_), @tmpupperkeys));
                                            
          push(@searchdb, $j);
          next;
                      
       }
       elsif ($j eq "dp") {
          next if (grep(!&isdomname($_), @tmpkeys));
                      
          push(@searchdb, $j);
          next;
                      
       }
       elsif ($j eq "li") {
          next if (grep(!&islimerick($_), @tmpupperkeys));
                      
          push(@searchdb, $j);
          next;
                      
       }
                   
       # print STDERR "-$j-";
                   
       push(@searchdb, $j);
                
    }
 
    return @searchdb;
                
}

sub referenceorder {
    
    #
    # check if we are actually referenced once ?
    
    if (($ALLPOINTSTO!~ /(^| )$a( |$)/) && ($ALLPOINTSTO!~ /(^| )$b( |$)/)) {
       
       return $ATTL{$a} cmp $ATTL{$b};
       
    }
    
    return -1 if ($ALLPOINTSTO!~ /(^| )$a( |$)/);
    
    return  1 if ($ALLPOINTSTO!~ /(^| )$b( |$)/);
        
    local($order)=0;
    
    local($pointstoa)=join(" ", " ", @POINTSTO{split(/ /, $RECUR{$a})}, " ");
    local($pointstob)=join(" ", " ", @POINTSTO{split(/ /, $RECUR{$b})}, " ");
    $pointstoa=~ s/ $a//g;
    $pointstob=~ s/ $b//g;
    
    # print "$a - $b = $pointstoa - $pointstob\n";
    
    #
    # check if we reference others ?
    
    return $ATTL{$a} cmp $ATTL{$b} if (($pointstoa=~ /^ +$/) && ($pointstob=~ /^ +$/));
    
    return  1 if  ($pointstoa=~ /^ +$/);
    
    return -1 if  ($pointstob=~ /^ +$/);
    
    $order-- if ($pointstoa=~ / $b /);
       
    $order++ if ($pointstob=~ / $a /);
    
    return $ATTL{$a} cmp $ATTL{$b} if ($order==0);
        
    return $order;
    
}


#
# lookupandprint - will find all matches for all keys, and will output
#                the objects found, if they indeed match all the keys.
#                will also generate an array with keys that should be
#                looked up recursively because they are referenced in
#                the printed objects
#
# Exit codes (set in $result):
#         0 - no match (if $result was not defined yet)
#         1 - OK, something was output (always)

sub lookupandprint {
    local($output, *db, *keys, *references, $types, *result, $options)=@_;
    
    #
    # it makes the code easier if we call with empty lists...
    
    # print STDERR "db contents: ", $db{"bl2-test"}, "\n";
    
    print STDERR "($$) lookupandprint - keys: ", join("*", @keys), " options: $options\n" if ($opt_V);
    
    return 0 if (!@keys);
    
    local(%entry)=();
    local($printed)=0;
    
    local($i,$key,$offset,$attribute,$newtype,@object);
    
    # print STDERR "($$) lookupandprint - keys: ", join(",",@keys), " options: $options types: $types result: $result fast: ",$options & $FASTOPTION," nonrecursive: ",$options & $NONRECURSIVEOPTION,"\n" if $opt_V;

    foreach $offset (sort { $a <=> $b } &dbmatch(*db, *keys, $types, $INTERSECTIONOPTION | ($options & ($ALLLESSSPECIFICOPTION | $MORESPECIFICOPTION | $ALLMORESPECIFICOPTION | $EXACTMATCHOPTION)))) {
       
       print STDERR "offset: ", $offset, " keys: ", join("*", @keys), "\n" if ($opt_V);
       
       $key=join("\t", $offset, $db[1]);
       
       next if ($DISPLAYED{$key});
       
       $DISPLAYED{$key}=1;
       
       alarm $KEEPOPEN;
       
       if ($options & $FASTOPTION) {
          
          #
          # enable paragraph mode
          
          local($/)="";
          
          seek($db, $offset, 0);
          @object=split(/\n+/, scalar(<db>));
          
          foreach (@object) {
             
             if ((/^ *\*?([^\*\s][^\:\s]*) *\:/) && ($OBJATSQ{$ATTR{$1}})) {
                
                $newtype=$ATTR{$1};
                
                last;
                
             }
             
          }
          
          if (($newtype) && ($types=~ /(^| )$newtype( |$)/)) {
          
             print $output "\n", join("\n", @object), "\n";
             
             $printed=1;
          
          }
           
       }
       else {

          $newtype=&enread($db, *entry, $offset);
          
          print STDERR "$newtype - $types\n" if ($opt_V);
          
          if ($types=~ /(^| )$newtype( |$)/) {
          
             # print STDERR "$newtype - $types\n";
          
             &enwrite($output, *entry, 1, 0, !($options & $NOSYNTACTICSUGAR));
             
             $printed=1;
          
          }
       
       }
       
       
       # print STDERR "recursive: $opt_r ", $newtype, $RECUR{$newtype}, $options, $NONRECURSIVEOPTION, $options & $NONRECURSIVEOPTION,"\n";
       
       if (($RECUR{$newtype}) && (!($options & $NONRECURSIVEOPTION))) {
          
          foreach $attribute (split(/ /, $RECUR{$newtype})) {
             
             foreach (split(/ /, $POINTSTO{$attribute})) {
                
                #print STDERR "attr: $attribute ent: $entry{$attribute} pointto: $_\n";
                
                if ($references{$_}) {
                   $references{$_}=join("\n", $references{$_}, $entry{$attribute}) if (($_) && ($entry{$attribute}));
                }
                else {
                   $references{$_}=$entry{$attribute} if (($_) && ($entry{$attribute}));
                }
                
             }
             
          }
          
          # print STDERR "keys: ", join(" ", keys %references), "\n";
          # print STDERR "values: ", join(" ", values %references), "\n";
          
             
       }
       
    }
    
    $result=1 if ($printed);
    
    return $printed;
    
}

#
# whois - main lookup loop. will output all objects found for all sources
#                requested. will also process the recursive lookups generated
#                by lookupandprint()
#

sub whois {
    local($input, $output, $searchstring, $name, $rhost) = @_;
    
    if ($opt_t) {
       
       $opt_t = $ATTR{$opt_t} if $ATTR{$opt_t};

       alarm $KEEPOPEN;
       
       if ($OBJATSQ{$opt_t}) {
          &Template($output,$opt_t);
       }
       else {
# Patch from David, 961010, replaced this; AMRM, 961010
#          print $output "\n% No template available for object \"$opt_t\"";
          print $output "\n% No template available for object \"$opt_t\"\n";
       }

       return 1;

    }
    elsif ($opt_g) {
       
       if ($opt_D) {
          &dogetserials($output,$opt_g,$name,$rhost);      
          return 1;
       }
       else {
          &exitonerror("***This server is not able to provide updates***\n***please contact \<$HUMAILBOX\> for more information***");
       }
       
    }
    elsif ($searchstring=~ /^\s*(HELP|HOWTO)\s*$/i) {
       open (HELP, $HELP);
       alarm $KEEPOPEN;
       print $output <HELP>, "\n";
       close(HELP);
       return 1;
    }
    elsif ($opt_U) {
        
       &whoisupdate($input, $output, $name, $rhost, $opt_U);
       return 1;
       
    }
    else {

       local(%dummyentry)=();
       local($result)=0;
       local($whoisdb)='whoisdb';
       local(%whoisdb,@whoisdb);
       
       local(%references)=();
       local(@references)=();
       local(@splitdbs)=();
       
       local($source,$types);
       local(%doubles,@splitdbs,@sources,@searchdb,@keys,@longreferences,@longkeys);
       
       #
       # global var for the objects that have already been shown
       
       %DISPLAYED=();
       
       if ($opt_a) {
          @sources=split(/ /, $ALLLOOK);
       }
       elsif ($opt_s) {
          @sources=split(/\,/, $opt_s);
       }
       else {
          @sources=split(/ /, $DEFLOOK);
       }
 
       print STDERR "($$) whois - sources: ", join(",",@sources), "\n" if $opt_V;
       
       if ($opt_i) {
          
          local(%done)=();
          
          local(@tmp,@pointsto,$db);
          
          print STDERR "opt_T: $opt_T opt_i: $opt_i\n" if ($opt_V);
          
          if ($opt_T) {
             @tmp=split(/ /, $opt_T);
          }
          else {
             @tmp=keys %OBJATSQ;
          }
          
          print STDERR "opt_T: $opt_T opt_i: $opt_i tmp:",join(" ",@tmp),"\n" if ($opt_V);   
          
          @splitdbs=();
          
          @pointsto=split(/ /, $opt_i);
          
          foreach $db (@tmp) {
             
             next if ($done{$db});
             $done{$db}++;
             
             push(@splitdbs, $db) if (grep($OBJATSQ{$db}=~ /(^| )$_( |$)/ ,@pointsto));
             
          }
          
          print STDERR "splitdbs: ", join(" ", @splitdbs), "\n" if ($opt_V);
          
       }
       else {
          
          @splitdbs=&findsplitdbs(*keys);

          @keys=&makekeys($searchstring, "", *longkeys);
       
          print STDERR "($$) whois - searchstring: -$searchstring- keys: ", join("*", @keys), " longkeys: ", join("*", @longkeys), "\n" if ($opt_V);
          
          return 0 if (!@keys);
          
          if ($opt_T) {
             @splitdbs=&findsplitdbs(*keys, split(/ /, $opt_T));
          }
          else {
             @splitdbs=&findsplitdbs(*keys, keys %OBJATSQ);
          }
       
          return 0 if (!@splitdbs);
          
       }
       
       $opt_T=join(" ", @splitdbs);
       
       local(%files)=();
       
       foreach $source (@sources) {
          
          #
          # we might have different sources in one dbfile
          # so skip the source if we already did this dbfile
          
          next if ($files{$source} eq $DBFILE{$source});
          $files{$source}=$DBFILE{$source};
          
          if ($SPLIT{$source}) {
             @searchdb=@splitdbs;
          }
          else {
             @searchdb=($source);
          }
         
          # print STDERR "keys: ", join(" ", keys(%OBJATSQ)), "\n";
          # print STDERR "oldorder: ", join(" ", keys(%OBJATSQ)), "\n";
          # print STDERR "order: ", join(" ", sort referenceorder keys(%OBJATSQ)), "\n";
         
          foreach $j (sort referenceorder @searchdb) {
             
             $dbfile=$DBFILE{$source};
             $dbfile.=".".$j if ($SPLIT{$source});    
             
             if ($opt_i) {
                
                if ($SPLIT{$source}) {
                   
                   @keys=&makekeys($searchstring, join(" ",grep($OBJATSQ{$j}=~ /(^| )$_( |$)/, split(/ /, $opt_i))), *longkeys);
                
                }
                else {
                   
                   @keys=&makekeys($searchstring, join(" ",grep($ALLPOINTSTOATTR=~ /(^| )$_( |$)/, split(/ /, $opt_i))), *longkeys);
                   
                }
                
             }
             
             next if ((!@keys) || (!&dbopen(*whoisdb, *dummyentry, 0, $dbfile)));
             &dbclopen(*dummyentry, 0, $dbfile) if ($CLASSLESSDBS{$j});
            
             print STDERR "($$) whois - search $dbfile\n" if ($opt_V);
             
             if ($opt_i) {
                
                if ($SPLIT{$source}) {
                   
                   foreach (grep($OBJATSQ{$j}=~ /(^| )$_( |$)/, split(/ /, $opt_i))) {
                      @keys=&makekeys($searchstring, $_, *longkeys);
                      &lookupandprint($output, *whoisdb, *longkeys, *references, $opt_T, *result, $opt_F | $opt_L | $opt_m | $opt_M | $opt_r | $opt_S | $EXACTMATCHOPTION) ||
                      &lookupandprint($output, *whoisdb, *keys, *references, $opt_T, *result, $opt_F | $opt_L | $opt_m | $opt_M | $opt_r | $opt_S | $EXACTMATCHOPTION);
                   }
                
                }
                else {
                   
                   foreach (grep($ALLPOINTSTOATTR=~ /(^| )$_( |$)/, split(/ /, $opt_i))) {
                      @keys=&makekeys($searchstring, $_, *longkeys);
                      &lookupandprint($output, *whoisdb, *longkeys, *references, $opt_T, *result, $opt_F | $opt_L | $opt_m | $opt_M | $opt_r | $opt_S | $EXACTMATCHOPTION) ||
                      &lookupandprint($output, *whoisdb, *keys, *references, $opt_T, *result, $opt_F | $opt_L | $opt_m | $opt_M | $opt_r | $opt_S | $EXACTMATCHOPTION);
                   }
                   
                }
                
             }
             else {
                &lookupandprint($output, *whoisdb, *longkeys, *references, $opt_T, *result, $opt_F | $opt_L | $opt_m | $opt_M | $opt_r | $opt_S | $EXACTMATCHOPTION) ||
                &lookupandprint($output, *whoisdb, *keys, *references, $opt_T, *result, $opt_F | $opt_L | $opt_m | $opt_M | $opt_r | $opt_S | $EXACTMATCHOPTION);
             }
             
             # print STDERR "($$) whois - references: ", join(" ", %references), " result: $result\n";
             
             if (($SPLIT{$source}) && ($references{$j})) {
          
                # print STDERR "($$) whois - recursion: ", $references{$j}, "\n";
          
                @references=();
                %doubles=();
                   
                foreach (split(/\n+/, delete($references{$j}))) {
                   
                   next if ($doubles{$_});
                   $doubles{$_}++;
                   
                   print STDERR "lookup reference: ", $_, "\n" if ($opt_V);
                   
                   @references=&makekeys($_, "", *longreferences);
                   
                   # print STDERR &findsplitdbs(*references, keys %OBJATSQ);
                   
                   if (grep($j eq $_, &findsplitdbs(*references, keys %OBJATSQ))) {
                      &lookupandprint($output, *whoisdb, *longreferences, *references, $j, *result, $opt_F | $opt_L | $opt_m | $opt_M | $NONRECURSIVEOPTION | $opt_S | $EXACTMATCHOPTION) ||
                      &lookupandprint($output, *whoisdb, *references, *references, $j, *result, $opt_F | $opt_L | $opt_m | $opt_M | $NONRECURSIVEOPTION | $opt_S | $EXACTMATCHOPTION)
                   }
                   
                }
                
             }
             
             
             if ($SPLIT{$source}) {
                
                &flush($output);
                
                &dbclclose() if ($CLASSLESSDBS{$j});
                &dbclose(*whoisdb);  
             
             }
          
          }
          
          if (%references) {
          
             foreach $type (keys %references) {
             
                %doubles=();
                
                if ($SPLIT{$source}) {
                
                   $dbfile=$DBFILE{$source}.".".$type;
                   
                   next if (!&dbopen(*whoisdb, *dummyentry, 0, $dbfile));
                
                   &dbclopen(*dummyentry, 0, $dbfile) if ($CLASSLESSDBS{$type});
                
                }
                
                foreach (split(/\n+/, delete($references{$type}))) {
                
                   next if ($doubles{$_});
                   $doubles{$_}++;
                   
                   @references=&makekeys($_, "", *longreferences);

                   &lookupandprint($output, *whoisdb, *longreferences, *references, $type, *result, $opt_F | $opt_L | $opt_m | $opt_M | $NONRECURSIVEOPTION | $opt_S | $EXACTMATCHOPTION) ||
                   &lookupandprint($output, *whoisdb, *references, *references, $type, *result, $opt_F | $opt_L | $opt_m | $opt_M | $NONRECURSIVEOPTION | $opt_S | $EXACTMATCHOPTION)
                   
                }
                
                if ($SPLIT{$source}) {
                   
                   &flush($output);
                
                   &dbclclose() if ($CLASSLESSDBS{$type});
                   &dbclose(*whoisdb);
                   
                }
                
             }
                
          }
          
          if (!$SPLIT{$source}) {
          
             &flush($output);
          
             &dbclclose() if ($CLASSLESSDBS{$source});
             &dbclose(*whoisdb);
             
          }
       
       }   

       return $result;
       
    }
    
    print STDERR "($$) left whois\n" if $opt_V;
}

sub clearflags {

    $opt_a=0;
    $opt_g="";
    $opt_i="";
    $opt_F=0;
    $opt_L=0;
    $opt_m=0;
    $opt_M=0;
    $opt_r=0;
    $opt_s="";
    $opt_S=0;
    $opt_t="";
    $opt_T="";
    $opt_Type=0;
    $opt_U="";
    $opt_Version="";

}

sub getflags {
    
    local(@flags)=();
    
    push(@flags, "a") if ($opt_a);
    push(@flags, "g") if ($opt_g);
    push(@flags, "i") if ($opt_i);
    push(@flags, "F") if ($opt_F);
    push(@flags, "k") if ($opt_k);
    push(@flags, "L") if ($opt_L);
    push(@flags, "m") if ($opt_m);
    push(@flags, "M") if ($opt_M);
    push(@flags, "r") if ($opt_r);
    push(@flags, "s") if ($opt_s);
    push(@flags, "S") if ($opt_S);
    push(@flags, "t") if ($opt_t);
    push(@flags, "T") if ($opt_Type);
    push(@flags, "U") if ($opt_U);
    push(@flags, "V".$opt_Version) if ($opt_Version);
    
    return join("\:", @flags);

}
     
                        
#
# parse - parses the command line string for special options and sets
#                appropriate variables

sub parse {
    local($string)=@_;

    print STDERR "($$) got in parse\n" if $opt_V;

    #
    # Reset all command line arguments, except -k

    &clearflags();
    
    $string=~ s/\s+/ /g;
    $string=~ s/^ //;
    $string=~ s/ $//;
    
    while ($string=~ /^-/) {
       
       if ($string=~ s/^\-(\S*[arkFLmMS]+\S*) *//) {
          
          local($option)=$1;
          
          $opt_a=1 if ($option=~ s/a//g);
          $opt_r=$NONRECURSIVEOPTION if ($option=~ s/r//g);
          $opt_k=1 if ($option=~ s/k//g);
          $opt_F=$FASTOPTION if ($option=~ s/F//g);
          $opt_L=$ALLLESSSPECIFICOPTION if ($option=~ s/L//g);
          $opt_m=$MORESPECIFICOPTION if ($option=~ s/m//g);
          $opt_M=$ALLMORESPECIFICOPTION if ($option=~ s/M//g);
          $opt_S=$NOSYNTACTICSUGAR if ($option=~ s/S//g);
          
          $string="-".$option." ".$string if ($option!~ /^ *$/);
          
          next;
          
        }
        
        if ($string =~ s/^\-g +(\S+) *//) {
            $opt_g = $1;
            next;
        }
        
        if ($string=~ s/^\-i +(\S+) *//) {
           
           if ($opt_i) {
              $opt_i=join(" ", $opt_i, split(/\,/, $1));
           }
           else {
              $opt_i=join(" ", split(/\,/, $1));
           }
           
           next;
           
        }
         
        if ($string =~ s/^\-U +(\S+) +(\S+) *//) {
        
           $opt_U = $1." ".$2;
           
           next;
           
        }
                
# Patch from David replaced this, 961010, AMRM.
#
#        if ($string =~ s/^\-(s) +(\S+) *//) {
#            local($src) = $2;
#
#            $src=~ tr/a-z/A-Z/;
#            $src=join(",", split(/\,+/, $src));
#            
#            if ($opt_s) {
#               $opt_s=join(",",$opt_s,$src);
#            }
#            else {
#               $opt_s=$src;
#            }
#            
#            next;
#        }

	 if ($string =~ s/^\-s +(\S+) *//) {
            local($src)=$1;
            
            $src=~ tr/a-z/A-Z/;
            
            local(@requestedsources)=split(/\,+/, $src);
            local(@acceptedsources)=grep($DBFILE{$_}, @requestedsources);
            
            if (scalar(@requestedsources)!=scalar(@acceptedsources)) {
               
               local(@notacceptedsources)=grep(!$DBFILE{$_}, @requestedsources);
               local($availablesources);
               ($availablesources=$ALLLOOK) =~ s/ /\,/g;
               
               alarm $KEEPOPEN;
               print $output "\n% Ignored \"".join(",",@notacceptedsources)."\",
% Choose sources from: \"".$availablesources."\"\n";  
               &flush($output);
               
            }
            
            if ($opt_s) {
               $opt_s=join(",",$opt_s,@acceptedsources) if (@acceptedsources);
            }
            else {
               $opt_s=join(",", @acceptedsources) if (@acceptedsources);
            }
            
            next;
        }
            
        
        if ($string =~ s/^\-V *([a-zA-Z]+\d+[\d\.]*) *//) {
            $opt_Version=$1;
            next;
        }
        
        if ($string =~ s/^\-T +(\S+) *//) {
           
           local($type);
           local(@onlysearch)=();
           
           foreach $type (split(/\,+/,$1)) {
            
              $type=$ATTR{$type} if $ATTR{$type};
           
              if ($OBJATSQ{$type}) {
              
                 push(@onlysearch, $type);
                 
              }
              else {
# Patch from David replaced the following, 961010, AMRM.                 
#                 alarm $KEEPOPEN;
#                 print $output "\n% Request for unknown object type \"$ATTL{$type}\" ignored";  
#                 &flush($output);
             	   
		$type=$ATTL{$type} if ($ATTL{$type});
                 alarm $KEEPOPEN;
                 print $output "\n% Request for unknown object type \"$type\" ignored\n";
                 &flush($output);

              }
              
           }
           
           if ($opt_T) {
              $opt_T=join(" ", @onlysearch);
           }
           else {
              $opt_T=join(" ", @onlysearch);
           }
           
           $opt_Type=1;
           
           next;
           
        }
        
        if ($string =~ s/^\-t +(\S+) *//)  {
            $opt_t=$1;
            
            next;
            
        }
        
        last;
        
    }
    
    if (!$LOWPRIORITY) {
       
       $LOWPRIORITY=1;
       
       if ((!$opt_F) && (($opt_k) || ($opt_m) || ($opt_M))) {
           
          alarm $KEEPOPEN;
          print $output "\n% Server is running at low priority for -M, -m and -k queries\n";
          &flush($output);
          
       }
    
       system("$RENICECMD 10 $$ > /dev/null 2>/dev/null") if (($opt_k) || ($opt_m) || ($opt_M) || ($opt_g) || ($opt_U));
    
    }
    
    print STDERR "($$) left parse, searchstring: $string\n" if $opt_V;

    if ($opt_i) {
       
       # print STDERR "opti: $opt_i\n";
       
       local($type);
       
       local(%allobjects)=();
       local(%allattributes)=();
       
       foreach $type (split(/ /, $opt_i)) {
       
          if ($ATTR{$type}) {
             
             $type=$ATTR{$type};
             
             if ($OBJATSQ{$type}) {
                
                $allobjects{$type}=1;
             
             }
             elsif ($POINTSTO{$type}) {
                
                $allattributes{$type}=1;
                
             }
             else {
                    
                alarm $KEEPOPEN;
                print $output "\n% ignored attribute \"$ATTL{$type}\" since it is not indexed as a reference\n";  
                &flush($output);
                    
             }
                 
          }
          else {
                 
             alarm $KEEPOPEN;
             print $output "\n% unknown attribute type \"$ATTL{$type}\" ignored\n";  
             &flush($output);
              
          }
      
       }
       
       if (%allobjects) {
          
          local($attribute);
          local(%objecthasreference)=();
          
          foreach $attribute (keys %ATTR) {
             
             foreach (keys %allobjects) {
                if ($POINTSTO{$attribute}=~ /(^| )$_( |$)/) {
                   $allattributes{$attribute}=1;
                   $objecthasreference{$_}=1;   
                }
             }
             
          }
          
          foreach (keys %allobjects) {
             
             if (!$objecthasreference{$_}) {
                
                alarm $KEEPOPEN;
                print $output "\n% ignored \"$ATTL{$_}\" since it is not indexed as a reference by other objects\n";  
                &flush($output);
                
             }
             
          }
       
       }        
          
       $opt_i=join(" ", keys %allattributes);
       
       # print STDERR "opti: $opt_i\n";
       
       if ($opt_i=~ /^\s*$/) {
          $name.="(".$rhost.")" if ($name ne $rhost);
          &syslog("QRYLOG","($$) [".&getflags()."] 0 $name no attributes specified for -i query");
          &exitwhoisd(0, "% no attributes specified for -i query");  
       }
       
    }

    print STDERR "($$) whoisd called with flags:".&getflags()."\n" if ($opt_V);
    
    return $string;
    
}


sub whoisupdate {
    local($input, $output, $name, $rhost, $logstr) = @_;
        
    print STDERR "($$) whoisupdate(name:'$name',ip:'$rhost',log:'$logstr') called\n" if $opt_V;
    
    alarm $KEEPOPEN;
    
    if (&allowupdate($rhost,*WHOISUPDFROM)) {
       
       local($line);
       local($ent)="";
       
       print STDERR "($$) whoisupdate accepted\n" if $opt_V;
       
       while ((! -e $UPDATELOCK) && ($line=<$input>) && ($line!~ /^\s*\.\s*$/)) {
          
          alarm $KEEPOPEN;
          
          if ($line=~ /^\s*$/) {
             
             if ($ent=~ /\S+/) {
                
                print STDERR "($$) whoisupdate got entry" if $opt_V;
                
                alarm 0;

                local($OLDTIME)=&MakeRegular($TIME);
                
                ($DATE,$TIME)=&getYYMMDDandHHMMSS();
    
                if (!$FROMHOST) {
                
                   #
                   # no shell spoofing
                
                   $logstr =~ s/[^\w\.\-\s]//g;
                   
                   $FROMHOST=$name;
                   $FROMHOST.="(".$rhost.")" if ($name!~ /^\d+\.\d+\.\d+\.\d+$/);
                   
                   # print STDERR "update from: ", $FROMHOST, "\n";
                   
                   $logstr=~ s/^\s+//;
                   $logstr=~ s/\s+$//;
                   $logstr=~ s/\s+/\@/;
                   $NETWORKUPDATE=$FROMHOST." ".$logstr;
                   
                   &ReplaceGlobalVars(*FWNETWORKTXT);
                   &ReplaceGlobalVars(*NOTINETWORKTXT);
                   
                }
                else {
                  $FWNETWORKTXT=~ s/$OLDTIME/$TIME/;
                  $NOTINETWORKTXT=~ s/$OLDTIME/$TIME/;
                }
                
                $opt_A=0;
                $opt_m="";
                $opt_M=0;
                $opt_T=0;
                $opt_v=1;
                
                &dbupdate($ent, $output);
                &flush($output);
                                
                alarm $KEEPOPEN;
                
                $ent="";
             
             }
          
          }
          else {
             
             print STDERR "($$) whoisupdate got line: $line" if $opt_V;
             
             $ent.=$line;
          
          }
   
       }
       
       if (-e $UPDATELOCK) {
          alarm $KEEPOPEN;
          print $output "\n% Network updates are temporarely disabled, please try again later\n";
       }
       
       # give a chance to the client to close the connection before we do...
       
       sleep 1;
       
   }
   else {
   
      &syslog("AUDITLOG", "Network update authorization failure: $name ($rhost) $logstr");
      &exitonerror("***You are not authorized to do network updates***");
   
   }
   
}


sub GetVersionOne {
    local($output, $version, $source, $from, $to)=@_;

    local($i)=$from;
          
    alarm $KEEPOPEN;      
    print $output "\n\%START Version: $version $source $from-$to\n\n";
    
    $i=$from;
    
    local($basename)=$LOGFILE{"SERIALDIR"}.$source.".";   
    local($oldbasename)=$LOGFILE{"OLDSERIALDIR"}.$source.".";   
    
    # print STDERR "base: $basename oldbase: $oldbasename from: $i to: $to\n";
       
    while ($i<=$to) {
       
       if (-f $basename.$i) {
          open(INP,"<".$basename.$i);
       }
       else {
          open(INP,"<".$oldbasename.$i);
       }
       
       &lock(INP);
       @input=<INP>;
       &unlock(INP); close(INP);
       
       alarm $KEEPOPEN;
       print $output @input, "\n";
       
       $i++;
    }
    
    alarm $KEEPOPEN;   
    print $output "\%END ", $source, "\n";
} 

sub dogetserials {
    local($output,$option,$name,$rhost)=@_;
    
    print STDERR "($$) dogetserials - string: $string name: $name rhost: $rhost access: ",join(" ", @GETUPDATESFROM),"\n" if $opt_V;
        
    if (&allowupdate($rhost,*GETUPDATESFROM)) {
   
       if ($option=~ /^([\w\.\-]+)\:(\d+)\:(\d+)\-(\S+)$/) {
          local($source)=$1;
          local($version)=$2;
          local($from)=$3;
          local($to)=$4;
               
          local($i);
               
          local($first)=&getoldestserial($source);
          local($last)=&getcurrentserial($source);
               
          print STDERR "from: $from to: $to first: $first last: $last\n" if $opt_V;
          
          if ($from=~ /^LAST$/i) {
             $from=$last;
          }
          
          if ($to=~ /^LAST$/i) {
             $to=$last;
             if ($from==scalar($to+1)) {
                &syslog("QRYLOG","($$) [g] 0 $name ($rhost) $option");
                &exitwhoisd(0, "% Warning (1): There are no newer updates available");  
             }
          }
               
          if (($to<$first) || ($from>$to)) {
          
             &syslog("QRYLOG","($$) [g] 0 $name ($rhost) -g syntax error 2: $first-$last, $option");
             &exitonerror("2: Invalid range: $from-$to don\'t exist\n");
          }
               
          if ($to>$last) {
             local($range)=++$last;
          
             &syslog("QRYLOG","($$) [g] 0 $name ($rhost) -g syntax error 3: $first-$last, $option");
             &exitonerror("3: Invalid range: serial(s) $range-$to don\'t exist\n");
          }
       
          if ($from<$first) {
             local($range)=--$first;
          
             &syslog("QRYLOG","($$) [g] 0 $name ($rhost) -g syntax error 4: $first-$last, $option");
             &exitonerror("4: Invalid range: serial(s) $from-$first don\'t exist\n");
          }
       
          if ($version==1) {
             &GetVersionOne($output,1,$source,$from,$to);
          }
          else {
             
             #
             # I am sorry but we currently only support version 1 ...
             
             &GetVersionOne($output,$UPDATEVERSION,$source,$from,$to);
          
          }  
          
          return 1;
          
       }
       else {
          &syslog("QRYLOG","($$) [g] 0 $name ($rhost) syntax error 1: $option");
          &exitonerror("1: Syntax error in -g argument: $option\n");
       }
    }
    else {
       &syslog("AUDITLOG", "Get serial updates authorization failure: $name ($rhost) $option");
       &syslog("QRYLOG","($$) [g] 0 $name ($rhost) authorization failure for query $option");
       &exitonerror("***You are not authorized to get updates***\n***please contact \<$HUMAILBOX\> for more information***");
    }
    
}


sub dowhoislookup {
   local($input, $output, $name, $rhost)=@_;
   
   local($result,$query);
   local($logstr)=$name;
   
   if (@ARGV) {
      $logstr="direct whoisd query";
   }
   elsif (!$opt_D) {
      $logstr="inetd daemon query";
   }
   elsif ($name!=$rhost) {
      $logstr.="(".$rhost.")";
   }
   
   for (;;) {

      $result=0;
      

# Patch from David replaced this, 961010, AMRM
#
#      if (@ARGV) {
#         $query=join(" ", grep($_ ne "-k", @ARGV));
#      }
#      else {
#         alarm $KEEPOPEN;
#         $query=<$input>;
#      }
	
	$!="";
      
      if ((@ARGV) && (!$opt_k)) {
         $query=join(" ", @ARGV);
      }
      else {
         alarm $KEEPOPEN;
         $query=<$input>;
      }
      
      print STDERR "($$) dowhoislookup - query: -$query- errorcode: -$!-\n" if (
$opt_V);


# End of patch from David, 961010, AMRM
      
      alarm $KEEPOPEN;
    
      if ($REPLYBANNER!~ /^\s*$/) {
         print $output "\n", $REPLYBANNER;
         &flush($output);
         alarm $KEEPOPEN;
      }
      
      
      ($query)=&parse($query);
      
      alarm $KEEPOPEN;
      
      print STDERR "($$) whoisd lookup: $query\n" if $opt_V;
      
      #
      # quit if we have an empty query for the second time
      #
      # (connection is probably closed)
         
      if (($LOWPRIORITY) && ($opt_k) && ($query=~ /^\s*$/)) {
            
         print $output "\n\n";
         close($output);
            
         last;
            
      }
      
      if (($query=~ /\w/) || ($opt_t) || ($opt_g) || ($opt_U)) {
          
         ($result)=&whois($input, $output, $query, $name, $rhost);
         alarm $KEEPOPEN;
         print $output "\n", $NOMATCH if ($result==0);
            
      }
      elsif ($query=~ /^\s*$/) {
            
         $result=0;
            
         print $output "\n\% No search key specified\n";
         
      }
      
      else {
         
         #
         # we want at least some alphanumeric stuff ...
         
         $result=0;
         
         print $output "\n\% Cannot lookup non-alphanumeric keys\n";
      
      }
      
      #
      # close connection and do logging afterwards to gain more speed
      
      if (!$opt_k) {
         alarm $KEEPOPEN;
         print $output "\n\n";
         close($output);
      }
      
      #
      # Log this query

      local($flags)=&getflags();
            
      &syslog("QRYLOG","($$) [$flags] $result $logstr - $query");
      
      #      
      # stop querying if not keep connection open is specified
      
      last if (!$opt_k);
      
      alarm $KEEPOPEN;
      print $output "\n\n";
      &flush($output);
      
   }
   
   return $result;

}

#
# Main program

local($arg);

while ($arg=shift(@ARGV)) {
   
   if ($arg eq "-D") {
      $opt_D=1;
   }
   elsif ($arg eq "-V") {
      $opt_V=1;
   }
   elsif ($arg eq "-p") {
      $opt_p=shift(@ARGV);
      die "wrong/no argument specified for \"-p\" switch\n" if ($opt_p=~ /^\s*$/);
   }
   else {
      unshift(@ARGV, $arg);
      last;
   }
   
}


#
# Read config file from RIPEDBCNF, or set to default.

$conffile=$ENV{"RIPEDBCNF"};
$conffile= "DEFCONFIG" unless $conffile;
&rconf($conffile);

$result=0;

print STDERR "whoisd daemon ($$) - running in debug mode\n" if ($opt_V);


#
# If there are other command line options,
# do not run as daemon, but process the command line and exit.

if ((@ARGV) || (!$opt_D)) {

   #
   # setup alarm handler
   
   $SIG{'ALRM'} = 'alarmhandler';

   &dowhoislookup(STDIN, STDOUT, "", "");
   
   exit 0;
   
} 
elsif (!$opt_V) {
   
   # 
   # detach from tty

   exit 0 if (fork()>0);

   if (open(FILE, "/dev/tty")) {
   
      if (!ioctl(FILE,(0x20000000|(ord('t')<<8)|113),0)) {   
         print STDERR "whoisd daemon ($$) - ioctl: $!\n" if ($opt_V);
      }
   
      close(FILE);
   
   }
   
   close(STDIN) if (-t STDIN);
   
}

#
# we are running in daemon mode now

#
# what port do we connect to and which protocol do we use ???

local($port,$proto);

if ($opt_p) {
   ($port,$proto)=&getwhoisportandproto($opt_p);
}
else {
   ($port,$proto)=&getwhoisportandproto("");
}

print STDERR "whoisd daemon ($$) - will connect to port: $port with protocol: $proto\n" if ($opt_V);

socket(S, AF_INETVALUE, SOCK_STREAMVALUE, $proto) || &fatalerror("socket: $!");

#
# || &fatalerror("setsockopt: $!"); commented out.
#
# didn't work for perl4 & ( BSDI || Linux )
#

setsockopt(S, $SOL_SOCKET, $SO_REUSEADDR, 1); # || &fatalerror("setsockopt: $!");

while (!bind(S, pack(SOCKADDRVALUE, AF_INETVALUE, $port, "\0\0\0\0"))) {

    if ($bindcount>=20) {
       print STDERR "whoisd: bind() failed 20 times, giving up\n";
       &syslog("ERRLOG", "whoisd cannot bind() for 20 times, giving up");
       exit 1;
    }
    else {
       print STDERR "whoisd daemon ($$) - bind: $!, trying again after 7 seconds\n" if ($opt_V);
       $bindcount++;
       sleep 7;
    }

}

&syslog("ERRLOG", "whoisd ($$) needed $bindcount binds before succeeding") if ($bindcount);

listen(S, 50) || &fatalerror("listen: $!");

#
# we can now change our UID

print STDERR "whoisd daemon ($$) - changing (UID,GID) from: ($>,$)) to: ($UID,$GID)\n" if ($opt_V);

$)=$GID;

&fatalerror("Couldn\'t change GID from $) to $GID") if ($)!=$GID);

$>=$UID;

&fatalerror("Couldn\'t change UID from $> to $UID") if ($>!=$UID);

print STDERR "whoisd daemon ($$) - changed (UID,GID) to: ($>,$))\n" if ($opt_V);

$oldhandle=select(S); $| = 1; select($oldhandle);

# We have come this far, let's write the PID to $PIDFILE, useful for
# killing and stuff.

if (open(PID, ">$LOCKDIR$PIDFILE.$port")) {
   print PID "$$\n";
   close(PID);
}
else {
   &syslog("ERRLOG", "cannot write to $LOCKDIR$PIDFILE.$port: $!");
}

$SIG{'INT'}='quitdaemonhandler';
$SIG{'KILL'}='quitdaemonhandler';
$SIG{'TERM'}='quitdaemonhandler';

&ReplaceGlobalVars(*REPLYBANNER);
&ReplaceGlobalVars(*NOMATCH);

#
# Main waiting loop, wait for connection, and fork of child to process
# the incoming request

for (;!$QUIT;) {
    
    ($address=accept(NS,S)) || (!$QUIT) || &fatalerror("accept (returncode: \'$address\'): $!");
    
    last if ($QUIT);
     
    #
    # fork as savely as possible (see Camel book, fork)
    
    FORK: {
    
       if (($child=fork())) {
        
          #
          # here is the parent process
       
          while (waitpid(-1, WNOHANGVALUE)>0) {}; 
      
       }
       elsif (defined($child)) {
          
          #
          # this is the child process
          
          #
          # exit as cleanly as possible when we do a shutdown
        
          $SIG{'INT'}='quithandler';
          $SIG{'KILL'}='quithandler';
          $SIG{'TERM'}='quithandler';
        
          #
          # setup alarm handler
        
          $SIG{'ALRM'} = 'alarmhandler';
        
          #
          # Set alarm to timeout after people did not send anything
        
          alarm $KEEPOPEN;
        
          local($af,$port,$inetaddr) = unpack(SOCKADDRVALUE,$address);
          local($rhost)=join(".",unpack('C4', $inetaddr));

          local($name)=(gethostbyaddr($inetaddr,AF_INETVALUE))[0];
        
          #
          # address might not have been reverse delegated...
        
          $name=$rhost if ($name=~ /^\s*$/);

          print STDERR "($$) child connection $name ($rhost)\n" if $opt_V;

          &dowhoislookup(NS, NS, $name, $rhost);
        
          print STDERR "($$) exit connection [$rhost]\n" if $opt_V;
        
          exit 0;
          
       }
       elsif ($!=~ /No\s+more\s+process/i) {
       
          #
          # looks like a recoverable fork error, let's try again
          
          sleep 5;
          
          redo FORK;
       
       }
       else {
          
          &fatalerror("Could not fork: $!") if (!$QUIT);
          
       }
        
    }
    
}

&syslog(AUDITLOG, "whois daemon ($$) killed by signal ($QUIT)") if ($QUIT);

unlink("$LOCKDIR$PIDFILE.$port");

exit 0;

