#include <stdlib.h>
#include "ep_internal.h"
//#include "cr.h"

LG_context_t* ep_ctx;
gchar* ep_temporary_directory = NULL;
gchar* ep_ca_file = NULL;
gchar* ep_DN = NULL;
gchar* ep_crl_dir = NULL;

/*
  EP_init: initializes the system.
 */
void EP_init(LG_context_t* ctx, gchar* tmp_dir,
             gchar* ca_file, gchar* crl_dir) {
  ep_ctx = ctx;

  ep_temporary_directory = g_strdup(tmp_dir);
  ep_ca_file = g_strdup(ca_file);
//  ep_DN = g_strdup(DN);
  ep_crl_dir = g_strdup(crl_dir);
  MM_init();
}

/*
  EP_end: terminates the system.
 */
void EP_end() {
  if ( ep_temporary_directory )
  {
    g_free(ep_temporary_directory);
  }
  if ( ep_ca_file )
  {
    g_free(ep_ca_file);
  }
  if ( ep_DN )
  {
    g_free(ep_DN);
  }
  if ( ep_crl_dir )
  {
    g_free(ep_crl_dir);
  }
}


/*
  Unfolds input.

  stream  - The input stream
  is_mail - is it mail?
  input_structure - pointer to the input structure to be filled in

  return  - success/error code

  This functions is the most important entry point of EP, its where
  the input is parsed.
 */
int EP_unfold(FILE* stream, gboolean is_mail, char *X509cert_file, 
                                  EP_input_structure_t **input_structure)
{
  int retval = EP_OK;
  GString *string_stream; /* no need to free - done in ep_a_d_free */
  ep_authenticated_data_t *data;

  LG_log(ep_ctx, LG_DEBUG, ">EP_unfold entered with parameters is mail %s X.509 certificate %s",
                               is_mail ? "yes" : "no", X509cert_file ? "yes" : "no");
  retval = ep_stream_to_string(stream, &string_stream);
  if (retval == EP_FATAL)
  {
    return retval;
  }
  data = ep_authenticated_data_new();

  data->data->data = string_stream;

  data->data->type = (is_mail) ? EP_MAIL : EP_TEXT;

  if (X509cert_file)
  {
    /* this update came from web/sync updates */
    /* verify the certificate and get dname 
       set the credential according to the verify result */
    LG_log(ep_ctx, LG_FATAL, "EP_unfold: web/sync update");
//  Temporary disable x509 from syncupdates
//    retval = ep_check_X509_cert(X509cert_file, &ep_DN);
    retval = EP_FAIL;

    if (retval == EP_FATAL)
    {
      return retval;
    }
    else if ( retval == EP_OK )
    {
      LG_log(ep_ctx, LG_DEBUG, "EP_unfold: create validated credential");
      data->credentials = g_list_append(NULL, CR_credential_new(CR_X509, ep_DN, TRUE));
    }
    else
    {
      LG_log(ep_ctx, LG_DEBUG, "EP_unfold: create invalidated credential");
      data->credentials = g_list_append(NULL, CR_credential_new(CR_X509, ep_DN ? ep_DN : "unk DN", FALSE));
    }
  }
  else
  {
    /* this is not a web/sync update so credentials will be
       set up while exploding the data */
    data->credentials = NULL;
  }
  
  LG_log(ep_ctx, LG_FATAL, "EP_unfold: will explode the data");
  *input_structure = ep_explode(data);
  LG_log(ep_ctx, LG_FATAL, "EP_unfold: exploded the data");
  LG_log(ep_ctx, LG_FATAL, "EP_unfold: will call ep_authenticated_data_free");
  ep_authenticated_data_free(data);
  LG_log(ep_ctx, LG_FATAL, "EP_unfold: called ep_authenticated_data_free");

  LG_log(ep_ctx, LG_DEBUG, "<EP_unfold exiting with value %s", EP_ret2str(retval));
  return retval;
}

/*
  Returns candidate keywords.

  input - Input structure.

  return - Candidate keywords.
 */
gchar* EP_get_candidate_keywords(EP_input_structure_t* input) {
  ep_input_structure_t* inp;
  GString* result;
  gchar** tokens;
  int curr_token;
  gchar* return_result;

  inp = (EP_input_structure_t*) input;
  if (!inp->mail_info) {
    return "";
  }
  tokens = g_strsplit(inp->mail_info->subject->str, " ", 0);
  result = g_string_new("");

  for (curr_token = 0; tokens[curr_token]; curr_token++) {
    if (tokens[curr_token][0] == 0) {
      continue;
    }
    if (curr_token) {
      g_string_append(result, ",");
    }
    g_string_append(result, tokens[curr_token]);
  }
 
  g_strfreev(tokens);
  return_result = result->str;
  g_string_free(result, FALSE);
  return return_result;
}

/*
  For each blob in an input structure apply a function

  input     - the input structure
  func      - the function
  user_data - extra data to be passed to the function
*/
void EP_blobs_foreach(EP_input_structure_t* input,
  ObjFunc func, gpointer user_data) {
  GList* blobs;

  blobs = ((ep_input_structure_t*)input)->flattened_authenticated_data;
  if (blobs == NULL) {
    return;
  }
  do {
    func(blobs->data, user_data);
  } while (blobs = g_list_next(blobs));
}

/*
  Returns the requested attribute of a mail_info.

  input  - the input structure

  field  - the field

  return - the address
*/
const gchar* EP_get_mail_hdr_field(EP_input_structure_t* input, gchar* field) {
  ep_input_structure_t* inp;

  inp = (ep_input_structure_t*)input;
  if (inp->mail_info) {
    if (strcasecmp(field, "from") == 0) {
      return inp->mail_info->from->str;
    }
    if (strcasecmp(field, "subject") == 0) {
      return inp->mail_info->subject->str;
    }
    if (strcasecmp(field, "cc") == 0) {
      return inp->mail_info->cc->str;
    }
    if (strcasecmp(field, "date") == 0) {
      return inp->mail_info->date->str;
    }
    if (strcasecmp(field, "replyto") == 0) {
      return inp->mail_info->reply_to->str;
    }
    if (strcasecmp(field, "message-id") == 0) {
      return inp->mail_info->message_id->str;
    }
    return NULL;
  }
  else {
    return NULL;
  }
}

/*
  Returns a blob from a blob_credential.

  input  - the blob_credential

  return - the blob
*/
const gchar* EP_get_blob(EP_blob_credential_t* input) {
  return ((ep_authenticated_data_t*)input)->data->data->str;
}

/*
  Returns a credential from a blob_credential

  input  - the blob_credential

  return - the credential 
*/
GList* EP_get_credentials(EP_blob_credential_t* input) {
  return ((ep_authenticated_data_t*)input)->credentials;
}

//void ep_authenticated_data_free(ep_authenticated_data_t* data);
void ep_input_structure_remove_auth_data(gpointer data, gpointer user) {
  ep_authenticated_data_t* auth_data;

  auth_data = (ep_authenticated_data_t*) data;
  ep_authenticated_data_free(auth_data);
}

void EP_input_structure_free(EP_input_structure_t* input) {
  ep_input_structure_t* structure;

  structure = (ep_input_structure_t*) input;
  if (structure->mail_info) {
    ep_mail_info_free(structure->mail_info);
  }
  g_list_foreach(structure->flattened_authenticated_data, ep_input_structure_remove_auth_data, NULL);
  g_list_free(structure->flattened_authenticated_data);
    
  g_free(structure);

}

/*
  Frees a mail_info structure.

  mail_info - mail_info structure.
*/
void ep_mail_info_free(ep_mail_info_t* mail_info) {
  g_string_free(mail_info->from, TRUE);
  g_string_free(mail_info->subject, TRUE);
  g_string_free(mail_info->date, TRUE);
  g_string_free(mail_info->reply_to, TRUE);
  g_string_free(mail_info->cc, TRUE);
  g_string_free(mail_info->message_id, TRUE);
  g_free(mail_info);
}

/*
  Explodes a piece of input.

  data   - data to explode

  return - the representative input structure
*/
ep_input_structure_t* ep_explode(ep_authenticated_data_t* data) {

  LG_log(ep_ctx, LG_DEBUG, ">ep_explode entered ");

  switch (data->data->type) {
    case EP_TEXT:
      LG_log(ep_ctx, LG_DEBUG, "<ep_explode exiting [return ep_text_driver(data)] ");
      return ep_text_driver(data);
    case EP_MAIL:
      LG_log(ep_ctx, LG_DEBUG, "<ep_explode exiting [return ep_mail_driver(data)] ");
      return ep_mail_driver(data);
    break;
  }

  LG_log(ep_ctx, LG_DEBUG, "<ep_explode exiting ");
}

/*
  ep_stream_to_string - Converts a file to a GString

  stream - FILE stream.

  return - a GString with the whole file contents

  The file is supposedly a text string (without 0s).
*/
int ep_stream_to_string(FILE* stream, GString **return_string) {
#define ESTS_BUFFER_SIZE 1025
#define ESTS_READ_SIZE 1024
  int retval = EP_OK;
  char buffer[ESTS_BUFFER_SIZE];

  LG_log(ep_ctx, LG_DEBUG, ">ep_stream_to_string entered ");

  *return_string = g_string_new("");
  while (!feof(stream)) {
    memset(buffer, 0, ESTS_BUFFER_SIZE);
    fread(buffer, 1, ESTS_READ_SIZE, stream);
    if ( ferror(stream) )
    {
      return EP_FATAL;
    }
    g_string_append(*return_string, buffer);
  }

  LG_log(ep_ctx, LG_DEBUG, "<ep_stream_to_string exiting with string [\n%s]", 
                                       (*return_string)->str );
  return retval;
}

/*
  Creates a new input_structure.

  return - input_structure.
 */
ep_input_structure_t* ep_input_structure_new() {
  ep_input_structure_t * input;

  input = g_malloc0(sizeof(ep_input_structure_t));
  input->mail_info = NULL;
  input->flattened_authenticated_data = NULL;

  return input;
}

/*
  Creates a new authenticated_data structure.

  return - authenticated_data structure.
*/
ep_authenticated_data_t* ep_authenticated_data_new() {
  ep_authenticated_data_t* data;

  data = g_malloc0(sizeof(ep_authenticated_data_t));
  data->data = ep_exploded_data_new();
  data->credentials = NULL;

  return data;
}

/*
  Frees an authenticated_data structure.

  data - authenticated_data structure.
 */
void ep_authenticated_data_free(ep_authenticated_data_t* data) {
  ep_exploded_data_free(data->data);
  CR_credential_list_free(data->credentials);
  g_free(data);
}

ep_exploded_data_t* ep_exploded_data_new() {
  ep_exploded_data_t* data;

  data = g_malloc0(sizeof(ep_exploded_data_t));
  data->data = g_string_new("");

  return data;
}

/*
  Frees an exploded_data structure.

  data - expolded_data structure.
 */
void ep_exploded_data_free(ep_exploded_data_t* data) {
  g_string_free(data->data, TRUE);
  g_free(data);
}

/*
  Creates a new mail_info structure.

  from    - From:
  subject - Subject:

  return - New mail_info structure.
 */
ep_mail_info_t* ep_mail_info_new(gchar* from, gchar* subject,
  const gchar* date, const gchar* reply_to, const gchar* cc,
  const gchar* message_id) {
  ep_mail_info_t* mail_info;

  mail_info = g_malloc0(sizeof(ep_mail_info_t));
  mail_info->from = g_string_new(from ? from : "");
  mail_info->subject = g_string_new(subject ? subject : "");
  mail_info->date = g_string_new(date ? date : "");
  mail_info->reply_to = g_string_new(reply_to ? reply_to : "");
  mail_info->cc = g_string_new(cc ? cc : "");
  mail_info->message_id = g_string_new(message_id ? message_id : "");

  return mail_info;
}

