/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * VMTP protocol specific code
 */

#include <general.h>
#include <protospec.h>


/*  */

#define VMTP_CALL_CODE 		VMTP_OK
#define VMTP_RETURN_CODE	VMTP_OK
#define VMTP_IDEMP_RETURN_CODE	(VMTP_OK | VU_DGM)

#define VMTP_MCB_DATA		28

#define SERVER_QSIZE		10
#define ETHER_TTL		1

char *vmtp_code2str();

/*  */

vmtp_create_client()
{
    pprintf("vmtp_create_client()\n");

} /* vmtp_create_client */

/*  */

vmtp_conn_request(addr, a_ch, errind)
    struct address_t *addr;
    channel_t *a_ch;
    error_t   *errind;
{
    struct vmtphdl *hdl;
    
    pprintf("vmtp_conn_request(0x%x, 0x%x, 0x%x)\n", addr, a_ch, errind);
    
    if ((hdl = CreateVmtpClient(1, "")) == NULL) {
	eprintf(EF_SYSCALL, COMMUNICATION, "CreateVmtpClient", "ConnRequest",
	       getsyserr());
	errind->er_ret = NOTOK;
	return;
    }

    a_ch->handle = hdl;
    a_ch->addr = *addr;
    
    errind->er_ret = OK;
    errind->er_code = 0;
} /* vmtp_conn_request */

/*  */

ConnRequestEther(addr, a_ch, errind)
    struct address_t *addr;
    channel_t *a_ch;
    error_t   *errind;
{
    int s;
    
    pprintf("ConnRequestEther(0x%x, 0x%x, 0x%x)\n", addr, a_ch, errind);
    
    vmtp_conn_request(addr, a_ch, errind);
    if (errind->er_ret != OK)
      return;
    
    errind->er_ret = set_ttl(a_ch->handle, ETHER_TTL);
    if (failed(errind)) {
	eprintf(EF_IN3, COMMUNICATION, "set_ttl", "ConnRequestEther");
	return;
    }

} /* ConnRequestEther */



vmtp_disc_request(ch, errind)
    channel_t ch;
    error_t   *errind;
{
    int s;

    pprintf("vmtp_disc_request(%d, 0x%x)\n", ch, errind);
    
    if (DestroyVmtpClient(ch.handle)== NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "DestroyVmtpClient()", 
		"DiscRequest",
	       getsyserr());
	errind->er_ret = NOTOK;
	return;
    }

    errind->er_ret = OK;
    errind->er_code = 0;
} /* vmtp_disc_request */

/*  */

vmtp_create_server(aa_server, aa_addr, errind)
    struct server_t **aa_server;
    struct address_t **aa_addr;
    error_t   *errind;
{
    struct vmtphdl *hdl;
    struct vmtpeid eid;
    long code;
    
    pprintf("vmtp_create_server(0x%x, 0x%x, 0x%x)\n",
		aa_server, aa_addr, errind);

    if ((hdl = CreateVmtpServer(1, "", SERVER_QSIZE, 0)) == NULL) {
	eprintf(EF_SYSCALL, COMMUNICATION, "CreateVmtpServer", "CreateServer",
	       getsyserr());
	errind->er_ret = NOTOK;
	return;
    }

    if (getmyeid(hdl, &eid) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "getmyeid", "CreateServer",
	       getsyserr());
	errind->er_ret = NOTOK;
	return;
    }

    *aa_server = (struct server_t *)malloc(sizeof(struct server_t));
    if (*aa_server == NULL) {
	eprintf(EF_IN3, INTERNAL, RESOURCE, "CreateServer");
	errind->er_ret = NOTOK;
	return;
    }
    *aa_addr = (struct address_t *)malloc(sizeof(struct address_t));
    if (*aa_addr == NULL) {
	eprintf(EF_IN3, INTERNAL, RESOURCE, "CreateServer");
	errind->er_ret = NOTOK;
	free((char *)*aa_server);
	return;
    }
    (*aa_server)->handle = hdl;
    (*aa_server)->addr = eid;

    *(*aa_addr) = eid;
    
    errind->er_ret = OK;
    errind->er_code = 0;
    pprintf("<<CreateServer\n");
} /* vmtp_create_server */

/*  */

vmtp_destroy_server(a_server, errind)
    struct server_t *a_server;
    error_t   *errind;
{
    
    pprintf("vmtp_destroy_server(%d, 0x%x)\n", a_server->handle->vmtp_socket, 
	    errind);

    if (DestroyVmtpServer(a_server->handle) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "DestroyVmtpServer()", 
		"DestroyServer",
	       getsyserr());
	free((char *)a_server);
	errind->er_ret = NOTOK;
	return;
    }
	
    free((char *)a_server);
    errind->er_ret = OK;
    errind->er_code = 0;
} /* vmtp_destroy_server */

/*  */

vmtp_call(ch, srcbuf, srclen, dstbuf, dstlen, errind)
    channel_t ch;
    char *srcbuf, *dstbuf;
    int srclen, dstlen;
    error_t *errind;
{
    union vmtpmsg msg;
    int rseglen;
    
    pprintf("vmtp_call(to %d: 0x%x:%x, 0x%x, %d, 0x%x, %d)\n",
	    ch.handle->vmtp_socket, ch.addr.ve_fltm, ch.addr.ve_inaddr,
	    srcbuf, srclen, dstbuf, dstlen);

    msg.crq_timeout = VMTP_WAIT_FOREVER;
    msg.crq_eid = ch.addr;
    msg.crq_code = VMTP_CALL_CODE;
    if (srclen > VMTP_MCB_DATA) {
	msg.crq_code |= VU_SDA;
	msg.crq_segptr = srcbuf;
	msg.crq_segsize = srclen;
    } else if (srclen) {
	bcopy(srcbuf, msg.crq_longdata, srclen);
    }
    if (dstlen > VMTP_MCB_DATA) {
	msg.crq_replysegptr = dstbuf;
	rseglen = msg.crq_replysegsize = dstlen;
    } else
        rseglen = msg.crq_replysegsize = 0;

    pprintf("calling transact\n");
    errind->er_ret = transact(ch.handle->vmtp_socket, &msg, 
			      sizeof(union vmtpmsg));
    pprintf("transact: %d, 0x%x\n", errind->er_ret, msg.crp_code);
    
    errind->er_code = msg.crp_code;
    if (failed(errind)) {
	eprintf(EF_SYSCALL, COMMUNICATION, "transact", "RPCCall", getsyserr());
	return;
    }

    if (rseglen) {
      if (!vmtp_segmentdata(msg.crq_code)) {
	eprintf(EF_IN3, COMMUNICATION, "transact", 
		"RPCCall - no segment returned");
	errind->er_ret = NOTOK;
	errno = 0;
	return;
      }
      if (rseglen != msg.crp_segsize) {
	eprintf(EF_IN3, COMMUNICATION, "transact",
		"RPCCall - unexpected segment length in response");
	errind->er_ret = NOTOK;
	errno = 0;
	return;
      }
    }
    if (dstlen != 0 && dstlen <= VMTP_MCB_DATA) {
	bcopy(msg.crq_longdata, dstbuf, dstlen);
    }
    pprintf("<<vmtp_call\n");
} /* vmtp_call */

/*  */

vmtp_await_call(a_ch, dstbuf, dstlen, errind)
    channel_t *a_ch;
    char *dstbuf;
    int dstlen;
    error_t *errind;
{
    union vmtpmsg msg;
    int rsegsize;

    pprintf("vmtp_await_call(from %d, 0x%x, %d)\n", a_ch->handle->vmtp_socket,
	    dstbuf, dstlen);

    msg.srr_timeout = VMTP_WAIT_FOREVER;
    msg.crq_code = VM_GETCLREQ;
    ZERO_EID(msg.srr_client);
    if (dstlen > VMTP_MCB_DATA) {
	msg.srr_reqsegptr = dstbuf;
	rsegsize = msg.srr_reqsegsize = dstlen;
    } else
        rsegsize = msg.srr_reqsegsize = 0;

    errind->er_ret = transact(a_ch->handle->vmtp_socket, &msg, 
			      sizeof(union vmtpmsg));
    errind->er_code = msg.srq_code;
    pprintf("RPCAwaitCall transact: %d, 0x%x\n", errind->er_ret, msg.crp_code);
    if (failed(errind)) {
	eprintf(EF_SYSCALL, COMMUNICATION, "transact", "RPCAwaitCall", 
		getsyserr());
	return;
    }

    if (rsegsize) {
      if (!vmtp_segmentdata(msg.srq_code)) {
	eprintf(EF_IN3, COMMUNICATION, "transact",
		"RPCAwaitCall - no segment returned");
	errind->er_ret = NOTOK;
	errno = 0;
	return;
      }
      if (msg.srq_segsize != rsegsize) {
	eprintf(EF_IN3, COMMUNICATION, "transact",
		"RPCAwaitCall - unexpected segment length returned");
	errind->er_ret = NOTOK;
	errno = 0;
	return;
      }
    }
    a_ch->addr = msg.srq_client;
    if (dstlen <= VMTP_MCB_DATA && dstlen != 0) {
	bcopy(msg.crq_longdata, dstbuf, dstlen);
    }
    pprintf("<<vmtp_await_call received from 0x%x:%x)\n", 
	    a_ch->addr.ve_fltm,
	    a_ch->addr.ve_inaddr);
    
} /* vmtp_await_call */

/*  */

vmtp_return_call(ch, srcbuf, srclen, errind)
    channel_t ch;
    char *srcbuf;
    int srclen;
    error_t *errind;
{
    union vmtpmsg msg;
    
    pprintf("vmtp_return_call(to %d: 0x%x:%x, 0x%x, %d)\n",
	    ch.handle->vmtp_socket, ch.addr.ve_fltm, ch.addr.ve_inaddr,
	    srcbuf, srclen);

    msg.srp_eid = ch.addr;
    msg.srp_code = VM_SENDCLRSP;
    msg.srp_respcode = VMTP_RETURN_CODE;
    if (srclen > VMTP_MCB_DATA) {
	msg.srp_code |= VU_SDA;
	msg.srp_respcode |= VU_SDA;
	msg.srp_segptr = srcbuf;
	msg.srp_segsize = srclen;
    } else if (srclen) {
	bcopy(srcbuf, msg.crq_longdata, srclen);
    }
    ZERO_EID(msg.srp_fserver);

    errind->er_ret = transact(ch.handle->vmtp_socket, &msg, 
			      sizeof(union vmtpmsg));
    errind->er_code = msg.crp_code;
    pprintf("RPCReturn transact: %d, 0x%x\n", errind->er_ret, msg.crp_code);
    if (failed(errind)) {
	eprintf(EF_SYSCALL, COMMUNICATION, "transact", "RPCReturn", 
		getsyserr());
	return;
    }
    pprintf("<<vmtp_return_call\n");
} /* vmtp_return_call */

/*  */

RPCReturnIdemp(ch, srcbuf, srclen, errind)
    channel_t ch;
    char *srcbuf;
    int srclen;
    error_t *errind;
{
    union vmtpmsg msg;
    
    pprintf("RPCReturnIdemp(to %d: 0x%x:%x, 0x%x, %d)\n",
	    ch.handle->vmtp_socket, ch.addr.ve_fltm, ch.addr.ve_inaddr,
	    srcbuf, srclen);

    msg.srp_eid = ch.addr;
    msg.srp_code = VM_SENDCLRSP;
    msg.srp_respcode = VMTP_IDEMP_RETURN_CODE;
    if (srclen > VMTP_MCB_DATA) {
	msg.srp_code |= VU_SDA;
	msg.srp_respcode |= VU_SDA;
	msg.srp_segptr = srcbuf;
	msg.srp_segsize = srclen;
    } else if (srclen) {
	bcopy(srcbuf, msg.crq_longdata, srclen);
    }
    ZERO_EID(msg.srp_fserver);

    errind->er_ret = transact(ch.handle->vmtp_socket, &msg, 
			      sizeof(union vmtpmsg));
    errind->er_code = msg.crp_code;
    if (failed(errind)) {
	eprintf(EF_SYSCALL, COMMUNICATION, "transact", "RPCReturnIdemp", 
		getsyserr());
	return;
    }
    pprintf("<<RPCReturnIdemp\n");
} /* RPCReturnIdemp */

/*  */

vmtp_report_error(err, str)
    error_t *err;
    char *str;
{
    
    pprintf("vmtp_report_error(0x%x)\n", err);
    
    if (err == NULL) {
	eprintf(EF_SYSCALL, COMMUNICATION, "Error", str,
	       getsyserr());
    } else if (err->er_ret == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "VMTP protocol", str,
	       getsyserr());
    } else {
	eprintf("%s in %s: %s return code %s (0x%x)\n", COMMUNICATION,
		"VMTP protocol", str,
		vmtp_code2str(err->er_code), err->er_code);
    }
} /* vmtp_report_error */

/*  */

/* Move to vmtplib? */

char *vmtp_code2str(code)
    long code;
{
    static char buf[100];
    
    switch (code&0x00ffffff) {
    case VMTP_OK: 		return "VMTP_OK";
    case VMTP_RETRY: 		return "VMTP_RETRY";
    case VMTP_RETRYALL: 	return "VMTP_RETRYALL";
    case VMTP_BUSY: 		return "VMTP_BUSY";
    case VMTP_NONEXIST: 	return "VMTP_NONEXIST";
    case VMTP_MIGRATED: 	return "VMTP_MIGRATED";
    case VMTP_NOPERMISSN: 	return "VMTP_NOPERMISSN";
    case VMTP_NOTAWAIT: 	return "VMTP_NOTAWAIT";
    case VMTP_PROERROR: 	return "VMTP_PROERROR";
#ifdef notdef
    case VMTP_FREQRFSD: 	return "VMTP_FREQRFSD";
#endif notdef
    case VMTP_BADTID: 		return "VMTP_BADTID";
    case VMTP_NOSTREAMING: 	return "VMTP_NOSTREAMING"; 
    case VMTP_NORUNREC: 	return "VMTP_NORUNREC";
    case VMTP_REXMTTIMEO: 	return "VMTP_REXMTTIMEO";
    case VMTP_USERTIMEO: 	return "VMTP_USERTIMEO";
    case VMTP_RESPDISC: 	return "VMTP_RESPDISC";
    case VMTP_NOSECURITY: 	return "VMTP_NOSECURITY";
    case VMTP_BADREPLY: 	return "VMTP_BADREPLY";
    case VMTP_SECUREREQ: 	return "VMTP_SECUREREQ";
#ifdef notdef
    case VMTP_NOMULTICAST: 	return "VMTP_NOMULTICAST";
#endif notdef
    case VMTP_MANYRETRIES: 	return "VMTP_MANYRETRIES";
    case VMTP_NOPRINCIPAL: 	return "VMTP_NOPRINCIPAL";
    case VMTP_NOKEY: 		return "VMTP_NOKEY";
#ifdef notdef
    case VMTP_CACHEMISS: 	return "VMTP_CACHEMISS";
#endif notdef
	
    default:
	sprintf(buf, "unknown (0x%X)", code);
	return buf;
    }
} /* vmtp_code2str */
    

