/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * Handling messages sent to system like MSG_CREATE and MSG_DESTROY
 */

#include <general.h>
#include <ipc.h>
#include <ipc_pres.h>

/*
 * Not: messages to system refer to an entity by placing it's address in
 * the from field. e.g. to create demon process/procedure with index 3 do:
 *	msg->msg_to.ba_addr = BA_SYSTEM;
 *	msg->msg_to.ba_index = ??
 *	msg->msg_from.ba_addr = BA_SERVER;
 *	msg->msg_from.ba_index = 3;
 *	msg->msg_code = MSG_CREATE;
 *	if (bcp_send_msg(msg) == NOTOK) { ... }
 */

/*
 * Exports:
 *	handle_system_msg(msg: msg_t *)
 *
 *	int local_addr(addr: bcpaddr_t)
 *	int remote_addr(addr: bcpaddr_t)
 *	procstate_free()
 */

/*  */

/* Forwards */

int handle_create();
int handle_destroy();
int handle_timeq();

/* Local data - exported to ipc.c and ipc_sig.c */

int 		system_client_nproc = 0;
commstate_t 	*system_client_procs = NULLC;
int 		system_server_nproc = 0;
commstate_t 	*system_server_procs = NULLC;

/*  */

int handle_system_msg(msg)
    msg_t *msg;
{
    int ret;
    
    tprintf("handle_system_msg(0x%x)\n", msg);
    
    switch (msg->msg_code) {
    case MSG_CREATE:
	if (!(msg->msg_from.ba_addr &
	      (BA_CLIENT|BA_CLIENTS|BA_SERVER|BA_SERVERS))) {
	    eprintf(EF_IN3, PROTOCOL, "Bad from address", "handle_system_msg");
	    dprintf("\tFrom %s\n", bcpaddr2str(msg->msg_from));
	    ret = NOTOK;
	} else {
	    ret = handle_create(msg);
	}
	break;
    case MSG_DESTROY:
	if (!(msg->msg_from.ba_addr &
	      (BA_CLIENT|BA_CLIENTS|BA_SERVER|BA_SERVERS))) {
	    eprintf(EF_IN3, PROTOCOL, "Bad from address", "handle_system_msg");
	    dprintf("\tFrom %s\n", bcpaddr2str(msg->msg_from));
	    ret = NOTOK;
	    
	} else {
	    ret = handle_destroy(msg);
	}
	break;

    case MSG_TIMEQ:
	ret = handle_timeq(msg);
	break;
	
    default:
	eprintf("Message with code %s sent to system - ignored!\n",
		code2str(msg->msg_code));
	ret = NOTOK;
	break;
    }
    
    if (ret == NOTOK) {
	eprintf(EF_IN3, INTERNAL, "Failure", "handle_system_msg");
	(void)send_failure(msg);
    }
	
    return OK;
} /* handle_system_msg */

/*  */

static int handle_timeq(msg)
    msg_t *msg;
{
    static char buf[512];
    struct timeval tv;
    struct timezone dummy;
    
    tprintf("handle_timeq(0x%x)\n", msg);

    if (msg->msg_datalen)
	free(msg->msg_data);

    msg->msg_to = msg->msg_from;

    if (gettimeofday(&tv, &dummy) == NOTOK) {
	eprintf(EF_SYSCALL, INTERNAL, "gettimeofday", "handle_timeq",
		getsyserr());
	return NOTOK;
    }

    sprintf(buf, "%ld %ld", tv.tv_sec, tv.tv_usec);

    msg->msg_data = buf;
    msg->msg_datalen = strlen(buf) + 1;
    msg->msg_code = MSG_TIME;
    if (send_msg(msg) == NOTOK) {
	eprintf(EF_IN3, INTERNAL, "send_msg", "handle_timeq");
	return NOTOK;
    }
    return OK;
} /* handle_timeq */

/*  */

int handle_create(msg)
    msg_t *msg;
{
    tprintf("handle_create(0x%x)\n", msg);

    if (msg->msg_from.ba_addr & BA_CLIENT) {
	if (alloc_procs(msg, &system_client_nproc,
			&system_client_procs) == NOTOK)
	    return NOTOK;
    } else if (msg->msg_from.ba_addr & BA_SERVER) {
	if (alloc_procs(msg, &system_server_nproc,
			&system_server_procs) == NOTOK)
	    return NOTOK;
    } else {
	eprintf(EF_IN3, INTERNAL, PARAMETER, "handle_create");
	return NOTOK;
    }
    (void)alloc_childstate(msg->msg_from.ba_index + 1);

    if (msg->msg_from.ba_addr & BA_CLIENT) {
	if (do_create(msg, &system_client_nproc,
		      &system_client_procs) == NOTOK)
	    return NOTOK;
    } else if (msg->msg_from.ba_addr & BA_SERVER) {
	if (do_create(msg, &system_server_nproc,
		      &system_server_procs) == NOTOK)
	    return NOTOK;
    }
    
    return OK;
} /* handle_create */

/*  */

static int alloc_procs(msg, a_nproc, a_procs)
    msg_t *msg;
    int *a_nproc;
    commstate_t **a_procs;
{
    register i;
    int index = msg->msg_from.ba_index;
    msg_t rqs, *rq = &rqs;
    int nproc = *a_nproc;
    commstate_t *procs = *a_procs;

    tprintf("alloc_procs(0x%x, 0x%X, 0x%x)\n", msg, a_nproc, a_procs);

    if (index >= nproc) {
	commstate_t *tmp;

	tmp = (commstate_t *)malloc((index+1)*sizeof(commstate_t));
	if (tmp == NULLC) {
	    eprintf(EF_IN3, RESOURCE, "Out of memory", "alloc_procs");
	    return NOTOK;
	}
	for (i = 0; i < nproc; i++) {
	    tmp[i] = procs[i];
	}
	for (i = nproc; i < index+1; i++) {
	    tmp[i].c_read = tmp[i].c_write = NOTOK;
	    tmp[i].c_pid = tmp[i].c_pgrp = 0;
	}
	if (procs != NULLC)
	    free((char *)procs);
	nproc = index + 1;
	procs = tmp;
    }

    *a_nproc = nproc;
    *a_procs = procs;

    return OK;
} /* alloc_procs */

/*  */

/*
 * The storage allocated in alloc_procs is reused or freed by itself, so
 * this function is only used whenchecking for dynamic storage "leaks".
 */
procstate_free()
{
    if (system_server_nproc != 0)
	free((char *)system_server_procs);
    system_server_nproc = 0;
    if (system_client_nproc != 0)
	free((char *)system_client_procs);
    system_client_nproc = 0;
    childstate_free();
} /* procstate_free */

/*  */

static int do_create(msg, a_nproc, a_procs)
    msg_t *msg;
    int *a_nproc;
    commstate_t **a_procs;
{
    register i;
    int index = msg->msg_from.ba_index;
    msg_t rqs, *rq = &rqs;
    int nproc = *a_nproc;
    commstate_t *procs = *a_procs;

    switch (get_state(msg->msg_from)) {
    case DEAD:
	/* create the process */
	if (create_process(msg->msg_from, &procs[index]) == NOTOK)
	    return NOTOK;

	if (set_state(msg->msg_from, UNKNOWN) == NOTOK) {
	    eprintf(EF_IN3, INTERNAL,
		    "Can't set state to UNKNOWN",
		    "handle_create");
	    return NOTOK;
	}
	break;

    case UNKNOWN:
    default:
	eprintf ("Process in bad state %s: pid %d, read %d, write %d\n",
		state2str(get_state(msg->msg_from)),
		procs[index].c_pid,
		procs[index].c_read,
		procs[index].c_write);
	return NOTOK;
    }
    
    return OK;
} /* do_create */

/*  */

int handle_destroy(msg)
    msg_t *msg;
{
    int i;
    bcpaddr_t addr;
    
    tprintf("handle_destroy(0x%x)\n", msg);

    addr = msg->msg_from;
    if (addr.ba_addr & BA_CLIENTS) {
	bcpaddr_t tmp;

	tmp.ba_addr = BA_CLIENTS;
	for (i = 0; i < system_client_nproc; i++) {
	    tmp.ba_index = i;
	    if (do_destroy(tmp, system_client_nproc,
			   system_client_procs) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "do_destroy", "handle_destroy");
		return NOTOK;
	    }
	}
    } else if (addr.ba_addr & BA_CLIENT
	       && addr.ba_index < system_client_nproc) {
	if (do_destroy(addr, system_client_nproc,
		       system_client_procs) == NOTOK) {
	    eprintf(EF_IN3, INTERNAL, "do_destroy", "handle_destroy");
	    return NOTOK;
	}
    }

    if (addr.ba_addr & BA_SERVERS) {
	bcpaddr_t tmp;

	tmp.ba_addr = BA_CLIENTS;
	for (i = 0; i < system_server_nproc; i++) {
	    tmp.ba_index = i;
	    if (do_destroy(tmp, system_server_nproc,
			   system_server_procs) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "do_destroy", "handle_destroy");
		return NOTOK;
	    }
	}
    } else if (addr.ba_addr & BA_SERVER
	       && addr.ba_index < system_server_nproc) {
	if (do_destroy(addr, system_server_nproc,
		       system_server_procs) == NOTOK) {
	    eprintf(EF_IN3, INTERNAL, "do_destroy", "handle_destroy");
	    return NOTOK;
	}
    }

    return OK;
} /* handle_destroy */

/*  */

int do_destroy(addr, nproc, procs)
    bcpaddr_t addr;
    int nproc;
    commstate_t *procs;
{
    int index = addr.ba_index;
    int pid, j;
#ifdef BSD
    union wait stat;
#endif BSD
    
    tprintf("do_destroy(%d, %d, 0x%x)\n", index, nproc, procs);
    
    if (get_state(addr) == DEAD || procs[index].c_pid == 0) {
	dprintf("do_destroy : Process %d already in DEAD state\n",
		index);
	return OK;
    }
    
    if (kill(procs[index].c_pid, SIGKILL) == NOTOK
	&& errno != ESRCH) {
	eprintf(EF_SYSCALL, INTERNAL, "kill", "handle_destroy",
		getsyserr());
	return NOTOK;
    }
    
 retry:
#ifdef BSD
    pid = wait3(&stat, 0, (struct rusage *)NULL);
#endif BSD
#ifdef SYSV
    pid = wait(NULL);
#endif SYSV
    if (pid != procs[index].c_pid) {
	dprintf(EF_IN3, INTERNAL, "wait3() returns wrong pid",
		"handle_destroy");
	dprintf("\tpid = %d, expected %d\n",
		pid, procs[index].c_pid);
	for (j = 0; j < nproc; j++) {
	    if (pid == procs[j].c_pid) {
		bcpaddr_t tmp;

		tmp.ba_addr = addr.ba_addr;
		tmp.ba_index = j;
		(void)set_state(tmp, DEAD);
		procs[j].c_pid = procs[j].c_pgrp = 0;
		goto retry;
	    }
	}
	eprintf(EF_IN3, INTERNAL, "wait3() returns a bogus pid",
		"handle_destroy");
	return NOTOK;
    }
    /*
     * Defer closing the channels until the MSG_CLOSED is
     * received (since MSG_CLOSED is an acknowledgement of the DESTROY)
     */
    (void)set_state(addr, DEAD);
    procs[index].c_pid = procs[index].c_pgrp = 0;
    
    return OK;
} /* do_destroy */

/*  */

#ifndef __SOCKET__
#define __SOCKET__
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif __SOCKET__

/*
 * For some versions of Apollo:s INADDR_LOOPBACK has caused problem.
 * In these situations it has worked to remove the variable. So far
 * we have not encountered any problems by doing so.
 */

#ifndef INADDR_LOOPBACK
#define INADDR_LOOPBACK (u_long)0x7f000001 /* 127.0.0.1 */
#endif

static int pgrp = 0;	/* reset to 0? */

/*
 * Setup the communication paths (bidirectional) between 'me' and
 * the process to be forked, and fork it.
 */
int create_process(addr, cstate)
    bcpaddr_t addr;
    commstate_t *cstate;
{
    int index = addr.ba_index;
    int server_socket, len;
    struct sockaddr_in server_address, client_address;
    int so;
    
    tprintf("create_process(0x%x:%d, 0x%x)\n",
	    addr.ba_addr, addr.ba_index, cstate);
    
    if ((server_socket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "socket call",
		"create_process", getsyserr());
	return NOTOK;
    }
    server_address.sin_family = AF_INET;
    server_address.sin_port = 0;
    server_address.sin_addr.s_addr = INADDR_ANY;

    /* BUG FIX */
    bzero(server_address.sin_zero, sizeof (server_address.sin_zero));

    if (bind(server_socket, &server_address, sizeof(server_address)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "bind call", "create_process",
	       getsyserr());
	return NOTOK;
    }
    
    if (listen(server_socket, 1) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "listen call", "create_process",
		getsyserr());
	return NOTOK;
    }
    len = sizeof(struct sockaddr_in);
    if (getsockname(server_socket, &server_address, &len) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "getsockname", "create_process",
		getsyserr());
	return NOTOK;
    }

    dprintf("create_process: server_address %d:0x%x: %d (len %d)\n",
	    server_address.sin_family, server_address.sin_addr.s_addr,
	    server_address.sin_port, len);
    
    switch (cstate->c_pid = fork()) {
    case OK:
	tprintf("create_process: fork - child\n");
	reset_myaddr();
    retry1:
	len = sizeof(struct sockaddr_in);
	if ((so = accept(server_socket, &client_address, &len)) == NOTOK) {
	    if (errno == EINTR) {
		eprintf(EF_SYSCALL, COMMUNICATION, "accept", "create_process",
			getsyserr());
		goto retry1;
	    }
	    eprintf(EF_SYSCALL, COMMUNICATION, "accept", "create_process",
		    getsyserr());
	    exit(1);
	}
	dprintf("create_process: accepted conn from %d:0x%x: %d (len %d)\n",
	    client_address.sin_family, client_address.sin_addr.s_addr,
	    client_address.sin_port, len);
	
	cstate->c_read = cstate->c_write = so;
	cstate->c_pid = getpid();
#ifdef BSD_PGRP
	if (pgrp == 0)
	    pgrp = cstate->c_pid;
	cstate->c_pgrp = pgrp;
	if (setpgrp(0, cstate->c_pgrp) == NOTOK) {
       	    eprintf(EF_SYSCALL, INTERNAL, "setpgrp", "create_procedure");
	    exit(0);
	}
#else SYSV
#ifdef notdef
	(void)setpgrp();
#endif notdef	
	cstate->c_pgrp = 0;
#endif BSD_PGRP
	
	/* close unused fd's in bcp_init() */
	child_procedure(addr);
	exit(0);
	/* NOTREACHED */
	
    default:
	tprintf("create_process: fork - parent\n");
	(void)close(server_socket);
	so = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (so == NOTOK) {
	    eprintf(EF_SYSCALL, COMMUNICATION, "socket call",
		    "create_process", getsyserr());
	    return NOTOK;
	}
	client_address.sin_family = AF_INET;
	client_address.sin_port = 0;
	client_address.sin_addr.s_addr = INADDR_ANY;
	
	/* BUG FIX */
	bzero(client_address.sin_zero, sizeof (client_address.sin_zero));
	
	if (bind(so, &client_address, sizeof(client_address)) == NOTOK) {
	    eprintf(EF_SYSCALL, COMMUNICATION, "bind call", "create_process",
		    getsyserr());
	    return NOTOK;
	}
#ifdef notdef	
	if (get_myipaddr(&server_address.sin_addr) == NOTOK) {
	    eprintf(EF_IN3, COMMUNICATION, "get_myipaddr", "create_process");
	    return NOTOK;
	}
#else
	server_address.sin_addr.s_addr = ntohl(INADDR_LOOPBACK);
#endif
	
    retry:
	if (connect(so, &server_address, sizeof(server_address)) == NOTOK) {
	    eprintf(EF_SYSCALL, COMMUNICATION, "connect call", "create_process",
		    getsyserr());
	    if (errno == ECONNREFUSED || errno == EINTR
		/* || errno == EAFNOSUPPORT */) {
		/* give the child a chance to accept */
		bench_delay((double)1.0);
		goto retry;
	    }
	    eprintf(EF_SYSCALL, COMMUNICATION, "connect call failed",
		    "create_process",
		    getsyserr());
	    return NOTOK;
	}
	cstate->c_read = cstate->c_write = so;
#ifdef BSD_PGRP
	if (pgrp == 0)
	    pgrp = cstate->c_pid;
	cstate->c_pgrp = pgrp;
#else
	cstate->c_pgrp = 0;
#endif BSD_PGRP
	break;
	
    case NOTOK:
   	eprintf(EF_IN3, RESOURCE, "No more processes", "create_process");
	/* return a diagnostics string */
	return NOTOK;
    }
    return OK;
} /* create_process */

/*  */

child_procedure(addr)
    bcpaddr_t addr;
{
    if (addr.ba_addr == BA_CLIENT)
	client_procedure(addr);
    else if (addr.ba_addr == BA_SERVER)
	server_procedure(addr);
    else
	eprintf(EF_IN3, INTERNAL, PARAMETER, "child_procedure");
} /* child_procedure */

/*  */

int local_addr(addr)
    bcpaddr_t addr;
{
    int i;

    if (addr.ba_addr & BA_CLIENTS) {
	for (i = 0; i < system_client_nproc; i++)
	    if (system_client_procs[i].c_pid != 0)
		return 1;
    } else if (addr.ba_addr & BA_CLIENT) {
	if (addr.ba_index < system_client_nproc
	    && system_client_procs[addr.ba_index].c_pid != 0)
		return 1;
    }
    if (addr.ba_addr & BA_SERVERS) {
	for (i = 0; i < system_server_nproc; i++)
	    if (system_server_procs[i].c_pid != 0)
		return 1;
    } else if (addr.ba_addr & BA_SERVER) {
	if (addr.ba_index < system_server_nproc
	    && system_server_procs[addr.ba_index].c_pid != 0)
		return 1;
    }
    return 0;
} /* local_addr */

/*  */

/*
 * Note: since system_{client,server}_nproc reflects number of processes
 * on this machine only they can't be used to decide wheather all
 * processes are local.
 */
int remote_addr(addr)
    bcpaddr_t addr;
{
    int i;

    if (addr.ba_addr & (BA_BENCH | BA_SYSTEM | BA_SYSTEMS))
	return 1;

    if (addr.ba_addr & BA_CLIENTS) {
	return 1;	
    } else if (addr.ba_addr & BA_CLIENT) {
	if (addr.ba_index >= system_client_nproc) {
	    return 1;
	}
	if (system_client_procs[addr.ba_index].c_pid == 0)
		return 1;
    }
    if (addr.ba_addr & BA_SERVERS) {
	return 1;
    } else if (addr.ba_addr & BA_SERVER) {
	if (addr.ba_index >= system_server_nproc) {
	    return 1;
	}
	if (system_server_procs[addr.ba_index].c_pid == 0)
		return 1;
    }
    return 0;
} /* remote_addr */

/*  */

extern char *diag;
extern char diagstr[];

#define BUFSIZE 2048

static send_failure(msg)
    msg_t *msg;
{
    static char buf[BUFSIZE];
    int ret;

    diag = diagstr;	/* reset buffer of diagnostics messages */

    sprintf(buf, "Failure from %s: %s\n",
	    bcpaddr2str(get_myaddr()), diag);

    if (msg->msg_datalen && msg->msg_data)
	free(msg->msg_data);
    
    msg->msg_datalen = strlen(buf) + 1;
    msg->msg_data = buf;
    msg->msg_code = MSG_FAILURE;
    msg->msg_to.ba_addr = BA_BENCH;
    msg->msg_to.ba_index = 0;
    ret = send_msg(msg);
    msg->msg_data = NULL;
    msg->msg_datalen = 0;

    return ret;
} /* send_failure */
