/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * The Benchmark Control Protocol - transport between machines and between
 * processes.
 */

#include <general.h>
#include <ipc.h>

/*
 * Exports:
 *	transport_server_close() -> OK
 *	transport_server_init() -> OK/NOTOK
 *	transport_server_listen() -> OK/NOTOK
 *	transport_connect_server(addr: bcpaddr_t) -> socket/NOTOK
 *			connect to the SYSTEM server with the address specified
 *
 *	transport_send_msg(msg: msg_t *) -> OK/NOTOK
 *	transport_recv_msg(mask: int; msg: msg_t *) -> OK/NOTOK
 *	transport_mask(a_mask: int *; a_nfd: int *) -> OK/NOTOK
 */

/*  Implementation dependent #include's */

#ifndef __SOCKET__
#define __SOCKET__
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif __SOCKET__


#ifndef __NETDB__
#define __NETDB__
#include <netdb.h>
#endif __NETDB__

/*  Exports and Imports */

extern int portnum;

/*  */

static int transport_socket = NOTOK,
    server_socket = NOTOK;

/*  */

int transport_server_close()
{
    if (server_socket != NOTOK) {
	(void)close(server_socket);
	server_socket = NOTOK;
    }
    return OK;
} /* transport_server_close */

int transport_send_msg(msg)
   msg_t *msg;
{
     return fd_send_msg(&transport_socket, msg);
}

int transport_recv_msg(mask, msg)
    int mask;
    msg_t *msg;
{
    if (mask & (1 << transport_socket)) {
	return fd_recv_msg(&transport_socket, msg);
    } else {
	eprintf(EF_IN3, INTERNAL, "Bad mask", "transport_recv_msg");
	return NOTOK;
    }
}

transport_mask(a_mask, a_nfd)
    int *a_mask, *a_nfd;
{
    if (transport_socket != NOTOK) {
	*a_mask = 1<<transport_socket;
	*a_nfd = transport_socket + 1;
    } else
	*a_nfd = 0;
} /* transport_mask */

/*  */

int transport_connect_server(addr)
    bcpaddr_t addr;
{
    int s, len;
    struct sockaddr_in otheraddr, myaddr;
    
    tprintf("transport_connect_server(0x%x:0x%x)\n",
	    addr.ba_addr, addr.ba_index);

    if ((s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "socket call",
		"transport_connect_server", getsyserr());
	return NOTOK;
    }
    
    myaddr.sin_family = AF_INET;
    myaddr.sin_port = 0;
    myaddr.sin_addr.s_addr = INADDR_ANY;

    /* BUG FIX */
    bzero(myaddr.sin_zero, sizeof (myaddr.sin_zero));

    if (bind(s, &myaddr, sizeof(myaddr)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "bind call",
		"transport_connect_server",
	       getsyserr());
	return NOTOK;
    }

    otheraddr.sin_family = AF_INET;
    otheraddr.sin_port = ntohs(portnum);
    otheraddr.sin_addr.s_addr = ntohl(addr.ba_index);

    if (connect(s, &otheraddr, sizeof(otheraddr)) == NOTOK) {
	if (errno == ECONNREFUSED) {
	    dprintf("Failed connecting to the benchmark demon\n");
	    return NOTOK;
	}
	eprintf(EF_SYSCALL, COMMUNICATION, "connect call",
		"transport_connect_server",
	       getsyserr());
	return NOTOK;
    }

    tprintf("transport_connect_server - done\n");
    
    return s;
} /* transport_connect_server */

/*  */

int transport_server_init()
{
    int s;
    struct sockaddr_in myaddr;

    tprintf("transport_server_init()\n");

    if (server_socket != NOTOK)
	(void)close(server_socket);

    if ((s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "socket call",
		"transport_server_init",
	       getsyserr());
	return NOTOK;
    }

    myaddr.sin_family = AF_INET;
    myaddr.sin_port = ntohs(portnum);
    myaddr.sin_addr.s_addr = INADDR_ANY;

    /* BUG FIX */
    bzero(myaddr.sin_zero, sizeof (myaddr.sin_zero));

    if (bind(s, &myaddr, sizeof(myaddr)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "bind call",
		"transport_server_init",
	       getsyserr());
	return NOTOK;
    }

    if (listen(s, 1) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "listen call",
		"transport_server_init",
	       getsyserr());
	return NOTOK;
    }

    server_socket = s;
    return OK;
} /* transport_server_init */

/*  */

int transport_server_listen()
{
    struct sockaddr_in otheraddr;
    int len = sizeof(otheraddr);
    
    tprintf("transport_server_listen()\n");

    if (server_socket == NOTOK || transport_socket != NOTOK)
	return OK;
    
    if ((transport_socket = accept(server_socket, &otheraddr, &len)) == NOTOK) {
        if (errno == EINTR) {
	    return OK;
	}
	eprintf(EF_SYSCALL, COMMUNICATION, "accept",
		"transport_server_listen",
	       getsyserr());
	return NOTOK;
    }
    
    if (len != sizeof(otheraddr)) {
	eprintf(EF_IN4X, COMMUNICATION, "accept",
		"transport_server_listen",
		"Bad length on connecting address");
	dprintf("Reurned %d, expected %d\n", len, sizeof(otheraddr));
	return NOTOK;
    }
    
    return OK;
} /* transport_server_listen */

