/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * The state machine running a single benchmark server procedure
 */

#include <general.h>
#include <ipc.h>
#include <ipc_pres.h>

/* Forwards */
void server_procedure();

/*  */

extern char diagstr[];
extern char *diag;

/* Module local data */

static struct bench *be = NULLBE;
static int pid;
static int myindex;
static struct server_t *server = NULL;
static statistics_t stats, *stat = &stats;

/*  */

void server_procedure(myaddr)
    bcpaddr_t myaddr;
{
    msg_t rqs, *rq = &rqs, rps, *rp = &rps;
    state_t state = UNKNOWN;

    pid  = getpid();
    
    tprintf("server_procedure:%d(0x%x:%d)\n",
	    pid, myaddr.ba_addr, myaddr.ba_index);

    diag = diagstr;
    diag[0] = '\0';
    myindex = myaddr.ba_index;
    
    /* close unused fd's - in forker!! */

    /* just in case - MSG_CREATE could have been multicast */
    myaddr.ba_addr = BA_SERVER;

    init_stdio();

    if (bcp_init(myaddr) == NOTOK) {
	eprintf(EF_IN3, INTERNAL, "bcp_init", "server_procedure");
	eprintf("\tPid %d exiting!\n", pid);
	return;
    }

    rq->msg_to.ba_addr = BA_BENCH;
    rq->msg_to.ba_index = 0;
    rq->msg_datalen = 0;
    
    state = IDLE;

    rq->msg_code = MSG_OK;
    if (send_msg(rq) == NOTOK) {
	eprintf(EF_IN4X, INTERNAL, "send_msg", "server_procedure",
		"Failed for MSG_OK");
	(void)send_failure(rq);
	state = UNKNOWN;
    }

    while (state != UNKNOWN) {
	int secs = childstate2timeout(state);

	if (await_recv_msg(rq, secs) == NOTOK) { /* was bcp_.. */
	    eprintf(EF_IN3, INTERNAL, "await_recv_msg", "server_procedure");
	    (void)send_failure(rq);
	    return;
	}

	rp->msg_to = rq->msg_from;
	if (rp->msg_to.ba_addr == BA_SYSTEM)
	    rp->msg_to.ba_addr = BA_BENCH;
	rp->msg_datalen = 0;

	switch(rq->msg_code) {

	case MSG_TIMEOUT:
	    switch (state) {
	    case EXECUTING:
		eprintf(EF_IN3, INTERNAL, "MSG_TIMEOUT",
			"state EXECUTING");
		break;
	    case IDLE:
	    case INIT:
		dprintf("Timed out in state %s - continuing\n",
			state2str(state));
		break;

	    case WAIT_INIT:
	    default:
		eprintf("Timed out in illegal state %s - aborting\n",
			state2str(state));
		rp->msg_code = MSG_FAILURE;
		state = IDLE;
		rp->msg_to.ba_addr = BA_BENCH;
		(void)send_msg(rp);
		break;
	    }
	    break;

	case MSG_RESET:
	    eprintf("Received a MSG_RESET!\n");
	    rp->msg_code = OK;
	    (void)send_msg(rp);
	    state = IDLE;
	    break;
	    
	case MSG_CREATE:
	case MSG_DESTROY:
	    /* should't get this - report the message */
	    eprintf(EF_IN3, PROTOCOL, "Received a bad message",
		    "server_procedure");
	    dprintf("\tCode %s, from 0x%x:%d\n", code2str(rq->msg_code),
		    rq->msg_from.ba_addr, rq->msg_from.ba_index);
	    dprintf("\tin state %s\n", state2str(state));
	    rp->msg_code = MSG_PROERROR;
	    state = IDLE;
	    (void)send_msg(rp);
	    break;
	
	case MSG_PROERROR:
	    eprintf("Received a MSG_PROERROR message\n");
	    dprintf("\tCode %s, from 0x%x:%d\n", code2str(rq->msg_code),
		    rq->msg_from.ba_addr, rq->msg_from.ba_index);
	    dprintf("\tin state %s\n", state2str(state));
	    eprintf("Protocol error: resetting benchmark\n");
	    state = IDLE;
	    (void)send_aborted(rp);
	    break;

	case MSG_FATAL:
	    eprintf("Benchmark killed by an external signal\n");
	    state = UNKNOWN;
	    (void)send_aborted(rp);
	    break;

	case MSG_ABORT:
	    eprintf("Benchmark aborted by %s\n",
		    bcpaddr2str(rq->msg_from));
	    if (state == INIT) {
		if (destroy_server(server) == NOTOK) {
		    eprintf(EF_IN3, COMMUNICATION, "destroy_server",
			    "server procedure");
		    send_failure(rp);
		    state = IDLE;
		    break;
		}
	    }
	    state = IDLE;
	    (void)send_aborted(rp);
	    break;
	    
	case MSG_CLOSED:
	    eprintf(EF_IN3, PROTOCOL,
		    "Unexpectedly lost contact with main (demon) process",
		    "server_procedure");
	    state = UNKNOWN;

	default:
	    caseonstate(&state, rq, rp);
	    break;
	}
	if (rq->msg_datalen) {
	    dprintf("freeing rq->msg_data: 0x%x\n", rq->msg_data);
	    free(rq->msg_data);
	    rq->msg_datalen = 0;
	}
	if (rp->msg_datalen) {
	    dprintf("freeing rp->msg_data: 0x%x\n", rp->msg_data);
	    free(rp->msg_data);
	    rp->msg_datalen = 0;
	}
	stprintf("In state %s\n", state2str(state));
    } /* while (state != UNKNOWN) */

} /* server_procedure */

/*  */

static caseonstate(a_state, rq, rp)
    state_t *a_state;
    msg_t *rq, *rp;
{
    state_t state = *a_state;
    struct address_t *server_addr = NULL;

    switch (state) {

/*  */
	
    case IDLE:
	switch (rq->msg_code) {
            int length;
	    
	case MSG_SPEC:
	    if (rq->msg_from.ba_addr != BA_BENCH) {
		handle_proerror(rq, rp, state);
		break;
	    }
	    length = rq->msg_datalen;
	    if (be != NULLBE)
		bench_free(be);
	    be = str2basic(rq->msg_data, &length);
	    if (be == NULLBE) {
		eprintf(EF_IN3, INTERNAL, "str2basic", "server_proc");
		send_failure(rq);
		break;
	    }

	    if (server_addr != NULL) {
		address_free(server_addr);
		server_addr = NULL;
	    }
	    if (create_server(&server, &server_addr) == NOTOK) {
		eprintf(EF_IN3, COMMUNICATION, "create_server",
			"server procedure");
		send_failure(rp);
		state = IDLE;
		break;
	    }
	    state = INIT;
	    rp->msg_data = address2str(server_addr, &rp->msg_datalen);
	    address_free(server_addr);
	    server_addr = NULL;
	    if (rp->msg_data == NULL) {
		eprintf(EF_IN4, INTERNAL, PROTOCOL, "address2str",
			"server_procedure");
		send_failure(rp);
		state = IDLE;
	    }
	    rp->msg_code = MSG_ADDR;
	    (void)send_msg(rp);
	    break;

	case MSG_STOP:
	    /* ignore */
	    break;
	case MSG_TERMINATE:
	    rp->msg_code = MSG_TERMINATED;
	    (void)send_result(rp, stat);
	    break;
	    
	case MSG_NOTOK:
	case MSG_FAILURE:
	    report_failure(rq, "waiting for MSG_SPEC");
	    state = IDLE;
	    (void)send_aborted(rp);
	    break;
	default:
	    handle_proerror(rq, rp, state);
	    state = IDLE;
	    break;
	}
	break;

/*  */
	
    case INIT:
	switch (rq->msg_code) {
	case MSG_START:
	    /*
	     * start the benchmark execution - only listening to interupting
	     * messages
	     */
	    interupted = 0;
	    rp->msg_code = MSG_STARTED;
	    (void)send_msg(rp);
	    
	    state = EXECUTING;

	    if (exec_server(server, be, stat) == NOTOK) {
		/* diagnostics in diag */
		send_failure(rq);
		state = INIT;
		break;
	    }

	    if (interupted) {
		dprintf("Benchmark procedure was interupted!\n");
		break;
	    }
	    break;

	case MSG_STOP:
	case MSG_TERMINATE:
	    if (rq->msg_code == MSG_STOP)
		rp->msg_code = MSG_STOPPED;
	    else
		rp->msg_code = MSG_TERMINATED;
	    (void)send_result(rp, stat);
	    state = INIT;
	    break;
	    
	case MSG_NOTOK:
	case MSG_FAILURE:
	    report_failure(rq, "benchmark execution");
	    state = IDLE;
	    (void)send_aborted(rp);
	    break;
	default:
	    handle_proerror(rq, rp, state);
	    state = IDLE;
	    break;
	}
	break;
	
/*  */
	
    case EXECUTING:
	/* interupting messages caught here! */
	switch (rq->msg_code) {

	case MSG_STOP:
	case MSG_TERMINATE:
	    if (rq->msg_code == MSG_STOP)
		rp->msg_code = MSG_STOPPED;
	    else
		rp->msg_code = MSG_TERMINATED;
	    (void)send_result(rp, stat);
	    state = INIT;
	    break;

	case MSG_NOTOK:
	case MSG_FAILURE:
	    report_failure(rq, "executing benchmark");
	    state = IDLE;
	    (void)send_aborted(rp);
	    break;
	default:
	    handle_proerror(rq, rp, state);
	    state = IDLE;
	    break;
	}
	break;

/*  */
	
    case UNKNOWN:
    case CREATING:
    case SETUP:
    case WAIT_INIT:
    case WAIT_STARTED:
    case WAIT_STOPPED:
    case WAIT_TERM:
    case TAREDOWN:
    case RESETTING:
    case FAILING:
    case ABORTING:
	dprintf("%s: %s %s in state %s\n", INTERNAL, PROTOCOL,
		"server_procedure",
		state2str(state));
	break;

    }

    *a_state = state;
} /* caseonstate */

/*  */

static int handle_proerror(rq, rp, state)
    msg_t *rq, *rp;
    int state;
{
    dprintf("%s: %s %s in state %s\n", PROTOCOL, "Received a",
	    code2str(rq->msg_code),
	    state2str(state));
    rp->msg_code = MSG_PROERROR;
    return send_msg(rp);
} /* handle_proerror */

/*  */

static int send_result(msg, stat)
    msg_t *msg;
    statistics_t *stat;
{
    tprintf("send_result(0x%x, 0x%x)\n", msg, stat);

    msg->msg_to.ba_addr = BA_BENCH;
    msg->msg_data = statistics2str(stat, &msg->msg_datalen);
    if (msg->msg_data == NULL) {
	eprintf(EF_IN4, INTERNAL, PROTOCOL, "statistics2str", "send_result");
	send_failure(msg);
	return NOTOK;
    }
    return send_msg(msg);
} /* send_result */
    
/*  */

#define BUFSIZE 2048

static send_failure(msg)
    msg_t *msg;
{
    static char buf[BUFSIZE];
    int ret;

    diag = diagstr;	/* reset buffer of diagnostics messages */
    
    sprintf(buf, "Failure from SERVER:%d: %s\n",
	    myindex, diag);

    if (msg->msg_datalen && msg->msg_data) {
	dprintf("send_failure: freeing 0x%x\n", msg->msg_data);
	free(msg->msg_data);
    }
    
    msg->msg_datalen = strlen(buf) + 1;
    msg->msg_data = buf;
    msg->msg_code = MSG_FAILURE;
    msg->msg_to.ba_addr = BA_BENCH;
    ret = send_msg(msg);
    msg->msg_data = NULL;
    msg->msg_datalen = 0;

    return ret;
} /* send_failure */

/*  */

static int send_aborted(msg)
    msg_t *msg;
{
    msg->msg_code = MSG_ABORTED;
    msg->msg_to.ba_addr = BA_BENCH;
    msg->msg_datalen = 0;
    return send_msg(msg);
} /* send_aborted */

/*  */

static report_failure(msg, str)
    msg_t *msg;
    char *str;
{
    eprintf("Error occured while %s:\n\t", str);
    if (msg->msg_datalen)
	eprintf("%s\n", msg->msg_data);
    else
	eprintf("Unspecified error - message code %s\n",
		code2str(msg->msg_code));
} /* report_failure */
