/*
 * Copyright (c) 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 * Internet Rover miscellaneous "C" Tools
 *	Note - This code may be called by signal handlers, and as such 
 *	you should avoid using global variables.
 *
 */
#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <time.h>
#include <string.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <memory.h>
#include <sys/param.h>
#include "ctools.h"

#define LOCKDIR "/tmp/"
#define MAXTRIES 70
#define SECSDAY (60*60*24)
#define SECSHR  (60*60)
#define SECSMIN 60

#ifndef FALSE
#define FALSE 0
#define TRUE 1
#endif

#ifdef S_IRWXU
#define S_RWALL (S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH)
#define S_RWOWNERGROUP (S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH )
#else
#define S_RWALL	0000666
#define S_RWOWNERGROUP 0000664
#endif

/*static char err_buf[BUFSIZ];	/* for use in error logging */

extern char *progname;

int EmptyIfFileDoesntExist=1;	/*** Do we empty non-existent files? ***/


char *itoa(n, s)
int n;
char *s;
{
	int i, sign;

	if ((sign=n) <0)
		n = -n;
	i=0;
	do
	{
		s[i++] = n % 10 + '0';
	} while ((n /= 10) >0);
	if (sign<0)
		s[i++]='-';
	s[i]='\0';
	reverse(s);
	return(s);
}

char *ltoa(n, s)
long n;
char *s;
{
	long i,sign;

	if ((sign=n) <0)
		n = -n;
	i=0;
	do
	{
		s[i++] = n % 10 + '0';
	} while ((n /= 10) >0);
	if (sign<0)
		s[i++]='-';
	s[i]='\0';
	reverse(s);
	return(s);
}

reverse(s)
register char *s;
{
	char c;
	register int i, j;

	for (i = 0, j = strlen(s) - 1; i < j; i++, j--) {
		c = s[i];
		s[i] = s[j];
		s[j] = c;
	}
}

/*
 *  The following routine should be replaced with syslog()
 */
void Log(line, logfile)
char *line; 	/* line to be logged */
char *logfile; 	/* file to log it in */
{
	long TimeNow;
	int fd,length;
	struct tm *tm;
	static int logfilevirgin = 1;
	static char StaticLogFile[MAXPATHLEN];
	char Time[27], RealFileName[MAXPATHLEN];
	extern void exit();
	char err_buf[BUFSIZ];

	if (logfilevirgin) {
		if (logfile==NULL) return; /* bad 1st call */
		strcpy(StaticLogFile, logfile);
		logfilevirgin = 0;
	}
	if (logfile == NULL) 
		logfile=StaticLogFile;

	time(&TimeNow);
	tm = gmtime(&TimeNow);
	sprintf(RealFileName,"%s.%2.2d%2.2d%2.2d",
		logfile,tm->tm_year%100,tm->tm_mon+1,tm->tm_mday);

	strcpy(Time,asctime(tm)); 
	length = strlen(Time);
	Time[length - 1]=' '; /* clip the trailing newline */

	length = strlen(line);

	if ((fd = open(RealFileName,O_CREAT | O_WRONLY | O_APPEND, S_RWALL)) == -1) {
		sprintf(err_buf, "Log: Open Log (%s)",RealFileName);
		fprintf(stderr, err_buf);
		exit(1);
	}
	if ( write(fd,Time,strlen(Time)) != strlen(Time)) {
		sprintf(err_buf, "Log: Write Date to Log (%s)",RealFileName);
		fprintf(stderr, err_buf);
		exit(1);
	}
	if ( write(fd,line,length) != length ) {
		sprintf(err_buf, "Log: Write Log (%s)",RealFileName);
		fprintf(stderr, err_buf);
		exit(1);
	}
	if (close(fd) < 0) 
		syserr("fclose log");
	if ( geteuid() == 0 ) {	/* root? Chown to rover */
		if (chown ( RealFileName, getuid(), getgid())) {
				sprintf(err_buf, "Log: Chown failed (%s)",
								RealFileName);
				fprintf(stderr, err_buf);
				exit(1);
		}
	}
#ifdef ROVERGROUP
	chmod( RealFileName, S_RWOWNERGROUP ); /* Log file for all rover users */
#else
	chmod( RealFileName, S_RWALL ); 	/* Log file for all */
#endif
}

/*****************************************************************************
 *    GetTime - Returns a string in the form we most often want to deal with *
 *                    Mon Mar 23 13:23\0                                     *
 *****************************************************************************/
char *GetTime(s)
char *s;
{
	time_t t;
	struct tm *tm;

	(void) time(&t);
	tm = gmtime(&t);
	sprintf(s, "%s", asctime(tm));
	s[16]=' ';
	s[17]='\0';
	return(s);
}

/****************************************************************
 * FileChanged: Return whether or not the file has changed      *
 ****************************************************************/
FileChanged(filename, lastmodtime)
char *filename; 	/* file we are interested in     */
long *lastmodtime; 	/* Last Modification Time returned from fstat */
{
	int rc,tries;
	struct stat st;
	char hostname[MAXHOSTNAMELEN];
	char err_buf[BUFSIZ];

	/*  	It is possible that the hostfile is editted and checked
		in using a Source Code Control System that removes the 
		file in question.  The file will get checked out again,
		but may not exist for a few seconds and that is OK. */
	for( tries=0; tries<3; tries++ )
		if ((rc = stat(filename, &st)) == 0) break;
		else sleep(5);	/* Allow time for file to be created */

	if( rc != 0 ) {
		sprintf( err_buf, "stat failed on %s : ", filename );
		perror( err_buf );
		gethostname(hostname,  sizeof(hostname));
		sprintf(err_buf, "stat failed rc=%d in %s on %s FileChanged(%s) about to EMPTY FILE\n", rc, progname, hostname, filename);
		Log(err_buf, (char *) NULL);
		if (EmptyIfFileDoesntExist)
			Empty_File(filename);	
			/**MAYBE A PROBLEM..IF NFS FAILS, empty file?*/
		else {
			sprintf(err_buf, "EMPTY ABORTED %s %s %s\n",
						progname, hostname, filename);
			Log(err_buf, (char *) NULL);
		}
		return(TRUE);	/* FILE IS NEW !!! It has changed */
	}
	if (*lastmodtime != st.st_mtime) {
		*lastmodtime = st.st_mtime;
		return(TRUE);	/* File HAS changed */
	}
		return(FALSE);	/* File HAS NOT changed */
	}

/****************************************************************
 * LastMod:	Return 0 if File Doesn't exist or Mod Time	*
 ****************************************************************/
time_t LastMod(filename)
char *filename; 		/* file we are interested in     */
{
	struct stat buf;

	if (stat(filename, &buf) != 0 ) 
		return((time_t) 0);
	return(buf.st_mtime);
}


/****************************************************************
 *        Empty_File:   Empty a file                            *
 ****************************************************************/
void Empty_File(File)
char *File;
{
	int fd;
	char hostname[MAXHOSTNAMELEN];
	char err_buf[BUFSIZ];

	if ((fd=open(File,O_WRONLY | O_CREAT | O_TRUNC, 0644)) == -1) {
		sprintf(err_buf,"Empty_File: open %s failed",File);
		syserr(err_buf);
	}
	if (close(fd) < 0)
		syserr("Empty_File: close");
	gethostname(hostname,  sizeof(hostname));
	sprintf(err_buf,"host=%s progname=%s Empty_File(File=%s) EMPTY\n",
		hostname,progname,File);
	printf(err_buf);
	fprintf(stderr,err_buf);
	Log(err_buf, (char *) NULL);
	if (geteuid() == 0)    /* root? Chmod to r/w everyone */
		if (chown ( File, getuid(), getgid())) {
				sprintf(err_buf, "Log: Chown failed (%s)",
								File);
				fprintf(stderr, err_buf);
		}
#ifdef ROVERGROUP
	chmod( File, S_RWOWNERGROUP ); /* Log file for all rover users */
#else
	chmod( File, S_RWALL ); 	/* Log file for all */
#endif

}

void panic(str)
char *str;
{
	char err_buf[BUFSIZ];

	sprintf(err_buf,"panic(): ERROR: %s\n",str);
	Log(err_buf,  (char *) NULL);
	fprintf(stderr,err_buf);
	perror(str);
	exit(1);
}

void syserr(msg)
char *msg;
{
	extern int errno, sys_nerr;
	extern char *sys_errlist[];
	char err_buf[BUFSIZ];

	perror(msg);
	sprintf(err_buf,"syserr: %s (%d; %s)\n", msg, errno,sys_errlist[errno]);
	Log(err_buf,  (char *) NULL );
	if (errno > 0 && errno < sys_nerr) 
		fprintf(stderr,"; %s)\n",sys_errlist[errno]);
	else
		fprintf(stderr,")\n");
	exit(1);
}

char lock(name)
char *name;    /* File name */
{
	char *path, *lockpath();
	int fd, tries;
	extern int errno;
	static int recursion=0;
	struct stat st;
	time_t t;
	char err_buf[BUFSIZ];

	path = lockpath(name);
	tries = 0;
	while(((fd = open(path, O_WRONLY | O_CREAT | O_EXCL, 0666)) == -1) && (errno == EEXIST)){
		if (++tries >= MAXTRIES) 
			break;
		sleep(1);		/* straight 1 second backoff */
	}
	if (tries >= MAXTRIES) { 	/* See if someone stuck the lock */
		sprintf(err_buf,"%s pid %d: Error...%s File LOCKed - Checking how long it was locked\n",progname, getpid(),name);
		Log(err_buf, (char *) NULL);
		printf(err_buf);
		if (stat(path, &st) < 0) {
			sprintf(err_buf,"lock: LOCK file didn't exist after 3 tries - lets try again\n");
			printf(err_buf);
			return(lock(name));	
		}
		t = time((time_t *) NULL);
		if (t - st.st_ctime > 80 ) {
			sprintf(err_buf, "%s: %s Lock file LOCKED for more than 80 seconds.%ul-%ul=%ul...I'm unlocking the lock\n",progname,name,t,st.st_ctime,t-st.st_ctime);
			Log(err_buf, (char *) NULL);
			fprintf(stderr, err_buf);
			if (unlink(path) < 0) {
				syserr("Unable to unlink LOCK FILE");
				/*NOTREACHED*/
			}
			if (!recursion) {
				recursion++;
				if (lock(name)) {
					recursion=0;
					return(TRUE);
				}
				else 
					return(FALSE);
			}
			else {
				sprintf(err_buf,"LOCK FILE LOCKED, I FREED IT, BUT STILL COULDN'tLOCK THE FILE!!Giving up\n");
				Log( err_buf,  (char *) NULL );
				syserr(err_buf);
				/*NOTREACHED*/
			}
		}
		else return(FALSE);
	}
	if (fd == -1 || close(fd) < 0)
		syserr("lock: ");
		/*NOTREACHED*/
	return(TRUE);
}

void unlock(name)
char *name;  /* free lock */
{
	char err_buf[BUFSIZ];

	if (unlink(lockpath(name)) < 0) {
		sprintf(err_buf,"%s Error: LOCK %s- someone unlocked the lockfile on me!\n",progname,name);
		Log(err_buf, (char *) NULL);
	}
}

char *lockpath(name)
char *name; /* generate lock file path */
{
	static char path[MAXPATHLEN];

	strcpy(path,name);
	return(strcat(path, ".LOCK"));
}

/***********************************************************************
 *   LogChecking : Print to a file what node we are currently checking *
 ***********************************************************************/
void LogChecking(str, filename)
char *str;
char *filename;
                       /* node name and file name to log checking to */
{
	FILE *fp;
	char MyTime[40]; /* Read Buffer */

	if ((fp = fopen(filename,"w")) == NULL)
		syserr("LogChecking: fopen checking file");
	fprintf(fp,"%s %s",GetTime(MyTime),str);
	if (fclose(fp) == EOF) 
		syserr("LogChecking: fclose checking");
}

char *_Day(t)
time_t t;
{
	struct tm *tm = localtime(&t);

	switch( tm->tm_wday ) {
	case 0:	return("Sun");
	case 1:	return("Mon");
	case 2:	return("Tue");
	case 3:	return("Wed");
	case 4:	return("Thu");
	case 5:	return("Fri");
	case 6:	return("Sat");
	default: return("DAY");
	}
}

char *_Time(t)
time_t t;
{
	struct tm *tm = localtime(&t);
	static char Time[10];
	
	sprintf(Time,"%02.2d:%02.2d",tm->tm_hour,tm->tm_min);
	return( Time );
}


char *ElapsedTime(TimeStamp)
time_t TimeStamp; 		/* Elapsed Seconds */
{
	static char buffer[20];
	int eday, ehr, emin;
	time_t TimeNow = time(&TimeNow);
	long diff = TimeNow - TimeStamp;

	if (eday= diff / SECSDAY ) 
		diff %= (eday*SECSDAY);
	if (ehr = diff / SECSHR ) 
		diff %= (ehr*SECSHR);
	emin= diff / SECSMIN;
	if ( eday ) 
		sprintf( buffer," %2.2d:%2.2d:%2.2d", eday, ehr, emin );
	else if ( ehr ) 
		sprintf( buffer,"    %2.2d:%2.2d", ehr, emin );
	else if ( emin ) 
		sprintf( buffer,"      :%2.2d", emin );
	else 
		sprintf(buffer,"       %2.2d",0);
	return( buffer );
}

char *PrintTime(t)
     long int t;
{
char *EventTime;

   EventTime=ctime( &t );
   EventTime[4]=EventTime[11];  /* Put date here */
   EventTime[5]=EventTime[12];  /* Put date here */
   EventTime[6]=EventTime[13];  /* Put date here */
   EventTime[7]=EventTime[14];  /* Put date here */
   EventTime[8]=EventTime[15];  /* Put date here */
   EventTime[9]=' ';
   EventTime[10]='\0';              /* Mon 03:34 \0 */
   return(EventTime);
}

char *trim(line)
char *line;
{
	register char *p;

	for(p = line; *p != '\0'; p++)
		if ((*p == '\t') || (*p == '\n')) 
			*p = ' ';
	return(line);
}

/************************************************************************/
/* The following code is used to produce a datestr of the form YYMMDD	*/
/*   Author:   WB Norton	01/10/92				*/
/************************************************************************/
char *cvt_time_to_datestr(timenow)
long int *timenow;
{
	static char buffer[20];
	struct tm *timestruct;

	timestruct = localtime( timenow );
	sprintf(buffer,"%.2d%.2d%.2d\n", timestruct->tm_year, 
			timestruct->tm_mon+1, timestruct->tm_mday);
	return(buffer);
}

char *datestr(DAYS)
int DAYS;
{
	static char datestring[10];
	time_t timenow;

	timenow = time((time_t *) NULL) - (DAYS * 24 * 60 * 60);
	strcpy(datestring,cvt_time_to_datestr( &timenow )); 
	datestring[strlen(datestring)-1]='\0';	/* kill \n */
	return( datestring );
}
