/*
 * Copyright (c) 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

/************************************************************************
 *			map.c -	Draw a status map			*
 *									*
 *	This code should be split into functional groups, for fast 	*
 *	compilation							*
 *	Global variables should	be minimized, putting some of these into*
 *	userData field of the drawarea widget.				*
 *									*
 ************************************************************************/
#include <stdio.h>
#include <memory.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/wait.h>
#include <time.h>
#include <math.h>
#include "motif.h"
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>
#include <signal.h>

#include "CommonDefs.h"
#include "StatusFileio.h"
#include "pathnames.h"
#include "ProcessNeighborTable.h"
#include "xmap.h"
#include "Linkfileio.h"

#define MAXLINKS 1000
#define MAX_ARGS  32

extern void BuildMyAppResources(), CreateMenus();

int verbose = 0, ShowASs = 0, ReDrawNecessary = 0, MapHasChanged = 0;
char progname[MAXPATHLEN], *Class;
struct NetworkType Network, NewNetwork;
Widget drawarea, NodePB[MAXMAPNODES], MoveWidget;

static int BellActive = 0, BellVolume = 0, NetworkType, Moving = 0;
static int CheckStatusFileInterval = 5, CollectorDiedInterval = 5;
static int debug = FALSE;

static char *DefaultPingkyDir = DEFAULT_PINGKY_DIR;
static char hostfile[MAXPATHLEN];	/* List	of nodes we are	checking */
static char cycletime[MAXPATHLEN];	/* How long did	pingky take to cycle */
static char logfile[MAXPATHLEN];	/* Where do we log things */
static char StateFile[MAXPATHLEN];
static char MapFile[MAXPATHLEN];
char Error_Log[MAXPATHLEN];		/* Where node/link transitions log */
char *Community="public";
char *pingkydir;			/* set in main - used to find data */

/*****************************************************************************
 * ProcessCmd - replace $1 and	$2 with	names of Nodes/Links user clicked on *
 *****************************************************************************/
char *ProcessCmd(buffer, node1, node2)
char *buffer, *node1, *node2;
{
	static char cmd[BUFSIZ];

	if (debug)
		printf("ProcessCmd(%s, %s, %s)", buffer, node1, node2);
	cmd[0] = '\0';
	while  (*buffer != '\0')
		if ((*buffer == '$') && (*(buffer + 1) != '\0')) {
			if (*(buffer + 1) == '1') {
				strcat(cmd, node1);
				buffer += 2;
			} else if (*(buffer + 1) == '2') {
				strcat(cmd, node2);
				buffer += 2;
			} else {
				strncat(cmd, "$", 1);
				buffer++;
			}
		}
		else {
			strncat(cmd, buffer, 1);
			buffer++;
		}
	if (debug)
		printf("ProcessCmd() returns [%s]\n", cmd);
	return(cmd);
}

static char *NetClass="NETWORKCLASS=";

/************************************************************************
 *  FeedSystem() - The user selected a menu item or invoked an action 	*
 *		routine.  Pass all the goodies to the script specified.	*
 ************************************************************************/
void FeedSystem(n1, n2, Cmd)
char *n1, *n2, *Cmd;
{
	struct NodeType *np1, *np2;
	char *argv[MAX_ARGS];
	register int i;
	char *s;
	register char *p;
	extern char Version[];
	extern char *Class;
	/*
	 *  These are all going into our environment.
	 */
	static char node2name[BUFSIZ], node2type[BUFSIZ], node2state[BUFSIZ],
		node1name[BUFSIZ], node1type[BUFSIZ], node1state[BUFSIZ],
		NetworkClass[BUFSIZ], networkstatefile[BUFSIZ]; 
	static char node1uniqueid[BUFSIZ], node2uniqueid[BUFSIZ],
		node1ifaddr[BUFSIZ], node2ifaddr[BUFSIZ], circuit[BUFSIZ],
		TTLastOpened[BUFSIZ], TTNextAction[BUFSIZ],
		node1ifnum[BUFSIZ], node2ifnum[BUFSIZ],
		ProgName[BUFSIZ], LogFile[BUFSIZ], mapfile[BUFSIZ],
		LinkType[BUFSIZ], CommunityString[BUFSIZ],
		PingkyVersion[BUFSIZ], PingkyDir[BUFSIZ];

	char tmp[100], *name;

	struct LinkDetail *LPtr;

	if (debug)
		printf("FeedSystem(%s, %s, %s)\n", n1, n2, Cmd);


	sprintf( PingkyVersion, "PINGKYVERSION=%s", Version );
	putenv( PingkyVersion);
	sprintf( PingkyDir, "PINGKYDIR=%s", pingkydir );
	putenv( PingkyDir);
	/*
	 * Find the node entry associated with this node
	 *
	 */
	if (( np1 = FindNode(&Network, n1)) == NULL) {
		fprintf(stderr, "couldn't find %s\n", n1);
		return;
	}
	sprintf(CommunityString, "COMMUNITY=%s", Community);
	putenv(CommunityString);

	NetworkClass[0]='\0';
	strcpy( tmp, Class );
	strcpy( NetworkClass, NetClass );
	strcat( NetworkClass, tmp );
	putenv(NetworkClass);
	sprintf(networkstatefile, "NETWORKSTATEFILE=%s", StateFile);
	putenv(networkstatefile);
	sprintf(node1uniqueid, "NODE1UNIQUEID=%s", n1);
	putenv(node1uniqueid);
	sprintf(LogFile, "LOGFILE=%s", logfile);
	putenv(LogFile);
	sprintf(ProgName, "PROGNAME=%s", progname);
	putenv(ProgName);
	sprintf(mapfile, "MAPFILE=%s", MapFile);
	putenv(mapfile);
	if ( (name=GetMapNodeName( n1 )) == NULL ) name=np1->Name;
	sprintf(node1name, "NODE1NAME=%s",  name );
	putenv(node1name);
	sprintf(node1type, "NODE1TYPE=%s", PrintNodeType( np1->Type));
	putenv(node1type);
	sprintf(node1state, "NODE1STATE=%s", PrintState( np1->State));
	putenv(node1state);

	/********* Blank out the LINK specific environment vars ****/
	sprintf(node1ifaddr, "NODE1IFADDR=");
	sprintf(node1ifnum, "NODE1IFNUM=");
	sprintf(circuit, "CIRCUIT=");
	sprintf(TTLastOpened, "TTLASTOPENED=");
	sprintf(TTNextAction, "TTNEXTACTION=");
	sprintf(node2name, "NODE2NAME=");
	sprintf(node2uniqueid, "NODE2UNIQUEID=");
	sprintf(node2type, "NODE2TYPE=");
	sprintf(node2state, "NODE2STATE=");
	sprintf(node2ifnum, "NODE2IFNUM=");
	sprintf(node2ifaddr, "NODE2IFADDR=");
	sprintf(LinkType, "LINKTYPE=");
	/*
	 *  If n2 is non-NULL, then this is a LINK; 
	 *  otherwise it is a NODE.
	 */
	if (n2 != NULL) {
		putenv("OBJECTTYPE=LINK");
		if ((np2 = FindNode(&Network, n2)) == NULL) {
			fprintf(stderr, "FindNode() couldn't find %s\n", n2);
			return;
		}
		sprintf(node2uniqueid, "NODE2UNIQUEID=%s", n2);
		if ( (name=GetMapNodeName( n2 )) == NULL ) name=np2->Name;
		sprintf(node2name, "NODE2NAME=%s", name);
		sprintf(node2type, "NODE2TYPE=%s", PrintNodeType(np2->Type));
		sprintf(node2state, "NODE2STATE=%s", PrintState(np2->State));
	
		if ( (LPtr=FindPtPtLink(n1, n2))!=NULL) {
			/*printf("%s %d %s %d %s\n",LPtr->LocalIPAddr,LPtr->LocalDSUNumber, LPtr->Circuit, LPtr->RemoteDSUNumber, LPtr->RemoteIPAddr);*/
			if ( strcmp(LPtr->LocalNode,n1) == 0 ) {
				sprintf(node1ifaddr, "NODE1IFADDR=%s", 
							LPtr->LocalIPAddr );
				sprintf(node2ifaddr, "NODE2IFADDR=%s", 
							LPtr->RemoteIPAddr );
				sprintf(node1ifnum, "NODE1IFNUM=%d", 
							LPtr->LocalDSUNumber );
				sprintf(node2ifnum, "NODE2IFNUM=%d", 
							LPtr->RemoteDSUNumber );
			}
			else {
				sprintf(node1ifaddr, "NODE1IFADDR=%s", 	
							LPtr->RemoteIPAddr );
				sprintf(node2ifaddr, "NODE2IFADDR=%s", 
							LPtr->LocalIPAddr );
				sprintf(node1ifnum, "NODE1IFNUM=%d", 
							LPtr->RemoteDSUNumber );
				sprintf(node2ifnum, "NODE2IFNUM=%d", 
							LPtr->LocalDSUNumber );
			}
			sprintf(circuit, "CIRCUIT=%s", LPtr->Circuit);
			sprintf(LinkType, "LINKTYPE=%d", LPtr->LinkType);
			if ( LPtr->TTLastOpened )
				sprintf(TTLastOpened, "TTLASTOPENED=%s", 
						ctime(&LPtr->TTLastOpened));
			if ( LPtr->TTNextAction )
				sprintf(TTNextAction, "TTNEXTACTION=%s", 
						ctime(&LPtr->TTNextAction));

		} 
	}
	else {
		putenv("OBJECTTYPE=NODE");
	}
	putenv(node1ifaddr);
	putenv(node1ifnum);
#ifdef TESTING
	putenv(node2uniqueid);
	putenv(node2name);
	putenv(node2type);
	putenv(node2state);
	putenv(node2ifaddr);
	putenv(node2ifnum);
	putenv(TTLastOpened);
	putenv( TTNextAction );
	putenv(circuit);
#endif


	/*
	 *  Substitute n1 for "$1" in Cmd and likewise for n2 and "$2".
	 */
	s = ProcessCmd(Cmd, n1, n2);
	if (debug)
		printf("FeedSystem: going to execvp(%s)\n", s);

	/*
	 *  Split s into args suitable for exec().
	 */
	for (i = 0, p = strtok(s, DELIMITERS); p != NULL; 
				i++, p = strtok((char *) NULL, DELIMITERS))
		argv[i] = p;
	argv[i] = NULL;

	{
		register int j;
		printf("Rover EXECUTING: ");
		for (j = 0; j <= i; j++)
			printf("%s ", argv[j]);
		printf("\n");
	}

	switch (vfork()) {
	case -1 :
		syserr("vfork");
		/*NOTREACHED*/
	case  0 :
		execvp(argv[0], argv);
		syserr("execvp");
		/*NOTREACHED*/
	default:
		if (debug)
			printf("Successfully forked off\n");
		break;
	}
}

/************************************************************************
 *  system1 - Action Routine:  feed the	users command to the system	*
 ************************************************************************/
void system1(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	Arg myArgs[10];
	XmString xmstr;
	char *name, *n1, *n2;
	char cmd[BUFSIZ];
	int i, x, y, Node1, Node2;
	extern char *GetIPAddr();

	if (debug)
		printf("system1()\n");

	XDefineCursor(XtDisplay(w), XtWindow(w), 
			XCreateFontCursor(XtDisplay(w), XC_watch));
	cmd[0] = '\0';

	/* fetch the user's command */
	for (i = 0; i < *num_params; i++ ) {
		strcat(cmd, params[i]);
		strcat(cmd, " ");
	}

	if (w == drawarea) {		/* LINK OBJECT */
		x = event->xbutton.x;
		y = event->xbutton.y;
		if (Link2Nodes(x, y, &Node1, &Node2) == 1) {
			n1 = MapNode[Node1].IPAddr;
			n2 = MapNode[Node2].IPAddr;
			FeedSystem(n1, n2, cmd);
		} else {
			printf("Couldn't find link\n");
			XDefineCursor(XtDisplay(w), XtWindow(w), 
				XCreateFontCursor(XtDisplay(w), XC_left_ptr));
		}
	} else {			/* NODE OBJECT */
		XtSetArg(myArgs[0], XmNlabelString, &xmstr);
		XtGetValues(w, myArgs, 1);
		if (XmStringGetLtoR(xmstr, XmSTRING_DEFAULT_CHARSET, &name) !=	True) {
			printf("Odd - couldn't get label on node - aborting\n");
			XDefineCursor( XtDisplay(w), XtWindow(w), 
				XCreateFontCursor(XtDisplay(w), XC_left_ptr));
		} else {
			n1 = GetIPAddr(name);
			n2 = NULL;
			FeedSystem(n1, n2, cmd);
		}
	}
	XDefineCursor(XtDisplay(w), XtWindow(w), 
				XCreateFontCursor(XtDisplay(w), XC_left_ptr));
}

/************************************************************************
 *	AddNodePB() - Add a Node to the	Map				*
 *		This involves creating a Node structure, + PushButton	*
 *		to Parent (drawarea)					*
 ************************************************************************/
void AddNodePB(MapIndex)
int MapIndex;
{
	int i = MapIndex, ac = 0;
	Arg myArgs[10];
	XmString xmstr;

	if (debug)
		printf("AddNodePB(%d)\n", MapIndex);

	xmstr = XmStringCreate(MapNode[i].Name, XmSTRING_DEFAULT_CHARSET);
	XtSetArg(myArgs[ac], XmNx, MapNode[i].x); 
	ac++;
	XtSetArg(myArgs[ac], XmNy, MapNode[i].y); 
	ac++;
	XtSetArg(myArgs[ac], XmNlabelString, xmstr); 
	ac++;
	NodePB[i] = XtCreateManagedWidget(MapNode[i].Name, 
			xmPushButtonWidgetClass, drawarea, myArgs, ac);
}

/************************************************************************
 * DrawMap() - Color the Nodes and draw	the links, where the colors	*   
 *				reflect	the node/link state.		*   
 *		Add code here to deal with T1/T3 links:	GetLineWidth()	*
 ************************************************************************/
static void DrawMap()
{
	static int virgin = 1;
	extern int PlaybackDialogActive;
	extern void AllocateColors(), ColorNodes(), ColorLinks();

#ifdef MANYEXPOSE
	static time_t thisdraw, lastdraw;
	extern time_t time();
#endif
	if (debug)
		printf("DrawMap()\n");

#ifdef MANYEXPOSE
	thisdraw = time(&thisdraw);
	if (((thisdraw - lastdraw) < 0 ) && !PlaybackDialogActive) {
		/*
		 *  Don't draw multiple times.
		 *  We are getting lots of these within a few seconds...Why?
		 */
		printf("Ignoring redraw request\n");
		return;
	}
#endif
	if (virgin) {
		AllocateColors(drawarea);
		virgin = 0;
	}
	ColorNodes(&drawarea, &Network, ShowASs);
	ColorLinks(&drawarea, &Network, ShowASs);
	ReDrawNecessary = 0;	/* We have drawn the map,so ReDrawNotNecessary*/
#ifdef MANYEXPOSE
	lastdraw = time(&lastdraw);
#endif
}

/**************************************************************************
 * ClearAndReadStatusFile() - Read the specified network status file and  *
 *			free the structures associated with the old one.  *
 **************************************************************************/
void ClearAndReadStatusFile(CallerStateFile)
char *CallerStateFile;
{
	if (debug)
		printf("ClearAndReadStatusFile()\n");
	if (CallerStateFile == NULL) 
		CallerStateFile = StateFile;
	if (debug)
		printf("Network Status File is %s\n", CallerStateFile);
	ReadConfig(&NewNetwork, NetworkType, CallerStateFile);
	FreeConfig(&Network);
	memcpy((char *) &Network, (char *) &NewNetwork, 
					sizeof(struct NetworkType));
	memset((char *) &NewNetwork, 0, sizeof(struct NetworkType));
}

/************************************************************************
 *  ClearAndDrawMap() - Force a redraw 					*
 *	Note that this will JUST Clear and Draw				*
 ************************************************************************/
ClearAndDrawMap()
{
	if (debug)
		printf("ClearAndDrawMap\n");
	XClearWindow(XtDisplay(drawarea), XtWindow(drawarea));
	DrawMap();
}

/*****************************************************************************
 *  ReDrawMap() - Force a redraw ( and Reread the status file )		     *
 *		Note that this will FREE and ReRead the NetworkStatusFile    *
 *****************************************************************************/
void ReDrawMap()
{
	extern int PlaybackDialogActive;

	if (debug)
		printf("ReDrawMap()\n");
	if (!PlaybackDialogActive)
		ClearAndReadStatusFile((char *) NULL);
	ClearAndDrawMap();
}

/************************************************************************
 *	ObjectStateTransition() - Interface to the drawing routines	*
 *			When objects change state, or would like to	*
 *			( as in playback ) call this routine.		*
 ************************************************************************/
void ObjectStateTransition(tp)
struct TransitionCallback *tp;
{
	struct LinkType *l;
	struct NodeType *n;
	int x, y;
	extern int PlaybackDialogActive;
	extern void ColorNode(), ColorLink(), exit();
	char cmd[100];
	extern char *GetShowNode();

	if (debug)
		printf("ObjectStateTransition(0x%x)\n", tp);

	if ((tp->Type > 10) || (tp->OldState > 10) || (tp->NewState > 10)) {
		fprintf(stderr,
			"ObjectStateTransition() - bad data - ignored\n");
		return;
	}

	switch (tp->Type) {
	case NODETRANSITION:
		if (debug)
			printf("%s NODE State Transition %s %s %s->%s\n",
			  PrintNodeType(tp->cptr.N.Type), tp->cptr.N.Name, 
			  tp->cptr.N.IPAddress, PrintState(tp->OldState),
			  PrintState(tp->NewState));
		if (tp->PleaseDoIt) {
			n = FindNode(&Network, tp->cptr.N.IPAddress);
			if (n == NULL)
				fprintf(stderr, "I tried to migrate the state but couldn't find the node: %s\n",tp->cptr.N.IPAddress);
			else {
				n->State = tp->NewState;
				if (BellActive && 
					isViewable(tp->cptr.N.IPAddress)) {
					XBell(XtDisplay(drawarea), BellVolume);
					if ( tp->cptr.N.State == DOWN ) {
						sprintf(cmd,"%s -S %s -a -title View_of_%s&",progname, tp->cptr.N.IPAddress,tp->cptr.N.IPAddress);
						printf("%s -S %s -a -title View_of_%s &\n",progname, tp->cptr.N.IPAddress,tp->cptr.N.IPAddress);
						system(cmd);
					}
				}
			}
		}
		if (!ReDrawNecessary)
			if ( ( tp->cptr.N.Type == AS ) && (ShowASs==0)) break;
			ColorNode(&tp->cptr.N, ShowASs);
		break;

	case LINKTRANSITION:
		if (debug)
			printf("%s LINK State Transition %s %s %s->%s\n",
			  PrintLinkType(tp->cptr.L.Type), tp->cptr.L.Node1,
			  tp->cptr.L.Node2, PrintState(tp->OldState),
			  PrintState(tp->NewState));
		if ((GetXY(tp->cptr.L.Node1, &x, &y) == 0 ) || 
		    (GetXY(tp->cptr.L.Node2, &x, &y) == 0 ))
			return;
		if (tp->PleaseDoIt) {
			l = FindLink(&Network, tp->cptr.L.Node1,
					tp->cptr.L.Node2, tp->cptr.L.Type);
			if (l == NULL) {
				fprintf(stderr, "Cannot find link %s-%s\n",
					tp->cptr.L.Node1,tp->cptr.L.Node2);
				break;
			}
			if (debug)
				printf("Found link %s-%s; migrating %d -> %d\n",
						l->Node1, l->Node2, l->State, 
						tp->NewState);
			l->State = tp->NewState;
			if (BellActive && isViewable(tp->cptr.L.Node1) && 
			    isViewable(tp->cptr.L.Node2)) {
				XBell(XtDisplay(drawarea), BellVolume);
					if ( tp->cptr.L.State == DOWN ) {
						sprintf(cmd,"%s -S %s -a -title %s-%s&",progname, tp->cptr.L.Node1,tp->cptr.L.Node1,tp->cptr.L.Node2 );
						printf("%s -S %s -a -title %s-%s &\n",progname, tp->cptr.L.Node1,tp->cptr.L.Node1,tp->cptr.L.Node2 );
						system(cmd);
					}
					if ( tp->cptr.L.State == UP ) {
						if (( strcmp(tp->cptr.L.Node1, GetShowNode() ) == 0 )  || 
						( strcmp(tp->cptr.L.Node2, GetShowNode() ) == 0 )) {
							fprintf(stderr,"Problem Cleared up exitting\n");
							exit(1);
						}
					}
				}
		}
		if ((ReDrawNecessary == 0 ) && (((tp->OldState == UP) || 
		    (tp->OldState == NR)) && ((tp->NewState == UP) || 
		    (tp->NewState == NR)))) {
			if (debug)
				printf("tp->OldState=%d tp->NewState=%d\n",
						tp->OldState,tp->NewState);
			ColorLink(&drawarea, l);
		}
		else {
			if (debug)
				printf("Setting ReDrawNecessary\n");
			ReDrawNecessary = 1;  /*Need a clear and redraw*/
		}
		break;

	case NEWNODE:
		printf("NEW %s NODE %s %s %s\n", 
			  PrintNodeType(tp->cptr.N.Type),
			  tp->cptr.N.Name, tp->cptr.N.IPAddress,
			  PrintState(tp->cptr.N.State));
		if (strlen(tp->cptr.N.IPAddress) < 2 ) 
			return;
		if (strlen(tp->cptr.N.Name) < 2 ) 
			return;
		if (debug)
			printf("Setting ReDrawNecessary\n");
		ReDrawNecessary = 1;

		if ( ShouldDisplayNode( &NewNetwork, &tp->cptr.N, ShowASs ) ) {
			fprintf(stderr,"Adding New Node %s %s ShowASs=%d because ShouldDisplayNode said to\n",tp->cptr.N.IPAddress, tp->cptr.N.Name, ShowASs);
			AddNodePB(AddMapNode(tp->cptr.N.Name, 
				tp->cptr.N.IPAddress, -1, -1));
		}
		break;

	case NEWLINK:
		printf(" NEW %s LINK: %s %s %s\n",
			  PrintLinkType(tp->cptr.L.Type), tp->cptr.L.Node1,
			  tp->cptr.L.Node2, PrintState(tp->NewState));
		if (strlen(tp->cptr.L.Node1) < 2 ) 
			return;
		if (strlen(tp->cptr.L.Node2) < 2 ) 
			return;
		if (BellActive)
			XBell(XtDisplay(drawarea), BellVolume);
		if (debug)
			printf("Setting ReDrawNecessary\n");
		ReDrawNecessary = 1;
		break;

	default:
		fprintf(stderr, 
		  "ObjectStateTransition() Unknown TRANSITION type: %d\n",
		  tp->Type);
		break;
	}
}

/****************************************************************************
 * CheckStatusFile() - This routine called periocially to reread status file*
 *					if necessary and redraw	the map.    *
 *		Optimize here -	only redraw if file changes...		    *
 *		Optimize here -	only redraw lines that have changed	    *
 ****************************************************************************/
void CheckStatusFile(id)
XtIntervalId id;
{
	time_t TimeNow;
	struct stat buf, dummy;
	static time_t StateFileModTime;
	extern int PlaybackDialogActive;
	static int virgin = 1;
	char *StateLockFile;
	extern char *lockpath();
	extern void CollectorDied();
	extern int ReadLinkWidthFileIfNecessary();

	if (debug)
		printf("CheckStatusFile(0x%x)\n", id);

	if (PlaybackDialogActive) {
		fprintf(stderr,
			"PlaybackDialog active - ignoring timer callback\n");
		return;
	}
	if (stat(StateFile, &buf) < 0) {
		perror("stat");
		fprintf(stderr, "Can't open statefile %s\n", StateFile);
		goto end;
	}
	if (StateFileModTime == buf.st_mtime)	{
		time(&TimeNow);
		if ((TimeNow - StateFileModTime) > (60 * CollectorDiedInterval))
			CollectorDied();
		goto end; 		/* No Status File Change */
	}

	/*
	 * If the file has changed size, but LOCKED, it is probably
	 * being updated right now!! .... Skip till next time.
	 */
	StateLockFile = lockpath(StateFile);
	if (stat(StateLockFile, &dummy) == 0 ) {
		fprintf(stderr,"Can't stat statefile (LOCKED)\n");
		goto end;
	}

	/* State File Mod Time Changed */
	StateFileModTime = buf.st_mtime;
	if (virgin) {	/* We already read the status file ! */
		virgin = 0;
		ReDrawNecessary = 1;
	} else {
		if (debug)
			printf("CheckStatusFile(): reading NewNet from Status File\n");
		ReadConfig(&NewNetwork, NetworkType, StateFile);
		DiffNetworks(&NewNetwork, &Network, ObjectStateTransition);
		FreeConfig(&Network);   /* Free original Net */
		memcpy((char *) &Network, (char *) &NewNetwork, 
						sizeof(struct NetworkType));
		memset((char *) &NewNetwork, 0, sizeof(struct NetworkType));
	}

end:
	if ( ReadLinkWidthFileIfNecessary() ) ReDrawNecessary=1;
	if (ReDrawNecessary) {
		XClearWindow(XtDisplay(drawarea), XtWindow(drawarea));
		DrawMap();
	}
	XtAddTimeOut((long) CheckStatusFileInterval * 1000, CheckStatusFile, 
							(Widget) NULL);
}

/************************************************************************
 * PlaceIt() - Place the Node at its new coordinates.			*
 *		Update MapFile,	Node coordinates, etc. and redraw map	*
 *									*
 * Replace this	code with a rubberband box that	allows users to	see the	*
 * links stretch during	interactive placement.				*
 ************************************************************************/
/*ARGSUSED*/
static void PlaceIt( w, event,	params,	num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	Position x, y;
	Arg myArgs[10];
	Dimension width, height;
	int centerx, centery;

	if (debug)
		printf("PlaceIt\n");

	if (Moving == 0) 
		return;
	XtSetArg(myArgs[0], XmNwidth, &width);
	XtSetArg(myArgs[1], XmNheight, &height);
	XtGetValues(w, myArgs, 2);
	x = event->xbutton.x;
	centerx = x + (width / 2);	/* get the center x value of the node */
	if (centerx % 10 < 5) 
		centerx = centerx - centerx % 10;
	else 
		centerx = centerx + 10 - centerx % 10;
	x = centerx - (width / 2);
	y = event->xbutton.y;
	centery = y + (height / 2);	/* get the center y value of the node */
	if (centery % 10 < 5) 
		centery = centery - centery % 10;
	else 
		centery = centery + 10 - centery % 10;
	y = centery - (height / 2);

	XtSetArg(myArgs[0], XmNx, x);
	XtSetArg(myArgs[1], XmNy, y);
	XtSetArg(myArgs[2], XmNshadowThickness, 2);
	XtSetValues(MoveWidget, myArgs, 3);
	MapNode[Moving].x = x;	/*+DELTA;*/
	MapNode[Moving].y = y;	/*+DELTA;*/

	Moving = 0;
	ReDrawMap();
	MapHasChanged = 1;
}

/************************************************************************
 * MoveIt() -	Mark the Node for new node placement			*
 *									*
 * Replace this	code with a rubberband box that	allows users to	see the	*
 * links stretch during	interactive placement.				*
 ************************************************************************/
/*ARGSUSED*/
static void MoveIt(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	int j;
	Arg myArgs[10];
	XmString xmstr;
	char *IPAddr, *name;
	extern char *GetIPAddr();

	if (debug)
		printf("MoveIt()\n");

	XtSetArg(myArgs[0], XmNlabelString, &xmstr);
	XtGetValues(w, myArgs, 1);
	if (XmStringGetLtoR(xmstr, XmSTRING_DEFAULT_CHARSET, &name) !=	True) {
		printf("MoveIt:	Error getting NodeName\n");
		return;
	}
	if ((IPAddr = GetIPAddr(name)) == NULL) {
		printf("MoveIt(): Error	with Name\n");
		return;
	}

	if (Moving != 0) {	/* Don't allow two widgets to be highlighted */
		XtSetArg(myArgs[0], XmNshadowThickness, 3);
		XtSetValues(MoveWidget, myArgs, 1);
	}
	MoveWidget = w;
	Moving = GetXY(IPAddr, &j,&j);
	XtSetArg(myArgs[0], XmNshadowThickness, 4);
	XtSetValues(w, myArgs, 1);
}

/************************************************************************
 *   SaveMap() - Save map to disk					*
 ************************************************************************/
/*ARGSUSED*/
static void SaveMap(w, client_data, call_data)
Widget w;
caddr_t client_data;
XtCallbackList *call_data;
{
	extern void WriteMap();

	if (debug)
		printf("SaveMap()\n");
	WriteMap(MapFile);
	MapHasChanged = 0;
}


/************************************************************************
 * Quit() -   Close down the X-Application and exit			*
 *									*
 *	Alot of	the cleanup is expected	to be done by the system	*
 ************************************************************************/
/*ARGSUSED*/
static void Quit(w, client_data, call_data)
Widget w;
caddr_t	client_data;
XtCallbackList *call_data;
{
	static int times;
	extern void exit(), WriteMap();

	if (debug)
		printf("Quit()\n");
	if (MapHasChanged && (client_data != NULL))
		WriteMap(MapFile);
	else {
		if ((client_data == NULL) && (times++ < 1))
			return;
	}
	XCloseDisplay(XtDisplay(drawarea));
	exit(1);
}

/****************************************************************************
 * aquit() -   Action Routine to quit the application			*
 ****************************************************************************/
/*ARGSUSED*/
void aquit(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	if (debug)
		printf("aquit\n");
	Quit(w, (caddr_t) NULL, (XtCallbackList *) NULL);
}

/************************************************************************
 * ExposureCallback() -	Event Handler version of this routine		*
 *			This routine gets called to handle exposures	*
 *			Try to optimize	line redraws here		*
 ************************************************************************/
/*ARGSUSED*/
void MyExposeCallback(w, client_data, call_data)
Widget w;
caddr_t	client_data;
XmDrawingAreaCallbackStruct *call_data;
{
	if (debug)
		printf("MyExposeCallback\n");
	if (call_data->event->xexpose.count != 0) 
		return;
	DrawMap();
}

extern void printresusage(), PlaybackDialog(), ChangeNameDialog();

XtActionsRec XmapActionsTable[]	= {
	{"SaveMap", SaveMap },
	{"Quit", aquit },
	{"ResourceCheck", printresusage },
	{"system", system1 },
	{"MoveIt", MoveIt },
	{"PlaceIt", PlaceIt},
	{"Refresh", ReDrawMap },
	{"Playback", PlaybackDialog },
	{"ChangeName", ChangeNameDialog },
};

/************************************************************************
 *  progtoClassName() - Convert this program name to a X ClassName	*
 ************************************************************************/
static char *progtoClassName(progname)
char *progname;
{
	static char ClassName[100];

	if (debug)
		printf("progtoClassName(%s)\n", progname);

	if (progname[0] == 'x') {
		ClassName[0] = 'X';
		if (strlen(progname) > 1) {
			ClassName[1] = toupper(progname[1]);
			strcpy(&ClassName[2], &progname[2]);
		} else 
			ClassName[2] = '\0';
	} else {
		ClassName[0] = toupper(progname[0]);
		if (strlen(progname) > 1)
			strcpy(&ClassName[1], &progname[1]);
		else 
			ClassName[2] = '\0';
	}
	if (debug)
		printf("progtoClassName() returns %s\n", ClassName);
	return(ClassName);
}

/************************************************************************
 *  Usage() - Describe the syntax of the command line parameters.	*
 ************************************************************************/
static void Usage()
{
	fprintf(stderr, "Usage: %s [-a] [-l #] [-c community ] [-i #] [-C #] [ -s StateFile ] [ -m MAPFILE ] [ X resources ]\n", progname);
	fprintf(stderr,"\
-a  Show Autonmous Systems           (value=%d)\n\
-l # Specifies line default thickness \n\
-b # Specifies bell volume (0-100) (value=%d)\n\
-i # Specifies interval to check Network Status File in secs (value=%d)\n\
-c community specifies community name to pass to invokes scripts \n\
-D  Turns on debugging output\n\
-C # Specifies Elapsed Time before calling Collector Dead in mins(value=%d)\n\
-v verbosity\n\
-s StateFile is the Network Status File\n\
-m MAPFILE contains the names and x,y coordinates for the MAP\n\n",
	    ShowASs,BellVolume,CheckStatusFileInterval, CollectorDiedInterval);
	exit(1);
}

static XtCallbackRec callbacks[] = {
	{(XtCallbackProc) MyExposeCallback, (caddr_t) NULL },
	{(XtCallbackProc) NULL, (caddr_t) NULL }
};

/*
 * The grim reaper collects the dead children
 */
void reaper()
{
        union wait status;

        while (wait3(&status,WNOHANG,0) > 0)
        {
                printf("Child returned\n");
        }
}


/************************************************************************
 *  main() - Main code -  Start Up X and all of main widgets		*
 ************************************************************************/
main(argc, argv)
int argc;
char *argv[];
{
	Widget toplevel, mainwindow, mainform, menubar, filepulldown, 
		filecascade, save, quit, quitnosave, mappulldown, mapcascade, 
		refresh, playback, aboutpulldown, aboutcascade, 
		linkwidthpulldown, linkwidthcascade;
	Arg myArgs[10];
	int i, c;
	XSetWindowAttributes winatt;
	char *p, *getenv();
	extern void PlaybackDialog(), CreateLinkWidthWidgets(),
		CheckStatusFile(), ReadMap(), CreateAboutWidgets(),
		SetLineWidth();
	extern char *optarg, Version[];
	extern int optind;

        /*
         * Set up for signals
         */

        signal( SIGCHLD, reaper );

	printf("%s\n", Version);
	if ((p = strrchr(argv[0], '/')) == NULL)
		strcpy(progname, argv[0]);
	else 
		strcpy(progname, p + 1);
	Class = progtoClassName(progname);
	printf("%s (Class=%s) starting\n", progname, Class);
	toplevel = XtInitialize(progname, Class, NULL, 0, &argc, argv);

	if ((pingkydir = getenv("PINGKYDIR")) == NULL) 
		pingkydir = DefaultPingkyDir;
	sprintf(cycletime, "%s/%s", pingkydir, NEWROVER_CYCLE);
	sprintf(logfile, "%s/%s", pingkydir, PROBLEM_LOG);
	sprintf(hostfile, "%s/%s", pingkydir, HOST_FILE);
	sprintf(Error_Log, "%s/%s", pingkydir, ROVER_LOG);
	sprintf(StateFile, "%s/%s.STATUS", pingkydir, progname);
	sprintf(MapFile, "%s/%s.MAP", pingkydir, progname);

	while ((c = getopt(argc, argv, "S:C:Dab:c:l:vs:m:")) != EOF)
		switch (c) {
		case 'S':
			SetMapShow(optarg);
			sprintf(MapFile, "%s/%s.MAP", pingkydir, optarg);
			break;
		case 'a':
			ShowASs = 1;
			break;
		case 'l':
			SetLineWidth(atoi(optarg));
			break;
		case 'b':
			BellVolume = atoi(optarg);
			BellActive = 1;
			break;
		case 'i':
			CheckStatusFileInterval = atoi(optarg);
			break;
		case 'c':
			Community = optarg;
			break;
		case 'D':
			debug = TRUE;
			break;
		case 'C':
			CollectorDiedInterval = atoi(optarg);
			break;
		case 'v':
			verbose = 1;
			break;
		case 's':
			strcpy(StateFile,optarg);
			break;
		case 'm':
			strcpy(MapFile,optarg);
			break;
		default:
			Usage();
			break;
		}

	XtAddActions( XmapActionsTable,	XtNumber(XmapActionsTable) );

	/*
	 *  Read Network Status File.
	 */
        /*if (debug)*/
                printf("Reading %s as the Network STATUS File\n", StateFile);
        ReadConfig(&Network, NetworkType, StateFile);
        Read_LinkDetail_File( progname );

        /*
         *  Read Map File.
         */

	/*if (debug)*/
		printf("Using %s as the map file\n", MapFile);
	ReadMap(MapFile);

	/*
	 *  Now create all of the widgets.
	 */
	i = 0;
	mainwindow = XmCreateMainWindow(toplevel, "mainwindow",	myArgs,	i);
	XtManageChild(mainwindow);
	menubar	= XmCreateMenuBar(mainwindow, "menubar", myArgs, 0);
	XtManageChild(menubar);

	aboutpulldown =	XmCreatePulldownMenu(menubar, "aboutpulldown", myArgs, 0);
	XtSetArg(myArgs[0], XmNsubMenuId, aboutpulldown);
	aboutcascade = XmCreateCascadeButton( menubar, "About...", myArgs, 1);
	XtManageChild( aboutcascade );
	CreateAboutWidgets(pingkydir, aboutpulldown);

	filepulldown = XmCreatePulldownMenu(menubar, "filepulldown", myArgs, 0);
	XtSetArg( myArgs[0], XmNsubMenuId, filepulldown);
	filecascade = XmCreateCascadeButton( menubar, "File", myArgs, 1);
	XtManageChild( filecascade );

	save = XmCreatePushButtonGadget(filepulldown, "SaveMap", myArgs, 0);
	XtManageChild(save);
	XtAddCallback(save, XmNactivateCallback, SaveMap, (caddr_t) NULL);

	quit = XmCreatePushButtonGadget(filepulldown, "Quit", myArgs, 0);
	XtManageChild(quit);
	XtAddCallback(quit, XmNactivateCallback, Quit,	(caddr_t) "Save");

	quitnosave = XmCreatePushButtonGadget(filepulldown, "Quit NO SAVE", 
								myArgs, 0);
	XtManageChild( quitnosave );
	XtAddCallback(quitnosave, XmNactivateCallback, Quit, (caddr_t) NULL);

	mappulldown= XmCreatePulldownMenu(menubar, "mappulldown", myArgs, 0);
	XtSetArg( myArgs[0], XmNsubMenuId, mappulldown );
	mapcascade = XmCreateCascadeButton( menubar, "Map", myArgs, 1);
	XtManageChild( mapcascade );
	refresh	= XmCreatePushButtonGadget(mappulldown, "Refresh", myArgs, 0);
	XtManageChild( refresh );
	XtAddCallback(refresh, XmNactivateCallback, ReDrawMap, (caddr_t) NULL);


	linkwidthpulldown= XmCreatePulldownMenu(menubar, "linkwidthpulldown", 
								myArgs, 0);
	XtSetArg( myArgs[0], XmNsubMenuId, linkwidthpulldown );
	linkwidthcascade = XmCreateCascadeButton( menubar, "LinkWidth", 
								myArgs, 1);
	XtManageChild( linkwidthcascade );
	CreateLinkWidthWidgets( pingkydir, linkwidthpulldown );

#define PLAYBACK  /*experimental*/
#ifdef PLAYBACK
	playback = XmCreatePushButtonGadget(mappulldown, "playback", myArgs, 0);
	XtManageChild( playback );
	XtAddCallback(playback , XmNactivateCallback, PlaybackDialog,
							(caddr_t) NULL);
#endif

	mainform = XmCreateForm(mainwindow, "mainform",	myArgs,	0);
	XtManageChild(mainform);

	XtSetArg( myArgs[0], XmNexposeCallback, callbacks );
	drawarea = XmCreateDrawingArea(mainform, "draw", myArgs, 1 );

	for (i = 1; i < NumMapNodes; i++)
		AddNodePB(i);
	XtManageChild(drawarea);

	BuildMyAppResources( toplevel );	/* Build Menu sub system */
	if (NumMapNodes > 1)
		CreateMenus( NodePB[1] );
	else
		printf("No nodes defined:  cannot create node/link menus\n");

	XtRealizeWidget(toplevel);

	/*
	 * Enable backing store so the window is automatically restored by the
	 * server.
	 */
	winatt.backing_store = WhenMapped;
	XChangeWindowAttributes(XtDisplay(toplevel), XtWindow(drawarea), 
					CWBackingStore, &winatt);
	CheckStatusFile((XtIntervalId) NULL);
	XtMainLoop();
	/*NOTREACHED*/
}
