/*
 * Copyright (c) 1989, 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 */

#include <curses.h>
#include <time.h>
#include <string.h>
#include "CommonDefs.h"
#include "problem.h"
#include "pathnames.h"
#include "ctools.h"
#include "display.h"

struct Filters Filter[MAXFILTERS];
int FutureAlarmsExist=0;	/* Boolean - are there alarms in the future */
int FilterFuture=1;		/* Should we filter them from sight? */


/**********************************************************
 *  Message: Put a message to the user on the bottom line * 
 **********************************************************/
void Message(msg)
char msg[];
{
	standout();
	mvaddstr(LINES-1,0,msg);
	standend();
	clrtoeol();
	refresh();
}

/********************************************************
 * PrintTime:  Print the current time on the screen.	*
 ********************************************************/
Print_Time()		/* Print the current time on the screen */
{
	time_t TimeNow;
	char buffer[27];
	struct tm *tm;
	extern char *tz;

	time( &TimeNow );
	strcpy(buffer,ctime( &TimeNow ));
	buffer[16]='\0';
	strcat(buffer," ");
	strcat(buffer,tz);
	buffer[20]='\0';
	mvaddstr(0,59,buffer);
	tm=gmtime( &TimeNow );
	strcpy(buffer,asctime(tm));
	buffer[16]='\0';
	strcat(buffer," GMT");
	mvaddstr(1,59,buffer);
}

/********************************************************
 * DrawFrame: Draw a frame in which problems get logged	*
 ********************************************************/
DrawFrame()
{
	char TopLine[80],buffer[80];
	int i;
	extern int TimeStampFormat;
	extern char DisplayType[];
	extern char *title;

	clear();
	sprintf(TopLine,"%s          %s", DisplayType, title);
	mvaddstr(0,0,TopLine);
	strcpy(buffer, "Filters: ");
	for(i = 0; i < MAXFILTERS && Filter[i].Name != NULL; i++)
		if (Filter[i].Selected == FALSE) {
			strcat(buffer,Filter[i].Name); 
			strcat(buffer," ");
		}
	if ( strlen(buffer)>9) 
		mvaddstr(2,10,buffer);
	switch( TimeStampFormat ) {
	case HUMANTIME:
		mvaddstr(3,0,"# Reported                Problem                          StatusLine");
		break;
	case LONGTIME:
		mvaddstr(3,0,"# longtime		    Problem                          StatusLine");
		break;
	case ELAPSEDSEC:
		mvaddstr(3,0,"# ElapsedSecs		    Problem                          StatusLine");
		break;
	case ELAPSEDMIN:
		mvaddstr(3,0,"# Elapsed Time            Problem                          StatusLine");
		break;
	default:
		break;
	}

	mvaddstr(4,0,"  ========= --------------------------------------  --------------------------");
}

/*****************************************************************
 * ReadProblems:  Read Problems and Display on the screen nicely *
 *****************************************************************/
ReadProblems()
{
int rc;

	rc=Problem_Manager( CHECK, (char *) NULL, (char *) NULL, (char *) NULL, (char *) NULL ); /* Reread prob list */
	DrawProblems();
	return( rc );
}

/****************************************************************
 * DrawProblems:  Draw Problems on the screen in a nice format	*
 ****************************************************************/
DrawProblems()
{
	int i = 0, y = 5;
	char buffer[100];
	/* In Coreversion of existing problems */
	extern struct ProblemType ProblemArray[];
	extern int NumProblems, CurrentLine;
	extern char *PrintTimeStamp();

	FutureAlarmsExist=0;
	for( i=0; i<MAXFILTERS; i++ ) Filter[i].count=0;
	sprintf(buffer,"[%3d]",NumProblems);
	mvaddstr(2,74,buffer);
	if ( CurrentLine > NumProblems ) CurrentLine=0;
	if (NumProblems==0) {
		move(y,0);
		clrtoeol();
		mvaddstr(y++,30,"**** No Problems ****");
		CurrentLine=0;
	}
	else {
		for( i=0,CountProblem((struct ProblemType *) NULL); i<NumProblems; i++)
			CountProblem( &ProblemArray[i] );
		if ( CurrentLine <= 0 ) {	/* Not scrolling now */
			for(i=0; (i<NumProblems) && (y<LINES-2); i++,y++) {
				if ( FilterProblem( &ProblemArray[i] ) == 1 ) {
					sprintf(buffer,"%2.2d",i+1);
					mvaddstr(y,0,buffer);
					sprintf(buffer," %9.9s %s %s %s",
					    PrintTimeStamp( ProblemArray[i].TimeStamp ),
					    ProblemArray[i].Name,ProblemArray[i].UniqueID,
					    ProblemArray[i].TestName);
					mvaddstr(y,3,buffer);
					clrtoeol();
					if ( ProblemArray[i].StatusLine[0] != '\0' )
						mvaddstr(y,STATUSX,ProblemArray[i].StatusLine);
				}
				else y--;	/* ProblemFiltered*/
			}
		}
		else {
			for(i=CurrentLine; i<NumProblems && y<LINES-2; i++,y++) {
				if ( FilterProblem( &ProblemArray[i] ) == 1 ) {
					/*if ( ProblemArray[i].StatusLine[0] == '\0' ) standout();*/
					sprintf(buffer,"%2.2d",i+1);
					mvaddstr(y,0,buffer);
					/*if ( ProblemArray[i].StatusLine[0] == '\0' ) standend();*/
					sprintf(buffer," %9.9s %s %s %s",
					    PrintTimeStamp( ProblemArray[i].TimeStamp ),
					    ProblemArray[i].Name,ProblemArray[i].UniqueID,
					    ProblemArray[i].TestName);
					mvaddstr(y,3,buffer);
					clrtoeol();
					mvaddstr(y,STATUSX,ProblemArray[i].StatusLine);
				} else y--;
			}
		}
	}
	if (i<NumProblems) {
		mvaddstr(2,55,"[ MORE ... ]");
	}
	else {
		mvaddstr(2,55,"            ");
		for(; y<LINES-2; y++) {
			move( y, 0 );
			clrtoeol();
		}
	}
	if ( FilterFuture && FutureAlarmsExist ) {
		mvaddstr(1,0,"<Future Alarms Exist>");
	} else {
		mvaddstr(1,0,"                     ");
	}
}

/****************************************************************
 * GotoCmdLine:  Got the Command Prompt 			*
 ****************************************************************/
GotoCmdLine()
{
	move(LINES-2,0); 
	clrtoeol();
	standout();
	addstr("Command:");
	standend();
	move(LINES-2,9);
}

/****************************************************************
 * ClearMessage:  Clear the message line			*
 ****************************************************************/
void ClearMessage()
{
	move(LINES-1,0);
	clrtoeol();
	refresh();
}

SetUpdating( claim )
int claim;
{
	extern int Updating;

	if ( claim )
		mvaddstr(2,40,"*Claiming*");
	else
		mvaddstr(2,40,"*Updating*");
	refresh();
	Updating=1;
}

ClearUpdating()
{
	extern int Updating;
	mvaddstr(2,40,"          "); 
	refresh();
	Updating=0;
	ClearMessage();
}


/****************************************************************
 *		Update - Update the status of a problem		*
 ****************************************************************/
void Update( claim )
int claim;
{
	char ProblemList[100];	/* array of nodes to be updated at once */
	int i;
	char buffer[200];
	extern int NumProblems;
	extern struct ProblemType ProblemArray[];
	char TestName[100], Name[100], UniqueID[100];
	extern char Updateable;	/* Don't Allow updating by default */
	extern int Updating;

	for(i=0; i<100; i++) ProblemList[i]=0;
	if (NumProblems==0) {
		Message("  There are no problems to update, Bonehead!");
		sleep(5);
		ClearMessage();
		return;
	}
	if (Updateable) {
		SetUpdating( claim );
		if ( claim ) {
			Message("   Enter the problem # that you want to Claim");
			mvaddstr(LINES-2,10,"laim Problem # ");
		}
		else {
			Message("   Enter the problem # that you want to Update   ");
			mvaddstr(LINES-2,10,"pdate Problem # ");
		}
		refresh();
		if ((i=GetUpdateList())==0) {
			ClearMessage();
			ClearUpdating();
			Updating=0;
			return;
		}
		if (i>0 && i<=NumProblems) {
			strcpy(Name,ProblemArray[i-1].Name);
			strcpy(TestName,ProblemArray[i-1].TestName);
			strcpy(UniqueID,ProblemArray[i-1].UniqueID);
			if ( claim )
				Message("  Enter your initials and brief note (if any)     ");
			else
				Message("  Enter the StatusLine as you want it to appear   ");
			move(4+i,STATUSX);
			clrtoeol();
			refresh();
			buffer[0]='\0';
			getstr(buffer);
			buffer[MAXSTATUSLINE-1]='\0';
			if ( claim )
				Problem_Manager( CLAIM_PROBLEM, TestName, Name, UniqueID, buffer );
			else
				Problem_Manager( UPDATE_PROBLEM, TestName, Name, UniqueID, buffer );
		}
		else {
			Message("   Illegal Problem Number   ");
			sleep(5);
			ClearMessage();
		}
		ClearUpdating();
	}
}

/****************************************************************
 *		Advance Problem Clock				*
 ****************************************************************/
void AdvanceClock( )
{
	int i;
	char buffer[200];
	extern int NumProblems;
	extern struct ProblemType ProblemArray[];
	char TestName[100], Name[100], UniqueID[100];
	extern char Updateable;	/* Don't Allow updating by default */
	extern int Updating;
	time_t TimeNow;
	char futuretime[100];

	if (NumProblems==0) {
		Message("  There are no problems to update, Bonehead!");
		sleep(5);
		ClearMessage();
		return;
	}
	if (Updateable) {
		SetUpdating( 0 );
		Message("   Enter the problem # that you want to futurize   ");
		mvaddstr(LINES-2,10,"Advance clock for Problem # ");
		refresh();
		if ((i=GetUpdateList())==0) {
			ClearMessage();
			ClearUpdating();
			Updating=0;
			return;
		}
		if (i>0 && i<=NumProblems) {
			strcpy(Name,ProblemArray[i-1].Name);
			strcpy(TestName,ProblemArray[i-1].TestName);
			strcpy(UniqueID,ProblemArray[i-1].UniqueID);
				Message("  Enter the number of minutes to advance the alarm timestamp ");
			move(4+i,STATUSX);
			clrtoeol();
			refresh();
			buffer[0]='\0';
			getstr(buffer);
			i=atoi(buffer);
			if ( i > 0 ) {
				time(&TimeNow);
				sprintf(futuretime,"%ul",TimeNow+(atoi(buffer)*60));
				Problem_Manager( FUTURIZE_PROBLEM, TestName, Name, UniqueID, futuretime);
			}
		}
		else {
			Message("   Illegal Problem Number   ");
			sleep(5);
			ClearMessage();
		}
		ClearUpdating();
	}
}

