/*
 * Copyright (c) 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 * All I/O with PROBLEM FILE uses these routines
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "CommonDefs.h"
#include "problem.h"
#include "ctools.h"

#define NO_PROBLEM_FOUND	-1
#define MAX_LOCK_ATTEMPTS	3

#ifdef S_IRWXU
#define S_RWALL (S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH)
#else
#define S_RWALL 0000666
#endif

int In_Critical_Section = 0;

static char buffer[BUFSIZ];	/* buffer for a variety of purposes */

void PrintProblems(n,Problem)
int n;
struct ProblemType *Problem;
{
	register int i;

	for(i = 0; i < n; i++) {
		if (Problem[i].Name[0] != '\0')
			printf("TimeStamp=%s Day: %s Time:%s Node: %s [%s] Failed %s Test StatusLine: %s \n",
				ltoa(Problem[i].TimeStamp, buffer),
				_Day(Problem[i].TimeStamp), 
				_Time(Problem[i].TimeStamp),Problem[i].Name,
			    	Problem[i].UniqueID,
				Problem[i].TestName,
				Problem[i].StatusLine);
	}
}

/***************************************************************************
 * Problem_Exists:  Returns index of Problem if it exists and -1 otherwise *
 ***************************************************************************/
int Problem_Exists(UniqueID,Service,Problem,n)
char *UniqueID;
char *Service;
struct ProblemType *Problem;
int n;
{
	register int i;

	for (i = 0; i < n; i++) {
		if (strcmp(Problem[i].UniqueID,UniqueID) == 0)
			if (strcmp(Problem[i].TestName,Service) == 0)
				return(i);
	}
	return(NO_PROBLEM_FOUND);
}


/*****************************************************************************
 * ReadProblemFile:  Read the Problems from file into our internal format.   *
 *****************************************************************************/
int ReadProblemFile(Problem,file) /* Read and Store hostfile - return # read */
struct ProblemType *Problem;      /* Network Entities Structure Array        */
char *file;                       /* Filename to read these from             */
{
	register int i;
	FILE *fp;
	char *p;
	extern long atol();

	/* mark this as a critical section */
	In_Critical_Section = 1;

	/*
	 *  Try to lock the Problem File.  If we cannot do it after
	 *  our maximum number of allowable attempts, bail out.
	 */
	for (i = MAX_LOCK_ATTEMPTS; i > 0; i--)
		if (lock(file))
			break;
	if (i == 0)
		panic("Read_ProblemFile: PROBLEM.FILE LOCKED");

	/*
	 *  It's locked.  Now, open it and read the problems.
	 */
	if ((fp = fopen(file,"r")) == NULL) {
		fprintf(stderr,"fopen reading PROBLEM FILE\n");
		Empty_File( file ); /* create an empty file with right perms */
		if ((fp = fopen(file,"r")) == NULL)
			syserr("fopen readingPROBLEM FILE");
	}
	for (i = 0; fgets(buffer, sizeof(buffer), fp) != NULL; ) {

		/* comment? */
		if (buffer[0] == '#')
			continue; 

		/* pull off timestamp */
		if ((p = strtok(buffer, DELIMITERS)) == NULL) 
			continue;
		Problem[i].TimeStamp = (time_t) atol(p);

		/* pull off node name */
		if ((p = strtok((char *) NULL, DELIMITERS)) == NULL) 
			continue;
		strncpy(Problem[i].Name, p, MAXNODENAME); 
		Problem[i].Name[MAXNODENAME - 1] = '\0';

		/* pull off unique id */
		if ((p = strtok((char *) NULL, DELIMITERS)) == NULL) 
			continue;
		strncpy(Problem[i].UniqueID, p, MAXUNIQUEID);
		Problem[i].UniqueID[MAXUNIQUEID - 1] = '\0';

		/* pull off test name */
		if ((p = strtok((char *) NULL, DELIMITERS)) == NULL)
			continue;
		strncpy(Problem[i].TestName, p, MAXTESTNAME);
		Problem[i].TestName[MAXTESTNAME - 1] = '\0';

		/* pull off NetMgmtProtocols */
		Problem[i].StatusLine[0] = '\0';
		while((p = strtok((char *) NULL, DELIMITERS)) != NULL)  {
			strncat(Problem[i].StatusLine, p, MAXSTATUSLINE - strlen(Problem[i].StatusLine));
			strncat(Problem[i].StatusLine, " ", MAXSTATUSLINE - strlen(Problem[i].StatusLine));
		}
		Problem[i].StatusLine[MAXSTATUSLINE - 1] = '\0';

		if (++i > MAXPROBLEMS)	/* we've got all that we can handle */
			break;
	}
	fclose(fp);
	unlock(file);
	Problem[i].Name[0] = '\0';
	In_Critical_Section = 0;
	return(i);
}

static int CompareRoutine(p1, p2)
struct ProblemType *p1,*p2;
{
	/*
	 *  Should return '1' if 1 is greater than 2.
	 *  Should return '-1' if 1 is less than 2.
	 */
	if (( p1->StatusLine[0] == '\0' ) && ( p2->StatusLine[0] != '\0' ))
		return(-1);
	if (( p1->StatusLine[0] != '\0' ) && ( p2->StatusLine[0] == '\0' ))
		return(1);

	/* BOTH ARE COMMENTED OR NOT COMMENTED - Sort by Time */
	return( p2->TimeStamp - p1->TimeStamp );
}

/*****************************************************************************
 * WriteProblemFile:  Write our problems to disk.                            *
 *****************************************************************************/
WriteProblemFile(Problem, Problem_File, n)
struct ProblemType *Problem;
char *Problem_File;
int n;
{
	FILE *fp;
	register int i;

	/* mark this as a critical section */
	In_Critical_Section = 1;

	/*
	 *  Try to lock the Problem File.  If we cannot do it after
	 *  our maximum number of allowable attempts, bail out.
	 */
	for (i = MAX_LOCK_ATTEMPTS; i > 0; i--)
		if (lock(Problem_File))
			break;
	if (i == 0)
		panic("WriteProblemFile: PROBLEM.FILE LOCKED");

	/*
	 *  Open the Problem File, sort the problems, and then write them
	 *  to disk.
	 */
	if ((fp = fopen(Problem_File,"w")) == NULL)
		syserr("fopen PROBLEM FILE");
	qsort((char *) Problem, n, sizeof(struct ProblemType), CompareRoutine);
	for (i = 0; i < n; i++) {
		if (Problem[i].Name[0] != '\0') {
			fprintf(fp,"%s %s %s %s %s\n",
				ltoa(Problem[i].TimeStamp, buffer),
				Problem[i].Name,
				Problem[i].UniqueID,
				Problem[i].TestName,
				Problem[i].StatusLine);
		}
	}
	fclose(fp);
	In_Critical_Section = 0;
	unlock(Problem_File);
	chmod(Problem_File, S_RWALL);
}
