/*
 * Copyright (c) 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 *	dsuchk2.c  - This code deals with identifying circuits
 *		Call ReadConfigFiles() to read in Link Data files
 */
 
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/param.h>
#include <dirent.h>
#include <time.h>
#include "CommonDefs.h"
#include "Linkfileio.h"
#include "mib.h"

/************************************************************************
 * ClearLastSampleFiles() - remove last collected sample files.		*
 ************************************************************************/
void ClearLastSampleFiles()
{
	DIR *dirp;
	struct dirent *dp;
	register char *s;
	char f[MAXPATHLEN];

	/*
	 *  Remove all files in /tmp that have a ".lastsample" suffix.
	 */
	if ((dirp = opendir("/tmp")) == NULL) {
		perror("opendir");
		return;
	}
	for (dp = readdir(dirp); dp != NULL; dp = readdir(dirp)) {
		if ((s = strrchr(dp->d_name, '.')) == NULL)
			continue;
		s++;
		if (strcmp(s, "lastsample"))
			continue;
		sprintf(f, "/tmp/%s", dp->d_name);
		if (unlink(f) < 0)
			perror("unlink");
	}
	closedir(dirp);
}

/************************************************************************
 * WriteDataLine() - Append a data line to the Last Sample Output File	* 
 *		This file contains LinkDetail DSU Counter information 	*
 *		retrieved from DSUs on both ends of the link.		*
 *									*
 *		These files can then be used as supporting evidence	*
 *		when link problems are reported.			*
 ************************************************************************/
void WriteDataLine(Node, dsunum, dataline)
char *Node;
int dsunum;
char *dataline;
{
	extern struct LinkDetail * FindLinkDetail();
	char OutputFileName[MAXPATHLEN];
	FILE *stream;
	struct LinkDetail *LPtr;
	struct stat outstat;
	extern time_t time();
	time_t TimeNow;
	char *ip_to_pop(), *pops_to_circuit();

	TimeNow = time((time_t *) NULL);
	if ((LPtr = FindLinkDetail( Node, dsunum )) == NULL) {
		fprintf( stderr, "WriteDataLine - can't write %s dsu %d data - no link available\n", Node, dsunum );
		return;
	}
	
	if ( strcmp( LPtr->LocalNode, LPtr->RemoteNode ) == 0 ) 
		return;

	/**** Create or append to last sample Output file ****/
	sprintf( OutputFileName, "/tmp/%s%s-%s%s.lastsample", LPtr->LocalPOP, 
			LPtr->LocalNode, LPtr->RemoteNode, LPtr->RemotePOP );

	if ( stat( OutputFileName, &outstat ) != 0 ) {    /* New File */
        	if (( stream=fopen( OutputFileName, "w" )) == NULL ) {
                	fprintf( stderr, "Can't open lastsample output file %s<\n", OutputFileName);
			perror("fopen(): ");
                	return;
		}
		fprintf( stream, "%s\n%s-%s-%s-%d-%s-%d-%s-%s-%s\n",
				ctime( &TimeNow ), ip_to_pop(Node), LPtr->LocalNode, LPtr->LocalIPAddr, LPtr->LocalDSUNumber, LPtr->Circuit,
				LPtr->RemoteDSUNumber ,LPtr->RemoteIPAddr,LPtr->RemoteNode, LPtr->RemotePOP );
        } else {
        	if (( stream=fopen( OutputFileName, "a" )) == NULL ) {
                	fprintf( stderr, "Can't open lastsample output file %s<\n", OutputFileName);
			perror("fopen(): ");
                	return;
        	}
	}
	fprintf( stream, "%s %s\n",ip_to_pop(Node),dataline ); 
        fclose( stream );
        return;
}

/************************************************************************
 * AppendData() - Write stream to stream2.  Used for appending support  *
 *		info to the Circuit TT.					*
 ************************************************************************/
static void AppendData(stream, stream2)
FILE *stream, *stream2;
{
	char buffer[BUFSIZ];

	while( fgets( buffer, sizeof(buffer), stream ) != NULL )
		fputs(  buffer , stream2 );
}

/************************************************************************
 * DealWithDirtyLinks() - Process each LinkDetail marked as dirty.		*
 *			Append all counters from both sides of the link	*
 *			to the DIRTY file.  If this file doesn't exist  *
 *			create it and mail it to the ds3action mailing  * 
 *			list.						*
 ************************************************************************/
void DealWithDirtyLinks(pingkydir)
char *pingkydir;
{
	register struct LinkDetail *lp,*LPtr;
	char DirtyFileName[MAXPATHLEN], SampleFileName[MAXPATHLEN], 
		cmd[BUFSIZ];
	FILE *stream, *stream2;
	long TimeNow=time(&TimeNow);
	union wait status;
	char lbuff[40], rbuff[40];
	extern char *datestr();

       /*********************************************************
 	*  For each LinkDetail, 				*
	*	Assure both DSUs responded,			*
	*	if DIRTY Link		 			*
	*		Append TimeStamp and data to DIRTY FILE	*
	*		if ( TT ! OPEN )			*
	*			Open Ticket			*
	* 			Set Next Action			*
	*	if ( NextAction Time )				*
	*		Handle Update				*
	*		Set Next Action				*
	*  		if (24 hours has elapsed )		*
	*			if ( isclean )			*
	*				Send CLOSE Ticket	*
	*				Set Next Action to 0	*
	*							*
        *********************************************************/
	for (lp = LinkDetailHead; lp != NULL; lp = lp->Next) {
		/*
		 *  Did either side NOT answer?
		 */
		if (lp->LinkType == MIB_IFTYPE_PROPPOINTTOPOINTSERIAL) {
			if (TimeNow - lp->LocalTimeStamp >= 120)
				fprintf(stderr,
				  "Node:DSU pair didn't report!:  %s : %d ---- %d %s\n",
				  lp->LocalNode, lp->LocalDSUNumber, lp->RemoteDSUNumber, lp->RemoteNode);
			if (TimeNow - lp->RemoteTimeStamp >= 120) 
				fprintf(stderr,
				  "Node:DSU pair didn't report!:  %s: %d --- %d %s\n",
				  lp->RemoteNode,lp->RemoteDSUNumber, lp->LocalDSUNumber, lp->LocalNode);
		}
		if (lp->LocalDSUSerialNumberVerified && lp->RemoteDSUSerialNumberVerified )
			printf("%s %d %d %s VERIFIED\n",lp->LocalNode,lp->LocalDSUNumber,lp->RemoteDSUNumber, lp->RemoteNode);

		sprintf(DirtyFileName, "%s/%s%s-%s%s.DIRTY.%s", 
			pingkydir, lp->LocalPOP, lp->LocalNode, lp->RemoteNode, 
			lp->RemotePOP, datestr(0));
		sprintf(SampleFileName, "/tmp/%s%s-%s%s.lastsample", 
			lp->LocalPOP, lp->LocalNode, lp->RemoteNode, 
			lp->RemotePOP );


		/*
		 *  If ( DIRTY LinkDetail )
		 */
		if (lp->Dirty  || lp->TTNextAction) {

        		if ((stream = fopen( SampleFileName, "r" )) == NULL ) {
                		fprintf( stderr, "DealWithDirtyLink(): Can't open Sample output file: %s<\n",SampleFileName);
				continue;
			}
        		if ((stream2 = fopen( DirtyFileName, "a" )) == NULL ) {
                		fprintf( stderr, "DealWithDirtyLink(): Can't open DirtyFile output file %s\n", DirtyFileName);
				continue;
			}
			AppendData( stream, stream2 );
			fclose( stream );
			fclose( stream2 );

			/****************************/
			/*****  if ( TT ! OPEN ) ****/
			/****************************/

			if ( lp->TTNextAction == 0 ) {	/* NEW TT */
				fprintf(stderr,
				  "Opening Trouble Ticket File %s\n",
				  DirtyFileName);
				lp->TTLastOpened=TimeNow;

				switch (fork()) {

				case -1 :
					perror("fork");
					break;

				case 0 :
					/* child */
					execlp("Notify", "Notify", "OPEN",
						DirtyFileName, 
						lp->LocalPOP,
						lp->LocalNode, 
						itoa(lp->LocalDSUNumber,lbuff),
						lp->Circuit,
						itoa(lp->RemoteDSUNumber,rbuff),
						lp->RemoteNode,
						lp->RemotePOP,
						NULL);
					perror("execlp");
					break;

				default :
					/* parent */
					wait(&(status.w_status));
					break;
				}

				lp->TTNextAction=TimeNow+4*60*60;
			}
		}

		/**********************************/
		/*****  if ( NextAction Time ) ****/
		/**********************************/
		if ((lp->TTNextAction == 0) || (TimeNow <= lp->TTNextAction)) {
			continue;
		}

		fprintf(stderr, "Updating Trouble Ticket File %s\n",
								DirtyFileName);
		switch (fork()) {

		case -1 :
			perror("fork");
			break;

		case 0 :
			/* child */
			execlp("Notify", "Notify", "UPDATE",
				DirtyFileName, lp->LocalPOP, lp->LocalNode, 
				itoa(lp->LocalDSUNumber,lbuff), lp->Circuit,
				itoa(lp->RemoteDSUNumber,rbuff), lp->RemoteNode,
				lp->RemotePOP, NULL);
			perror("execlp");
			break;

		default :
			/* parent */
			wait(&(status.w_status));
			break;
		}

		lp->TTNextAction = TimeNow + 4 * 60 * 60;

		/*
		 *  If 24 hours has not elapsed, we're done here.
		 *  If 24 hours has elapsed, try to clean up.
		 */
		if ( TimeNow - lp->TTLastOpened <= (24 * 60 * 60)) {
			continue;
		}

		switch (fork()) {

		case -1 :
			perror("fork");
			break;

		case 0 :
			/* child */
			execlp("isclean", "isclean", lp->LocalNode, 
				itoa(lp->LocalDSUNumber,lbuff), lp->RemoteNode,
				itoa(lp->RemoteDSUNumber,rbuff), NULL);
			perror("execlp");
			break;

		default :
			/* parent */
			wait(&(status.w_status));
			break;
		}

		if (status.w_retcode != 0) {
			fprintf(stderr,
			  "Can't close ticket - isclean said it wasn't\n");
			continue;
		}

		/*
		 *  The clean-up went OK; try to close the TT.
		 */
		switch (fork()) {

		case -1 :
			perror("fork");
			break;

		case 0 :
			/* child */
			execlp("Notify", "Notify", "CLOSE",
				DirtyFileName, lp->LocalPOP, lp->LocalNode, 
				itoa(lp->LocalDSUNumber, lbuff), lp->Circuit,
				itoa(lp->RemoteDSUNumber, rbuff), lp->RemoteNode,
				lp->RemotePOP, NULL);
			perror("execlp");
			break;
	
		default :
			/* parent */
			wait(&(status.w_status));
			break;
		}
		lp->TTNextAction = 0;
	}
}
	
#ifdef STANDALONE

main()
{

	Read_POP_TO_CIRCUIT_MAP_File( "POP_TO_CIRCUIT" );
	Read_IP_TO_POP_MAP( "IP_TO_POP" );
	Read_LinkDetail_File(  );
	PrintLinkDetail();
}
#endif
