
/************************************************************************
 *	Bill Norton	Enhanced SNMP API				*
 *									*
 *	High-Level Calls:						*
 *		Get() - Provides a SNMP_GET for any number of hosts	*
 *		GetNext() - SNMP GET NEXT for any number of hosts	*
 *		GetTable() - SNMP GET NEXT Until End of Table		*
 *									*
 *	Each of these is called with the following parameters:		*
 *		AddressListHead - List of AddressList structures	*
 *		Community - Community Name for all queries		*
 *			    ( If NULL, it is assumed that caller 	*
 *			     already set up the Community field in the  *
 *			     AddressList Structures.		     )  *
 *		QueryList - List of Variables to Query for.		*
 *			    ( If NULL, it is assumed that the caller	*
 *			     already set up the VarLists for each 	*
 *			     AddressList Structures.		     )  *
 *									*
 *	If a mixture of Gets, GetNexts, and GetTables is desired, you	*
 * will have to "roll your own".  This means setting up the VarLists	*
 * and AddressList structures to accomodate your unique mix of queries. *
 *									*
 *	Comvenience Routines:						*
 *		To create AddressList Structures:			*
 ************************************************************************/
#include <string.h>
#include <time.h>
#include <sys/time.h>
#ifdef _AIX
#include <sys/select.h>		
#endif

#include "snmptable2.h"		/* Defines Application-FastSNMPAPI Interface */
#include "snmpapi.h"		/* Defines FastSNMPAPI-CMU Interface routines*/

#include "ctools.h"

int MaxSecsB4Retry=5;
int MaxSNMPRetries=5;

int DebugOnFailure=0;

char *progname;
static int TimedOut=0;
static int Retries=0;

struct bindings *save_SNMP_var_Head,*save_SNMP_var_Tail;	/* mine */

/*extern char *SNMP_errormsg[];  /* SNMP Error messages -replaced by routine @wbn */

int verbose=0;

struct AddressListType *SendSNMPRequest_Address_List_Head=NULL;

/************************************************************************
 * SendSNMPRequest() - Send SNMP queries to hosts in AddressList struct *
 *                      provided the sockets aren't BUSY                *
 ************************************************************************/
void SendSNMPRequest(A)
     struct AddressListType *A;
{
struct VarList *v;
struct bindings *btail;
int num_reqs;         /* Number of instance IDs in request packet */
char rpacket[2048];   /* buffer for ASN-encoded SNMP request 	*/
char req[100][50];     /* Array of up to 100 requests 		*/
char *preq[100];       /* Array of up to 100 ptrs to instance IDs */
int len;
int nfds;

	num_reqs=0;
	for( v=A->QueryListHead; v!=NULL; v=v->Next ) {
		if ( A->Busy == 1 ) {
			fprintf(stderr,"SendSNMPRequest() Interrupted mid-var. Shouldn't happen often\n");
			continue;
		}
		if ( v->Head == NULL ) {	
			/*if (A->SNMP_Cmd==SNMPTABLE) {  /* First instance=0 */

			/** Append a .0 if we aren't asking for a particular 
				instance				**/
			if ( v->Prefix[strlen(v->Prefix)-1]=='.' ) {
/*#define APPEND_0_TO_TABLE_QUERY*/
#ifdef APPEND_0_TO_TABLE_QUERY
				/* Append a 0 to end of IDENTIFIER */
				strcpy( req[num_reqs], v->Prefix ); 
				strcat(req[num_reqs],"0");
#else
				/* kill appended . */
				strcpy( req[num_reqs], v->Prefix ); 
				req[num_reqs][strlen(v->Prefix)-1]='\0';
#endif
			}
			else {
				strcpy( req[num_reqs], v->Prefix ); 
				/* 1st query and user has specified OID */
			}
		}
		else {
			for(btail=v->Head; btail->Next!=NULL; btail=btail->Next);
			strcpy( req[num_reqs], btail->instance); 
		}
		preq[num_reqs]=req[num_reqs];
		num_reqs++;
	}
        len = make_SNMP_request( (A->SNMP_Cmd==SNMPGET)?1:2, A->Community, 
				num_reqs, preq, 0L, rpacket, sizeof(rpacket));
        if ( len < 0 ) {
                fprintf(stderr,"make_SNMP_request() failed (len=%d)\n",len);
        }
	if ( verbose ) fprintf(stderr,"SendSNMPRequest(): Sending %d bytes to %s\n",len,A->Address);
	if (sendto( A->SNMP_socket, rpacket, len, 0, &A->snmp_dest, 
							sizeof(A->snmp_dest))==-1)
		perror("sendto(): ");
	else;
}	



/************************************************************************
 * BlastSNMPPackets() - For each Address in list, SendSNMPRequest()     *
 ************************************************************************/
void BlastSNMPPackets()
{
struct AddressListType *A;
long TimeNow=time(&TimeNow);
int retranscount=0;
static struct itimerval value,ovalue;


	if ( Retries >= MaxSNMPRetries ){
		TimedOut=1;
		return;
	}
	for( A=SendSNMPRequest_Address_List_Head; A!=NULL; A=A->Next ) {
		if (( A->QueryListHead != NULL ) &&
			( TimeNow - A->LastResponded >= MaxSecsB4Retry )) { 
			if ( Retries )
				if (verbose) fprintf(stderr,"Time=%ld Retransmitting SNMP Packet to %s on socket %d\n",
					A->LastResponded,A->Address,A->SNMP_socket);
			else 
				if (verbose) fprintf(stderr,"Transmitting SNMP Packet to %s on socket %d\n",A->Address,A->SNMP_socket);
			
			SendSNMPRequest( A );
			retranscount++;
		}
		else if (verbose) fprintf(stderr,"%s on socket %d responded of is DONE\n",A->Address,A->SNMP_socket);
	}
	Retries++;
	if (verbose) fprintf(stderr,"****** %d packets transmitted ******\n",retranscount);
	value.it_interval.tv_sec = 0;
        value.it_interval.tv_usec = 0;
        value.it_value.tv_sec = MaxSecsB4Retry;
        value.it_value.tv_usec = 0;
	signal( SIGALRM, BlastSNMPPackets );
        setitimer(ITIMER_REAL, &value, &ovalue);
        /* signal(SIGALRM, FastPing); */
	/* alarm( MaxSecsB4Retry ); */
}

/************************************************************************
 * ProcessAwaitingPacket() - Receive and Process received SNMP response *
 ************************************************************************/
int ProcessAwaitingPacket(A)
     struct AddressListType *A;
{
int flags=0;
char rpacket[1024];
struct VarList *v,*v2;
extern struct bindings *save_SNMP_var_Head;
struct bindings *binding_head;
struct bindings *b,*b2,*btail,*btest;
struct sockaddr from;
int fromlen,recvlen,rc;
char *badvar;		/* pointer to errored variable */

struct bindings *AddBindingToChain();

    if (( A==NULL ) || ( A->QueryListHead==NULL ) ) {
	fprintf(stderr,"received packet from invalid AddressList.\n");
	return 0;
    }
    A->Busy=1;		/* Don't interupt me to send get nexts */
	/****** Receive the Packet ********/
    recvlen = recvfrom( A->SNMP_socket, rpacket,sizeof(rpacket),flags, 
			&from, &fromlen);
    if ( verbose ) printf("%d chars received from socket %d for %s\n",recvlen,A->SNMP_socket,A->Address);

    save_SNMP_var_Head = binding_head = NULL;
    A->LastResponded=time(&A->LastResponded);
	/*************************************************************
	 ** parse_SNMP_packet will automatically call save_SNMP_var **
	 ** which will have the bindings chained together           **
         *************************************************************/
    if ((rc=parse_SNMP_packet(rpacket, recvlen, &binding_head, &badvar))!=0) {
		fprintf(stderr,"parse_SNMP_packet from %s %s failed rc=%d %s\n",
			A->Address,IDtoName(badvar),rc,SNMP_errormsg(rc));
		if ( DebugOnFailure ) {
		   if ( io_debug == 0 ) {
			io_debug=1;
    			rc=parse_SNMP_packet(rpacket, recvlen, &binding_head);
			io_debug=0;
		   }
		}
			/***** SNMP Returned an ERROR from HOST *****/
			/* For this release, just terminate query */
		for( v=A->QueryListHead; v!=NULL; v=v->Next )
			MarkDone( A, v );	/* Do we watn to clear answers
						so it looks like the node didn't
						respond?  No. . . */
		A->Busy=0;
		return(1);
    }

    for(b=binding_head; b!=NULL; b=b->Next) 
	save_SNMP_var_Head=AddBindingToChain(save_SNMP_var_Head, b->instance, b->value);

    if (verbose) {
	for (btest=save_SNMP_var_Head; btest != NULL; btest = btest->Next)
		fprintf(stderr, "chain ins %s val %s\n", btest->instance, btest->value);
	for (v=A->QueryListHead; v != NULL; v = v->Next)
		fprintf(stderr, "varlist chain %x\n", v);
    };

		/****** FOR EACH BINDING WE GOT  ********/
    for( 	b=save_SNMP_var_Head, v=A->QueryListHead; 
		b!=NULL && v!=NULL; 
		v=v2,b=b2 	) {
if (verbose) fprintf(stderr,"received a binding from %s %s %s %s\n",A->Address,v->Prefix,b->instance,b->value);
	b2=b->Next;	/* Keep the next pointers around because we may */
	v2=v->Next;	/* Dequeue these structures leaving us stranded */
        b->Next=NULL;	/* Dequeue this received binding */

	/******************************************************************
	 *  Main Switch for SNMPGET/SNMPNEXT versus SNMPTABLE, where more *
	 * work is involved.						  *
	 ******************************************************************/
	switch( A->SNMP_Cmd ) {
	case SNMPGET:
	case SNMPNEXT:
				v->Head=b; /* There will only be ONE binding */
				if (verbose) fprintf(stderr,"MARKDONE (Get) %s %s (%s)\n",A->Address,v->VarName, v->Prefix);
				MarkDone( A, v ); /* This Query DONE */
				break;
	case SNMPTABLE:
	if ( strncmp( b->instance, v->Prefix, strlen(v->Prefix)) != 0 ) {
		if (verbose) fprintf(stderr,"MARKDONE (Table) %s %s (%s) (%s)\n",A->Address,v->VarName, b->instance, v->Prefix);
		MarkDone( A, v );      /* Mark this Query Complete, 	*
					* put on Done Chain 		*/
	}
	else {
		/* Go to the end of bindings list, and check for duplicates *
		 * on the way.						    */
            for( btail=v->Head; btail!=NULL; btail=btail->Next ) {
               if ( strcmp(b->instance,btail->instance) == 0 ) {
                   if (verbose) fprintf(stderr,"Already got this var!\n");
    		   save_SNMP_var_Head = FreeBindingChain(b);
    		   A->Busy=0;	
		   return(1);
		}
		   if ( btail->Next == NULL ) break;
               }
	    if ( btail == NULL ) {
			v->Head=b;	/* Put on head */
			if (verbose) fprintf(stderr,"Putting %s %s on the head\n",
				A->Address, b->instance );
	    }
            else {
			btail->Next=b;			/* Put on tail */
			if (verbose) fprintf(stderr,"Putting %s %s on the tail after %s\n",
				A->Address, b->instance,btail->instance );
	    }
	}
	break;
	default:
		fprintf(stderr, "Internal Error: Unrecognized SNMP_Cmd Type=%d\n",A->SNMP_Cmd);
		break;
	}  /* End Switch */
    }
    save_SNMP_var_Head = NULL;
    A->Busy=0;	
    if ( A->QueryListHead != NULL ) {
        if (verbose) fprintf(stderr,"SENDING GET-NEXT FOR %s \n", A->Address);
    	SendSNMPRequest(A);
    }
    return(0);
}


/************************************************************************
 * ReadResponse() - Find Address struct that owns SNMP socket for which *
 *                      there is a SNMP packet awaiting.                *
 ************************************************************************/
int ReadResponse(socket)
     int socket;
{
struct AddressListType *A;

	for( A=SendSNMPRequest_Address_List_Head; A!=NULL; A=A->Next ) {
		if ( A->SNMP_socket == socket ) {
			return(ProcessAwaitingPacket( A ));
		}
	}
	fprintf(stderr,"ReadResponse( socket=%d ) No Such Socket!\n",socket);
	return(1);
}

/************************************************************************
 *	MakeFDS() - Given a pointer to a list of AddressList Structures *
 *		make a select() file descriptor bit mask.		*
 ************************************************************************/
int MakeFDS( A, fdmask )
struct AddressListType *A;
fd_set *fdmask;
{
int GotOne=0;

	FD_ZERO( fdmask );
	for( ; A!=NULL; A=A->Next ) 
		if ( A->QueryListHead ) {
			FD_SET( A->SNMP_socket, fdmask );
			GotOne++;
		}
	return(GotOne);
}

/************************************************************************
 *	QueryNetwork()  -Main Query Command				*
 *			-Invoked directly or indirectly via Get(), 	*
 *			GetNext(), GetTable().				*
 *			-Assumes AddressList Structure is set up with	*
 *			Community and VarList Chain and SNMP_Cmd.	*
 ************************************************************************/
struct AddressListType *QueryNetwork(AddressListHead)
     struct AddressListType *AddressListHead;
{
int nfds,i,Done;
struct AddressListType *A;
struct VarList *v;
fd_set rfds;
fd_set readfds;
int MaxSocket=0;

	TimedOut=0;
	Retries=0;
		/* Initialize the data structure */
	MakeFDS( AddressListHead, &readfds );
	for( A=AddressListHead; A!=NULL; A=A->Next ) {
		A->LastResponded=0;
		A->ResponseListHead=NULL;
		A->Busy=0;
		if ( A->SNMP_socket > MaxSocket ) MaxSocket=A->SNMP_socket;
	}
	SendSNMPRequest_Address_List_Head=AddressListHead;
	if (verbose) DisplayAddressChain( SendSNMPRequest_Address_List_Head );
	BlastSNMPPackets( );
	/*********** Here do a select waiting for response ************/
	for( ; !TimedOut; ) {
		if (MakeFDS( AddressListHead, &rfds )==0) break; /* DONE? */
                nfds = select( MaxSocket+1, &rfds, NULL, NULL, NULL);
		if (verbose) fprintf(stderr,"number of filedescriptors that matched criteria=%d\n",nfds);
		if ( nfds >0 ) {
                	for (i = 0; i <= MaxSocket; i++) {
                        	if ( FD_ISSET( i, &rfds ) ) {
					ReadResponse( i );
                        	}
                	}
		}
        }
	/* For EACH UNANSWERED VARLIST - free any partial answers */
	for( A=SendSNMPRequest_Address_List_Head; A!=NULL; A=A->Next ) {
		for( v=A->QueryListHead; v!=NULL; v=v->Next ) {
				v->Head=FreeBindingChain( v->Head );
		}
	}
	return( SendSNMPRequest_Address_List_Head );
 	/***********************************************************
	 * ResponseListHead contains completed answers,		   *
 	 * QueryListHead contains incomplete queries 		   *
	 ***********************************************************/
}

/************************************************************************
 * Get() - Send and receive SNMP packets given:                    	*
 * struct AddressListType *AddressListHead - set of Addresses to Query  *
 * char *Community - community string to use with all queries           *
 * struct VarList *QueryList - a list of variables to query             *
 *	If NULL, it is assumed that caller already set up VarLists 	*
 *	for the Address Chain 						*
 ************************************************************************/
struct AddressListType *Get(AddressListHead, Community, QueryList)
     struct AddressListType *AddressListHead;
     char *Community;
     struct VarList *QueryList; 
{
struct AddressListType *A;
void Dup_Community_onto_AddressList();

	for( A=AddressListHead; A!=NULL; A=A->Next) {
		A->SNMP_Cmd=SNMPGET;
	}
	if ( Community != NULL )
		Dup_Community_onto_AddressList( AddressListHead, Community );
	if ( QueryList != NULL )
		Dup_Var_List_onto_AddressList( AddressListHead, Community, QueryList );
	return( QueryNetwork( AddressListHead ));
}

/************************************************************************
 * GetNext() - Send and receive SNMP packets given:                    	*
 * struct AddressListType *AddressListHead - set of Addresses to Query  *
 *									*
 * char *Community - community string to use with all queries           *
 * struct VarList *QueryList - a list of variables to query             *
 *	If NULL, it is assumed that caller already set up VarLists 	*
 *	for the Address Chain 						*
 ************************************************************************/
struct AddressListType *GetNext(AddressListHead, Community, QueryList)
     struct AddressListType *AddressListHead;
     char *Community;
     struct VarList *QueryList; 
{
struct AddressListType *A;
void Dup_Community_onto_AddressList();

	for( A=AddressListHead; A!=NULL; A=A->Next)
		A->SNMP_Cmd=SNMPNEXT;
	if ( Community != NULL )
		Dup_Community_onto_AddressList( AddressListHead, Community );
	if ( QueryList != NULL )
		Dup_Var_List_onto_AddressList( AddressListHead, Community, QueryList );
	return( QueryNetwork( AddressListHead ));
}

/************************************************************************
 * GetTable() - Send and receive SNMP packets given:                    *
 * struct AddressListType *AddressListHead - set of Addresses to Query  *
 * char *Community - community string to use with all queries           *
 * struct VarList *QueryList - a list of variables to query             *
 *	If NULL, it is assumed that caller already set up VarLists 	*
 *	for the Address Chain 						*
 ************************************************************************/
struct AddressListType *GetTable(AddressListHead, Community, QueryList)
     struct AddressListType *AddressListHead;
     char *Community;
     struct VarList *QueryList; 
{
struct AddressListType *A;
void Dup_Community_onto_AddressList();

	for( A=AddressListHead; A!=NULL; A=A->Next)
		A->SNMP_Cmd=SNMPTABLE;
	if ( Community != NULL )
		Dup_Community_onto_AddressList( AddressListHead, Community );
	if ( QueryList != NULL )
		Dup_Var_List_onto_AddressList( AddressListHead, Community, QueryList );
	return( QueryNetwork( AddressListHead ));
}
