/***********************************************************************
 * SortStatus.c - Sort the Status File into a set of AddProblem and    *
 *		DelProblem requests.				       *
 ***********************************************************************/
#include <stdio.h>
#include <time.h>
#include <ctype.h>
#include <string.h>
#include <sys/param.h>
#include "CommonDefs.h"
#include "StatusFileio.h"
#include "problem.h"
#include "pathnames.h"


static char *DefaultPingkyDir = DEFAULT_PINGKY_DIR;
static char Checking_File[MAXPATHLEN];
static char cycletime[MAXPATHLEN];
static char logfile[MAXPATHLEN];
static char Problem_File[MAXPATHLEN];
static char hostname[MAXHOSTNAMELEN];
static char StateFile[MAXPATHLEN];
char ELog[MAXPATHLEN];
char *Error_Log=ELog;

/*	Split the Network into a list of Up/Down/Busy/NR Nodes 		*/
static struct NodeType *UPNodeHead,*DOWNNodeHead,
*NRNodeHead,*BUSYNodeHead,
*UPNodeTail,*DOWNNodeTail,
*NRNodeTail,*BUSYNodeTail,
*TestNodeHead,*TestNodeTail;

struct LinkType *UPLinkHead,*DOWNLinkHead,*NRLinkHead,
		*TestLinkHead,*MOVEDLinkHead;
struct LinkType *UPLinkTail,*DOWNLinkTail,*NRLinkTail,
		*TestLinkTail,*MOVEDLinkTail;
struct LinkType *NextLink,*l;

char *progname;

#define ADD(N,HEAD,TAIL) { 					\
				if ( HEAD == NULL ) { 		\
					HEAD=TAIL=N; 		\
				}				\
				else {				\
					TAIL->Next=N;		\
					TAIL=N;			\
				}				\
				TAIL->Next=NULL;		\
			}

int Down( IPAddress )
char *IPAddress;
{
        register struct NodeType *n=DOWNNodeHead;

        while( n != NULL )
                if ( strcmp( n->IPAddress , IPAddress ) == 0 ) return( 1 );
                else n=n->Next;
        return( 0 );
}

int Busy( IPAddress )
char *IPAddress;
{
	register struct NodeType *n=BUSYNodeHead;

	while( n != NULL )
		if ( strcmp( n->IPAddress , IPAddress ) == 0 ) return( 1 );
		else n=n->Next;
	return( 0 );
}

struct NodeType *SSFindNode( IPAddress )
char *IPAddress;
{
	register struct NodeType *n=UPNodeHead;

	for( n=DOWNNodeHead;n != NULL; n=n->Next )
		if ( strcmp(n->IPAddress,IPAddress) == 0 ) return( n );
	for( n=UPNodeHead;n != NULL; n=n->Next )
		if ( strcmp(n->IPAddress,IPAddress) == 0 ) return( n );
	for( n=NRNodeHead;n != NULL; n=n->Next )
		if ( strcmp(n->IPAddress,IPAddress) == 0 ) return( n );
	for( n=BUSYNodeHead;n != NULL; n=n->Next )
		if ( strcmp(n->IPAddress,IPAddress) == 0 ) return( n );
	return(NULL);
}

struct LinkType *SSFindLink( IPAddress )
char *IPAddress;
{
	register struct LinkType *l=UPLinkHead;

	for( l=DOWNLinkHead;l != NULL; l=l->Next )
		if (( strcmp(l->Node1,IPAddress) == 0 ) || ( strcmp(l->Node2,IPAddress) == 0 ))
			return( l );
	for( l=UPLinkHead;l != NULL; l=l->Next )
		if (( strcmp(l->Node1,IPAddress) == 0 ) || ( strcmp(l->Node2,IPAddress) == 0 ))
			return( l );
	for( l=NRLinkHead;l != NULL; l=l->Next )
		if (( strcmp(l->Node1,IPAddress) == 0 ) || ( strcmp(l->Node2,IPAddress) == 0 ))
			return( l );
	return(NULL);
}

void DisplayLinkChainWComments( Network, LPtr )
struct NetworkType *Network;
struct LinkType *LPtr;
{
	char buf[100],*p;
	char LocalName[100],RemoteName[100],LinkType[100];
	struct NodeType *LocalNode,*RemoteNode;
	extern struct NodeType *FindNode();
	long TimeNow=time(&TimeNow);

	for( ;LPtr != NULL;LPtr = LPtr->Next ) {
		if (LPtr->Type==EGPLINK) continue;
		if ((LocalNode=SSFindNode(LPtr->Node1))==NULL) {
			fprintf(stderr,"Can't find LocalNode %s\n",LPtr->Node1);
			continue;
		}
		if ((RemoteNode=SSFindNode(LPtr->Node2))==NULL) {
			fprintf(stderr,"Can't find RemoteNode %s\n",LPtr->Node2);
			continue;
		}
		/******MAKE LOCAL AND REMOTE NAME ******/
		if ( strcmp(LocalNode->Name,"NotKnown")!=0){
			strcpy(LocalName,LocalNode->Name);
			if (!isdigit(LocalName[0]))
				p=strtok(LocalName,". \t\n");
		}
		else
			strcpy(LocalName,LocalNode->IPAddress);

		if ( strcmp(RemoteNode->Name,"NotKnown")!=0) {
			strcpy(RemoteName,RemoteNode->Name);
			if (!isdigit(RemoteName[0]))
				p=strtok(RemoteName,". \t\n");
		}
		else 
			strcpy(RemoteName,RemoteNode->IPAddress);

		strcpy(buf,LocalName);
		strcat(buf,"-");
		strcat(buf,RemoteName);
		LinkType[0]='\0';
		switch( LPtr->Type ) {
			case ASLINK: sprintf(LinkType,"AS"); break;
			case ISISLINK: sprintf(LinkType,"ISIS"); break;
			case INPLINK: sprintf(LinkType,"INPLINK"); break;
			default: sprintf(LinkType,"LINK"); break;
		}
		/*** PROCESS ***/
		if ( LPtr->State == DOWN ) {
			if (( LPtr->Type == ASLINK ) && ( TimeNow-LPtr->TimeStamp < 2*60+30 )) continue;	/* Don't report Down AS for 2 1/2 minutes */
			Problem_Manager( ADD_PRIMARY_PROBLEM, PrintState( LPtr->State ),
			    LinkType,buf,NULL);
		}
		else
			if ( LPtr->State == UP || LPtr->State == MOVED ) {
				if ( LPtr->State == MOVED ) /*DEBUG*/
					fprintf(stderr,"Removing problems with MOVED link\n");
				Problem_Manager( DELETE_PROBLEM, PrintState(DOWN),
				    LinkType,buf,NULL);
			}
	}
}

void DisplayNodeChainWComments( Network, NPtr )
struct NetworkType *Network;
struct NodeType *NPtr;
{
	char buffer[100];
	long TimeNow=time(&TimeNow);

	for( ; NPtr!=NULL; NPtr = NPtr->Next ) {
		if (NPtr->Type==EGPPEER) continue;
		if (  ( NPtr->State == DOWN ) || 
		    ( ( NPtr->State == BUSY) && 
		      ( TimeNow-NPtr->TimeStamp > 5*60 )) ||
		    ( (NPtr->State==NR) && (NPtr->Type==PCP) ) ||
		    ( ( NPtr->State == NR ) && 
                      /*(SSFindLink( NPtr->IPAddress)==NULL) && original */
                      ( NPtr->Type == AS ) &&    /* ANY AS delay showing */
		      ( TimeNow-NPtr->TimeStamp > 5*60 ) ) 
		   ) {
			if ( strcmp( NPtr->Name, "NotKnown" ) == 0 ) {
				sprintf(buffer,"%s",PrintNodeType(NPtr->Type));
				Problem_Manager( ADD_PRIMARY_PROBLEM, 
				    PrintState( NPtr->State ),
				    buffer,NPtr->IPAddress,NULL);
			}
			else {
				sprintf(buffer,"%s_%s",PrintNodeType(NPtr->Type),NPtr->Name);
				Problem_Manager( ADD_PRIMARY_PROBLEM, 
				    PrintState( NPtr->State ),
				    buffer,NPtr->IPAddress,NULL);
			}
		}
		if ( NPtr->State == UP ) {
			if ( strcmp( NPtr->Name, "NotKnown" ) == 0 ) {
				sprintf(buffer,"%s",PrintNodeType(NPtr->Type));
				Problem_Manager(DELETE_PROBLEM,PrintState(DOWN),
				    buffer,NPtr->IPAddress,NULL);
				Problem_Manager(DELETE_PROBLEM,PrintState(BUSY),
				    buffer,NPtr->IPAddress,NULL);
				Problem_Manager(DELETE_PROBLEM,PrintState(NR),
				    buffer,NPtr->IPAddress,NULL);
			}
			else {
				sprintf(buffer,"%s_%s",PrintNodeType(NPtr->Type),NPtr->Name);
				Problem_Manager(DELETE_PROBLEM,PrintState(DOWN),
				    NPtr->Name,NPtr->IPAddress,NULL);
				Problem_Manager(DELETE_PROBLEM,PrintState(BUSY),
				    NPtr->Name,NPtr->IPAddress,NULL);
				Problem_Manager(DELETE_PROBLEM,PrintState(NR),
				    NPtr->Name,NPtr->IPAddress,NULL);
			}
		}
	}
}

void GroupNodes( Network )
struct NetworkType Network;
{
	struct NodeType *NextNode,*n;

	for ( n=NextNode=Network.NodeHead; n!=NULL; n=NextNode ) {
		NextNode=n->Next;
		if ( TestNode( n->Name )) {
			/*printf("ADDING TEST NODE: %s \n",n->Name);*/
			ADD(n,TestNodeHead,TestNodeTail);
			continue;
		}
		switch( n->State ) {
		case DOWN:
			ADD(n,DOWNNodeHead,DOWNNodeTail);
			break;
		case BUSY:
			ADD(n,BUSYNodeHead,BUSYNodeTail);
			break;
		case NR:
			ADD(n,NRNodeHead,NRNodeTail);
			break;
		case UP:
			ADD(n,UPNodeHead,UPNodeTail);
			break;
		default:
			printf("Bad State=%d\n",n->State);
			break;
		}
	}
}

void GroupLinks( Network )
struct NetworkType Network;
{
	struct LinkType *NextLink,*l;

	for ( l=NextLink=Network.LinkHead; l!=NULL; l=NextLink ) {
		NextLink=l->Next;
		if ( ( TestNode( l->Node1 )) || ( TestNode( l->Node2 )) ) {
			/*printf("ADDING TEST LINK %s-%s\n",l->Node1,l->Node2);*/
			ADD(l,TestLinkHead,TestLinkTail);
			continue;
		}
		switch( l->State ) {
		case DOWN:
			if ((!Down( l->Node1 )) && (!Down( l->Node2 ))){
				ADD(l,DOWNLinkHead,DOWNLinkTail);
			}
			else
				ADD(l,TestLinkHead,TestLinkTail);
			break;
		case NR:
			if ((!Busy( l->Node1 )) && (!Busy( l->Node2 ))){
				ADD(l,NRLinkHead,NRLinkTail);
			}
			else
				ADD(l,TestLinkHead,TestLinkTail);
			break;
		case UP:
			ADD(l,UPLinkHead,UPLinkTail);
			break;
		case MOVED:
				ADD(l,MOVEDLinkHead,MOVEDLinkTail);
			break;
		default:
			printf("Bad State=%d\n",l->State);
			break;
		}
	}
}


static int NetworkType;
char *GetStateFile( prog, parm )
char *prog,*parm;
{
	static char statefile[100];
	char *p;

	if ( parm != NULL ) {
		if ((p=strrchr( parm , '/' ))==NULL) strcpy( statefile,
		    parm );
		else strcpy( statefile, p+1 );
		if ((p=strrchr( statefile, '.' ))!=NULL) *p='\0';
		NetworkType=Str_To_NetworkType( statefile );
		return( parm );
	}
	if ((p=strrchr( prog , '/' ))==NULL) strcpy( statefile, prog );
	else strcpy( statefile, p+1 );
	p=statefile;
	while( isupper(*p) || isdigit(*p) ) p++;
	*p='\0';
	NetworkType=Str_To_NetworkType( statefile );
	strcat( statefile, ".STATUS" );
	return( statefile );
}

Usage()
{
	fprintf(stderr,"Usage: %s -P ccccc -s <NetworkStatusFile>\n\n",progname);
	exit(1);
}

/*int ResolveNames=0;*/
main( argc, argv )
int argc;
char *argv[];
{
	struct NetworkType Network;
	char buf[100];
	long LastTimeStatusFileChanged;
	char *pingkydir,*getenv();
	char buffer[200];
	extern char Version[];
	int c;
	extern char *optarg;
	char *SFile;

	progname=argv[0];
        while ((c = getopt(argc, argv, "P:s:")) != EOF)
                switch (c) {
                case 'P':
                        SetProblemPrefix( optarg );
                        break;
                case 's':
                        SFile=optarg;
                        break;
		default:
			Usage();
			/*NOTREACHED*/	
	}
	if ( SFile == NULL ) {
		Usage();
	}
	if ((pingkydir=getenv("PINGKYDIR"))==NULL) pingkydir=DefaultPingkyDir;
	sprintf(Checking_File, "%s/%s", pingkydir, CHECK_FILE);
	sprintf(cycletime, "%s/%s", pingkydir, PINGD_CYCLE);
	sprintf(logfile, "%s/%s", pingkydir, PROBLEM_LOG);
	sprintf(Problem_File, "%s/%s", pingkydir, PROBLEM_FILE);
	sprintf(ELog, "%s/%s", pingkydir, ROVER_LOG);

	gethostname( hostname, sizeof(hostname));
	strtok( hostname, ". \t\n");
#ifdef VERBOSE
	sprintf(buffer,"Starting %s %s\n",hostname,Version);
	printf(buffer);
	Log(buffer,logfile);
#endif

	buf[0]='\0';
	Problem_Manager( CHECK, NULL, NULL, NULL, NULL );

	GetStateFile( argv[0], SFile );
	strcpy( StateFile, pingkydir); 
	strcat(StateFile, "/");
	strcat( StateFile, SFile );

	InitNetwork( &Network , NetworkType, StateFile );
	while( 1 ) {
		if ( FileChanged( StateFile, &LastTimeStatusFileChanged) ) {
			FreeNodeChain( DOWNNodeHead );
			FreeNodeChain( BUSYNodeHead );
			FreeNodeChain( NRNodeHead );
			FreeNodeChain( UPNodeHead );
			FreeNodeChain( TestNodeHead );
			FreeLinkChain( UPLinkHead );
			FreeLinkChain( DOWNLinkHead );
			FreeLinkChain( NRLinkHead );
			FreeLinkChain( TestLinkHead );
			FreeLinkChain( MOVEDLinkHead );
			ReadConfig( &Network , NetworkType , StateFile);
			FileChanged( StateFile, &LastTimeStatusFileChanged );
		}
		else {
			continue;
		}
		GroupNodes( Network );  /* Group Nodes into UP,BUSY,DOWN,NR */
		GroupLinks( Network );  /* Group Links into UP, DOWN and NR */

		if ( DOWNNodeHead != NULL )
			DisplayNodeChainWComments( &Network, DOWNNodeHead );
		if ( BUSYNodeHead != NULL )
			DisplayNodeChainWComments( &Network, BUSYNodeHead );
		if ( NRNodeHead != NULL )
			DisplayNodeChainWComments( &Network, NRNodeHead );
		if ( UPNodeHead != NULL )
			DisplayNodeChainWComments( &Network, UPNodeHead );

		if ( DOWNLinkHead != NULL )
			DisplayLinkChainWComments( &Network, DOWNLinkHead );
		if ( NRLinkHead != NULL )
			DisplayLinkChainWComments( &Network, NRLinkHead );
		if ( UPLinkHead != NULL )
			DisplayLinkChainWComments( &Network, UPLinkHead );
		if ( MOVEDLinkHead != NULL )
			DisplayLinkChainWComments( &Network, MOVEDLinkHead );
	
#ifndef FOREVERLOOP	
		break;
#endif
		sleep( 30 );
	}
	exit( 0 );
}
