/*
 * $Header: /noc/network/netlog/src/RCS/menu.y,v 2.4 1992/06/01 19:05:36 aggarwal Exp $
 */

/*+ 
** FUNCTION:
**	Display the main menu and get user's selection. Returns the input
** character which the 'decider' uses to call the respective program.
**
**	1. Start up curses
**	2. display menu, highliting the selection at 'pselect'
**	3. parse input (here uses the yacc function 'get_valid_selection')
**	  i) highlite the new selection and redisplay menu
** 	 ii) if selected using RET, then return the selection's char.
**
** I tried using the new curses 'keypad' function for the arrow keys.
** Interestingly, if the startup_curses() function is after the main,
** the keypad() function does not work. Thus was deleted.
**
** Define TEST while compiling to get a sample test program.
**
** AUTHOR:
**	Vikas Aggarwal, vikas@jvnc.net, March 1990
**
**/

/* Copyright 1992 JvNCnet, Princeton University

 Permission to use, copy, modify and distribute this software and its
 documentation for any purpose is hereby granted without fee, provided
 that the above copyright notice appear in all copies and that both
 that copyright notice and this permission notice appear in supporting
 documentation, and that the name of JvNCnet or Princeton University
 not be used in advertising or publicity pertaining to distribution of
 the software without specific, written prior permission.  Princeton
 University makes no representations about the suitability of this
 software for any purpose.  It is provided "as is" without express or
 implied warranty.

 PRINCETON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 FITNESS. IN NO EVENT SHALL PRINCETON UNIVERSITY BE LIABLE FOR ANY
 DAMAGES WHATSOEVER, INCLUDING DAMAGES RESULTING FROM LOSS OF USE, DATA
 OR PROFITS, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/


/*
 *	$Log: menu.y,v $
 * Revision 2.4  1992/06/01  19:05:36  aggarwal
 * Changed the '\%' to a '%%'
 *
 * Revision 2.3  1992/05/31  01:20:21  aggarwal
 * Caved in to popular demand- the 'ctrl' now treats its args as int and
 * not a character (removed the 'c' in the define)
 *
 * Revision 2.2  1992/05/08  15:31:08  aggarwal
 * Wasn't returning JUNK if the user hit a RETURN on startup. Added check.
 *
 * Revision 2.1  1992/04/27  12:30:10  aggarwal
 * Now doing an 'undef' for CTRL since Ultrix was doing an 'c & 037' instead
 * of a 'c' & 037.
 *
 * Revision 2.0  1992/04/26  01:04:31  aggarwal
 * Self sustaining independent kinda module, and the arrow keys work (yippee).
 * 
 * Revision 1.1  90/03/01  14:33:28  aggarwal
 * Initial revision
 * 
 */

%token	OKCHAR 	UPARROW  DNARROW REFRESH TAB JUNK

%{						/* start C definitions	*/
#ifndef lint
  static char rcsid[] = "$RCSfile: menu.y,v $ $Revision: 2.4 $Date$" ;
#endif

#include <curses.h>

/*
 * For 4.2 curses.
 */
#ifndef cbreak
#define cbreak() crmode()
#endif
#ifndef nocbreak
#define nocbreak() nocrmode()
#endif

/*
 * Usually defined in ttychars.h.
 */
#ifndef ESC		/* ESCAPE character */
#define ESC 033
#endif

#ifdef CTRL		/* Some implementations do a: c & 037 (assume int) */
#undef CTRL		/* .. so undefine it and use our version */
#endif	/* CTRL */
#define CTRL(c)         (c & 037)	/* treat as an integer */

#ifndef max
#define max( _a , _b)		( (_a) > (_b) ?  (_a) : (_b) )
#endif

#define	OPT_X	10			/* location of options on terminal  */
#define	OPT_Y	6

static int debug ;			/* for debugging, shows kbd input */
static int numoptions, pselect;		/* total options, selection index */
static char **options, *prompt ;	/* array of options and prompt */
static char uparrow[10], dnarrow[10] ;	/* arrow key sequences  */
%}

/* ****** */
%%						/* start yacc rules	*/
 final:		/* empty */			/* use recursion..	*/
                | final valid			/* for endless loop	*/
                ;
 valid:	        character
                | arrow
                | refresh
                | tab
                | junk
                ;
 character:	OKCHAR				/* gets char in yyval	*/
                        { if (debug) fprintf(stderr, "Got OKCHAR\n"); 
			   display_menu() ; }
 arrow:		UPARROW
                        { if (debug) fprintf(stderr,"Got UPARROW\n") ;
			  pselect = (pselect + numoptions - 1)%numoptions ;
			  display_menu(); }
                | DNARROW
                        { if (debug) fprintf(stderr,"Got DNARROW\n") ;
			  pselect = (pselect + 1)%numoptions ; 
		           display_menu() ; }
                ;
 refresh:	REFRESH
                        { if (debug) fprintf(stderr,"Got REFRESH\n") ;
			   wrefresh(curscr) ; }
                ;
 tab:		TAB
                        { if (debug) fprintf(stderr,"Got TAB\n") ;
			  pselect = (pselect + numoptions - 1)%numoptions ;
			  display_menu(); }
 junk:		JUNK
                       { if (debug) fprintf(stderr,"Got JUNK\n") ; }
                ;
%%						/* end yacc rules	*/

/****************** Start yacc related functions **********************/

yyerror(s)
{
    fprintf(stderr, "menu parser: %s\n", s);
}

/*+			yylex
** FUNCTION:
**	Our li'l intelligent lexical analyser. Returns a zero only
** if 'pselect' is not -1.
** Sets the value of the static global variable 'pselect' to reflect
** the array index of the selection made if a char is read in.
*/

yylex()
{
    int input ;					/* int for keypad keys	*/
    register int i;
    char *s ;					/* tmp character ptr */

    input = getch() ;

    if (debug)
      fprintf(stderr, "(yylex): Got %%d (%c)\n", input,
	      (input < ' ' || input > '~') ? '?' : input) ; 

    for (i = 0; i < numoptions ; ++i)
      if ( toupper(*(options[i])) == toupper(input) )
      {
	  pselect = i ;			/* modify selection value	*/
	  yyval = input ;		/* in case its needed		*/
	  return (OKCHAR);
      }

    switch (input)
    {
    case CTRL('l'):		/* refresh screen */
      return (REFRESH) ;
      break;
       
    case '+':
    case '=':
    case '\t':			/* TAB */
    case CTRL('n'):		/* next line */
      return (DNARROW);
       break ;
       
    case '-':
    case CTRL('p'):		/* previous line */
      return (UPARROW);
       break ;
       
#ifdef KEY_UP			/* in new advanced curses */
    case KEY_UP:
       return (UPARROW);	
       break;
#endif
#ifdef KEY_DOWN
    case KEY_DOWN:
       return(DNARROW);
       break;
#endif

    /* In case end of user input using RET or end of file, return
    ** a 0 to yacc only if pselect has a valid value.
    */
    case '\0':
    case '\n':
    case EOF:
      if (pselect >= 0)
	return (0);				/* signify end to yacc	*/
      else
	return(JUNK) ;				/* No selection made */
       break;

    default:					/* check for arrow keys */
      i = check_arrows (input) ;
      if (debug)
	fprintf(stderr, "(yylex): check_arrow returned %d\n", i) ;
      if ( i == 1)
	return (UPARROW) ;
      else if ( i == 2)
	return (DNARROW) ;
      else 
	return(JUNK);				/* not recognized key	*/
      break;
   }	/* end switch */

}						/* end: yylex		*/


/*+ 
** FUNCTION:
** 	Checks to see if user input can match an arrow key. Reads in
** chars as long as a match with either one exists, flushes input and
** sends 0 if no match with either arrow key. Returns 1 for UP 2 for
** DN arrow.
**
** General escape seq = ESC-O-A from 'tgetent', but the terminal returns
** ESC-[-A instead. Due to mass confusion, and lack of documentation on this
** topic, I am following the 'tcsh' gurus and mapping the 'O' to '['.
**/

check_arrows(c)
     int c ;
{
    static int maxread ; 
    int cread ;			/* char read so far */
    int input = c , uflag = 1, dflag = 1;
    char *u = uparrow , *d = dnarrow ;

    if (!maxread)
      maxread = max (strlen(uparrow) , strlen(dnarrow)) ;

    if (*u == '\0')
      uflag = 0;
    if (*d == '\0')
      dflag = 0;

    for (cread = maxread ; cread > 0 ; --cread )
    {
	if (input == '[')	/* map into 'O', see comment above */
	  input = 'O' ;		/* .. since 'tgetent' diff from kbd */

	if (debug)
	  fprintf(stderr, "(check_arrows): checking input:%d with %d/%d\n",
		  input, *u, *d) ;

	if (dflag && *d)		/* check with DN arrow */
	  if (input != *d++)		/* dnarrow mismatch */
	    dflag = 0 ;

	if (uflag && *u)		/* check with UParrow */
	  if (input != *u++)		/* uparrow mismatch */
	    uflag = 0 ;

	if (cread ==  1)		/* don't read next character */
	  break ;			/* needed since reading in for loop */

	if (dflag || uflag)
	  input = getch() ;		/* read next character */
	else
	  break ;			/* both mismatched  */

    }	/* end for()  */

    if (dflag)
      return (2) ;
    else if (uflag)
      return (1) ;
    else
      return (0) ;

} 	/* End check_arrows()  */

/********************* End yacc related functions ************************/

#ifdef TEST
char *menu_options[] = {	/* must be defined statically */
    "Option 1",
    "Choice 2",
    "Do not have...",
    "Same first characters",
    "Use ^p, ^n, TAB, +, -",
    "Choice 3 (note what happens if same first chars)",
    "Quit",
    ""
  } ;

main (ac, av)
     char **av;
{
    char *menu_title = "Testing Module menu.y" ;
    char *menu_prompt = "Enter first character or TAB or +/-: " ;
    extern char *optarg ;
    extern int optind, opterr ;
    int c ;

    while ( (c = getopt(ac, av, "d")) != EOF)
      switch (c)
      {
       case 'd':
	  debug = 1 ;
	  break ;
       default:
	  break ;
      }

    menu (menu_options, menu_title, menu_prompt);

    exit (0) ;
}
#endif TEST

/**** ######## ****/

menu (menu_options, menu_title, menu_prompt)
     char *menu_options[];	/* Array of options strings */
     char *menu_title ;		/* Menu title to be displayed */
     char *menu_prompt;	  	/* User prompt string */
{
    pselect = -1 ;		/* make it refer to no 'option' element	*/
    numoptions = 0;
    options = menu_options ;	/* use the localized variable */
    prompt  = menu_prompt ;

    while (*options[numoptions])
      ++numoptions; 		/* count total number of options */

    startup_curses() ;		/* initialization sissy stuff */
    write_headers(menu_title);	/* fancy headers stuff */
    display_menu() ;	  	/* display the main menu on the screen	*/
    yyparse() ;
    endwin();					/* end curses */

    printf("\n\n");
    fflush (stdout);
    return (*options[pselect]) ;
}					/* end: menu	*/

/************************************************************************/

/*+ 			startup_curses
** FUNCTION:
** 	Start up curses - set scrolling, crmode, leave cursor after
** last update and anything you can thing of. Extract the arrow key
** sequences.
**/
startup_curses()
{
    char termtype[128], bp[1024] ;		/* needed by tgetent */
    
    initscr() ;
    cbreak();
    noecho();					/* else messes up curses */
    scrollok(stdscr, TRUE);			/* Allow scrolling	*/
    leaveok(stdscr, FALSE);			/* leave cursor at end	*/

    /*
     * Extract arrow keys
     */
    strcpy (termtype, (char *)getenv("TERM")) ;
    if (tgetent (bp, termtype) != 1)
      uparrow[0] = '\0' , dnarrow[0] = '\0' ;	/* NULL strings */
    else
    {
	char *s = (char *)uparrow ;
	tgetstr("ku", &s) ;
	s = (char *)dnarrow ;
	tgetstr("kd", &s) ;
    }
    
}	/* end: startup_curses	*/


/*		write_headers
** FUNCTION:
** 	Write out the header stuff on the terminal 
**/
write_headers(mtitle)
     char *mtitle ;				/* menu title */
{
    extern int LINES, COLS;			/* defined in curses.h	*/
    int titlelen = strlen(mtitle);
    static int st_x, st_y ;		     	/* start xy location	*/
    register i;					/* temp index		*/

    (int)st_x = (COLS/2)-(titlelen/2) - 2;	/* Offset from center	*/
    st_y = OPT_Y - 5;				/* hdr offset from top	*/

    move(st_y, st_x) ;
    for (i = 0; i < titlelen + 4; ++i)
      printw("*");
    printw("\n");
    mvprintw(stdscr->_cury, st_x, "* %s *\n", mtitle);
    move(stdscr->_cury, st_x) ;
    for (i = 0; i < titlelen + 4; ++i)
      printw("*");
    printw("\n");

}	/* end: write headers		*/


/*+ 		display_menu
** FUNCTION:
** 	Display the main menu on the screen. Displays the headers and then
** starts writing out the choices from the location (OPT_X, OPT_Y). Thus
** the first option is churned out at (OPT_X, OPT_Y). Highlights the option
** at location 'pselect'. Notice the two blank spaces while writing out
** the option string - to erase the '* ' in case it was there.
**/
display_menu()
{
    register int i;
    extern WINDOW *stdscr;			/* defined in curses.h */

    move(OPT_Y, OPT_X);				/* Move to start loc	*/
    for (i = 0; i < numoptions; ++i)		/* same indentation	*/
      if ( i == pselect )			/* highlight selection	*/
      {
	  mvprintw(stdscr->_cury, OPT_X - 2, "* ") ;
	  standout();
	  mvprintw(stdscr->_cury, OPT_X, "%s\n", options[i]) ;
	  standend();
      }
      else
	mvprintw(stdscr->_cury, OPT_X - 2, "  %s\n", options[i]) ;
    write_prompt() ;
/*
**    touchwin(stdscr);
*/
    refresh() ;
}					/* end: display menu		*/


/*		write_prompt
** FUNCTION:
** 	Write out the prompt on the terminal 
**/
write_prompt()
{
    static int y, x = 2 ;

    y = OPT_Y + numoptions + 3 ;
    mvprintw (y, x, "%s", prompt);
}					/* end: write prompt		*/
