/*
 * Copyright 1992 SynOptics Communications, Inc.  All Rights Reserved.
 * SynOptics grants a non-exclusive license to use, copy, modify, and distribute
 * this software for any purpose and without fee, provided that this copyright
 * notice and license appear on all copies and supporting documentation.
 * SynOptics makes no representations about the suitability of this software for
 * any particular purpose.  The software is supplied "AS IS", and SynOptics
 * makes no warranty, either express or implied, as to the use, operation,
 * condition, or performance of the software.
 * SynOptics retains all title and ownership in the software.
 *
 * file: SMIB.Y - Grammar for SNMP Concise MIB and Concise Traps
 *
 * Thanks go to Dean Throop at DG for help with error productions.
 *
 * $Workfile:   smib.y  $
 * $Revision:   1.4  $ $Date:   08 Jul 1992 17:44:40  $
 * $Log:   R:/MIBTOOLS/V1.0/SMIC/SRC/SMIB.Y_V  $
 * 
 *    Rev 1.4   08 Jul 1992 17:44:40   gfoster
 * Removed unnecessary revision comment lines added by
 * PVCS to make revision history easier to read.
 * 
 *    Rev 1.3   08 Jul 1992 16:55:32   gfoster
 * Added keyword "IMPLIED" and modified
 * grammar so that either "IMPLIED" could
 * be used where ever "NOLENGTH" is used.
 * 
 *    Rev 1.2   29 Jun 1992 19:40:08   gfoster
 * Cleaned up by removing some irrelevant comments,
 * removed "extra" yyerrok function calls, and fixed
 * the text of some of the error/warning messages.
 * 
 * Casted the value of "-1L" in the calls to
 * checkOIDcomp to "(ULONG)-1L" to prevent
 * signed/unsigned mismatch.
 * 
 *    Rev 1.1   19 Jun 1992 16:35:14   gfoster
 * Copyright text was reformated.
 * 
 * Major changes to add error productions.
 * 
 *    Rev 1.0   27 May 1992 16:25:56   gfoster
 * Initial revision.
 *
*/

%{

/* file: SMPAR.C - from SMIB.Y - Parser for Concise MIB and Concise Traps
*
*/

#include <stdio.h>
#include <string.h>

#include "tds.h"
#include "smscdefs.h"
#include "smstdefs.h"
#include "smsydefs.h"
#include "smic.h"


MIBSYM *pMod;
MIBSYM *pOid;
MIBSYM *pSeq;
MIBSYM *pTc;
MIBSYM *pTr;
MIBSYM *pComp;
MIBSYN syn;
BOOL fNoStrSav;

#define YYSTATIC

%}

/* values for tokens */
%union {
  NUMVAL numval;
  STRVAL strval;
  SYMVAL symval;
  KWVAL  kwval;
}

/* directives */
%token <kwval> dirADDOPT
%token <kwval> dirALIASMODULE
%token <kwval> dirALIASSYMBOL
%token <kwval> dirHELP
%token <kwval> dirINCLUDE
%token <kwval> dirPOPOPT
%token <kwval> dirPRINTOPT
%token <kwval> dirPUSHOPT
%token <kwval> dirREMOVEOPT


/* Keywords */
%token <kwval> kwACCESS
%token <kwval> kwBEGIN
%token <kwval> kwCOUNTER
%token <kwval> kwDEFINITIONS   kwDEFVAL        kwDEPRECATED    kwDESCRIPTION
%token <kwval> kwEND           kwENTERPRISE
%token <kwval> kwFROM
%token <kwval> kwGAUGE
%token <kwval> kwIDENTIFIER    kwIMPLIED       kwIMPORTS       kwINDEX
%token <kwval>                 kwINTEGER       kwIPADDRESS 
%token <kwval> kwMANDATORY
%token <kwval> kwNETWORKADDRESS kwNOLENGTH     kwNOT_ACCESSIBLE kwNULL
%token <kwval> kwOBJECT        kwOBJECT_TYPE   kwOBSOLETE      kwOCTET
%token <kwval>                 kwOF            kwOPAQUE        kwOPTIONAL
%token <kwval> kwREAD_ONLY     kwREAD_WRITE    kwREFERENCE
%token <kwval> kwSEQUENCE      kwSIZE          kwSMI           kwSTATUS
%token <kwval>                 kwSTRING        kwSYNTAX
%token <kwval> kwTIMETICKS     kwTRAP_TYPE
%token <kwval> kwVARIABLES
%token <kwval> kwWRITE_ONLY
         
/* special tokens */
%token <kwval> chLCB            /* "{" */
%token <kwval> chRCB            /* "}" */
%token <kwval> chLPR            /* "(" */
%token <kwval> chRPR            /* ")" */
%token <kwval> chSEMI           /* ";" */
%token <kwval> chCOMMA          /* "," */
%token <kwval> chMINUS          /* "-" */
%token <kwval> chDOT            /* "." */
%token <kwval> chOTHER          /* any other character */

%token <kwval> tokDOTDOT        /* ".." */
%token <kwval> tokBADDOTDOT     /* bad ".." */
%token <kwval> tokIS            /* "::=" */
%token <kwval> tokBADIS         /* bad "::=" */
%token <kwval> tokBADSTR        /* bad string */

/* name, strings, number, bit string, hex string, and IP address */
%token <strval> tokNAME         /* a name */
%token <strval> tokSTRING       /* a quoted string */
%token <numval> tokNUMBER       /* an unsigned number */
%token <strval> tokBSTR         /* a bit string */
%token <strval> tokHSTR         /* a hex string */
%token <numval> tokIP           /* an IP address */

/* type definitions */
%type <strval>  optDesc
%type <strval>  optRefer
%type <symval>  OTdef
%type <symval>  SEQdef
%type <symval>  TCdef
%type <symval>  TTdef
%type <numval>  objectAccess
%type <numval>  objectStatus
%type <numval>  srVal
%type <symval>  oidValue
%type <numval>  OTaccess
%type <numval>  OTstatus
%type <strval>  TTenterprise
%type <kwval>   synType

/* start symbol */
%start mibDef

%%

mibDef
  : moduleAndDirectiveList {
      /* check to see if any modules defined - one should be */
      if (cModUsed == 0)
          yyerror("One or more MIB modules must be defined"); }
  ;

moduleAndDirectiveList
  : moduleOrDirective
  | moduleAndDirectiveList moduleOrDirective
  ;

moduleOrDirective
  : module
  | directive

  /* error production - eat up unknown char */
  | anyExceptDirOrMod {
      yyerror("Start of MIB module or Compiler directives expected - item skipped"); }
  ;

anyExceptDirOrMod
  /* all tokens other than directives or a module name */
  : kwACCESS    | kwBEGIN       | kwCOUNTER     | kwDEFINITIONS | kwDEFVAL
  | kwDEPRECATED | kwDESCRIPTION | kwEND        | kwENTERPRISE  | kwFROM
  | kwGAUGE     | kwIDENTIFIER  | kwIMPLIED     | kwIMPORTS     | kwINDEX
  | kwINTEGER   | kwIPADDRESS | kwMANDATORY   | kwNETWORKADDRESS | kwNOLENGTH
  | kwNOT_ACCESSIBLE | kwNULL   | kwOBJECT      | kwOBJECT_TYPE
  | kwOBSOLETE  | kwOCTET       | kwOF          | kwOPAQUE      | kwOPTIONAL
  | kwREAD_ONLY | kwREAD_WRITE  | kwREFERENCE   | kwSEQUENCE
  | kwSIZE      | kwSMI         | kwSTATUS      | kwSTRING      | kwSYNTAX
  | kwTIMETICKS | kwTRAP_TYPE   | kwVARIABLES   | kwWRITE_ONLY
  | chLCB       | chRCB         | chLPR         | chRPR         | chSEMI
  | chCOMMA     | chMINUS       | chDOT         | chOTHER       | tokDOTDOT
  | tokBADDOTDOT | tokIS        | tokBADIS
  | tokBADSTR   | tokSTRING     | tokNUMBER
  | tokBSTR     | tokHSTR       | tokIP
  ;

directive
  :   {
      fNoStrSav = fNoStr;
      fNoStr = FALSE; }
    directiveItem {
      fNoStr = fNoStrSav; }
  ;

directiveItem
  /* include a file - quoted filename */
  : dirINCLUDE tokSTRING {
      openInclude($2.pStr); }

  /* include a file - error recovery */
  | dirINCLUDE error {
      yyerror("quoted file name expected for \"#include\" directive");
      /* yyerrok; */}

  /* print help */
  | dirHELP {
      dirHelp(); }  

  /* alias a module - moduleName aliasName */
  | dirALIASMODULE tokNAME tokNAME {
      addMODalias($2.pStr, $3.pStr); }

  /* alias a module - error recovery */
  | dirALIASMODULE error {
      yyerror("Module name and alias name expected for \"#aliasModule\" directive");
      /* yyerrok; */}

  /* alias a name in a module - moduleName objectName aliasName */
  | dirALIASSYMBOL tokNAME tokNAME tokNAME {
      addSYMalias($2.pStr, $3.pStr, $4.pStr); }

  /* alias a name in a module - error recovery */
  | dirALIASSYMBOL error {
      yyerror("Module name, object name, and its alias name expected for \"#aliasSymbol\" directive");
      /* yyerrok; */}

  /* push option settings */
  | dirPUSHOPT {
      pushOpt(); }

  /* pop option settings */
  | dirPOPOPT {
      popOpt(); }

  /* add option settings */
  | dirADDOPT tokSTRING {
      addOpt($2.pStr); }

  /* add option settings - error recovery */
  | dirADDOPT error {
      yyerror("Quoted option string expected for \"#addOpt\" directive");
      /* yyerrok;*/}

  /* remove option settings */
  | dirREMOVEOPT tokSTRING {
      removeOpt($2.pStr); }

  /* remove option settings - error recovery */
  | dirREMOVEOPT error {
      yyerror("Quoted option string expected for \"#removeOpt\" directive");
      /* yyerrok; */}

  /* print option settings */
  | dirPRINTOPT {
      printOpt(); }
  ;

module
  : moduleHeader
    optImports {
      fixupImport(pMod); }
    definition
    optDefinitionList
    kwEND {
      moduleEnd:
      /* check that IMPORTED items are referenced */
      checkIMPs(pMod);

      /* check that all OID items have forward reference removed */
      checkOIDs(pMod);

      /* check that all Sequence items have forward ref removed */
      checkSEQs(pMod);

      /* check that all textual conventions referenced */
      checkTCs(pMod);

      /* check that all vars in traps are defined */
      checkTRs(pMod);

      /* make all keywords undefined */
      makeKWundef();
      pMod = NULL; }

  /* error production */
  | moduleHeader
    optImports {
      fixupImport(pMod); }
    kwEND {
      yyerror("Module must contain definitions - terminating");
      YYABORT; }
  | error
    kwEND {
      yyerror("Unrecognizable item in module - skipping to end");
      yyerrok;
      goto moduleEnd; }  
  ;

moduleHeader
  : tokNAME optID kwDEFINITIONS moduleIS kwBEGIN {
      pMod = addMODname($1.pStr); }

  /* error productions */
  | tokNAME error kwBEGIN {
      yyerrok;  
      modHeaderErr:
      yyerror("Syntax for module header is\n  <moduleName> [<oidValue>] \"DEFINITIONS\" \"::=\" \"BEGIN\"");
      pMod = addMODname($1.pStr); }
  | tokNAME optID kwDEFINITIONS {
      goto modHeaderErr; }
  | tokNAME optID tokIS {
      goto modHeaderErr; }
  | tokNAME optID tokBADIS {
      goto modHeaderErr; }
  ;

moduleIS
  : tokIS
  /* error production */
  | tokBADIS {
      yyerror("Must use \"::=\""); }
  ;


optID
  :
  | chLCB compList chRCB {
      finishMODid(); }

  /* error production */
  | chLCB error chRCB {
      yyerror("Syntax for <oidValue> is\n  \"{\" { {<name>[\"(\"<number>\")\"]} | <number> }...\"}\"");
      yyerrok; }
  ;

compList
  : comp
  | compList comp
  ;

comp
  : tokNAME
  | tokNAME chLPR tokNUMBER chRPR
  | tokNUMBER
  ;

/* The optional IMPORTs statement */
/*?? Consider turning on recognition of the OBJECT-TYPE and
     TRAP-TYPE macros (if not already on) as actions here
     and checking the use count at the end of the module.
     If they have not been defined, but are used, then generate
     a warning.
     Also consider not making OBJECT-TYPE, TRAP-TYPE, etc keywords
     until the complete IMPORTs have been processed. If they are
     IMPORTed twice, the second one looks like a keyword! */
optImports
  :
  | kwIMPORTS importsList chSEMI

  /* error production */
  | kwIMPORTS error chSEMI {
      yyerrok;
      importsErr:
      yyerror("Syntax for IMPORTS is\n  \"IMPORTS\" {<itemList> \"FROM\" <moduleName>}... \";\""); }
  | kwIMPORTS error {
      goto importsErr; }
  ;

importsList
  : imports
  | importsList imports
  ;

imports
  : importNameList kwFROM tokNAME {
      finishImport(pMod, $3.pStr); }

  /* error productions */
  | importNameList error notFrom {
      yyerror("Missing semicolon at end of IMPORTs - terminating");
      YYABORT; }
/* this version causes too many errors */
/*  | importNameList error { */
/*        yyerror("Missing semicolon at end of IMPORTs"); } */
  | importNameList error kwFROM tokNAME {
      yyerror("Problem with IMPORT item list, syntax is\n  <item>[\",\" <item>]... \"FROM\" <moduleName>");
      yyerrok;  
      finishImport(pMod, $4.pStr); }
  | importNameList kwFROM tokNAME chCOMMA {
      yyerror("Extra comma after module name");
      finishImport(pMod, $3.pStr); }
  ;

notFrom
  : tokIS       | kwOBJECT_TYPE | kwTRAP_TYPE   | kwSEQUENCE
  | kwOBJECT    | kwEND         | kwDEFINITIONS | kwBEGIN
  ;

importNameList
  : importName
  | importNameList importComma importName
  ;

importComma
  : chCOMMA
  /* error - missing comma */
  | {
      yyerror("Missing comma between imported items"); }
  ;
  
importName
  : tokNAME {
      newIMPI(pMod->ut.mod.pImodTL, $1.pStr); }
  ;

/* the definitions in a module */
optDefinitionList
  :
  | definitionList
  ;

definitionList
  : definition
  | definitionList definition

  /* error production */
  | kwIMPORTS {
      yyerror("IMPORTS must be at the very start of a module - terminating");
      YYABORT; }
  ;

definition
  : SMIdefinition
  | OIDdefinition
  | OTdefinition
  | SEQdefinition
  | TTdefinition
  | TCdefinition

  /* error production */
  | DIRstart {
      yyerror("Compiler directives not allowed in module definition - terminating");
      YYABORT; }
  | tokNAME mostAnyExceptDefinition {
      yyerror("Unrecognized definition type - item skipped"); }
  ;

DIRstart
  : dirADDOPT   | dirALIASMODULE | dirALIASSYMBOL
  | dirHELP     | dirINCLUDE     | dirPOPOPT
  | dirPRINTOPT | dirPUSHOPT     | dirREMOVEOPT
  ;

mostAnyExceptDefinition
  /* all tokens other than definitions, SMI, directives,
   * SEQUENCE, tokNAME, IMPORTS, or END */
  : kwACCESS    | kwBEGIN       | kwCOUNTER     | kwDEFINITIONS | kwDEFVAL
  | kwDEPRECATED | kwDESCRIPTION | kwENTERPRISE  | kwFROM
  | kwGAUGE     | kwIDENTIFIER  | kwIMPLIED     | kwINDEX       | kwINTEGER
  | kwIPADDRESS | kwMANDATORY   | kwNETWORKADDRESS | kwNOLENGTH
  | kwNOT_ACCESSIBLE | kwNULL   | 
  | kwOBSOLETE  | kwOCTET       | kwOF          | kwOPAQUE      | kwOPTIONAL
  | kwREAD_ONLY | kwREAD_WRITE  | kwREFERENCE
  | kwSIZE      | kwSTATUS      | kwSTRING      | kwSYNTAX
  | kwTIMETICKS | kwVARIABLES   | kwWRITE_ONLY
  | chLCB       | chRCB         | chLPR         | chRPR         | chSEMI
  | chCOMMA     | chMINUS       | chDOT         | chOTHER       | tokDOTDOT
  | tokBADDOTDOT
  | tokBADSTR   | tokSTRING     | tokNUMBER
  | tokBSTR     | tokHSTR       | tokIP
  ;


SMIdefinition
  : kwSMI tokNAME {
      addSMIname($2.pStr, pMod); }
  /* error production */
  | kwSMI error {
      yyerror("Name of SMI item expected");
      /* yyerrok; */ }
  ;

OIDdefinition
  : tokNAME kwOBJECT kwIDENTIFIER OIDis chLCB tokNAME tokNUMBER chRCB {
      addOIDname($1.pStr, $6.pStr, $7.ul, pMod); }
  | tokNAME kwOBJECT kwIDENTIFIER OIDis chLCB tokNUMBER chRCB {
      addOIDname($1.pStr, NULL, $6.ul, pMod); }

  /* error productions */
  | tokNAME kwOBJECT error chRCB {
      OIDerr:
      yyerror("Syntax for OID is\n  <oidName> \"OBJECT\" \"IDENTIFIER\" \"::=\" <oidValid>");
      yyerrok; }
  | tokNAME kwIDENTIFIER error chRCB {
      goto OIDerr; }
  | tokNAME kwOBJECT kwIDENTIFIER OIDis chLCB error chRCB {
      yyerror("Syntax for <oidValue> is\n  { <OIDname> <number> } | <number>");
      yyerrok; }
  ;

OIDis
  : tokIS
  /* error production */
  | tokBADIS {
      yyerror("Must use \"::=\""); }
  ;


OTdefinition
  : OTdef kwOBJECT_TYPE
        OTsyntax
        OTaccess
        OTstatus
        optDesc
        optRefer
        optIndex
        optDefval
        OTis chLCB tokNAME tokNUMBER chRCB {
          if ($1.pSym != NULL)
              finishOT($1.pSym, &syn, (USHORT)($4.ul), (USHORT)($5.ul),
                       $6.pStr, $7.pStr, $12.pStr, $13.ul); }
  /* error productions */
  | OTdef kwOBJECT_TYPE
        OTsyntax
        OTaccess
        OTstatus
        optDesc
        optRefer
        optIndex
        optDefval
        OTis chLCB error chRCB {
      yyerror("Syntax for <oidValue> is  <oidName> <number>");
      yyerrok; }
  | OTdef kwOBJECT_TYPE
        error
        OTis chLCB tokNAME tokNUMBER chRCB {
      yyerror(
"Syntax is\n  <objName> \"OBJECT-TYPE\" <syntax> <access> <status> <optDesc> <optRefer> <optIndex> <optDefval> \"::=\" <oidVal>");
      yyerrok; }
  ;

OTsyntax
  : kwSYNTAX objectSyntax
  /* error productions */
  | {
      yyerror("Missing SYNTAX clause"); }
  | kwSYNTAX error {
      yyerror("Problem with SYNTAX clause");
      /* yyerrok;*/ }
  ;

OTaccess
  : kwACCESS objectAccess {
      $$ = $2; }
  /* error productions */
/*  | { */
/*      yyerror("Missing ACCESS clause"); } */
  | kwACCESS error {
      yyerror("Problem with ACCESS");
      /* yyerrok;*/ }
  ;

OTstatus
  : kwSTATUS objectStatus {
      $$ = $2; }
  /* error productions */
/*  | { */
/*      yyerror("Missing STATUS clause"); } */
  | kwSTATUS error {
      yyerror("Problem with STATUS");
      /*yyerrok;*/ }
  ;

OTis
  : tokIS
  /* error production */
  | tokBADIS {
      yyerror("Must use \"::=\""); }
  ;


OTdef
  : tokNAME {
      syn.usSyntax = MIBSYNbad; /* assume BAD syntax will be specified */
      pOid = addOTname($1.pStr, pMod);
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.pSym = pOid; }
  ;

SEQdefinition
  : SEQdef kwSEQUENCE chLCB seqMemberList chRCB {
      if ($1.pSym != NULL)
          pSeq = finishSEQ($1.pSym); }
  /* error productions */
  | SEQdef kwSEQUENCE chLCB seqMemberList chCOMMA chRCB {
      yyerror("Extra comma at end of sequence item list");
      if ($1.pSym != NULL)
          pSeq = finishSEQ($1.pSym); }
  | SEQdef kwSEQUENCE error chRCB {
      yyerror("Problem with sequence item list, syntax is\n  <seqName> \"::=\" \"SEQUENCE\" \"{\" <seqItem> [\",\"<seqItem>]... \"}\"");
      yyerrok;
      if ($1.pSym != NULL)
          pSeq = finishSEQ($1.pSym); }
  ;

SEQdef
  : tokNAME SEQis {
      pSeq = addSEQname($1.pStr, pMod);
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.pSym = pSeq; }
  ;

SEQis
  : tokIS
  /* error production */
  | tokBADIS {
      yyerror("Must use \"::=\""); }
  | {
      yyerror("Syntax is\n  <seqName> \"::=\" \"SEQUENCE\" \"{\" <seqItems> \"}\""); }
  ;

TCdefinition
  : TCdef simpleAndEnum {
      if ($1.pSym != NULL)   
          pTc = finishTC($1.pSym, &syn); }
  /* error production */
  | TCdef error {
      yyerror("Problem with textual convention, syntax is\n  <tcName> \"::=\" <simpleSyntax>");
      /*yyerrok;*/ } 
  ;

TCdef
  : tokNAME TCis {
      pTc = addTCname($1.pStr, pMod);
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.pSym = pTc; }
  ;

TCis
  : tokIS
  /* error production */
  | tokBADIS {
      yyerror("Must use \"::=\""); }
  ;

TTdefinition
  : TTdef kwTRAP_TYPE
        TTenterprise
        optVar
        optDesc
        optRefer
        TTis tokNUMBER {
          if ($1.pSym != NULL)
              pTr = finishTR($1.pSym, $3.pStr, $5.pStr, $6.pStr, $8.ul); }
  /* error productions */
  | TTdef kwTRAP_TYPE
        error
        TTis tokNUMBER {
      yyerror(
"Syntax is\n  <trapName> \"TRAP-TYPE\" <enterprise> <optVars> <optDesc> <optRefer> \"::=\" <value>");
      yyerrok; }
  ;

TTdef
  : tokNAME {
      pTr = addTRname($1.pStr, pMod);
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.pSym = pTr; }
  ;

TTis
  : tokIS
  /* error production */
  | tokBADIS {
      yyerror("Must use \"::=\""); }
  ;

TTenterprise
  : kwENTERPRISE tokNAME {
      $$ = $2; }
  /* error productions */
  | kwENTERPRISE error {
      yyerror("Syntax is \"ENTERPRISE\" <oidName>");
      /*yyerrok;*/ }
  | {
      yyerror("Missing ENTERPRISE clause"); }
  ;



/********************************************************************/
/*
*  Supporting productions
*
*/

/* description */
optDesc
  : {
      descAction:
      $$.pszFn = NULL;
      $$.usLineNo = 0;
      $$.usColNo = 0;
      $$.pStr = NULL; }
  | kwDESCRIPTION tokSTRING {
      $$ = $2; }
  /* error production */
  | kwDESCRIPTION error {
      yyerror("Syntax is  \"DESCRIPTION\" <quotedString>");
      goto descAction; }
  ;

/* reference */
optRefer
  : {
      refAction:
      $$.pszFn = NULL;
      $$.usLineNo = 0;
      $$.usColNo = 0;
      $$.pStr = NULL; }
  | kwREFERENCE tokSTRING {
      $$ = $2; }
  /* error production */
  | kwREFERENCE error {
      yyerror("Syntax is  \"REFERENCE\" <quotedString>");
      goto refAction; }
  ;


/* supporting productions used for Object-type template */
objectSyntax
  : simpleAndEnum
  | kwSEQUENCE kwOF tokNAME {
      checkSyntax(&syn, $3.pStr, MIBSYNseqOf, MIBSRno, 0L, 0L, pMod); }
  /* error production */
  | kwSEQUENCE kwOF tokNAME chLPR kwSIZE chLPR srVal chRPR chRPR {
      checkSyntax(&syn, $3.pStr, MIBSYNseqOf, MIBSRno, 0L, 0L, pMod); 
      objSyn:
      yyerror("A sequence name can not be qualified with a size or range");}
  | kwSEQUENCE kwOF tokNAME chLPR kwSIZE chLPR srVal SZdotdot srVal chRPR chRPR {
      checkSyntax(&syn, $3.pStr, MIBSYNseqOf, MIBSRno, 0L, 0L, pMod);
      goto objSyn;}
  | kwSEQUENCE error {
      yyerror("Syntax is \"SEQUENCE\" \"OF\" <sequenceName>");
      /*yyerrok;*/ }
  ;

simpleAndEnum
  : simpleSyntax
  | enumHeader enumList chRCB
  /* error productions */
  | enumHeader enumList chCOMMA chRCB {
      yyerror("Extra comma at end of enumerated value list"); }
  | enumHeader error chRCB {
      yyerrok;
      enumErr:
      yyerror("Problem with enumerated list, syntax is\n  <enumName>\"(\"<number>\")\" [\",\" <enumName>\"(\"<number>\")\"]..."); }
  | enumHeader error {
      goto enumErr; }
  ;

enumHeader
  : kwINTEGER chLCB {
      checkSyntax(&syn, NULL, MIBSYNenum, MIBSRno, 0L, 0L, pMod); }
  ;

enumList
  : enum
  | enumList enumComma enum
  ;

enumComma
  : chCOMMA
  /* error production */
  | {
      yyerror("Missing comma between enumerated items"); }
  ;

enum
  : tokNAME chLPR tokNUMBER chRPR  {
      if ($3.ul == 0L) {
        yywarning("Value for enumerated item \"%s\" should not be zero",
                ($1.pStr)->pszVal);
      }
      addENUMitem($1.pStr, $3.ul, &syn); }
  /* error productions */
  | tokNAME chLPR chMINUS tokNUMBER chRPR {
      yyerror("Negative numbers are not valid enumerated values"); }
  | tokNAME chLPR error chRPR  {
      yyerror("Enumerated value must be a number");
      yyerrok; }
  | tokNAME error {
      yyerror("Problem with enumerated value, syntax is\n  <enumName>\"(\"<number>\")\"");
      /* yyerrok;*/ }
  ;

objectAccess
  : kwREAD_ONLY {
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.ul = MIBACCro; }
  | kwREAD_WRITE {
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.ul = MIBACCrw; }
  | kwWRITE_ONLY {
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.ul = MIBACCwo; }
  | kwNOT_ACCESSIBLE {
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.ul = MIBACCna; }
  | tokNAME {
      yyerror("Unknown ACCESS type \"%s\"", $1.pStr->pszVal);
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.ul = MIBACCbad; } 
  ;

objectStatus
  : kwOPTIONAL {
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.ul = MIBSTAop; }
  | kwMANDATORY {
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.ul = MIBSTAma; }
  | kwDEPRECATED {
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.ul = MIBSTAde; }
  | kwOBSOLETE {
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.ul = MIBSTAob; }
  | tokNAME {
      yyerror("Unknown STATUS type \"%s\"", $1.pStr->pszVal);
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.ul = MIBSTAbad; }
  ;

optIndex
  :
  | kwINDEX chLCB indexItemList chRCB
  /* error production */
  | kwINDEX chLCB indexItemList chCOMMA chRCB {
      yyerror("Extra comma at end of index item list"); }
  | kwINDEX error chRCB {
      yyerrok;
      indexErr:
      yyerror("Problem with index item list, syntax is\n  \"INDEX\" \"{\" <indexItem> [\",\"<indexItem>]... \"}\""); }
  | kwINDEX error {
      goto indexErr; }
  ;

indexItemList
  : indexItem
  | indexItemList indexComma indexItem
  ;

indexComma
  : chCOMMA
  /* error production */
  | {
      yyerror("Missing comma between index items"); }
  ;


indexItem
  : tokNAME {
      if (pOid != NULL)
          addINDXitem(pOid, MIBITobj, $1.pStr, 0L, 0L, pMod); }
  | noLenOrImpl tokNAME {
      if (pOid != NULL)
          addINDXitem(pOid, MIBITnlobj, $2.pStr, 0L, 0L, pMod); }
  | kwINTEGER {
      if (pOid != NULL)
          addINDXitem(pOid, MIBITint, NULL, 0L, 0xFFFFFFFF, pMod); }
  | kwINTEGER chLPR srVal INDXintdotdot srVal chRPR{
      if (pOid != NULL)
          addINDXitem(pOid, MIBITint, NULL, $3.ul, $5.ul, pMod); }
  | kwOCTET kwSTRING {
      if (pOid != NULL)
          addINDXitem(pOid, MIBIToct, NULL, 0L, 0L, pMod); }
  | kwOCTET kwSTRING chLPR kwSIZE chLPR srVal chRPR chRPR {
      if (pOid != NULL)
          addINDXitem(pOid, MIBITfloct, NULL, $6.ul, 0L, pMod); }
  | kwOCTET kwSTRING chLPR kwSIZE chLPR srVal INDXoctdotdot srVal chRPR chRPR {
      if (pOid != NULL)
          addINDXitem(pOid, MIBIToct, NULL, $6.ul, $8.ul, pMod); }
  | noLenOrImpl kwOCTET kwSTRING {
      if (pOid != NULL)
          addINDXitem(pOid, MIBITnloct, NULL, 0L, 0L, pMod); }
  | noLenOrImpl kwOCTET kwSTRING chLPR kwSIZE chLPR srVal INDXoctdotdot srVal chRPR chRPR {
      if (pOid != NULL)
          addINDXitem(pOid, MIBITnloct, NULL, $7.ul, $9.ul, pMod); }
  | kwOBJECT kwIDENTIFIER {
      if (pOid != NULL)
          addINDXitem(pOid, MIBIToid, NULL, 0L, 0L, pMod); }
  | noLenOrImpl kwOBJECT kwIDENTIFIER {
      if (pOid != NULL)
          addINDXitem(pOid, MIBITnloid, NULL, 0L, 0L, pMod); }
  | kwNETWORKADDRESS {
      if (pOid != NULL)
          addINDXitem(pOid, MIBITnaddr, NULL, 0L, 0L, pMod); }
  | kwIPADDRESS {
      if (pOid != NULL)
          addINDXitem(pOid, MIBITipaddr, NULL, 0L, 0L, pMod); }
  /* error production */
  | noLenOrImpl kwOCTET kwSTRING chLPR kwSIZE chLPR srVal chRPR chRPR {
      yyerror("\"NOLENGTH\" qualifier is only valid with varying length OCTET STRINGs");
      if (pOid != NULL)
          addINDXitem(pOid, MIBITfloct, NULL, $7.ul, 0L, pMod); }
  ;

noLenOrImpl
  : kwNOLENGTH
  | kwIMPLIED
  ;

INDXintdotdot
  : tokDOTDOT
  /* error productions */
  | {
      yyerror("Missing \"..\""); }
  | tokBADDOTDOT {
      indxIntDot:
      yyerror("Must use \"..\""); }
  | chDOT {
      goto indxIntDot; }
  ;

INDXoctdotdot
  : tokDOTDOT
  /* error productions */
  | {
      yyerror("Missing \"..\""); }
  | tokBADDOTDOT {
      indxOctDot:  
      yyerror("Must use \"..\""); }
  | chDOT {
      goto indxOctDot; }
  ;


optDefval
  :
  | defValHead defValMiddle chRCB
  /* error productions */
  | defValHead error chRCB {
      defValErr:
      fNoStr = fNoStrSav;
      yyerror("Problem with value for DEFVAL clause"); }
  | defValHead error {
      goto defValErr; }
  | kwDEFVAL error {
      yyerror("Problem with DEFVAL clause, syntax is\n  \"DEFVAL\" \"{\" <value> \"}\""); }
  ;


defValHead
  : kwDEFVAL {
      fNoStrSav = fNoStr;
      fNoStr = FALSE; }
  ;

defValMiddle
  : chLCB defval {
      fNoStr = fNoStrSav; }
  ;

defval
  : tokSTRING {
      if (pOid != NULL)
          addDV(pOid, MIBDFVstr, $1.pStr, 0L, NULL); }
  | tokBSTR {
      if (pOid != NULL)
          addDV(pOid, MIBDFVbstr, $1.pStr, 0L, NULL); }
  | tokHSTR {
      if (pOid != NULL)
          addDV(pOid, MIBDFVhstr, $1.pStr, 0L, NULL); }
  | tokNUMBER {
      if (pOid != NULL)
          addDV(pOid, MIBDFVint, NULL, $1.ul, NULL); }
  | chMINUS tokNUMBER {
      if (pOid != NULL)
          addDV(pOid, MIBDFVneg, NULL, $2.ul, NULL); }
  | tokNAME {
      if (pOid != NULL)
          addDV(pOid, MIBDFVna, $1.pStr, 0L, NULL); }
  | chLCB oidValue chRCB {
      if ((pOid != NULL) && ($2.pSym != NULL))
              addDV(pOid, MIBDFVoid, NULL, 0L, $2.pSym); }
  | tokIP {
      if (pOid != NULL)
          addDV(pOid, MIBDFVip, NULL, $1.ul, NULL); }
  ;


oidValue
  : tokNAME {
      pComp = checkOIDcomp(NULL, $1.pStr, (ULONG)-1L, pMod);
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.pSym = pComp; }
  | tokNAME chLPR tokNUMBER chRPR {
      pComp = checkOIDcomp(NULL, $1.pStr, $3.ul, pMod);
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.pSym = pComp; }
  | tokNUMBER {
      pComp = checkOIDcomp(&OidRoot, NULL, $1.ul, pMod);
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.pSym = pComp; }
  | oidValue tokNAME {
      pComp = checkOIDcomp($1.pSym, $2.pStr, (ULONG)-1L, pMod);
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.pSym = pComp; }
  | oidValue tokNAME chLPR tokNUMBER chRPR {
      pComp = checkOIDcomp($1.pSym, $2.pStr, $4.ul, pMod);
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.pSym = pComp; }
  | oidValue tokNUMBER {
      pComp = checkOIDcomp($1.pSym, NULL, $2.ul, pMod);
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.pSym = pComp; }
  ;


/* support productions for object-type and sequence definitions */
simpleSyntax
  : tokNAME {
      checkSyntax(&syn, $1.pStr, MIBSYNtc, MIBSRno, 0L, 0L, pMod); }

  | tokNAME chLPR kwSIZE chLPR srVal SZdotdot srVal chRPR chRPR {
      checkSyntax(&syn, $1.pStr, MIBSYNtc, MIBSRvs, $5.ul, $7.ul, pMod); }

  | tokNAME chLPR kwSIZE chLPR srVal chRPR chRPR {
      checkSyntax(&syn, $1.pStr, MIBSYNtc, MIBSRfs, $5.ul, $5.ul, pMod); }

  | tokNAME chLPR srVal RNGdotdot srVal chRPR {
      checkSyntax(&syn, $1.pStr, MIBSYNtc, MIBSRpp, $3.ul, $5.ul, pMod); }

  | tokNAME chLPR chMINUS srVal RNGdotdot srVal chRPR {
      checkSyntax(&syn, $1.pStr, MIBSYNtc, MIBSRnp, $4.ul, $6.ul, pMod); }

  | tokNAME chLPR chMINUS srVal RNGdotdot chMINUS srVal chRPR {
      checkSyntax(&syn, $1.pStr, MIBSYNtc, MIBSRnn, $4.ul, $7.ul, pMod); }

  | synType {
      checkSyntax(&syn, NULL, $1.us, MIBSRno, 0L, 0L, pMod); }

  | synType chLPR kwSIZE chLPR srVal SZdotdot srVal chRPR chRPR {
      checkSyntax(&syn, NULL, $1.us, MIBSRvs, $5.ul, $7.ul, pMod); }

  | synType chLPR kwSIZE chLPR srVal chRPR chRPR {
      checkSyntax(&syn, NULL, $1.us, MIBSRfs, $5.ul, $5.ul, pMod); }

  | synType chLPR srVal RNGdotdot srVal chRPR {
      checkSyntax(&syn, NULL, $1.us, MIBSRpp, $3.ul, $5.ul, pMod); }

  | synType chLPR chMINUS srVal RNGdotdot srVal chRPR {
      checkSyntax(&syn, NULL, $1.us, MIBSRnp, $4.ul, $6.ul, pMod); }

  | synType chLPR chMINUS srVal RNGdotdot chMINUS srVal chRPR {
      checkSyntax(&syn, NULL, $1.us, MIBSRnn, $4.ul, $7.ul, pMod); }

  /* error productions */
  | synType chLPR kwSIZE error {
      yyerror("Problem with size clause, syntax is\n  <syntaxType> \"(\" \"SIZE\" \"(\" <small> [\"..\" <large>] \")\" \")\"");
      /* yyerrok; */
      checkSyntax(&syn, NULL, $1.us, MIBSRno, 0L, 0L, pMod); }
  | synType kwSIZE chLPR srVal SZdotdot srVal chRPR {
      yyerror("Problem with size clause, syntax is\n  <syntaxType> \"(\" \"SIZE\" \"(\" <small> [\"..\" <large>] \")\" \")\"");
      checkSyntax(&syn, NULL, $1.us, MIBSRvs, $4.ul, $6.ul, pMod); }
  | synType kwSIZE chLPR srVal chRPR {
      yyerror("Problem with size clause, syntax is\n  <syntaxType> \"(\" \"SIZE\" \"(\" <small> [\"..\" <large>] \")\" \")\"");
      checkSyntax(&syn, NULL, $1.us, MIBSRfs, $4.ul, $4.ul, pMod); }
  | synType kwSIZE error {
      yyerror("Problem with size clause, syntax is\n  <syntaxType> \"(\" \"SIZE\" \"(\" <small> [\"..\" <large>] \")\" \")\"");
      /* yyerrok; */
      checkSyntax(&syn, NULL, $1.us, MIBSRno, 0L, 0L, pMod); }
  | synType chLPR chMINUS error {
      yyerror("Problem with range clause, syntax is\n  <syntaxType> \"(\" <low> \"..\" <high> \")\"");
      /* yyerrok; */
      checkSyntax(&syn, NULL, $1.us, MIBSRno, 0L, 0L, pMod); }
  | synType chLPR srVal chRPR {
      yyerror("Problem with range clause, syntax is\n  <syntaxType> \"(\" <low> \"..\" <high> \")\"");
      checkSyntax(&syn, NULL, $1.us, MIBSRno, 0L, 0L, pMod); }
  | synType chLPR srVal error {
      yyerror("Problem with range clause, syntax is\n  <syntaxType> \"(\" <low> \"..\" <high> \")\"");
      /* yyerrok; */
      checkSyntax(&syn, NULL, $1.us, MIBSRno, 0L, 0L, pMod); }
  | synType srVal error {
      yyerror("Problem with range clause, syntax is\n  <syntaxType> \"(\" <low> \"..\" <high> \")\"");
      /* yyerrok; */
      checkSyntax(&syn, NULL, $1.us, MIBSRno, 0L, 0L, pMod); }

  | tokNAME chLPR kwSIZE error {
      yyerror("Problem with size clause, syntax is\n  <syntaxType> \"(\" \"SIZE\" \"(\" <small> [\"..\" <large>] \")\" \")\"");
      /* yyerrok; */
      checkSyntax(&syn, $1.pStr, MIBSYNtc, MIBSRno, 0L, 0L, pMod); }
  | tokNAME kwSIZE chLPR srVal SZdotdot srVal chRPR {
      yyerror("Problem with size clause, syntax is\n  <syntaxType> \"(\" \"SIZE\" \"(\" <small> [\"..\" <large>] \")\" \")\"");
      checkSyntax(&syn, $1.pStr, MIBSYNtc, MIBSRvs, $4.ul, $6.ul, pMod); }
  | tokNAME kwSIZE chLPR srVal chRPR {
      yyerror("Problem with size clause, syntax is\n  <syntaxType> \"(\" \"SIZE\" \"(\" <small> [\"..\" <large>] \")\" \")\"");
      checkSyntax(&syn, $1.pStr, MIBSYNtc, MIBSRfs, $4.ul, $4.ul, pMod); }
  | tokNAME kwSIZE error {
      yyerror("Problem with size clause, syntax is\n  <syntaxType> \"(\" \"SIZE\" \"(\" <small> [\"..\" <large>] \")\" \")\"");
      /* yyerrok; */
      checkSyntax(&syn, $1.pStr, MIBSYNtc, MIBSRno, 0L, 0L, pMod); }
  | tokNAME chLPR chMINUS error {
      yyerror("Problem with range clause, syntax is\n  <syntaxType> \"(\" <low> \"..\" <high> \")\"");
      /* yyerrok; */
      checkSyntax(&syn, $1.pStr, MIBSYNtc, MIBSRno, 0L, 0L, pMod); }
  | tokNAME chLPR srVal chRPR {
      yyerror("Problem with range clause, syntax is\n  <syntaxType> \"(\" <low> \"..\" <high> \")\"");
      checkSyntax(&syn, $1.pStr, MIBSYNtc, MIBSRno, 0L, 0L, pMod); }
  | tokNAME chLPR srVal error {
      yyerror("Problem with range clause, syntax is\n  <syntaxType> \"(\" <low> \"..\" <high> \")\"");
      /* yyerrok; */
      checkSyntax(&syn, $1.pStr, MIBSYNtc, MIBSRno, 0L, 0L, pMod); }
  | tokNAME srVal error {
      yyerror("Problem with range clause, syntax is\n  <syntaxType> \"(\" <low> \"..\" <high> \")\"");
      /* yyerrok; */
      checkSyntax(&syn, $1.pStr, MIBSYNtc, MIBSRno, 0L, 0L, pMod); }
  ;

/* well known syntax types */
synType
  : kwINTEGER {
      $$ = $1;
      $$.us = MIBSYNint; }
  | kwOCTET kwSTRING {
      $$ = $1;
      $$.us = MIBSYNoctstr; }
  | kwOBJECT kwIDENTIFIER {
      $$ = $1;
      $$.us = MIBSYNoid; }
  | kwNULL {
      $$ = $1;
      $$.us = MIBSYNnull; }
  | kwNETWORKADDRESS {
      $$ = $1;
      $$.us = MIBSYNnaddr; }
  | kwIPADDRESS {
      $$ = $1;
      $$.us = MIBSYNipaddr; }
  | kwCOUNTER {
      $$ = $1;
      $$.us = MIBSYNcntr; }
  | kwGAUGE {
      $$ = $1;
      $$.us = MIBSYNgauge; }
  | kwTIMETICKS {
      $$ = $1;
      $$.us = MIBSYNticks; }
  | kwOPAQUE {
      $$ = $1;
      $$.us = MIBSYNopaque; }
  /* error productions */
  | kwOCTET {
      $$ = $1;
      $$.us = MIBSYNoctstr;
      synOctErr:
      yyerror("Syntax name is \"OCTET\" \"STRING\""); }
  | kwSTRING {
      $$ = $1;
      $$.us = MIBSYNoctstr;
      goto synOctErr; }
  | kwOBJECT {
      $$ = $1;
      $$.us = MIBSYNoid;
      synOIDerr:
      yyerror("Syntax name is \"OBJECT\" \"IDENTIFER\""); }
  | kwIDENTIFIER {
      $$ = $1;
      $$.us = MIBSYNoid;
      goto synOIDerr; }
  ;


/* ".." for ranges */
RNGdotdot
  : tokDOTDOT
  /* error productions */
  | {
      yyerror("Missing \"..\""); }
  | tokBADDOTDOT {
      yyerror("Must use \"..\""); }
  | chDOT {
      yyerror("Must use \"..\""); }
  ;

/* ".." for sizes */
SZdotdot
  : tokDOTDOT
  /* error productions */
  | {
      yyerror("Missing \"..\""); }
  | tokBADDOTDOT {
      yyerror("Must use \"..\""); }
  | chDOT {
      yyerror("Must use \"..\""); }
  ;

/* numeric value for size or range */
srVal
  : tokNUMBER
  | tokBSTR {
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.ul = bstrToVal($1.pStr); }
  | tokHSTR {
      $$.pszFn = $1.pszFn;
      $$.usLineNo = $1.usLineNo;
      $$.usColNo = $1.usColNo;
      $$.ul = hstrToVal($1.pStr); }
  ;

/* supporting productions for defined sequence template */
seqMemberList
  : seqMember  
  | seqMemberList seqComma seqMember
  ;

seqComma
  : chCOMMA
  /* error production */
  | {
      yyerror("Missing comma between sequence items"); }
  ;


seqMember
  : tokNAME simpleSyntax {
      if (pSeq != NULL)
          addSEQitem(pSeq, $1.pStr, &syn, pMod); }
  /* error productions */
  | tokNAME {yyerror("Extra comma");} chCOMMA simpleSyntax {
      if (pSeq != NULL)
          addSEQitem(pSeq, $1.pStr, &syn, pMod); }
  | tokNAME error {
      yyerror("Problem with sequence item syntax");
      /* yyerrok; */ }
  ;


/* supporting productions for traps */
optVar
  :
  | kwVARIABLES chLCB varList chRCB
  /* error production */
  | kwVARIABLES chLCB varList chCOMMA chRCB {
      yyerror("Extra comma at end of trap variable list"); }
  | kwVARIABLES error chRCB {
      yyerrok;
      varErr:
      yyerror("Syntax is\n  \"VARIABLES\" \"{\" <variable> [\",\"<variable>]... \"}\""); }
  | kwVARIABLES error {
      goto varErr; }
  ;

varList
  : var
  | varList varComma var
  ;

varComma
  : chCOMMA
  /* error production */
  | {
      yyerror("Missing comma between trap variables"); }
  ;

var
  : tokNAME {
      if (pTr != NULL)
          addVAR(pTr, $1.pStr, pMod); }
  ;


/* end of file: SMIB.Y */
