/*
 * Copyright 1992 SynOptics Communications, Inc.  All Rights Reserved.
 * SynOptics grants a non-exclusive license to use, copy, modify, and
 * distribute this software for any purpose and without fee, provided
 * that this copyright notice and license appear on all copies and
 * supporting documentation.
 * SynOptics makes no representations about the suitability of this
 * software for any particular purpose.  The software is supplied
 * "AS IS", and SynOptics makes no warranty, either express or implied,
 * as to the use, operation, condition, or performance of the software.
 * SynOptics retains all title and ownership in the software.
 *
 * file: SMMOD.C - Module (and SMI) functions
 *
 * $Revision:   1.1  $ $Date:   19 Jun 1992 16:11:18  $
 * $Log:   R:/MIBTOOLS/V1.0/SMIC/SRC/SMMOD.C_V  $
 * 
 *    Rev 1.1   19 Jun 1992 16:11:18   gfoster
 * Copyright text was reformated.
 * 
 * Fixed bug that caused imported items to be reported in
 * the next imported module if the module containing the
 * items didn't exist.
 * 
 *    Rev 1.0   27 May 1992 15:59:24   gfoster
 * Initial revision.
 *
*/

#include <stdio.h>

#ifdef MS_DOS
#include <stdlib.h>
#endif /* MS_DOS */

#include <ctype.h>

#include "tds.h"
#include "smscdefs.h"
#include "smstdefs.h"
#include "smsydefs.h"
#include "smic.h"


/** addMODname - add module name
*
* call with:
*   pNa - name of module
*
* returns:
*   ptr to module name or NULL if error
*/
    MIBSYM *
#ifdef __STDC__
addMODname(STRTAB *pNa)
#else
addMODname(pNa)
    STRTAB *pNa;
#endif /* __STDC__ */
{
    MIBSYM *pSym;
    MIBSYM *pMod;


    /* check if module already defined */
    for (pSym = pNa->pSym; pSym != NULL; pSym = pSym->pSym) {
        if ((pSym->usType & MIBSYMmask) == MIBSYMmod) {
            yyerror("duplicate definition of module \"%s\"",
                    pNa->pszVal);
#ifdef OLD
            yyterm();
            return(NULL);
#endif
            /*?? note: maybe a new nonduplicate name should be used */
            /*?? continueing here will cause the dup module name
                 to be "hidden", but maybe this is OK */
            break;
        }
    }

    /* check case of name */
    if (!isupper(*(pNa->pszVal))) {
        yywarning("Name of Module \"%s\" must start with uppercase letter",
                pNa->pszVal);
    }

    /* allocate module */
    pMod = newMOD(pNa);

    if (pMod != NULL) {
        /* assume has imports, so add import list */
        newMODREF(pMod);
    }
    return(pMod);

} /* addMODname */


/** finishMODid - finished processing of ID for module
*
*/
    VOID
#ifdef __STDC__
finishMODid(VOID)
#else
finishMODid()
#endif /* __STDC__ */
{
    yywarning("Extraneous OID definition for module ignored");

} /* finishMODid */


/** finishImport - check the names of the import now that the
*                  module name is given
*
* call with:
*   pMod - module containing imports
*   pNa - name of module on IMPORTs clause
*
* returns:
*   ptr to module or NULL if error
*/
    MIBSYM *
#ifdef __STDC__
finishImport(MIBSYM *pMod, STRTAB *pNa)
#else
finishImport(pMod, pNa)
    MIBSYM *pMod;
    STRTAB *pNa;
#endif /* __STDC__ */
{
    MIBSYM *pSym;
    MIBSYM *pImpMod;
    MIBMODREF *pModRef;
    MIBIMPI *pImpi;
    MIBSYM *pImp;
    USHORT usType;
    BOOL fDupImport;
    BOOL fCkItems = TRUE;


    /* check if module exists */
    for (pSym = pNa->pSym; pSym != NULL; pSym = pSym->pSym) {
        if ((usType = (pSym->usType & MIBSYMmask)) == MIBSYMmod)
            break;
        if ((usType == MIBSYMalias) &&
                (((pSym->ut.al.pAlSym)->usType & MIBSYMmask) == MIBSYMmod)) {
            /* alias to module */
            pSym->ut.al.cUse++;
            pSym = pSym->ut.al.pAlSym;
            break;
        }
    }
    if (pSym == NULL) {
        yyerror("module \"%s\" not defined", pNa->pszVal);
#ifdef OLD
        yyterm();
        return(NULL);
#else
        /* allocate one */
        pSym = newMOD(pNa);
        if (pSym == NULL)
            return(NULL);
        fCkItems = FALSE;       /* don't check IMPORTED items */
#endif
    }

    /* check for circular definition */
    if (pSym == pMod) {
        yyerror("module \"%s\" can not be IMPORTed in itself",
                pMod->pszName);
#ifdef OLD
        yyterm();
        return(NULL);
#else
        fCkItems = FALSE;       /* don't check IMPORTED items */
#endif
    }

    pImpMod = pSym;
    pModRef = pMod->ut.mod.pImodTL;
    pModRef->pMod = pImpMod;


    /* check each name in import list to see if item from IMPORT module */
    /* also special checks for certain types */
    if (fCkItems) {
        for (pImpi = pModRef->pImpIL; pImpi != NULL; pImpi = pImpi->pNext) {
            /* see if import symbol in IMPORTed module */
            for (pSym = (pImpi->pNa)->pSym, fDupImport = FALSE; pSym != NULL;
                    pSym = pSym->pSym) {
                if (pSym->pMod == pMod)
                    /* found dupplicate name being imported */
                    fDupImport = TRUE;
                else if (pSym->pMod == pImpMod)
                    /* found module containing item */
                    break;
            }
            /* check if item defined in "FROM" module */
            if (pSym == NULL) {
                yyerror("Item \"%s\" not defined in module \"%s\"",
                        (pImpi->pNa)->pszVal, pImpMod->pszName);
#ifdef OLD
                yyterm();
                return(NULL);
#else
                continue;       /* skip item */
#endif
            }
        
            /* check if another imported item has the same name */
            if (fDupImport) {
                yyerror("Item \"%s\" from module \"%s\" has the same name as object IMPORTed from another module",
                        (pImpi->pNa)->pszVal, pImpMod->pszName);
                continue;
            }

            /* check if item is an alias */
            if ((usType = (pSym->usType & MIBSYMmask)) == MIBSYMalias) {
                /* get real symbol and type */
                pSym->ut.al.cUse++;
                pSym = pSym->ut.al.pAlSym;
                usType = pSym->usType & MIBSYMmask;
            }

            /* check if item imported in original module */
            if (usType == MIBSYMimp) {
                yyerror("Item \"%s\" from module \"%s\" is imported",
                        (pImpi->pNa)->pszVal, pImpMod->pszName);
#ifdef OLD
                yyterm();
                return(NULL);
#else
                continue;       /* skip item */
#endif
            }

            /* check if item is a sequence */
            if (usType == MIBSYMseq) {
                yyerror("Item \"%s\" from module \"%s\" is a sequence",
                        (pImpi->pNa)->pszVal, pImpMod->pszName);
#ifdef OLD
                yyterm();
                return(NULL);
#else
                continue;
#endif
            }

            /* check if SMI item */
            if (usType == MIBSYMsmi) {
                /* make name known in scanner */
                if (!addSMIitem(pSym->pszName)) {
                    yyerror("finishImport: Internal error with SMI name \"%s\"",
                            pSym->pszName);
                    yyterm();
                    return(NULL);
                }
            }


            /* add import object */
            pImp = newIMP(pMod, pSym, pImpi->pNa);
        }
    }

    /* allocate new module ref */
    newMODREF(pMod);

    return(pMod);

} /* finishImport */


/** fixupImport - return extra import module reference
*
* call with:
*   pMod - module
*
* returns:
*   ptr to module
*/
    MIBSYM *
#ifdef __STDC__
fixupImport(MIBSYM *pMod)
#else
fixupImport(pMod)
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBMODREF *pModRef;
    MIBMODREF *pT;


    /* remove last module reference */
    pModRef = pMod->ut.mod.pImodTL;
    if (pMod->ut.mod.pImodTL == pMod->ut.mod.pImodHL) {
        /* only one item on list */
        pMod->ut.mod.pImodTL = (pMod->ut.mod.pImodHL = NULL);
    } else {
        /* find item before last one */
        for (pT = pMod->ut.mod.pImodHL; pT->pNext != pModRef; pT = pT->pNext) {
        }
        pT->pNext = NULL;
        pMod->ut.mod.pImodTL = pT;
    }
    pModRef->pNext = pModRefAvail;
    pModRefAvail = pModRef;
    cModRefUsed--;

    return(pMod);

} /* fixupImport */


/** checkIMPs - check reference counts for imports in module
*
* NOTE: must first get use counts for SMI imported object
*
* call with:
*   pMod - module
*/
    VOID
#ifdef __STDC__
checkIMPs(MIBSYM *pMod)
#else
checkIMPs(pMod)
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pT;
    MIBSYM *pSym;


    for (pT = pImpGHL; pT != NULL; pT = pT->pNext) {
        if (pT->pMod != pMod)
            continue;

        pSym = pT->ut.imp.pImpSym;
        if (pSym->usType == MIBSYMsmi) {
            /* get use count for SMI item */
            pT->ut.imp.cUse = getKWuse(pSym->ut.smi.iKW);
        }

        if (pT->ut.imp.cUse == 0) {
            if (!fNoUnUsedWarn)
                yywarning("\"%s\" imported but not used",
                        pT->pszName);
        }
    }
} /* checkIMPs */


/** addSMIname - add an SMI name
*
* call with:
*   pNa - name to add
*   pMod - module containing name
*
* returns:
*   ptr to symbol or NULL for error
*/
    MIBSYM *
#ifdef __STDC__
addSMIname(STRTAB *pNa, MIBSYM *pMod)
#else
addSMIname(pNa, pMod)
    STRTAB *pNa;
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pSym;
    USHORT usStatus;
    USHORT iKW;


    usStatus = checkSMIname(pNa->pszVal, &iKW);
    switch(usStatus) {
    case SMISTnsmi:
        yyerror("Item \"%s\" is not defined in SMI",
                pNa->pszVal);
        break;

    case SMISTad:
        yyerror("Item \"%s\" is already defined",
                pNa->pszVal);
        break;

    case SMISTsmi:
        if (addSMIitem(pNa->pszVal)) {
            pSym = newSMI(pNa, pMod);
            pSym->ut.smi.iKW = iKW;
            return(pSym);
        }
        break;
    }
#ifdef OLD
    yyterm();
#endif

    return(NULL);

} /* addSMIname */


/* end of file: SMMOD.C */
