#include "easy.h"
#include "easydefs.h"

#include <stdio.h>
#ifdef SYSVSTR
#include <string.h>
#else
#include <strings.h>
#endif
#include <stdlib.h>
#include <time.h>

#define CREATEPROC "createproc"
#define MAXSTRLEN  8192

/* 
  ndf_file -format:

  Record#1:  NX NY NZ   (mesh dimensions)

  The rest:  procno PVM_hostname exe_name [args]
        or:  procno_A,procno_B,procno_C PVM_hostname exe_name [args]
	or:  procno_A-procno_B PVM_hostname exe_name [args]
	or:  procno_A-procno_B,procno_C PVM_hostname exe_name [args]

	PVM_hostname can be one of
	(a) hostname (cs.utk.edu, crocus)
	(b) arch (e.g. SUN4)
	(c) * (wildcard == any host)

*/


static char *read_line(fp, outstr, maxoutstrlen)
     FILE *fp;
     char *outstr;
     int maxoutstrlen;
{
  static char s[MAXSTRLEN];
  int ch;
  int count = 0;
  char *p = outstr ? outstr : s;
  int maxstrlen = outstr ? maxoutstrlen : MAXSTRLEN;
  char *startp = p;
  int len;

  if (!fp) return NULL;
  if (feof(fp) || ferror(fp)) return NULL;

  while ( (ch = getc(fp)) != EOF && count < maxstrlen ) {
    if ( ch == '\n' ) break;
    *p++ = ch;
    count++;
  }
  *p = '\0';

  len = strlen(startp);

  if (len > 0 && !outstr) {
    /* Check for backquote or dollar in string */
    if (strchr(startp,'`') || strchr(startp,'$')) {
      FILE *pipe;
      char *cmd = (char *)malloc((len+20) * sizeof(*cmd));
      if (cmd) {
	sprintf(cmd,"echo \"%s\"",startp);
	pipe = popen(cmd,"r");
	if (pipe) {
	  startp = read_line(pipe,s,sizeof(s));
	  pclose(pipe);
	}
	else {
	  fprintf(stderr,"read_line: Cannot execute '%s'\n",startp);
	  exit(1);
	}
	free(cmd);
      }
    } /* if (strchr(startp,'`') || strchr(startp,'$')) */
  }
  
  return startp;
}


static char *scan_str(line, advanced)
     char *line; 
     int *advanced;
{
  static char s[MAXSTRLEN];
  char *linestartp = line;
  char *p;

  
  s[0] = '\0';

  if (line) {
    while ( *line == ' ' || *line == '\t' ) line++;
    switch ( *line ) {

    case '#':
      *line = '\0';
      break;

    case '!': /* Execute command after ! */
      {
	FILE *pipe;
	p = s;
	for (;;) {
	  line++;
	  if (*line == ' ' || *line == '\0') {
	    if (*line == ' ') line++;
	    break;
	  }
	  *p++ = *line;
	} /* for (;;) */
	*p = '\0';
	pipe = popen(s,"r"); /* Execute command and get output  */
	if (pipe) {
	  read_line(pipe,s,sizeof(s));
	  pclose(pipe);
	}
	else {
	  fprintf(stderr,"scan_str: Cannot execute '%s'\n",s);
	  exit(1);
	}
      }
      break;

    case '"':  /* String is between " " */
      p = s;
      for (;;) {
	line++;
	if (*line == '"' || *line == '\0') {
	  if (*line == '"') line++;
	  break;
	}
	*p++ = *line;
      } /* for (;;) */
      *p = '\0';
      break;

    default: /* Search for the next blank or end */
      p = s;
      for (;;) {
        if (*line == ' ' || *line == '#' || *line == '\0') {
	  if (*line == '#') *line = '\0';
	  break;
	}
        *p++ = *line++;
      } /* for (;;) */
      *p = '\0';
      break;
    } /* switch ( *line ) */
    while ( *line == ' ' || *line == '\t' ) line++;

  }

  else {
    fprintf(stderr,"scan_str: Unexpected input error\n");
    exit(1);
  }

  if (advanced) *advanced += line - linestartp;

  return s;
}

 
static int char_count(number)
     int number;
{
  int count = 1;
  if (number < 0) number = -number;
  while (number >= 10) {
    number /= 10;
    count++;
  }
  return count;
}


static int getnum(length, s, num, next, limit)
     int length; 
     char *s;
     int *num; 
     int *next; 
     int limit;
{
  int number;

  if (*s == '\0' || *s == ',' || *s == ' ') {
    *num = -1;
    *next = *++s;
    return ++length;
  }
  else if (*s == '-') {
    return getnum(++length,++s,num,next,limit);
  }
  else if (*s == '*') {
    *num = limit-1;
    *next = *++s;
    return ++length;
  }
  else if (sscanf(s,"%d",&number) == 1) {
    int len = char_count(number);
    *num = number;
    *next = s[len];
    return length + len;
  }
  else
    return length;
}


static int *read_procno(line, n, limit, advanced)
     char *line;
     int *n; 
     int limit; 
     int *advanced;
{
  char *procnums = scan_str(line,advanced);

  if (procnums) {
    int *procno = (int *)calloc(1,sizeof(*procno)); 
    int count=0;
    int k=0;
    int low=0, high=-1, num, prev=',', next;
    int len;
    
    while ((len = getnum(0,procnums,&num,&next,limit)) > 0) {
      if (num >= 0 && num < limit) {
	if (prev == ',') { /* e.g. 1,2,5 */
	  low = num;
	  high = num;
	  prev = next;
	}
	else if (prev == '-') {  /* e.g. 0-5 */
	  high = num;
	  prev = ',';
	}
      }
      else {
	prev=',';
      }
      if ((next == ',' || next == '\0') && (high >= low)) {
	int i;
	count += high - low + 1;
	procno = (int *)realloc(procno, count * sizeof(*procno));
	for (i=low; i<=high; i++) procno[k++] = i;
	low = 0;
	high = -1;
	prev=',';
	len++; /* skip over the ',' ! */
      }
      if (next == '\0') break;
      procnums += len;
    } /* while getnum() */

    if (n) *n = count;
    return procno;
  }
  else {
    if (n) *n = 0;
    return NULL;
  }

}


static char *read_hostname(line, advanced)
     char *line;
     int *advanced;
{
  static char hostname[64];
  char *p = scan_str(line,advanced);
  if (p) {
    strncpy(hostname,p,sizeof(hostname));
  }
  else {
    hostname[0] = '\0';
  }
  return hostname;
}


static char *read_exename(line, advanced)
     char *line;
     int *advanced;
{
  static char exename[512];
  char *p = scan_str(line,advanced);
  if (p) {
    strncpy(exename,p,sizeof(exename));
  }
  else {
    exename[0] = '\0';
  }
  return exename;
}


int createproc(ndf_file)
     char *ndf_file;
{
  int i;
  int xyzdim[3];
  FILE *ndf = fopen(ndf_file,"r");
  int *proctable;
  char *line;

  if (!ndf_file) {
    fprintf(stderr,"%s: Node definition file is NULL\n",
	    CREATEPROC);
    return 0;
  }

  ndf = fopen(ndf_file,"r");

  if (!ndf) {
    perror(ndf_file);
    return 0;
  }


  attachproc(); /* Side-effects:
                   - active_cluster_id is set to next free in cluster_id[]-table 
 		   */

  NX = NY = NZ = 0;
  do {
    line = read_line(ndf,NULL,0);
    if (!line) {
      perror(ndf_file);
      return 0;
    }
    while ( *line == ' ' || *line == '\t') line++;
  } while ( *line == '#' );

  sscanf(line,"%d %d %d",&NX,&NY,&NZ);

  if (NX < 1) NX = 1;
  if (NY < 1) NY = 1;
  if (NZ < 1) NZ = 1;

  xyzdim[0] = NX;
  xyzdim[1] = NY;
  xyzdim[2] = NZ;

  NUMNODES = NX * NY * NZ;
  NODEID = (int *)malloc( NUMNODES * sizeof(int));
  BCASTNODEID = NODEID;

  proctable = (int *)malloc(NUMNODES * sizeof(*proctable));
  for (i=0; i<NUMNODES; i++) {
    proctable[i] = -1;
  }

  i = 0;
  while ( (line = read_line(ndf,NULL,0)) != NULL ) {
    int *procno;
    char *hostname;
    char *exename;
    int numargs;
    char **args = NULL;
    int numexes = 0;
    int method=PvmTaskHost;
    int advanced=0;

    procno = read_procno(line,&numexes,NUMNODES,&advanced);

    if (numexes <= 0) continue;

    i += numexes;

    if (i > NUMNODES) {
      fprintf(stderr,
	      "%s: Nodecount not consistent with (NX,NY,NZ)=(%d,%d,%d)\n",
	      CREATEPROC,NX,NY,NZ);
      fprintf(stderr,
	      "%s: Last numexes=%d, nodecount=%d\n",
	      CREATEPROC,numexes,i);
      return 0;
    }

    hostname = read_hostname(line+advanced,&advanced);
    if ( strcmp(hostname,"*") == 0 ) 
      method = PvmTaskDefault;
    else if ( hostname[0] >= 'A' && hostname[0] <= 'Z' ) /* Arch presumably */
      method = PvmTaskArch;
    else 
      method = PvmTaskHost;

    exename = read_exename(line+advanced,&advanced);

    {
      int save_advanced_so_far = advanced;
      numargs = 0;
      for (;;) {
	char *p = scan_str(line+advanced,&advanced);
	if (!p) break;
	if ( *p == '\0' ) break;
	numargs++;
      } 
      advanced = save_advanced_so_far;
    }

    if (numargs > 0) {
      int j;
      args = (char **)malloc((numargs+1)* sizeof(char *));
      for (j=0; j<numargs; j++) {
	int len;
	char *tmparg = scan_str(line+advanced,&advanced);
	len = strlen(tmparg);
	args[j] = (char *)malloc((len+1) * sizeof(char));
	strcpy(args[j],tmparg);
      } /* for j */
      args[numargs] = NULL;
    } /* if (numargs > 0) */

    if (procno) {
      /* -- Start programs here -- */
      int j;
      for (j=0; j<numexes; j++) {
	if (proctable[procno[j]] != -1) {
	  fprintf(stderr,"%s: Node #%d already initiated\n",
		  CREATEPROC,procno[j]);
	}
	else {
	  proctable[procno[j]] = procno[j];
	}
      } /* for (j=0; j<numexes; j++) */

      {
	int *taskid = (int *)malloc(numexes * sizeof(*taskid));
	int retcode;

	retcode = 
	  pvm_spawn(exename, args, method, hostname,
		    numexes, taskid);
	if (retcode != numexes) {
	  fprintf(stderr,"%s: Problems in starting exe '%s@%s'\n",
		  CREATEPROC,exename,hostname);
	  fprintf(stderr,"%s: Only %d of %d executables started\n",
		  CREATEPROC,
		  (retcode>=0) ? retcode : 0, numexes);
	  pvm_perror(exename);
	  exit(retcode);
	}
	for (j=0; j<numexes; j++) {
	  NODEID[procno[j]] = taskid[j];
	}
	free(taskid);
      }

      free(procno);
    }

    if (numargs > 0) {
      int j;
      for (j=0; j<numargs; j++) {
	free(args[j]);
      } /* for j */
      free(args);
    }

    if (i>=NUMNODES) break;

  } /* for (;;) */

  fclose(ndf);

  if (proctable) {
    free(proctable);
    proctable = NULL;
  }

  if (i != NUMNODES) {
    fprintf(stderr,
	    "%s: # of exes specified (=%d) is different from started (=%d)\n",
	    CREATEPROC,NUMNODES,i);
    return 0;
  }

  pvm_notify(PvmTaskExit,MSG_SHUTDOWN,NUMNODES,NODEID);

  { /* Check whether NODEs have the same data rep. fmt. 
       i.e. XDRring can be then removed.
       Check also if HOST has different data rep. fmt. than NODEs do have. */

    struct hostinfo *hostp;
    int nhost, narch;

    pvm_config(&nhost,&narch,&hostp);

    if (narch > 1) { 
      /* At least there are different datareps. But have NODEs the same ? */

      INITHOW_host = PvmDataDefault; /* XDR at least between HOST & NODEs */

      if (NUMNODES > 1) {
	int i,j;
	int node_archcode0, node_archcodesSUM;
	/* From PVM archcode.c source file */

	for (j=0; j<NUMNODES; j++) {
	  int node_pvmd_tids_j = pvm_tidtohost(NODEID[j]);
	  for (i=0; i<nhost; i++) {

	    if (hostp[i].hi_tid == node_pvmd_tids_j) {
	      int archcode = getarchcode(hostp[i].hi_name); /* Get ARCH-code */

	      if (j==0) {
		node_archcode0 = archcode;
		node_archcodesSUM = archcode;
	      }
	      else
		node_archcodesSUM += archcode;

	      break;
	    } /* if (hostp[i].hi_tid == node_pvmd_tids_j) */
	  } /* for (i=0; i<nhost; i++) */
	} /* for (j=0; j<NUMNODES; j++) */
	
	/* A simple test */
	if (node_archcodesSUM == (node_archcode0 * NUMNODES))
	  INITHOW_node = PvmDataRaw;
	else
	  INITHOW_node = PvmDataDefault;

      }
      else {
	INITHOW_node = PvmDataRaw; /* Stupidly trivial case ! */
      }
      
    }
    else { 
      /* All share the same data rep. fmt.; So RAW-xfer is possible; Bypass XDR */
      INITHOW_host = INITHOW_node = PvmDataRaw;
    }
  }

  /* Send start up message to every node */

#ifdef PICL
  {
    char *p = getenv("EASY_PICLTRACE");
    if (p) {
      char time_string[7];
      struct tm *localtp;
      PICLTRACE = 1;
      HOST_TIMESTAMP = time(NULL);
      localtp = localtime((time_t *)&HOST_TIMESTAMP);
      strftime(time_string,sizeof(time_string),
	       "%H%M%S",
	       localtp);
      sscanf(time_string,"%d",&HOST_TIMESTAMP);
      traceenable("host",0);
    }
    else {
      PICLTRACE = 0;
    }
  }
#endif /* PICL */

  pvm_initsend(PvmDataDefault);

  pvm_pkint(&active_cluster_id,1,1);
  pvm_pkint(&NUMNODES,1,1);
  pvm_pkint(NODEID,NUMNODES,1);
  pvm_pkint(xyzdim,3,1);
  pvm_pkint(&INITHOW_host,1,1);  /* XDRring between HOST & NODE ? */
  pvm_pkint(&INITHOW_node,1,1);  /* XDRring between NODES ? (e.g. global opers) */
  
  pvm_pkint(&PICLTRACE,1,1);
  pvm_pkint(&HOST_TIMESTAMP,1,1);

  pvm_mcast(NODEID,NUMNODES,MSG_STARTUP);

  setdatatype(BYTE1);
  setstride(1);

  check_printfmsg_arrival();

  return NUMNODES;
}

int getarchcode(arch)   /* A temporary hack to obtain arch code 
			   For example SGI and SUN4 may have the same
			   arch code i.e. data representation */
     char *arch;
{
  static char *prevarch = NULL;
  static int x=0;

  if (!prevarch) {
    prevarch = arch;
    return ++x;
  }
  else if (strcmp(arch,prevarch) == 0) {
    return x;
  }
  else
    return ++x;
}
