/* Copyright (c) 1995,1996 NEC Corporation.  All rights reserved.            */
/*                                                                           */
/* The redistribution, use and modification in source or binary forms of     */
/* this software is subject to the conditions set forth in the copyright     */
/* document ("COPYRIGHT") included with this distribution.                   */
#define __NOLOGUPDATEPROTO

#include "socks5p.h"

#include "log.h"       /* Prototypes & defines.                              */
#include "threads.h"   /* MUTEX & THREAD_SELF                                */

IFTHREADED(extern MUTEX_T env_mutex;)
IFTHREADED(extern MUTEX_T lt_mutex;)

void *S5LogDefaultHandle = NULL;
int   S5LogShowThreadIDS   = 0;

#ifdef HAVE_STDARG_H
#include <stdarg.h>
#define VA_START(a, b) va_start((a), (b))
#define va_alist ...
#define va_dcl
#else
#include <varargs.h>
#define VA_START(a, b) va_start((a))
#endif

#ifdef HAVE_NL_TYPES_H
#include <nl_types.h>
#endif

#ifdef HAVE_SYSLOG_H
#include <syslog.h>
#ifndef LOG_DAEMON
#define LOG_DAEMON LOG_USER
#endif
#endif

typedef struct {
#ifdef HAVE_NL_TYPES_H
    nl_catd catalog;
#define NL_START(x, y)     (x)->catalog = catopen((y), 0)
#define NL_UPDATE(x, e, f) if ((x)->catalog != (nl_catd)-1) (f) = catgets((x)->catalog, 0, (e), (f))
#define NL_CLOSE(x)        if ((x)->catalog != (nl_catd)-1) catclose((x)->catalog); (x)->catalog = (nl_catd)-1
#elif defined(HAVE_DGETTEXT)
    char domain[MAXNAMELEN];
#define NL_START(x, y)     strncpy((x)->domain, name, MAX(MAXNAMELEN, strlen(name))); (x)->domain[MAX(MAXNAMELEN, strlen(name))] = '\0'
#define NL_UPDATE(x, e, f) (f) = dgettext((x)->domain, (f))
#define NL_CLOSE(x)
#else
#define NL_START(x, y) 
#define NL_UPDATE(x, e, f) 
#define NL_CLOSE(x)
#endif
    int level;
    int how;
} S5LogHandle;

static void replacePercentM(const char *inbuffer, char *outbuffer, int olen) {
    register const char *t2;
    register char *t1, ch;

    if (!outbuffer || !inbuffer) return;

    for (t1 = outbuffer; (ch = *inbuffer) && t1-outbuffer < olen ; ++inbuffer)
	if (inbuffer[0] == '%' && inbuffer[1] == 'm') 
	    for (++inbuffer, t2 = strerror(GETERRNO()); t2 && (*t1 = *t2++); t1++);
	else *t1++ = ch;
    
    *t1 = '\0';
}

void S5LogvUpdate(const void *handle, int level, int msgID, const char *oformat, va_list pvar) {
    char fmt_cpy[2*1024 + 2*10], format[2*1024 + 2*10];
    S5LogHandle *h = (S5LogHandle *)handle;
    int serrno = GETERRNO();
    static int dontLoop = 0;


    *fmt_cpy = '\0';

    /* If S5LogHandle has not been initialized, do it before we start;       */
    /* Saves the mess of making sure it gets called ahead of time.           */
    if (h == NULL) {
	S5LogStart(&S5LogDefaultHandle, -1, -1, "libsocks5");
	h = (S5LogHandle *)S5LogDefaultHandle;
    }

    /* If the handle is invalid, don't log.                                  */
    /* If the maximum log level is too low for this message, don't log.      */
    /* If something that we call forces us to log a message, don't log.      */
    if (!h || level > h->level || dontLoop) return;

    /* Mark that we're in the log...                                         */
    dontLoop++;

    /* Change the format if the message is in the catalog...                 */
    NL_UPDATE(h, msgID, oformat);

    if (!oformat) { dontLoop--; return; }

    /* Print the pid & maybe the thread id in format here.  Skip forward if  */
    /* you don't want it later (e.g. if syslogging).                         */
    sprintf(format, "%05d:", (int)getpid());
    if (S5LogShowThreadIDS) sprintf(format+strlen(format), "%06d:", (int)THREAD_SELF());
    strcat(format, " ");
    
    replacePercentM(oformat, format + strlen(format), sizeof(format) - strlen(format));
    vsprintf(fmt_cpy, format, pvar);

    /* Log to the Local log facility, e.g. Stderr on Unix and maybe a window */
    /* or something on NT.  Neither system can deal with a NULL format so    */
    /* check that here too.                                                  */
    if (h->how & S5_LOG_LOCAL && format) {

	fprintf(stderr, "%s\n", fmt_cpy);
	fflush(stderr);
    }

    /* Log to the system logging facility -- e.g. Syslog on Unix & the       */
    /* EventLog on Windows NT.                                               */
    if (h->how & S5_LOG_SYSTEM) {
#ifdef HAVE_SYSLOG_H
	int slfac = LOG_NOTICE;
	int offset = 6; 

#ifdef USE_SYSLOG_LEVELS
	switch(level) {
	    case S5_LOG_ERROR:   slfac = LOG_ERR;    break;
	    case S5_LOG_INFO:    slfac = LOG_NOTICE; break;
	    case S5_LOG_WARNING: slfac = LOG_INFO; break;
	    default:             slfac = LOG_DEBUG;
	}
#endif

	/* skip "%05d:", and maybe another " "                               */
	if (!S5LogShowThreadIDS) offset++;
	syslog(slfac, fmt_cpy + offset);
#else
#endif
    }

    SETERRNO(serrno); /* restore errno, just in case...?                     */
    dontLoop--;
    return;
}

void  S5LogUpdate(const void *handle, int level, int msgID, const char *format, va_alist) va_dcl {
    va_list pvar;
    VA_START(pvar, format);
    S5LogvUpdate(handle, level, msgID, format, pvar);
    va_end(pvar);
}

void S5LogStart(void **vhp, int how, int level, const char *name) {
    S5LogHandle **hp = (S5LogHandle **)vhp;
    char tbuf[1024], buf[1024], *tmp;
    time_t now = time(NULL);
    int allocated = 0;

    sprintf(buf, "%s", name);

    /* Find out from environment how we should log                           */
    if (how == -1) {
	how = 0;
	MUTEX_LOCK(env_mutex);
	if (getenv("SOCKS5_LOG_SYSLOG")) how |= S5_LOG_SYSTEM;
	if (getenv("SOCKS5_LOG_STDERR")) how |= S5_LOG_LOCAL;
	MUTEX_UNLOCK(env_mutex);
    }

    /* Find out from environment what kind of level we should use            */
    if (level == -1) {
	MUTEX_LOCK(env_mutex);
	if ((tmp = getenv("SOCKS5_LOG_LEVEL"))) {
	    /* Might be nice to check for strings like "quiet", "debug" here */
	    level = atoi(tmp);
	} else if ((tmp = getenv("SOCKS5_QUIET"))) {
	    level = 0;
	} else if ((tmp = getenv("SOCKS5_DEBUG"))) {
	    if (*tmp) level = S5_LOG_DEBUG(atoi(tmp));
	    else level = 0xff;
	} else if ((tmp = getenv("SOCKS5_WARNINGS"))) {
	    level = S5_LOG_WARNING;
	}
	MUTEX_UNLOCK(env_mutex);
    }
    
    if (*hp) {
	/* If we are just modifying an preexisting handle, only change it if */
	/* the environment variables said to...                              */
	if (how   !=  0) (*hp)->how   = how;
	if (level != -1) (*hp)->level = level;
    } else if (((*hp) = (S5LogHandle *)malloc(sizeof(S5LogHandle))) == NULL) {
	/* no where to store things, so just skip it.                        */
	return;
    } else {
	allocated = 1;
	(*hp)->how     = how;
	(*hp)->level   = level;

	NL_START(*hp, buf);

    }

    if ((*hp)->how   == 0)  (*hp)->how   = S5_LOG_SYSTEM;
    if ((*hp)->level == -1) (*hp)->level = S5_LOG_INFO;
    
#ifdef HAVE_SYSLOG_H
    /* Only the first person to call this gets to open the log, that way the */
    /* monitor doesn't clobber the server's socks5 name in the logs.         */
    if ((*hp)->how & S5_LOG_SYSTEM){
	static int logopened = 0;
	
	if (!logopened) {
	    openlog(name, LOG_PID, SYSLOG_FAC);
	    logopened = 1;
	}
    }
#endif

    MUTEX_LOCK(lt_mutex);
    strftime(tbuf, sizeof(tbuf), "%c", REAL(localtime)(&now));
    S5LogUpdate((*hp), S5_LOG_DEBUG(0), 0, "%s Logging (re)started at %s", name, tbuf);
    MUTEX_UNLOCK(lt_mutex);
}

void S5LogEnd(void *vh) {
    S5LogHandle *h = (S5LogHandle *)vh;

    if (!h) return;
    NL_CLOSE(h);

#ifdef HAVE_SYSLOG_H
    closelog();
#endif

    
    free(h);
}

