/*
    Filter Manager - Management software for the Drawbridge package
    Copyright (C) 1993 David K. Hess, Douglas Lee Schales, David R. Safford

    Please see the file `COPYING' for the complete copyright notice.

    nit.c - Version 1.0 - 4/21/93
*/

/*************************************************************************/
/* nit_open()
 * 1)   Open the NIT device
 * 2)	Set it into message-discard mode.
 * 3)	Push an instance of the packet-filtering module on it that passes
 *	through only packets whose Ethernet type field is type.
 * 4)	Bind it to device.
 *
 * Return the resulting descriptor, or -1 on error.
 * NOTE: this nit fd is in BLOCKING MODE!
 *
 * nit_write()
 * write packet to NIT:
 *     *buf is complete ethernet packet with header, minimum 60 bytes.
 *     Note that NIT will change ether.src to local interface's addr.
 *     Note that broadcasts WILL be echoed upstream WITHOUT the eaddr set!
 * return length written, or -1 on error
 */
#include "fm.h"

#define NIT_DEV "/dev/nit"

int nit_open(char *device,u_short type)
{
    struct strioctl si;
    struct ifreq    ifr;
    struct ether_header eh;
    struct packetfilt pf;
    register u_short *fwp;
    register int    fd;

    if ((fd = open(NIT_DEV, O_RDWR)) < 0) {
	fprintf(messages,"nit open: %s\n",sys_errlist[errno]);
	exit(1);
    }

    if (ioctl(fd, I_SRDOPT, (char *) RMSGD) < 0) {
	fprintf(messages,"ioctl (I_SRDOPT): %s\n",sys_errlist[errno]);
	exit(1);
    }

    /*
     * Push the packet filtering module. and set filter for ether.type ==
     * type.
     */
    if (ioctl(fd, I_PUSH, "pf") < 0) {
	fprintf(messages,"nit filter module push: %s\n",sys_errlist[errno]);
	exit(1);
    }

    /*
     * Push a filter for the specified protocol.
     */
    fwp = &pf.Pf_Filter[0];
    *fwp++ = ENF_PUSHWORD +
	((u_int) & eh.ether_type - (u_int) & eh.ether_dhost) / sizeof(u_short);
    *fwp++ = ENF_PUSHLIT | ENF_EQ;
    *fwp++ = type;		/* already in network order */
    pf.Pf_FilterLen = fwp - &pf.Pf_Filter[0];
    pf.Pf_Priority = 5;		/* unimportant, so long as > 2 */

    si.ic_cmd = NIOCSETF;
    si.ic_timout = INFTIM;
    si.ic_len = sizeof pf;
    si.ic_dp = (char *) &pf;

    if (ioctl(fd, I_STR, (char *) &si) < 0) {
	fprintf(messages,"nit filter push: %s\n",sys_errlist[errno]);
	exit(1);
    }

    /*
     * do the bind
     */
    (void) strncpy(ifr.ifr_name, device, sizeof ifr.ifr_name - 1);
    ifr.ifr_name[sizeof ifr.ifr_name - 1] = '\0';
    si.ic_cmd = NIOCBIND;
    si.ic_len = sizeof ifr;
    si.ic_dp = (char *) &ifr;

    if (ioctl(fd, I_STR, (char *) &si) < 0) {
	fprintf(messages,"nit bind: %s\n",sys_errlist[errno]);
	exit(1);
    }

    ioctl(fd, I_FLUSH, (char *)FLUSHR);

    return fd;
}

int nit_write(int fd,char *b,int len)
{
    struct sockaddr sa;
    struct strbuf   cbuf, dbuf;
    register int    offset = sizeof(sa.sa_data);

    sa.sa_family = AF_UNSPEC;
    bcopy(b, sa.sa_data, offset);
    cbuf.maxlen = cbuf.len = sizeof sa;
    cbuf.buf = (char *) &sa;
    dbuf.maxlen = dbuf.len = len - offset;
    dbuf.buf = b + offset;

    if (putmsg(fd, &cbuf, &dbuf, 0) < 0) {
	fprintf(messages,"nit write: %s\n",sys_errlist[errno]);
	return -1;
    }

    return len;
}
