/* socket.c -- various socket routines to communicate with the nameserver
 *
 * Copyright (C) 1992, Bradley C. Spatz, bcs@ufl.edu
 * Last edited: Wed Apr 22 11:06:07 1992 by bcs (Bradley C. Spatz) on frenulum
 *
 * 3 public entry points are provided here:
 *    read_ns(char *data)
 *    write_ns(char *data)
 *    connect_to_server(char *server, *service)
 */

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include "replies.h"

/* Declare 2 global file pointers for communication with the namserver. */
static FILE *fp_to_ns, *fp_from_ns;
static int sock;

/* Provide some routines to read and write to the nameserver. */
read_ns(data)
char *data;
{
   /* Read from the nameserver, stripping of the newline at the end. */
   fgets(data, 1024, fp_from_ns);
   data[strlen(data) -1] = '\0';
#ifdef DEBUG
   fprintf(stderr, "read from ns: '%s'\n", data);
#endif
}

write_ns(data)
char *data;
{
   fputs(data, fp_to_ns);
   fflush(fp_to_ns);
#ifdef DEBUG
   fprintf(stderr, "wrote to ns: '%s'\n", data); 
#endif
}


/* A generic error routine for the connection sequence herewithin. */
static void bail(server, service, msg)
char *server;
char *service;
char *msg;
{
   fprintf(stderr, "Unable to connect to server %s at port %s.\n",
	   server, service);
   perror(msg);
   exit(1);
}


/* Establish a socket-based connection to the nameserver.  We check that
 * the server hostname (server) is ok and can deal with a numeric or
 * name the port (service).  This routine also checks the connection
 * once it has been established.
 */
int connect_to_server(server, service)
char *server;
char *service;
{
   static struct sockaddr_in sa;    /* Address of the server. */
   static struct hostent *hp;       /* Host entry for server. */
   struct servent *sp;    	    /* Nameserver service entry. */
   int port, code;
   char buf[80];

   /* Check out the server hostname. */
   if ((hp = gethostbyname(server)) != NULL) {
      bcopy((char *) hp->h_addr, (char *) &sa.sin_addr, hp->h_length);
      sa.sin_family = hp->h_addrtype;
   }
   else {
      bail(server, service, "unknown host");
   }
  
   /* Find the proper port. */
   port = atoi(service);
   if (port == 0) {
      /* By name. */
      if ((sp = getservbyname(service, "tcp")) != NULL) {
	 sa.sin_port = sp->s_port;
      }
      else {
	 bail(server, service, "bad port");
      }
   }
   else {
      /* By number. */
      sa.sin_port = htons((unsigned short) port);
   }

   /* Allocate a socket to use and make it for the Internet.  Oh boy. */
   if ((sock = socket(hp->h_addrtype, SOCK_STREAM, 0)) < 0) {
      bail(server, service, "socket");
   }

   /* Now connect to the server. */
   if (connect(sock, (struct sockaddr *) &sa, sizeof(sa)) < 0) {
      bail(server, service, "connect()");
   }

   /* Check that the path to and from the server is sound. */
   if ((fp_to_ns = fdopen(sock, "w")) == NULL) {
      bail(server, service, "fdopen() to");
   }
   if ((fp_from_ns = fdopen(sock, "r")) == NULL) {
      bail(server, service, "fdopen() from");
   }
}


close_connection()
{
   fclose(fp_to_ns);
   fclose(fp_from_ns);
   close(sock);
}
