/*
 * status.c: handles the status line updating, etc for IRCII 
 *
 * Written By Michael Sandrof
 * Extensive modifications by Jake Khuon and others (epic software labs)
 *
 * Copyright(c) 1990  Michael Sandrof
 * Copyright 1997, 1998 EPIC Software Labs
 * See the COPYRIGHT file, or do a HELP IRCII COPYRIGHT 
 */

#include "irc.h"
#include "dcc.h"
#include "term.h"
#include "status.h"
#include "server.h"
#include "vars.h"
#include "hook.h"
#include "input.h"
#include "commands.h"
#include "window.h"
#include "screen.h"
#include "mail.h"
#include "output.h"
#include "names.h"
#include "ircaux.h"
#include "alias.h"

#ifdef Char
#undef Char
#endif
#define Char const char

/*
 * Maximum number of "%" expressions in a status line format.  If you change
 * this number, you must manually change the sprintf() in make_status 
 */
#define MAX_FUNCTIONS 40

static	Char	*convert_format 	(char *, int);
static	Char	*status_nickname 	(Window *);
static	Char	*status_query_nick 	(Window *);
static	Char	*status_right_justify 	(Window *);
static	Char	*status_chanop 		(Window *);
static	Char	*status_channel 	(Window *);
static	Char	*status_server 		(Window *);
static	Char	*status_mode 		(Window *);
static	Char	*status_umode 		(Window *);
static	Char	*status_insert_mode 	(Window *);
static	Char	*status_overwrite_mode 	(Window *);
static	Char	*status_away 		(Window *);
static	Char	*status_oper 		(Window *);
static	Char	*status_user0 		(Window *);
static	Char	*status_user1 		(Window *);
static	Char	*status_user2 		(Window *);
static	Char	*status_user3 		(Window *);
static	Char	*status_user4 		(Window *);
static	Char	*status_user5 		(Window *);
static	Char	*status_user6 		(Window *);
static	Char	*status_user7 		(Window *);
static	Char	*status_user8 		(Window *);
static	Char	*status_user9 		(Window *);
static	Char	*status_user10 		(Window *);
static	Char	*status_user11 		(Window *);
static	Char	*status_user12 		(Window *);
static	Char	*status_user13 		(Window *);
static	Char	*status_user14 		(Window *);
static	Char	*status_user15 		(Window *);
static	Char	*status_user16 		(Window *);
static	Char	*status_user17 		(Window *);
static	Char	*status_user18 		(Window *);
static	Char	*status_user19 		(Window *);
static	Char	*status_dcc 		(Window *);
static	Char	*status_hold 		(Window *);
static	Char	*status_version 	(Window *);
static	Char	*status_clock 		(Window *);
static	Char	*status_hold_lines 	(Window *);
static	Char	*status_window 		(Window *);
static	Char	*status_mail 		(Window *);
static	Char	*status_refnum 		(Window *);
static	Char	*status_null_function 	(Window *);
static	Char	*status_notify_windows 	(Window *);
static	Char	*status_voice 		(Window *);
static 	Char 	*status_cpu_saver_mode 	(Window *);
static 	Char 	*status_position 	(Window *);
static	Char	*status_scrollback	(Window *);
static	Char	*status_windowspec	(Window *);
static	Char	*status_percent		(Window *);
static	Char	*status_test		(Window *);

/* These are used as placeholders for some expandos */
static	char	*mode_format 		= (char *) 0;
static	char	*umode_format 		= (char *) 0;
static	char	*query_format 		= (char *) 0;
static	char	*clock_format 		= (char *) 0;
static	char	*hold_lines_format 	= (char *) 0;
static	char	*channel_format 	= (char *) 0;
static	char	*cpu_saver_format 	= (char *) 0;
static	char	*mail_format 		= (char *) 0;
static	char	*nick_format		= (char *) 0;
static	char	*server_format 		= (char *) 0;
static	char	*notify_format 		= (char *) 0;

/* The raw formats for the status bars */
static	Char	*status_format[3] 	= {NULL, NULL, NULL};

/* The callback functions that correspond to the expandos in the status bar */
static	Char	*(*status_func[3][MAX_FUNCTIONS]) (Window *);

/* The number of callback functions in the above variable. */
static	int	func_cnt[3];


/*
 * This is how we store status line expandos.
 */
struct status_formats {
	int	map;
	char 	key;
	Char	*(*callback_function)(Window *);
	char	**format_var;
	int	format_set;
};

/*
 * This is the list of possible expandos.  Note that you should not use
 * the '{' character, as it would be confusing.  It is already used for 
 * specifying the map.
 */
struct status_formats status_expandos[] = {
{ 0, 'A', status_away,           NULL, 			-1 },
{ 0, 'B', status_hold_lines,     &hold_lines_format,	STATUS_HOLD_LINES_VAR },
{ 0, 'C', status_channel,        &channel_format,	STATUS_CHANNEL_VAR },
{ 0, 'D', status_dcc, 	         NULL, 			-1 },
{ 0, 'F', status_notify_windows, &notify_format,	STATUS_NOTIFY_VAR },
{ 0, 'H', status_hold,		 NULL,			-1 },
{ 0, 'I', status_insert_mode,    NULL,			-1 },
{ 0, 'K', status_scrollback,	 NULL,			-1 },
{ 0, 'L', status_cpu_saver_mode, &cpu_saver_format,	STATUS_CPU_SAVER_VAR },
{ 0, 'M', status_mail,		 &mail_format,		STATUS_MAIL_VAR },
{ 0, 'N', status_nickname,	 &nick_format,		STATUS_NICK_VAR },
{ 0, 'O', status_overwrite_mode, NULL,			-1 },
{ 0, 'P', status_position,       NULL,			-1 },
{ 0, 'Q', status_query_nick,     &query_format,		STATUS_QUERY_VAR },
{ 0, 'R', status_refnum,         NULL, 			-1 },
{ 0, 'S', status_server,         &server_format,     	STATUS_SERVER_VAR },
{ 0, 'T', status_clock,          &clock_format,      	STATUS_CLOCK_VAR },
{ 0, 'U', status_user0,		 NULL, 			-1 },
{ 0, 'V', status_version,	 NULL, 			-1 },
{ 0, 'W', status_window,	 NULL, 			-1 },
{ 0, 'X', status_user1,		 NULL, 			-1 },
{ 0, 'Y', status_user2,		 NULL, 			-1 },
{ 0, 'Z', status_user3,		 NULL, 			-1 },
{ 0, '#', status_umode,		 &umode_format,	     	STATUS_UMODE_VAR },
{ 0, '%', status_percent,	 NULL, 			-1 },
{ 0, '*', status_oper,		 NULL, 			-1 },
{ 0, '+', status_mode,		 &mode_format,       	STATUS_MODE_VAR },
{ 0, '.', status_windowspec,	 NULL, 			-1 },
{ 0, '=', status_voice,		 NULL, 			-1 },
{ 0, '>', status_right_justify,	 NULL, 			-1 },
{ 0, '@', status_chanop,	 NULL, 			-1 },
{ 0, '0', status_user0,		 NULL, 			-1 },
{ 0, '1', status_user1,		 NULL, 			-1 },
{ 0, '2', status_user2,		 NULL, 			-1 },
{ 0, '3', status_user3,		 NULL, 			-1 },
{ 0, '4', status_user4,		 NULL, 			-1 },
{ 0, '5', status_user5,		 NULL, 			-1 },
{ 0, '6', status_user6,		 NULL, 			-1 },
{ 0, '7', status_user7,		 NULL, 			-1 },
{ 0, '8', status_user8,		 NULL, 			-1 },
{ 0, '9', status_user9,		 NULL, 			-1 },
{ 1, '0', status_user10,	 NULL, 			-1 },
{ 1, '1', status_user11,	 NULL, 			-1 },
{ 1, '2', status_user12,	 NULL, 			-1 },
{ 1, '3', status_user13,	 NULL, 			-1 },
{ 1, '4', status_user14,	 NULL, 			-1 },
{ 1, '5', status_user15,	 NULL, 			-1 },
{ 1, '6', status_user16,	 NULL, 			-1 },
{ 1, '7', status_user17,	 NULL, 			-1 },
{ 1, '8', status_user18,	 NULL, 			-1 },
{ 1, '9', status_user19,	 NULL, 			-1 },
{ 1, 'T', status_test,		 NULL,			-1 }
};
#define NUMBER_OF_EXPANDOS (sizeof(status_expandos) / sizeof(struct status_formats))

/*
 * convert_sub_format: This is used to convert the formats of the
 * sub-portions of the status line to a format statement specially designed
 * for that sub-portions.  convert_sub_format looks for a single occurence of
 * %c (where c is passed to the function). When found, it is replaced by "%s"
 * for use is a sprintf.  All other occurences of % followed by any other
 * character are left unchanged.  Only the first occurence of %c is
 * converted, all subsequence occurences are left unchanged.  This routine
 * mallocs the returned string. 
 */
static	char	*convert_sub_format (char *format, char c)
{
	char	buffer[BIG_BUFFER_SIZE + 1];
	int	pos = 0;
	int	dont_got_it = 1;

	if (!format)
		return format;		/* NULL in, NULL out */

	while (*format && pos < BIG_BUFFER_SIZE - 4)
	{
		if (*format != '%')
		{
			buffer[pos++] = *format++;
			continue;
		}

		format++;
		if (*format == c && dont_got_it)
		{
			dont_got_it = 0;
			buffer[pos++] = '%';
			buffer[pos++] = 's';
		}
		else
		{
			buffer[pos++] = '%';
			buffer[pos++] = '%';
			buffer[pos++] = *format;
		}
		format++;
	}

	buffer[pos] = 0;
	return m_strdup(buffer);
}


/*
 * This walks a raw format string and parses out any expandos that it finds.
 * An expando is handled by pre-fetching any string variable that is used
 * by the callback function, the callback function is registered, and a
 * %s format is put in the sprintf()-able return value (stored in buffer).
 * All other characters are copied as-is to the return value.
 */
static	Char	*convert_format (char *format, int k)
{
	char	buffer[BIG_BUFFER_SIZE + 1];
	int	pos = 0;
	int	cp;
	int	map;
	char	key;
	int	i;

	cp = func_cnt[k];
	while (*format && pos < BIG_BUFFER_SIZE - 4)
	{
		if (*format != '%')
		{
			buffer[pos++] = *format++;
			continue;
		}

		/* Its a % */
		map = 0;

		/* Find the map, if neccesary */
		if (*++format == '{')
		{
			char	*endptr;

			format++;
			map = strtoul(format, &endptr, 10);
			if (*endptr != '}')
			{
				/* Unrecognized */
				continue;
			}
			format = endptr + 1;
		}

		key = *format++;

		/* Choke once we get to the maximum number of expandos */
		if (cp >= MAX_FUNCTIONS)
			continue;

		for (i = 0; i < NUMBER_OF_EXPANDOS; i++)
		{
			if (status_expandos[i].map != map ||
			    status_expandos[i].key != key)
				continue;

			if (status_expandos[i].format_var)
				new_free(status_expandos[i].format_var);
			if (status_expandos[i].format_set != -1)
				*(status_expandos[i].format_var) = 
					convert_sub_format(get_string_var(status_expandos[i].format_set), key);

			buffer[pos++] = '%';
			buffer[pos++] = 's';

			status_func[k][cp++] = 
				status_expandos[i].callback_function;
			break;
		}
	}

	func_cnt[k] = cp;
	buffer[pos] = 0;
	return m_strdup(buffer);
}

void	build_status (char *format)
{
	int i,k;

	for (k = 0; k < 3; k++)
	{
		new_free((char **)&status_format[k]);
		func_cnt[k] = 0;

		if (k == 0)
			format = get_string_var(STATUS_FORMAT_VAR);
		else if (k == 1)
			format = get_string_var(STATUS_FORMAT1_VAR);
		else if (k == 2)
			format = get_string_var(STATUS_FORMAT2_VAR);

		if (format)
			status_format[k] = convert_format(format, k);

		for (i = func_cnt[k]; i < MAX_FUNCTIONS; i++)
			status_func[k][i] = status_null_function;
	}
	update_all_status();
}


/*
 * This just sucked beyond words.  I was always planning on rewriting this,
 * but the crecendo of complaints with regards to this just got to be too 
 * irritating, so i fixed it early.
 */
void	make_status (Window *window)
{
	int	status_line;
	u_char	buffer	    [BIG_BUFFER_SIZE + 1];
	u_char	lhs_buffer  [BIG_BUFFER_SIZE + 1];
	u_char	rhs_buffer  [BIG_BUFFER_SIZE + 1];
	Char	*func_value [MAX_FUNCTIONS];
	u_char	*ptr;

	for (status_line = 0; status_line < window->double_status + 1; status_line++)
	{
		u_char	lhs_fillchar[6],
			rhs_fillchar[6],
			*fillchar = lhs_fillchar,
			*lhp = lhs_buffer,
			*rhp = rhs_buffer,
			*cp,
			*start_rhs = 0,
			*str;
		int	in_rhs = 0,
			pr_lhs = 0,
			pr_rhs = 0,
			line,
			*prc = &pr_lhs, 
			i;

		fillchar[0] = fillchar[1] = 0;

		/*
		 * If status line gets to one, then that means that
		 * window->double_status is not zero.  That means that
		 * the status line we're working on is STATUS2.
		 */
		if (status_line)
			line = 2;

		/*
		 * If status_line is zero, and window->double_status is
		 * not zero (double status line is on) then we're working
		 * on STATUS1.
		 */
		else if (window->double_status)
			line = 1;

		/*
		 * So status_line is zero and window->double_status is zero.
		 * So we're working on STATUS (0).
		 */
		else
			line = 0;


		/*
		 * Sanity check:  If the status format doesnt exist, dont do
		 * anything for it.
		 */
		if (!status_format[line])
			continue;

		/*
		 * Run each of the status-generating functions from the the
		 * status list.  Note that the retval of the functions is no
		 * longer malloc()ed.  This saves 40-some odd malloc/free sets
		 * each time the status bar is updated, which is non-trivial.
		 */
		for (i = 0; i < MAX_FUNCTIONS; i++)
			func_value[i] = (status_func[line][i])(window);

		/*
		 * If the REVERSE_STATUS_LINE var is on, then put a reverse
		 * character in the first position (itll get translated to
		 * the tcap code in the output code.
		 */
		if (get_int_var(REVERSE_STATUS_LINE_VAR))
			*buffer = REV_TOG , str = buffer + 1;
		else
			str = buffer;

		/*
		 * Now press the status line into "buffer".  The magic about
		 * setting status_format is handled elsewhere.
		 */
		snprintf(str, BIG_BUFFER_SIZE - 1, status_format[line],
		    func_value[0], func_value[1], func_value[2], 
		    func_value[3], func_value[4], func_value[5],
		    func_value[6], func_value[7], func_value[8], func_value[9],
		    func_value[10], func_value[11], func_value[12],
		    func_value[13], func_value[14], func_value[15],
		    func_value[16], func_value[17], func_value[18],
		    func_value[19], func_value[20], func_value[21],
		    func_value[22], func_value[23], func_value[24],
		    func_value[25], func_value[26], func_value[27],
		    func_value[28], func_value[29], func_value[30],
		    func_value[31], func_value[32], func_value[33],
		    func_value[34], func_value[35], func_value[36], 
		    func_value[37], func_value[38], func_value[39]); 

		/*
		 * If the user wants us to, pass the status bar through the
		 * expander to pick any variables/function calls.
		 * This is horribly expensive, but what do i care if you
		 * want to waste cpu cycles? ;-)
		 */
		if (get_int_var(STATUS_DOES_EXPANDOS_VAR))
		{
			int  af = 0;

			str = expand_alias(buffer, empty_string, &af, NULL);
			strmcpy(buffer, str, BIG_BUFFER_SIZE);
			new_free(&str);
		}


		/*
		 * This converts away any ansi codes in the status line
		 * in accordance with the currenet settings.  This leaves us
		 * with nothing but logical characters, which are then easy
		 * to count. :-)
		 */
		str = strip_ansi(buffer);

		/*
		 * Count out the characters.
		 * Walk the entire string, looking for nonprintable
		 * characters.  We count all the printable characters
		 * on both sides of the %> tag.
		 */
		ptr = str;
		cp = lhp;
		lhs_buffer[0] = rhs_buffer[0] = 0;

		while (*ptr)
		{
			/*
			 * The FIRST such %> tag is used.
			 * Using multiple %>s is bogus.
			 */
			if (*ptr == '\f' && start_rhs == NULL)
			{
				ptr++;
				start_rhs = ptr;
				fillchar = rhs_fillchar;
				in_rhs = 1;
				*cp = 0;
				cp = rhp;
				prc = &pr_rhs;
			}

			/*
			 * Skip over color attributes if we're not
			 * doing color.
			 */
			else if (*ptr == '\003')
			{
				const u_char *end = skip_ctl_c_seq(ptr, NULL, NULL);
				while (ptr < end)
					*cp++ = *ptr++;
			}

			/*
			 * If we have a ROM character code here, we need to
			 * copy it to the output buffer, as well as the fill
			 * character, and increment the printable counter by
			 * only 1.
			 */
			else if (*ptr == ROM_CHAR)
			{
				fillchar[0] = *cp++ = *ptr++;
				fillchar[1] = *cp++ = *ptr++;
				fillchar[2] = *cp++ = *ptr++;
				fillchar[3] = *cp++ = *ptr++;
				fillchar[4] = 0;
				*prc += 1;
			}

			/*
			 * Is it NOT a printable character?
			 */
			else if ((*ptr == REV_TOG) || (*ptr == UND_TOG) ||
				 (*ptr == ALL_OFF) || (*ptr == BOLD_TOG) ||
				 (*ptr == BLINK_TOG))
					*cp++ = *ptr++;

			/*
			 * XXXXX This is a bletcherous hack.
			 * If i knew what was good for me id not do this.
			 */
			else if (*ptr == 9)	/* TAB */
			{
				fillchar[0] = ' ';
				fillchar[1] = 0;
				do
					*cp++ = ' ';
				while (++(*prc) % 8);
				ptr++;
			}

			/*
			 * So it is a printable character.
			 */
			else
			{
				*prc += 1;
				fillchar[0] = *cp++ = *ptr++;
				fillchar[1] = 0;
			}

			/*
			 * Dont allow more than CO printable characters
			 */
			if (pr_lhs + pr_rhs >= window->screen->co)
			{
				*cp = 0;
				break;
			}
		}
		*cp = 0;

		/* What will we be filling with? */
		if (get_int_var(STATUS_NO_REPEAT_VAR))
		{
			lhs_fillchar[0] = ' ';
			lhs_fillchar[1] = 0;
			rhs_fillchar[0] = ' ';
			rhs_fillchar[1] = 0;
		}

		/*
		 * Now if we have a rhs, then we have to adjust it.
		 */
		if (start_rhs)
		{
			int numf = 0;

			numf = window->screen->co - pr_lhs - pr_rhs -1;
			while (numf-- >= 0)
				strmcat(lhs_buffer, lhs_fillchar, 
						BIG_BUFFER_SIZE);
		}

		/*
		 * No rhs?  If the user wants us to pad it out, do so.
		 */
		else if (get_int_var(FULL_STATUS_LINE_VAR))
		{
			int chars = window->screen->co - pr_lhs - 1;

			while (chars-- >= 0)
				strmcat(lhs_buffer, lhs_fillchar, 
						BIG_BUFFER_SIZE);
		}

		strcpy(buffer, lhs_buffer);
		strmcat(buffer, rhs_buffer, BIG_BUFFER_SIZE);
		strmcat(buffer, ALL_OFF_STR, BIG_BUFFER_SIZE);
		new_free(&str);

		if (dumb_mode)
		{
			do_hook(STATUS_UPDATE_LIST, "%d %d %s", 
				window->refnum, 
				status_line, 
				buffer);
			continue;
		}

		/*
		 * Simplified this optimization:
		 * If the status line has changed, then we 
		 * redraw it all.  If it hasnt changed, we dont redraw it.
		 * Simple?
		 */
		if (!window->status_line[status_line] ||
			strcmp(buffer, window->status_line[status_line]))
		{
			/*
			 * Roll the new back onto the old
			 */
			malloc_strcpy(&window->status_line[status_line], buffer);

			/*
			 * Output the status line to the screen
			 */
			if (!dumb_mode)
			{
				output_screen = window->screen;
				term_move_cursor(0, window->bottom + status_line);
#if 0
				term_clreol();
#endif
				output_line(buffer);
				cursor_in_display();
				term_bold_off();
			}
		}
	}

	cursor_to_input();
}


/* Some useful macros */
/*
 * This is used to get the current window on a window's screen
 */
#define CURRENT_WINDOW window->screen->current_window

/*
 * This tests to see if the window IS the current window on its screen
 */
#define IS_CURRENT_WINDOW (window->screen->current_window == window)

/*
 * This tests to see if all expandoes are to appear in all status bars
 */
#define SHOW_ALL_WINDOWS (get_int_var(SHOW_STATUS_ALL_VAR))

/*
 * "Current-type" window expandoes occur only on the current window for a 
 * screen.  However, if /set show_status_all is on, then ALL windows act as
 * "Current-type" windows.
 */
#define DISPLAY_ON_WINDOW (IS_CURRENT_WINDOW || SHOW_ALL_WINDOWS)


/*
 * These are the functions that all of the status expandoes invoke
 */


/*
 * This is your current nickname in the window.
 */
static	Char 	*status_nickname (Window *window)
{
static	char	my_buffer[64];

	snprintf(my_buffer, 63, nick_format, 
			get_server_nickname(window->server));
	return my_buffer;
}

/*
 * This displays the server that the window is connected to.
 */
static	Char	*status_server (Window *window)
{
	char	*rest,
		*name;
	char	*next;
static	char	my_buffer[64];
	size_t	len;
	char	c;

	/*
	 * If there is only one server, dont bother telling the user
	 * what it is.
	 */
	if (connected_to_server == 1)
		return empty_string;

	/*
	 * If this window isnt connected to a server, say so.
	 */
	if (window->server == -1)
		return "No Server";

	/*
	 * If the user doesnt want this expando, dont force it.
	 */
	if (!server_format)
		return empty_string;

	/* Figure out what server this window is on */
	name = get_server_name(window->server);

	/*
	 * If the first segment before the first dot is a number,
	 * then its an ip address, and use the whole thing.
	 */
	if (strtoul(name, &next, 10) && *next == '.')
	{
		snprintf(my_buffer, 63, server_format, name);
		return my_buffer;
	}

	/*
	 * Get the stuff to the left of the first dot.
	 */
	if (!(rest = strchr(name, '.')))
	{
		snprintf(my_buffer, 63, server_format, name);
		return my_buffer;
	}

	/*
	 * If the first segment is 'irc', thats not terribly
	 * helpful, so get the next segment.
	 */
	if (!strncmp(name, "irc", 3))
	{
		name = rest + 1;
		rest = strchr(name + 1, '.');
	}

	/*
	 * If the name of the server is > 60 chars, crop it back to 60.
	 */
	if ((len = rest - name) > 60)
		len = 60;

	/*
	 * Plop the server into the server_format and return it.
	 */
	c = name[len];
	name[len] = 0;
	snprintf(my_buffer, 63, server_format, name);
	name[len] = c;

	return my_buffer;
}

/*
 * This displays any nicks that you may have as your current query in the
 * given window.
 */
static	Char	*status_query_nick (Window *window)
{
	static char my_buffer[BIG_BUFFER_SIZE + 1];

	if (window->query_nick && query_format)
	{
		snprintf(my_buffer, BIG_BUFFER_SIZE, 
				query_format, window->query_nick);
		return my_buffer;
	}

	return empty_string;
}

/*
 * This forces anything to the right of this point to be right-justified
 * (if possible) on the status bar.  Note that the right hand side is always
 * cropped, and not the left hand side.  That is to say, this is a hint that
 * if the left hand side is too short, that the "filler" should be put here
 * at this point rather than at the end of the status line.
 */
static	Char	*status_right_justify (Window *window)
{
	static char my_buffer[] = "\f";
	return my_buffer;
}

/*
 * Displays whatever windows are notifying, and have notified
 * (usually when output happens on hidden windows that you marked with 
 * /window notify on)
 */
static	Char	*status_notify_windows (Window *window)
{
	int	doneone = 0;
	char	buf2[81];
static	char	my_buffer[81];

	/*
	 * This only goes to a current-type window.
	 */
	if (!DISPLAY_ON_WINDOW)
		return empty_string;

	*my_buffer = 0;
	*buf2 = 0;

	/*
	 * Look for any notifying windows that have had some output since 
	 * they have been hidden and collect their refnums.
	 */
	window = NULL;
	while (traverse_all_windows(&window))
	{
		if (window->miscflags & WINDOW_NOTIFIED)
		{
			if (doneone++)
				strmcat(buf2, ",", 80);
			strmcat(buf2, ltoa(window->refnum), 80);
		}
	}

	/*
	 * Only do the sprintf if there are windows to process.
	 */
	if (doneone && notify_format)
		snprintf(my_buffer, 80, notify_format, buf2);

	return my_buffer;
}

/*
 * Displays what time it is on the current-type window
 */
static	Char	*status_clock (Window *window)
{
	static	char	my_buffer[81];

	if (get_int_var(CLOCK_VAR) && clock_format && DISPLAY_ON_WINDOW)
		snprintf(my_buffer, 80, clock_format, update_clock(GET_TIME));
	else
		strcpy(my_buffer, empty_string);

	return my_buffer;
}

/*
 * The current channel mode for the current channel (if any)
 */
static	Char	*status_mode (Window *window)
{
	char	*mode = (char *) 0;
static  char    my_buffer[81];

	*my_buffer = 0;
	if (window->current_channel && window->server != -1)
	{
                mode = get_channel_mode(window->current_channel,window->server);
		if (mode && *mode && mode_format)
		{
			/*
			 * This gross hack is required to make sure that the 
			 * channel key doesnt accidentally contain anything 
			 * dangerous...
			 */
			if (get_int_var(STATUS_DOES_EXPANDOS_VAR))
			{
				char *mode2 = alloca(strlen(mode) * 2 + 1);
				double_quote(mode, "$", mode2);
				mode = mode2;
			}

			snprintf(my_buffer, 80, mode_format, mode);
		}
	}
	return my_buffer;
}


/*
 * Your user mode for the server in this window.
 */
static	Char	*status_umode (Window *window)
{
	char	localbuf[20];
static	char	my_buffer[81];

	/*
	 * If we are only on one server and this isnt the current-type 
	 * window, then dont display it here.
	 */
	if (connected_to_server == 1 && !DISPLAY_ON_WINDOW)
		return empty_string;

	/*
	 * Punt if the window isnt connected to a server.
	 */
	if (window->server < 0)
		return empty_string;

	strmcpy(localbuf, get_umode(window->server), 19);
	if (!*localbuf)
		return empty_string;

	snprintf(my_buffer, 80, umode_format, localbuf);
	return my_buffer;
}

/*
 * Figures out whether or not youre a channel operator on the current
 * channel for this window.
 */
static	Char	*status_chanop (Window *window)
{
	char	*text;

	if (window->current_channel && window->server != -1 &&
	    get_channel_oper(window->current_channel, window->server) &&
	    (text = get_string_var(STATUS_CHANOP_VAR)))
		return text;

	return empty_string;
}


/*
 * Figures out how many lines are being "held" (never been displayed, usually
 * because of hold_mode or scrollback being on) for this window.
 */
static	Char	*status_hold_lines (Window *window)
{
	int	num;
static	char	my_buffer[81];

	if ((num = (window->lines_held / 10) * 10))
	{
		snprintf(my_buffer, 80, hold_lines_format, ltoa(num));
		return my_buffer;
	}

	return empty_string;
}

/*
 * Figures out what the current channel is for the window.
 */
static	Char	*status_channel (Window *window)
{
	char 	channel[IRCD_BUFFER_SIZE + 1];
static	char	my_buffer[IRCD_BUFFER_SIZE + 1];
	int	num;

	if (!window->current_channel || window->server == -1)
		return empty_string;

	if (get_int_var(HIDE_PRIVATE_CHANNELS_VAR) && 
	    is_channel_private(window->current_channel, window->server))
		strmcpy(channel, "*private*", IRCD_BUFFER_SIZE);
	else
		strmcpy(channel, window->current_channel, IRCD_BUFFER_SIZE);

	num = get_int_var(CHANNEL_NAME_WIDTH_VAR);
	if (num > 0 && strlen(channel) > num)
		channel[num] = 0;

	snprintf(my_buffer, IRCD_BUFFER_SIZE, channel_format, channel);
	return my_buffer;
}

/*
 * Figures out if you are a channel voice for the current channel in this
 * window.  For whatever reason, this wont display if youre also a channel
 * operator on the current channel in this window.
 */
static	Char	*status_voice (Window *window)
{
	char *text;

	if (window->current_channel && window->server != -1 &&
	    get_channel_voice(window->current_channel, window->server) &&
	    !get_channel_oper(window->current_channel, window->server) &&
	    (text = get_string_var(STATUS_VOICE_VAR)))
		return text;

	return empty_string;
}

/*
 * Displays how much mail we think you have.
 */
static	Char	*status_mail (Window *window)
{
	char	*number;
static	char	my_buffer[81];

	/*
	 * The order is important here.  We check to see whether or not
	 * this is a current-type window *FIRST* because most of the time
	 * that will be false, and check_mail() is very expensive; we dont
	 * want to do it if we're going to ignores its result.
	 */
	if (get_int_var(MAIL_VAR) && mail_format && 
	    DISPLAY_ON_WINDOW && (number = check_mail()))
	{
		snprintf(my_buffer, 80, mail_format, number);
		return my_buffer;
	}

	return empty_string;
}

/*
 * Display if "insert mode" is ON for the input line.
 */
static	Char	*status_insert_mode (Window *window)
{
	char	*text;

	if (get_int_var(INSERT_MODE_VAR) && DISPLAY_ON_WINDOW &&
	    (text = get_string_var(STATUS_INSERT_VAR)))
		return text;

	return empty_string;
}

/*
 * Displays if "insert mode" is OFF for the input line.
 */
static	Char	*status_overwrite_mode (Window *window)
{
	char	*text;

	if (!get_int_var(INSERT_MODE_VAR) && DISPLAY_ON_WINDOW &&
	    (text = get_string_var(STATUS_OVERWRITE_VAR)))
		return text;

	return empty_string;
}

/*
 * Displays if you are AWAY (protocol away) on the current server for
 * the window.
 */
static	Char	*status_away (Window *window)
{
	char	*text;

	/*
	 * If we're only on one server, only do this for the
	 * current-type window.
	 */
	if (connected_to_server == 1 && !DISPLAY_ON_WINDOW)
		return empty_string;

	if (window->server != -1 && 
	    server_list[window->server].away && 
	    (text = get_string_var(STATUS_AWAY_VAR)))
		return text;

	return empty_string;
}


/*
 * This is a generic status_userX variable.  All of these go to the
 * current-type window, although i think they should go to all windows.
 */
#define STATUS_VAR(x) \
static	Char	*status_user ## x ## (Window *window)			\
{									\
	char	*text;							\
									\
	if ((text = get_string_var(STATUS_USER ## x ## _VAR)) && 	\
	     DISPLAY_ON_WINDOW)						\
		return text;						\
	else								\
		return empty_string;					\
}

STATUS_VAR(0) STATUS_VAR(1) STATUS_VAR(2) STATUS_VAR(3) STATUS_VAR(4)
STATUS_VAR(5) STATUS_VAR(6) STATUS_VAR(7) STATUS_VAR(8) STATUS_VAR(9)
STATUS_VAR(10) STATUS_VAR(11) STATUS_VAR(12) STATUS_VAR(13) STATUS_VAR(14)
STATUS_VAR(15) STATUS_VAR(16) STATUS_VAR(17) STATUS_VAR(18) STATUS_VAR(19)

static	Char	*status_hold (Window *window)
{
	char *text;

	if (window->holding_something && 
	    (text = get_string_var(STATUS_HOLD_VAR)))
		return text;
	else
		return empty_string;
}

static	Char	*status_oper (Window *window)
{
	char *text;

	if (window->server != -1 && get_server_operator(window->server) &&
	    (text = get_string_var(STATUS_OPER_VAR)) &&
	    (connected_to_server != 1 || DISPLAY_ON_WINDOW))
		return text;
	else
		return empty_string;
}

static	Char	*status_window (Window *window)
{
	char *text;

	if ((number_of_windows_on_screen(window) > 1) && 
	    IS_CURRENT_WINDOW && (text = get_string_var(STATUS_WINDOW_VAR)))
		return text;
	else
		return empty_string;
}

static	Char	*status_refnum (Window *window)
{
	static char my_buffer[81];

	strcpy(my_buffer, window->name ? window->name : ltoa(window->refnum));
	return my_buffer;
}

static	Char	*status_version (Window *window)
{
	if (DISPLAY_ON_WINDOW)
		return (char *)irc_version; /* XXXX */

	return empty_string;
}


static	Char	*status_null_function (Window *window)
{
	return empty_string;
}

/*
 * This displays the DCC "Progress Meter", which goes to the window that
 * has level DCC.  OR, if "current_window_level" includes DCC, then this
 * goes to the current window.
 */
static	Char *status_dcc (Window *window)
{
	if ((current_window_level & LOG_DCC && IS_CURRENT_WINDOW) ||
			(window->window_level & LOG_DCC))
		return DCC_get_current_transfer();

	return empty_string;
}

/*
 * This displays something if the client is in 'cpu saver' mode.
 * A discussion of that mode should be found in irc.c, so i wont get
 * into it here.
 */
static	Char *status_cpu_saver_mode (Window *window)
{
	static char my_buffer[81];

	if (cpu_saver && cpu_saver_format)
	{
		snprintf(my_buffer, 80, cpu_saver_format, "CPU");
		return my_buffer;
	}

	return empty_string;
}

/*
 * This is a private expando that i use for testing.  But if you want to
 * use it, more power to you!  I reserve the right to change this expando
 * at my whims.  You should not rely on anything specific happening here.
 */
static	Char *status_position (Window *window)
{
	static char my_buffer[81];

	snprintf(my_buffer, 80, "(%d-%d)", 
			window->screen->input_line,
			window->screen->input_cursor);
#if 0
			window->lines_scrolled_back,
			window->distance_from_display);
#endif
	return my_buffer;
}

/*
 * This returns something if this window is currently in scrollback mode.
 * Useful if you sometimes forget!
 */
static	Char *status_scrollback (Window *window)
{
	char *stuff;

	if (window->scrollback_point && 
	    (stuff = get_string_var(STATUS_SCROLLBACK_VAR)))
		return stuff;
	else
		return empty_string;
}

static	Char	*status_windowspec	(Window *window)
{
	static char my_buffer[81];

	if (window->status_special)
		strmcpy(my_buffer, window->status_special, 80);
	else
		*my_buffer = 0;

	return my_buffer;
}

static	Char	*status_percent		(Window *window)
{
	static	char	percent[] = "%";
	return	percent;
}

static	Char	*status_test		(Window *window)
{
	static	char	retval[] = "TEST";
	return retval;
}

