# listserv-bkt
# 6-Feb-1993 totty@cs.uiuc.edu

# This file contains an alternative implementation of the listserv mailing
# list service.  It attempts to be more robust and user friendly than the
# original eitech offering, but the code is based heavily on the eitech code.
#
# The mailing list file names have been changed slightly to make them more
# uniform.  All mailing list files now have '.list' extensions.  Information
# about mailing lists is stored in '.info' files.  One line descriptions of
# each mailing list is stored in '.oneliner' files.
#
# All responses are now returned as strings.  It might make sense to re-add in
# the more general response types in the future.
#
#------------------------------------------------------------------------------
#
# 25-Jun-92 weber@eitech.com updated to new parameter format
# 1-Jun-92 weber@eitech.com
#
# This service mimicks some of the commands in the listserv package
# for mailing list maintenance.  Mailing lists must be manually created;
# e.g., to create a list called "mylist", do a "touch ~/Lists/mylist" and
# then add the following line to your system's /etc/aliases file:
# "mylist: :include:~/Lists/mylist" (where you have to replace the "~"
# with services' home directory on your system).

##############################################################################
#
#	listserv switches envelope inputs
#
##############################################################################

proc listserv {switches envelope inputs} {
	set olddir [pwd] ; cd ~/Lists

	set bodyfilename $olddir/[getfield $inputs FILE]

	catch {
	    set switches [string trimleft $switches\ [exec cat $bodyfilename]]}
	if {[llength $switches] == 0} {set switches help}

	set command [string tolower [lindex $switches 0]]

	case $command {
		{ subscribe } \
		    { set reply_text [listserv-subscribe $switches] }
		{ subscribe-address } \
		    { set reply_text [listserv-subscribe-address $switches] }
		{ unsubscribe signoff } \
		    { set reply_text [listserv-unsubscribe $switches] }
		{ recipients review } \
		    { set reply_text [listserv-recipients $switches] }
		{ information info } \
		    { set reply_text [listserv-information $switches] }
		{ lists } \
		    { set reply_text [listserv-lists $switches] }
		{ help } \
		    { set reply_text [listserv-help $switches] }
		default \
		    { set reply_text [listserv-bad-command $switches] }
	}

	cd $olddir

	setfield response STRING "[listserv-header]$reply_text"
	return [mailout [turnaround $envelope] $response]
}

##############################################################################
#
#	subscribe mlist [realname...]
#
##############################################################################

proc listserv-subscribe { args } {
	upvar envelope env

	set args [lindex $args 0]

	if {[llength $args] < 2} \
	{
		return [listserv-error-message $args "syntax error" \
			"[lindex args 0] mailing-list \[realname...\]" \
			"[lindex args 0] free-widgets-announce Bill Clinton"]
	}

	set mlist [string tolower [lindex $args 1]]
	set username [getfield $env REPLYTO]
	if {[llength $args] == 2} \
		{set realname "???"} \
		{set realname [lrange $args 2 end]}

	if {! [file exists $mlist.list]} \
	{
		return [listserv-error-message $args \
			"no local mailing list '$mlist'" \
			"[lindex args 0] mailing-list \[realname...\]" \
			"[lindex args 0] free-widgets-announce Bill Clinton"]
	}

	set found [listserv-find-address-in-mlist $username $mlist]
	if {$found != ""} \
	{
		return [listserv-error-message $args \
			"'$found' already on mailing list '$mlist'" \
			"[lindex args 0] mailing-list \[realname...\]" \
			"[lindex args 0] free-widgets-announce Bill Clinton"]
	}
	listserv-add-address-to-mlist $username $realname $mlist
	set r "'$username' added to the '$mlist' mailing list"
	if {[file readable $mlist.welcome]} \
		{append r "\n\n" [exec cat $mlist.welcome]}
	return $r
}

##############################################################################
#
#	subscribe-address mlist address [realname...]
#
##############################################################################

proc listserv-subscribe-address { args } {
	upvar envelope env

	set args [lindex $args 0]

	if {[llength $args] < 3} {
	    return [listserv-error-message $args "syntax error" \
	        "[lindex args 0] mailing-list address \[realname...\]" \
	        "[lindex args 0] free-widgets-bugs the-pres@gov Bill Clinton"]
	}
	set mlist [string tolower [lindex $args 1]]
	set username [lindex $args 2]
	if {[llength $args] == 3} \
		{set realname "???"} \
		{set realname [lrange $args 3 end]}

	if {! [file exists $mlist.list]} \
	{
		return [listserv-error-message $args \
			"no local mailing list '$mlist'" \
		        "[lindex args 0] mailing-list address \[realname...\]" \
		        "[lindex args 0] free-widgets-bugs the-pres@gov Bill Clinton"]
	}

	set found [listserv-find-address-in-mlist $username $mlist]
	if {$found != ""} \
	{
		return [listserv-error-message $args \
			"'$found' already on mailing list '$mlist'" \
		        "[lindex args 0] mailing-list address \[realname...\]" \
		        "[lindex args 0] free-widgets-bugs the-pres@gov Bill Clinton"]
	}
	listserv-add-address-to-mlist $username $realname $mlist
	set r "'$username' added to the '$mlist' mailing list"
	if {[file readable $mlist.welcome]} \
		{append r "\n\n" [exec cat $mlist.welcome]}
	return $r
}

##############################################################################
#
#	unsubscribe mlist
#
##############################################################################

proc listserv-unsubscribe { args } {
	set args [lindex $args 0]

	if {[llength $args] != 2} \
	{
		return [listserv-error-message $args "syntax error" \
			"[lindex args 0] mailing-list" \
			"[lindex args 0] free-widgets-announce"]
	}

	set mlist [string tolower [lindex $args 1]]
	set username [getfield $env REPLYTO]

	if {! [file exists $mlist.list]} \
	{
		return [listserv-error-message $args \
			"no local mailing list '$mlist'" \
			"[lindex args 0] mailing-list" \
			"[lindex args 0] free-widgets-announce"]
	}

	set found [listserv-find-address-in-mlist $username $mlist]
	if {$found == ""} \
	{
		return [listserv-error-message $args \
			"no address '$username' in mailing list '$mlist'" \
			"[lindex args 0] mailing-list" \
			"[lindex args 0] free-widgets-announce"]
	}

	set d [listserv-remove-address-from-mlist $username $mlist]
	return "these addresses removed from the '$mlist' mailing list:\n\n$d"
}

##############################################################################
#
#	recipients mlist
#
##############################################################################

proc listserv-recipients { args } {
	set args [lindex $args 0]

	if {[llength $args] != 2} \
	{
		return [listserv-error-message $args "syntax error" \
			"[lindex args 0] mailing-list" \
			"[lindex args 0] free-widgets-announce"]
	}

	set mlist [string tolower [lindex $args 1]]

	if {! [file exists $mlist.list]} \
	{
		return [listserv-error-message $args \
			"no local mailing list '$mlist'" \
			"[lindex args 0] mailing-list" \
			"[lindex args 0] free-widgets-announce"]
	}

	set m [exec cat $mlist.list]
	return "these are the members of the '$mlist' mailing list:\n\n$m"
}

##############################################################################
#
#	information mlist
#
##############################################################################

proc listserv-information { args } {
	set args [lindex $args 0]

	if {[llength $args] != 2} \
	{
		return [listserv-error-message $args "syntax error" \
			"[lindex args 0] mailing-list" \
			"[lindex args 0] free-widgets-announce"]
	}

	set mlist [string tolower [lindex $args 1]]

	if {! [file exists $mlist.list]} \
	{
		return [listserv-error-message $args \
			"no local mailing list '$mlist'" \
			"[lindex args 0] mailing-list" \
			"[lindex args 0] free-widgets-announce"]
	}

	if {! [file readable $mlist.info]} \
		{ return "I have no info about the '$mlist' mailing list" } \
		{ return "info about '$mlist':\n\n[exec cat $mlist.info]" }
}

##############################################################################
#
#	lists
#
##############################################################################

proc listserv-lists { args } {
	set args [lindex $args 0]

	if {[llength $args] != 1} \
	{
		return [listserv-error-message $args "syntax error" \
			"[lindex args 0]" "[lindex args 0]"]
	}

	set text "local mailing lists:\n\n"
	foreach l [glob {*.list}] {
		regsub "(.*)\.list" $l {\1} basename
		append text " $basename"
		if [file readable $basename.oneliner] {
			append text " ([exec cat $basename.oneliner])" }
		append text "\n"
	}
	return "$text"
}

##############################################################################
#
#	help
#
##############################################################################

proc listserv-help { args } {
	set man ~/src/man/listserv.man
	if {[file readable $man]} \
		{ return "help for listserv:\n\n[exec cat $man]" } \
		{ return "sorry, I have no help file for listserv" }
}

##############################################################################
#
# bad command handler
#
##############################################################################

proc listserv-bad-command {args} \
{
	set args [lindex $args 0]

	return [listserv-error-message $args \
		"unknown command '[lindex $args 0]'" \
		"<listserv-command> <listserv-command-args>*" \
		"subscribe free-widgets-announce Bill Clinton"]
}

##############################################################################
#
# some reply message formatting helpers
#
##############################################################################

proc listserv-header { } {
	return ""
}

proc listserv-error-message { command error usage example } {
	append r "       error: ${error}\n\n"
	append r "your command: ${command}\n\n"
	append r "       usage: ${usage}\n"
	append r "     example: ${example}\n"

	return $r
}

##############################################################################
#
# code to manipulate mailing list files
#
##############################################################################

proc listserv-find-address-in-mlist { address mlist } {
	set in [open $mlist.list r]
	set line_found ""
	while {! [eof $in]} \
	{
		gets $in line
		if [regexp -nocase "$address (.*)" $line] \
		{
			set line_found $line ; break
		}
	}
	close $in
	return $line_found
}

proc listserv-add-address-to-mlist { address realname mlist } {
	set out [open $mlist.list a]
	puts $out [format "%s (%s)" $address $realname]
	close $out
}

proc listserv-remove-address-from-mlist { address mlist } {
	set in [open $mlist.list r]
	set out [open $mlist.list.before-delete w]
	set lines_deleted = ""
	while {! [eof $in]} \
	{
		gets $in line
		if {![regexp -nocase "$address (.*)" $line]} \
		{
			puts $out $line
			append lines_deleted $line
		}
	}
	close $in ; close $out ; exec cp $mlist.list.before-delete $mlist.list
	return $lines_deleted
}
