/* export.c  1994 may 19  [gh]
+-----------------------------------------------------------------------------
| Abstract:
|    Function to generate MIME headers.
|
| History:
|    2.0 94 may 19 [gh] Release of version 2.0
|    1.1 94 feb 01 [gh] Improved documentation and MIME support.
|    1.0 94 jan 03 [gh] Wrote first version.
|
| Authorship:
|    Copyright (c) 1994 Gisle Hannemyr.
|    Permission is granted to hack, make and distribute copies of this program
|    as long as this copyright notice is not removed.
|    Flames, bug reports, comments and improvements to:
|       snail: Gisle Hannemyr, Brageveien 3A, 0452 Oslo, Norway
|       email: gisle@oslonett.no
+---------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "mimelite.h"

/*---( globals )------------------------------------------------------------*/

static int XPos;    /* Position om o/p line (not to exceed 76 positions).   */
static int BPos;    /* Postition in BASE64 o/p buffer.			    */
static unsigned char BBuf[3];


/*---( constants )----------------------------------------------------------*/

static char BaseTable[64] = {
    'A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P',
    'Q','R','S','T','U','V','W','X','Y','Z','a','b','c','d','e','f',
    'g','h','i','j','k','l','m','n','o','p','q','r','s','t','u','v',
    'w','x','y','z','0','1','2','3','4','5','6','7','8','9','+','/'
};


/*---( export )-------------------------------------------------------------*/

/*
| Abs: Check whether character need to be quoted.
| Ret: Zero if we may need to quote, else one.
*/ 
static int plainp(int cc)
{
    return(((cc >= 33) && (cc <= 60)) || ((cc >= 62) && (cc <= 126)));
} /* plainp */


/*
| Abs: Do the quoted-printable encoding.
| Ref: The rule references are to RFC-1521, sec. 5.1.
| Des: In the worst case, we may need to inject up to 4 characters, so to make
|      sure lines stay within the 76 character limit, we inject soft line
|      breaks at 72 characters.
*/ 
static void quotedp(int cc, int (*nextc)(void), FILE *stream)
{
    unsigned char tmp[10];
    int dd;

    if (cc == '\n') {
	fputc('\n', stream);
	XPos = 0;
	return;
    } else if ((XPos >= 75) || ((!plainp(cc)) && (XPos >= 73))) { /* Rule #5 */
	fputs("=\n", stream);
	XPos = 0;
    }

    if (plainp(cc)) {
	fputc(cc, stream);
	XPos++;
    } else if ((cc == 9) || (cc == 32)) {
	dd = (*nextc)();
	if ((dd == '\n') || (dd == '\r') || (dd == '\0')) {
	    fputc(cc, stream);
	    fputs("=\n", stream);
	    XPos = 0;
	} else {
	    fputc(cc, stream);
	    XPos++;
	}
	dd = ml_foldchar(dd);
	quotedp(dd, nextc, stream);
    } else if (cc == 61) {
	fputs("=3D", stream);
	XPos += 3;
    } else {
	sprintf(tmp, "=%02X", cc);
	fputs(tmp, stream);
	XPos += 3;
    } /* if (all possible char values) */
} /* quotedp */


/*
| Abs: Flush base64 encoded buffer.
*/ 
static void encodebase64(FILE *stream)
{
    int ii, lim;
    unsigned char outbuf[4];

    switch (BPos) {
      case 0: return;
      case 1: lim = 2; break;
      case 2: lim = 3; break;
      case 3: lim = 4; break;
    } /* switch */

    outbuf[0] =  (BBuf[0] & 0xFC) >> 2;
    outbuf[1] = ((BBuf[0] & 0x03) << 4) | ((BBuf[1] & 0xF0) >> 4);
    outbuf[2] = ((BBuf[1] & 0x0F) << 2) | ((BBuf[2] & 0xC0) >> 6);
    outbuf[3] =   BBuf[2] & 0x3F;
    for (ii = 0; ii < lim; ii++) fputc(BaseTable[outbuf[ii]], stream);
    for (ii = lim; ii < 4; ii++) fputc('=', stream);
/*  fprintf(stream, "[%d]", BPos); ** DB */

    BPos  = 0;
    XPos += 4;
    if (XPos >= 76) {
	fputc('\n', stream);
	XPos = 0;
    } /* if */
} /* encodebase64*/


/*
| Abs: Change a complete message into MIME encoding.
| Par: headcte = head contents transfer encoding to use (see Des: field)
|      bodycte = body contents transfer encoding to use (see Des: field)
|      bodycst = the minimum character set needed for the body.
|      nextchr = pointer to a function (supplied by caller) to get next
|                character of message to be encoded
|      ostream = stream to write encoded message to
| Des: - If headcte is the special value CE_STRIP_7, then the header should be
|        stripped to seven bits. If headcte is CE_UNCODED, then the header is
|        exported unchanged, If headcte is the special value CE_NO_HEAD, then
|	 there is no heading and the function will assume that the first byte
|        returned by nextchr is the first byte of the body.  No other values
|        for headcte are at the moment legal (i.e. mimelite does _not_ support
|        exporting messages with RFC-1522-type encoded headers).  You should
|        use CE_NO_HEAD if you want to process the header yourself (e.g. when
|        exporting binary files using BASE64 encoding, CE_UNCODED if you
|        believe that 8-bit transport is available, and CE_STRIP_7 otherwise.
|      - The combination where bodycst is CS_IR002 (7 bit ASCII) and bodycte
|        is CE_QUOTEDP is legal, but is considered bad form.  If the message
|        only use 7 bit characters, bodycte should be CE_UNCODED.
*/
void ml_mimemessage(int headcte, int bodycte, int bodycst,
		    int (*nextchr)(void), FILE *ostream)
{
    int cc;
    int inhead, wasanl;


    inhead = headcte != CE_NO_HEAD;
    wasanl = 0; /* To detect end of header.   */
    XPos   = 0; /* Position on line.	      */
    BPos   = 0; /* Position in BASE64 buffer  */

    while (cc = (*nextchr)()) {
        if (inhead && ('\n' == cc)) {
	    if (wasanl) {
		inhead = 0;
		fputs("Mime-Version: 1.0\nContent-Type: text/plain; charset=", ostream);
		switch (bodycst) {
		  case CS_IR002:   fputs("us-ascii",	     ostream); break;
		  case CS_ISOL1:   fputs("iso-8859-1",	     ostream); break;
		  default      :   fputs("x-unknown",	     ostream);
		} /* switch */
		fputs("\nContent-Transfer-Encoding: ", ostream);
		switch (bodycte) {
		  case CE_UNCODED: fputc(bodycst == CS_IR002 ? '7' : '8', ostream);
				   fputs("bit",		     ostream); break;
		  case CE_QUOTEDP: fputs("quoted-printable", ostream); break;
		  case CE_BASE064: fputs("base64",	     ostream); break;
		  default        : fputs("x-unknown",	     ostream);
		} /* switch */
		fputc('\n', ostream);
		fputc('\n', ostream);
		if (!(cc = (*nextchr)())) break; /* first char of body */
	    } else wasanl = 1;
	} else {
	    if ('\r' != cc) wasanl = 0;
	} /* if .. else */

	if (inhead) {	/* heading */

	    if (CE_STRIP_7 == headcte) cc = ml_foldcto7(cc);
	    fputc(cc, ostream);

	} else {	/* body    */

	    cc = ml_foldchar(cc);
	    switch (bodycte) {
	      case CE_UNCODED: fputc(cc, ostream);	      break;
	      case CE_QUOTEDP: quotedp(cc, nextchr, ostream); break;
	      case CE_BASE064: BBuf[BPos++] = cc;
			       if (BPos == 3) encodebase64(ostream);
			       break;
	      default        : fputc('?', ostream);
	    } /* switch */

	} /* if (inhead) proces head; else process body; */
    } /* while */
    if (bodycte == CE_BASE064) {
	encodebase64(ostream);
	fputc('\n', ostream);
    } /* if (CE_BASE064) */

} /* ml_mimemessage */


/* EOF */
