/*
 * snprintf.c -- formatted output to a string
 *
 * This is a simple version of snprintf() for systems
 * that don't have a native version.  This is a simple
 * version using temporary files.
 *
 * Returns the number of bytes that would have been output by printf.
 * Does not check whether a negative value if passed in for n
 * (as it's unsigned); some implementations cast n to int and than
 * compare with 0.
 *
 * Originally written by Casper Dik (Casper.Dik@Holland.Sun.COM)
 */

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>

int
snprintf (char *buf, size_t n, const char *fmt, ...)
{
    va_list ap;
    int ret;

    va_start(ap, fmt);
    ret = vsnprintf(buf, n, fmt, ap);
    va_end(ap);
    return ret;
}

static char template[] = "/tmp/snprintfXXXXXX";

int
vsnprintf (char *buf, size_t n, const char *fmt, va_list ap)
{
    char templ[sizeof(template)];
    int s;
    int ret, cnt = 0, nread;
    FILE *fp;

    strcpy(templ,template);

    if ((s = mkstemp(templ)) < 0)
	return -1;

    (void) unlink(templ);

    if ((fp = fdopen(s, "w+")) == NULL) {
	close (s);
	return -1;
    }

    ret = vfprintf(fp, fmt, ap);

    if (ret < 0 || fseek(fp, 0, SEEK_SET) < 0) {
	fclose(fp);
	return -1;
    }

    /*
     * read at most n-1 characters, since
     * we need to leave last space for '\0'
     */
    while (cnt < (n-1) && (nread = fread(buf + cnt, 1, n - cnt - 1, fp)) > 0)
	cnt += nread;

    buf[cnt] = '\0';  /* cnt is at most n-1 */

    fclose(fp);
    return ret;
}

