/*
   ``Sort of SPS for OSF/1'' -- a `ps' that shows process descendancy info

   By Matti Aarnio <mea@nic.funet.fi>  1996

*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <values.h> /* MAXINT et.al. */
#include <sys/table.h>
#include <sys/user.h>
#include <alloca.h>
#include <malloc.h>
#include <errno.h>
#include <pwd.h>

extern void idcache_init();
extern void idcache_read();
extern char *id_ttyname();
extern char *id_uidname();

struct tbl_sysinfo  tb_sysinfo;
struct tbl_tblstats tb_tblstats;
struct tbl_procinfo *tb_procs;
int exiting = 0, zombies = 0;

/* Reuse couple variables.. */
#define pi_CHILD    pi_jobc
#define pi_SIBLING  pi_cursig

int option_c = 0; /* show only the saved command name, not argv!   */
int option_l = 0; /* lots of meaningless process oriented info     */
int option_u = 0; /* show times as user-time only, not user+system */
char *option_uname = NULL;
int option_uuid = -1;
int option_pid = -1;
int option_i = 0; /* initialize /etc/passwd, and /dev/ -tables */
int option_n = 0; /* numerical only */

void print_argv(pp,pid)
struct tbl_procinfo *pp;
int pid;
{
	char buf[8192];
	int buflen = 8192;
	int rc;

	memset(buf,0,buflen);
	rc = table(TBL_ARGUMENTS,pid,buf,1,buflen);
	if (rc == 1) {
	  for (rc = 0; rc < buflen; ++rc) {
	    if (buf[rc] == 0) {
	      if (buf[rc+1]==0)
		break;
	      buf[rc] = ' ';
	    }
	  }
	  buf[buflen-1] = 0;
	  for (rc = strlen(buf)-1; rc >= 0; --rc)
	    if (buf[rc] == ' ') buf[rc] = 0;
	    else break;
	  printf("%s",buf);
	} else {
	  printf("(%s)",pp->pi_comm);
	}
}

void errtrap(rc,str)
int rc;
char *str;
{
	if (rc == -1) {
	  fprintf(stderr,"sps: %s lookup failed with error %s\n",str,
		  strerror(errno));
	  exit(1);
	}
}

char *pi_status_strings[] = { "<empty>", "RUN", "EXIT", "Zombie" };

static int pi_compare(p2, p1)
struct tbl_procinfo *p2, *p1;
{
	int rc;

	rc = p2->pi_ppid - p1->pi_ppid;
	if (rc == 0)
	  rc = p2->pi_pid - p1->pi_pid;

/* printf("cmp: (%5d,%5d) - (%5d,%5d) -> rc = %d\n",
   p2->pi_ppid,p2->pi_pid,
   p1->pi_ppid,p1->pi_pid, rc); */

	return rc;
}

char *pi_timestr(tv,urc)
struct timeval tv;
int urc;
{
	static char buf[30];
	char *fmt;

	if (urc != 1) {
	  strcpy(buf," **:**.***");
	  return buf;
	}
	if (tv.tv_sec > (60*99))
	  fmt = "%d:%02d.%03d"; /* over 99 minutes.. */
	else
	  fmt = " %02d:%02d.%03d";
	sprintf(buf,fmt,
		tv.tv_sec / 60, tv.tv_sec % 60,
		(tv.tv_usec+500)/1000 /* millisecs */);
	return buf;
}

void print_proc(pp, prefix)
struct tbl_procinfo *pp;
char *prefix;
{
	int i, urc;
	struct user u;
	struct timeval rutim;
	long ru_maxrss, ru_ixrss, ru_idrss, ru_isrss, tds_sum;

	urc = table(TBL_UAREA, pp->pi_pid, &u, 1, sizeof(u));
	if (urc == 1) {

	  ru_maxrss = u.u_ru.ru_maxrss;
	  ru_ixrss = u.u_ru.ru_ixrss;
	  ru_idrss = u.u_ru.ru_idrss;
	  ru_isrss = u.u_ru.ru_isrss;

	  tds_sum  = ( /* u.u_tsize + */ u.u_dsize + u.u_ssize) * 16;

	  if (option_u)
	    rutim = u.u_ru.ru_utime;
	  else {
	    rutim = u.u_ru.ru_utime;
	    rutim.tv_sec  += u.u_ru.ru_stime.tv_sec;
	    rutim.tv_usec += u.u_ru.ru_stime.tv_usec;
	    if (rutim.tv_usec > 1000000) {
	      rutim.tv_sec += 1;
	      rutim.tv_usec -= 1000000;
	    }
	  }
	} else {
	  rutim.tv_sec = 0;
	  rutim.tv_usec = 0;
	  ru_maxrss = ru_ixrss = ru_idrss = ru_isrss = 0;
	}

	if (option_l) {
	  printf("%8s %5d %5d %7s %08x %-6s %s ",
		 id_uidname(pp->pi_uid, option_n),
		 pp->pi_pid,
		 pp->pi_ppid,
		 id_ttyname(pp->pi_ttyd, option_n),
		 pp->pi_flag,
		 pi_status_strings[pp->pi_status],
		 pi_timestr(rutim,urc));
	  /* printf("%6d %6d %6d %6d ",
	     ru_maxrss, ru_ixrss, ru_idrss, ru_isrss); */
	  printf("%6dk %6dk ",
		  tds_sum, ru_maxrss);
	} else {
	  printf("%8s %5d %s ",
		 id_uidname(pp->pi_uid,option_n),
		 pp->pi_pid,pi_timestr(rutim,urc));
	}
	printf("%s",prefix);
	if (option_c)
	  printf("%s",pp->pi_comm);
	else
	  print_argv(pp,pp->pi_pid);
	printf("\n");
	fflush(stdout);
}

void print_proc_tree(ppa,idx,prefix)
struct tbl_procinfo *ppa;
int idx;
char *prefix;
{
	int len = strlen(prefix);
	int once = 1;

	while (idx >= 0) {
	  if (ppa[idx].pi_uid >= 0)
	    print_proc(&ppa[idx], prefix);
	  ppa[idx].pi_uid = -3;

	  if (once && len >= 3 && prefix[len-2]=='-') {
	    once = 0;
	    if (prefix[len-3] == '\\')
	      prefix[len-3] = ' ';
	    prefix[len-2] = ' ';
	    prefix[len-1] = ' ';
	  }

	  /* print child processes */
	  if (ppa[idx].pi_CHILD != idx) {
	    char *newpathprefix = (char*)alloca(len+4);
	    if (newpathprefix != NULL)
	      if (ppa[idx].pi_SIBLING >= 0)
		sprintf(newpathprefix,"%s%s",prefix,"|--");
	      else
		sprintf(newpathprefix,"%s%s",prefix,"\\--");
	    else
	      newpathprefix="*";
	    print_proc_tree(ppa,ppa[idx].pi_CHILD, newpathprefix);
	  }
	  /* print brother processes */
	  idx = ppa[idx].pi_SIBLING;
	}
}

void search_proc_tree(ppa,idx) /* uses external criteria */
struct tbl_procinfo *ppa;
int idx;
{
	int i;

	while (idx >= 0) {

	  if (option_uuid >= 0 &&
	      option_uuid == ppa[idx].pi_uid) {
	    i = (int)ppa[idx].pi_SIBLING;
	    ppa[idx].pi_SIBLING = -1;
	    print_proc_tree(ppa, idx, "");
	    ppa[idx].pi_uid = -3; /* printed, don't come to this anymore.. */
	    ppa[idx].pi_SIBLING = i;
	  } else {
	    /* something else .. */
	  }

	  /* search child processes */
	  if ((int)ppa[idx].pi_CHILD != idx) {
	    search_proc_tree(ppa,(int)ppa[idx].pi_CHILD);
	  }
	  /* search brother processes */
	  idx = (int)ppa[idx].pi_SIBLING;
	}
}

int proc_find_pid(ppa,cnt,pid)
struct tbl_procinfo *ppa;
int cnt, pid;
{
	int i;
	for (i = 0; i < cnt; ++i)
	  if (ppa[i].pi_pid == pid)
	    return i;
	return -1;
}

void proc_sibling_link(ppa,cnt)
struct tbl_procinfo *ppa;
int cnt;
{
	int pppid = -1; /* previous ppid */
	int i, j;

	for (i = 0; i < cnt; ++i) {
	  if (ppa[i].pi_ppid != pppid) {
	    pppid = ppa[i].pi_ppid;
	    j = proc_find_pid(ppa,cnt,pppid);
	    if (j >= 0)
	      ppa[j].pi_CHILD = i;
	  } else {
	    ppa[i-1].pi_SIBLING = i;
	  }
	}
}

void proc_report(procs, pused, palloc)
struct tbl_procinfo *procs;
int pused, palloc;
{
	int pid_idx = 0;
	if (option_pid >= 0) {
	  pid_idx = proc_find_pid(procs,pused,option_pid);
	  if (pid_idx < 0) {
	    printf("Sorry, pid=%d not around\n",option_pid);
	    return;
	  } else {
	    procs[pid_idx].pi_SIBLING = -1;
	  }
	}

	if (option_l)
	  printf("     UID   PID  PPID     TTY    Flags Status    CPUtime SWPSIZE     RSS CMD\n");
	else
	  printf("     UID   PID    CPUtime CMD\n");
	/* All processes are childs of the INIT,
	   and INIT is the first process.. */
	if (option_uuid >= 0)
	  search_proc_tree(procs,pid_idx);
	else
	  print_proc_tree(procs,pid_idx,"");
	printf("Sum: %d procs %d exiting %d zombies\n",
	       pused, exiting, zombies);
}


void read_procinfo(count, highindex)
int count, highindex;
{
#define PROCS_SET 128
	struct tbl_procinfo tb_pset[PROCS_SET];
	int i, setbase, rc, cnt;
	struct tbl_procinfo *procs;

	cnt = exiting = zombies = 0;

	tb_procs = (void*)malloc(sizeof(struct tbl_procinfo) * count);
	cnt = 0;
	setbase = 0;
	while (setbase < highindex) {
	  rc = table(TBL_PROCINFO, setbase,
		     &tb_pset[0], PROCS_SET, sizeof(struct tbl_procinfo));
	  errtrap(rc,"procinfo");
	  setbase += PROCS_SET;
	  for (i=0; i < PROCS_SET; ++i) {
	    if (tb_pset[i].pi_status != PI_EMPTY && cnt < count ) {
	      memcpy(&tb_procs[cnt],&tb_pset[i],sizeof(tb_pset[0]));
	      ++cnt;
	    }
	  }
	}

	/* Count them, and collect pointers */
	procs = tb_procs;
	for (i = 0; i < count; ++i, ++procs) {
	  if (procs->pi_status == PI_EXITING)
	    ++exiting;
	  else if (procs->pi_status == PI_ZOMBIE)
	    ++zombies;
	  procs->pi_CHILD   = -1;
	  procs->pi_SIBLING = -1;
	}
	/* Sort them */
	qsort(tb_procs,count,sizeof(*procs),pi_compare);
	proc_sibling_link(tb_procs,count);
}

int get_uidbyuname(uname)
char *uname;
{
	struct passwd *pw = getpwnam(uname);
	if (!pw) return -1;
	return pw->pw_uid;
}

void usage()
{
	fprintf(stderr,"sps: [-clu][-p pid][-U uname]  -- ``Sort of SPS for Solaris''\n");
	fprintf(stderr,"  -c    -- show saved command name, not full argv\n");
	fprintf(stderr,"  -l    -- show lots of details about process\n");
	fprintf(stderr,"  -u    -- show accumulated user+childs time\n");
	fprintf(stderr,"  -p pid -- show all processes that are childs of this process\n");
	fprintf(stderr,"  -U uname -- all processes of this user\n");
	fprintf(stderr,"  -n    -- show devs and uids in numeric format\n");
	fprintf(stderr,"  -i    -- initialize /etc/passwd, and /dev/ -tables\n");
	exit(64);
}

int main(argc,argv)
int argc;
char *argv[];
{
	int rc;
	int c;

	while ((c = getopt(argc,argv,"?cilnup:U:")) != -1) {
	  switch (c) {
	    case 'c':
	        option_c = 1;
		break;
	    case 'i':
		option_i = 1;
		break;
	    case 'l':
		option_l = 1;
		break;
	    case 'n':
		option_n = 1;
		break;
	    case 'u':
		option_u = 1;
		break;
	    case 'p':
		option_pid = atoi(optarg);
		break;
	    case 'U':
		option_uname = optarg;
		option_uuid = get_uidbyuname(option_uname);
		break;
	    case '?':
	    default:
		usage();
		break;
	  }
	}


	if (option_i)
	  idcache_init();
	else
	  idcache_read();

	rc = table(TBL_SYSINFO, 0, &tb_sysinfo, 1, sizeof(tb_sysinfo));
	errtrap(rc,"sysinfo");
	rc = table(TBL_TBLSTATS, 0, &tb_tblstats, 1, sizeof(tb_tblstats));
	errtrap(rc,"tblstats");
	/* Read the processes */
	read_procinfo(tb_tblstats.tb_procu,tb_tblstats.tb_proca);

	printf("sps: stats: procu %d proca %d\n",
	       tb_tblstats.tb_procu, tb_tblstats.tb_proca);

	/* Print them */
	proc_report(tb_procs, tb_tblstats.tb_procu, tb_tblstats.tb_proca);

	return 0;
}
