/*
 * Yacc grammar for NewsClip
 */

%{
#include "nc.h"
 /*
  * Newsclip(TM) Compiler Source Code.
  * Copyright 1989 Looking Glass Software Limited.  All Rights Reserved.
  * Unless otherwise licenced, the only authorized use of this source
  * code is compilation into a binary of the newsclip compiler for the
  * use of licenced Newsclip customers.  Minor source code modifications
  * are allowed before compiling.
  */
%}

/*
 * TERMINAL DECLARATIONS
 *
 */

%union {
	nodep	treeval;		/* maybe need tree and list types? */
	char	*strval;
	int	intval;
}

%token		YACCEPT		YARRAY		YBREAK
		YCASE		YCONTINUE	YPARSE
		YTDATABASE	YTDATE
		YDEFAULT	YDUALCASE	YELSE
		YEXTERN		YFOR		YGOTO
		YHEADER		YIF		YIN
		YHAS		YTINT
		YTNEWSGROUP	YREJECT
		YSWITCH 	YTUSERID
		YWHILE 		YILLCH 
		YADJUST		YPROCEDURE	
		YFORWARD	YRETURN		YTSTRING


%token 		EQ_OP	NE_OP	LE_OP	GE_OP
		INC_OP	DEC_OP	AND_OP	OR_OP

%token <intval>	YNEWSGROUP YINT	YQUERYGROUP

%token <strval>	YID		YSTRING

%type <treeval> argdec_list		variable	typelist
		argdec	statement labeled_statement	compound_statement
		statement_list simple_statement	assignment	routine
		selection_statement	forstat		fortest
		iteration_statement	jump_statement	explist	
		elist	primary_expr	postfix_expr	unary_expr
		cast_expr	multiplicative_expr	
		additive_expr	relational_expr	equality_expr	and_expr	
		exclusive_or_expr	inclusive_or_expr
		logical_and_expr	logical_or_expr	conditional_expr
		expr constant_expr	varident	routident
		routid		decl	localdecs
		local_decl	local_decl_list


%type <intval>	type simtype




%%
/*
 * PRODUCTIONS
 */

program:
	progel
		|
	program progel
		;

progel:
	decl	{
		extern int got_error;
		if( got_error < SYNTAX_ERROR ) {
			out( $1 );
			coutc( '\n' );
			}
		treefree( $1 );
		}
		|
	procedure
		|
	function
		|
	error {
		parerror( "Invalid declaration or routine" );
		}
		;
	


push_table:
	{ push_table(); }
		;
pop_table:
	{ pop_table(); }
		;


local_decl:
	YEXTERN type YID ';'
	{ $$ = extern_var( $3, $2 ); }
		|
	YEXTERN type YID '(' typelist ')' ';'
	{ $$ = extern_func( $3, (dtype)$2, (listp)$5, TRUE ); }
		|
	YEXTERN YPROCEDURE YID '(' typelist ')' ';'
	{ $$ = extern_func( $3, (dtype)0, (listp)$5, TRUE ); }
		|
	type YID ';'
	{ $$ = declare_var( $2, (dtype)$1 ); }
		|
	error ';'
		{
		parerror( "Invalid declaration" );
		$$ = NIL;
		}
		;
decl:
	YFORWARD type YID '(' typelist ')' ';'
	{ $$ = extern_func( $3, (dtype)$2, (listp)$5, FALSE ); }
		|
	YFORWARD YPROCEDURE YID '(' typelist ')' ';'
	{ $$ = extern_func( $3, (dtype)0, (listp)$5, FALSE ); }
		|
	YHEADER simtype YID ':' YSTRING ';' {
		$$ = declare_var( $3, (dtype)$2 ); 
		select_header( $5, $$, "" );
		}
		|
	YHEADER simtype YARRAY YID ':' YSTRING ',' YSTRING ';' {
		$$ = declare_var( $4, (dtype)(arrayof($2)) ); 
		select_header( $6, $$, $8 );
		}
		|
	local_decl
		;

local_decl_list:
	local_decl
	{ $$ = (nodep)freshlist( $1 ); }
		|
	local_decl_list local_decl
	{ $$ = (nodep)llist( $1, $2 ); }
		;



procedure:
	YPROCEDURE routid push_table '(' argdec_list ')' routine {
		routine_decl( $2, $5, $7, 0, ST_PROC );
		}
		pop_table
		;
function:
	type routid push_table '(' argdec_list ')' routine {
		routine_decl( $2, $5, $7, $1, ST_FUNC );
		}
		pop_table
		;

routid:
	YID
	{ $$ = gen_declare( $1 ); }
		;

typelist:
	type
	{ $$ = (nodep)freshlist( tree(N_INT, (nodep)$1 ) ); }
		|
	typelist ',' type
	{ $$ = (nodep)llist( $1, tree(N_INT, (nodep)$3 ) ); }
		|
	{ $$ = NIL; }
		;
		

argdec_list:
	argdec
	{ $$ = (nodep)freshlist( $1 ); }
		|
	argdec_list ',' argdec
	{ $$ = (nodep)llist( $1, $3 ); }
		|
	{ $$ = NIL; }
		;

argdec:
	type YID {
		$$ = declare_arg( $2, (dtype)$1 );
		}
		|
	error { 
		parerror( "Invalid argument declaration" );
		$$ = NIL;
		}
		;
		

type:
	simtype
		|
	simtype YARRAY
	{ $$ = arrayof($1); }
		|
	YTDATABASE
	{ $$ = T_DATABASE; }
		;
simtype:
	YTINT 
	{ $$ = T_INTEGER; }
	| YTDATE 
	{ $$ = T_DATE; }
	| YTUSERID 
	{ $$ = T_USERNAME; }
	| YTSTRING 
	{ $$ = T_STRING; }
	| YTNEWSGROUP
	{ $$ = T_NEWSGROUP; }
	;
	

routine:
	'{' localdecs statement_list '}'
		{
 		$$ = tree( N_ROUTINE, $2, $3 );
		}
		|
	'{' localdecs '}'
		{
 		$$ = tree( N_ROUTINE, $2, NIL );
		}
		;

localdecs:
	local_decl_list
		|
	{ $$ = NIL; }
		;

/*
 * And now for general statement
 */

statement:
	labeled_statement
		|
	compound_statement
		|
	simple_statement
		|
	selection_statement
		|
	iteration_statement
		|
	jump_statement
		|
	error ';' {
		parerror( "statement syntax error" );
		$$ = NIL;
		}
		;

labeled_statement:
	YID ':' statement {
		$$ = tree( N_LABELED, declare_lab( $1 ), $3 );
		}
		|
	YCASE constant_expr ':' statement {
		$$ = tree( N_CASE, $2, $4 );
		}
		|
	YDEFAULT ':' statement {
		$$ = tree( N_DEFAULT, $3 );
		}
		;

compound_statement:
	'{' '}' {
		$$ = tree( N_COMPOUND, NIL );
		}
		|
	'{' statement_list '}' {
		$$ = tree( N_COMPOUND, $2 );
		}
		|
	'{' error '}' {
		parerror( "Invalid compound statement" );
		$$ = NIL;
		}
		;

statement_list:
	statement
	{ $$ = (nodep)freshlist( $1 ); }
		|
	statement_list statement {
		$$ = (nodep)llist( $1, $2 );
		}
		;
simple_statement:
	assignment ';'
		{ $$ = tree( N_SEMISTAT, $1 ); }
		|
	YADJUST expr ';' {
		$$ = tree( N_ADJUST, $2 );
		}
		|
	routident '(' explist ')' ';' {
		$$ = tree( N_CALL, $1, $3 );
		}
		;

assignment:
	variable '=' expr
		{ $$ = tree( N_ASSIGN, $1, $3 ); }
		|
	YPARSE variable '=' expr
		{ $$ = tree( N_PARSE, $2, $4, NIL ); }
		|
	YPARSE variable '=' expr ',' expr
		{ $$ = tree( N_PARSE, $2, $4, $6 ); }
		|
	variable '=' YARRAY expr
		{ $$ = tree( N_ARINIT, $1, $4 ); }
		|
	variable INC_OP
		{ $$ = tree( N_POSTINC, $1 ); }
		|
	variable DEC_OP
		{ $$ = tree( N_POSTDEC, $1 ); }
		|
	INC_OP variable
		{ $$ = tree( N_PREINC, $2 ); }
		|
	DEC_OP variable
		{ $$ = tree( N_PREDEC, $2 ); }
		|
		{ $$ = NIL; }
		;

selection_statement:
	YIF '(' expr ')' statement {
		$$ = tree( N_IF, $3, $5 );
		}
		|
	YIF '(' expr ')' statement YELSE statement {
		$$ = tree( N_IFELSE, $3, $5, $7 );
		}
		|
	YSWITCH '(' expr ')' compound_statement {
		$$ = tree( N_SWITCH, $3, $5 );
		}
		;

forstat:
	assignment
		|
	unary_expr
		;
fortest:
	expr
		|
	{ $$ = NIL; }
		;


iteration_statement:
	YWHILE '(' expr ')' statement {
		$$ = tree( N_WHILE, $3, $5 );
		}
		|
	YFOR '(' forstat ';' fortest ';' forstat ')' statement {
		$$ = tree( N_FOR, $3, $5, $7, $9 );
		}
		|
	YFOR '(' variable YIN expr ')' statement {
		$$ = tree( N_FOREACH, $3, $5, $7 );
		}
		;

jump_statement:
	YGOTO YID ';' {
		$$ = tree( N_GOTO, goto_lookup($2) );
		}
		|
	YCONTINUE ';' {
		$$ = tree( N_CONTINUE );
		}
		|
	YBREAK ';' {
		$$ = tree( N_BREAK );
		}
		|
	YRETURN ';' {
		$$ = tree( N_RETURN, NIL);
		}
		|
	YRETURN expr ';' {
		$$ = tree( N_RETURN, $2 );
		}
		|
	YACCEPT ';' {
		$$ = tree( N_ACCEPT );
		}
		|
	YACCEPT YIF expr ';' {
		$$ = tree( N_IFACCEPT, $3 );
		}
		|
	YREJECT ';' {
		$$ = tree( N_REJECT );
		}
		|
	YREJECT YIF expr ';' {
		$$ = tree( N_IFREJECT, $3 );
		}
		;
		
explist:
	elist
		|
		{ $$ = NIL; }
		;

elist:
	expr
	{ $$ = (nodep)freshlist( $1 ); }
		|
	elist ',' expr {
		$$ = (nodep)llist( $1, $3 );
		}
		;

variable
	: varident
	| varident '[' expr ']'
	{ $$ = tree( N_INDEX, $1, $3 ); }
	;
	

primary_expr
	: variable
	| routident '(' explist ')'
	{ $$ = tree( N_FUNCALL, $1, $3 ); }
	| YINT
	{ $$ = tree( N_INT, (nodep)$1 ); }
	| YSTRING
	{ $$ = tree( N_STRING, (nodep)allocstring($1) ); }
	| YNEWSGROUP
	{ $$ = tree( N_NGROUP, (nodep)$1 ); }
	| YQUERYGROUP
	{ $$ = tree( N_QGROUP, (nodep)$1 ); }
	| '(' expr ')'
	{ $$ = tree( N_PAREN, $2 ); }
	;

postfix_expr
	: primary_expr
	;

unary_expr
	: postfix_expr
	| '-' cast_expr
	{ $$ = tree( N_UMINUS, $2 ); }
	| '!' cast_expr
	{ $$ = tree( N_NOT, $2 ); }
	;

cast_expr
	: unary_expr
	;

multiplicative_expr
	: cast_expr
	| multiplicative_expr '*' cast_expr
	{ $$ = tree( N_MULT, $1, $3 ); }
	| multiplicative_expr '/' cast_expr
	{ $$ = tree( N_DIVIDE, $1, $3 ); }
	| multiplicative_expr '%' cast_expr
	{ $$ = tree( N_MODULUS, $1, $3 ); }
	;

additive_expr
	: multiplicative_expr
	| additive_expr '+' multiplicative_expr
	{ $$ = tree( N_PLUS, $1, $3 ); }
	| additive_expr '-' multiplicative_expr
	{ $$ = tree( N_MINUS, $1, $3 ); }
	;


relational_expr
	: additive_expr
	| relational_expr '<' additive_expr
	{ $$ = tree( N_LT, $1, $3 ); }
	| relational_expr '>' additive_expr
	{ $$ = tree( N_GT, $1, $3 ); }
	| relational_expr LE_OP additive_expr
	{ $$ = tree( N_LE, $1, $3 ); }
	| relational_expr GE_OP additive_expr
	{ $$ = tree( N_GE, $1, $3 ); }
	;

equality_expr
	: relational_expr
	| equality_expr EQ_OP relational_expr
	{ $$ = tree( N_EQ, $1, $3 ); }
	| equality_expr NE_OP relational_expr
	{ $$ = tree( N_NE, $1, $3 ); }
	| equality_expr YIN relational_expr
	{ $$ = tree( N_IN, $1, $3 ); }
	| equality_expr '!' YIN relational_expr
	{ $$ = tree( N_NOT_IN, $1, $4 ); }
	| equality_expr YHAS relational_expr
	{ $$ = tree( N_HAS, $1, $3 ); }
	| equality_expr '!' YHAS relational_expr
	{ $$ = tree( N_NOT_HAS, $1, $4 ); }
	;

and_expr
	: equality_expr
	| and_expr '&' equality_expr
	{ $$ = tree( N_BAND, $1, $3 ); }
	;

exclusive_or_expr
	: and_expr
	| exclusive_or_expr '^' and_expr
	{ $$ = tree( N_XOR, $1, $3 ); }
	;

inclusive_or_expr
	: exclusive_or_expr
	| inclusive_or_expr '|' exclusive_or_expr
	{ $$ = tree( N_BOR, $1, $3 ); }
	;

logical_and_expr
	: inclusive_or_expr
	| logical_and_expr AND_OP inclusive_or_expr
	{ $$ = tree( N_AND, $1, $3 ); }
	;

logical_or_expr
	: logical_and_expr
	| logical_or_expr OR_OP logical_and_expr
	{ $$ = tree( N_OR, $1, $3 ); }
	;

conditional_expr
	: logical_or_expr
	| logical_or_expr '?' logical_or_expr ':' conditional_expr
	{ $$ = tree( N_QUERY, $1, $3, $5 ); }
	;

expr
	: conditional_expr
	| error {
		parerror( "Syntax error in expression" );
		$$ = tree( N_INT, (nodep)0 );
		}
	;

constant_expr
	: conditional_expr
	;

varident:
	YID {
	$$ = symlookup( $1, ST_VAR, 0 );
	}
	;
routident:
	YID {
	$$ = symlookup( $1, ST_FUNC, ST_PROC );
	}
	;
%%
