/* $Header: cursor.c,v 1.3 86/02/27 19:20:16 notes Exp $ */

/*
 * Terminal driving routines
 */

#include "parms.h"
#include "structs.h"

static char bufcap[256];
static char *CM;			/* cursor motion control */
static char *CL;			/* clear screen control */
static char *CE;			/* clear to end of line */
static char *IS;			/* initialization string */
static char *SO;			/* standout mode */
static char *SE;			/* end standout mode */
static char *TI;			/* initialization string for cm */
static char *TE;			/* initialization end string for cm */
static char *ML;			/* memory lock */
static char *MU;			/* memory unlock */
static int cur_row;
static int cur_col;
static int top_col;			/* last printable character for top */
static int bot_col;			/* same for the bottom line */

char *tgoto();				/* decodes the cursor via termcap */

/*
 * charout - output 1 character
 * used by termcap tputs routine
 */
charout(c)
{
	putchar(c);
}

/*
 *  at(row,col) places cursor at row,col
 *	row = 1 to 24    ( y coords )
 * 	column = 1 to 80   ( x coords )
 */
at(row, col)
{
	register char  *p;

	if (CM) {
		if (row <= 0)
			row += nrows;		/* wraparound - cute */
		if (col <= 0)
			col += ncols;
		p = tgoto(CM, col - 1, row - 1);
		tputs(p, 1, charout);
	} else {
		if (cur_row != row)
			putchar('\n');
		else
			putchar(' ');
	}
	cur_row = row;
	cur_col = col;
}

/*
 * erase() erases the screen
 */
erase()
{
	if (CL)
		tputs(CL, 1, charout);
	else
		printf("\n\n");
	cur_row = 1;			/* back in top of screen */
	cur_col = 1;			/* back in top of screen */
	top_col = 0;
	bot_col = 0;
}

clear_eol()
{
	int i;
	int last_col;

	/*
	 * No need to do anything if there are no characters to
	 * the right (cur_col > last_col).
	 *
	 * last_col is determined in a rather crude way.
	 * We actually only keep track of the rightmost position for
	 * the bottom line (bot_col) and the lines above it (top_col)
	 */
	last_col = (cur_row == nrows) ? bot_col : top_col;
	if (cur_col > last_col)
		return;
	if (CE)
		tputs(CE, 1, charout);
	else {
		for (i = cur_col; i <= last_col; i++)
			putchar(' ');
		for (i = cur_col; i <= last_col; i++)
			putchar('\b');
	}
	/* update the line position indicators */
	if (cur_row == nrows)
		bot_col = cur_col;
	/* can't update top_col because there are many possible lines */
}

/* 
 *	Brute-force version of above: don't be smart, just do what I say.
 *      (This is because column counters are not updated by printf.)
 */

ceol ()
{
	int i;

        if (CE != NULL)
	    tputs (CE, 1, charout);
	else {
		for (i = cur_col; i <= ncols; i++)
			putchar(' ');
		for (i = cur_col; i <= ncols; i++)
			putchar('\b');
	}
}

cursget(myterm)
	char *myterm;
{
	char *tgetstr();
	char *q;
	register int i;				/* rows, cols */
	char bp[1024];				/* termcap stuff */

	if (tgetent(bp, myterm) != 1)
		return(-1);
	q = bufcap;
	CM = tgetstr("cm", &q);			/* get the cursor motion */
	CL = tgetstr("cl", &q);			/* get the clear screen */
	CE = tgetstr("ce", &q);			/* get the clear to eol */
	IS = tgetstr("is", &q);
	SO = tgetstr("so", &q);
	SE = tgetstr("se", &q);
	TI = tgetstr("ti", &q);
	TE = tgetstr("te", &q);
        ML = tgetstr("ml", &q);			/* memory lock */
        MU = tgetstr("mu", &q);			/* memory unlock */

	if ((i = tgetnum("li")) != -1)
		nrows = i;				/* rows on screen */
	if (nrows > 24)
		nindex += nrows - 24;
	if ((i = tgetnum("co")) != -1)
		ncols = i;				/* cols on screen */
	if (IS)
		tputs(IS, 1, charout);
	if (CM && TI)
		tputs(TI, 1, charout);
	return(0);
}

cursend()
{
	at(0, 1);
	if (CM && TE)
		tputs(TE, 1, charout);
	clear_eol();
}

/*
 * get the coordinates of the cursor
 */
curpos(row, col)
int *row, *col;
{
	*row = cur_row;
	*col = cur_col;
}

/*
 * a putchar with no interpretation of characters
 * updates current cursor position
 */
putch(c)
{
	switch (c) {
	case '\t':
		cur_col = ((cur_col + 7) & ~07) + 1;
		break;
	case '\n':
		cur_row++;
		/* FALL INTO ... */
	case '\r':
		cur_col = 1;
		break;
	case '\b':
		if (cur_col > 1)
			cur_col--;
		break;
	default:
		if (c >= 040 && c < 0177)
			cur_col++;
		break;
	}
	putchar(c);
	/* update the line position indicators */
	if (cur_row == nrows) {
		if (cur_col > bot_col)
			bot_col = cur_col;
	} else if (cur_col > top_col)
		top_col = cur_col;
}

standout(flag)
{
	if (flag) {
		if (SO)
			tputs(SO, 1, charout);
		else
			putchar('\007');
	} else {
		if (SE)
			tputs(SE, 1, charout);
	}
}

/*
 * can_lock() tells us if we can use memory lock
 */
int can_lock()
{
	return (ML && MU);		/* only if both exist */
}


/*
 * m_lock() locks memory
 */
m_lock()
{
	if (ML)
		tputs(ML, 1, charout);
}

/*
 * m_ulock() unlocks memory
 */
m_ulock()
{
	if (MU)
		tputs(MU, 1, charout);
}
