/* $Header: /g1/users/staff/gore/exp/notes/src/lib/RCS/recsio.c,v 2.0 89/04/16 01:08:05 gore Exp $ */

/* The include for <sys/stat.h> must follow "structs.h", since the latter */
/* includes <types.h>, and some systems barf on multiple includes of      */
/* <types.h> (can't declare a typedef twice).                             */

#include <errno.h>
#include "parms.h"
#include "structs.h"
#include <time.h>
#include <sys/stat.h>
#ifdef SYSV
#  include <fcntl.h>
#endif

#ifdef BSD
#  ifndef __FCNTL_HEADER__
#    include <sys/fcntl.h>
#  endif
#endif BSD

#ifndef __STRING_POOL__
#  include "stringpool.h"
#endif

#ifndef O_RDWR
#  define O_RDWR	2
#endif  O_RDWR

/*
 *  init(io,p), finish(io)  struct io_f *io, char *p
 *    init opens the three i/o files and initializes session stats
 *
 *    finish(io) closes all those files.
 *
 *  getnrec, putnrec, getrrec, putrrec
 *  getdscr, putdscr, gettrec, puttrec
 *  each gets or puts physical records inside its appropriate file.
 *
*/

extern int   errno;
extern char *sys_errlist[];

long lseek();					/* for sake of lint */

init(io, p)
struct io_f *io;
char   *p;
{
    char fn[WDLEN];				/* build scratch names */
    char *getname();
    struct stat  stat_buf;
    int  stat_result;

    if (chkpath(p) != 0)
	return(QUITBAD);
    safecpy(io->nf, p, NNLEN);

    /* We check for the existance of the access file, since the directory */
    /*  itself might have been created in order to make a sub-newsgroup.  */

    sprintf(fn, "%s/%s/%s", spooldir, mapnfname(p), ACCESS);
    stat_result = stat(fn,&stat_buf);
    if(stat_result < 0) {
	switch(errno) {
	    case ENOTDIR:
		fprintf(stderr,
		"init: path contains non-directory, file= \"%s\", errno= %d\n",
			fn);
		printuid();
		break;
	    case ENOENT:
		if(verbose_mode)	/* Maybe it has been auto-deleted, */
		    fprintf(stderr,	/* so only complain in verbose mode */
		    "init: file does not exist, file= \"%s\", errno= %d\n",
		    fn, errno);
		break;
	    case EACCES:
		fprintf(stderr,
	    "init: cannot search directory in path, file= \"%s\", errno= %d\n",
		    fn, errno);
		printuid();
		break;
	    default:
		x(1,
	    "init: fatal error in parameters to stat(), fn= \"%s\", errno= %d",
		  fn, errno);
	}
	return(QUITNEX);
    }

    if (opennf(io, p) < 0)
	return(QUITBAD);			/* bad luck opening */

    getdscr(io, &io->descr);
    getperms(io, 0, getname(0));	       /* go establish access rights */

    io->nrspwrit = io->nnotwrit = 0;	/* set up stats */
    io->nrspread = io->nnotread = 0;
    io->nnotxmit = io->nrspxmit = 0;
    io->nnotrcvd = io->nrsprcvd = 0;
    io->nnotdrop = io->nrspdrop = 0;
    io->norphans = 0;
    io->xstring[0] = io->xauth[0] = '\0';	/* clear search strings */

    time(&io->entered);			/* get entry time */

    return(io->access);			/* return the access bits */
}

opennf(io, p)
struct io_f *io;
char *p;
{
    char fn[WDLEN];

#ifdef MSGID_INDEX
    datum vkey, version;
    char dbm_db_version[512]; /* comment holds our version number */

    sprintf(fn, "%s/%s/%s", spooldir, mapnfname(p), INDEXM);
    io->msgindexp = dbm_open(fn, O_RDWR, 0);
    if (io->msgindexp == NULL)
        return(-1);

    /* Check for version string in database */
    vkey.dptr = DB_VERS_KEY;
    vkey.size = strlen(DB_VERS_KEY) + 1;
    version = dbm_fetch(io->msgindexp, vkey);
    x(version.dptr == NULL,
      "opennf: msg-ID index has no version, nf= \"%s\"", p);
    if (strcmp(version.dptr, MSGID_INDEX_VERSION) !=0)
      {
	x(1,
	  "opennf: msg-ID index is wrong version, index= \"%s\", code= \"%s\"",
	  dbm_db_version, MSGID_INDEX_VERSION);
	/* OK, I should remake the index instead, but I'm tired. */
	/* Sorry. */
      }
#endif /* MSGID_INDEX */

    sprintf(fn, "%s/%s/%s", spooldir, mapnfname(p), TEXT);
    if ((io->fidtxt = fopen(fn, "r+")) == NULL) {
#ifdef MSGID_INDEX
        dbm_close(io->msgindexp);
#endif /* MSGID_INDEX */
	return(-1);					/* bad nf */
    }

    sprintf(fn, "%s/%s/%s", spooldir, mapnfname(p), INDEXN);
    if ((io->fidndx = open(fn, O_RDWR)) < 0) {
#ifdef MSGID_INDEX
        dbm_close(io->msgindexp);
#endif /* MSGID_INDEX */
	fclose(io->fidtxt);
	return(-1);
    }

    sprintf(fn, "%s/%s/%s", spooldir, mapnfname(p), INDEXR);
    if ((io->fidrdx = open(fn, O_RDWR)) < 0) {
#ifdef MSGID_INDEX
        dbm_close(io->msgindexp);
#endif /* MSGID_INDEX */
	fclose(io->fidtxt);
	close(io->fidndx);
	return(-1);					/* bad nf */
    }

    /*
     * Just figure out the name of the string.pool file; it will be
     * opened/closed later.
     */
    sprintf(fn, "%s/%s/%s", spooldir, mapnfname(p), STRPOOL);
    io->str_pool_fn = malloc(strlen(fn)+1);
    strcpy(io->str_pool_fn, fn);

    return(0);						/* all's well */
}


finish (io)
struct io_f *io;
{
    long left;
    time_t lvtime;				/* for days used */

    lock(io, 'n');					/* update statistics */
    getdscr(io, &io->descr);
    io->descr.d_notwrit += io->nnotwrit;
    io->descr.d_rspwrit += io->nrspwrit;
    io->descr.d_notread += io->nnotread;
    io->descr.d_rspread += io->nrspread;
    io->descr.d_notxmit += io->nnotxmit;
    io->descr.d_rspxmit += io->nrspxmit;
    io->descr.d_notrcvd += io->nnotrcvd;
    io->descr.d_rsprcvd += io->nrsprcvd;
    io->descr.d_notdrop += io->nnotdrop;
    io->descr.d_rspdrop += io->nrspdrop;
    io->descr.d_orphans += io->norphans;
    io->descr.entries++;			/* count of entries */
    time(&left);
    lvtime = left;
    io->descr.walltime += left - io->entered;   /* time spent in nf */
    if ((lvtime / DAYSECS) != (io->descr.d_lastuse / DAYSECS)) {
	io->descr.d_daysused++;
	io->descr.d_lastuse = lvtime;
    }
    putdscr(io, &io->descr);			/* update the block */
    unlock(io, 'n');

    closenf(io);
}

closenf(io)
struct io_f *io;
{
#ifdef MSGID_INDEX
    dbm_close(io->msgindexp);		/* never returns any status */
#endif /* MSGID_INDEX */

    x(fclose(io->fidtxt) == EOF,
        "finish: failed fclose of text file for %s",
	    io -> nf);
    x(close(io->fidndx) < 0,
        "finish: failed fclose of note index for %s",
	    io -> nf);
    x(close(io->fidrdx) < 0, 
        "finish: failed fclose of resp index for %s",
	    io -> nf);
}



/*
 * In all of {get,put}{n,r}rec below, explicitly cast one
 * of the members of the rhs of the "where" assignment
 * to long.  This prevents a problem in 16 bit machines
 * whereby
 *	sizeof (*descr) + n * sizeof (*note)
 * is evaulated thusly:
 *	<long> = <int> + <int> * <int>
 * i.e.	<long> = <int>
 * thus losing some of the information.  You will have this
 * problem on your machine as soon as "n * sizeof (*not)" exceeds
 * MAXINT  (or maybe MAXUNSIGNED.)
 */

/* n is the number of the note to get.  0 is policy note */
getnrec(io, n, note)
struct io_f *io;
int n;
struct note_f *note;
{
    long where;				/* going to seek here eventually */
    struct descr_f *descr;		/* for sizeof below */

    x(n < 0,
        "getnrec: got a negative record number (%d)", n);
    where = sizeof (*descr) + (long)n * sizeof (*note);

    x(lseek(io->fidndx, where, 0) < 0,
        "getnrec: lseek failed in note index for %s", io -> nf);

    x(read(io->fidndx, (char *)note, sizeof(*note)) < sizeof(*note),
	"getnrec: read failed in note index for %s", io -> nf);
}

/* n is the number of the note to put.  0 is policy note */
putnrec(io, n, note)
struct io_f *io;
int n;
struct note_f *note;
{
    long where;				/* going to seek here eventually */
    struct descr_f *descr;		/* for sizeof below */

    x(n < 0,
        "putnrec: got a negative record number (%d)", n);
    where = sizeof (*descr) + (long)n * sizeof (*note);
    x(lseek(io->fidndx, where, 0) < 0, 
        "putnrec: lseek failed in note index for %s", io -> nf);
    x(write(io->fidndx, (char *)note, sizeof(*note)) < sizeof(*note),
	"putnrec: write failed in note index for %s", io -> nf);
}

getdscr(io, descr)
struct io_f *io;
struct descr_f *descr;
{

    x(lseek(io->fidndx, 0L, 0) < 0, 
        "getdscr: lseek failed in note index for %s", io -> nf);
    x(read(io->fidndx, (char *)descr, sizeof(*descr)) < sizeof(*descr),
	"getdscr: read failed in note index for %s", io -> nf);
}

putdscr(io, descr)
struct io_f *io;
struct descr_f *descr;
{

    x(lseek(io->fidndx, 0L, 0) < 0, 
        "putdscr: lseek failed in note index for %s", io -> nf);
    x(write(io->fidndx, (char *)descr, sizeof(*descr)) < sizeof(*descr),
	"putdscr: write failed in note index for %s", io -> nf);
}

getrrec(io, n, resp)
struct io_f *io;
int n;
struct resp_f *resp;	/* n is the number of the resp to get */
{
    long where;				/* going to seek here eventually */
    int a;				/* size of free link */

    x(n < 0, 
        "getrrec: got a negative record number (%d)", n);

    where = sizeof a + (long)n * sizeof (*resp);
    x(lseek(io->fidrdx, where, 0) < 0, 
        "getrrec: lseek failed in resp index for %s", io -> nf);    
    x(read(io->fidrdx, (char *)resp, sizeof(*resp)) < sizeof(*resp),
	"getrrec: read failed in resp index for %s", io -> nf);
}


putrrec(io, n, resp)
struct io_f *io;
int n;			/* n is the number of the resp to put */
struct resp_f *resp;

{
    long where;				/* going to seek here eventually */
    int a;					/* size of free link */

    x(n < 0, 
        "putrrec: got a negative record number (%d)", n);
    where = sizeof a + (long)n * sizeof (*resp);
    x(lseek(io->fidrdx, where, 0) < 0,
        "putrrec: lseek failed in resp index for %s", io -> nf);    
    x(write(io->fidrdx, (char *)resp, sizeof(*resp)) < sizeof(*resp),
	"putrrec: write failed in resp index for %s", io -> nf);
}

printuid()
{
	printf("uid= %d, euid= %d, gid= %d, egid= %d\n",
		getuid(),
		geteuid(),
		getgid(),
		getegid());
}


#ifdef MSGID_INDEX

#define MAKE_DATUM(d,str)	{ \
				d.dptr = str;\
				d.dsize = strlen(str)+1;\
				}

putmsgid(io, notenum, respnum, msgp)
     struct io_f *io;
     int notenum;
     int respnum;
     struct msg_f *msgp;
{
  DBM  *db = io->msgindexp;
  datum art;
  datum straight_key;		/* Message-ID with domain */
  datum stripped_key;		/* Message-ID without domain */

  char  msgid[BUFLEN];
  char  stripped_msgid[BUFLEN];
  char  ngrp_buf[BUFLEN];
  char  date[BUFLEN];

  int   has_domain;
  struct tm *tm;
  struct tm *localtime();

  /* Grab the message-ID and make a stripped version */
  gethdr(io, msgp, msgp->m_artid, msgid);
  safecpy(stripped_msgid, msgid, BUFLEN);
  has_domain = stripid(stripped_msgid);

  /* Stuff them both into keys */
  MAKE_DATUM(straight_key, msgid);
  if (has_domain)
    MAKE_DATUM(stripped_key, stripped_msgid);
  
  /* Make our article data to store */
  /* This string looks a lot like a News history file */
  tm = localtime(&msgp->m_rcvtime);
#ifdef USG
  sprintf(date,"%2.2d/%2.2d/%d %2.2d:%2.2d",
	  tm->tm_mon+1, tm->tm_mday, tm->tm_year,tm->tm_hour, tm->tm_min);
#else
  sprintf(date,"%02d/%02d/%d %02d:%02d",
	  tm->tm_mon+1, tm->tm_mday, tm->tm_year,tm->tm_hour, tm->tm_min);
#endif /* USG */

  sprintf(ngrp_buf, "%s\t%s/%d:%d", date, io->nf, notenum, respnum);
  MAKE_DATUM(art, ngrp_buf);

  lock(io, 'm');			/* ENTER CRITICAL */

  dbm_store(db, straight_key, art, DBM_REPLACE);
  x(dbm_error(db),
    "putmsgid: IO error from dbm_store, errno= %d (\"%s\"), key= \"%s\"",
    errno, sys_errlist[errno], straight_key.dptr);

  if (has_domain)
    {
      /* store with domains stripped so that we can reject stuff from Notes */
      dbm_store(db, stripped_key, art, DBM_REPLACE);
      x(dbm_error(db),
	"putmsgid: IO error from dbm_store, errno= %d (\"%s\"), key= \"%s\"",
	errno, sys_errlist[errno], stripped_key.dptr);
    }

  unlock(io, 'm');			/* EXIT CRITICAL */
}


delmsgid(io, msgp)
     struct io_f *io;
     struct msg_f *msgp;
{
  DBM  *db = io->msgindexp;
  datum d;
  datum art;
  datum straight_key;		/* Message-ID with domain */
  datum stripped_key;		/* Message-ID without domain */

  char  msgid[BUFLEN];
  char  stripped_msgid[BUFLEN];
  char  tmp_buf[BUFLEN];
 
  int   has_domain;

  /* Grab the message-ID and make a stripped version */
  gethdr(io, msgp, msgp->m_artid, msgid);
  safecpy(stripped_msgid, msgid, BUFLEN);
  has_domain = stripid(stripped_msgid);

  /* Stuff it into a key */
  MAKE_DATUM(straight_key, msgid);
  if (has_domain)
    MAKE_DATUM(stripped_key, stripped_msgid);
  
  lock(io, 'm');			/* ENTER CRITICAL */

  dbm_delete(db, straight_key);
  x(dbm_error(db),
    "putmsgid: IO error from dbm_delete, errno= %d (\"%s\"), key= \"%s\"",
    errno, sys_errlist[errno], straight_key.dptr);

  if (has_domain)
    {
      dbm_delete(db, stripped_key);
      x(dbm_error(db),
	"putmsgid: IO error from dbm_delete, errno= %d (\"%s\"), key= \"%s\"",
	errno, sys_errlist[errno], stripped_key.dptr);
    }

  unlock(io, 'm');			/* EXIT CRITICAL */
}

#endif /* MSGID_INDEX */
