/*
** AWK style split routines
**
** If the split character is NULL, this routine will split on space/tab
** and eat long stretches of same (that is, no null fields will result).
** This is similar to AWK's treatment of fields when no field separator is
** specified.
**
** There are three routines:
**	split()		- nondestructive split (copies to malloc'd string)
**	split_d()	- destructive split (mung's up source)
**	split_free()	- free() memory that split() malloc'd
**
**	split and split_d return FAIL on error
**
** Erik E. Fair <fair@ucbarpa.berkeley.edu>,	Nov 6, 1987
*/
#include <stdio.h>

extern	int	errno;
extern	char	*Pname;		/* global variable containing program name */
extern	char	*malloc();
extern	char	*errmsg();

static	char	*e_null = "%s: %s() called with a NULL pointer.\n";

#define	TRUE	1
#define	FALSE	0
#define	FAIL	(-1)

/*
** Checkpoint to make sure that we don't free() something that wasn't
** malloc'd. Basically, we malloc an array that is CHK_ADJ larger than
** we need, and hand the user a pointer to our array + CHK_ADJ, so that
** he won't fiddle the stuff we save above that point.
*/
#define	C_MAGIC		(-2)
#define	C_STRING	(-1)
#define	CHK_MAGICK	(char *)05252
#define	CHK_ADJ		2	/* how many backward subscripts there are */

/*
** We have to find out the length of the string (for the non-destructive
** split case), and since we have to make a pass over the string to get
** that information, we might as well count fields while we're at it so
** that we can malloc the right number of bytes for the array of pointers.
*/
static void
count(s, length, NF, FS)
register char	*s, FS;
int	*length, *NF;
{
	register int	l = 0, n = 0;

	if (FS == '\0') {
		register int	blank = TRUE;

		while(*s != '\0') {
			if ((FS = *s++) == ' ' || FS == '\t') {
				if (!blank)
					blank = TRUE;
			} else {
				if (blank) {
					n++;
					blank = FALSE;
				}
			}
			l++;
		}
	} else {
		while(*s != '\0') {
			if (*s++ == FS)
				n++;
			l++;
		}
	}
	*length = l;
	*NF = n;
}

/*
** Common split code
*/
static int
common(s, p, FS, destroy)
register char	*s, FS;
char	***p;
int	destroy;	/* boolean - modify the source string? */
{
	register char	*cp, **array;
	register int	n = 0;

	if (s == (char *)NULL || p == (char ***)NULL) {
		fprintf(stderr, e_null, Pname, "split");
		abort();
	}

	*p = (char **)NULL;

	/*
	** Get the memory we need
	*/
	{
		int	NF, length;
		char	*e_malloc = "%s: split() could not malloc(%u): %s\n";

		count(s, &length, &NF, FS);

		if (length == 0)
			return(0);	/* nothing to split - go away */

		if (destroy) {
			cp = s;
		} else if ((cp = malloc((unsigned)++length)) == (char *)NULL) {
			fprintf(stderr, e_malloc, Pname, length, errmsg(errno));
			return(FAIL);
		}

		NF = (NF + 1 + CHK_ADJ) * sizeof(char *);
		if ((array = (char **)malloc((unsigned)NF)) == (char **)NULL) {
			fprintf(stderr, e_malloc, Pname, NF, errmsg(errno));
			return(FAIL);
		}
	}

	/*
	** Store the stuff we need for split_free to take care of
	** cleanup neatly.
	*/
	array += CHK_ADJ;
	array[C_MAGIC] = CHK_MAGICK;
	array[C_STRING] = (destroy ? (char *)NULL : cp);

	if (FS == '\0') {
		/*
		** Special case: field separator is NULL, so we use both
		** space and tab as field separators. I'm copying things
		** as I go, and compacting the string in so doing. This
		** should result in no null fields.
		*/
		register int	blank = TRUE;

		while(*s != '\0') {
			if (*s == ' ' || *s == '\t') {
				if (!blank) {
					*cp++ = '\0';	/* terminate */
					blank = TRUE;
				}
				s++;			/* skip blank */
			} else {
				if (blank) {
					array[n++] = cp;	/* begin next */
					blank = FALSE;
				}
				*cp++ = *s++;		/* copy non-blank */
			}
		}
		/*
		** For the !destroy case, we could realloc the string
		** here, but the space savings would have to be pretty
		** big to warrant it: we'd also have to adjust all those
		** pointers in the array that we just set up, if realloc
		** moves the string while compacting it (which the man
		** page says it might).
		*/
	} else {
		/*
		** Normal case: find all occurrences of "FS" and
		** null them as field terminators; add new string
		** starts to the array of pointers to strings.
		** Null fields are valid.
		*/
		for(array[n++] = cp; *s != '\0'; s++) {
			if (*s == FS) {
				*cp++ = '\0';	/* terminate previous */
				array[n++] = cp;	/* begin next */
			} else {
				*cp++ = *s;
			}
		}
	}

	*p = array;
	return(n);
}

split_d(s, p, FS)
char	*s, ***p, FS;
{
	return(common(s, p, FS, TRUE));		/* destructive */
}

split(s, p, FS)
char	*s, ***p, FS;
{
	return(common(s, p, FS, FALSE));	/* non-destructive */
}

/*
** Free up the memory allocated by common cleanly
*/
void
split_free(p)
char ***p;
{
	char	*name = "split_free";

	if (p == (char ***)NULL || *p == (char **)NULL) {
		fprintf(stderr, e_null, Pname, name);
		abort();
	}

	if ((*p)[C_MAGIC] != CHK_MAGICK) {
		fprintf(stderr, "%s: %s(): arena corrupted!\n", Pname, name);
		abort();
	}

	if ((*p)[C_STRING] != (char *)NULL)
		(void) free((*p)[C_STRING]);
	(void) free((char *)(*p - CHK_ADJ));
	*p = (char **)NULL;
}
