//	Copyright (c) 1994, University of Kansas, All Rights Reserved
//
//	Class:		'HText'
//	Include File:	gridtext.h
//	Purpose:	Implement the 'member' functions of the HText 'class'
//	Remarks/Portability/Dependencies/Restrictions:
//		Unlike other WWW clients, no display will be done here in
//		gridtext.  In fact, many unique things will be taking place.
//		Structures were moved to the gridtext.h header file for
//		access by other client modules.
//	Revision History:
//		02-01-94	created, see gridtext.h for details.
//		02-25-94	Began modification to serialize all changes
//				in anchors, styles, and paragraphs by writing
//				pointer values to file that will never change.
//				All functions modified to assume that the
//				HText structure is valid as is the image
//				file.  If not, expect errors.  This was
//				done for the sake of speed in a slow dos
//				WWW client.
#define Uses_TProgram
#include"gridtext.h"
#include"trace.h"
#include"globals.h"
#include"turlview.h"
#include"ttempnam.h"
#include"tform.h"
#include<stdlib.h>
#include<string.h>

//	These functions will be called by "C" source code.
extern "C"	{
#include"htfont.h"

//	The default style to use in case of no others being set.
static HTStyle HTS_default = {	0,
				"(Unstyled)",
				"",
				(HTFont)0,
				1.0,
				HT_BLACK,
				0,
				0,
				0,
				0,
				0,
				HT_LEFT,
				1,
				0,
				0,
				NO,
				NO,
				0,
				0,
				0
};


extern HText *HText_new(HTParentAnchor *HTPAp_anchor)	{
//	Purpose:	Construct a new HText object.
//	Arguments:	HTPAp_anchr	The parent anchor of the HText object.
//	Return Value:	HText *	The new HText object.
//	Remarks/Portability/Dependencies/Restrictions:
//		Does nothing more than member initialization.
//	Revision History:
//		02-01-94	created
#ifndef RELEASE
	trace("creating new HText object.");
#endif // RELEASE
	//	Allocate a new HText structure.
	HText *HTp_text = new HText;
	if(HTp_text == NULL)
		return(NULL);

	//	Assign in the parent anchor.
	HTp_text->HTPAp_node_anchor = HTPAp_anchor;

	//	Initially no views are owning this HText.
	HTp_text->usi_Views = 0U;
	//	No styles or paragraphs yet appended.
	HTp_text->B_hasParagraph = False;
	HTp_text->B_hasStyle = False;

	//	Create the temporary file name.
	HTp_text->TTNp_fspname = new TTempName(cp_TempDir);

	//	No forms as of yet.
	HTp_text->HTLp_forms = NULL;
	HTp_text->B_inForm = False;

	//	Open the image stream in the temporary directory.
	if(HTp_text->TTNp_fspname != NULL)	{
		HTp_text->fsp_image = new fstream(HTp_text->TTNp_fspname->
			getName(), ios::trunc | ios::out | ios::binary);
	}

	//	If everything allocated correctly return.
	if(HTp_text->fsp_image != NULL && HTp_text->TTNp_fspname != NULL)
	{
		//	Set the DosLynx HText collection to know there is a
		//	new HText in memory and is begin loaded.
		HTp_text->B_isLoading = True;
		TNSCp_LoadedHTexts->insert((void *)HTp_text);

		//	Check to see if the number of current HTexts is over
		//	the specified limit.  If so, try to find the amount
		//	over not in use and free them, from the beginning
		//	which are the older documents.
		//	Exclude this newly created HText from the search.
		if(usi_MaxLoadedHTexts < TNSCp_LoadedHTexts->getCount())
		{
			HText *HTp_remove;
			unsigned short int usi_over = TNSCp_LoadedHTexts->
				getCount() - usi_MaxLoadedHTexts;
			for(signed short int ssi_search = 0; ssi_search <
				TNSCp_LoadedHTexts->getCount() - 2;
				ssi_search++)	{
				HTp_remove = (HText *)(TNSCp_LoadedHTexts->
					at(ssi_search));
				if(HTp_remove->usi_Views != 0U)
					continue;
				HText_free(HTp_remove);
				//	Reset ssi_search since HText_free
				//	messed with the number in the
				//	collection.
				ssi_search--;
				if(--usi_over == 0U)
					break;
			}
		}

		//	Set WWW to know that the parent anchor is now
		//	associated with this HText.
		HTAnchor_setDocument(HTPAp_anchor, (HyperDoc *)HTp_text);

		//	Set TURLV_current to own this HText.
		TURLV_current->registerText(HTp_text);

		//	Set the current style of the HText to the default.
		HText_beginStyle(HTp_text, &HTS_default);
		return(HTp_text);
	}

	//	Not everything initialized correctly, return NULL after
	//	releasing memory taken.
	if(HTp_text->fsp_image != NULL && HTp_text->TTNp_fspname != NULL)
	{
		HTp_text->fsp_image->close();
		delete(HTp_text->fsp_image);
	}
	if(HTp_text->TTNp_fspname != NULL)
		delete(HTp_text->TTNp_fspname);

	return(NULL);
}

extern HText *HText_new2(HTParentAnchor *HTPAp_anchor, HTStream *HTSp_stream)
{
//	Purpose:	Construct a new HText object associated with a
//			HTStream.
//	Arguments:	HTPAp_anchor	The parent anchor of this HText.
//			HTSp_stream	The stream to direct output.
//	Return Value:	HText *	The new HText object.
//	Remarks/Portability/Dependencies/Restrictions:
//		Assignment of a HTStream to a HText object is currently not
//			supported.
//	Revision History:
//		02-01-94	created

	//	Just call the default constructor until support for a
	//	HTStream is implemented.
	return(HText_new(HTPAp_anchor));

#ifdef TODO
#error Implement a HTStream hook into HText.
#endif // TODO
}

extern void HText_free(HText *HTp_text)	{
//	Purpose:	Destroy a HText object.
//	Arguments:	HTp_text	The HText to release from memory.
//	Return Value:	void
//	Remarks/Portability/Dependencies/Restrictions:
//		Will also delete any temporary file associated with the
//		HText object.
//	Revision History:
//		02-01-94	created
//		04-28-94	Modified so that we take ourselves out of
//				the global collection containing the loaded
//				HTexts so that functions calling us don't
//				worry about it.

	doslynxmessage("Freeing " << HTp_text->HTPAp_node_anchor->address);

	//	Take this out of the loaded HText collection.
	TNSCp_LoadedHTexts->remove((void *)HTp_text);
	TNSCp_LoadedHTexts->pack();

	//	Tell WWW to no longer associate this HText with its parent
	//	anchor.
	HTAnchor_setDocument(HTp_text->HTPAp_node_anchor, (HyperDoc *)NULL);

	//	Delete any anchors from this parent anchor.
	HTAnchor_delete(HTp_text->HTPAp_node_anchor);

	//	Attempt to delete the image stream.
	delete(HTp_text->TTNp_fspname);

	//	Delete the forms collection and the contained forms objects.
	if(HTp_text->HTLp_forms != NULL)	{
		while(HTList_isEmpty((HTp_text->HTLp_forms)) == NO)	{
			delete((TForm *)HTList_removeLastObject(HTp_text->
				HTLp_forms));
		}
		HTList_delete((HTp_text->HTLp_forms));
		HTp_text->HTLp_forms = NULL;
	}

	//	Free the actual HText object.
	delete(HTp_text);
}

extern void HText_beginAppend(HText *HTp_text)	{
//	Purpose:	Prepare the HText object for appending.
//	Arguments:	HTp_text	The HText object to append to.
//	Return Value:	void
//	Remarks/Portability/Dependencies/Restrictions:
//		Doesn't do a thing, just for compatibility with WWW.
//	Revision History:
//		02-01-94	created
#ifndef RELEASE
	trace("preparing HText for appending.");
#endif // RELEASE
}

extern void HText_endAppend(HText *HTp_text)	{
//	Purpose:	End appending to the HText object.
//	Arguments:	HTp_text	The HText object to end appends.
//	Return Value:	void
//	Remarks/Portability/Dependencies/Restrictions:
//		Closes the image file opened upon HText creation.
//	Revision History:
//		02-01-94	created

	//	Finish any open styles and paragraphs.
	HText_endParagraph(HTp_text);
	HText_endStyle(HTp_text);

#ifndef RELEASE
	trace("ending appends to HText.");
#endif // RELEASE
	//	Close then free the image file.
	//	The actual file will not be deleted here, but will be deleted
	//	in HText_free when it is called.  The name of the file is
	//	preserved in TTNp_fspname.
	HTp_text->fsp_image->close();
	delete(HTp_text->fsp_image);
	HTp_text->fsp_image = NULL;

	//	Set the HText object to no longer be loading.
	HTp_text->B_isLoading = False;
}

extern void HText_beginParagraph(HText *HTp_text)	{
//	Purpose:	Begin a paragraph in the HText object.
//	Arguments:	HTp_text	The HText object to create a paragraph
//					for.
//	Return Value:	void
//	Remarks/Portability/Dependencies/Restrictions:
//		Will end any unended paragraphs before starting a new
//		paragraph.
//	Revision History:
//		02-01-94	created
//		02-17-94	Updated code to include the index of the
//				paragraph.
//		02-25-94	Serialized new paragraph.

	//	Close any open paragraphs.
	HText_endParagraph(HTp_text);

	//	Put in the embedded character.
	HTp_text->fsp_image->put(c_Embedded);
	HTp_text->fsp_image->put(c_AppendParagraph);
	//	If some type of error occured while writing, print a message
	//	and close the file.
	if(HTp_text->fsp_image->bad())	{
		doslynxmessage("An error occured while writing to file " <<
			HTp_text->TTNp_fspname->getName());
		HTp_text->fsp_image->close();

		//	Cause the application to exit.
		TEvent TE_quit;
		TE_quit.what = evMessage;
		TE_quit.message.command = cmQuit;
		TProgram::application->handleEvent(TE_quit);
	}

	//	Set flag so that endParagraph knows that there is one to end.
	HTp_text->B_hasParagraph = True;
}

extern void HText_endParagraph(HText *HTp_text)	{
//	Purpose:	End a paragraph in the HText object.
//	Arguments:	HTp_text	The HText object to end the paragraph
//					in.
//	Return Value:	void
//	Remarks/Portability/Dependencies/Restrictions:
//		Will first check to see if a paragraph need be ended.
//	Revision History:
//		02-01-94	created.
//		02-25-94	Modified so does nothing....
//				Maybe in the future.

	//	Only the very first time this function is called there will
	//	be no need to close a paragraph.
	if(HTp_text->B_hasParagraph == False)
		return;

	//	Do some special processing when a paragraph ends if needed.
}

extern void HText_beginStyle(HText *HTp_text, HTStyle *HTSp_style)	{
//	Purpose:	Start a particular style in a HText object.
//	Arguments:	HTp_text	The HText to set the style for.
//			HTSp_style	The style to set to.
//	Return Value:	void.
//	Remarks/Portability/Dependencies/Restrictions:
//		Will end any open styles first.
//	Revision History:
//		02-01-94	created
//		02-25-94	Writing style pointer to disk.

	//	End any open styles.
	HText_endStyle(HTp_text);

	//	Put in the embedded character.
	HTp_text->fsp_image->put(c_Embedded);
	HTp_text->fsp_image->put(c_SetStyle);
	//	Here's the trick, write a pointer to the style to file.
	HTp_text->fsp_image->write((const char *)(&HTSp_style),
		sizeof(HTStyle *));
	//	If some type of error occured while writing, print a message
	//	and close the file.
	if(HTp_text->fsp_image->bad())	{
		doslynxmessage("An error occured while writing to file " <<
			HTp_text->TTNp_fspname->getName());
		HTp_text->fsp_image->close();

		//	Cause the application to exit.
		TEvent TE_quit;
		TE_quit.what = evMessage;
		TE_quit.message.command = cmQuit;
		TProgram::application->handleEvent(TE_quit);
	}

	//	Set a flag so that endStyle will know that there is a style
	//	to end.
	HTp_text->B_hasStyle = True;
}

extern void HText_endStyle(HText *HTp_text)	{
//	Purpose:	End the use of a style in a HText object.
//	Arguments:	HTp_text	The HText object to end the style.
//	Return Value:	void
//	Remarks/Portability/Dependencies/Restrictions:
//		Will first check and see if a style needs to be ended.
//	Revision History:
//		02-01-94	created
//		02-25-94	Changed to do nothing, maybe in the future....

	//	Only the very first time this function is called will there
	//	be no need to close a style.
	if(HTp_text->B_hasStyle == False)
		return;

	//	Do nothing unless special processing is needed to end a style.
}

extern void HText_beginAnchor(HText *HTp_text, HTChildAnchor *HTCAp_anchor)
{
//	Purpose:	Begin an anchor in a HText object.
//	Arguments:	HTp_text	The HText object to begin the anchor
//					in.
//			HTCAp_anchor	The child anchor beginning the anchor
//					in HText.
//	Return Value:	void
//	Remarks/Portability/Dependencies/Restrictions:
//	Revision History:
//		02-01-94	created
//		02-25-94	Modified to serialize the pointer value.

	//	Put in the embedded character.
	HTp_text->fsp_image->put(c_Embedded);
	HTp_text->fsp_image->put(c_BeginAnchor);
	//	Here's the trick, write a pointer to the anchor to file.
	HTp_text->fsp_image->write((const char *)(&HTCAp_anchor),
		sizeof(HTChildAnchor *));
	//	If some type of error occured while writing, print a message
	//	and close the file.
	if(HTp_text->fsp_image->bad())	{
		doslynxmessage("An error occured while writing to file " <<
			HTp_text->TTNp_fspname->getName());
		HTp_text->fsp_image->close();

		//	Cause the application to exit.
		TEvent TE_quit;
		TE_quit.what = evMessage;
		TE_quit.message.command = cmQuit;
		TProgram::application->handleEvent(TE_quit);

	}
}

extern void HText_endAnchor(HText *HTp_text)	{
//	Purpose:	End an anchor in a HText object.
//	Arguments:	HTp_text	The HText to end the anchor in.
//	Return Value:	void
//	Remarks/Portability/Dependencies/Restrictions:
//		Unlike HText_endParagraph and HText_endStyle, there will
//		always be an anchor to close; there is no need to check.
//	Revision History:
//		02-01-94	created
//		02-25-94	Modified to serialize a marker to end the
//				anchor.

	//	Put in the embedded character.
	HTp_text->fsp_image->put(c_Embedded);
	HTp_text->fsp_image->put(c_EndAnchor);
	//	If some type of error occured while writing, print a message
	//	and close the file.
	if(HTp_text->fsp_image->bad())	{
		doslynxmessage("An error occured while writing to file " <<
			HTp_text->TTNp_fspname->getName());
		HTp_text->fsp_image->close();
		//	Cause the application to exit.
		TEvent TE_quit;
		TE_quit.what = evMessage;
		TE_quit.message.command = cmQuit;
		TProgram::application->handleEvent(TE_quit);
	}
}

extern void HText_appendCharacter(HText *HTp_text, char c_append)	{
//	Purpose:	Add a character to the HText object.
//	Arguments:	HTp_text	The HText object to append to.
//			c_append	The character to add.
//	Return Value:	void
//	Remarks/Portability/Dependencies/Restrictions:
//		Character not actually added to HText but appended to the
//		image file.
//	Revision History:
//		02-01-94	created

	//	Write the character to the binary file stream.
	HTp_text->fsp_image->put(c_append);

	//	If some type of error occured while writing, print a message
	//	and close the file.
	if(HTp_text->fsp_image->bad())	{
		doslynxmessage("An error occured while writing to file " <<
			HTp_text->TTNp_fspname->getName());
		HTp_text->fsp_image->close();
		//	Cause the application to exit.
		TEvent TE_quit;
		TE_quit.what = evMessage;
		TE_quit.message.command = cmQuit;
		TProgram::application->handleEvent(TE_quit);
	}
}

extern void HText_appendImage(HText *HTp_text, HTChildAnchor *HTCAp_anc,
	const char *cp_alt, int ssi_alignment, BOOL B_isMap)	{
//	Purpose:	Append an image in the HTML rendering.
//	Arguments:	HTp_text	The hyper document in which to insert
//					the image.
//			HTCAp_anc	The anchor pointing to the image.
//			cp_alt		The title of the picture.
//			ssi_alignment	How the picture should be aligned on
//					the screen.
//			B_isMap		Whether or not portions are
//					selectable?
//	Return Value:	void
//	Remarks/Portability/Dependencies/Restrictions:
//		Created just to fill where pictures should be with an anchor.
//	Revision History:
//		03-07-94	created

	auto char *cp_append = "<IMAGE>";

	if(cp_alt != NULL)	{
		if(*cp_alt != '\0')	{
			cp_append = (char *)cp_alt;
		}
	}

	HText_appendText(HTp_text, cp_append);
}

extern void HText_appendText(HText *HTp_text, const char *cp_append)	{
//	Purpose:	Add a string to the HText object.
//	Arguments:	HTp_text	The HText object to add to.
//	Return Value:	void
//	Remarks/Portability/Dependencies/Restrictions:
//		Characters are not actually added to HText but appended to the
//		image file.
//	Revision History:
//		02-01-94	created

	for(char *cp = (char *)cp_append; *cp != '\0'; cp++)	{
		HText_appendCharacter(HTp_text, *cp);
	}
}

extern BOOL HText_selectAnchor(HText *HTp_text, HTChildAnchor *HTCAp_anchor)
{
//	Purpose:	Selects the specified anchor in HText.
//	Arguments:	HTp_text	The HText object with the anchor.
//			HTCAp_anchor	The anchor to select.
//	Return Value:	BOOL	TRUE	The anchor was selected.
//				FALSE	The anchor could not be selected.
//	Remarks/Portability/Dependencies/Restrictions:
//		The only reason why an anchor is unable to be selected is
//		if the anchor does not exists in the Anchor collection.
//		Also, assume that if an anchor is selected, the HText is
//		viewable in the current TURLView.  Since multiple views
//		of the same HText can exists, we can not record the selected
//		anchor in the HText structure, but must call a TURLView
//		function to set the selected anchor for each different view.
//	Revision History:
//		02-01-94	created

#ifndef RELEASE
	trace("selecting an anchor in HText.");
#endif // RELEASE

	//	Have the view select the anchor.
	if(TURLV_current->selectAnchor(HTCAp_anchor) == False)	{
		//	Anchor wasn't found.
		doslynxmessage("Attempt to select invalid anchor failed.");
		return(FALSE);
	}
	return(TRUE);
}

extern BOOL HText_select(HText *HTp_text)	{
//	Purpose:	View the HText object NOW.
//	Arguments:	HTp_text	The HText object to be viewed.
//	Return Value:	BOOL	TRUE	The HText was selected.
//				FALSE	The HText was not selected.
//	Remarks/Portability/Dependencies/Restrictions:
//		This function won't be called unless a request to load is
//		made.
//		The HText was loaded at some time and is still in memory, be
//		it in none, one, or more windows.
//		Inform the calling window that this HText exists, by simply
//		manually assigning a pointer to this HText.
//	Revision History:
//		02-01-94	created.

	//	Use the global variable of the URL view to point to us.
	TURLV_current->registerText(HTp_text);
	return(TRUE);
}

extern void HText_beginForm(HText *HTp_text, const char *cp_action,
	const char *cp_method)	{
//	Purpose:	Begin the creation of a Form in our hypertext object.
//	Arguments:	HTp_text	The hypertext object owning this new
//						form.
//			cp_action	The destination of our output.
//			cp_method	The method to use to send the data,
//						could be as a URL or could
//						be sent as a post....
//	Return Value:	void
//	Remarks/Portability/Dependencies/Restrictions:
//	Revision History:
//		04-24-94	created

#ifndef RELEASE
	trace("Begin form " << cp_action << " " << cp_method);
#endif // RELEASE

	//	Create a new form object.
	auto TForm *TFp_form = new TForm(cp_action, cp_method);

	//	Make sure the forms list exists.
	if(HTp_text->HTLp_forms == NULL)	{
		HTp_text->HTLp_forms = HTList_new();
	}

	//	Insert the form into the list of forms in HText.  The current
	//		form can always be considered the last form.
	HTList_addObject((HTp_text->HTLp_forms), (void *)TFp_form);

	//	Mark that we are currently in a form.
	HTp_text->B_inForm = True;

	//	We won't need to write anything to file since the form won't
	//		be a displayable item itself, it's other parts will
	//		be though....
#ifndef RELEASE
	trace("Returning");
#endif // RELEASE
}

extern void HText_endForm(HText *HTp_text)	{
//	Purpose:	End the creation of a Form in the HyperText object.
//	Arguments:	void
//	Return Value:	void
//	Remarks/Portability/Dependencies/Restrictions:
//		Really we do nothing here....  Just put in for the future.
//	Revision History:
//		04-24-94	created

	//	Mark that we are no longer in a form.
	HTp_text->B_inForm = False;
}

extern BOOL HText_inForm(HText *HTp_text)	{
//	Purpose:	Return wether or not currently handling a form.
//	Arguments:	HTp_text	The hypertext being referred to.
//	Return Value:	BOOL	TRUE	Yes, in a form.
//				FALSE	No, not in a form.
//	Remarks/Portability/Dependencies/Restrictions:
//	Revision History:
//		05-12-94	created

	return((BOOL)(HTp_text->B_inForm));
}

}; // extern "C"