/****************************************************************************
 * pnt_ren_mthd.c
 * Author Joel Welling
 * Copyright 1991, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module provides renderer methods for the Painter's Algorithm renderer.
*/

#include <stdio.h>
#include <math.h>
#include "p3dgen.h"
#include "pgen_objects.h"
#include "ge_error.h"
#include "assist.h"
#include "paintr_strct.h"
#include "painter.h"

/* 
The following is designed to figure out what machine this is running
on, and include appropriate definition files.
*/
#ifdef unix
#define USE_UNIX
#endif
#ifdef CRAY
#undef USE_UNIX
#endif
#ifdef ardent
#undef USE_UNIX
#endif
#ifdef CRAY
#include "unicos_defs.h"
#endif
#ifdef ardent
#include "unicos_defs.h"
#endif
#ifdef USE_UNIX
#include "unix_defs.h"
#endif

/* Notes-
*/

/* Space for default color map */
static P_Renderer_Cmap default_map;

/* default color map function */
static void default_mapfun(float *val, float *r, float *g, float *b, float *a)
/* This routine provides a simple map from values to colors within the
 * range 0.0 to 1.0.
 */
{
  /* No debugging; called too often */
  *r= *g= *b= *a;
}

static P_Void_ptr def_cmap( char *name, double min, double max,
		  void (*mapfun)(float *, float *, float *, float *, float *) )
/* This function stores a color map definition */
{
  P_Renderer *self= (P_Renderer *)po_this;
  P_Renderer_Cmap *thismap;
  METHOD_IN

  ger_debug("pnt_ren_mthd: def_cmap");

  if (max == min) {
    ger_error("pnt_ren_mthd: def_cmap: max equals min (val is %f)", max );
    METHOD_OUT;
    return( (P_Void_ptr)0 );    
  }

  if ( RENDATA(self)->open ) {
    if ( !(thismap= (P_Renderer_Cmap *)malloc(sizeof(P_Renderer_Cmap))) )
      ger_fatal( "pnt_ren_mthd: def_cmap: unable to allocate %d bytes!",
		sizeof(P_Renderer_Cmap) );

    strncpy(thismap->name, name, MAXSYMBOLLENGTH-1);
    (thismap->name)[MAXSYMBOLLENGTH-1]= '\0';
    thismap->min= min;
    thismap->max= max;
    thismap->mapfun= mapfun;
    METHOD_OUT
    return( (P_Void_ptr)thismap );
  }
  METHOD_OUT
  return( (P_Void_ptr)0 );
}

static void install_cmap( P_Void_ptr mapdata )
/* This method causes the given color map to become the 'current' map. */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  ger_debug("pnt_ren_mthd: install_cmap");
  if ( RENDATA(self)->open ) {
    if (mapdata) CUR_MAP(self)= (P_Renderer_Cmap *)mapdata;
    else ger_error("pnt_ren_mthd: install_cmap: got null color map data.");
  }
  METHOD_OUT
}

static void destroy_cmap( P_Void_ptr mapdata )
/* This method causes renderer data associated with the given color map
 * to be freed.
 */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  ger_debug("pnt_ren_mthd: destroy_cmap");
  if ( RENDATA(self)->open ) free( mapdata );
  METHOD_OUT  
}

static void map_color( P_Renderer *self, float val, 
		      float *r, float *g, float *b, float *a )
/* This routine invokes the color map. */
{
  /* no debugging; called too often */

  /* Scale, clip, and map. */
  val= ( val-MAP_MIN(self) )/( MAP_MAX(self)-MAP_MIN(self) );
  if ( val > 1.0 ) val= 1.0;
  if ( val < 0.0 ) val= 0.0;
  (*MAP_FUN(self))( &val, r, g, b, a );
}

static P_Void_ptr def_camera( P_Camera *cam )
/* This method defines the given camera */
{
  P_Renderer *self= (P_Renderer *)po_this;
  P_Camera *saved_cam;
  METHOD_IN

  ger_debug("pnt_ren_mthd: def_camera");
  if ( RENDATA(self)->open ) {
    saved_cam= po_create_camera(cam->name, &(cam->lookfrom), &(cam->lookat),
				&(cam->up), cam->fovea, cam->hither, cam->yon);
    saved_cam->background.ctype= cam->background.ctype;
    saved_cam->background.r= cam->background.r;
    saved_cam->background.g= cam->background.g;
    saved_cam->background.b= cam->background.b;
    saved_cam->background.a= cam->background.a;
  }
  else saved_cam= (P_Camera *)0;

  METHOD_OUT
  return( (P_Void_ptr)saved_cam );
}

static void set_camera( P_Void_ptr primdata )
/* This makes the given camera the current camera */
{
  P_Renderer *self= (P_Renderer *)po_this;
  P_Camera *cam;
  float fov;
  int i;
  METHOD_IN

  ger_debug("pnt_ren_mthd: set_camera");
  cam= (P_Camera *)primdata;
  if ( RENDATA(self)->open ) {
	float *trans_camera, *Rotx, *Roty, *Rotz;
	float a, b, c, d, a_prime, b_prime, d_prime,length;
	Pnt_Vectortype *up, *forward, *rotated_upvector;
	Pnt_Pointtype *origin,*camera;
  	float deg_to_rad= 3.14159265/180;
	int i;

	if (cam->yon == 0.0)
		cam->yon = 0.00001;
	origin = pnt_makepoint_rec( cam->lookat.x, cam->lookat.y, 
				cam->lookat.z );
	camera = pnt_makepoint_rec( cam->lookfrom.x, cam->lookfrom.y, 
				cam->lookfrom.z );
	/* Background color record */
	BACKGROUND(self).r= cam->background.r;
	BACKGROUND(self).g= cam->background.g;
	BACKGROUND(self).b= cam->background.b;
	BACKGROUND(self).a= cam->background.a;

	/* what if Camera_Up is not a unit vector ? */
	/* and what if Camera_Up is not perpendicular to Forward */
	up = pnt_makevector_rec( cam->up.x, cam->up.y, 
	        	     cam->up.z );
	forward = pnt_make_directionvector(camera,origin);

	if (VIEWMATRIX(self)) free( VIEWMATRIX(self) );
	VIEWMATRIX(self) = pnt_make3dScale(1.0,1.0,1.0);
	trans_camera = pnt_make3dTrans(
	     0.0 - camera->x, 0.0 - camera->y, 0.0 - camera->z);
	pnt_append3dMatrices(VIEWMATRIX(self),trans_camera);

	a = forward->x;
	b = forward->y;
	c = forward->z;
	d = (float) sqrt( (b*b + c*c) );

	Rotx = pnt_make3dScale(1.0,1.0,1.0);
	Roty = pnt_make3dScale(1.0,1.0,1.0);
	Rotz = pnt_make3dScale(1.0,1.0,1.0);

	/*  Need check for d=0 */
	if (d != 0.0)
		{
		Rotx[5] = c/d;
		Rotx[6] = b/d;
		Rotx[9] = 0.0-b/d;
		Rotx[10] = c/d;
		}
	else
		{
		Rotx[5] = 1.0;
		Rotx[6] = 0.0;
		Rotx[9] = 0.0;
		Rotx[10] = 1.0;
		}

	Roty[0] =  d;
	Roty[2] =  a;
	Roty[8] =  0.0 - a;
	Roty[10] = d;

	pnt_append3dMatrices(VIEWMATRIX(self),Rotx);
	pnt_append3dMatrices(VIEWMATRIX(self),Roty);

	rotated_upvector = pnt_vector_matrix_mult3d(up,VIEWMATRIX(self));
	a_prime = rotated_upvector->x;
	b_prime = rotated_upvector->y;
	d_prime = (float) sqrt(a_prime*a_prime + b_prime*b_prime);
	
	if (d_prime != 0.0)
		{
		Rotz[0] = b_prime/d_prime;
		Rotz[1] = a_prime/d_prime;
		Rotz[4] = (0.0 - a_prime)/d_prime;
		Rotz[5] = b_prime/d_prime;
		}
	else
		{
		Rotz[0] = 1.0;
		Rotz[1] = 0.0;
		Rotz[4] = 0.0;
		Rotz[5] = 1.0;
		}
	pnt_append3dMatrices(VIEWMATRIX(self),Rotz);
	
	fov = fabs(cam->fovea) * deg_to_rad;
	for (i=0; i<15; i++) EYEMATRIX(self)[i]= 0.0;
	EYEMATRIX(self)[0] = 1.0 / tan((double) fov/2.0);   
	EYEMATRIX(self)[5] = 1.0 / tan((double) fov/2.0);   

	EYEMATRIX(self)[8] = -1.0;   /*  Offset origin to middle of screen */
	EYEMATRIX(self)[9] = -1.0;   /*  Screen dimensions are 2.0  by 2.0 */
	EYEMATRIX(self)[10] = 1.0 / (1.0 - cam->hither/cam->yon);
	EYEMATRIX(self)[11] = -1.0;
	EYEMATRIX(self)[14] = (0.0 + cam->hither)/ 
	  (1.0 - cam->hither/cam->yon);
	ZMAX(self) = cam->hither;
	ZMIN(self) = cam->yon;

	/* Make sure the renderer doesn't keep old camera info in its caches */
	pnt_recache(self);

	free(rotated_upvector);
	free(up);
	free(forward);
	free(origin);
	free(camera);
	free(trans_camera);
	free(Rotx);
	free(Roty);
	free(Rotz);
  }
  METHOD_OUT
}

static void destroy_camera( P_Void_ptr primdata )
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  ger_debug("pnt_ren_mthd: destroy_camera");
  if ( RENDATA(self)->open ) {
    free(primdata);
  }
  METHOD_OUT  
}

static void ren_print( VOIDLIST )
/* This is the print method for the renderer */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  ger_debug("pnt_ren_mthd: ren_print");
  if ( RENDATA(self)->open ) 
    ind_write("RENDERER: Painter renderer '%s', device %s, outfile %s, open",
	      self->name, DEVICENAME(self),OUTFILE(self)); 
  else 
    ind_write("RENDERER: Painter renderer '%s', device %s, outfile %s, closed",
	      self->name, DEVICENAME(self),OUTFILE(self)); 
  ind_eol();
  METHOD_OUT
}

static void ren_open( VOIDLIST )
/* This is the open method for the renderer */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN
  ger_debug("pnt_ren_mthd: ren_open");
  RENDATA(self)->open= 1;
  METHOD_OUT
}

static void ren_close( VOIDLIST )
/* This is the close method for the renderer */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN
  ger_debug("pnt_ren_mthd: ren_close");
  RENDATA(self)->open= 0;
  METHOD_OUT
}

static void ren_destroy( VOIDLIST )
/* This is the destroy method for the renderer */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN
  ger_debug("pnt_ren_mthd: ren_destroy");

  if (RENDATA(self)->open) ren_close();
  pnt_shutdown_renderer();

  if (ASSIST(self)) {
    METHOD_RDY( ASSIST(self) );
    (*(ASSIST(self)->destroy_self))();
  }
  if (DPOLYBUFFER(self)) free((P_Void_ptr)DPOLYBUFFER(self));
  if (DEPTHBUFFER(self)) free((P_Void_ptr)DEPTHBUFFER(self));
  if (DCOORDBUFFER(self)) free((P_Void_ptr)DCOORDBUFFER(self));
  if (DCOLORBUFFER(self)) free((P_Void_ptr)DCOLORBUFFER(self));
  if (DEVICENAME(self)) free((P_Void_ptr)DEVICENAME(self));
  if (OUTFILE(self)) free((P_Void_ptr)OUTFILE(self));
  if (VIEWMATRIX(self)) free((P_Void_ptr)VIEWMATRIX(self));
  if (EYEMATRIX(self)) free((P_Void_ptr)EYEMATRIX(self));

  RENDATA(self)->initialized= 0;
  free( (P_Void_ptr)RENDATA(self) );
  free( (P_Void_ptr)self );
  METHOD_DESTROYED
}

static void internal_render(P_Renderer *self, P_Gob *gob, float *thistrans)
{
   P_Attrib_List newattrlist;
   P_Transform *atrans;
   float *newtrans;
   P_Gob_List *kidlist;

   ger_debug(
      "pnt_ren_mthd: internal_render: rendering object given by gob <%s>",
	     gob->name);

   if ( gob->has_transform )
	{
	atrans = transpose_trans( duplicate_trans( &(gob->trans) ) );
	newtrans = pnt_mult3dMatrices( atrans->d, thistrans );
	destroy_trans(atrans);
	}
   else
	newtrans  = thistrans;
   if (gob->attr) {
     METHOD_RDY(ASSIST(self));
     (*(ASSIST(self)->push_attributes))( gob->attr );
   }
   /* There must be children to render, because if this was a primitive
    * this method would not be getting executed.
    */
   kidlist= gob->children;
   while (kidlist) {
     RECENTTRANS(self)= newtrans;
     METHOD_RDY(kidlist->gob);
     (*(kidlist->gob->render_to_ren))(self, (P_Transform *)0, 
				      (P_Attrib_List *)0);
     kidlist= kidlist->next;
   }
   if (newtrans != thistrans) free(newtrans);
   if (gob->attr) {
     METHOD_RDY(ASSIST(self));
     (*(ASSIST(self)->pop_attributes))( gob->attr );
   }
}

static int compare(a,b)
depthtable_rec *a,*b;
{
	if (a->key > b->key)
		return(1);
	if (a->key < b->key)
		return(-1);
	return(0);
}

static void draw_DepthBuffer(P_Renderer *self)
{
  register int i;

  ger_debug("pnt_ren_mthd: draw_DepthBuffer");
  
  if (DEPTHCOUNT(self) >= MAXDEPTHPOLY(self))
    ger_fatal("ERROR, was about to overflow!");
  for (i=0;i<DEPTHCOUNT(self);i++)
    pnt_render_polyrec(self,i);
  DEPTHCOUNT(self)=0;			/* reset DepthBuffer */
}

static void ren_gob( P_Void_ptr primdata, P_Transform *thistrans, 
		    P_Attrib_List *thisattrlist )
/* This routine renders gob */
{
  P_Renderer *self= (P_Renderer *)po_this;
  int top_level_call= 0;
  METHOD_IN

  if (RENDATA(self)->open) {
    if (primdata) {
      P_Gob *thisgob;
      int error=0,color_mode=1; 
      P_Transform *newtrans= (P_Transform *)0;

      thisgob= (P_Gob *)primdata;
      
      ger_debug("pnt_ren_mthd: ren_gob: rendering object given by gob <%s>", 
		thisgob->name);

      if (thisattrlist) {
	METHOD_RDY(ASSIST(self));
	(*(ASSIST(self)->push_attributes))( thisattrlist );
      }

      /* If thistrans is non-null, this is a top-level call to
       * ren_gob, as opposed to a recursive call.  
       */
      if (thistrans) {
	top_level_call= 1;
	newtrans = transpose_trans( duplicate_trans(thistrans) );
	internal_render(self,thisgob,newtrans->d);
	destroy_trans(newtrans);
      }
      else {
	float *oldtrans= RECENTTRANS(self);
	internal_render(self,thisgob,RECENTTRANS(self));
	RECENTTRANS(self)= oldtrans;
      }

      if (thisattrlist) {
	METHOD_RDY(ASSIST(self));
	(*(ASSIST(self)->pop_attributes))( thisattrlist );
      }

      if (top_level_call) { /* Actually draw the model */
	/* This is the sort that implements Painter's Algorithm */
	qsort((char *)DEPTHBUFFER(self),(unsigned) DEPTHCOUNT(self),
	      sizeof(depthtable_rec),compare);
	
	wrbegp( &error );		/* begin picture  */
	wrtcsm( &color_mode,&error );	/* use direct color */
	/* background color */
	wrbgdc( &(BACKGROUND(self).r), &(BACKGROUND(self).g), 
	       &(BACKGROUND(self).b), &(BACKGROUND(self).a) );
	wrbgpb( &error );		/* begin picture body  */
	if (error) ger_fatal("ERROR: Bad Picture Initialization!");
	draw_DepthBuffer(self);
	wrendp(&error);
	if (error) ger_fatal("ERROR: Bad Picture End");

	/*  Reset fill indices for disposable records  */
	DEPTHCOUNT(self) = 0;
	DEPTHPOLYCOUNT(self) = 0;
	DCOLORCOUNT(self) = 0;
	DCOORDINDEX(self) = 0;
      }

    }
    else ger_error("pnt_ren_mthd: ren_gob: got a null data pointer.");
  }
  METHOD_OUT
}

static void internal_traverse(P_Renderer *self, P_Gob *gob, float *thistrans)
{
   P_Attrib_List newattrlist;
   P_Transform *atrans;
   float *newtrans;
   P_Gob_List *kidlist;

   ger_debug(
      "pnt_ren_mthd: internal_traverse: traversing object given by gob <%s>",
	     gob->name);

   if ( gob->has_transform )
	{
	atrans = transpose_trans( duplicate_trans( &(gob->trans) ) );
	newtrans = pnt_mult3dMatrices( atrans->d, thistrans );
	destroy_trans(atrans);
	}
   else
	newtrans  = thistrans;
   if (gob->attr) {
     METHOD_RDY(ASSIST(self));
     (*(ASSIST(self)->push_attributes))( gob->attr );
   }
   /* There must be children to render, because if this was a primitive
    * this method would not be getting executed.
    */
   kidlist= gob->children;
   while (kidlist) {
     RECENTTRANS(self)= newtrans;
     METHOD_RDY(kidlist->gob);
     (*(kidlist->gob->traverselights_to_ren))(self, (P_Transform *)0, 
				      (P_Attrib_List *)0);
     kidlist= kidlist->next;
   }
   if (newtrans != thistrans) free(newtrans);
   if (gob->attr) {
     METHOD_RDY(ASSIST(self));
     (*(ASSIST(self)->pop_attributes))( gob->attr );
   }
}

static void traverse_gob( P_Void_ptr primdata, P_Transform *thistrans, 
		    P_Attrib_List *thisattrlist )
/* This routine traverses a gob looking for lights. */
{
  P_Renderer *self= (P_Renderer *)po_this;
  int top_level_call= 0;
  METHOD_IN

  if (RENDATA(self)->open) {
    if (primdata) {
      P_Gob *thisgob;
      P_Transform *newtrans= (P_Transform *)0;

      thisgob= (P_Gob *)primdata;
      
      ger_debug(
	 "pnt_ren_mthd: traverse_gob: traversing object given by gob <%s>", 
		thisgob->name);

      if (thisattrlist) {
	METHOD_RDY(ASSIST(self));
	(*(ASSIST(self)->push_attributes))( thisattrlist );
      }

      /* If thistrans is non-null, this is a top-level call to
       * traverse_gob, as opposed to a recursive call.  
       */
      if (thistrans) {
	top_level_call= 1;
	newtrans = transpose_trans( duplicate_trans(thistrans) );
	/* Initialize light table and ambient color buffer */
	DLIGHTCOUNT(self)= 0;
	AMBIENTCOLOR(self).r = 0.0;
	AMBIENTCOLOR(self).g = 0.0;
	AMBIENTCOLOR(self).b = 0.0;
	AMBIENTCOLOR(self).a = 0.0;
	internal_traverse(self,thisgob,newtrans->d);
	destroy_trans(newtrans);
      }
      else {
	float *oldtrans= RECENTTRANS(self);
	internal_traverse(self,thisgob,RECENTTRANS(self));
	RECENTTRANS(self)= oldtrans;
      }

      if (thisattrlist) {
	METHOD_RDY(ASSIST(self));
	(*(ASSIST(self)->pop_attributes))( thisattrlist );
      }

    }
    else ger_error("pnt_ren_mthd: traverse_gob: got a null data pointer.");
  }
  METHOD_OUT
}

static void destroy_gob( P_Void_ptr primdata )
/* This routine destroys the renderer's data for the gob. */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  if (RENDATA(self)->open) {
    P_Gob *gob;
    gob= (P_Gob *)primdata;
    ger_debug("pnt_ren_mthd: destroy_gob: destroying gob <%s>",gob->name);
  }
  METHOD_OUT
}

static P_Void_ptr def_sphere(char *name)
/* This routine defines a sphere */
{
  P_Renderer *self= (P_Renderer *)po_this;
  P_Void_ptr result;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: def_sphere");

    METHOD_RDY(ASSIST(self));
    result= (*(ASSIST(self)->def_sphere))();

    METHOD_OUT
    return( (P_Void_ptr)result );
  }  
  METHOD_OUT
  return((P_Void_ptr)0);
}

static void ren_sphere( P_Void_ptr rendata, P_Transform *trans, 
			 P_Attrib_List *attr )
/* This routine renders a sphere */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: ren_sphere");

    METHOD_RDY(ASSIST(self));
    (*(ASSIST(self)->ren_sphere))(rendata,trans,attr);
  }  
  METHOD_OUT
}

static void destroy_sphere( P_Void_ptr rendata )
/* This routine renders a sphere */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: destroy_sphere");

    METHOD_RDY(ASSIST(self));
    (*(ASSIST(self)->destroy_sphere))(rendata);
  }  
  METHOD_OUT
}

static P_Void_ptr def_cylinder(char *name)
/* This routine defines a cylinder */
{
  P_Renderer *self= (P_Renderer *)po_this;
  P_Void_ptr result;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: def_cylinder");

    METHOD_RDY(ASSIST(self));
    result= (*(ASSIST(self)->def_cylinder))();

    METHOD_OUT
    return( (P_Void_ptr)result );
  }  
  METHOD_OUT
  return((P_Void_ptr)0);
}

static void ren_cylinder( P_Void_ptr rendata, P_Transform *trans, 
			 P_Attrib_List *attr )
/* This routine renders a cylinder */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: ren_cylinder");

    METHOD_RDY(ASSIST(self));
    (*(ASSIST(self)->ren_cylinder))(rendata,trans,attr);
  }  
  METHOD_OUT
}

static void destroy_cylinder( P_Void_ptr rendata )
/* This routine renders a cylinder */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: destroy_cylinder");

    METHOD_RDY(ASSIST(self));
    (*(ASSIST(self)->destroy_cylinder))(rendata);
  }  
  METHOD_OUT
}

static P_Void_ptr def_torus(char *name, double major, double minor)
/* This routine defines a torus */
{
  P_Renderer *self= (P_Renderer *)po_this;
  P_Void_ptr result;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: def_torus");

    METHOD_RDY(ASSIST(self));
    result= (*(ASSIST(self)->def_torus))(major,minor);

    METHOD_OUT
    return( (P_Void_ptr)result );
  }  
  METHOD_OUT
  return((P_Void_ptr)0);
}

static void ren_torus( P_Void_ptr rendata, P_Transform *trans, 
			 P_Attrib_List *attr )
/* This routine renders a torus */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: ren_torus");

    METHOD_RDY(ASSIST(self));
    (*(ASSIST(self)->ren_torus))(rendata,trans,attr);
  }  
  METHOD_OUT
}

static void destroy_torus( P_Void_ptr rendata )
/* This routine renders a torus */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: destroy_torus");

    METHOD_RDY(ASSIST(self));
    (*(ASSIST(self)->destroy_torus))(rendata);
  }  
  METHOD_OUT
}

static void get_coords(Pnt_Polytype *record, P_Vlist *vlist)
{
  int length, i;
  float *x, *y, *z;

  METHOD_RDY(vlist);
  length= vlist->length;
  if ( !(record->xcoords= (float *)malloc(length*sizeof(float))) )
    ger_fatal("pnt_ren_mthd: get_coords: unable to allocate %d floats for x!",
	      length);
  if ( !(record->ycoords= (float *)malloc(length*sizeof(float))) )
    ger_fatal("pnt_ren_mthd: get_coords: unable to allocate %d floats for y!",
	      length);
  if ( !(record->zcoords= (float *)malloc(length*sizeof(float))) )
    ger_fatal("pnt_ren_mthd: get_coords: unable to allocate %d floats for z!",
	      length);
  x= record->xcoords;
  y= record->ycoords;
  z= record->zcoords;
  METHOD_RDY(vlist);
  for ( i=0; i<length; i++ ) {
    *x++= (*(vlist->x))(i);
    *y++= (*(vlist->y))(i);
    *z++= (*(vlist->z))(i);
  }
}

static void get_ind_coords(Pnt_Polytype *record, P_Vlist *vlist, 
			   int *indices, int length)
{
  int i;
  float *x, *y, *z;

  if ( !(record->xcoords= (float *)malloc(length*sizeof(float))) )
    ger_fatal(
	"pnt_ren_mthd: get_ind_coords: unable to allocate %d floats for x!",
	      length);
  if ( !(record->ycoords= (float *)malloc(length*sizeof(float))) )
    ger_fatal(
	"pnt_ren_mthd: get_ind_coords: unable to allocate %d floats for y!",
	      length);
  if ( !(record->zcoords= (float *)malloc(length*sizeof(float))) )
    ger_fatal(
        "pnt_ren_mthd: get_ind_coords: unable to allocate %d floats for z!",
	      length);
  x= record->xcoords;
  y= record->ycoords;
  z= record->zcoords;
  METHOD_RDY(vlist);
  for ( i=0; i<length; i++ ) {
    *x++= (*(vlist->x))(*indices);
    *y++= (*(vlist->y))(*indices);
    *z++= (*(vlist->z))(*indices++);
  }
}

static void get_one_coord(Pnt_Polytype *record, P_Vlist *vlist, int i)
{
  if ( !(record->xcoords= (float *)malloc(sizeof(float))) )
    ger_fatal(
       "pnt_ren_mthd: get_one_coord: unable to allocate 1 float for x!");
  if ( !(record->ycoords= (float *)malloc(sizeof(float))) )
    ger_fatal(
       "pnt_ren_mthd: get_one_coord: unable to allocate 1 float for y!");
  if ( !(record->zcoords= (float *)malloc(sizeof(float))) )
    ger_fatal(
       "pnt_ren_mthd: get_one_coord: unable to allocate 1 float for z!");
  METHOD_RDY(vlist);
  *(record->xcoords)= (*(vlist->x))(i);
  *(record->ycoords)= (*(vlist->y))(i);
  *(record->zcoords)= (*(vlist->z))(i);
}

static Pnt_Colortype *get_ave_color(P_Vlist *vlist)
{
  int i,length;
  Pnt_Colortype *result;

  METHOD_RDY(vlist);
  length= vlist->length;
  if ( !(result= (Pnt_Colortype *)malloc(sizeof(Pnt_Colortype))) )
    ger_fatal(
       "pnt_ren_mthd: get_ave_color: unable to allocate %d bytes!",
	      sizeof(Pnt_Colortype));
  result->r= result->g= result->b= 0.0;
  result->a= 1.0; /* opacity currently ignored */
  METHOD_RDY(vlist);
  for ( i=0; i<length; i++ ) {
    result->r += (*(vlist->r))(i);
    result->g += (*(vlist->g))(i);
    result->b += (*(vlist->b))(i);
  }
  result->r /= length;
  result->g /= length;
  result->b /= length;
  return(result);
}

static Pnt_Colortype *get_ind_ave_color(P_Vlist *vlist, int *indices,
					int length)
{
  int i;
  Pnt_Colortype *result;

  if ( !(result= (Pnt_Colortype *)malloc(sizeof(Pnt_Colortype))) )
    ger_fatal(
       "pnt_ren_mthd: get_ind_ave_color: unable to allocate %d bytes!",
	      sizeof(Pnt_Colortype));
  result->r= result->g= result->b= 0.0;
  result->a= 1.0; /* opacity currently ignored */
  METHOD_RDY(vlist);
  for ( i=0; i<length; i++ ) {
    result->r += (*(vlist->r))(*indices);
    result->g += (*(vlist->g))(*indices);
    result->b += (*(vlist->b))(*indices++);
  }
  result->r /= length;
  result->g /= length;
  result->b /= length;
  return(result);
}

static Pnt_Colortype *get_one_color(P_Vlist *vlist, int i)
{
  Pnt_Colortype *result;

  METHOD_RDY(vlist);
  if ( !(result= (Pnt_Colortype *)malloc(sizeof(Pnt_Colortype))) )
    ger_fatal(
       "pnt_ren_mthd: get_one_color: unable to allocate %d bytes!",
	      sizeof(Pnt_Colortype));
  result->a= 1.0; /* opacity currently ignored */
  METHOD_RDY(vlist);
  result->r= (*(vlist->r))(i);
  result->g= (*(vlist->g))(i);
  result->b= (*(vlist->b))(i);
  return(result);
}

static Pnt_Colortype *get_val_color(P_Renderer *self, P_Vlist *vlist)
{
  int length, i;
  Pnt_Colortype *result;
  float r, g, b, a;

  METHOD_RDY(vlist);
  length= vlist->length;
  if ( !(result= (Pnt_Colortype *)malloc(sizeof(Pnt_Colortype))) )
    ger_fatal(
       "pnt_ren_mthd: get_val_color: unable to allocate %d bytes!",
	      sizeof(Pnt_Colortype));
  result->r= result->g= result->b= 0.0;
  result->a= 1.0; /* opacity currently ignored */
  METHOD_RDY(vlist);
  for ( i=0; i<length; i++ ) {
    map_color(self, (*(vlist->v))(i), &r, &g, &b, &a);
    result->r += r;
    result->g += g;
    result->b += b;
  }
  result->r /= length;
  result->g /= length;
  result->b /= length;
  return(result);
}

static Pnt_Colortype *get_ind_val_color(P_Renderer *self, P_Vlist *vlist,
					int *indices, int length)
{
  int i;
  Pnt_Colortype *result;
  float r, g, b, a;

  if ( !(result= (Pnt_Colortype *)malloc(sizeof(Pnt_Colortype))) )
    ger_fatal(
       "pnt_ren_mthd: get_ind_val_color: unable to allocate %d bytes!",
	      sizeof(Pnt_Colortype));
  result->r= result->g= result->b= 0.0;
  result->a= 1.0; /* opacity currently ignored */
  METHOD_RDY(vlist);
  for ( i=0; i<length; i++ ) {
    map_color(self, (*(vlist->v))(*indices++), &r, &g, &b, &a);
    result->r += r;
    result->g += g;
    result->b += b;
  }
  result->r /= length;
  result->g /= length;
  result->b /= length;
  return(result);
}

static Pnt_Colortype *get_one_val_color(P_Renderer *self, P_Vlist *vlist, 
					int i)
{
  Pnt_Colortype *result;
  float r, g, b, a;

  if ( !(result= (Pnt_Colortype *)malloc(sizeof(Pnt_Colortype))) )
    ger_fatal(
       "pnt_ren_mthd: get_one_val_color: unable to allocate %d bytes!",
	      sizeof(Pnt_Colortype));
  result->a= 1.0; /* opacity currently ignored */
  METHOD_RDY(vlist);
  map_color(self, (*(vlist->v))(i), &r, &g, &b, &a);
  result->r += r;
  result->g += g;
  result->b += b;
  return(result);
}

static Pnt_Polytype *get_polyrec( P_Renderer *self, P_Vlist *vlist, 
				 primtype polytype )
{
  Pnt_Polytype *result;

  if ( !(result= (Pnt_Polytype *)malloc( sizeof(Pnt_Polytype) ) ) )
    ger_fatal("pnt_ren_mthd: get_polyrec: unable to allocate %d bytes!\n",
	      sizeof(Pnt_Polytype));

  METHOD_RDY(vlist);
  result->numcoords= vlist->length;
  result->type= polytype;
  get_coords(result,vlist);
  METHOD_RDY(vlist);
  switch (vlist->type) {
  case P3D_CVTX:   
  case P3D_CNVTX:
    result->color= (Pnt_Colortype *)0;
    break;
  case P3D_CCVTX:  
  case P3D_CCNVTX: 
    result->color= get_ave_color(vlist);
    break;
  case P3D_CVVTX:
  case P3D_CVVVTX: /* ignore second value */
  case P3D_CVNVTX: 
    result->color= get_val_color(self,vlist);
    break;
  default: 
    ger_error("pnt_ren_mthd: get_polyrec: unknown vlist type %d!",vlist->type);
  }

  return(result);
}

static Pnt_Polytype *get_mesh_polyrec( P_Renderer *self, P_Vlist *vlist,
				      int *indices, int *facet_lengths, 
				      int nfacets, primtype polytype )
{
  Pnt_Polytype *result, *runner;
  int i;

  if ( !(result= (Pnt_Polytype *)malloc( nfacets*sizeof(Pnt_Polytype) ) ) )
    ger_fatal("pnt_ren_mthd: get_mesh_polyrec: unable to allocate %d bytes!\n",
	      nfacets*sizeof(Pnt_Polytype));

  runner= result;
  METHOD_RDY(vlist);
  for (i=0; i<nfacets; i++) {
    runner->numcoords= *facet_lengths;
    runner->type= polytype;
    get_ind_coords(runner, vlist, indices, *facet_lengths);
    switch (vlist->type) {
    case P3D_CVTX:   
    case P3D_CNVTX:
      runner->color= (Pnt_Colortype *)0;
      break;
    case P3D_CCVTX:  
    case P3D_CCNVTX: 
      runner->color= get_ind_ave_color(vlist, indices, *facet_lengths);
      break;
    case P3D_CVVTX:
    case P3D_CVVVTX: /* ignore second value */
    case P3D_CVNVTX: 
      runner->color= get_ind_val_color(self, vlist, indices, *facet_lengths);
      break;
    default: 
      ger_error("pnt_ren_mthd: get_mesh_polyrec: unknown vlist type %d!",
		vlist->type);
    }
    /* Advance to the next polygon */
    indices += *facet_lengths;
    facet_lengths++;
    runner++;
  }

  return(result);
}

static Pnt_Polytype *get_tri_polyrec( P_Renderer *self, P_Vlist *vlist,
				     primtype type )
{
  Pnt_Polytype *result, *runner;
  int i,nfacets;
  int indices[3];

  nfacets= vlist->length - 2;

  if ( !(result= (Pnt_Polytype *)malloc( nfacets*sizeof(Pnt_Polytype) ) ) )
    ger_fatal("pnt_ren_mthd: get_tri_polyrec: unable to allocate %d bytes!\n",
	      nfacets*sizeof(Pnt_Polytype));

  runner= result;
  METHOD_RDY(vlist);
  for (i=0; i<nfacets; i++) {
    if (i%2) { /* odd, so we flip vertex order */
      indices[0]= i+1;
      indices[1]= i;
      indices[2]= i+2;
    }
    else { /* even- normal vertex order */
      indices[0]= i;
      indices[1]= i+1;
      indices[2]= i+2;
    }
    runner->numcoords= 3;
    runner->type= type;
    get_ind_coords(runner, vlist, indices, 3);
    switch (vlist->type) {
    case P3D_CVTX:   
    case P3D_CNVTX:
      runner->color= (Pnt_Colortype *)0;
      break;
    case P3D_CCVTX:  
    case P3D_CCNVTX: 
      runner->color= get_ind_ave_color(vlist, indices, 3);
      break;
    case P3D_CVVTX:
    case P3D_CVVVTX: /* ignore second value */
    case P3D_CVNVTX: 
      runner->color= get_ind_val_color(self, vlist, indices, 3);
      break;
    default: 
      ger_error("pnt_ren_mthd: get_tri_polyrec: unknown vlist type %d!",
		vlist->type);
    }
    /* Advance to the next polygon */
    runner++;
  }

  return(result);
}

static Pnt_Polytype *get_multiple_polyrecs( P_Renderer *self, P_Vlist *vlist, 
				 primtype polytype )
{
  Pnt_Polytype *result, *runner;
  int i;

  if ( !(result=(Pnt_Polytype *)malloc( vlist->length*sizeof(Pnt_Polytype) )) )
    ger_fatal(
       "pnt_ren_mthd: get_multiple_polyrecs: unable to allocate %d bytes!\n",
	      vlist->length*sizeof(Pnt_Polytype));

  runner= result;
  for (i=0; i<vlist->length; i++) {
    runner->numcoords= 1;
    runner->type= polytype;
    get_one_coord(runner,vlist,i);
    switch (vlist->type) {
    case P3D_CVTX:   
    case P3D_CNVTX:
      runner->color= (Pnt_Colortype *)0;
      break;
    case P3D_CCVTX:  
    case P3D_CCNVTX: 
      runner->color= get_one_color(vlist,i);
      break;
    case P3D_CVVTX:
    case P3D_CVVVTX: /* ignore second value */
    case P3D_CVNVTX: 
      runner->color= get_one_val_color(self,vlist,i);
      break;
    default: 
      ger_error("pnt_ren_mthd: get_multiple_polyrecs: unknown vlist type %d!",
		vlist->type);
    }
    runner++;
  }

  return(result);
}

static P_Void_ptr def_polymarker(char *name, P_Vlist *vlist)
/* This routine defines a polymarker */
{
  P_Renderer *self= (P_Renderer *)po_this;
  Pnt_Objecttype *result;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: def_polymarker");
    if (vlist->length < 1)
      ger_error(
 "pnt_ren_mthd: def_polymarker: %d isn't enough vertices for a polymarker!",
		vlist->length);
    else {
      if ( !(result= (Pnt_Objecttype *)malloc(sizeof(Pnt_Objecttype))) )
	ger_fatal("pnt_ren_mthd: def_polymarker: unable to allocate %d bytes!",
		  sizeof(Pnt_Objecttype));
      result->polygons= get_multiple_polyrecs( self, vlist, POLYMARKER ); 
      result->num_polygons= vlist->length;     
      METHOD_OUT
      return( (P_Void_ptr)result );
    }
  }  
  METHOD_OUT
  return((P_Void_ptr)0);
}

static P_Void_ptr def_polyline(char *name, P_Vlist *vlist)
/* This routine defines a polyline */
{
  P_Renderer *self= (P_Renderer *)po_this;
  Pnt_Objecttype *result;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: def_polyline");
    if (vlist->length < 2)
      ger_error(
	"pnt_ren_mthd: def_polyline: %d isn't enough vertices for a polyline!",
	vlist->length);
    else {
      if ( !(result= (Pnt_Objecttype *)malloc(sizeof(Pnt_Objecttype))) )
	ger_fatal("pnt_ren_mthd: def_polyline: unable to allocate %d bytes!",
		  sizeof(Pnt_Objecttype));
      result->polygons= get_polyrec( self, vlist, POLYLINE ); 
      result->num_polygons= 1;
      METHOD_OUT
      return( (P_Void_ptr)result );
    }
  }  
  METHOD_OUT
  return((P_Void_ptr)0);
}

static P_Void_ptr def_polygon(char *name, P_Vlist *vlist)
/* This routine defines a polyline */
{
  P_Renderer *self= (P_Renderer *)po_this;
  Pnt_Objecttype *result;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: def_polygon");
    if (vlist->length < 3)
      ger_error(
	"pnt_ren_mthd: def_polygon: %d isn't enough vertices for a polygon!",
	vlist->length);
    else {
      if ( !(result= (Pnt_Objecttype *)malloc(sizeof(Pnt_Objecttype))) )
	ger_fatal("pnt_ren_mthd: def_polygon: unable to allocate %d bytes!",
		  sizeof(Pnt_Objecttype));
      result->polygons= get_polyrec( self, vlist, POLYGON ); 
      result->num_polygons= 1;
      METHOD_OUT
      return( (P_Void_ptr)result );
    }
  }  
  METHOD_OUT
  return((P_Void_ptr)0);
}

static void ren_object(P_Void_ptr primdata, P_Transform *trans, 
			 P_Attrib_List *attr)
{
  P_Renderer *self= (P_Renderer *)po_this;
  Pnt_Objecttype *object;
  Pnt_Colortype color;
  P_Color *pcolor;
  int back_cull;
  P_Transform *newtrans;
  METHOD_IN

  /* The mechanism by which this gets called guarantees attr is null. */

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: ren_object");
    object= (Pnt_Objecttype *)primdata;
    if (!object) {
      ger_error("pnt_ren_mthd: ren_object: null object data found.");
      METHOD_OUT
      return;
    }

    /* Inherit needed attributes */
    METHOD_RDY(ASSIST(self));
    pcolor= (*(ASSIST(self)->color_attribute))(COLORSYMBOL(self));
    back_cull= (*(ASSIST(self)->bool_attribute))(BACKCULLSYMBOL(self));
    rgbify_color(pcolor);
    color.r= pcolor->r;
    color.g= pcolor->g;
    color.b= pcolor->b;
    color.a= pcolor->a;

    /* The assist object can call this routine with non-null transforms
     * under some circumstances, so we need to handle that case.
     */
    if (trans) {
      P_Transform *newtrans;
      float *totaltrans;
      newtrans = transpose_trans( duplicate_trans(trans) );
      totaltrans = pnt_mult3dMatrices( newtrans->d, RECENTTRANS(self) );
      pnt_render_primitive(self, object, totaltrans, back_cull, &color);
      destroy_trans(newtrans);
      free( (P_Void_ptr)totaltrans );
    }
    else {
      pnt_render_primitive(self, object, RECENTTRANS(self), back_cull, &color);
    }
  }  

  METHOD_OUT
}

static void destroy_polyrec(Pnt_Polytype *rec)
{
  if (rec->xcoords) free( (P_Void_ptr)(rec->xcoords) );
  if (rec->ycoords) free( (P_Void_ptr)(rec->ycoords) );
  if (rec->zcoords) free( (P_Void_ptr)(rec->zcoords) );
  if (rec->color) free( (P_Void_ptr)(rec->color) );
  free( (P_Void_ptr)rec );
}

static void destroy_objecttype(P_Void_ptr primdata)
{
  P_Renderer *self= (P_Renderer *)po_this;
  Pnt_Objecttype *obj= (Pnt_Objecttype *)primdata;
  Pnt_Polytype *rec;
  int i;
  METHOD_IN

  rec= obj->polygons;
  for (i=0; i<obj->num_polygons; i++)
    destroy_polyrec( rec++ );
  free( (P_Void_ptr)obj );

  METHOD_OUT
}

static P_Void_ptr def_tristrip(char *name, P_Vlist *vlist)
/* This routine defines a triangle strip */
{
  P_Renderer *self= (P_Renderer *)po_this;
  Pnt_Objecttype *result;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: def_tristrip");
    if (vlist->length < 3)
      ger_error(
	"pnt_ren_mthd: def_tristrip: %d isn't enough vertices for a tristrip!",
		vlist->length);
    else {
      if ( !(result= (Pnt_Objecttype *)malloc(sizeof(Pnt_Objecttype))) )
	ger_fatal("pnt_ren_mthd: def_tristrip: unable to allocate %d bytes!",
		  sizeof(Pnt_Objecttype));
      result->polygons= get_tri_polyrec( self, vlist, POLYGON );
      result->num_polygons= vlist->length - 2;
      
      METHOD_OUT
      return( (P_Void_ptr)result );
    }
  }
  METHOD_OUT
  return((P_Void_ptr)0);
}

static P_Void_ptr def_bezier(char *name, P_Vlist *vlist)
/* This routine defines a 4-by-4 bicubic Bezier patch */
{
  P_Renderer *self= (P_Renderer *)po_this;
  P_Void_ptr result;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: def_bezier");

    METHOD_RDY(ASSIST(self));
    result= (*(ASSIST(self)->def_bezier))(vlist);

    METHOD_OUT
    return( result );
  }  
  METHOD_OUT
  return((P_Void_ptr)0);
}

static void ren_bezier( P_Void_ptr rendata, P_Transform *trans, 
			 P_Attrib_List *attr )
/* This routine renders a 4-by-4 bicubic Bezier patch */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: ren_bezier");

    METHOD_RDY(ASSIST(self));
    (*(ASSIST(self)->ren_bezier))(rendata,trans,attr);

  }  
  METHOD_OUT
}

static void destroy_bezier( P_Void_ptr rendata )
/* This routine destroys a 4-by-4 bicubic Bezier patch */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: destroy_bezier");

    METHOD_RDY(ASSIST(self));
    (*(ASSIST(self)->destroy_bezier))(rendata);

  }  
  METHOD_OUT
}

static P_Void_ptr def_mesh(char *name, P_Vlist *vlist, int *indices,
                           int *facet_lengths, int nfacets)
/* This routine defines a general mesh */
{
  P_Renderer *self= (P_Renderer *)po_this;
  Pnt_Objecttype *result;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: def_mesh");
    if (vlist->length < 3)
      ger_error(
		"pnt_ren_mthd: def_mesh: %d isn't enough vertices for a mesh!",
		vlist->length);
    else if (nfacets<1)
      ger_error(
		"pnt_ren_mthd: def_mesh: %d isn't enough facets for a mesh!",
		nfacets);
    else {
      if ( !(result= (Pnt_Objecttype *)malloc(sizeof(Pnt_Objecttype))) )
	ger_fatal("pnt_ren_mthd: def_mesh: unable to allocate %d bytes!",
		  sizeof(Pnt_Objecttype));
      result->polygons= get_mesh_polyrec( self, vlist, indices,
					 facet_lengths, nfacets, POLYGON ); 
      result->num_polygons= nfacets;
      
      METHOD_OUT
      return( (P_Void_ptr)result );
    }
  }
  METHOD_OUT
  return((P_Void_ptr)0);
}

static P_Void_ptr def_text( char *name, char *tstring, P_Point *location, 
                           P_Vector *u, P_Vector *v )
/* This method defines a text string */
{
  P_Renderer *self= (P_Renderer *)po_this;
  P_Void_ptr result;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: def_text");
    METHOD_RDY(ASSIST(self));
    result= (*(ASSIST(self)->def_text))( tstring, location, u, v );
    METHOD_OUT
    return( (P_Void_ptr)result );
  }  
  METHOD_OUT
  return((P_Void_ptr)0);
}

static void ren_text( P_Void_ptr rendata, P_Transform *trans, 
			 P_Attrib_List *attr )
/* This method renders a text string */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: ren_text");
    METHOD_RDY(ASSIST(self));
    (*(ASSIST(self)->ren_text))(rendata, trans, attr);
  }  
  METHOD_OUT
}

static void destroy_text( P_Void_ptr rendata )
/* This method destroys a text string */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: destroy_text");

    METHOD_RDY(ASSIST(self));
    (*(ASSIST(self)->destroy_text))(rendata);

  }  
  METHOD_OUT
}

static P_Void_ptr def_light( char *name, P_Point *location, P_Color *color )
/* This method defines a positional light source */
{
  P_Renderer *self= (P_Renderer *)po_this;
  Pnt_Lighttype *result;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: def_light");

    if ( !(result= (Pnt_Lighttype *)malloc(sizeof(Pnt_Lighttype))) )
      ger_fatal("pnt_ren_mthd: def_ambient: unable to allocate %d bytes!",
		sizeof(Pnt_Lighttype));

    rgbify_color(color);
    result->color.r= color->r;
    result->color.g= color->g;
    result->color.b= color->b;
    result->color.a= color->a;
    result->loc.x= location->x;
    result->loc.y= location->y;
    result->loc.z= location->z;
    METHOD_OUT
    return( (P_Void_ptr)result );
  }  
  METHOD_OUT
  return((P_Void_ptr)0);
}

static void traverse_light( P_Void_ptr primdata, P_Transform *trans, 
			 P_Attrib_List *attr )
{
  P_Renderer *self= (P_Renderer *)po_this;
  Pnt_Lighttype *light;
  register float *rtrans= RECENTTRANS(self);
  float lx,ly,lz;
  METHOD_IN

  if (RENDATA(self)->open) {
    int lightindex;

    ger_debug("pnt_ren_mthd: traverse_light");

    light= (Pnt_Lighttype *)primdata;
    if (!light) {
      ger_error("pnt_ren_mthd: traverse_light: null object data found.");
      METHOD_OUT
      return;
    }
    
    lightindex= pnt_new_DLightIndex(self);
    lx= light->loc.x;
    ly= light->loc.y;
    lz= light->loc.z;
    DLIGHTBUFFER(self)[lightindex].loc.x=
      rtrans[0]*lx + rtrans[4]*ly + rtrans[8]*lz + rtrans[12];
    DLIGHTBUFFER(self)[lightindex].loc.y=
      rtrans[1]*lx + rtrans[5]*ly + rtrans[9]*lz + rtrans[13];
    DLIGHTBUFFER(self)[lightindex].loc.z=
      rtrans[2]*lx + rtrans[6]*ly + rtrans[10]*lz + rtrans[14];
    DLIGHTBUFFER(self)[lightindex].color.r= light->color.r;
    DLIGHTBUFFER(self)[lightindex].color.g= light->color.g;
    DLIGHTBUFFER(self)[lightindex].color.b= light->color.b;
    DLIGHTBUFFER(self)[lightindex].color.a= light->color.a;
  }  
  METHOD_OUT
}

static void destroy_light( P_Void_ptr primdata )
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN
  ger_debug("pnt_ren_mthd: destroy_light");
  free( primdata );
  METHOD_OUT
}

static P_Void_ptr def_ambient( char *name, P_Color *color )
/* This method defines an ambient light source */
{
  P_Renderer *self= (P_Renderer *)po_this;
  Pnt_Colortype *result;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: def_ambient");

    if ( !(result= (Pnt_Colortype *)malloc(sizeof(Pnt_Colortype))) )
      ger_fatal("pnt_ren_mthd: def_ambient: unable to allocate %d bytes!",
		sizeof(Pnt_Colortype));
    rgbify_color(color);
    result->r= color->r;
    result->g= color->g;
    result->b= color->b;
    result->a= color->a;
    METHOD_OUT
    return( (P_Void_ptr)result );
  }  
  METHOD_OUT
  return((P_Void_ptr)0);
}

static void traverse_ambient( P_Void_ptr primdata, P_Transform *trans, 
			 P_Attrib_List *attr )
{
  P_Renderer *self= (P_Renderer *)po_this;
  Pnt_Colortype *color;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: traverse_ambient");

    color= (Pnt_Colortype *)primdata;
    if (!color) {
      ger_error("pnt_ren_mthd: traverse_ambient: null object data found.");
      METHOD_OUT
      return;
    }
    AMBIENTCOLOR(self).r += color->r;
    AMBIENTCOLOR(self).g += color->g;
    AMBIENTCOLOR(self).b += color->b;
    AMBIENTCOLOR(self).a += color->a;
  }  
  METHOD_OUT
}

static void destroy_ambient( P_Void_ptr primdata )
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN
  ger_debug("pnt_ren_mthd: destroy_ambient");
  free( primdata );
  METHOD_OUT
}

static void ren_donothing( P_Void_ptr primdata, P_Transform *thistrans, 
		    P_Attrib_List *thisattrlist )
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN
  ger_debug("pnt_ren_mthd: ren_donothing: doing nothing.");
  METHOD_OUT
}

static P_Void_ptr def_gob( char *name, P_Gob *gob )
/* This method defines a gob */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: def_gob");
    METHOD_OUT
    return( (P_Void_ptr)gob ); /* Stash the gob where we can get it later */
  }  
  METHOD_OUT
  return((P_Void_ptr)0);
}

static void hold_gob( P_Void_ptr primdata )
/* This routine holds gob */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: hold_gob: doing nothing");
  }
  METHOD_OUT
}

static void unhold_gob( P_Void_ptr primdata )
/* This routine unholds gob */
{
  P_Renderer *self= (P_Renderer *)po_this;
  METHOD_IN

  if (RENDATA(self)->open) {
    ger_debug("pnt_ren_mthd: unhold_gob: doing nothing");
  }
  METHOD_OUT
}

static void setup_Buffers( P_Renderer *self )
{   
  DEPTHPOLYCOUNT(self) =0;
  DPOLYBUFFER(self) = (Pnt_DPolytype *)
    malloc( MAXDEPTHPOLY(self)*sizeof(Pnt_DPolytype));
  if (!DPOLYBUFFER(self))
    ger_fatal("ERROR: COULD NOT ALLOCATE %d POLYGONS\n",
	      MAXDEPTHPOLY(self));
  DEPTHCOUNT(self) = 0;
  DEPTHBUFFER(self) = (depthtable_rec *)
    malloc( MAXPOLYCOUNT(self)*sizeof(depthtable_rec) );
  if (!DEPTHBUFFER(self))
    ger_fatal("ERROR: COULD NOT ALLOCATE %d POLYGON RECORDS\n",
	      MAXPOLYCOUNT(self));
  DLIGHTCOUNT(self)= 0;
  DLIGHTBUFFER(self)= (Pnt_Lighttype *)
    malloc( MAXDLIGHTCOUNT(self)*sizeof(Pnt_Lighttype) );
  if (!DLIGHTBUFFER(self))
    ger_fatal(
	 "pnt_ren_mthd: setup_Buffers: could not allocate %d light recors!",
	      MAXDLIGHTCOUNT(self));
}

P_Renderer *po_create_painter_renderer( char *device, char *datastr )
/* This routine creates a ptr-generating renderer object */
{
  P_Renderer *self;
  P_Renderer_data *rdata;
  static int sequence_number= 0;
  int length,i;

  ger_debug("po_create_painter_renderer: device= <%s>, datastr= <%s>",
	    device, datastr);

  /* Check to make sure this is the only instance, since
   * DRAWCGM can only handle one open device.
   */
  if (sequence_number>0) {
    ger_error(
"Sorry, only one instance of the Painter renderer is supported (for now).");
    return((P_Renderer *)0);
  }

  /* Create memory for the renderer */
  if ( !(self= (P_Renderer *)malloc(sizeof(P_Renderer))) )
    ger_fatal("po_create_painter_renderer: unable to allocate %d bytes!",
              sizeof(P_Renderer) );

  /* Create memory for object data */
  if ( !(rdata= (P_Renderer_data *)malloc(sizeof(P_Renderer_data))) )
    ger_fatal("po_create_painter_renderer: unable to allocate %d bytes!",
              sizeof(P_Renderer_data) );
  self->object_data= (P_Void_ptr)rdata;

  /* Fill out default color map */
  strcpy(default_map.name,"default-map");
  default_map.min= 0.0;
  default_map.max= 1.0;
  default_map.mapfun= default_mapfun;

  /* Fill out public and private object data */
  sprintf(self->name,"painter%d",sequence_number++);
  rdata->open= 0;  /* renderer created in closed state */
  CUR_MAP(self)= &default_map;

  ASSIST(self)= po_create_assist(self);

  BACKCULLSYMBOL(self)= create_symbol("backcull");
  TEXTHEIGHTSYMBOL(self)= create_symbol("text-height");
  COLORSYMBOL(self)= create_symbol("color");
  MATERIALSYMBOL(self)= create_symbol("material");
  
  length = 1 + strlen(device);
  DEVICENAME(self) = malloc(length);
  strcpy(DEVICENAME(self),device);
  
  length = 1 + strlen(datastr);
  OUTFILE(self) = malloc(length);
  strcpy(OUTFILE(self),datastr);
  
  DCOORDINDEX(self)=0;
  MAXDCOORDINDEX(self) = INITIAL_MAX_DCOORD;
  DCOORDBUFFER(self) = (float *) malloc(MAXDCOORDINDEX(self)*sizeof(float));
  
  DCOLORCOUNT(self) = 0;
  MAXDCOLORCOUNT(self) = INITIAL_MAX_DCOLOR;
  DCOLORBUFFER(self) = (Pnt_Colortype *)
    malloc( MAXDCOLORCOUNT(self)*sizeof(Pnt_Colortype) );

  if ( !DCOORDBUFFER(self) || !DCOLORBUFFER(self) )
    ger_fatal("po_create_painter_renderer: memory allocation failed!");

  MAXPOLYCOUNT(self) = INITIAL_MAX_DEPTHPOLY;
  MAXDEPTHPOLY(self) = MAXPOLYCOUNT(self);
  
  MAXDLIGHTCOUNT(self)= INITIAL_MAX_DLIGHTS;

  pnt_init_renderer(self);
  
  setup_Buffers(self);

  RENDATA(self)->initialized= 1;

  /* Fill in all the methods */
  self->def_sphere= def_sphere;
  self->ren_sphere= ren_sphere;
  self->destroy_sphere= destroy_sphere;

  self->def_cylinder= def_cylinder;
  self->ren_cylinder= ren_cylinder;
  self->destroy_cylinder= destroy_cylinder;

  self->def_torus= def_torus;
  self->ren_torus= ren_torus;
  self->destroy_torus= destroy_torus;

  self->def_polymarker= def_polymarker;
  self->ren_polymarker= ren_object;
  self->destroy_polymarker= destroy_objecttype;

  self->def_polyline= def_polyline;
  self->ren_polyline= ren_object;
  self->destroy_polyline= destroy_objecttype;

  self->def_polygon= def_polygon;
  self->ren_polygon= ren_object;
  self->destroy_polygon= destroy_objecttype;

  self->def_tristrip= def_tristrip;
  self->ren_tristrip= ren_object;
  self->destroy_tristrip= destroy_objecttype;

  self->def_bezier= def_bezier;
  self->ren_bezier= ren_bezier;
  self->destroy_bezier= destroy_bezier;

  self->def_mesh= def_mesh;
  self->ren_mesh= ren_object;
  self->destroy_mesh= destroy_objecttype;

  self->def_text= def_text;
  self->ren_text= ren_text;
  self->destroy_text= destroy_text;

  self->def_light= def_light;
  self->ren_light= ren_donothing;
  self->light_traverse_light= traverse_light;
  self->destroy_light= destroy_light;

  self->def_ambient= def_ambient;
  self->ren_ambient= ren_donothing;
  self->light_traverse_ambient= traverse_ambient;
  self->destroy_ambient= destroy_ambient;

  self->def_gob= def_gob;
  self->ren_gob= ren_gob;
  self->light_traverse_gob= traverse_gob;
  self->hold_gob= hold_gob;
  self->unhold_gob= unhold_gob;
  self->destroy_gob= destroy_gob;

  self->print= ren_print;
  self->open= ren_open;
  self->close= ren_close;
  self->destroy_self= ren_destroy;

  self->def_camera= def_camera;
  self->set_camera= set_camera;
  self->destroy_camera= destroy_camera;

  self->def_cmap= def_cmap;
  self->install_cmap= install_cmap;
  self->destroy_cmap= destroy_cmap;

  return( self );
}
